

/*---------------------------------------------------------------------------
 * I've taken the liberty to assume:
 *    - 64 K RAM
 *    - Instrument tables IMMEDIATELY follow table of pointers.
 *    - Files (on the default disk) with the extension .SND will be
 * included in the "SOUND LIBRARY".
 *--------------------------------------------------------------------------*/

#include <stdio.h>
#include <maestro.h>
			/* Parameters for the send_bytes routine:	 */
int nlen;		/*	nlen   = number of bytes to send to 8051 */
			/* ??? nbuf[0 ... nlen-1] ???			 */
char herror;		/*	herror = error flag;			 */
			/*		 Set herror to 0 before call.	 */
			/*		 If herror # 0 on return, some	 */
			/*		 error has occurred.		 */
static unsigned tbl_loc = BEG_ORCH; /* RAM address at which to load
				     * the instrument table
				     * (just beyond the previous
				     * instrument).			 */
static unsigned ptr_loc = BEG_PTR;  /* RAM address at which to load
				     * pointers to the instrument tables.*/
static char bin_hedr[] = {0xFF,0xFF,'B','I','N','A','R','Y','\0'};
			/* Header used to flag binary file after	 */
			/* file pre-processing routine converts 	 */
			/* text data to binary data.			 */
extern char dump;	/* dump option flag.				 */

/*---------------------------------------------------------------------------
 * 'reset' re-initializes the static pointers which point to RAM locations
 * for loading instrument tables and the corresponding pointer table.
 * The buffers containing the orchestro list and the player allocations
 * are also re-initialized.  This allows for re-allocation of everything
 * as if 'maestro' had been started from scratch.
 *--------------------------------------------------------------------------*/
reset ()
{
	extern int buf_end;	 /* Last character in orch_buf. 	      */
	extern int o_len;	 /* Number of instruments in orchestra.
				  * i.e. Length of olist.		      */
	extern int p_inst[8];
				/* Array of pointers into the orchestra
				 * list to specify which instrument
				 * each of the 8 players is playing.	     */
	int i;

	tbl_loc = BEG_ORCH;
	ptr_loc = BEG_PTR;
	buf_end = 0;
	o_len = 0;
	for (i = 0; i <= 7; ++i) {
		p_inst[i] = -1;
	}
}

/*---------------------------------------------------------------------------
 * orch adds the specified instrument table files to the orchestra.
 * The array filename points to the specified file names.
 *---------------------------------------------------------------------------
 * First, the instrument table files are loaded into RAM one at a time.
 * Then orch loads into RAM a pointer table of the RAM addresses
 * for those files.
 *--------------------------------------------------------------------------*/
orch(filename, nof)
char *filename[];	/* Array of filenames of files to be loaded into
			 * the instrument table area of RAM.		    */
int nof;		/* Number of files to be loaded.		    */
{
	char ptr_buf[BEG_ORCH-BEG_PTR];
			/* Buffer of pointers to instrument tables loaded
			 * into RAM; one pointer for each instrument loaded
			 * (stored as two bytes: msb, lsb).  This table is
			 * also loaded into RAM (at ptr_loc).		    */
	int j;		/* Index into ptr[];				    */
	int len;	/* temporary variable (ultimately unnecessary)	    */
			/* length of file table 			    */
	int ret_code=0; /* Returns 0 if all was successful, ERR if not.     */
	int i, k;

	j = 0;		/* leave room for LRAM command in buffer	    */
	for (i = 0; i < nof; ++i) {
		if (ptr_loc + j  >= BEG_ORCH) {
			printf("Pointer table ran into Instrument tables.\n");
			printf("All instruments beyond %s not loaded.\n",
				filename[i-1]);
		   printf("ORCH Terminated early.\n");
		/* Special requirements: olist now has more instruments
		 *	than the orchestra!  o_len is too big!		    */

		   return;
		}
		ptr_buf[j++] = tbl_loc >> 8;	/* pointer msb		    */
		ptr_buf[j++] = tbl_loc & LSB;	/* pointer lsb		    */
		if ((len = lram(filename[i], tbl_loc)) == ERR) {
			ret_code = ERR;
			printf("Error loading file %s\n",filename[i]);
			printf("%s not added to orchestra.\n\n",filename[i]);
			j -= 2;
		}
		else {
		     tbl_loc += len;
		     if (dump) {
			printf("Table length is %4xH.  ",len);
			printf("Store next table at %4xH.\n\n",tbl_loc);
			printf("Type any character to go on.\n");
			getch();
		     }
		}
	}

	/* Now load table of pointers to the instrument tables just loaded. */
	/* (Later: Why bother saving the whole table?)			    */
	/* (Later: Why change "working code"?)                              */
	nbuf[0] = LRAM; 	     /* 8051's LRAM command byte         */
	nbuf[1] = ptr_loc >> 8;      /* destination address (msb).	 */
	nbuf[2] = ptr_loc & LSB;     /* destination address (lsb).	 */
	nbuf[3] = j >> 8;	     /* number of bytes in table (msb).  */
	nbuf[4] = j & LSB;	     /* number of bytes in table (lsb).  */
	for (i = 0; i < j; ++i)  nbuf[i+5] = ptr_buf[i];
	nlen = j+5;
	if (dump) {
		printf("\n  LRAM command sent: \n");
		printf("       Command byte: %2x\n",nbuf[0]);
		printf("     Location (msb): %2x\n",nbuf[1]);
		printf("     Location (lsb): %2x\n",nbuf[2]);
		printf("Size of table (msb): %2x\n",nbuf[3]);
		printf("Size of table (lsb): %2x\n",nbuf[4]);
		printf("Pointer table sent: \n");
		for (i=5; i<nlen; ++i) printf("%22x\n",nbuf[i]);
		printf("Press any key to go on.\n");
		getch();
	}
	herror = 0;
	send_bytes();
	if (herror != 0) {
		if (1-dump) cursor(23,1);
		printf("\nError talking to 8051. ");
		printf("(Handshake error = %d)\n",herror);
		ret_code = ERR;
	}

	ptr_loc += j;
	return(ret_code);
}

/**************************************************************************
 * This routine sends the LRAM (load ram) command sequence and the contents
 * of the specified instrument file stored on disk to the 8051 (using the
 * send_bytes routine included in JCODE.OBJ).  In response, the 8051 will
 * load the instrument file into its ram.  The LRAM command sequence
 * is as follows:
 *	byte 0	-  16H
 *	byte 1	-  ram destination address (msb)
 *	byte 2	-  ram destination address (lsb)
 *	byte 3	-  length: # of bytes to load into ram (msb)
 *	byte 4	-  length: # of bytes to load into ram (lsb)
 *	byte 5 ... byte (length - 1 + 5)  -  data bytes to be loaded
 *
 *************************************************************************/
lram(instname, rloc)
char instname[];	/* file from which to extract instrument table	 */
unsigned rloc;		/* RAM location for instrument table		 */
{
	static char ext[] = ".SND";
	char file[14];	/* Copy of inst name plus .SND extension.	 */
	int i,j;

	for (i=0; (file[i] = instname[i]) != '\0'; i++);
	for (j=0; j < 5; file[i+j] = ext[j++]); /* Append extenstion.	     */

	if (read_bin(file) == ERR) {
		printf("Bad data: task terminated; ");
		printf("(nothing was loaded into RAM).\n");
		return(ERR);
	}
	if ((rloc + nlen - 5) > MAXTABL) {
		printf("RAM capacity would have been exceeded ");
		printf("(so I didn't load this file).  Sorry.\n");
		return(ERR);
	}
	nbuf[0] = LRAM; 		/* LRAM command byte	     */
	nbuf[1] = rloc >> 8;		/* 8051 ram destination msb  */
	nbuf[2] = rloc & LSB;		/* 8051 ram destination lsb  */
	nbuf[3] = nlen >> 8;		/* length msb		     */
	nbuf[4] = nlen - 5 & LSB;	/* length lsb		     */
	if (dump) {
		printf("\nInstrument: %s\n", instname);
		printf("\n  LRAM command sent: \n");
		printf("       Command byte: %2x\n",nbuf[0]);
		printf("     Location (msb): %2x\n",nbuf[1]);
		printf("     Location (lsb): %2x\n",nbuf[2]);
		printf("Size of table (msb): %2x\n",nbuf[3]);
		printf("Size of table (lsb): %2x\n",nbuf[4]);
		printf("Byte %4x H of table (first): %2x\n",1,nbuf[5]);
		printf("Byte %4x H of table (last):  %2x\n",
			nlen-5,nbuf[nlen-1]);
	}
	herror = 0;
	send_bytes();
	if (herror != 0) {
		if (1-dump) cursor(23,1);
		printf("\nError talking to 8051. ");
		printf("(Handshake error = %d)\n",herror);
	}
	return(nlen - 5);
}

/*-----------------------------------------------------------------------------
 * alocvoic(v,n) tells the 8051 driver to allocate voice v to the nth instrument
 * stored in the orchestra.  (If n = -1, voice v will be silenced.)
 *-----------------------------------------------------------------------------
 */
alocvoic(voice,inst_num)
int	voice, inst_num;
{
	extern int o_len;	/* Number of instruments in orchestra.	     */
	int i;		    /* Ultimately unnecessary.			     */

	/* 8051's command byte to allocate voice assignments is composed of
	 * 5 bits to specify the command (most significant bits) and
	 * 3 bits to specify the voice (least significant bits).	     */

	if (inst_num < 0 || inst_num >= o_len) {
		printf("\nError encountered in allocating voice.");
		return(ERR);
	}
	nbuf[0] = AVOIC | voice;	/* bitwise inclusive OR 	     */
	nbuf[1] = inst_num;
	nlen	= 2;
	herror	= 0;
	if (dump) {
		cursor(23,1);
		printf("LVOIC command most recently sent:   ");
		printf("  %2x  %2x  ",nbuf[0],nbuf[1]);
	}
	send_bytes();
	if (herror != 0) {
		cursor(24,1);
		printf("Error talking to 8051.");
		printf("(herror = %d)",herror);
	}
	return(0);

}

/*************************************************************************
 * This routine reads the file fp and, given the following rules,
 * prepares nbuf and nlen with data to be sent to send_bytes.
 *
 *    - Only input lines beginning with 'DB' will be accepted
 *	as data byte input for AMY.  All other lines will be ignored.
 *    - Each data byte is defined by a string of valid hex characters
 *	ending with 'H'.
 *
 * Then, the file fp is deleted and a new, binary file with the same name
 * and an extension of .SND is created using the data extracted from the
 * original text file.
 *
 * Upon return, nbuf[5 ... nlen-1] contains the data bytes for send_bytes.
 *
 *************************************************************************/
#define len_inpt 80    /* Maximum characters accepted as string input.	     */

massage(file)
char file[];	    /* file from which to extract instrument table   */
{
	FILE *fp, *fopen();
	char byte[len_inpt];		/* Input string for string to hex
					 * conversion.			     */
	static char ext[] = ".SND";
	int fd; 			/* File descriptor for binary file.  */
	int i;

	if ((fp = fopen(file, "r")) == NULL) {
		printf("massage: can't open %s\n", file);
		return(ERR);
	}
	nlen = 5;
	while ((i = rstring(fp,byte,len_inpt)) != EOF) {
		/*   New input line to process: 			     */
		if (byte[0] == 'D' && byte[1] == 'B' && byte[2] == '\0') {
			/* Process line which begins with 'DB':              */
			while (rstring(fp,byte,len_inpt) == 0 ) {
				if (nlen > MAXNBUF) {
				     printf("File exceeded size constraint.\n");
				     fclose(fp);
				     return(ERR);
				}
				if (hex(byte,&nbuf[nlen++]) == ERR) {
				     fclose(fp);
				     return(ERR);
				}
			}
			if (nlen > MAXNBUF) {
				printf("File exceeded size constraint.\n");
				fclose(fp);
				return(ERR);
			}
			if (hex(byte,&nbuf[nlen++]) == ERR) {
				fclose(fp);
				return(ERR);
			}
		}
		else if (i == EOL);	/* ignore blank line		     */

		else while (rstring(fp,byte,len_inpt) == 0 ); /* ignore all other lines */
	}
	fclose(fp);
	if (nlen == 5) return(ERR);	/* The file didn't make sense to me;
					 * I got no data from it.	     */
	if (unlink(file) == -1) {	/* Destroy old file.		     */
		printf("Error unlinking %s.\n",file);
		return(ERR);
	}
	i = strlen(file);
	file[i-3] = ext[1];		/* Change extension to .SND.	     */
	file[i-2] = ext[2];
	file[i-1] = ext[3];

	fd = creat(file,0x8001);	/* Create new, binary file.	     */
	i = write(fd,bin_hedr,9);	/* Write converted data into new file*/
	if ((write(fd, &nbuf[5], nlen-5) != nlen-5) || (i != 9)) {
		close(fd);
		printf("Error encountered on writing new binary file.\n");
		printf("File contents might have been lost.  Sorry.\n");
		printf("Press any key to go on.\n");
		getch();
		return(ERR);
	}
	if (dump) printf("Number of data bytes = %5d: ",nlen-5);
	close(fd);
	return(0);	/* All seems to have worked as expected.	     */
			/* The file has been converted to binary.	     */
}

/*************************************************************************
 * Reads contents of binary file (excluding file header) into nbuf[5 . . .].
 *
 * (If file does not begin with the expected file header, massage() is
 * invoked to attempt to read the file as a text file (and convert it to
 * a binary file).)
 *************************************************************************/
read_bin(file)
char file[];		/* file from which to extract instrument table	     */
{
	int fd; 	/* File descriptor.				     */
	int fsize;	/* Number of bytes actually read from file.	     */
	char chk_hedr[9];

	fd = open(file,0x8000); /* Open file to read in untranslated mode.   */
	read(fd, chk_hedr, 9);
	if (strcmp(chk_hedr,bin_hedr) != 0) {
		close(fd);
		printf("%s is not in binary form expected.\n",file);
		if (massage(file) == ERR) return(ERR);
		printf("\n%s seemed to be a text file, so I went ", file);
		printf("ahead and imposed\nmy pre-processing gimmick on it.");
		printf("  Hence, the file is now a binary file.\n");
		printf("Sorry 'bout that if you didn't want me to do it.  ");
		printf("I am so bad sometimes . . .\n");
		printf("Type any key to go on.\n");
		getch();
		return(0);
	}
	nlen = 5;
	fsize = read(fd, &nbuf[nlen], MAXNBUF-5);
	if (fsize < 0 ) {
		printf("I couldn't read the file, %s\n",file);
		return(ERR);
	}
	if (fsize == MAXNBUF-5) {
		printf("The file, %s, is too large.\n",file);
		return(ERR);
	}
	nlen += fsize;
	return(0);
}

/*************************************************************************
 * Convert string to hex byte and return the value.
 *
 * The string must consist of only one byte's worth of valid alphanumeric
 * hex characters followed by an 'H'.  Otherwise ERR is returned.
 *
 * The integral value of the input string will be returned and will be
 * stored at the address passed as the second parameter.
 *
 *************************************************************************/
hex(s,d)
char s[];
int  *d;
{
	int i, h, result;

	/* Delete trailing 'H'  */
	for (i = 0; s[i] != '\0'; i++);
	if (s[--i] == 'H') s[i] = '\0';
	else return(ERR);

	result = 0;
	for ( i = 0; s[i] == '0'; i++); /* Skip leading 0's             */
	if ((h = c2h(s[i])) != -1) {
		i++;
		result = h;
		if ((h = c2h(s[i])) != -1) {
			i++;
			result = (result << 4) | h;
		}
	}
	if (s[i] == '\0')  return(*d = result);
	return(ERR);
}

/*************************************************************************
 * Return the integral value of a valid hex character.
 * (Return ERR if character passed is invalid.)
 *************************************************************************/
c2h(c)
int c;
{
	if (c >= '0' && c <= '9') return(c - '0');
	if (c >= 'a' && c <= 'f') return(c - 'a' + 10);
	if (c >= 'A' && c <= 'F') return(c - 'A' + 10);
	return(ERR);
}

/*************************************************************************
 * Get a positive 3-digit decimal number from console.
 * Special responses to the following inputs:
 *		<CR> --> terminate number with less than 3 digits.
 *		 '*' --> dump option is toggled and 0-'*' is returned.
 *		 ESC --> - ESC is returned.
 * invalid character --> ERR is returned.
 *************************************************************************/
getnum()
{
	int value = 0;	/*	    value of input.			     */
	char c[4];
	int i,j;

	for (i = 0; ((c[i] = getch()) != 13) && (i < 4); ++i) {
		if (c[i] == ESC) return(0 - ESC);
		else if (c[i] == '*') {
			dump = dump ^ 1;	/* exclusive-or 	     */
			return(0 - '*');
		}
		else if (c[i] == '\b') {
			if (i == 0) {
				--i;
			}
			else {
			/*	printf("\b \b"); <--Doesn't work! (Why not?)  */
				putch('\b');
				putch(' ');
				putch('\b');
				i -= 2;
			}
		}
		else putch(c[i]);
	}
	if (i == 4 || i == 0) return(ERR);

	for (j = 0; j < i; ++j) {
		if (c[j] >= '0' && c[j] <= '9')
			value = value * 10 + (c[j] - '0');
		else return(ERR);
	}
	return(value);
}
/*************************************************************************
 * Reads the next string from the file fp, character by character into s[]
 * until the next blank, comma, tab, or end of line.
 * Returns either: EOF if the string was the last of the file.
 *		   EOL if the string was the last on that input line.
 *		   0   if neither of the above (including the case where
 *		       the string length has reached lim).
 *************************************************************************/
rstring(fp, s, lim)
FILE *fp;
char s[];
int lim;	/* Max number of characters to read into s[]		 */
{
	int c, i;

	s[0] = '\0';
						/* ignore leading blanks */
	while ((c = getc(fp)) == ' ' || c == '\t');
	for ( i = 0; i <= lim-1; ++i) {
		if (c == EOF )
			return (EOF);
		if (c == '\n')
			return(EOL);
		if (c == ',' || c == ' ' || c == '\t')
			return(0);
		s[i] = c;
		s[i+1] = '\0';                  /* end of string marker */
		c = getc(fp);
	}
	return(0);
}

/*-----------------------------------------------------------------------------
 *   Jack Palevich's MBOX.C
 */

#define DEBUG if(testing)
#define MAXSNG 26
#define VOICES 4
extern	int gfname(), parse();
long	clock_t;

int	music_p[VOICES];	/* -1 for not-playing, otherwise points to score */
char	note, testing, music_s;

#define MBUFLEN 10000
char mbuf[MBUFLEN];	/* input buffer for the music composer file */
int msize;	/* # of characters in existance */

#define MAXNBUF 500
extern char name_buf[]; /* Buffer of file names.			 */

mbox(){
	char *flist[MAXSNG];
	int nomf,i,j,jrem;	 /* number of music files */
	char c;

	testing = 0;

	nomf = gfname("????????MUS", flist, MAXSNG, name_buf, MAXNBUF);
	if (nomf <= 0) {
		CLEAR;
		printf("There are no music files in the default directory.\n");
		printf("Type any character to return.\n");
		getch();
		return;
	}
	j = nomf / 2;
	jrem = nomf % 2;

      for(;;){	/* menu loop */
	putbar(TOP);

	printf("               AMY Music Box v1.2  ");
	putright();
	putbar(MID);

	for(i = 0; i < j; i += 1) {
		printf("\t %c) %-20s %c) %-20s ", i + 'a',  flist [i],
			i + j + jrem + 'a',flist [i + j + jrem]);
		putright();
	}
	if(jrem) {
		printf("\t %c) %-20s", j + 'a', flist[j]);
		putright();
	}
	putbar(MID);

	printf("  Type the letter of the piece you want (a-%c),   ",
		nomf + 'a' - 1);
	putright();
	printf("  or '*' to toggle debugging, or ESC to return.  ");
	putright();

	putbar(BOT);

	for(;;){

		c = getch();	/* unbuffered input */

		if(c == 27){
			printf("Good bye!\n");
			fflush(stdout);
			return;      /* escape out of program */
		}
		if (c == ' '){
			monitor();
			continue;
		}

		if(c == '*'){
			testing = 1-testing;
			printf("Testing %s\n", testing ? "on" : "off");
			fflush(stdout);
			continue;
		}

		c = c - 'a';

		if (c < 0 || c >= nomf)
		{
			printf("Don't type that!\n");
			fflush(stdout);
			continue;
		}
		else break;
	}

	printf("\nPlaying %c) %-20s\n", c + 'a', flist[c]);
	fflush(stdout);

	play(flist [c]);

	if (testing)
	{
		printf("Type space bar to go on.\n");
		fflush(stdout);
		getch();
	}
      }
}

/* Allow the user to send out individual amy codes by hand */

monitor(){
	char c1, c2;
	printf("<");
	fflush(stdout);
	c1 = getch();
	printf("%c", c1);
	fflush(stdout);
	c2 = getch();
	printf("%c", c2);
	fflush(stdout);
	herror = 0;
	send_amy((c2h(c1) << 4) | c2h(c2));
	if (herror != 0)
	{
		printf( "Handshake error %d", herror);
	}
	printf(">");
	fflush(stdout);
}

/* play a piece of music
 */

play(fname)
char *fname;
{
	static char ext[] = ".MUS";
	char file[14];	/* Copy of song name plus .MUS extension.	 */
	int fp, i, j;

	for (i = 0; (file[i] = fname[i]) != '\0'; i++);
	for (j = 0; j < 5; file[i+j] = ext[j++]); /* Append extension.	 */
	fp = open(file, 0x8000);

	if(fp <= 0){
		printf("Couldn't open '%s'.\n", file);
		return;
	}

	DEBUG{
		printf("I'm playing %s\n",fname);
		fflush(stdout);
	}

	msize = read(fp, mbuf, MBUFLEN);
	close(fp);
	if(msize < 0){
		printf("Couldn't read the music file.\n");
		printf("Type any character to go on.\n");
		getch();
		return;
	}
	if(msize == MBUFLEN){
		printf("Music file too large.\n");
		printf("Type any character to go on.\n");
		getch();
		return;
	}
	DEBUG printf("There are %d bytes in the file\n", msize);

	if(parse(mbuf, msize, music_p, testing) != 0){
		fflush(stdout);
		return;
	}

	if(testing){
		printf("Type space to see the score table:\n");
		fflush(stdout);
		if (getch() == ' ')
			print_score(music_p);
	}

	printf("Playing the music now. (any key will stop it)\n");
	fflush(stdout);
	music_s = -1;
	herror = 0;	/* handshake error */
	i = init_clock();	/* returns 0 if no printer port found */
	DEBUG{
		printf("Using the printer at %xh.\n", i);
		fflush(stdout);
	}

	music_s = 0;
	while(music_s == 0){
		if (herror)
		{
			printf("Handshake error %d\n", herror);
			fflush(stdout);
			if (! testing)
			{
				printf("Press any key to go on\n");
				fflush(stdout);
				getch();
			}
			break;
		}
		if (kbhit() != 0)
		{
			getch();
			break;
		}
		if (testing)
		{
			printf("music_p: ");
			for(i = 0; i < VOICES; i += 1)
				printf("%4d ", music_p[i]);
			printf("\n");
			fflush(stdout);
		}
	}
	quit_clock();
	send_amy(0x13); /* silence command */

}
/* === end of mbox === */

