/*
 * parse.c of mbox
 * Parse a music file into a score.
 * John Palevich 8-22-83..9-19-83
 */

#include <a:stdio.h>

#define AV_NU	0
#define AV_GO	1
#define AV_PP	2
#define AV_TR	3
#define AV_VO	4
#define AV_DI	5
#define AV_CO	6

#define MAX_SCORE	10000

char	score[MAX_SCORE];
char	tempo;	/* tempo of piece (1..255) */

int	wait_p, build_p;

#define PHRASE_N 10
#define VOICE_N 4
char *phrases[PHRASE_N], *voices[VOICE_N], *misc, *eob;

int  ttp, tp[VOICE_N];

char xpitch[7] = {
		 0,	 /* C3 */
		 2,	 /* D3 */
		 4,	 /* E3 */
		 5,	 /* F3 */
		 7,	 /* G3 */
		 9,	 /* A3 */
		 11};	 /* B3 */

int	xaccidental[4] = {0, 1, -1, 0};

char	xduration[12] = {
		2, 3,	 /* thirtysecond */
		4, 6,	 /* sixteenth	 */
		8, 12,	 /* eighth	 */
		16, 24,  /* quarter	 */
		32, 48,  /* half	 */
		64, 96}; /* whole	 */

char	xspacing[12] = {
		1, 1,	/* thirtysecond */
		2, 2,	/* sixteenth	*/
		3, 3,	/* eighth	*/
		4, 4,	/* quarter	*/
		5, 5,	/* half 	*/
		6, 6};	/* whole	*/

/* parse a music file into a score */
parse(m, len, mp, testing)
	char *m;
	int len, *mp, testing;
{
	int i, n_p, n_v, v;
	if(testing){
		printf("Parsing music file.\n");
	}

	/* find the phrase and voice and miscellaneous records */

	for(i = 0; i < PHRASE_N; i += 1)phrases[i] = 0;
	for(i = 0; i < VOICE_N; i += 1)voices[i] = 0;

	i = 0;		/* Point to start of file */
	while(i < len){
		if(m[i] == 255){
			if(testing)printf("End at %d\n", i);
			eob = m + i;
			break;
		}
		else if(m[i] != 170){
			printf("Bad end-of-file %d\n", m[i]);
			return -1;
		}
		v = m[i + 1] / 2;

		if(v <= 9){
			if(testing)printf("Phrase %d\n", v);
			i += 2;
			phrases[v] = m + i;
			while(m[i] != 255){
				if(i < len)i += 2;
				else {
					printf("No end of record\n");
					return -1;
				}
			}
			i += 1;
		}
		else if(v <= 13){
			v -= 10;
			if(testing)printf("Arrange Voice %d\n", v);
			i += 2;
			voices[v] = m + i;

			while(m[i] != 255){
				if(i < len)i += 2;
				else {
					printf("No end to Arrange record\n");
					return -1;
				}
			}
			i += 1;
		}
		else if(v == 64){
			if(testing)printf("Miscellaneous Information Record\n");
			i += 2;
			misc = m + i;

			if(i + 5 <= len && m[i + 5] != 255)i += 5;
				else {
					printf(
					 "Misc. record has wrong length.\n");
					return -1;
				}
		}
		else {
			printf("Wierd record: %d.\n", v + v);
			return -1;
		}
	} /* end while */

	len = eob - m;

	/* Report on file format */

	if(testing){
		printf("Misc record: %d\n", misc - m);

		for(i = 0; i < VOICE_N; i += 1){
			if(voices[i] == 0)printf("voice %d undefined\n",i);
			else printf("voice %d: %d\n", i, voices[i] - m);
		}

		for (i = 0; i < PHRASE_N; i += 1)
		{
			if(phrases[i] == 0)printf("phrase %d undefined\n", i);
			else printf("phrase %d: %d\n", i,
				phrases[i] - m);
		}
		fflush(stdout);
		printf("Type SPACE BAR to see raw bytes, anything else to skip\n");
		if (getch() == ' ')
		{
			printf("Type any key to stop this\n");
			for (i = 0; i < len; i += 1)
			{
				if (kbhit() != 0)
				{
					getch();
					break;
				}
				if( i % 10 == 0)printf("[%4d - %4d] ",i, i+9);
				printf("%4d", m[i]);
				if (i % 10 == 9)putchar('\n');
			}
			putchar('\n');
			fflush(stdout);
		}
	}

	tempo = misc[2];
	if(testing) printf("Tempo was %d, -> %d\n", misc[2], tempo);

	/* Now use voice tables to build score */
	return build(testing, mp);
}

/*
 * Build the score for each voice.
 */

build(testing, mp)
	int testing;
	int *mp;
{
	int i, j, k, ph;

	char *vp, vline;	/* the voice we're playing */
	int voice_p[VOICE_N],	/* pointer into voice table */
	    pp, phrase_p[VOICE_N],  /* which phrase this voice is playing */
	    ip, iph_p[VOICE_N],     /* how far into the phrase are we? */
	    volume,		/* volume register for AV_VO */
	    count,		/* counter for AV_CO command */
	    transpose,		/* transpose offset */
	    tbp[VOICE_N],	/* beginning of each voice's notes */
	    tep[VOICE_N];	/* end of ... */

	tbp[0] = ttp = 0;

	for(i = 0; i < VOICE_N; i += 1)
	{	/* don't change indent level */
	vp = voices[i];
	vline = 1;
	count = 1;  /* null count to begin with */
	volume	= 4; /* default volume */
	transpose = 0;

	/* now interpret this voice */

	if (testing) printf("     Parsing voice %d\n", i);

	while(1)
	{
		if (testing)
			printf("[%2d] ", vline);
		j = (vline - 1) << 1;
		vline += 1;	/* Post-increment line number */
		if (vp[j] == 255)
		{
			if(testing)printf("End of Voice\n");
			break;	  /* end of this music piece */
		}
		switch (vp[j])
		{
		case AV_NU:
			if(testing)printf("Null\n");
			break;
		case AV_GO:
			count -= 1;
			if (count > 0)
			{
				vline = vp[j + 1];
				if(testing)printf("(%d)Go to %d\n",
					count, vline);
			}
			else
			{
				if(testing)printf("Don't go to %d\n",
					vp[j + 1]);
			}
			break;
		case AV_PP:
			ph = vp[j + 1];
			if(testing)printf("Play phrase %d\n", ph);
			vphrase(testing, transpose, ph, volume);
			break;
		case AV_TR:
			ph = vp[j + 1];
			if (ph & 128) ph = - (ph & 127);
			transpose += ph;
			if(testing)
				printf("Transpose %d to %d\n",
					ph, transpose);
			break;
		case AV_VO:
			volume = vp[j + 1];
			if(testing)printf("Volume %d\n", volume);
			break;
		case AV_DI:
			if(testing)printf("Display voice %d\n", vp[j + 1]);
			break;
		case AV_CO:
			count = vp[j + 1];
			if(testing)printf("Count %d\n", count);
			break;
		default:
			printf("Error: wierd command byte: %d\n", vp[j]);
			break;
		}

	}	/* elihw */

	/* now compress the ties */

	for(k = j = tbp[i]; j < ttp; k += 4, j += 4)
	{
		score[k] = score[j];
		score[k + 1] = score[j + 1];
		score[k + 2] = score[j + 2];
		score[k + 3] = score[j + 3];

		while (score[k + 3] == 0)
		{	/* this is tied to the next note */

			if (score[k] != score[j+4])
			{	/* this is a slur */
				score[k+2] -= 1;
				score[k+3] += 1;
			}
			else
			{	/* this is a tie -- merge both notes */
				score[k + 2] += score[j + 6];
				score[k + 3] += score[j + 7];
				j += 4;
			}
		}
	}
	ttp = k;

	score[ttp] = score[ttp+1] = score[ttp+2] = score[ttp+3]
		= 255;	 /* end of phrase */

	tep[i] = ttp + 3;
	ttp += 4;

	if(i < 3)tbp[i + 1] = ttp;

	}	/* don't change indent level */

	if(testing){
		for (i = 0; i < VOICE_N; i += 1)
			printf("Voice [%d] %4d bytes\n", i, tep[i] - tbp[i]+1);
	}

	/* construct the music pointer table */
	for(i = 0; i < VOICE_N; i += 1)
	{
		mp[i] = tbp[i];
		if (testing)
			printf("music_p[%d] = %d\n", i, mp[i]);
	}

	return 0;
}

/*
 * parse a phrase
 */

vphrase(testing, tran, p, volume)
	int testing, tran, p, volume;
{
	int i, j, k, mflag;
	char *v;

	if (testing)
		printf("vphrase(%d, %d):\n", tran, p);

	v = phrases[p];

	if (v == 0)
	{
		if (testing) printf(". . . which is undefined.\n" );
		return;
	}
	/*
	 * Here we convert from the ATARI Music Composer Phrase Record
	 *  format to an internal format.  At this point we drop
	 *  measure signatures and convert time & notes.
	 */
	v += 2; /* skip first note, which is junk */
	mflag = 1;	/* start of a measure */
	while (*v != 255)	/* end of phrase */
	{
		if (*v < 87)	/* a note */
		{
			i = v[0];
			j = i >> 2;
			if (volume == 0 || i == 85)    /* a rest */
			{
				score[ttp] = 128;	/* ibm "rest" code */
			}
			else
			{
				score[ttp] =
					( tran
					+ 12 * (j / 7)
					+ xpitch[j % 7]
					+ xaccidental[i & 3]
					- 12);	 /* C3, not C4 */
			}

			score[ttp + 1] = volume * 32; /* 0..7 -> 0..255 */
			if(mflag)
			{	/* accent 1st note in measure */
				mflag = 0;
				score[ttp + 1] += 16;
			}

			i = v[1];
			j = (i & 127) - 1;

			if (i & 128)	/* tie -- no release */
			{
				score[ttp + 2] = xduration[j];
				score[ttp + 3] = 0;	/* tie code */
			}
			else
			{
				score[ttp + 2] = xduration[j]
					- xspacing[j];
				score[ttp + 3] = xspacing[j];
			}

			ttp += 4;

			if (ttp >= MAX_SCORE)
			{
				printf("Too many notes...\n");
				return;
			}
		}
		else if (*v != 127)	/* measure bar */
		{
			if(testing)printf("Junk Byte: %d\n", *v);
		}
		else mflag = 1; /* new measure */
		v += 2;
	}
	if (testing)
		printf("%d notes in phrase %d.\n", v - phrases[p], p);
}

/*
 * print_score
 */

print_score(mp)
	int *mp;
{
	int v, i;
	for (v = 0; v < VOICE_N; v += 1)
	{
		printf("Voice %d\n ", v);
		for(i = mp[v]; i < MAX_SCORE; i += 4)
		{
			printf("[%4d] %3d %3d:%3d %3d\n", i, score[i],
				score[i + 1], score[i + 2], score[i + 3]);
			if (score[i + 2] == 255)
			{
				printf("\n");
				break;
			}
			if(kbhit())
			{
				getch();
				return;
			}
		}
	}
	if(i >= MAX_SCORE)printf("\nLarger than buffer.\n");
	fflush(stdout);
}
