	Page	58,132
	Title	KEYBOARD.ASM	Apple Emulator Keyboard Routines
;******************************************************************************
;
;   Name:	KEYBOARD.ASM	Apple Emulator Keyboard Routines
;
;   Group:	Emulator
;
;   Revision:	1.00
;
;   Date:	January 30, 1988
;
;   Author:	Randy W. Spurlock, modified for Atari by Mike Munoz
;
;******************************************************************************
;
;  Module Functional Description:
;
;		This module contains all the code for the Atari keyboard
;	routines.
;
;******************************************************************************
;
;  Changes:
;
;    DATE     REVISION				DESCRIPTION
;  --------   --------	-------------------------------------------------------
;   1/30/88	1.00	Original
;
;******************************************************************************
	Page
;
;  Public Declarations
;
	Public	Keyboard_Input		; Keyboard input routine
	Public	Clear_Keyboard		; Clear keyboard strobe routine
	Public	Key_Int 		; Keyboard interrupt routine
	Public	Set_LED 		; Set keyboard LED routine
	Public	Flush_Keyboard		; Flush keyboard routine
	Public	Check_Key		; Check for keyboard input routine
	Public	Get_Key 		; Get keyboard scan code routine
	Public	Key_Reset		; Keyboard reset routine
	Public	Key_Status		; Keyboard status byte
	Public	Last_Key		; Last actual keyboard code
	Public	Last_Scan		; Last keyboard scan code
	Public	int_flag
	Public  nmi_flag
	Public	Access_flag
	Public Atari_Reset
;
;  External Declarations
;
	Extrn	System_Access:Near	; System reset routine	 (Atari)
	Extrn	System_Request:Near	; System request routine (Atari)
	Extrn	Exit:Near		; Exit program		(Atari)
	Extrn	System_Flag:Byte	; Atari emulator system flag byte(DATA)
	Extrn	Original_Int_9
	Extrn	exit_flag:Byte
	Extrn	sBOT:Word
	Extrn	Hsync:Word
	Extrn	sTOP:Word
;
;  LOCAL Equates
;
DATA_PORT	Equ	60h		; Keyboard data port address (60h)
COMMAND_PORT	Equ	64h		; Keyboard command port address (64h)
OUTPUT_FULL	Equ	01h		; Output buffer full flag bit
INPUT_EMPTY	Equ	02h		; Input buffer empty flag bit
KEY_CODE	Equ	7Fh		; Keyboard scan code mask value
KEY_MASK	Equ	07h		; Keyboard status mask value
KEY_SHIFT	Equ	03h		; Keyboard shift value
LED_MASK	Equ	0E0h		; Keyboard status LED mask value
LED_SHIFT	Equ	05h		; Keyboard status LED shift value
INT_PORT	Equ	20h		; Interrupt controller port
INT_ACK 	Equ	20h		; Interrupt acknowledge value
KEY_DOWN	Equ	80h		; Key held down flag bit
KEY_STROBE	Equ	80h		; Key has been pressed flag bit
TAB		Equ	09h		; Tab key translated code value
UP_ARROW	Equ	0Bh		; Up arrow translated code value
LEFT_ARROW	Equ	08h		; Left arrow translated code value
RIGHT_ARROW	Equ	15h		; Right arrow translated code value
DOWN_ARROW	Equ	0Ah		; Down arrow translated code value
DEL_CODE	Equ	7Fh		; DEL key translated code value
DISABLE_KEY	Equ	0ADh		; Disable keyboard function code
ENABLE_KEY	Equ	0AEh		; Enable keyboard function code
LED_SET 	Equ	0EDh		; Set keyboard LED's function code
TIME_OUT	Equ	0FFFFh		; 8042 time out counter value
Key_LED 	Equ	0097h		; BIOS keyboard LED status byte (40:97h)
;
;  Define any include files needed
;
	Include 	Macros.inc	; Include the macro definitions
	Include 	Equates.inc	; Include the equate definitions
	Include		Atari.inc
	.286c				; Include 80286 instructions
	Page
;
;  Define the emulator code segment
;
Emulate Segment Word Public 'EMULATE'   ; Emulator code segment
	Assume	cs:Emulate, ds:Nothing, es:Nothing
	Subttl	Keyboard_Input	Keyboard Input Routine
	Page	+
;******************************************************************************
;
;	Keyboard_Input()
;
;		Read the keyboard data byte
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		AL    - Keyboard data
;
;******************************************************************************
		Even			; Force procedure to even address
Keyboard_Input	Proc	Near		; Keyboard input procedure
	mov	al,cs:[Key_Data]	; Read the keyboard data byte buffer
	ret				; Return to the caller
Keyboard_Input	Endp			; End of the Keyboard_Input procedure
	Subttl	Clear_Keyboard	Clear Keyboard Strobe Routine
	Page	+
;******************************************************************************
;
;	Clear_Keyboard()
;
;		Clear the keyboard strobe
;		Clear the any key down flag
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Clear_Keyboard	Proc	Near		; Clear keyboard input procedure
	and	Byte Ptr cs:[Key_Data],Not KEY_STROBE
	and	Byte Ptr cs:[Last_Key],Not KEY_STROBE
	and	Byte Ptr cs:[Any_Key],Not KEY_DOWN
	ret				; Return to the caller
Clear_Keyboard	Endp			; End of the Clear_Keyboard procedure
	Subttl	Key_Int 	Keyboard Interrupt Routine
	Page	+
;******************************************************************************
;
;	Key_Int()
;
;		Save the required registers
;		Get the key from the input port (60h)
;		Translate scan code into keyboard code value
;		If special keyboard scan code
;			Call correct routine to handle scan code
;		Else this is a standard keyboard scan code
;			If this is a make code
;					Set the keyboard strobe bit
;					Save as last keyboard code
;				If keyboard not in input mode
;					Set the any key down flag bit
;					Save the keyboard data value
;				Endif
;			Else this is a break code
;				If keyboard not in input mode
;					Clear the any key down flag bit
;			Endif
;		Endif
;		Save the last scan code value
;		Acknowledge the interrupt controller
;		Restore the required registers
;		Return to the caller (Interrupt return)
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Key_Int 	Proc	Near		; Keyboard interrupt procedure
	Save	ax,bx,cx,di		; Save the required registers
	in	al,DATA_PORT		; Get the keyboard scan code
	mov	ah,al			; Save scan code value in AH
	and	al,KEY_CODE		; Ignore break/make bit in scan code
	mov	bl,al			; Move scan code into BL register
	xor	bh,bh			; Convert scan code to full word
	mov	cl,cs:[Key_Status]	; Get the previous keyboard status
	mov	ch,cl			; Get a copy of the keyboard status
	; see if the current key is effected by caps lock (=0 or 80h)
	and	ch,cs:[bx+Caps_Table]	; Mask with the caps table entry
	rol	ch,1			; Shift caps status to shifted bit
	xor	cl,ch			; Set shift status to correct state
	and	cx,KEY_MASK		; Mask off all but desired bits
	shl	bx,KEY_SHIFT		; Convert scan code to table index
	add	bx,cx			; Compute actual table entry
	mov	bl,cs:[bx+Key_Translate]; Translate the keyboard scan code
	or	bl,bl			; Set the status bits correctly
	js	Special_Key		; Jump if this is a special key
	or	ah,ah			; Check for a make/break code
	js	Key_Break		; Jump if this is a break code
Key_Make:
	mov	di,SKSTAT
	and	Byte Ptr [di],not B2
	mov	cs:[Last_Key],bl	; Save last actual keyboard code
	test	cs:[System_Flag],INPUT	; Check for keyboard in input mode
	jnz	Key_Done		; Jump if NOT in normal keyboard mode
	or	Byte Ptr cs:[Any_Key],KEY_DOWN
	mov	cs:[Key_Data],bl	; Save keyboard data for 65C02
	rol	cl,6			; =0 or 80
	mov	di,KBCODE		; get atari KBCODE location
	or	bl,cl			; add shift and control bits
	mov	ds:[di],bl		; save in pokey memory area
	mov	di,IRQST
	and	Byte Ptr ds:[di],Not B6	; "other key" interrupt
	mov	cs:[int_flag],1
	jmp	Short Key_Done		; Go return to the caller
Key_Break:
	test	cs:[System_Flag],INPUT	; Check for keyboard in input mode
	jnz	Key_Done		; Jump if NOT in normal keyboard mode
	and	Byte Ptr cs:[Any_Key],Not KEY_DOWN
	mov	di,SKSTAT
	or	Byte Ptr [di],B2
	jmp	Short Key_Done		; Go return to the caller
Special_Key:
	shl	bl,1			; Convert key code to table index
	xor	bh,bh			; Convert table index to full word
	call	cs:[bx + Key_Table]	; Call the correct key routine
Key_Done:
	mov	al,INT_ACK		; Get interrupt acknowledge value
	out	INT_PORT,al		; Send acknowledgement to controller
	mov	cs:[Last_Scan],ah	; Save the last scan code value
	Restore ax,bx,cx,di		; Restore the required registers
	iret				; Return to the caller (Interrupt)
Key_Int 	Endp			; End of the Key_Int procedure
	Subttl	Flush_Keyboard	Flush Keyboard Routine
	Page	+
;******************************************************************************
;
;	Flush_Keyboard()
;
;		Clear the keyboard strobe
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Flush_Keyboard	Proc	Near		; Flush keyboard input procedure
	and	Byte Ptr cs:[Last_Key],Not KEY_STROBE
	ret				; Return to the caller
Flush_Keyboard	Endp			; End of the Flush_Keyboard procedure
	Subttl	Check_Key	Check for Keyboard Input Routine
	Page	+
;******************************************************************************
;
;	Check_Key()
;
;		Chcek for keyboard strobe
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		None
;		FL    - Zero flag set if no input available
;
;******************************************************************************
		Even			; Force procedure to even address
Check_Key	Proc	Near		; Check for keyboard input procedure
	test	cs:[Last_Key],KEY_STROBE; Check for a keyboard strobe
	ret				; Return to the caller
Check_Key	Endp			; End of the Check_Key procedure
	Subttl	Get_Key 	Get Keyboard Scan Code Routine
	Page	+
;******************************************************************************
;
;	Get_Key()
;
;		While no keyboard strobe
;			Wait for keyboard strobe
;		EndWhile
;		Clear the keyboard strobe
;		Get the keyboard scan code
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		AL    - Keyboard scan code
;
;******************************************************************************
		Even			; Force procedure to even address
Get_Key 	Proc	Near		; Clear keyboard input procedure
	test	cs:[Last_Key],KEY_STROBE; Check for a keyboard strobe
	jz	Get_Key 		; Jump if no key is available
	and	Byte Ptr cs:[Last_Key],Not KEY_STROBE
	mov	al,cs:[Last_Key]	; Get the last actual key scan code
	ret				; Return to the caller
Get_Key 	Endp			; End of the Get_Key procedure
	Subttl	Illegal_Key	Illegal Scan Code Routine
	Page	+
;******************************************************************************
;
;	Illegal_Key(Scan_Code)
;
;		Get the last scan code value (No scan code change)
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Scan code value
;		AH    - Original scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Illegal_Key	Proc	Near		; Illegal scan code procedure
	mov	ah,cs:[Last_Scan]	; Get last scan code for key save
	ret				; Return to the caller
Illegal_Key	Endp			; End of the Illegal_Key procedure
	Subttl	Control_Key	Control Key Routine
	Page	+
;******************************************************************************
;
;	Control_Key(Scan_Code)
;
;		If this is a make code
;			Set the control bit in the keyboard status
;		Else this is a break code
;			Clear the control bit in the keyboard status
;		Endif
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Control key scan code
;		AH    - Original scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Control_Key	Proc	Near		; Control key procedure
	or	ah,ah			; Check for a make or break code
	js	Control_Break		; Jump if this is a break code
Control_Make:
	or	Byte Ptr cs:[Key_Status],CTRLED
	jmp	Short Control_Done	; Go return to the caller
Control_Break:
	and	Byte Ptr cs:[Key_Status],Not CTRLED
Control_Done:
	ret				; Return to the caller
Control_Key	Endp			; End of the Control_Key procedure
	Subttl	Tab_Key 	Tab Key Routine
	Page	+
;******************************************************************************
;
;	Tab_Key(Scan_Code)
;
;		If NOT in joystick mode
;			If this is a make code
;				Get the tab key code
;				Set the any key down flag bit
;				Set the keyboard strobe bit
;				Save the keyboard data value
;			Endif
;		Else in the joystick mode
;			Call the joystick button 1 routine
;		Endif
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Tab key scan code
;		AH    - Original scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Tab_Key 	Proc	Near		; Tab key procedure
	test	Byte Ptr cs:[Key_Status],JOY_MODE
	jnz	Do_Button		; Jump if this is joystick mode
	or	ah,ah			; Check for a make or break code
	js	Tab_Done		; Jump if this is a break code
	mov	bl,TAB			; Get the tab translated code
	or	Byte Ptr cs:[Any_Key],KEY_DOWN
	or	bl,KEY_STROBE		; Set the keyboard strobe bit
	mov	cs:[Key_Data],bl	; Save the keyboard data value
	jmp	Short Tab_Done		; Go return to the caller
Do_Button:
	mov	di,TRIG0
	or	ah,ah
	js	Tab_Break
	mov	Byte Ptr ds:[di],0	; button down
	jmp	Tab_Done
Tab_Break:
	mov	Byte Ptr ds:[di],1	; button up
Tab_Done:
	ret				; Return to the caller
Tab_Key 	Endp			; End of the Tab_Key procedure
	Subttl	Alt_Key 	Alternate Key Routine
	Page	+
;******************************************************************************
;
;	Alt_Key(Scan_Code)
;
;		If this is a make code
;			Set the alternate bit in the keyboard status
;		Else this is a break code
;			Clear the alternate bit in the keyboard status
;		Endif
;		If in joystick mode
;			Call the joystick button 2 routine
;		Endif
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Control key scan code
;		AH    - Original scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Alt_Key 	Proc	Near		; Alternate key procedure
	or	ah,ah			; Check for a make or break code
	js	Alt_Break		; Jump if this is a break code
Alt_Make:
	or	Byte Ptr cs:[Key_Status],ALTED
	jmp	Short Alt_Test		; Go check for joystick mode
Alt_Break:
	and	Byte Ptr cs:[Key_Status],Not ALTED
Alt_Test:
	test	Byte Ptr cs:[Key_Status],JOY_MODE
	jz	Alt_Done		; Jump if this is NOT joystick mode
	mov	di,TRIG0
	mov	Byte Ptr ds:[di],0	; button held down
Alt_Done:
	ret				; Return to the caller
Alt_Key 	Endp			; End of the Alt_Key procedure
	Subttl	Shift_Key	Shift Key Routine
	Page	+
;******************************************************************************
;
;	Shift_Key(Scan_Code)
;
;		If this is a make code
;			Set the shift bit in the keyboard status
;		Else this is a break code
;			Clear the shift bit in the keyboard status
;		Endif
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Shift key scan code
;		AH    - Original scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Shift_Key	Proc	Near		; Shift key procedure
	or	ah,ah			; Check for a make or break code
	js	Shift_Break		; Jump if this is a break code
Shift_Make:
	or	Byte Ptr cs:[Key_Status],SHIFTED
	jmp	Short Shift_Done	; Go return to the caller
Shift_Break:
	and	Byte Ptr cs:[Key_Status],Not SHIFTED
Shift_Done:
	ret				; Return to the caller
Shift_Key	Endp			; End of the Shift_Key procedure
	Subttl	Caps_Lock	Caps Lock Key Routine
	Page	+
;******************************************************************************
;
;	Caps_Lock(Scan_Code)
;
;		If the last scan code does NOT match (NOT a repeat)
;			If this is a make code
;				Toggle the caps lock status bit
;			Endif
;		Endif
;		Call routine to update the LED status
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Caps lock key scan code
;		AH    - Original scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Caps_Lock	Proc	Near		; Caps lock key procedure
	cmp	ah,cs:[Last_Scan]	; Check for a repeat scan code
	je	Caps_Done		; Jump if this key is repeating
	or	ah,ah			; Check for a make or break code
	js	Caps_Done		; Jump if this is a break code
	xor	Byte Ptr cs:[Key_Status],CAPS_LOCKED
	mov	di,KBCODE
	mov	Byte Ptr ds:[di],03ch	; atari caps key
	mov	di,IRQST
	and	Byte Ptr ds:[di],Not B6	; "other key" interrupt
	mov	cs:[int_flag],1
Caps_Done:
	call	Set_LED 		; Update the keyboard LED status
	ret				; Return to the caller
Caps_Lock	Endp			; End of the Caps_Lock procedure
	Subttl	Home_Key	Home Key Routine
	Page	+
;******************************************************************************
;
;	Home_Key(Scan_Code)
;
;		If in joystick mode
;			Call the joystick up/left routine
;		Endif
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Home key scan code
;		AH    - Original scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Home_Key	Proc	Near		; Home key procedure
	test	Byte Ptr cs:[Key_Status],JOY_MODE
	jz	Home_Done		; Jump if this is NOT joystick mode
	mov	di,PORTA		; get joystick address
	or	ah,ah
	js	Home_Key_Break
	mov	Byte Ptr ds:[di], 0fah		; up/left
	jmp	Home_Done
Home_Key_Break:
	mov	Byte Ptr ds:[di], 0ffh		; center
Home_Done:
	ret				; Return to the caller
Home_Key	Endp			; End of the Home_Key procedure
	Subttl	Up_Key		Up Key Routine
	Page	+
;******************************************************************************
;
;	Up_Key(Scan_Code)
;
;		If NOT in joystick mode
;			If this is a make code
;				Get the up arrow key code
;				Set the any key down flag bit
;				Set the keyboard strobe bit
;				Save the keyboard data value
;			Endif
;		Else in the joystick mode
;			Call the joystick up routine
;		Endif
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Up key scan code
;		AH    - Original scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Up_Key		Proc	Near		; Up key procedure
	test	Byte Ptr cs:[Key_Status],JOY_MODE
	jnz	Do_Up			; Jump if this is joystick mode
	or	ah,ah			; Check for a make or break code
	js	Up_Done 		; Jump if this is a break code
	mov	bl,UP_ARROW		; Get the up arrow translated code
	or	Byte Ptr cs:[Any_Key],KEY_DOWN
	or	bl,KEY_STROBE		; Set the keyboard strobe bit
	mov	cs:[Key_Data],bl	; Save the keyboard data value
	mov	di,KBCODE
	mov	Byte Ptr ds:[di],08eh
	mov	di,IRQST
	and	Byte Ptr ds:[di],Not B6	; "other key" interrupt
	mov	cs:[int_flag],1
	jmp	Short Up_Done		; Go return to the caller
Do_Up:
	mov	di,PORTA		; get joystick address
	or	ah,ah
	js	Up_Break
	mov	Byte Ptr ds:[di], 0feh		; up
	jmp	Up_Done
Up_Break:
	mov	Byte Ptr ds:[di], 0ffh		; center
Up_Done:
	ret				; Return to the caller
Up_Key		Endp			; End of the Up_Key procedure
	Subttl	PgUp_Key	PgUp Key Routine
	Page	+
;******************************************************************************
;
;	PgUp_Key(Scan_Code)
;
;		If in joystick mode
;			Call the joystick up/right routine
;		Endif
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - PgUp key scan code
;		AH    - Original scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
PgUp_Key	Proc	Near		; PgUp key procedure
	test	Byte Ptr cs:[Key_Status],JOY_MODE
	jz	PgUp_Done		; Jump if this is NOT joystick mode
	mov	di,PORTA		; get joystick address
	or	ah,ah
	js	PgUp_Break
	mov	Byte Ptr ds:[di], 0f6h		; up/right
	jmp	PgUp_Done
PgUp_Break:
	mov	Byte Ptr ds:[di], 0ffh		; center
PgUp_Done:
	ret				; Return to the caller
PgUp_Key	Endp			; End of the PgUp_Key procedure
	Subttl	Left_Key	Left Key Routine
	Page	+
;******************************************************************************
;
;	Left_Key(Scan_Code)
;
;		If NOT in joystick mode
;			If this is a make code
;				Get the left arrow key code
;				Set the any key down flag bit
;				Set the keyboard strobe bit
;				Save the keyboard data value
;			Endif
;		Else in the joystick mode
;			Call the joystick left routine
;		Endif
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Left key scan code
;		AH    - Original scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Left_Key	Proc	Near		; Left key procedure
	test	Byte Ptr cs:[Key_Status],JOY_MODE
	jnz	Do_Left 		; Jump if this is joystick mode
	or	ah,ah			; Check for a make or break code
	js	Left_Done		; Jump if this is a break code
	mov	bl,LEFT_ARROW		; Get the left arrow translated code
	or	Byte Ptr cs:[Any_Key],KEY_DOWN
	or	bl,KEY_STROBE		; Set the keyboard strobe bit
	mov	cs:[Key_Data],bl	; Save the keyboard data value
	mov	di,KBCODE
	mov	Byte Ptr ds:[di],086h
	mov	di,IRQST
	and	Byte Ptr ds:[di],Not B6	; "other key" interrupt
	mov	cs:[int_flag],1
	jmp	Short Left_Done 	; Go return to the caller
Do_Left:
	mov	di,PORTA		; get joystick address
	or	ah,ah
	js	Left_Break
	mov	Byte Ptr ds:[di], 0fbh		; left
	jmp	Left_Done
Left_Break:
	mov	Byte Ptr ds:[di], 0ffh		; center
Left_Done:
	ret				; Return to the caller
Left_Key	Endp			; End of the Left_Key procedure
	Subttl	Center_Key	Center Key Routine
	Page	+
;******************************************************************************
;
;	Center_Key(Scan_Code)
;
;		If in joystick mode
;			Call the joystick center routine
;		Endif
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Center key scan code (Pad 5)
;		AH    - Original scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Center_Key	Proc	Near		; Center key procedure
	test	Byte Ptr cs:[Key_Status],JOY_MODE
	jz	Center_Done		; Jump if this is NOT joystick mode
	mov	di,PORTA		; get joystick address
	or	ah,ah
	js	Center_Break
	mov	Byte Ptr ds:[di], 0feh		; jump action
	mov	di,TRIG0
	mov	Byte Ptr ds:[di],0	; trigger on
	jmp	Center_Done
Center_Break:
	mov	Byte Ptr ds:[di], 0ffh		; center
	mov	di,TRIG0
	mov	Byte Ptr ds:[di],1	; trigger off
Center_Done:
	ret				; Return to the caller
Center_Key	Endp			; End of the Center_Key procedure
	Subttl	Right_Key	Right Key Routine
	Page	+
;******************************************************************************
;
;	Right_Key(Scan_Code)
;
;		If NOT in joystick mode
;			If this is a make code
;				Get the right arrow key code
;				Set the any key down flag bit
;				Set the keyboard strobe bit
;				Save the keyboard data value
;			Endif
;		Else in the joystick mode
;			Call the joystick right routine
;		Endif
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Right key scan code
;		AH    - Original scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Right_Key	Proc	Near		; Right key procedure
	test	Byte Ptr cs:[Key_Status],JOY_MODE
	jnz	Do_Right		; Jump if this is joystick mode
	or	ah,ah			; Check for a make or break code
	js	Right_Done		; Jump if this is a break code
	mov	bl,RIGHT_ARROW		; Get the right arrow translated code
	or	Byte Ptr cs:[Any_Key],KEY_DOWN
	or	bl,KEY_STROBE		; Set the keyboard strobe bit
	mov	cs:[Key_Data],bl	; Save the keyboard data value
	mov	di,KBCODE
	mov	Byte Ptr ds:[di],087h
	mov	di,IRQST
	and	Byte Ptr ds:[di],Not B6	; "other key" interrupt
	mov	cs:[int_flag],1
	jmp	Short Right_Done	; Go return to the caller
Do_Right:
	mov	di,PORTA		; get joystick address
	or	ah,ah
	js	Right_Break
	mov	Byte Ptr ds:[di], 0f7h		; right
	jmp	Right_Done
Right_Break:
	mov	Byte Ptr ds:[di], 0ffh		; center
Right_Done:
	ret				; Return to the caller
Right_Key	Endp			; End of the Right_Key procedure
	Subttl	End_Key 	End Key Routine
	Page	+
;******************************************************************************
;
;	End_Key(Scan_Code)
;
;		If in joystick mode
;			Call the joystick down/left routine
;		Endif
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - End key scan code
;		AH    - Original scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
End_Key 	Proc	Near		; End key procedure
	test	Byte Ptr cs:[Key_Status],JOY_MODE
	jz	End_Done		; Jump if this is NOT joystick mode
	mov	di,PORTA		; get joystick address
	or	ah,ah
	js	End_Break
	mov	Byte Ptr ds:[di], 0f9h		; down/left
	jmp	End_Done
End_Break:
	mov	Byte Ptr ds:[di], 0ffh		; center
End_Done:
	ret				; Return to the caller
End_Key 	Endp			; End of the End_Key procedure
	Subttl	Down_Key	Down Key Routine
	Page	+
;******************************************************************************
;
;	Down_Key(Scan_Code)
;
;		If NOT in joystick mode
;			If this is a make code
;				Get the down arrow key code
;				Set the any key down flag bit
;				Set the keyboard strobe bit
;				Save the keyboard data value
;			Endif
;		Else in the joystick mode
;			Call the joystick down routine
;		Endif
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Down key scan code
;		AH    - Original scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Down_Key	Proc	Near		; Down key procedure
	test	Byte Ptr cs:[Key_Status],JOY_MODE
	jnz	Do_Down 		; Jump if this is joystick mode
	or	ah,ah			; Check for a make or break code
	js	Down_Done		; Jump if this is a break code
	mov	bl,DOWN_ARROW		; Get the down arrow translated code
	or	Byte Ptr cs:[Any_Key],KEY_DOWN
	or	bl,KEY_STROBE		; Set the keyboard strobe bit
	mov	cs:[Key_Data],bl	; Save the keyboard data value
	mov	di,KBCODE
	mov	Byte Ptr ds:[di],08fh
	mov	di,IRQST
	and	Byte Ptr ds:[di],Not B6	; "other key" interrupt
	mov	cs:[int_flag],1
	jmp	Short Down_Done 	; Go return to the caller
Do_Down:
	mov	di,PORTA		; get joystick address
	or	ah,ah
	js	Down_Break
	mov	Byte Ptr ds:[di], 0fdh		; down
	jmp	Down_Done
Down_Break:
	mov	Byte Ptr ds:[di], 0ffh		; center
Down_Done:
	ret				; Return to the caller
Down_Key	Endp			; End of the Down_Key procedure
	Subttl	PgDn_Key	PgDn Key Routine
	Page	+
;******************************************************************************
;
;	PgDn_Key(Scan_Code)
;
;		If in joystick mode
;			Call the joystick down/right routine
;		Endif
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - PgDn key scan code
;		AH    - Original scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
PgDn_Key	Proc	Near		; PgDn key procedure
	test	Byte Ptr cs:[Key_Status],JOY_MODE
	jz	PgDn_Done		; Jump if this is NOT joystick mode
	mov	di,PORTA		; get joystick address
	or	ah,ah
	js	PgDn_Break
	mov	Byte Ptr ds:[di], 0f5h		; down/right
	jmp	PgDn_Done
PgDn_Break:
	mov	Byte Ptr ds:[di], 0ffh		; center
PgDn_Done:
	ret				; Return to the caller
PgDn_Key	Endp			; End of the PgDn_Key procedure
	Subttl	Ins_Key 	Ins Key Routine
	Page	+
;******************************************************************************
;
;	Ins_Key(Scan_Code)
;
;	Registers on Entry:
;
;		AL    - Ins key scan code
;		AH    - Original scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Ins_Key 	Proc	Near		; Ins key procedure
	or	ah,ah			; Check for a make/break code
	js	Ins_Key_Break	; Jump if this is a break code
Ins_Key_Make:
	mov	cs:[Key_Data],0b7h	; Save keyboard data for 65C02
	mov	di,KBCODE		; get atari KBCODE location
	mov	Byte Ptr ds:[di],0b7h		; save in pokey memory area
	mov	di,IRQST
	and	Byte Ptr ds:[di],Not B6	; "other key" interrupt
	mov	cs:[int_flag],1
Ins_Key_Break:
	ret				; Return to the caller
Ins_Key 	Endp			; End of the Ins_Key procedure
	Subttl	Del_Key 	Del Key Routine
	Page	+
;******************************************************************************
;
;	Del_Key(Scan_Code)
;
;		If NOT in joystick mode
;			If this is a make code
;				Get the DEL key code
;				Set the any key down flag bit
;				Set the keyboard strobe bit
;				Save the keyboard data value
;			Endif
;		Else in the joystick mode
;			If Ctrl key is down
;				Call joystick X resolution decrease routine
;			Else
;				If Alt key is down
;					Call joystick X center decrease routine
;				Endif for Alt key
;			Endif for Ctrl key
;		Endif for joystick mode
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Del key scan code
;		AH    - Original scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Del_Key 	Proc	Near		; Del key procedure
	or	ah,ah			; Check for a make or break code
	js	Del_Done		; Jump if this is a break code
	mov	bl,DEL_CODE		; Get the DEL key translated code
	or	Byte Ptr cs:[Any_Key],KEY_DOWN
	or	bl,KEY_STROBE		; Set the keyboard strobe bit
	mov	cs:[Key_Data],bl	; Save the keyboard data value
Del_Done:
	ret				; Return to the caller
Del_Key 	Endp			; End of the Del_Key procedure
	Subttl	Pad_Plus	Pad Plus Key Routine
	Page	+
;******************************************************************************
;
;	Pad_Plus(Scan_Code)
;
;	Registers on Entry:
;
;		AL    - Pad plus key scan code
;		AH    - Original scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Pad_Plus	Proc	Near		; Pad plus key procedure
	ret				; Return to the caller
Pad_Plus	Endp			; End of the Pad_Plus procedure
	Subttl	Pad_Minus	Pad Minus Key Routine
	Page	+
;******************************************************************************
;
;	Pad_Minus(Scan_Code)
;
;	Registers on Entry:
;
;		AL    - Pad minus key scan code
;		AH    - Original scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Pad_Minus	Proc	Near		; Pad minus key procedure
	ret				; Return to the caller
Pad_Minus	Endp			; End of the Pad_Minus procedure
	Subttl	Num_Lock	Num Lock Key Routine
	Page	+
;******************************************************************************
;
;	Num_Lock(Scan_Code)
;
;		Save the required registers
;		If the last scan code does NOT match (NOT a repeat)
;			If this is a make code
;				If neither shift/ctrl/alt keys are down
;					Toggle the joystick mode bit
;				Endif
;				If this is joystick mode
;					Call correct joystick routine
;				Endif for joystick mode
;			Endif this is a break code
;		Endif this key is repeating
;		Call routine to update the LED status
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		AL    - Num lock key scan code
;		AH    - Original scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Num_Lock	Proc	Near		; Num lock key procedure
	Save	bx			; Save the required registers
	cmp	ah,cs:[Last_Scan]	; Check for a repeat scan code
	je	Num_Done		; Jump if this key is repeating
	or	ah,ah			; Check for a make or break code
	js	Num_Done		; Jump if this is a break code
	xor	bh,bh			; Setup to get keyboard status
	mov	bl,cs:[Key_Status]	; Get the keyboard status byte
;	and	bl,KEY_MASK		; Save only the desired bits
;	jnz	Joy_Test		; Jump if other status bits present
	xor	Byte Ptr cs:[Key_Status],JOY_MODE
Joy_Test:
	test	Byte Ptr cs:[Key_Status],JOY_MODE
	jz	Num_Done		; Jump if NOT in joystick mode
Num_Done:
	call	Set_LED 		; Update the keyboard LED status
	Restore bx			; Restore the required registers
	ret				; Return to the caller
Num_Lock	Endp			; End of the Num_Lock procedure
	Subttl	Set_LED 	Set keyboard LED routine
	Page	+
;******************************************************************************
;
;	Set_LED()
;
;		Save the required registers
;		Wait for 8042 input buffer empty
;		Disable the keyboard for LED update
;		Send the set LED command to 8042
;		Wait for ACK from the 8042
;		Wait for 8042 input buffer empty
;		Get the keyboard status byte
;		Mask off all but the LED status bits
;		Toggle scroll lock status to correct state
;		Send the new LED status bits to 8042
;		Wait for ACK from the 8042
;		Wait for 8042 input buffer empty
;		Enable the keyboard after LED update
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Set_LED 	Proc	Near		; Set keyboard LED procedure
	Save	ax			; Save the required registers
	call	Wait_Input		; Call routine to wait for buffer empty
	jc	Set_Exit		; Jump if buffer never empty
	mov	al,DISABLE_KEY		; Get disable keyboard function code
	out	COMMAND_PORT,al 	; Disable the keyboard
	call	Wait_Output		; Wait for output buffer full
	jc	Skip_Input		; Jump if nothing waiting
	in	al,DATA_PORT		; Read any key that might be there
Skip_Input:
	call	Wait_Input		; Wait for input buffer to go empty
	jc	Set_Enable		; Jump if buffer never empty
	mov	al,LED_SET		; Get set LED function code
	out	DATA_PORT,al		; Send the set LED function code
	call	Wait_Output		; Wait for output buffer full
	jc	Set_Enable		; Jump if buffer never full
	in	al,DATA_PORT		; Get the ACK byte received
	call	Wait_Input		; Wait for input buffer empty
	jc	Set_Enable		; Jump if buffer never empty
	mov	al,cs:[Key_Status]	; Get the current keyboard status
	and	al,LED_MASK		; Mask off all but the LED indicators
	xor	al,SCROLL_LOCKED	; Toggle scroll lock to correct state
	shr	al,LED_SHIFT		; Shift bits into the correct position
	out	DATA_PORT,al		; Output the new LED status byte
	call	Wait_Output		; Wait for output buffer to fill
	jc	Set_Enable		; Jump if buffer never full
	in	al,DATA_PORT		; Get the ACK byte from the buffer
	call	Wait_Input		; Wait for the input buffer to go empty
Set_Enable:
	mov	al,ENABLE_KEY		; Get enable keyboard function code
	out	COMMAND_PORT,al 	; Enable the keyboard
Set_Exit:
	Restore ax			; Restore the required registers
	ret				; Return to the caller
Set_LED 	Endp			; End of the Set_LED procedure
	Subttl	Wait_Input	Wait for Input Buffer Empty routine
	Page	+
;******************************************************************************
;
;	Wait_Input()
;
;		Save the required registers
;		Get counter limit value
;		While counter > 0 and input buffer full
;			Wait till input buffer is empty
;			Decrement counter value
;			If counter is zero
;				Set carry flag indicating never empty
;			Endif
;		Endwhile
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		AL    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Wait_Input	Proc	Near		; Wait for input buffer empty procedure
	Save	cx			; Save the required registers
	mov	cx,TIME_OUT		; Get the time out counter value
Input_Loop:
	in	al,COMMAND_PORT 	; Read the 8042 status
	test	al,INPUT_EMPTY		; Check for input buffer empty
	jz	Input_Exit		; Jump if input buffer is empty
	loop	Input_Loop		; Loop until time out is expired
	stc				; Set carry flag indicating time out
Input_Exit:
	Restore cx			; Restore the required registers
	ret				; Return to the caller
Wait_Input	Endp			; End of the Wait_Input procedure
	Subttl	Wait_Output	Wait for Output Buffer Full routine
	Page	+
;******************************************************************************
;
;	Wait_Output()
;
;		Save the required registers
;		Get counter limit value
;		While counter > 0 and output buffer empty
;			Wait till output buffer is full
;			Decrement counter value
;			If counter is zero
;				Set carry flag indicating never full
;			Endif
;		Endwhile
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		AL    - Destroyed
;
;******************************************************************************
		Even			; Force procedure to even address
Wait_Output	Proc	Near		; Wait for output buffer full procedure
	Save	cx			; Save the required registers
	mov	cx,TIME_OUT		; Get the time out counter value
Output_Loop:
	in	al,COMMAND_PORT 	; Read the 8042 status
	test	al,OUTPUT_FULL		; Check for output buffer full
	jz	Output_Exit		; Jump if output buffer is full
	loop	Output_Loop		; Loop until time out is expired
	stc				; Set carry flag indicating time out
Output_Exit:
	Restore cx			; Restore the required registers
	ret				; Return to the caller
Wait_Output	Endp			; End of the Wait_Output procedure
	Subttl	Key_Reset	Keyboard Reset routine
	Page	+
;******************************************************************************
;
;	Key_Reset()
;
;		Save the required registers
;		Wait for 8042 input buffer empty
;		Disable the keyboard for LED update
;		Send the set LED command to 8042
;		Wait for ACK from the 8042
;		Wait for 8042 input buffer empty
;		Get the keyboard shift status (BIOS)
;		Mask off all but the LED status bits
;		Toggle scroll lock status to correct state
;		Send the new LED status bits to 8042
;		Wait for ACK from the 8042
;		Wait for 8042 input buffer empty
;		Enable the keyboard after LED update
;		Restore the required registers
;		Return to the caller
;
;	Registers on Entry:
;
;		None
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
		Even			; Force procedure to even address
Key_Reset	Proc	Near		; Keyboard reset procedure
	Save	ax			; Save the required registers
	call	Wait_Input		; Call routine to wait for buffer empty
	jc	Reset_Exit		; Jump if buffer never empty
	mov	al,DISABLE_KEY		; Get disable keyboard function code
	out	COMMAND_PORT,al 	; Disable the keyboard
	call	Wait_Output		; Wait for output buffer full
	jc	Input_Skip		; Jump if nothing waiting
	in	al,DATA_PORT		; Read any key that might be there
Input_Skip:
	call	Wait_Input		; Wait for input buffer to go empty
	jc	Reset_Enable		; Jump if buffer never empty
	mov	al,LED_SET		; Get set LED function code
	out	DATA_PORT,al		; Send the set LED function code
	call	Wait_Output		; Wait for output buffer full
	jc	Reset_Enable		; Jump if buffer never full
	in	al,DATA_PORT		; Get the ACK byte received
	call	Wait_Input		; Wait for input buffer empty
	jc	Reset_Enable		; Jump if buffer never empty
	mov	ah,SHIFT_STATUS 	; Get shift status function code
	int	KEYBOARD		; Get the shift/lock status byte
	shl	al,1			; Shift LED status bits into position
	and	al,LED_MASK		; Mask off all but the LED indicators
	shr	al,LED_SHIFT		; Shift bits into the correct position
	out	DATA_PORT,al		; Output the new LED status byte
	call	Wait_Output		; Wait for output buffer to fill
	jc	Reset_Enable		; Jump if buffer never full
	in	al,DATA_PORT		; Get the ACK byte from the buffer
	call	Wait_Input		; Wait for the input buffer to go empty
Reset_Enable:
	mov	al,ENABLE_KEY		; Get enable keyboard function code
	out	COMMAND_PORT,al 	; Enable the keyboard
Reset_Exit:
	mov	al,0ah
	mov	ah,0ch
	int	DOS			; call get key routine
	Restore ax			; Restore the required registers
	ret				; Return to the caller
Key_Reset	Endp			; End of the Key_Reset procedure

	Even
;******************************************************************************
;
;	Atari_Asterisk
;          Handle Special case characters for atari keyboard
;	   mapping.
;
;	Registers on Entry:
;		ah - original  IBM scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
Atari_Asterisk	Proc	Near
	or	ah,ah			; Check for a make/break code
	js	Asterisk_Key_Break	; Jump if this is a break code
Asterisk_Key_Make:
	mov	cs:[Key_Data],07h	; Save keyboard data for 65C02
	mov	di,KBCODE		; get atari KBCODE location
	mov	Byte Ptr ds:[di],07h		; save in pokey memory area
	mov	di,IRQST
	and	Byte Ptr ds:[di],Not B6	; "other key" interrupt
	mov	cs:[int_flag],1
Asterisk_Key_Break:
	ret				; Return to the caller
Atari_Asterisk	Endp

	Even
;******************************************************************************
;
;	Atari_Plus
;          Handle Special case characters for atari keyboard
;	   mapping.
;
;	Registers on Entry:
;		ah - original  IBM scan code
;		bl - tarnslated atari scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
Atari_Plus	Proc	Near
	or	ah,ah			; Check for a make/break code
	js	Plus_Key_Break	; Jump if this is a break code
Plus_Key_Make:
	mov	cs:[Key_Data],06h	; Save keyboard data for 65C02
	mov	di,KBCODE		; get atari KBCODE location
	mov	Byte Ptr ds:[di],06h		; save in pokey memory area
	mov	di,IRQST
	and	Byte Ptr ds:[di],Not B6	; "other key" interrupt
	mov	cs:[int_flag],1
Plus_Key_Break:
	ret				; Return to the caller
Atari_Plus	Endp

	Even
;******************************************************************************
;
;	Atari_LBracket
;          Handle Special case characters for atari keyboard
;	   mapping.
;
;	Registers on Entry:
;		ah - original  IBM scan code
;		bl - tarnslated atari scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
Atari_LBracket	Proc	Near
	or	ah,ah			; Check for a make/break code
	js	LBracket_Key_Break	; Jump if this is a break code
LBracket_Key_Make:
	mov	cs:[Key_Data],060h	; Save keyboard data for 65C02
	mov	di,KBCODE		; get atari KBCODE location
	mov	Byte Ptr ds:[di],060h		; save in pokey memory area
	mov	di,IRQST
	and	Byte Ptr ds:[di],Not B6	; "other key" interrupt
	mov	cs:[int_flag],1
LBracket_Key_Break:
	ret				; Return to the caller
Atari_LBracket	Endp

	Even
;******************************************************************************
;
;	Atari_RBracket
;          Handle Special case characters for atari keyboard
;	   mapping.
;
;	Registers on Entry:
;		ah - original  IBM scan code
;		bl - tarnslated atari scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
Atari_RBracket	Proc	Near
	or	ah,ah			; Check for a make/break code
	js	RBracket_Key_Break	; Jump if this is a break code
RBracket_Key_Make:
	mov	cs:[Key_Data],062h	; Save keyboard data for 65C02
	mov	di,KBCODE		; get atari KBCODE location
	mov	Byte Ptr ds:[di],062h		; save in pokey memory area
	mov	di,IRQST
	and	Byte Ptr ds:[di],Not B6	; "other key" interrupt
	mov	cs:[int_flag],1
RBracket_Key_Break:
	ret				; Return to the caller
Atari_RBracket	Endp

	Even
;******************************************************************************
;
;	Atari_Quote
;          Handle Special case characters for atari keyboard
;	   mapping.
;
;	Registers on Entry:
;		ah - original  IBM scan code
;		bl - tarnslated atari scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
Atari_Quote	Proc	Near
	or	ah,ah			; Check for a make/break code
	js	Quote_Key_Break	; Jump if this is a break code
Quote_Key_Make:
	mov	cs:[Key_Data],073h	; Save keyboard data for 65C02
	mov	di,KBCODE		; get atari KBCODE location
	mov	Byte Ptr ds:[di],073h		; save in pokey memory area
	mov	di,IRQST
	and	Byte Ptr ds:[di],Not B6	; "other key" interrupt
	mov	cs:[int_flag],1
Quote_Key_Break:
	ret				; Return to the caller
Atari_Quote	Endp

	Even
;******************************************************************************
;
;	Atari_Bar
;          Handle Special case characters for atari keyboard
;	   mapping.
;
;	Registers on Entry:
;		ah - original  IBM scan code
;		bl - tarnslated atari scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
Atari_Bar	Proc	Near
	or	ah,ah			; Check for a make/break code
	js	Bar_Key_Break	; Jump if this is a break code
Bar_Key_Make:
	mov	cs:[Key_Data],04fh	; Save keyboard data for 65C02
	mov	di,KBCODE		; get atari KBCODE location
	mov	Byte Ptr ds:[di],04fh		; save in pokey memory area
	mov	di,IRQST
	and	Byte Ptr ds:[di],Not B6	; "other key" interrupt
	mov	cs:[int_flag],1
Bar_Key_Break:
	ret				; Return to the caller
Atari_Bar	Endp

	Even
;******************************************************************************
;
;	Atari_LAngle
;          Handle Special case characters for atari keyboard
;	   mapping.
;
;	Registers on Entry:
;		ah - original  IBM scan code
;		bl - tarnslated atari scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
Atari_LAngle	Proc	Near
	or	ah,ah			; Check for a make/break code
	js	LAngle_Key_Break	; Jump if this is a break code
LAngle_Key_Make:
	mov	cs:[Key_Data],036h	; Save keyboard data for 65C02
	mov	di,KBCODE		; get atari KBCODE location
	mov	Byte Ptr ds:[di],036h		; save in pokey memory area
	mov	di,IRQST
	and	Byte Ptr ds:[di],Not B6	; "other key" interrupt
	mov	cs:[int_flag],1
LAngle_Key_Break:
	ret				; Return to the caller
Atari_LAngle	Endp
	Dw	Atari_RAngle		; Code 1F - >

	Even
;******************************************************************************
;
;	Atari_RAngle
;          Handle Special case characters for atari keyboard
;	   mapping.
;
;	Registers on Entry:
;		ah - original  IBM scan code
;		bl - tarnslated atari scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
Atari_RAngle	Proc	Near
	or	ah,ah			; Check for a make/break code
	js	RAngle_Key_Break	; Jump if this is a break code
RAngle_Key_Make:
	mov	cs:[Key_Data],037h	; Save keyboard data for 65C02
	mov	di,KBCODE		; get atari KBCODE location
	mov	Byte Ptr ds:[di],037h		; save in pokey memory area
	mov	di,IRQST
	and	Byte Ptr ds:[di],Not B6	; "other key" interrupt
	mov	cs:[int_flag],1
RAngle_Key_Break:
	ret				; Return to the caller
Atari_RAngle	Endp

	Even
;******************************************************************************
;
;	Atari_Reset
;          Handle Special case characters for atari keyboard
;	   mapping.
;
;	Registers on Entry:
;		ah - original  IBM scan code
;		bl - tarnslated atari scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
Atari_Reset	Proc	Near
	mov	di,NMIST
	mov	Byte Ptr ds:[di],B5
	mov	cs:[nmi_flag],1
	ret				; Return to the caller
Atari_Reset	Endp
	Even
;******************************************************************************
;
;	Atari_Reboot
;          Handle Special case characters for atari keyboard
;	   mapping.
;
;	Registers on Entry:
;		ah - original  IBM scan code
;		bl - tarnslated atari scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
Atari_Reboot	Proc	Near
	mov	cs:sTOP,32d
	mov	cs:sBOT,224d
	mov	cs:Hsync,128d
	mov	di,244h
	mov	Byte Ptr [di],1		; set cold start flag
	mov	di,NMIST
	mov	Byte Ptr ds:[di],B5
	mov	cs:[nmi_flag],1
	ret				; Return to the caller
Atari_Reboot	Endp

	Even
;******************************************************************************
;
;	Atari_Option
;          Handle Special case characters for atari keyboard
;	   mapping.
;
;	Registers on Entry:
;		ah - original  IBM scan code
;		bl - tarnslated atari scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
Atari_Option	Proc	Near
	mov	di,CONSOL
	or	ah,ah
	js	Atari_Option_Break
Atari_Option_Make:
	and	Byte Ptr [di],Not B2
	jmp	Atari_Option_Done
Atari_Option_Break:
	or	Byte Ptr [di],B2
Atari_Option_Done:
	ret				; Return to the caller
Atari_Option	Endp

	Even
;******************************************************************************
;
;	Atari_Select
;          Handle Special case characters for atari keyboard
;	   mapping.
;
;	Registers on Entry:
;		ah - original  IBM scan code
;		bl - tarnslated atari scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
Atari_Select	Proc	Near
	mov	di,CONSOL
	or	ah,ah
	js	Atari_Select_Break
Atari_Select_Make:
	and	Byte Ptr ds:[di],Not B1
	jmp	Atari_Select_Done
Atari_Select_Break:
	or	Byte Ptr ds:[di],B1
Atari_Select_Done:
	ret				; Return to the caller
Atari_Select	Endp
	Dw	Atari_Start		; Code 23

	Even
;******************************************************************************
;
;	Atari_Start
;          Handle Special case characters for atari keyboard
;	   mapping.
;
;	Registers on Entry:
;		ah - original  IBM scan code
;		bl - tarnslated atari scan code
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
Atari_Start	Proc	Near
	mov	di,CONSOL
	or	ah,ah
	js	Atari_Start_Break
Atari_Start_Make:
	and	Byte Ptr ds:[di],Not B0
	jmp	Atari_Start_Done
Atari_Start_Break:
	or	Byte Ptr ds:[di],B0
Atari_Start_Done:
	ret				; Return to the caller
Atari_Start	Endp

;******************************************************************************
;
;	Screen_UP
;		Moves screen up one scan line
;
;	Registers on Entry:
;		None
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
Screen_UP	Proc	Near
	or	ah,ah
	js	Screen_UP_Done		; keystroke break
	inc	cs:sBOT
	inc	cs:sTOP
Screen_UP_Done:
	ret
Screen_UP	endp

;******************************************************************************
;
;	Screen_DOWN
;		Moves screen DOWN one scan line
;
;	Registers on Entry:
;		None
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
Screen_DOWN	Proc	Near
	or	ah,ah
	js	Screen_DOWN_Done	; keystroke break
	dec	cs:sBOT
	dec	cs:sTOP
Screen_DOWN_Done:
	ret
Screen_DOWN	endp

;******************************************************************************
;
;	Speed_UP
;		increases the harware update time
;
;	Registers on Entry:
;		None
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
Speed_UP	Proc	Near
	add	cs:Hsync,32
	ret
Speed_UP	endp

;******************************************************************************
;
;	Speed_DOWN
;		decreases the hardware update speed
;
;	Registers on Entry:
;		None
;
;	Registers on Exit:
;
;		None
;
;******************************************************************************
Speed_DOWN	Proc	Near
	sub	cs:Hsync,8
	ret
Speed_DOWN	endp

Go_Exit		Proc	Near
		mov cs:exit_flag,1
		ret
Go_Exit		endp

;******************************************************************************
;
;	Define the keyboard translation table
;
;	None, Shift, Ctrl, Ctrl/Shift, Alt, Alt/Shift, Alt/Ctrl, Alt/Ctrl/Shift
;
;******************************************************************************
Key_Translate	Equ	This Byte	; Keyboard translation table
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 00h - Illegal
	Db	1Ch,1Ch,1Ch,80h,80h,80h,80h,80h ; Scan Code 01h - Escape key
	Db	1Fh,1Fh,1Fh,80h,80h,80h,80h,80h ; Scan Code 02h - Keyboard 1
	Db	1Eh,35h,1Eh,80h,80h,80h,80h,80h ; Scan Code 03h - Keyboard 2
	Db	1Ah,1Ah,1Ah,80h,80h,80h,80h,80h ; Scan Code 04h - Keyboard 3
	Db	18h,18h,18h,80h,80h,80h,80h,80h ; Scan Code 05h - Keyboard 4
	Db	1Dh,1Dh,1Dh,80h,80h,80h,80h,80h ; Scan Code 06h - Keyboard 5
	Db	1Bh,07h,1Bh,80h,80h,80h,80h,80h ; Scan Code 07h - Keyboard 6
	Db	33h,1Dh,33h,80h,80h,80h,80h,80h ; Scan Code 08h - Keyboard 7
	Db	35h,98h,35h,80h,80h,80h,80h,80h ; Scan Code 09h - Keyboard 8
	Db	30h,30h,30h,80h,80h,80h,80h,80h ; Scan Code 0Ah - Keyboard 9
	Db	32h,32h,32h,80h,80h,80h,80h,80h ; Scan Code 0Bh - Keyboard 0
	Db	0Eh,0Eh,0Eh,80h,80h,80h,80h,80h ; Scan Code 0Ch - Keyboard minus
	Db	0Fh,99h,0Fh,80h,80h,80h,80h,80h ; Scan Code 0Dh - Equal sign
	Db	34h,34h,34h,08h,80h,80h,80h,80h ; Scan Code 0Eh - Backspace
	Db	2Ch,2Ch,2Ch,80h,80h,80h,80h,80h ; Scan Code 0Fh - Tab
	Db	2Fh,2Fh,2Fh,80h,80h,80h,80h,80h ; Scan Code 10h - Letter Q
	Db	2Eh,2Eh,2Eh,80h,80h,80h,80h,80h ; Scan Code 11h - Letter W
	Db	2Ah,2Ah,2Ah,80h,80h,80h,80h,80h ; Scan Code 12h - Letter E
	Db	28h,28h,28h,80h,80h,80h,80h,80h ; Scan Code 13h - Letter R
	Db	2Dh,2Dh,2Dh,80h,80h,80h,80h,80h ; Scan Code 14h - Letter T
	Db	2Bh,2Bh,2Bh,80h,80h,80h,80h,80h ; Scan Code 15h - Letter Y
	Db	0Bh,0Bh,0Bh,80h,80h,80h,80h,80h ; Scan Code 16h - Letter U
	Db	0Dh,0Dh,0Dh,80h,80h,80h,80h,80h ; Scan Code 17h - Letter I
	Db	08h,08h,08h,80h,80h,80h,80h,80h ; Scan Code 18h - Letter O
	Db	0Ah,0Ah,0Ah,80h,80h,80h,80h,80h ; Scan Code 19h - Letter P
	Db	9Ah,80h,80h,80h,80h,80h,80h,80h ; Scan Code 1Ah - Left bracket
	Db	9Bh,80h,80h,80h,80h,80h,80h,80h ; Scan Code 1Bh - Right bracket
	Db	0Ch,0Ch,0Ch,80h,80h,80h,80h,80h ; Scan Code 1Ch - Enter
	Db	81h,81h,81h,81h,81h,81h,81h,81h ; Scan Code 1Dh - Ctrl
	Db	3Fh,3Fh,3Fh,80h,80h,80h,80h,80h ; Scan Code 1Eh - Letter A
	Db	3Eh,3Eh,3Eh,80h,80h,80h,80h,80h ; Scan Code 1Fh - Letter S
	Db	3Ah,3Ah,3Ah,80h,80h,80h,80h,80h ; Scan Code 20h - Letter D
	Db	38h,38h,38h,80h,80h,80h,80h,80h ; Scan Code 21h - Letter F
	Db	3Dh,3Dh,3Dh,80h,80h,80h,80h,80h ; Scan Code 22h - Letter G
	Db	39h,39h,39h,80h,80h,80h,80h,80h ; Scan Code 23h - Letter H
	Db	01h,01h,01h,80h,80h,80h,80h,80h ; Scan Code 24h - Letter J
	Db	05h,05h,05h,80h,80h,80h,80h,80h ; Scan Code 25h - Letter K
	Db	00h,00h,00h,80h,80h,80h,80h,80h ; Scan Code 26h - Letter L
	Db	02h,02h,02h,80h,80h,80h,80h,80h ; Scan Code 27h - Semi-colon
	Db	9Ch,1Eh,80h,80h,80h,80h,80h,80h ; Scan Code 28h - Single quote
; ATARI KEY
	Db	27h,27h,27h,80h,80h,80h,80h,80h ; Scan Code 29h - Accent mark
	Db	84h,84h,84h,84h,84h,84h,84h,84h ; Scan Code 2Ah - Left shift
	Db	9Dh,0Fh,80h,80h,80h,80h,80h,80h ; Scan Code 2Bh - Backslash
	Db	17h,17h,17h,80h,80h,80h,80h,80h ; Scan Code 2Ch - Letter Z
	Db	16h,16h,16h,80h,80h,80h,80h,80h ; Scan Code 2Dh - Letter X
	Db	12h,12h,12h,80h,80h,80h,80h,80h ; Scan Code 2Eh - Letter C
	Db	10h,10h,10h,80h,80h,80h,80h,80h ; Scan Code 2Fh - Letter V
	Db	15h,15h,15h,80h,80h,80h,80h,80h ; Scan Code 30h - Letter B
	Db	23h,23h,23h,80h,80h,80h,80h,80h ; Scan Code 31h - Letter N
	Db	25h,25h,25h,80h,80h,80h,80h,80h ; Scan Code 32h - Letter M
	Db	20h,9Eh,20h,80h,80h,80h,80h,80h ; Scan Code 33h - Comma
	Db	22h,9Fh,22h,80h,80h,80h,80h,80h ; Scan Code 34h - Period
	Db	26h,26h,26h,80h,80h,80h,80h,80h ; Scan Code 35h - Slash
	Db	84h,84h,84h,84h,84h,84h,84h,84h ; Scan Code 36h - Right shift
	Db	87h,87h,87h,87h,87h,87h,87h,87h ; Scan Code 37h - Prtsc/SysReq
	Db	83h,83h,83h,83h,83h,83h,83h,83h ; Scan Code 38h - Alt
	Db	21h,21h,21h,21h,80h,80h,80h,80h ; Scan Code 39h - Space bar
	Db	85h,85h,85h,85h,85h,85h,85h,85h ; Scan Code 3Ah - Caps lock
	Db	0A0h,0A0h,0A6h,0A6h,80h,80h,80h,80h ; Scan Code 3Bh - Function 1
	Db	0A1h,0A1h,0A1h,0A1h,80h,80h,80h,80h ; Scan Code 3Ch - Function 2
	Db	0A2h,0A2h,0A2h,0A2h,80h,80h,80h,80h ; Scan Code 3Dh - Function 3
	Db	0A3h,0A3h,0A3h,0A3h,80h,80h,80h,80h ; Scan Code 3Eh - Function 4
	Db	80h,96h,96h,96h,96h,96h,96h,96h ; Scan Code 3Fh - Function 5
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 40h - Function 6
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 41h - Function 7
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 42h - Function 8
	Db	97h,97h,97h,97h,97h,97h,97h,97h ; Scan Code 43h - Function 9
	Db	87h,87h,87h,87h,87h,87h,87h,87h ; Scan Code 44h - Function 10
	Db	88h,88h,88h,88h,88h,88h,88h,88h ; Scan Code 45h - Numlock
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 46h - Scroll lock
	Db	89h,89h,89h,89h,80h,80h,80h,80h ; Scan Code 47h - Home
	Db	8Ah,8Ah,8Ah,8Ah,80h,80h,80h,80h ; Scan Code 48h - Up arrow
	Db	8Bh,8Bh,0a4h,0a4h,0a7h,0a7h,0a7h,0a7h ; Scan Code 49h - PgUp
	Db	95h,95h,95h,95h,95h,95h,80h,80h ; Scan Code 4Ah - Pad minus
	Db	8Ch,8Ch,8Ch,8Ch,80h,80h,80h,80h ; Scan Code 4Bh - Left arrow
	Db	8Dh,8Dh,8Dh,8Dh,80h,80h,80h,80h ; Scan Code 4Ch - Pad 5
	Db	8Eh,8Eh,8Eh,8Eh,80h,80h,80h,80h ; Scan Code 4Dh - Right
	Db	94h,94h,94h,94h,94h,94h,80h,80h ; Scan Code 4Eh - Pad plus
	Db	8Fh,8Fh,8Fh,8Fh,80h,80h,80h,80h ; Scan Code 4Fh - End
	Db	90h,90h,90h,90h,80h,80h,80h,80h ; Scan Code 50h - Down arrow
	Db	91h,91h,0a5h,0a5h,0a8h,0a8h,0a8h,0a8h ; Scan Code 51h - PgDn
	Db	92h,92h,92h,92h,92h,92h,80h,80h ; Scan Code 52h - Ins
	Db	93h,93h,93h,93h,93h,93h,80h,80h ; Scan Code 53h - Del
	Db	87h,87h,87h,87h,87h,87h,87h,87h ; Scan Code 54h - System request
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 55h - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 56h - Illegal
	Db	87h,87h,87h,87h,87h,87h,87h,87h ; Scan Code 57h - F11/SysReq
	Db	86h,86h,86h,86h,86h,86h,86h,86h ; Scan Code 58h - F12/Access
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 59h - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 5Ah - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 5Bh - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 5Ch - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 5Dh - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 5Eh - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 5Fh - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 60h - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 61h - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 62h - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 63h - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 64h - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 65h - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 66h - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 67h - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 68h - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 69h - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 6Ah - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 6Bh - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 6Ch - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 6Dh - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 6Eh - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 6Fh - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 70h - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 71h - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 72h - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 73h - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 74h - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 75h - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 76h - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 77h - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 78h - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 79h - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 7Ah - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 7Bh - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 7Ch - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 7Dh - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 7Eh - Illegal
	Db	80h,80h,80h,80h,80h,80h,80h,80h ; Scan Code 7Fh - Illegal
;******************************************************************************
;
;	Define the keyboard caps lock table
;
;******************************************************************************
Caps_Table	Equ	This Byte	; Keyboard caps lock table
	Db	NOT_AFFECTED		; Scan Code 00h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 01h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 02h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 03h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 04h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 05h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 06h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 07h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 08h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 09h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 0Ah - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 0Bh - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 0Ch - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 0Dh - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 0Eh - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 0Fh - Non-Shifted key
	Db	AFFECTED		; Scan Code 10h - Shifted Key
	Db	AFFECTED		; Scan Code 11h - Shifted Key
	Db	AFFECTED		; Scan Code 12h - Shifted Key
	Db	AFFECTED		; Scan Code 13h - Shifted Key
	Db	AFFECTED		; Scan Code 14h - Shifted Key
	Db	AFFECTED		; Scan Code 15h - Shifted Key
	Db	AFFECTED		; Scan Code 16h - Shifted Key
	Db	AFFECTED		; Scan Code 17h - Shifted Key
	Db	AFFECTED		; Scan Code 18h - Shifted Key
	Db	AFFECTED		; Scan Code 19h - Shifted Key
	Db	NOT_AFFECTED		; Scan Code 1Ah - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 1Bh - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 1Ch - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 1Dh - Non-Shifted key
	Db	AFFECTED		; Scan Code 1Eh - Shifted Key
	Db	AFFECTED		; Scan Code 1Fh - Shifted Key
	Db	AFFECTED		; Scan Code 20h - Shifted Key
	Db	AFFECTED		; Scan Code 21h - Shifted Key
	Db	AFFECTED		; Scan Code 22h - Shifted Key
	Db	AFFECTED		; Scan Code 23h - Shifted Key
	Db	AFFECTED		; Scan Code 24h - Shifted Key
	Db	AFFECTED		; Scan Code 25h - Shifted Key
	Db	AFFECTED		; Scan Code 26h - Shifted Key
	Db	NOT_AFFECTED		; Scan Code 27h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 28h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 29h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 2Ah - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 2Bh - Non-Shifted key
	Db	AFFECTED		; Scan Code 2Ch - Shifted Key
	Db	AFFECTED		; Scan Code 2Dh - Shifted Key
	Db	AFFECTED		; Scan Code 2Eh - Shifted Key
	Db	AFFECTED		; Scan Code 2Fh - Shifted Key
	Db	AFFECTED		; Scan Code 30h - Shifted Key
	Db	AFFECTED		; Scan Code 31h - Shifted Key
	Db	AFFECTED		; Scan Code 32h - Shifted Key
	Db	NOT_AFFECTED		; Scan Code 33h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 34h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 35h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 36h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 37h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 38h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 39h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 3Ah - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 3Bh - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 3Ch - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 3Dh - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 3Eh - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 3Fh - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 40h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 41h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 42h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 43h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 44h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 45h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 46h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 47h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 48h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 49h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 4Ah - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 4Bh - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 4Ch - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 4Dh - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 4Eh - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 4Fh - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 50h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 51h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 52h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 53h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 54h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 55h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 56h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 57h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 58h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 59h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 5Ah - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 5Bh - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 5Ch - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 5Dh - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 5Eh - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 5Fh - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 60h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 61h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 62h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 63h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 64h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 65h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 66h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 67h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 68h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 69h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 6Ah - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 6Bh - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 6Ch - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 6Dh - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 6Eh - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 6Fh - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 70h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 71h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 72h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 73h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 74h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 75h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 76h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 77h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 78h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 79h - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 7Ah - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 7Bh - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 7Ch - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 7Dh - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 7Eh - Non-Shifted key
	Db	NOT_AFFECTED		; Scan Code 7Fh - Non-Shifted key
;******************************************************************************
;
;	Define the special keys table
;
;******************************************************************************
Key_Table	Equ	This Word	; Special key table
	Dw	Illegal_Key		; Code 00 - Illegal scan code
	Dw	Control_Key		; Code 01 - Control key
	Dw	Tab_Key 		; Code 02 - Tab key (Button 1)
	Dw	Alt_Key 		; Code 03 - Alternate key (Button 2)
	Dw	Shift_Key		; Code 04 - Shift key was pressed
	Dw	Caps_Lock		; Code 05 - Caps lock was pressed
	Dw	System_Access		; Code 06 - Ctrl. Scroll lock ()
	Dw	System_Request		; Code 07 - System Request (Break)
	Dw	Num_Lock		; Code 08 - Num lock (Joystick Mode)
	Dw	Home_Key		; Code 09 - Home key (Joystick up/left)
	Dw	Up_Key			; Code 0A - Up arrow (Joystick up)
	Dw	PgUp_Key		; Code 0B - PgUp key (Joystick up/right)
	Dw	Left_Key		; Code 0C - Left arrow (Joystick left)
	Dw	Center_Key		; Code 0D - Center key (Joystick center)
	Dw	Right_Key		; Code 0E - Right arrow (Joystick right)
	Dw	End_Key 		; Code 0F - End key (Joystick down/left)
	Dw	Down_Key		; Code 10 - Down arrow (Joystick down)
	Dw	PgDn_Key		; Code 11 - PgDn key (Joy. down/right)
	Dw	Ins_Key 		; Code 12 - Ins. key (Joy X inc.)
	Dw	Del_Key 		; Code 13 - Del. key (Joy X dec.)
	Dw	Pad_Plus		; Code 14 - Pad plus (Joy Y inc.)
	Dw	Pad_Minus		; Code 15 - Pad minus (Joy Y dec.)
	Dw	Go_Exit			; Code 16 - (ctrl F5)
	Dw	System_Request		; Code 17 - System Request
	Dw	Atari_Asterisk		; Code 18 - *
	Dw	Atari_Plus		; Code 19 - +
	Dw	Atari_LBracket		; Code 1A - [
	Dw	Atari_RBracket		; Code 1B - ]
	Dw	Atari_Quote		; Code 1C - '
	Dw	Atari_Bar		; Code 1D - |
	Dw	Atari_LAngle		; Code 1E - <
	Dw	Atari_RAngle		; Code 1F - >
	Dw	Atari_Reset		; Code 20
	Dw	Atari_Option		; Code 21
	Dw	Atari_Select		; Code 22
	Dw	Atari_Start		; Code 23
	Dw	Screen_UP		; Code 24
	Dw	Screen_DOWN		; Code 25
	Dw	Atari_Reboot		; code 26
	Dw	Speed_UP		; code 27
	Dw	Speed_DOWN		; code 28
;******************************************************************************
;
;	Define the keyboard status byte
;
;	-----------------
;	|7|6|5|4|3|2|1|0|
;	-----------------
;	 | | | | | | | |
;	 | | | | | | | -------> Shift key status (0 = Up, 1 = Down)
;	 | | | | | | ---------> Ctrl key status  (0 = Up, 1 = Down)
;	 | | | | | -----------> Alt key status (0 = Up, 1 = Down)
;	 | | | | -------------> ***** Reserved *****
;	 | | | ---------------> ***** Reserved *****
;	 | | -----------------> Scroll lock key status (1 = Locked)
;	 | -------------------> Num lock key status (1 = Locked)
;	 ---------------------> Caps lock key status (1 = Locked)
;
;******************************************************************************
Key_Status	Db	SCROLL_LOCKED + CAPS_LOCKED
;******************************************************************************
;
;	Define any other keyboard variables
;
;******************************************************************************
Last_Key	Db	?		; Last actual keyboard code
Last_Scan	Db	?		; Last actual keyboard scan code
Key_Data	Db	?		; Keyboard data value
Any_Key 	Db	?		; Any key down flag

;Memory for interrupts
int_flag	DB	0
nmi_flag	DB	0
Access_flag	DB	0
;******************************************************************************
;
;	Define the end of the Emulator Code Segment
;
;******************************************************************************
Emulate Ends
	End				; End of the Keyboard module
