/*
Copyright (c) 1998 Richard Lawrence

This program is free software; you can redistribute it and/or modify it under the terms 
of the GNU General Public License as published by the Free Software Foundation; either 
version 2 of the License, or (at your option) any later version. This program is 
distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details. You should have received a copy of the GNU
General Public License along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <windows.h>
#include "atari.h"
#include "colours.h"
#include "platform.h"
#include "winatari.h"
#include "graphics.h"
#include "resource.h"
#include "winatari.h"
#include "registry.h"

extern void Input_Reset( void );
extern void CleanupInput(void);
extern void WriteRegBinary( HKEY hkInput, char *item, unsigned char *data, int size );
extern void WriteRegDWORD( HKEY hkInput, char *item, DWORD value);
extern void RestoreSIO( void );
extern void SetSIOEsc( void );
extern unsigned	long	ulModesAvail;
extern void	CheckDDrawModes( void );
extern LPBITMAPINFO	lpbmi;
extern ULONG *atari_screen;
extern UBYTE	screenbuff[];
extern ULONG ulAtariState;
extern ULONG ulMiscStates;
extern void Clear_Sound( BOOL bPermanent );
extern void Restart_Sound( void );
extern void SetSafeDisplay( );
extern HWND	MainhWnd;
extern int	enable_sio_patch;

PALETTEENTRY	pe[256];

ULONG	ulAtari_HW_Nexttime;
ULONG	ulDeltaT = 0;
#ifdef USE_PERF_COUNTER
LARGE_INTEGER	lnTimeFreq;
#endif
long nSleepThreshold = 0;

int	Atari_Exit( int );

#ifdef USE_THREAD
DWORD WINAPI StartAtariThread( LPVOID pParam )
{
	char	*argv[2] = {"Atari800Win.exe", ""};
	/*main( 3, argv );*/
	main( 1, NULL );
	return 0;
}
#endif

void Start_Atari_Timer( void )
{
#ifdef USE_PERF_COUNTER
	LARGE_INTEGER	lnCurrentTime;

	QueryPerformanceCounter( &lnCurrentTime );
	ulAtari_HW_Nexttime = lnCurrentTime.LowPart + ulDeltaT;
#else
	ulAtari_HW_Nexttime = timeGetTime() + ulDeltaT;
#endif
}

void Atari_Initialise (int *argc, char *argv[])
{
	int i;

#ifdef USE_PERF_COUNTER
	memset( &lnTimeFreq, 0, sizeof( lnTimeFreq ) );
	memset( &ulAtari_HW_Nexttime, 0, sizeof( ulAtari_HW_Nexttime ) );
#endif

	if (tv_mode == TV_PAL)
	{
#ifdef USE_PERF_COUNTER
		QueryPerformanceFrequency( &lnTimeFreq );
		ulDeltaT = lnTimeFreq.LowPart / 50;
		nSleepThreshold = lnTimeFreq.LowPart / 1000L;
		nSleepThreshold *= (SLEEP_TIME_IN_MS + 1);
#else
		ulDeltaT = 20;
		nSleepThreshold = (SLEEP_TIME_IN_MS + 1);
#endif
	}
	else
	{
#ifdef USE_PERF_COUNTER
		QueryPerformanceFrequency( &lnTimeFreq );
		ulDeltaT = lnTimeFreq.LowPart / 60;
		nSleepThreshold = lnTimeFreq.LowPart / 1000L;
		nSleepThreshold *= (SLEEP_TIME_IN_MS + 1);
#else
		ulDeltaT = 17;
		nSleepThreshold = (SLEEP_TIME_IN_MS + 1);
#endif
	}

	if( !ulModesAvail )
		CheckDDrawModes();

	if( !lpbmi )
		lpbmi = (LPBITMAPINFO)malloc( sizeof( BITMAPINFOHEADER ) + sizeof( RGBQUAD ) * 256 );
	
	if( !lpbmi  )
		Atari_Exit( TRUE );

	lpbmi->bmiHeader.biSize = sizeof( BITMAPINFOHEADER );
	lpbmi->bmiHeader.biWidth = ATARI_VIS_WIDTH;
	lpbmi->bmiHeader.biHeight = -ATARI_HEIGHT;	/* Negative because we are a top-down bitmap */
	lpbmi->bmiHeader.biPlanes = 1;
	lpbmi->bmiHeader.biBitCount = 8;				/* Each byte stands for a color value */
	lpbmi->bmiHeader.biCompression = BI_RGB;		/* Uncompressed format */
	lpbmi->bmiHeader.biSizeImage = 0;
	lpbmi->bmiHeader.biXPelsPerMeter = 0;
	lpbmi->bmiHeader.biYPelsPerMeter = 0;
	lpbmi->bmiHeader.biClrUsed = 0;
	lpbmi->bmiHeader.biClrImportant = 0;

	for (i=0;i<256;i++)
    {
		int	rgb = colortable[i];
		
		lpbmi->bmiColors[i].rgbRed = pe[i].peRed = (rgb & 0x00ff0000) >> 16;
		lpbmi->bmiColors[i].rgbGreen = pe[i].peGreen = (rgb & 0x0000ff00) >> 8;
		lpbmi->bmiColors[i].rgbBlue = pe[i].peBlue = rgb & 0x000000ff;
		lpbmi->bmiColors[i].rgbReserved = pe[i].peFlags = 0;
    }

	Input_Reset();
}

void Toggle_FullSpeed( void )
{
	if( ulAtariState & ATARI_RUNNING )
	{
		if( !(ulAtariState & ATARI_PAUSED ))
		{
			ulAtariState |= ATARI_PAUSED;
			Clear_Sound( FALSE );
			SetWindowText( MainhWnd, "Atari800Win: Stopped (F9 to restart)" );
			if( MainhWnd )
				SetTimer( MainhWnd, TIMER_READ_JOYSTICK, 100, NULL); 
 		}
		else
		{
			ulAtariState &= ~ATARI_PAUSED;
			Restart_Sound();
			SetWindowText( MainhWnd, "Atari800Win: Running" );
		}
	}
}

void Toggle_Pause( void )
{
	if( ulMiscStates & ATARI_FULL_SPEED )
	{
		ulMiscStates &= ~ATARI_FULL_SPEED;
		SetWindowText( MainhWnd, "Atari800Win: Running" );
		Restart_Sound();
	}
	else
	{
		ulMiscStates |= ATARI_FULL_SPEED;
		SetWindowText( MainhWnd, "Atari800Win: Running full speed" );
		Clear_Sound( FALSE );
	}
}

void Toggle_SIO_Patch( void )
{
	if( enable_sio_patch )
	{
		enable_sio_patch = 0;
		RestoreSIO();
	}
	else
	{
		enable_sio_patch = 1;
		SetSIOEsc();
	}
}

int Atari_Exit (int panic)
{
	atari_screen = (ULONG *)screenbuff;

	/*Reset everything DirectDraw*/
	SetSafeDisplay( );

	ulAtariState = ATARI_UNINITIALIZED | ATARI_PAUSED | ATARI_CLOSING;

	/*Reset everything DirectSound*/
	Clear_Sound( TRUE );

	/*Reset everything DirectInput*/
	CleanupInput( );

	Remove_ROM( );

	if( lpbmi )
		free( lpbmi );
	lpbmi = NULL;

	if( panic )
	{
		char	message[LOADSTRING_STRING_SIZE];
		int		result;

		LoadString( NULL, IDS_WARNING_ATARI_PANIC, message, LOADSTRING_STRING_SIZE );
		result = MessageBox( NULL, message, "Atari800Win", MB_ICONSTOP|MB_YESNO );
		if( result == IDYES )
			WriteRegBinary( NULL, REG_DRIVE1, "Empty", 5 );

		exit( 1 );
	}
	return 0;
}

