/*
Copyright (c) 1998 Richard Lawrence

This program is free software; you can redistribute it and/or modify it under the terms 
of the GNU General Public License as published by the Free Software Foundation; either 
version 2 of the License, or (at your option) any later version. This program is 
distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details. You should have received a copy of the GNU
General Public License along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <stdio.h>
#include <fcntl.h>
#include <windows.h>
#include <crtdbg.h>
#include <ddraw.h>

#include "atari.h"
#include "monitor.h"
#include "platform.h"
#include "winatari.h"
#include "registry.h"
#include "graphics.h"
#include "resource.h"
#include "diskled.h"

/* This doesn't buy us anything. It's actually slower than blitting directly from system
   memory to the primary buffer (which is how we have to render because the Atari screen
   is drawn pixel by pixel). */
/* #define USE_FLIP_BUFFER 1 */

TCHAR	gcErrorString[ LOADSTRING_STRING_SIZE ];

extern void Restart_Sound( void );
extern void WriteRegDWORD( HKEY hkInput, char *item, DWORD value);
extern PALETTEENTRY	pe[];
extern ULONG ulAtariState;
/* extern int Sound_Initialise( void ); */
extern void Clear_Sound( BOOL bPermanent ); 
extern ULONG *atari_screen;
extern TVmode tv_mode;

int	nStatusSize = 18;
static LPDIRECTDRAW				lpDirectDrawObject = NULL;
/* Don't use SURFACE3, it doesn't work with NT 4.0 */
static LPDIRECTDRAWSURFACE		lpPrimary = NULL;
static LPDIRECTDRAWSURFACE		lpMemory = NULL;
#ifdef USE_FLIP_BUFFER
static LPDIRECTDRAWSURFACE		lpFrontBuff = NULL;
static LPDIRECTDRAWSURFACE		lpBackBuff = NULL;
#endif

static BOOL GetDDErrorString (HRESULT hResult, LPTSTR lpszErrorBuff, DWORD cchError);
static RECT	rcDest, rcSource, rcSrcClip;
LONG	origWndStyle = 0;
unsigned long	ulScreenMode = DISPLAY_MODE_DEFAULT;
int		nStartX = 0, nStartY = 0;

LPBITMAPINFO	lpbmi = NULL;		/* bitmap info for our screen bitmap */

UBYTE	screenbuff[ ATARI_WIDTH * (ATARI_HEIGHT+16) ];
static UBYTE	screentemp[ ATARI_STRETCH_VIS_SCREEN_SIZE ];

HWND	MainhWnd = NULL, hWnd = NULL;
HDC		h_screenDC = NULL;

unsigned	long	ulModesAvail = 0;

static unsigned long ulLastMode = 0;
static void Screen_Windowed_DDraw_384( UBYTE *screen );
static void Screen_Windowed_DDraw_768( UBYTE *screen );
static void Screen_DDraw_Full_Blt( UBYTE *screen );
#ifdef USE_FLIP_BUFFER
static void Screen_DDraw_Full_FlipBuff( UBYTE *screen );
#endif
static void Screen_NoDDraw_768( UBYTE *screen );
static void Screen_NoDDraw( UBYTE *screen );


void Atari_Set_Disk_LED( int unit, int state )
{
	ulDiskUnit = unit;
	ulDiskState = state;
}

void Draw_DiskLED( UBYTE *screen )
{
	if( ulDiskUnit <= LED_NO_UNIT )
	{
		int	i = DISKLED_FONT_HEIGHT;
		unsigned char *src = &DiskLED[ ulDiskUnit * DISKLED_FONT_CHARSIZE ]; 
		unsigned char *dest = &screen[ (rcSource.bottom - DISKLED_FONT_HEIGHT) * ATARI_WIDTH + rcSource.right - DISKLED_FONT_WIDTH ];
		if( ulDiskState == LED_WRITE )
			src += 9 * DISKLED_FONT_CHARSIZE;
		while( i-- )
		{
			*dest++ = *src++;
			*dest++ = *src++;
			*dest++ = *src++;
			*dest++ = *src++;
			*dest++ = *src++;
			dest+= ATARI_WIDTH - DISKLED_FONT_WIDTH;
		}
		ulDiskUnit = LED_NO_UNIT;
		ulDiskState = 0;
	}
}

static HRESULT WINAPI EnumModesCallback(LPDDSURFACEDESC lpDDSurfaceDesc, LPVOID lpContext)
{
	if( lpDDSurfaceDesc->ddpfPixelFormat.dwRGBBitCount == 8 )
	{
		if( lpDDSurfaceDesc->dwHeight == 600 && lpDDSurfaceDesc->dwWidth == 800 )
			ulModesAvail |= MODE_800_600;
		if( lpDDSurfaceDesc->dwHeight == 480 && lpDDSurfaceDesc->dwWidth == 640 )
			ulModesAvail |= MODE_640_480;
		if( lpDDSurfaceDesc->dwHeight == 400 && lpDDSurfaceDesc->dwWidth == 640 )
			ulModesAvail |= MODE_640_400;
		if( lpDDSurfaceDesc->dwHeight == 240 && lpDDSurfaceDesc->dwWidth == 320 )
			ulModesAvail |= MODE_320_240;
		if( lpDDSurfaceDesc->dwHeight == 200 && lpDDSurfaceDesc->dwWidth == 320 )
			ulModesAvail |= MODE_320_200;
	}
	return DDENUMRET_OK;
}

void	CheckDDrawModes( void )
{
	if( !lpDirectDrawObject )
	{
		/* This is for IDirectDraw2 objects (don't work with some cards, so much for compatibility */
/*		LPDIRECTDRAW lpDDT;
		if( SUCCEEDED( DirectDrawCreate(NULL,&lpDDT,0)))
		{
			if( SUCCEEDED( IDirectDraw_QueryInterface( lpDDT, &IID_IDirectDraw2, (void**)&lpDirectDrawObject)) )
			{
				IDirectDraw_Release( lpDDT );
				IDirectDraw_EnumDisplayModes( lpDirectDrawObject, 0, NULL, NULL, EnumModesCallback);
				IDirectDraw_Release( lpDirectDrawObject );
			}
			else
				IDirectDraw_Release( lpDDT );
			lpDirectDrawObject = NULL;
		}*/
		if( SUCCEEDED( DirectDrawCreate( NULL, &lpDirectDrawObject, 0 ) ) )
		{
			IDirectDraw_EnumDisplayModes( lpDirectDrawObject, 0, NULL, NULL, EnumModesCallback );
			IDirectDraw_Release( lpDirectDrawObject );
		}
		lpDirectDrawObject = NULL;
		return;
	}
	IDirectDraw_EnumDisplayModes( lpDirectDrawObject, 0, NULL, NULL, EnumModesCallback);
}

HRESULT ReleaseAllSurfaces( void )
{
	HRESULT	hResult, hFailure;
	hFailure = hResult = DD_OK;

	if( lpMemory )
	{
		hResult = IDirectDrawSurface_Release( lpMemory );
		if( FAILED( hResult ) )
			hFailure = hResult;
	}
	lpMemory = NULL;

#ifdef USE_FLIP_BUFFER
	if( lpBackBuff )
	{
		hResult = IDirectDrawSurface_Release( lpBackBuff );
		if( FAILED( hResult ) )
			hFailure = hResult;
	}

	if( lpFrontBuff )
	{
		hResult = IDirectDrawSurface_Release( lpFrontBuff );
		if( FAILED( hResult ) )
			hFailure = hResult;
	}
	lpFrontBuff = lpBackBuff = NULL;
#endif

	if( lpPrimary )
	{
		hResult = IDirectDrawSurface_Release( lpPrimary );
		if( FAILED( hResult ) )
			hFailure = hResult;
	}
	lpPrimary = NULL;
	return hFailure;
}

void SetSafeDisplay( )
{
	HRESULT	hResult;

	atari_screen = (ULONG *)screenbuff;

	if( ulScreenMode & WINDOWED_768_480 )
		Atari_DisplayScreen = Screen_NoDDraw_768;
	else
		Atari_DisplayScreen = Screen_NoDDraw;

	hResult = ReleaseAllSurfaces();

	if( MainhWnd && origWndStyle )
		SetWindowLong( MainhWnd, GWL_STYLE, origWndStyle );

	if( lpDirectDrawObject )
	{
		IDirectDraw_RestoreDisplayMode( lpDirectDrawObject );
		IDirectDraw_Release( lpDirectDrawObject );
	}
	lpDirectDrawObject = NULL; 

	if( FAILED( hResult ) )
	{
		LoadString( NULL, IDS_DDRAW_ERROR_RELEASE_ALL, gcErrorString, LOADSTRING_STRING_SIZE );
		MessageBox( NULL, gcErrorString, "Atari800Win", MB_OK );
	}

	if( !(ulAtariState & ATARI_CLOSING) )
	{
		SetWindowPos( MainhWnd, HWND_NOTOPMOST, nStartX, nStartY, 
			ATARI_VIS_WIDTH + GetSystemMetrics( SM_CXDLGFRAME )*2 + GetSystemMetrics( SM_CXEDGE )*2,
			ATARI_HEIGHT + GetSystemMetrics( SM_CYMENU)+ nStatusSize + GetSystemMetrics( SM_CYDLGFRAME )*2 +	
			GetSystemMetrics( SM_CYCAPTION ) + GetSystemMetrics( SM_CYEDGE ) * 2, SWP_FRAMECHANGED | SWP_SHOWWINDOW );
	}
	return;
}

static void ShowDDrawError( UINT nUID, HRESULT hResult, BOOL quit )
{
	TCHAR	error[256];
	TCHAR	fullstring[LOADSTRING_STRING_SIZE];
	TCHAR	action[LOADSTRING_STRING_SIZE];
	int		result;

	SetSafeDisplay();

	Clear_Sound( FALSE );

	ulScreenMode |= DDRAW_640_480;
	ulScreenMode |= DDRAW_NONE;
	WriteRegDWORD( NULL, REG_DDRAW_MODE, ulScreenMode);

	GetDDErrorString( hResult, error, 256 );
	LoadString( NULL, IDS_DDRAW_ERROR_PROMPT, gcErrorString, LOADSTRING_STRING_SIZE );
	LoadString( NULL, nUID, action, LOADSTRING_STRING_SIZE );
	wsprintf( fullstring, gcErrorString, action, error );
	LoadString( NULL, IDS_DDRAW_ERROR_HDR, gcErrorString, LOADSTRING_STRING_SIZE );
	result = MessageBox( hWnd, fullstring, gcErrorString, MB_YESNO );

	if( result == IDYES )
	{
		Clear_Sound( TRUE );
		ulAtariState = ATARI_UNINITIALIZED | ATARI_PAUSED;
		if( quit )
			PostMessage( MainhWnd, WM_CLOSE, 0, 0L );
	}
	Restart_Sound( );
	return;
}

void ComputeClipArea( void )
{
	RECT	rcWind, rcDesktop;
	int	nOffset = GetSystemMetrics( SM_CXEDGE );
	int nDivisor = 1;

	/* The following section covers clip behavoir for DDraw windowed modes (important, because the primary
	   surface is the desktop). These are computed once here and then every time a WM_MOVE is received in
	   Atari800WinView. I never had much luck with ClipObjects... */
	SystemParametersInfo( SPI_GETWORKAREA, 0, (LPVOID)&rcDesktop, 0 );
	GetWindowRect( hWnd, &rcWind );
	rcWind.left += nOffset;
	rcWind.top += nOffset;

	if( ulScreenMode & WINDOWED_768_480 )
		nDivisor = 2;

	memcpy( &rcSrcClip, &rcSource, sizeof( RECT ) );

	rcDest.left = rcWind.left - rcDesktop.left;
	if( rcDest.left < 0 )
	{
		rcSrcClip.left = (rcDesktop.left - rcDest.left) / nDivisor;
		rcDest.left = rcDesktop.left;
	}

	rcDest.right = rcDest.left + ATARI_VIS_WIDTH * nDivisor;
	if( rcDest.right > rcDesktop.right )
	{
		rcSrcClip.right -= (rcDest.right - rcDesktop.right) / nDivisor;
		rcDest.right = rcDesktop.right;
	}

	rcDest.top = rcWind.top - rcDesktop.top;
	if( rcDest.top < 0 )
	{
		rcSrcClip.top = (rcDesktop.top - rcDest.top) / nDivisor;
		rcDest.top = rcDesktop.top;
	}
		
	rcDest.bottom = rcDest.top + ATARI_HEIGHT * nDivisor;
	if( rcDest.bottom > rcDesktop.bottom )
	{
		rcSrcClip.bottom -= (rcDest.bottom - rcDesktop.bottom) / nDivisor;
		rcDest.bottom = rcDesktop.bottom ;
	}
	/* Negative because we are a top-down bitmap. Note that it is CORRECT to only adjust height
	   of the src bitmap. This is to prevent windows from blitting the wrong vertical portion 
	   (from the bottom up) if the bmiHeader.biHeight doesn't match the source rectangle. However
	   you always want width the FULL width, otherwise a blit will assume the memory area described
	   has shorter rows and will unevenly blit. Confused? good, we're not finished. In DDRAW
	   windowed modes the SRC rectangle will actually always be 1X, because we stretch it to memory
	   first then just directly BLT. In GDI mode it will already be stretched and then we'll 
	   StretchBlt for the color conversion. Weee. */
	if( lpbmi )
	{
		if( ulScreenMode & DDRAW_WIND )
			lpbmi->bmiHeader.biHeight = -(rcSrcClip.bottom - rcSrcClip.top);
		else
			lpbmi->bmiHeader.biHeight = -(rcSrcClip.bottom - rcSrcClip.top) * nDivisor;
	}

}

static HRESULT Init_Windowed_Mode( void )
{
	HRESULT	hResult;

	rcSource.left = (ATARI_WIDTH - ATARI_VIS_WIDTH) / 2;
    rcSource.top = 0;
	rcSource.right = ATARI_VIS_WIDTH + rcSource.left;
	rcSource.bottom = ATARI_HEIGHT;

	if( MainhWnd && origWndStyle )
		SetWindowLong( MainhWnd, GWL_STYLE, origWndStyle );

	ReleaseAllSurfaces();

	if( lpDirectDrawObject )
		IDirectDraw_RestoreDisplayMode( lpDirectDrawObject );

	if( ulScreenMode & DDRAW_NONE )
	{
		if( lpDirectDrawObject )
		{
			hResult = IDirectDraw_Release( lpDirectDrawObject );
			if( FAILED( hResult ) )
			{
				ShowDDrawError( IDS_DDERR_RELEASE_OBJ, hResult, FALSE );
				return hResult;
			}
		}
		lpDirectDrawObject = NULL;
		atari_screen = (ULONG *)screenbuff;

		if( ulScreenMode & WINDOWED_768_480 )
		{
			lpbmi->bmiHeader.biWidth = ATARI_STRETCH_VIS_WIDTH;
			lpbmi->bmiHeader.biHeight = -ATARI_STRETCH_HEIGHT;	/* Negative because we are a top-down bitmap */
		}
	}

	/* If we are recovering from a DirectDraw mode, the window frame will be whatever
	   size that mode was instead of what we want it to be, so set window size here */
	if( MainhWnd )
	{
		RECT	rcPos;
		int		height, width;

		GetWindowRect( MainhWnd, &rcPos );
		if( ((rcPos.left != 0) && (rcPos.left!=nStartX)) || ((rcPos.top!=0)&&(rcPos.top!=nStartY)))
		{
			nStartX = rcPos.left;
			nStartY = rcPos.top;
		}
		if( ulScreenMode & WINDOWED_384_240 )
		{
			height = ATARI_HEIGHT;
			width = ATARI_VIS_WIDTH;
		}
		else
		{
			height = ATARI_STRETCH_HEIGHT;
			width = ATARI_STRETCH_VIS_WIDTH;
		}
		SetWindowPos( MainhWnd, HWND_NOTOPMOST, nStartX, nStartY, 
			width + GetSystemMetrics( SM_CXDLGFRAME )*2 + GetSystemMetrics( SM_CXEDGE )*2,
			height + GetSystemMetrics( SM_CYMENU) + nStatusSize + GetSystemMetrics( SM_CYDLGFRAME )*2 +	
			GetSystemMetrics( SM_CYCAPTION ) + GetSystemMetrics( SM_CYEDGE ) * 2, SWP_SHOWWINDOW );
	}
	ulAtariState |= ATARI_HW_READY | ATARI_WINDOWS_READY;

	ComputeClipArea();

	return DD_OK;
}

int SetupPrimarySurface( void )
{
/*	LPDIRECTDRAWSURFACE lpDDSTemp;*/
	DDSURFACEDESC	ddsd;
	HRESULT hResult;

	ZeroMemory( &ddsd, sizeof( ddsd ) );
	ddsd.dwSize         = sizeof(ddsd);
	ddsd.dwFlags        = DDSD_CAPS;
	ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;

	_ASSERT( !lpPrimary );

	if( lpPrimary )
		ReleaseAllSurfaces();

	hResult = IDirectDraw_CreateSurface( lpDirectDrawObject, &ddsd, &lpPrimary, NULL);
	if( FAILED( hResult ) )
	{
		IDirectDrawSurface_Release( lpPrimary );
		lpPrimary = NULL;
		ShowDDrawError( IDS_DDERR_ALLOC_PRIMARY, hResult, FALSE );
		return 0;
	}

/*	hResult = IDirectDrawSurface_QueryInterface( lpDDSTemp, &IID_IDirectDrawSurface2, (void**)&lpPrimary);
	IDirectDrawSurface_Release( lpDDSTemp );
	if( FAILED( hResult ) )
	{
		lpPrimary = NULL;
		ShowDDrawError( IDS_DDERR_ALLOC_PRIMARY, hResult, FALSE );
		return 0;
	}*/

	if( (ulScreenMode & DDRAW_FULL) && (ulScreenMode & DDRAW_800_600) )
	{
		/* This is just SO much easier than a DDBLTFX operation.... */
		HDC	hdc;
		IDirectDrawSurface_GetDC( lpPrimary, &hdc );
		if( hdc )
		{
			PatBlt( hdc, 0, GetSystemMetrics( SM_CYMENU ), 800, 600-GetSystemMetrics( SM_CYMENU ), BLACKNESS );
			IDirectDrawSurface_ReleaseDC( lpPrimary, hdc );
		}
	}
	return 1;
}

int SetupMemorySurface( void )
{
/*	LPDIRECTDRAWSURFACE lpDDSTemp;	 */
	DDSURFACEDESC	ddsd;
	HRESULT hResult;

	_ASSERT( !lpMemory );
	if( ulScreenMode & DDRAW_FULL )
	{
		ZeroMemory( &ddsd, sizeof( ddsd ) );
		ddsd.dwSize = sizeof( ddsd );
		ddsd.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH | DDSD_PIXELFORMAT;
		ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_SYSTEMMEMORY;
		ddsd.dwHeight = ATARI_HEIGHT + 16;
		ddsd.dwWidth = ATARI_WIDTH;
		ddsd.ddpfPixelFormat.dwSize = sizeof( DDPIXELFORMAT );
		ddsd.ddpfPixelFormat.dwFlags = DDPF_RGB | DDPF_PALETTEINDEXED8;
		ddsd.ddpfPixelFormat.dwRGBBitCount = 8;
	}
	else
	{
		ZeroMemory( &ddsd, sizeof( ddsd ) );
		ddsd.dwSize = sizeof( ddsd );
		ddsd.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;
		ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_SYSTEMMEMORY;
		ddsd.dwHeight = ATARI_HEIGHT + 16;
		ddsd.dwWidth = ATARI_WIDTH;
	}

	hResult = IDirectDraw_CreateSurface( lpDirectDrawObject, &ddsd, &lpMemory, NULL);
	if( FAILED( hResult ) )
	{
		lpMemory = NULL;
		ShowDDrawError( IDS_DDERR_ALLOC_SYSMEM_SURFACE, hResult, FALSE );
		return 0;
	}

/*	hResult = IDirectDrawSurface_QueryInterface( lpDDSTemp, &IID_IDirectDrawSurface2, (void**)&lpMemory);
	IDirectDrawSurface_Release( lpDDSTemp );
	if( FAILED( hResult ) )
	{
		lpMemory = NULL;
		ShowDDrawError( IDS_DDERR_ALLOC_SYSMEM_SURFACE, hResult, FALSE );
		return 0;
	}*/

	ZeroMemory( &ddsd, sizeof( ddsd ) );
	ddsd.dwSize = sizeof( ddsd );
	IDirectDrawSurface_Lock( lpMemory, NULL, &ddsd, 0, NULL );
	atari_screen = ddsd.lpSurface;
	ZeroMemory( ddsd.lpSurface, ATARI_WIDTH * ATARI_HEIGHT );
	IDirectDrawSurface_Unlock( lpMemory, &ddsd );

	return 1;
}

#ifdef USE_FLIP_BUFFER
int SetupFlipBuffer( void )
{
/*	LPDIRECTDRAWSURFACE lpDDSTemp;	*/
	DDSURFACEDESC	ddsd;
	HRESULT hResult;

	ZeroMemory( &ddsd, sizeof( ddsd ) );
	ddsd.dwSize = sizeof( ddsd );
	ddsd.dwFlags = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
	ddsd.dwBackBufferCount = 2;
	ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE | DDSCAPS_FLIP | DDSCAPS_COMPLEX | DDSCAPS_VIDEOMEMORY;

	hResult = IDirectDraw_CreateSurface( lpDirectDrawObject, &ddsd, &lpFrontBuff, NULL);
	if( FAILED( hResult ) )
	{
		IDirectDrawSurface_Release( lpFrontBuff );
		lpFrontBuff = NULL;
		ShowDDrawError( IDS_DDERR_ALLOC_PRIMARY, hResult, FALSE );
		return 0;
	}
/*	hResult = IDirectDrawSurface_QueryInterface( lpDDSTemp, &IID_IDirectDrawSurface2, (void**)&lpFrontBuff);
	IDirectDrawSurface_Release( lpDDSTemp );
	if( FAILED( hResult ) )
	{
		ShowDDrawError( IDS_DDERR_ALLOC_PRIMARY, hResult, FALSE );
		return 0;
	} */
	else
	{
		DDSCAPS caps;
		ZeroMemory( &caps, sizeof( caps ) );
		caps.dwCaps = DDSCAPS_BACKBUFFER;
		hResult = IDirectDrawSurface_GetAttachedSurface( lpFrontBuff, &caps, &lpBackBuff);
		if( FAILED( hResult ) )
		{
			ShowDDrawError( IDS_DDERR_ALLOC_PRIMARY, hResult, FALSE );
			IDirectDrawSurface_Release( lpFrontBuff );
			lpFrontBuff = NULL;
			return 0;
		}
	}

	return 1;
}
#endif

int Init_Graphics( void )
{
	LPDIRECTDRAWPALETTE	lpDDPalette;
	HRESULT	hResult = DD_OK;

	if( ulLastMode == ulScreenMode )
		return 1;

	ulLastMode = ulScreenMode;
	lpbmi->bmiHeader.biWidth = ATARI_VIS_WIDTH;
	lpbmi->bmiHeader.biHeight = -ATARI_HEIGHT;	/* Negative because we are a top-down bitmap */
	
	/* Sometimes when changing from window to full screen, DirectSound becomes very
	   confused, so stop it here and start it re-init it again when we have finished
	   the screen change */
	/* Clear_Sound( TRUE ); */

	if( !origWndStyle )
		origWndStyle = GetWindowLong( MainhWnd, GWL_STYLE );

	ulScreenMode &= ~DDRAW_NO_MENU_MODE;

	ReleaseAllSurfaces();
	if( lpDirectDrawObject )
	{
		IDirectDraw_RestoreDisplayMode( lpDirectDrawObject );
		if( !(ulScreenMode & DDRAW_FULL) && (ulScreenMode & DDRAW_NONE) )
		{
			/* Release the object since we are going to be just GDI */
			IDirectDraw_Release( lpDirectDrawObject );
			lpDirectDrawObject = NULL; 
		}
	}

	if( !(ulScreenMode & DDRAW_FULL) && ((ulScreenMode & DDRAW_WIND) || (ulScreenMode & DDRAW_NONE)) )
	{
		if( Init_Windowed_Mode()!=DD_OK )
			return 0;

		/*If we're not going to be running DDraw windowed, return now, we're all set */
		if( ulScreenMode & DDRAW_NONE )
		{
			/* Sound_Initialise( ); */

			if( ulScreenMode & WINDOWED_768_480 )
				Atari_DisplayScreen = Screen_NoDDraw_768;
			else
				Atari_DisplayScreen = Screen_NoDDraw;
			return 1;
		}
	}

	/* Get the object if not already allocated */
	if( !lpDirectDrawObject )
	{
		/*LPDIRECTDRAW lpDDT; 
		if( SUCCEEDED( DirectDrawCreate(NULL,&lpDDT,0)))
		{
			if( SUCCEEDED( IDirectDraw_QueryInterface( lpDDT, &IID_IDirectDraw2, (void**)&lpDirectDrawObject)) )
				IDirectDraw_Release( lpDDT );
		}*/

		if( FAILED( DirectDrawCreate( NULL, &lpDirectDrawObject, 0 ) ) )
		{
			ShowDDrawError( IDS_DDERR_ALLOC_OBJ, hResult, FALSE );
			ulAtariState = ATARI_UNINITIALIZED | ATARI_PAUSED;
			PostMessage( MainhWnd, WM_CLOSE, 0, 0L );
			return 0;
		}
	}

	/* Handle the various FULL (exclusive screen) modes */
	if( ulScreenMode & DDRAW_FULL )
	{
		int	iMenuSize = GetSystemMetrics( SM_CYMENU );

		if( iMenuSize & 0x01 )
			iMenuSize++;

		if( MainhWnd )
			SetWindowLong( MainhWnd, GWL_STYLE, WS_BORDER );


		hResult = IDirectDraw_SetCooperativeLevel(lpDirectDrawObject, MainhWnd, 
						DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN | DDSCL_ALLOWREBOOT );

		if( FAILED( hResult ) )
		{
			ShowDDrawError( IDS_DDERR_SET_MODE, hResult, FALSE );
			return 0;
		}

		if( ulScreenMode & DDRAW_800_600 )
		{

			hResult = IDirectDraw_SetDisplayMode( lpDirectDrawObject, 800, 600, 8 );
			if( FAILED( hResult ) )
			{
				ShowDDrawError( IDS_DDERR_INIT_800, hResult, FALSE );
				return 0;
			}
			rcDest.left = ( 800 - (ATARI_VIS_WIDTH << 1) ) >> 1;
			rcDest.top =  ( 600 - (ATARI_HEIGHT << 1 ) ) >> 1;
			rcDest.right = rcDest.left + (ATARI_VIS_WIDTH << 1);
			rcDest.bottom = rcDest.top + (ATARI_HEIGHT << 1);
			rcSource.left = ATARI_HORZ_CLIP;
		    rcSource.top = 0;
			rcSource.right = ATARI_WIDTH - ATARI_HORZ_CLIP;
			rcSource.bottom = ATARI_HEIGHT;
		}

		if( ulScreenMode & DDRAW_640_480 )
		{
			hResult = IDirectDraw_SetDisplayMode( lpDirectDrawObject, 640, 480, 8 );
			if( FAILED( hResult ) )
			{
				ShowDDrawError( IDS_DDERR_INIT_640_480, hResult, FALSE );
				return 0;
			}
			rcDest.left = 0;
			rcDest.top = iMenuSize;
			rcDest.right = 640;
			rcDest.bottom = 480 - iMenuSize;

			rcSource.left = ATARI_HORZ_CLIP + ((ATARI_VIS_WIDTH - 320) >> 1);
			rcSource.right = ATARI_WIDTH - rcSource.left;
			rcSource.top = iMenuSize >> 1;
			rcSource.bottom = ATARI_HEIGHT - rcSource.top;
		}

		if( ulScreenMode & DDRAW_640_400 )
		{
			hResult = IDirectDraw_SetDisplayMode( lpDirectDrawObject, 640, 400, 8 );
			if( FAILED( hResult ) )
			{
				ShowDDrawError( IDS_DDERR_INIT_640_400, hResult, FALSE );
				return 0;
			}
			rcDest.left = 0;
			rcDest.top = 0;
			rcDest.right = 640;
			rcDest.bottom = 400;
			rcSource.left = ATARI_HORZ_CLIP + ((ATARI_VIS_WIDTH - 320) >> 1);
			rcSource.right = ATARI_WIDTH - rcSource.left;
		    rcSource.top = (ATARI_HEIGHT - 200) >> 1;
			rcSource.bottom = ATARI_HEIGHT - rcSource.top;
			ulScreenMode |= DDRAW_NO_MENU_MODE;
		}

		if( ulScreenMode & DDRAW_320_240 )
		{
			hResult = IDirectDraw_SetDisplayMode( lpDirectDrawObject, 320, 240, 8 );
			if( FAILED( hResult ) )
			{
				ShowDDrawError( IDS_DDERR_INIT_320_240, hResult, FALSE );
				return 0;
			}
			rcDest.left = 0;
			rcDest.top = iMenuSize;
			rcDest.right = 320;
			rcDest.bottom = 240 - iMenuSize;
			rcSource.left = ATARI_HORZ_CLIP + ((ATARI_VIS_WIDTH - 320) >> 1);
			rcSource.right = ATARI_WIDTH - rcSource.left;
		    rcSource.top = iMenuSize;
			rcSource.bottom = ATARI_HEIGHT - rcSource.top;
		}

		if( ulScreenMode & DDRAW_320_200 )
		{
			hResult = IDirectDraw_SetDisplayMode( lpDirectDrawObject, 320, 200, 8 );
			if( FAILED( hResult ) )
			{
				ShowDDrawError( IDS_DDERR_INIT_320_200, hResult, FALSE );
				return 0;
			}
			rcDest.left = 0;
			rcDest.top = 0;
			rcDest.right = 320;
			rcDest.bottom = 200;
			rcSource.left = ATARI_HORZ_CLIP + ((ATARI_VIS_WIDTH - 320) >> 1);
			rcSource.right = ATARI_WIDTH - rcSource.left;
		    rcSource.top = (ATARI_HEIGHT - 200) >> 1;
			rcSource.bottom = ATARI_HEIGHT - rcSource.top;
			ulScreenMode |= DDRAW_NO_MENU_MODE;
		}

		Atari_DisplayScreen = Screen_DDraw_Full_Blt;
	} 
	else // Handle the various non-exclusive screen modes. From the top of this routine, when we called Init_Windowed,
	{	 //we know when we get here lpPrimary=lpMemory=NULL
		if( origWndStyle && MainhWnd)
			SetWindowLong( MainhWnd, GWL_STYLE, origWndStyle );

		_ASSERT( !lpPrimary && !lpMemory );
		hResult = IDirectDraw_SetCooperativeLevel(lpDirectDrawObject, NULL, DDSCL_NORMAL);
		if( FAILED( hResult ) )
		{
			ShowDDrawError( IDS_DDERR_SET_MODE, hResult, FALSE );
			return 0;
		}

		if( ulScreenMode & WINDOWED_384_240 )
			Atari_DisplayScreen = Screen_Windowed_DDraw_384;
		else
			Atari_DisplayScreen = Screen_Windowed_DDraw_768;

	}

	/* Now that the cooperative mode is set, create a palette */
	hResult = IDirectDraw_CreatePalette( lpDirectDrawObject, DDPCAPS_8BIT | DDPCAPS_ALLOW256, pe, &lpDDPalette, NULL );
	if( FAILED( hResult ) )
	{
		ShowDDrawError( IDS_DDERR_ALLOC_PALETTE, hResult, FALSE );
		return 0;
	}

#ifndef USE_FLIP_BUFFER
	if( !SetupPrimarySurface() )
		return 0;

	if( ulScreenMode & DDRAW_FULL )
	{
		hResult = IDirectDrawSurface_SetPalette( lpPrimary, lpDDPalette );
		if( FAILED( hResult ) )
		{
			ShowDDrawError( IDS_DDERR_SETPAL_PRIMARY, hResult, FALSE );
			return 0;
		}
	}
#endif

	if( !SetupMemorySurface() )
		return 0;

	if( ulScreenMode & DDRAW_FULL )
	{
		hResult = IDirectDrawSurface_SetPalette( lpMemory, lpDDPalette );
		if( FAILED( hResult ) )
		{
			ShowDDrawError( IDS_DDERR_SETPAL_MEMORY, hResult, FALSE );
			return 0;
		}
	}

#ifdef USE_FLIP_BUFFER
	if( (ulScreenMode & DDRAW_FULL) && SetupFlipBuffer() )
	{
		hResult = IDirectDrawSurface_SetPalette( lpFrontBuff, lpDDPalette );
		if( SUCCEEDED( hResult ) )
			Atari_DisplayScreen = Screen_DDraw_Full_FlipBuff;
	}
#endif
	ulAtariState |= ATARI_HW_READY | ATARI_WINDOWS_READY;

	/* Sound_Initialise( ); */

	return 1;
}

static BOOL RestoreSurfaces( void )
{
	HRESULT hResult; 

	if( lpMemory )
	{
		if( FAILED( IDirectDrawSurface_IsLost( lpMemory ) ) )
		{
			hResult = IDirectDrawSurface_Restore( lpMemory );
			if( FAILED( hResult ) )
				return FALSE;
		}
	}

#ifdef USE_FLIP_BUFFER
	if( lpFrontBuff )
	{
		if( FAILED( IDirectDrawSurface_IsLost( lpFrontBuff ) ) )
		{
			hResult = IDirectDrawSurface_Restore( lpFrontBuff );
			if( FAILED( hResult ) )
				return FALSE;
		}
	}
#endif

	if( lpPrimary )
	{
		if( FAILED( IDirectDrawSurface_IsLost( lpPrimary ) ) ) 
		{
			hResult = IDirectDrawSurface_Restore( lpPrimary ) ;
			if( FAILED( hResult ) )
				return FALSE;
		}
	}
	return TRUE;
}

void Screen_Paused( UBYTE *screen )
{
	int i, j;
	UBYTE	*blank;
	
	for( i = 0; i < ATARI_HEIGHT; i++ )
	{
		blank = screen + i * ATARI_WIDTH + i % 2;
		for( j = 0; j < ATARI_WIDTH; j+=2 )
		{
			*blank = 0;
			blank +=2;
		}
	}
	Atari_DisplayScreen( screen );
}

static void Screen_Windowed_DDraw_384( UBYTE *screen )
{
	HDC	hdc;
	HRESULT hResult;

	if( ulAtariState & ATARI_PAUSED )
		return;
	
	/*Blts fail here in different bitplane depth modes. Easier to use StretchDIB           */
	/*Further, it is faster to use StetchBlt than the colorspace conversion below UNLESS we*/
	/*are stretching the output*/

	hResult = IDirectDrawSurface_GetDC( lpPrimary, &hdc );
	if( SUCCEEDED( hResult ) )
	{
		char *dest, *src, *stop;
		
		stop = &screentemp[0] + ATARI_VIS_SCREEN_SIZE;
		dest = &screentemp[0];
		src = &screenbuff[0] + ATARI_HORZ_CLIP;
		while( dest < stop )
		{
			memcpy( dest, src, ATARI_VIS_WIDTH );
			dest += ATARI_VIS_WIDTH;
			src += ATARI_WIDTH;
		}
		StretchDIBits( hdc, rcDest.left, rcDest.top, rcDest.right - rcDest.left, rcDest.bottom - rcDest.top, 
			rcSrcClip.left, rcSrcClip.top, rcSrcClip.right, rcSrcClip.bottom, 
			screentemp, lpbmi, DIB_RGB_COLORS, SRCCOPY);
	}
	else
		ShowDDrawError( IDS_DDERR_SURFACE_LOCK, hResult, TRUE );
	IDirectDrawSurface_ReleaseDC( lpPrimary, hdc );
	return;
}

static void Screen_Windowed_DDraw_768( UBYTE *screen )
{
	HDC	hdc;
	HRESULT	hResult;
	char *dest, *src, *stop;

	if( ulAtariState & ATARI_PAUSED )
		return;

	if( ulDiskState )
		Draw_DiskLED( screen );

	dest = screentemp;
	src = screenbuff + ATARI_HORZ_CLIP;
	stop = screentemp + ATARI_VIS_SCREEN_SIZE;
	while( dest < stop )
	{
		memcpy( dest, src, ATARI_VIS_WIDTH );
		dest += ATARI_VIS_WIDTH;
		src += ATARI_WIDTH;
	}
	hResult = IDirectDrawSurface_GetDC( lpMemory, &hdc );
	if( SUCCEEDED( hResult ) )
	{
		/* This StretchDIB basically does only the color space conversion to the memory surface */
		StretchDIBits( hdc, 0, 0, ATARI_VIS_WIDTH, ATARI_HEIGHT,
			0,  0, ATARI_VIS_WIDTH, ATARI_HEIGHT,
			screentemp, lpbmi, DIB_RGB_COLORS, SRCCOPY );
	}
	else
		ShowDDrawError( IDS_DDERR_SURFACE_LOCK, hResult, TRUE );
	IDirectDrawSurface_ReleaseDC( lpMemory, hdc );

	hResult = IDirectDrawSurface_Blt( lpPrimary, &rcDest, lpMemory, &rcSrcClip, DDBLT_ASYNC, NULL);

	if( FAILED( hResult ) && hResult != DDERR_WASSTILLDRAWING )
		ShowDDrawError( IDS_DDERR_PRIMARY_BLT, hResult, TRUE );
	return;
}

#ifdef USE_FLIP_BUFFER
static void Screen_DDraw_Full_FlipBuff( UBYTE *screen )
{
	DDSURFACEDESC	ddsd;
	HRESULT	hResult;

	if( ulDiskState )
		Draw_DiskLED( screen );

	IDirectDrawSurface_Blt( lpBackBuff, &rcDest, lpMemory, &rcSource, 0, NULL );

	//hResult = IDirectDrawSurface_Flip( lpFrontBuff, NULL, DDFLIP_WAIT );
	hResult = IDirectDrawSurface_Flip( lpFrontBuff, NULL, 0 );
    if( hResult == DDERR_SURFACELOST )
	{
		if( !RestoreSurfaces() )
			ShowDDrawError( IDS_DDERR_RESTORE_SURFACE, hResult, TRUE );
	}
	else if( FAILED( hResult ) && hResult != DDERR_WASSTILLDRAWING )
		ShowDDrawError( IDS_DDERR_PRIMARY_BLT, hResult, TRUE );
}
#endif

static void Screen_DDraw_Full_Blt( UBYTE *screen )
{
	HRESULT	hResult;

	if( ulDiskState )
		Draw_DiskLED( screen );

	hResult = IDirectDrawSurface_Blt( lpPrimary, &rcDest, lpMemory, &rcSource, DDBLT_ASYNC, NULL);
	if( SUCCEEDED( hResult ) || hResult == DDERR_WASSTILLDRAWING)
		return;

    if( hResult == DDERR_SURFACELOST )
	{
		if( !RestoreSurfaces() )
			ShowDDrawError( IDS_DDERR_RESTORE_SURFACE, hResult, TRUE );
	}
	else
		ShowDDrawError( IDS_DDERR_PRIMARY_BLT, hResult, TRUE );
	return;
}

static void Screen_NoDDraw_768( UBYTE *screen )
{
	char *dest, *src, *stop;

	if( ulDiskState )
		Draw_DiskLED( screen );

	dest = &screentemp[0];
	src = screenbuff + ATARI_HORZ_CLIP;
	stop = &screentemp[0] + ATARI_STRETCH_VIS_SCREEN_SIZE;
	while( dest < stop )
	{
		char *linep = dest;
		while( linep < dest + ATARI_STRETCH_VIS_WIDTH )
		{
			*linep++ = *src;
			*linep++ = *src++;
		}
		memcpy( linep, linep - ATARI_STRETCH_VIS_WIDTH, ATARI_STRETCH_VIS_WIDTH );
		dest += ATARI_STRETCH_VIS_WIDTH * 2;
		src += ATARI_HORZ_CLIP * 2;
	}
	StretchDIBits( h_screenDC, 0, 0, ATARI_STRETCH_VIS_WIDTH, ATARI_STRETCH_HEIGHT,
		0, 0, ATARI_STRETCH_VIS_WIDTH, ATARI_STRETCH_HEIGHT,
		screentemp, lpbmi, DIB_RGB_COLORS, SRCCOPY);
}

static void Screen_NoDDraw( UBYTE *screen )
{
	char *dest, *src, *stop;

	if( ulDiskState )
		Draw_DiskLED( screen );

	stop = &screentemp[0] + ATARI_VIS_SCREEN_SIZE;
	dest = &screentemp[0];
	src = &screenbuff[0] + ATARI_HORZ_CLIP;
	while( dest < stop )
	{
		memcpy( dest, src, ATARI_VIS_WIDTH );
		dest += ATARI_VIS_WIDTH;
		src += ATARI_WIDTH;
	}
	StretchDIBits( h_screenDC, 0, 0, ATARI_VIS_WIDTH, ATARI_HEIGHT, 
		0, 0, ATARI_VIS_WIDTH, ATARI_HEIGHT, 
		screentemp, lpbmi, DIB_RGB_COLORS, SRCCOPY);
}

void SafeShowScreen( void )
{
	if( ulScreenMode & DDRAW_NONE )
	{
		Screen_NoDDraw( (unsigned char *)atari_screen );
		return;
	}

	if( (ulScreenMode & DDRAW_WIND) )
	{
		if( ulScreenMode & WINDOWED_384_240 )
		{
			Screen_Windowed_DDraw_384( (unsigned char *)atari_screen );
			return;
		}

		if( ulScreenMode & WINDOWED_768_480 )
		{
			Screen_Windowed_DDraw_768( (unsigned char *)atari_screen );
			return;
		}
	}
	
	_ASSERT( ulScreenMode & DDRAW_FULL );
	Screen_DDraw_Full_Blt( (unsigned char *)atari_screen );
}


/*
**-----------------------------------------------------------------------------
** Name:    GetDDErrorString
** Purpose: outputs a debug string to debugger
**-----------------------------------------------------------------------------
*/

static BOOL GetDDErrorString (HRESULT hResult, LPTSTR lpszErrorBuff, DWORD cchError)
{
    DWORD  cLen;
    LPTSTR lpszError;
    TCHAR  szMsg[256];
	
    // Check parameters
    if (!lpszErrorBuff || !cchError)
    {
        // Error, invalid parameters
        return FALSE;
    }
	
    switch (hResult)
    {
		case DD_OK:
		/*
		* The request completed successfully.
		*/
        lpszError = TEXT("DD_OK");
        break;

		/*
		* This object is already initialized
		*/
	case DDERR_ALREADYINITIALIZED:
		lpszError = TEXT("DDERR_ALREADYINITIALIZED");
		break;
		
		/*
		* This surface can not be attached to the requested surface.
		*/
    case DDERR_CANNOTATTACHSURFACE:
        lpszError = TEXT("DDERR_CANNOTATTACHSURFACE");
        break;
		
		/*
		* This surface can not be detached from the requested surface.
		*/
    case DDERR_CANNOTDETACHSURFACE:
        lpszError = TEXT("DDERR_CANNOTDETACHSURFACE");
        break;
		
		/*
		* Support is currently not available.
		*/
    case DDERR_CURRENTLYNOTAVAIL:
        lpszError = TEXT("DDERR_CURRENTLYNOTAVAIL");
        break;
		
		/*
		* An exception was encountered while performing the requested operation
		*/
    case DDERR_EXCEPTION:
        lpszError = TEXT("DDERR_EXCEPTION");
        break;
		
		/*
		* Generic failure.
		*/
    case DDERR_GENERIC:
		lpszError = TEXT("DDERR_GENERIC");
		break;
		
		/*
		* Height of rectangle provided is not a multiple of reqd alignment
		*/
    case DDERR_HEIGHTALIGN:
        lpszError = TEXT("DDERR_HEIGHTALIGN");
        break;
		
		/*
		* Unable to match primary surface creation request with existing
		* primary surface.
		*/
    case DDERR_INCOMPATIBLEPRIMARY:
        lpszError = TEXT("DDERR_INCOMPATIBLEPRIMARY");
        break;
		
		/*
		* One or more of the caps bits passed to the callback are incorrect.
		*/
    case DDERR_INVALIDCAPS:
        lpszError = TEXT("DDERR_INVALIDCAPS");
        break;
		
		/*
		* DirectDraw does not support provided Cliplist.
		*/
    case DDERR_INVALIDCLIPLIST:
        lpszError = TEXT("DDERR_INVALIDCLIPLIST");
        break;
		
		/*
		* DirectDraw does not support the requested mode
		*/
    case DDERR_INVALIDMODE:
        lpszError = TEXT("DDERR_INVALIDMODE");
        break;
		
		/*
		* DirectDraw received a pointer that was an invalid DIRECTDRAW object.
		*/
    case DDERR_INVALIDOBJECT:
        lpszError = TEXT("DDERR_INVALIDOBJECT");
        break;
		
		/*
		* One or more of the parameters passed to the callback function are
		* incorrect.
		*/
    case DDERR_INVALIDPARAMS:
        lpszError = TEXT("DDERR_INVALIDPARAMS");
        break;
		
		/*
		* pixel format was invalid as specified
		*/
    case DDERR_INVALIDPIXELFORMAT:
        lpszError = TEXT("DDERR_INVALIDPIXELFORMAT");
        break;
		
		/*
		* Rectangle provided was invalid.
		*/
    case DDERR_INVALIDRECT:
        lpszError = TEXT("DDERR_INVALIDRECT");
        break;
		
		/*
		* Operation could not be carried out because one or more surfaces are locked
		*/
    case DDERR_LOCKEDSURFACES:
        lpszError = TEXT("DDERR_LOCKEDSURFACES");
        break;
		
		/*
		* There is no 3D present.
		*/
    case DDERR_NO3D:
        lpszError = TEXT("DDERR_NO3D");
        break;
		
		/*
		* Operation could not be carried out because there is no alpha accleration
		* hardware present or available.
		*/
    case DDERR_NOALPHAHW:
        lpszError = TEXT("DDERR_NOALPHAHW");
        break;
		
		
		/*
		* no clip list available
		*/
    case DDERR_NOCLIPLIST:
        lpszError = TEXT("DDERR_NOCLIPLIST");
        break;
		
		/*
		* Operation could not be carried out because there is no color conversion
		* hardware present or available.
		*/
    case DDERR_NOCOLORCONVHW:
        lpszError = TEXT("DDERR_NOCOLORCONVHW");
        break;
		
		/*
		* Create function called without DirectDraw object method SetCooperativeLevel
		* being called.
		*/
    case DDERR_NOCOOPERATIVELEVELSET:
        lpszError = TEXT("DDERR_NOCOOPERATIVELEVELSET");
        break;
		
		/*
		* Surface doesn't currently have a color key
		*/
    case DDERR_NOCOLORKEY:
        lpszError = TEXT("DDERR_NOCOLORKEY");
        break;
		
		/*
		* Operation could not be carried out because there is no hardware support
		* of the dest color key.
		*/
    case DDERR_NOCOLORKEYHW:
        lpszError = TEXT("DDERR_NOCOLORKEYHW");
        break;
		
		/*
		* No DirectDraw support possible with current display driver
		*/
    case DDERR_NODIRECTDRAWSUPPORT:
        lpszError = TEXT("DDERR_NODIRECTDRAWSUPPORT");
        break;
		
		/*
		* Operation requires the application to have exclusive mode but the
		* application does not have exclusive mode.
		*/
    case DDERR_NOEXCLUSIVEMODE:
        lpszError = TEXT("DDERR_NOEXCLUSIVEMODE");
        break;
		
		/*
		* Flipping visible surfaces is not supported.
		*/
    case DDERR_NOFLIPHW:
        lpszError = TEXT("DDERR_NOFLIPHW");
        break;
		
		/*
		* There is no GDI present.
		*/
    case DDERR_NOGDI:
        lpszError = TEXT("DDERR_NOGDI");
        break;
		
		/*
		* Operation could not be carried out because there is no hardware present
		* or available.
		*/
    case DDERR_NOMIRRORHW:
        lpszError = TEXT("DDERR_NOMIRRORHW");
        break;
		
		/*
		* Requested item was not found
		*/
    case DDERR_NOTFOUND:
        lpszError = TEXT("DDERR_NOTFOUND");
        break;
		
		/*
		* Operation could not be carried out because there is no overlay hardware
		* present or available.
		*/
    case DDERR_NOOVERLAYHW:
        lpszError = TEXT("DDERR_NOOVERLAYHW");
        break;
		
		/*
		* Operation could not be carried out because there is no appropriate raster
		* op hardware present or available.
		*/
    case DDERR_NORASTEROPHW:
		lpszError = TEXT("DDERR_NORASTEROPHW");
        break;
		
		/*
		* Operation could not be carried out because there is no rotation hardware
		* present or available.
		*/
    case DDERR_NOROTATIONHW:
        lpszError = TEXT("DDERR_NOROTATIONHW");
        break;
		
		/*
		* Operation could not be carried out because there is no hardware support
		* for stretching
		*/
    case DDERR_NOSTRETCHHW:
        lpszError = TEXT("DDERR_NOSTRETCHHW");
        break;
		
		/*
		* DirectDrawSurface is not in 4 bit color palette and the requested operation
		* requires 4 bit color palette.
		*/
    case DDERR_NOT4BITCOLOR:
        lpszError = TEXT("DDERR_NOT4BITCOLOR");
        break;
		
		/*
		* DirectDrawSurface is not in 4 bit color index palette and the requested
		* operation requires 4 bit color index palette.
		*/
    case DDERR_NOT4BITCOLORINDEX:
        lpszError = TEXT("DDERR_NOT4BITCOLORINDEX");
        break;
		
		/*
		* DirectDraw Surface is not in 8 bit color mode and the requested operation
		* requires 8 bit color.
		*/
    case DDERR_NOT8BITCOLOR:
        lpszError = TEXT("DDERR_NOT8BITCOLOR");
        break;
		
		/*
		* Operation could not be carried out because there is no texture mapping
		* hardware present or available.
		*/
    case DDERR_NOTEXTUREHW:
        lpszError = TEXT("DDERR_NOTEXTUREHW");
        break;
		
		/*
		* Operation could not be carried out because there is no hardware support
		* for vertical blank synchronized operations.
		*/
    case DDERR_NOVSYNCHW:
        lpszError = TEXT("DDERR_NOVSYNCHW");
        break;
		
		/*
		* Operation could not be carried out because there is no hardware support
		* for zbuffer blting.
		*/
    case DDERR_NOZBUFFERHW:
        lpszError = TEXT("DDERR_NOZBUFFERHW");
        break;
		
		/*
		* Overlay surfaces could not be z layered based on their BltOrder because
		* the hardware does not support z layering of overlays.
		*/
    case DDERR_NOZOVERLAYHW:
        lpszError = TEXT("DDERR_NOZOVERLAYHW");
        break;
		
		/*
		* The hardware needed for the requested operation has already been
		* allocated.
		*/
    case DDERR_OUTOFCAPS:
        lpszError = TEXT("DDERR_OUTOFCAPS");
        break;
		
		/*
		* DirectDraw does not have enough memory to perform the operation.
		*/
    case DDERR_OUTOFMEMORY:
        lpszError = TEXT("DDERR_OUTOFMEMORY");
        break;
		
		/*
		* DirectDraw does not have enough memory to perform the operation.
		*/
    case DDERR_OUTOFVIDEOMEMORY:
        lpszError = TEXT("DDERR_OUTOFVIDEOMEMORY");
        break;
		
		/*
		* hardware does not support clipped overlays
		*/
    case DDERR_OVERLAYCANTCLIP:
        lpszError = TEXT("DDERR_OVERLAYCANTCLIP");
        break;
		
		/*
		* Can only have ony color key active at one time for overlays
		*/
    case DDERR_OVERLAYCOLORKEYONLYONEACTIVE:
        lpszError = TEXT("DDERR_OVERLAYCOLORKEYONLYONEACTIVE");
        break;
		
		/*
		* Access to this palette is being refused because the palette is already
		* locked by another thread.
		*/
    case DDERR_PALETTEBUSY:
        lpszError = TEXT("DDERR_PALETTEBUSY");
        break;
		
		/*
		* No src color key specified for this operation.
		*/
    case DDERR_COLORKEYNOTSET:
        lpszError = TEXT("DDERR_COLORKEYNOTSET");
        break;
		
		/*
		* This surface is already attached to the surface it is being attached to.
		*/
    case DDERR_SURFACEALREADYATTACHED:
        lpszError = TEXT("DDERR_SURFACEALREADYATTACHED");
        break;
		
		/*
		* This surface is already a dependency of the surface it is being made a
		* dependency of.
		*/
    case DDERR_SURFACEALREADYDEPENDENT:
        lpszError = TEXT("DDERR_SURFACEALREADYDEPENDENT");
        break;
		
		/*
		* Access to this surface is being refused because the surface is already
		* locked by another thread.
		*/
    case DDERR_SURFACEBUSY:
        lpszError = TEXT("DDERR_SURFACEBUSY");
        break;
		
		/*
		* Access to this surface is being refused because no driver exists
		* which can supply a pointer to the surface.
		* This is most likely to happen when attempting to lock the primary
		* surface when no DCI provider is present.
		* Will also happen on attempts to lock an optimized surface.
		*/
    case DDERR_CANTLOCKSURFACE:
        lpszError = TEXT("DDERR_CANTLOCKSURFACE");
        break;
		
		/*
		* Access to Surface refused because Surface is obscured.
		*/
    case DDERR_SURFACEISOBSCURED:
        lpszError = TEXT("DDERR_SURFACEISOBSCURED");
        break;
		
		/*
		* Access to this surface is being refused because the surface is gone.
		* The DIRECTDRAWSURFACE object representing this surface should
		* have Restore called on it.
		*/
    case DDERR_SURFACELOST:
        lpszError = TEXT("DDERR_SURFACELOST");
        break;
		
		/*
		* The requested surface is not attached.
		*/
    case DDERR_SURFACENOTATTACHED:
        lpszError = TEXT("DDERR_SURFACENOTATTACHED");
        break;
		
		/*
		* Height requested by DirectDraw is too large.
		*/
    case DDERR_TOOBIGHEIGHT:
        lpszError = TEXT("DDERR_TOOBIGHEIGHT");
        break;
		
		/*
		* Size requested by DirectDraw is too large --	 The individual height and
		* width are OK.
		*/
    case DDERR_TOOBIGSIZE:
        lpszError = TEXT("DDERR_TOOBIGSIZE");
        break;
		
		/*
		* Width requested by DirectDraw is too large.
		*/
    case DDERR_TOOBIGWIDTH:
        lpszError = TEXT("DDERR_TOOBIGWIDTH");
        break;
		
		/*
		* Action not supported.
		*/
    case DDERR_UNSUPPORTED:
        lpszError = TEXT("DDERR_UNSUPPORTED");
        break;
		
		/*
		* FOURCC format requested is unsupported by DirectDraw
		*/
    case DDERR_UNSUPPORTEDFORMAT:
        lpszError = TEXT("DDERR_UNSUPPORTEDFORMAT");
        break;
		
		/*
		* Bitmask in the pixel format requested is unsupported by DirectDraw
		*/
    case DDERR_UNSUPPORTEDMASK:
        lpszError = TEXT("DDERR_UNSUPPORTEDMASK");
        break;
		
		/*
		* vertical blank is in progress
		*/
    case DDERR_VERTICALBLANKINPROGRESS:
        lpszError = TEXT("DDERR_VERTICALBLANKINPROGRESS");
        break;
		
		/*
		* Informs DirectDraw that the previous Blt which is transfering information
		* to or from this Surface is incomplete.
		*/
    case DDERR_WASSTILLDRAWING:
        lpszError = TEXT("DDERR_WASSTILLDRAWING");
        break;
		
		
		/*
		* Rectangle provided was not horizontally aligned on reqd. boundary
		*/
    case DDERR_XALIGN:
        lpszError = TEXT("DDERR_XALIGN");
        break;
		
		/*
		* The GUID passed to DirectDrawCreate is not a valid DirectDraw driver
		* identifier.
		*/
    case DDERR_INVALIDDIRECTDRAWGUID:
        lpszError = TEXT("DDERR_INVALIDDIRECTDRAWGUID");
        break;
		
		/*
		* A DirectDraw object representing this driver has already been created
		* for this process.
		*/
    case DDERR_DIRECTDRAWALREADYCREATED:
        lpszError = TEXT("DDERR_DIRECTDRAWALREADYCREATED");
        break;
		
		/*
		* A hardware only DirectDraw object creation was attempted but the driver
		* did not support any hardware.
		*/
    case DDERR_NODIRECTDRAWHW:
        lpszError = TEXT("DDERR_NODIRECTDRAWHW");
        break;
		
		/*
		* this process already has created a primary surface
		*/
    case DDERR_PRIMARYSURFACEALREADYEXISTS:
        lpszError = TEXT("DDERR_PRIMARYSURFACEALREADYEXISTS");
        break;
		
		/*
		* software emulation not available.
		*/
    case DDERR_NOEMULATION:
        lpszError = TEXT("DDERR_NOEMULATION");
        break;
		
		/*
		* region passed to Clipper::GetClipList is too small.
		*/
    case DDERR_REGIONTOOSMALL:
        lpszError = TEXT("DDERR_REGIONTOOSMALL");
        break;
		
		/*
		* an attempt was made to set a clip list for a clipper objec that
		* is already monitoring an hwnd.
		*/
    case DDERR_CLIPPERISUSINGHWND:
        lpszError = TEXT("DDERR_CLIPPERISUSINGHWND");
        break;
		
		/*
		* No clipper object attached to surface object
		*/
    case DDERR_NOCLIPPERATTACHED:
        lpszError = TEXT("DDERR_NOCLIPPERATTACHED");
        break;
		
		/*
		* Clipper notification requires an HWND or
		* no HWND has previously been set as the CooperativeLevel HWND.
		*/
    case DDERR_NOHWND:
        lpszError = TEXT("DDERR_NOHWND");
        break;
		
		/*
		* HWND used by DirectDraw CooperativeLevel has been subclassed,
		* this prevents DirectDraw from restoring state.
		*/
    case DDERR_HWNDSUBCLASSED:
        lpszError = TEXT("DDERR_HWNDSUBCLASSED");
        break;
		
		/*
		* The CooperativeLevel HWND has already been set.
		* It can not be reset while the process has surfaces or palettes created.
		*/
    case DDERR_HWNDALREADYSET:
        lpszError = TEXT("DDERR_HWNDALREADYSET");
        break;
		
		/*
		* No palette object attached to this surface.
		*/
    case DDERR_NOPALETTEATTACHED:
        lpszError = TEXT("DDERR_NOPALETTEATTACHED");
        break;
		
		/*
		* No hardware support for 16 or 256 color palettes.
		*/
    case DDERR_NOPALETTEHW:
        lpszError = TEXT("DDERR_NOPALETTEHW");
        break;
		
		/*
		* If a clipper object is attached to the source surface passed into a
		* BltFast call.
		*/
    case DDERR_BLTFASTCANTCLIP:
        lpszError = TEXT("DDERR_BLTFASTCANTCLIP");
        break;
		
		/*
		* No blter.
		*/
    case DDERR_NOBLTHW:
        lpszError = TEXT("DDERR_NOBLTHW");
        break;
		
		/*
		* No DirectDraw ROP hardware.
		*/
    case DDERR_NODDROPSHW:
        lpszError = TEXT("DDERR_NODDROPSHW");
        break;
		
		/*
		* returned when GetOverlayPosition is called on a hidden overlay
		*/
    case DDERR_OVERLAYNOTVISIBLE:
        lpszError = TEXT("DDERR_OVERLAYNOTVISIBLE");
        break;
		
		/*
		* returned when GetOverlayPosition is called on a overlay that UpdateOverlay
		* has never been called on to establish a destionation.
		*/
    case DDERR_NOOVERLAYDEST:
        lpszError = TEXT("DDERR_NOOVERLAYDEST");
        break;
		
		/*
		* returned when the position of the overlay on the destionation is no longer
		* legal for that destionation.
		*/
    case DDERR_INVALIDPOSITION:
        lpszError = TEXT("DDERR_INVALIDPOSITION");
        break;
		
		/*
		* returned when an overlay member is called for a non-overlay surface
		*/
    case DDERR_NOTAOVERLAYSURFACE:
        lpszError = TEXT("DDERR_NOTAOVERLAYSURFACE");
        break;
		
		/*
		* An attempt was made to set the cooperative level when it was already
		* set to exclusive.
		*/
    case DDERR_EXCLUSIVEMODEALREADYSET:
        lpszError = TEXT("DDERR_EXCLUSIVEMODEALREADYSET");
        break;
		
		/*
		* An attempt has been made to flip a surface that is not flippable.
		*/
    case DDERR_NOTFLIPPABLE:
        lpszError = TEXT("DDERR_NOTFLIPPABLE");
        break;
		
		/*
		* Can't duplicate primary & 3D surfaces, or surfaces that are implicitly
		* created.
		*/
    case DDERR_CANTDUPLICATE:
        lpszError = TEXT("DDERR_CANTDUPLICATE");
        break;
		
		/*
		* Surface was not locked.  An attempt to unlock a surface that was not
		* locked at all, or by this process, has been attempted.
		*/
    case DDERR_NOTLOCKED:
        lpszError = TEXT("DDERR_NOTLOCKED");
        break;
		
		/*
		* Windows can not create any more DCs
		*/
    case DDERR_CANTCREATEDC:
        lpszError = TEXT("DDERR_CANTCREATEDC");
        break;
		
		/*
		* No DC was ever created for this surface.
		*/
    case DDERR_NODC:
        lpszError = TEXT("DDERR_NODC");
        break;
		
		/*
		* This surface can not be restored because it was created in a different
		* mode.
		*/
    case DDERR_WRONGMODE:
        lpszError = TEXT("DDERR_WRONGMODE");
        break;
		
		/*
		* This surface can not be restored because it is an implicitly created
		* surface.
		*/
    case DDERR_IMPLICITLYCREATED:
        lpszError = TEXT("DDERR_IMPLICITLYCREATED");
        break;
		
		/*
		* The surface being used is not a palette-based surface
		*/
    case DDERR_NOTPALETTIZED:
        lpszError = TEXT("DDERR_NOTPALETTIZED");
        break;
		
		
		/*
		* The display is currently in an unsupported mode
		*/
    case DDERR_UNSUPPORTEDMODE:
        lpszError = TEXT("DDERR_UNSUPPORTEDMODE");
        break;
		
		/*
		* Operation could not be carried out because there is no mip-map
		* texture mapping hardware present or available.
		*/
    case DDERR_NOMIPMAPHW:
        lpszError = TEXT("DDERR_NOMIPMAPHW");
        break;
		
		/*
		* The requested action could not be performed because the surface was of
		* the wrong type.
		*/
    case DDERR_INVALIDSURFACETYPE:
        lpszError = TEXT("DDERR_INVALIDSURFACETYPE");
        break;
		
		/*
		* Device does not support optimized surfaces, therefore no video memory optimized surfaces
		*/
    case DDERR_NOOPTIMIZEHW:
        lpszError = TEXT("DDERR_NOOPTIMIZEHW");
        break;
		
		/*
		* Surface is an optimized surface, but has not yet been allocated any memory
		*/
    case DDERR_NOTLOADED:
        lpszError = TEXT("DDERR_NOTLOADED");
        break;
		
		/*
		* A DC has already been returned for this surface. Only one DC can be
		* retrieved per surface.
		*/
    case DDERR_DCALREADYCREATED:
        lpszError = TEXT("DDERR_DCALREADYCREATED");
        break;
		
		/*
		* An attempt was made to allocate non-local video memory from a device
		* that does not support non-local video memory.
		*/
    case DDERR_NONONLOCALVIDMEM:
        lpszError = TEXT("DDERR_NONONLOCALVIDMEM");
        break;
		
		/*
		* The attempt to page lock a surface failed.
		*/
    case DDERR_CANTPAGELOCK:
        lpszError = TEXT("DDERR_CANTPAGELOCK");
        break;
		
		/*
		* The attempt to page unlock a surface failed.
		*/
    case DDERR_CANTPAGEUNLOCK:
        lpszError = TEXT("DDERR_CANTPAGEUNLOCK");
        break;
		
		/*
		* An attempt was made to page unlock a surface with no outstanding page locks.
		*/
    case DDERR_NOTPAGELOCKED:
        lpszError = TEXT("DDERR_NOTPAGELOCKED");
        break;
		
		/*
		* There is more data available than the specified buffer size could hold
		*/
    case DDERR_MOREDATA:
        lpszError = TEXT("DDERR_MOREDATA");
        break;
		
		/*
		* The video port is not active
		*/
    case DDERR_VIDEONOTACTIVE:
        lpszError = TEXT("DDERR_VIDEONOTACTIVE");
        break;
		
		/*
		* Surfaces created by one direct draw device cannot be used directly by
		* another direct draw device.
		*/
    case DDERR_DEVICEDOESNTOWNSURFACE:
        lpszError = TEXT("DDERR_DEVICEDOESNTOWNSURFACE");
        break;
		
		/*
		* An attempt was made to invoke an interface member of a DirectDraw object
		* created by CoCreateInstance() before it was initialized.
		*/
    case DDERR_NOTINITIALIZED:
        lpszError = TEXT("DDERR_NOTINITIALIZED");
        break;

    default:
        /* Unknown DD/App Error */
        wsprintf (szMsg, "Error #%ld", (DWORD)hResult );
        lpszError = szMsg;
        break;
    }
	
    // Copy DD Error string to buff
    cLen = strlen (lpszError);
    if (cLen >= cchError)
    {
        cLen = cchError - 1;
    }
	
    if (cLen)
    {
        strncpy (lpszErrorBuff, lpszError, cLen);
        lpszErrorBuff[cLen] = 0;
    }
	
    return TRUE;
} // End GetDDErrorString

