//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include "wintypes.h"
#include <unistd.h>
#include <fcntl.h>
#include <math.h>
#include "carray.h"
#include "cfile.h"

#include "pub.def"

#define CMD_TB_FIRST 0x38

//list of all basic commands
char* aBasicCmds[] =
{
	"REM",			//00
	"DATA",			//01
	"INPUT",			//02
	"COLOR",			//03
	"LIST",			//04
	"ENTER",			//05
	"LET",			//06
	"IF",				//07
	"FOR",			//08
	"NEXT",			//09
	"GOTO",			//0a
	"GO TO",			//0b
	"GOSUB",			//0c
	"TRAP",			//0d
	"BYE",			//0e
	"CONT",			//0f
	"COM",			//10
	"CLOSE",			//11
	"CLR",			//12
	"DEG",			//13
	"DIM",			//14
	"END",			//15
	"NEW",			//16
	"OPEN",			//17
	"LOAD",			//18
	"SAVE",			//19
	"STATUS",		//1a
	"NOTE",			//1b
	"POINT",			//1c
	"XIO",			//1d
	"ON",				//1e
	"POKE",			//1f
	"PRINT",			//20
	"RAD",			//21
	"READ",			//22
	"RESTORE",		//23
	"RETURN",		//24
	"RUN",			//25
	"STOP",			//26
	"POP",			//27
	"?",				//28
	"GET",			//29
	"PUT",			//2a
	"GRAPHICS",		//2b
	"PLOT",			//2c
	"POSITION",		//2d
	"DOS",			//2e
	"DRAWTO",		//2f
	"SETCOLOR",		//30
	"LOCATE",		//31
	"SOUND",			//32
	"LPRINT",		//33
	"CSAVE",			//34
	"CLOAD",			//35
	"",				//36 silent let
	"ERROR -",		//37 last Atari Basic command
	"DPOKE",			//38
	"MOVE",			//39
	"-MOVE",			//3A
	"*F",				//3B
	"REPEAT",		//3C
	"UNTIL",			//3D
	"WHILE",			//3E
	"WEND",			//3F
	"ELSE",			//40
	"ENDIF",			//41
	"BPUT",			//42
	"BGET",			//43
	"FILLTO",		//44
	"DO",				//45
	"LOOP",			//46
	"EXIT",			//47
	"DIR",			//48
	"LOCK",			//49
	"UNLOCK",		//4A
	"RENAME",		//4B
	"DELETE",		//4C
	"PAUSE",			//4D
	"TIME$=",		//4E
	"PROC",			//4F
	"EXEC",			//50
	"ENDPROC",		//51
	"FCOLOR",		//52
	"*L",				//53
	"------------------------------",				//54
	"RENUM",			//55
	"DEL",			//56
	"DUMP",			//57
	"TRACE",			//58
	"TEXT",			//59
	"BLOAD",			//5A
	"BRUN",			//5B
	"GO#",			//5C
	"#",				//5D
	"*B",				//5E
	"PAINT",			//5F
	"CLS",			//60
	"DSOUND",		//61
	"CIRCLE",		//62
	"\%PUT",			//63
	"\%GET",			//64
};						  
						  
#define CMD_NUM ( sizeof( aBasicCmds ) / sizeof( aBasicCmds[ 0 ] ) )
						  
#define OP_TB_FIRST 0x55

#define OP_EOL			0x16
#define OP_NCONST		0x0E
#define OP_NHCONST 	0x0D
#define OP_SCONST		0x0F

#define OP_FIRST		0x0D

//list of all basic operands
char* aBasicOps[] = {	  
   "NHCONST",       //0D
	"NCONST",		  //0E
	"SCONST",		  //0F
	"NOUSE",			  //10
	"NOUSE",			  //11
	",",				  //12
	"$",				  //13
	":",				  //14
	";",				  //15
	"EOL",			  //16
	" GOTO ",		  //17
	" GOSUB ",		  //18
	" TO ",			  //19
	" STEP ",		  //1A
	" THEN ",		  //1B
	"#",				  //1C
	"<=",				  //1D
	"<>",				  //1E
	">=",				  //1F
	"<",				  //20
	">",				  //21
	"=",				  //22
	"^",				  //23
	"*",				  //24
	"+",				  //25
	"-",				  //26
	"/",				  //27
	" NOT ",			  //28
	" OR ",			  //29
	" AND ",			  //2A
	"(",				  //2B
	")",				  //2C
	"=",				  //2D
	"=",				  //2E
	"<=",				  //2F
	"<>",				  //30
	">=",				  //31
	"<",				  //32
	">",				  //33
	"=",				  //34
	"+",				  //35
	"-",				  //36
	"(",				  //37
	"(",				  //38
	"(",				  //39
	"(",				  //3A
	"(",				  //3B
	",",				  //3C
	"STR$",			  //3D
	"CHR$",			  //3E
	"USR",			  //3F
	"ASC",			  //40
	"VAL",			  //41
	"LEN",			  //42
	"ADR",			  //43
	"ATN",			  //44
	"COS",			  //45
	"PEEK",			  //46
	"SIN",			  //47
	"RND",			  //48
	"FRE",			  //49
	"EXP",			  //4A
	"LOG",			  //4B
	"CLOG",			  //4C
	"SQR",			  //4D
	"SGN",			  //4E
	"ABS",			  //4F
	"INT",			  //50
	"PADDLE",		  //51
	"STICK",			  //52
	"PTRIG",			  //53
	"STRIG",			  //54	last Atari Basic
	"DPEEK",			  //55
	"&",					//56
	"!",					//57
	"INSTR",				//58
	"INKEY$",			//59
	" EXOR ",			//5A
	"HEX$",				//5B
	"DEC",				//5C
	" DIV ",				//5D
	"FRAC",				//5E
	"TIME$",				//5F
	"TIME",				//60
	" MOD ",				//61
	" EXEC ",			//62
	"RND",				//63
	"RAND",				//64
	"TRUNC",				//65
	"%0",					//66
	"%1",					//67
	"%2",					//68
	"%3",					//69
	" GO# ",				//6A
	"UINSTR",			//6B
	"ERR",				//6C
	"ERL",				//6D
};

#define OPS_NUM ( sizeof( aBasicOps ) / sizeof( aBasicOps[ 0 ] ) )

#define VART_SCALAR		0x00
#define VART_ARRAYU		0x40
#define VART_ARRAY		0x41
#define VART_STRINGU		0x80
#define VART_STRING		0x81
#define VART_PROC			0xC1
#define VART_LABEL		0xC2

#define SHEADER PRG_NAME " v" PRG_VERSION " (c) " PRG_COPYRIGHT " " PRG_AUTHOR "\n"

#define HEADER SHEADER \
   PRG_DESC "\n" \
	"  Latest version can be found at " PRG_URL "\n" \
	"  Published under GPL. See GPL.TXT.\n" \
	"  Thanks to Russ Gilbert for his SALVAGE programs.\n\n"

#define USAGE HEADER "Usage:  " PRG_NAME " " PRG_USAGE

BOOL GenCode( CFile* pcf, WORD wStart, WORD wEnd );
BOOL GenLineCode( CFile* pcf, WORD wEnd );
BOOL GenCmdCode( CFile* pcf, WORD wLineStart, WORD wLineEnd );
BOOL GenOpsCode( CFile* pcf, WORD wTokEnd );
double ReadAtariBCD( CFile* pcf );

BOOL Vars_Load( CFile* pcf, WORD wVNT, WORD wVNTL, WORD wVVT, WORD NV );
void Var_Get( char* szRet, int iNum );
void Vars_Destroy();

FILE* g_fout = NULL;
BOOL g_bTB_Code = FALSE;
CArray caBasicVars;
BOOL g_bVerbose = FALSE;
BOOL g_bAtariOut = FALSE;
BOOL g_bExtOut = TRUE;
BOOL g_bHasErrors = FALSE;


#include "switches.cpp"

int main( int argc, char* argv[] )
{
	setbuf( stdout, NULL );
	setbuf( stderr, NULL );

	if ( !SWITCHES_Init( &argc, argv ) )
		return 1;

	if ( argc < 2 )
	{
		SWFN_HELP( USAGE );
		return 1;
	}

	if ( g_bExtOut )
		fprintf( stderr, SHEADER "\n" );
	else
		fprintf( stderr, HEADER );

	char* szInfile = argv[ 1 ];
	char* szOutfile = NULL;

	if ( argc > 2 )
		szOutfile = argv[ 2 ];

	CFile cf;

	if ( !cf.Open( szInfile ) )
	{
		fprintf( stderr, "Can't open input file '%s'\n", szInfile );
		return 1;
	}

	if ( szOutfile && !g_bAtariOut )
		g_fout = fopen( szOutfile, "wt" );

	if ( szOutfile && g_bAtariOut )
		g_fout = fopen( szOutfile, "wb" );

	if ( !g_fout )
		g_fout = stdout;

	//Don't emit additional info if you are in Atari mode
	if ( g_bAtariOut )
		g_bExtOut = FALSE;

	if ( g_bExtOut )
	{
		fprintf( g_fout, HEADER );
		fprintf( g_fout, "Input file: %s\n\n", szInfile );
	}

	WORD wFileLen = cf.GetLength();

	//read head
	WORD wLOMEM = cf.readLEw();
	WORD wVNT = cf.readLEw();
	WORD wVNTE = cf.readLEw();
	WORD wVVT = cf.readLEw();
	WORD wSTMTAB = cf.readLEw();
	WORD wSTMCUR = cf.readLEw();
	WORD wSTARP = cf.readLEw();

	WORD wCOR = wVNT - wLOMEM - 0x0E;

	wVNT -= wCOR;
	wVNTE -= wCOR;
	wVVT -= wCOR;
	wSTMTAB -= wCOR;
	wSTMCUR -= wCOR;
	wSTARP -= wCOR;

	WORD wVNTL = wVNTE - wVNT + 1;
	WORD wVVTE = wSTMTAB - 1;
	WORD wVVTL = wVVTE - wVVT + 1;
	WORD NV = wVVTL / 8;

	WORD wCodeLen = wSTMCUR - wSTMTAB;
	WORD wCodeLenCur = wSTARP - wSTMCUR;

	long lLenDiff = (long)wFileLen - (long)wSTARP;

	if ( g_bExtOut )
	{
		fprintf( g_fout, "Constants & pointers:\n" );
		fprintf( g_fout, "Start of Name Table      (VNT)   : %04X\n", wVNT );
		fprintf( g_fout, "End of Name Table        (VNTE)  : %04X\n", wVNTE );
		fprintf( g_fout, "Lenght of Name Table     (VNTL)  : %04X\n", wVNTL );

		fprintf( g_fout, "Start of Variable Table  (VVT)   : %04X\n", wVVT );
		fprintf( g_fout, "End of Variable Table    (VVTE)  : %04X\n", wVVTE );
		fprintf( g_fout, "Length of Variable Table (VVTL)  : %04X\n", wVVTL );

		fprintf( g_fout, "Number of Variables      (NV)    : %04X\n", NV );

		fprintf( g_fout, "Start of Code            (STMTAB): %04X\n", wSTMTAB );
		fprintf( g_fout, "Length of Code                   : %04X\n", wCodeLen );
		fprintf( g_fout, "Current command          (STMCUR): %04X\n", wSTMCUR );
		fprintf( g_fout, "Length of current command        : %04X\n", wCodeLenCur );
		fprintf( g_fout, "First byte after program (STARP) : %04X\n", wSTARP );
		fprintf( g_fout, "Length of file                   : %04X\n", wFileLen );
		fprintf( g_fout, "File len difference              : %08lX\n", (DWORD)lLenDiff );
		fprintf( g_fout, "\n" );
	}


	if ( NV * 8 != wVVTL )
	{
		fprintf( stderr, "Variable Table Length Mismatch!\n" );
		return 1;
	}

	if (lLenDiff <0 )
	{
		fprintf( stderr, "File Length Incorrect!\n" );
		return 1;
	}

	if ( !Vars_Load( &cf, wVNT, wVNTL, wVVT, NV ) )
		return 1;

	if ( g_bExtOut )
	{
		fprintf( g_fout, "Main code starts here:\n" );
	}

	GenCode( &cf, wSTMTAB, wSTMCUR );

	if ( g_bExtOut )
	{
		fprintf( g_fout, "\n" );

		fprintf( g_fout, "Immediate code starts here:\n" );
		GenCode( &cf, wSTMCUR, wSTARP );
		fprintf( g_fout, "\n" );
	}

	cf.Close();

	if ( g_bTB_Code & g_bExtOut )
		fprintf( g_fout, "\nTurbo BASIC file format.\n" );

	if ( g_bHasErrors )
		fprintf( stderr, "File may contain some errors!\n" );

	if ( g_fout != stdout )
		fclose( g_fout );

	fprintf( stderr, "Done!\n" );

	Vars_Destroy();

	return 0;
}

BOOL GenCode( CFile* pcf, WORD wStart, WORD wEnd )
{
	pcf->Seek( wStart, SEEK_SET );

	while( pcf->Tell() < wEnd )
	{
		if ( !GenLineCode( pcf, wEnd ) )
		{
			g_bHasErrors = TRUE;
			return FALSE;
		}

		if ( g_bAtariOut )
			fprintf( g_fout, "\x9B" );
		else
			fprintf( g_fout, "\n" );
	}

	return TRUE;
}

BOOL GenLineCode( CFile* pcf, WORD wEnd )
{
	if ( g_bVerbose )
		fprintf( g_fout, "[O:%08lX]", pcf->Tell() );

	WORD wLineNum = pcf->readLEw();

	fprintf( g_fout, "%d ", wLineNum );

	BYTE btLEnd = pcf->readb();

	WORD wLineStart = pcf->Tell() - 3;
	WORD wLineEnd = wLineStart + btLEnd;

	if ( g_bVerbose )
	{
		fprintf( g_fout, "[S:%04X E:%04X]", wLineStart, wLineEnd );
	}

	if ( ( wLineStart > wEnd ) || ( wLineEnd > wEnd ) )
	{
		fprintf( g_fout, "***Line size mismatch. (%04X-%04X)\n", wLineEnd, wEnd );
		return FALSE;
	}

	while( pcf->Tell() < wLineEnd )
	{
		if ( !GenCmdCode( pcf, wLineStart, wLineEnd ) )
			return FALSE;
	}

	return TRUE;
}

BOOL GenCmdCode( CFile* pcf, WORD wLineStart, WORD wLineEnd )
{
	BYTE btTEnd = pcf->readb();

	WORD wTokStart = pcf->Tell();
	WORD wTokEnd = wLineStart + btTEnd;

	if ( g_bVerbose )
	{
		fprintf( g_fout, "[C:%04X E:%04X]", wTokStart, wTokEnd );
	}

	if ( wTokStart >= wTokEnd )
	{
		fprintf( g_fout, "***Command size problem!\n" );
		return FALSE;
	}

	if ( wTokEnd > wLineEnd )
	{
		fprintf( g_fout, "***Command size mismatch. (%04X-%04X)\n", wTokEnd, wLineEnd );
		return FALSE;
	}

	BYTE btTok = pcf->readb();

	char szTok[ 50 ];

	if ( btTok < CMD_NUM ) 
	{
		strcpy( szTok, aBasicCmds[ btTok ] );

		if ( btTok >= CMD_TB_FIRST )
			g_bTB_Code = TRUE;
	}
	else
	{
		sprintf( szTok, "COM%02X", btTok );
		fprintf( stderr, "Unknown command %02X!!!\n", btTok );
		g_bHasErrors = TRUE;
	}

	if ( *szTok )
		fprintf( g_fout, "%s ", szTok );

	//REM & DATA
	if ( btTok < 2 )
	{
		BYTE c;

		for( ;; )
		{
			c = pcf->readb();

			if ( c == 0x9B )
				break;

			fprintf( g_fout, "%c", c );

		} 

		return TRUE;

	}

	while( pcf->Tell() < wTokEnd )
	{
		if ( !GenOpsCode( pcf, wTokEnd ) )
			return FALSE;
	}

	return TRUE;
}

BOOL GenOpsCode( CFile* pcf, WORD wTokEnd )
{
	BYTE btTok = pcf->readb();

	if ( btTok == OP_EOL )
	{
		//new line
		return TRUE;
	}

	if ( btTok == OP_NCONST )
	{
		//bcd num

		fprintf( g_fout, "%.10g", ReadAtariBCD( pcf ) );

		return TRUE;
	}

	if ( btTok == OP_NHCONST )
	{
		//hex num TBASIC
		g_bTB_Code = TRUE;

		fprintf( g_fout, "$%X", (unsigned short)ReadAtariBCD( pcf ) );
						
		return TRUE;
	}

	if ( btTok == OP_SCONST )
	{
		//string
		BYTE c;

		BYTE btLen = pcf->readb();

		fprintf( g_fout, "\"" );
		while( btLen-- )
		{
			c = pcf->readb();

			if ( c == 0x9B )
			{
				break;
			}

			if ( c == '\"' )
			{
				fprintf ( g_fout, "\"\"" );
			}
			else
				fprintf( g_fout, "%c", c );

		} 
		fprintf( g_fout, "\"" );
		return TRUE;
	}

	if ( btTok & 0x80 )
	{
		char szPom[ 255 ];

		Var_Get( szPom, btTok & 0x7F );
		fprintf( g_fout, szPom );

		return TRUE;
	}

	//func or op
	if ( ( btTok < OP_FIRST ) || ( btTok - OP_FIRST + 1 > (BYTE)OPS_NUM ) )
	{
		fprintf( g_fout, "UNKOP%02X", btTok );
		fprintf( stderr, "Unknown operand %02X!!!\n", btTok );
		g_bHasErrors = TRUE;
	}
	else
	{
		fprintf( g_fout, "%s", aBasicOps[ btTok - OP_FIRST ] );

		if ( btTok >= OP_TB_FIRST )
		{
			g_bTB_Code = TRUE;
		}
	}

	return TRUE;

}

//code for reading Atari BCD format
double ReadAtariBCD( CFile* pcf )
{
	double dRes = 0;

	//reads exponent
	BYTE btExp = pcf->readb();

	int iExp;

	if ( !btExp )
	{
		//if Exp==0, we're skipping it! (silently!)
		iExp = 0;
		pcf->skip( 5 );
	}
	else
	{
		//compute exponent
		iExp = ( btExp - 68 ) * 2;

		int i = 5;

		//read 5 pairs of numbers and combine 'em
		while ( i-- )
		{
			BYTE btPom = pcf->readb();

			BYTE btNum = ( btPom >> 4 ) * 10 + ( btPom &0xf );

			dRes *= 100;
			dRes += btNum;
		}

	}

	dRes *= pow( 10, iExp );

	return dRes;
}

BOOL Vars_Load( CFile* pcf, WORD wVNT, WORD wVNTL, WORD wVVT, WORD NV )
{
	char szVarTemp[ 256 ];
	char szVarGen[ 256 ];

	char* pStr = NULL;

	BOOL bTog = FALSE;

	pcf->Seek( wVNT, SEEK_SET );

	for( int i = 0; i < wVNTL; i++ )
	{
		if ( !pStr )
			pStr = szVarTemp;

		char c = pcf->readb();

		if ( c & 0x80 )
		{
			c &= 0x7F;
			bTog = TRUE;
		}

		*( pStr++ ) = c;
		*pStr = '\0';

		if ( pStr - szVarTemp > 254 )
		{
			*szVarTemp = '\0';
			bTog = TRUE;
		}

		if ( bTog )
		{
			int iVar = caBasicVars.GetSize();

			sprintf( szVarGen, "VAR%d", iVar );

			pStr = szVarTemp;

			int iLength = strlen( szVarTemp );

			if ( iLength )
			{
				for( int i = 0; i < iLength; i++ )
				{
					if ( !isprint( *( pStr++ ) ) )
					{
						strcpy( szVarTemp, szVarGen );
						break;
					}
				}

				if ( iVar )
				{
					for( int i = 0; i < iVar; i++ )
					{
						if( !strcmp( (char*)caBasicVars.GetAt( i ), szVarTemp ) )
						{
							fprintf( g_fout, "Dupped varname: %s\n", szVarTemp );
							strcpy( szVarTemp, szVarGen );
							break;
						}
					}
				}

			}
			else
			{
				strcpy( szVarTemp, szVarGen );
			}

			pStr = new char [ strlen( szVarTemp ) + 1 ];

			if ( pStr )
			{
				strcpy( pStr, szVarTemp );
				caBasicVars.Add( pStr );
			}
			else
			{
				printf( "Not enough memory!\n" );
				return FALSE;
			}

			bTog = FALSE;
			pStr = NULL;
		}
	}

	if ( g_bExtOut )
	{
		fprintf( g_fout, "Variable table:\n" );
	}

	pcf->Seek( wVVT, SEEK_SET );

	for( int i = 0 ; i < NV; i++ )
	{
		BYTE btType = pcf->readb();
		BYTE btNumber = pcf->readb();

		char* szType;

		char szText[ 50 ];

		switch( btType )
		{
			case VART_SCALAR:
			{
				szType = "SCALAR ";
				sprintf( szText, "%.10g", ReadAtariBCD( pcf ) );
				break;
			}

			case VART_ARRAYU:
			{
				szType = "ARRAYu ";
			  	char* szP = szText;
				szP += sprintf( szP, "SPoff: %04X ", pcf->readLEw() );
				szP += sprintf( szP, "Dim1: %d ", pcf->readLEw() );
				szP += sprintf( szP, "Dim2: %d ", pcf->readLEw() );
				break;
			}

			case VART_ARRAY:
			{
				szType = "ARRAY  ";
			  	char* szP = szText;
				szP += sprintf( szP, "SPoff: %04X ", pcf->readLEw() );
				szP += sprintf( szP, "Dim1: %d ", pcf->readLEw() );
				szP += sprintf( szP, "Dim2: %d ", pcf->readLEw() );
				break;
			}

			case VART_STRINGU:
			{
				szType = "STRINGu";
			  	char* szP = szText;
				szP += sprintf( szP, "SPoff: %04X ", pcf->readLEw() );
				szP += sprintf( szP, "Len: %d ", pcf->readLEw() );
				szP += sprintf( szP, "Dim: %d ", pcf->readLEw() );
				break;
			}

			case VART_STRING:
			{
				szType = "STRING ";
			  	char* szP = szText;
				szP += sprintf( szP, "SPoff: %04X ", pcf->readLEw() );
				szP += sprintf( szP, "Len: %d ", pcf->readLEw() );
				szP += sprintf( szP, "Dim: %d ", pcf->readLEw() );
				break;
			}

			case VART_PROC:
			{
				szType = "PROC   ";
			  	char* szP = szText;
				for( int i = 0 ; i < 6; i++ )
				{
					szP += sprintf( szP, "%02X ", pcf->readb() );
				}
				break;
			}

			case VART_LABEL:
			{
				szType = "LABEL  ";
			  	char* szP = szText;
				for( int i = 0 ; i < 6; i++ )
				{
					szP += sprintf( szP, "%02X ", pcf->readb() );
				}
				break;
			}

			default:
			{
				szType = "UNKNOWN";
			  	char* szP = szText;
				for( int i = 0 ; i < 6; i++ )
				{
					szP += sprintf( szP, "%02X ", pcf->readb() );
				}
				g_bHasErrors = TRUE;
			}
		}

		char szName[ 255 ];
		Var_Get( szName, btNumber );

		char cLastChar = szName[ strlen( szName ) - 1 ];

		switch( btType )
		{
			case VART_STRING:
			case VART_STRINGU:
			{
				if ( cLastChar != '$' )
				{
					strcat( szName, "$" );
					char* szStr = new char [ strlen( szName ) + 1 ];
					strcpy( szStr, szName );
					char* szOld = (char*) caBasicVars.GetAt( btNumber );

					if ( szOld )
						delete [] szOld;

					caBasicVars.SetAt( btNumber, szStr );

				}

				break;
			}

			case VART_ARRAY:
			case VART_ARRAYU:
			{
				if ( cLastChar == '(' )
				{
					szName[ strlen( szName ) - 1 ] = '\0';
					char* szStr = new char [ strlen( szName ) + 1 ];
					strcpy( szStr, szName );
					char* szOld = (char*) caBasicVars.GetAt( btNumber );

					if ( szOld )
						delete [] szOld;

					caBasicVars.SetAt( btNumber, szStr );

				}

				break;
			}

		}

		if ( g_bExtOut )
		{
			fprintf( g_fout, "%04X %s (%02X) %02X: %s %s\n",
					i+1, 
					szType, 
					btType, 
					btNumber, 
					szText,
					szName );
		}

	}

	if ( g_bExtOut )
		fprintf( g_fout, "\n" );

	return TRUE;
}

void Var_Get( char* szRet, int iNum )
{
	char* szVar = (char*) caBasicVars.GetAt( iNum );

	if ( szVar )
		strcpy( szRet, szVar );
	else
		sprintf( szRet, "VAR%d", iNum );
}

void Vars_Destroy()
{
	for( int i = 0; i < caBasicVars.GetSize(); i++ )
	{
		if ( caBasicVars.GetAt( i ) )
			delete [] ( char* ) caBasicVars.GetAt( i );
	}
}

