/*	AD&D version 1 spell book generation,
 *        written May, 1992, by Ken Jenks, 
 *        kjenks@gothamcity.jsc.nasa.gov
 *
 *	This program generates spell books for AD&D 1 Magic-Users and
 *      Illusionists.
 *
 *	Compile and run with no command line options to see 'usage.'
 */

#include <ctype.h>
#include <math.h>
#include <stdio.h>

	/* Global array used to determine if char already has a spell */
static int already_in_book[9][30];

#define MU	0
#define ILSN	1

/* Note to future developers: If you change the spell_list table, you'll
 * need to change the initial_mu_spells table, too. */

static char *spell_list[2][9][30] = {{

	/* 0=MU Spell Book Table */
{"Affect Normal Fires",	/* 1 */
"Burning Hands",	/* 2 */
"Charm Person",	/* 3 */
"Comprehend Languages",	/* 4 */
"Dancing Lights",	/* 5 */
"Detect Magic",	/* 6 */
"Enlarge",	/* 7 */
"Erase",	/* 8 */
"Feather Fall",	/* 9 */
"Find Familiar",	/* 10 */
"Friends",	/* 11 */
"Hold Portal",	/* 12 */
"Identify",	/* 13 */
"Jump",	/* 14 */
"Light",	/* 15 */
"Magic Missile",	/* 16 */
"Mending",	/* 17 */
"Message",	/* 18 */
"Nystul's Magic Aura",	/* 19 */
"Protection from Evil",	/* 20 */
"Push",	/* 21 */
"Read Magic",	/* 22: Read Magic is a special case; all MU's have it */

#define ReadMagic 22

"Shield",	/* 23 */
"Shocking Grasp",	/* 24 */
"Sleep",	/* 25 */
"Spider Climb",	/* 26 */
"Tenser's Floating Disc",	/* 27 */
"Unseen Servant",	/* 28 */
"Ventriloquism",	/* 29 */
"Write"},	/* 30 */

/* MU Level 2 */
{"Audible Glamer",	/* 1 */
"Continual Light",	/* 2 */
"Darkness, 15' Radius",	/* 3 */
"Detect Evil",		/* 4 */
"Detect Invisibility",	/* 5 */
"ESP",	/* 6 */
"Fool's Gold",	/* 7 */
"Forget",	/* 8 */
"Invisibility",	/* 9 */
"Knock",	/* 10 */
"Leomund's Trap",	/* 11 */
"Levitate",	/* 12 */
"Locate Object",	/* 13 */
"Magic Mouth",	/* 14 */
"Mirror Image",	/* 15 */
"Pyrotechnics",	/* 16 */
"Ray of Enfeeblement",	/* 17 */
"Rope Trick",	/* 18 */
"Scare",	/* 19 */
"Shatter",	/* 20 */
"Stinking Cloud",	/* 21 */
"Strength",	/* 22 */
"Web",	/* 23 */
"Wizard Lock",	/* 24 */
"",	/* 25 */
"",	/* 26 */
"",	/* 27 */
"",	/* 28 */
"",	/* 29 */
""},	/* 30 */

/* MU Level 3 */
{"Blink",	/* 1 */
"Clairaudience",	/* 2 */
"Clairvoyance",	/* 3 */
"Dispel Magic",	/* 4 */
"Explosive Runes",	/* 5 */
"Feign Death",	/* 6 */
"Fireball",	/* 7 */
"Flame Arrow",	/* 8 */
"Fly",	/* 9 */
"Gust of Wind",	/* 10 */
"Haste",	/* 11 */
"Hold Person",	/* 12 */
"Infravision",	/* 13 */
"Invisibility 10' Radius",	/* 14 */
"Leomund's Tiny Hut",	/* 15 */
"Lightning Bolt",	/* 16 */
"Monster Summoning I",	/* 17 */
"Phantasmal Force",	/* 18 */
"Protection from Evil 10' Radius",	/* 19 */
"Protection from Normal Missiles",	/* 20 */
"Slow",	/* 21 */
"Suggestion",	/* 22 */
"Tongues",	/* 23 */
"Water Breathing",	/* 24 */
"",	/* 25 */
"",	/* 26 */
"",	/* 27 */
"",	/* 28 */
"",	/* 29 */
""},	/* 30 */

/* MU Level 4 */
{"Charm Monster",	/* 1 */
"Confusion",	/* 2 */
"Dig",	/* 3 */
"Dimension Door",	/* 4 */
"Enchanted Weapon",	/* 5 */
"Extension I",	/* 6 */
"Fear",	/* 7 */
"Fire Charm",	/* 8 */
"Fire Shield",	/* 9 */
"Fire Trap",	/* 10 */
"Fumble",	/* 11 */
"Hallucinatory Terrain",	/* 12 */
"Ice Storm",	/* 13 */
"Massmorph",	/* 14 */
"Minor Globe of Invulnerability",	/* 15 */
"Monster Summoning II",	/* 16 */
"Plant Growth",	/* 17 */
"Polymorph Other",	/* 18 */
"Polymorph Self",	/* 19 */
"Rary's Mnemonic Enhancer",	/* 20 */
"Remove Curse",	/* 21 */
"Wall of Fire",	/* 22 */
"Wall of Ice",	/* 23 */
"Wizard Eye",	/* 24 */
"",	/* 25 */
"",	/* 26 */
"",	/* 27 */
"",	/* 28 */
"",	/* 29 */
""},	/* 30 */

/* MU Level 5 */
{"Airy Water",	/* 1 */
"Animal Growth",	/* 2 */
"Animate Dead",	/* 3 */
"Bigby's Interposing Hand",	/* 4 */
"Cloudkill",	/* 5 */
"Conjure Elemental",	/* 6 */
"Cone of Cold",	/* 7 */
"Contact Other Plane",	/* 8 */
"Distance Distortion",	/* 9 */
"Extension II",	/* 10 */
"Feeblemind",	/* 11 */
"Hold Monster",	/* 12 */
"Leomund's Secret Chest",	/* 13 */
"Magic Jar",	/* 14 */
"Monster Summoning II",	/* 15 */
"Mordenkainen's Faithful Hound",	/* 16 */
"Passwall",	/* 17 */
"Stone Shape",	/* 18 */
"Telekinesis",	/* 19 */
"Teleport",	/* 20 */
"Transmute Rock to Mud",	/* 21 */
"Wall of Force",	/* 22 */
"Wall of Iron",	/* 23 */
"Wall of Stone",	/* 24 */
"",	/* 25 */
"",	/* 26 */
"",	/* 27 */
"",	/* 28 */
"",	/* 29 */
""},	/* 30 */

/* MU Level 6 */
{"Anti-Magic Shell",	/* 1 */
"Bigby's Forceful Hand",	/* 2 */
"Control Weather",	/* 3 */
"Death Spell",	/* 4 */
"Disintigrate",	/* 5 */
"Enchant an Item",	/* 6 */
"Extension III",	/* 7 */
"Geas",	/* 8 */
"Glassee",	/* 9 */
"Globe of Invulnerability",	/* 10 */
"Guards and Wards",	/* 11 */
"Invisible Stalker",	/* 12 */
"Legend Lore",	/* 13 */
"Lower Water",	/* 14 */
"Monster Summoning IV",	/* 15 */
"Move Earth",	/* 16 */
"Otiluke's Freezing Sphere",	/* 17 */
"Part Water",	/* 18 */
"Project Image",	/* 19 */
"Reincarate",	/* 20 */
"Repulsion",	/* 21 */
"Spiritwrack",	/* 22 */
"Stone to Flesh",	/* 23 */
"Tenser's Transformation",	/* 24 */
"",	/* 25 */
"",	/* 26 */
"",	/* 27 */
"",	/* 28 */
"",	/* 29 */
""},	/* 30 */

/* MU Level 7 */
{"Bigby's Grasing Hand",	/* 1 */
"Cacodemon",	/* 2 */
"Charm Plants",	/* 3 */
"Delayed Blast Fireball",	/* 4 */
"Drawmij's Instant Summons",	/* 5 */
"Duo-Dimension",	/* 6 */
"Limited Wish",	/* 7 */
"Mass Invisibility",	/* 8 */
"Monster Summoning V",	/* 9 */
"Mordenkainen's Sword",	/* 10 */
"Phase Door",	/* 11 */
"Power Word, Stun",	/* 12 */
"Reverse Gravity",	/* 13 */
"Similicrum",	/* 14 */
"Statue",	/* 15 */
"Vanish",	/* 16 */
"",	/* 17 */
"",	/* 18 */
"",	/* 19 */
"",	/* 20 */
"",	/* 21 */
"",	/* 22 */
"",	/* 23 */
"",	/* 24 */
"",	/* 25 */
"",	/* 26 */
"",	/* 27 */
"",	/* 28 */
"",	/* 29 */
""},	/* 30 */

/* MU Level 8 */
{"Antipathy/Sympathy",	/* 1 */
"Bigby's Clenched Fist",	/* 2 */
"Clone",	/* 3 */
"Glassteel",	/* 4 */
"Incendiary Cloud",	/* 5 */
"Mass Charm",	/* 6 */
"Maze",	/* 7 */
"Mind Blank",	/* 8 */
"Monster Summoning VI",	/* 9 */
"Otto's Irresistable Dance",	/* 10 */
"Permanency",	/* 11 */
"Polymorph Any Object",	/* 12 */
"Power Word, Blind",	/* 13 */
"Serten's Spell Immunity",	/* 14 */
"Symbol",	/* 15 */
"Trap The Soul",	/* 16 */
"",	/* 17 */
"",	/* 18 */
"",	/* 19 */
"",	/* 20 */
"",	/* 21 */
"",	/* 22 */
"",	/* 23 */
"",	/* 24 */
"",	/* 25 */
"",	/* 26 */
"",	/* 27 */
"",	/* 28 */
"",	/* 29 */
""},	/* 30 */

/* MU Level 9 */
{"Astral Spell",	/* 1 */
"Bigby's Crushing Hand",	/* 2 */
"Gate",	/* 3 */
"Imprisonment",	/* 4 */
"Meteor Swarm",	/* 5 */
"Monster Summoning VII",	/* 6 */
"Power Word, Kill",	/* 7 */
"Prismatic Sphere",	/* 8 */
"Shape Change",	/* 9 */
"Temporal Stasis",	/* 10 */
"Time Stop",	/* 11 */
"Wish",	/* 12 */
"",	/* 13 */
"",	/* 14 */
"",	/* 15 */
"",	/* 16 */
"",	/* 17 */
"",	/* 18 */
"",	/* 19 */
"",	/* 20 */
"",	/* 21 */
"",	/* 22 */
"",	/* 23 */
"",	/* 24 */
"",	/* 25 */
"",	/* 26 */
"",	/* 27 */
"",	/* 28 */
"",	/* 29 */
""},	/* 30 */

}, /* end MU Spell Book Table */

{
	/* 1=Illusionist Spell Book Table */

/* Illusionist Level 1 */
{"Audible Glamer",	/* 1 */
"Change Self",	/* 2 */
"Color Spray",	/* 3 */
"Dancing Lights",		/* 4 */
"Darkness",	/* 5 */
"Detect Illusion",	/* 6 */
"Detect Invisibility",	/* 7 */
"Gaze Reflection",	/* 8 */
"Hypnotism",	/* 9 */
"Light",	/* 10 */
"Phantasmal Force",	/* 11 */
"Wall of Fog",	/* 12 */
"",	/* 13 */
"",	/* 14 */
"",	/* 15 */
"",	/* 16 */
"",	/* 17 */
"",	/* 18 */
"",	/* 19 */
"",	/* 20 */
"",	/* 21 */
"",	/* 22 */
"",	/* 23 */
"",	/* 24 */
"",	/* 25 */
"",	/* 26 */
"",	/* 27 */
"",	/* 28 */
"",	/* 29 */
""},	/* 30 */

/* Illusionist Level 2 */
{"Blindness",	/* 1 */
"Blur",	/* 2 */
"Deafness",	/* 3 */
"Detect Magic",	/* 4 */
"Fog Cloud",	/* 5 */
"Hypnotic Pattern",	/* 6 */
"Improved Phantasmal Force",	/* 7 */
"Invisibility",	/* 8 */
"Magic Mouth",	/* 9 */
"Mirror Image",	/* 10 */
"Misdirection",	/* 11 */
"Ventriloquism",	/* 12 */
"",	/* 13 */
"",	/* 14 */
"",	/* 15 */
"",	/* 16 */
"",	/* 17 */
"",	/* 18 */
"",	/* 19 */
"",	/* 20 */
"",	/* 21 */
"",	/* 22 */
"",	/* 23 */
"",	/* 24 */
"",	/* 25 */
"",	/* 26 */
"",	/* 27 */
"",	/* 28 */
"",	/* 29 */
""},	/* 30 */

/* Illusionist Level 3 */
{"Continual Darkness",	/* 1 */
"Continual Light",	/* 2 */
"Dispel Illusion",	/* 3 */
"Fear",	/* 4 */
"Hallucinatory Terrain",	/* 5 */
"Illusionary Script",	/* 6 */
"Invisibility 10' Radius",	/* 7 */
"Non-detection",	/* 8 */
"Paralyzation",	/* 9 */
"Rope Trick",	/* 10 */
"Spectral Force",	/* 11 */
"Suggestion",	/* 12 */
"",	/* 13 */
"",	/* 14 */
"",	/* 15 */
"",	/* 16 */
"",	/* 17 */
"",	/* 18 */
"",	/* 19 */
"",	/* 20 */
"",	/* 21 */
"",	/* 22 */
"",	/* 23 */
"",	/* 24 */
"",	/* 25 */
"",	/* 26 */
"",	/* 27 */
"",	/* 28 */
"",	/* 29 */
""},	/* 30 */

/* Illusionist Level 4 */
{"Confusion",	/* 1 */
"Dispel Exhaustion",	/* 2 */
"Emotion",	/* 3 */
"Improved Invisibility",	/* 4 */
"Massmorph",	/* 5 */
"Minor Creation",	/* 6 */
"Phantasmal Killer",	/* 7 */
"Shadow Monsters",	/* 8 */
"",	/* 9 */
"",	/* 10 */
"",	/* 11 */
"",	/* 12 */
"",	/* 13 */
"",	/* 14 */
"",	/* 15 */
"",	/* 16 */
"",	/* 17 */
"",	/* 18 */
"",	/* 19 */
"",	/* 20 */
"",	/* 21 */
"",	/* 22 */
"",	/* 23 */
"",	/* 24 */
"",	/* 25 */
"",	/* 26 */
"",	/* 27 */
"",	/* 28 */
"",	/* 29 */
""},	/* 30 */

/* Illusionist Level 5 */
{"Chaos",	/* 1 */
"Demi-Shadow Monsters",	/* 2 */
"Major Creation",	/* 3 */
"Maze",	/* 4 */
"Projected Image",	/* 5 */
"Shadow Door",	/* 6 */
"Shadow Magic",	/* 7 */
"Summon Shadow",	/* 8 */
"",	/* 9 */
"",	/* 10 */
"",	/* 11 */
"",	/* 12 */
"",	/* 13 */
"",	/* 14 */
"",	/* 15 */
"",	/* 16 */
"",	/* 17 */
"",	/* 18 */
"",	/* 19 */
"",	/* 20 */
"",	/* 21 */
"",	/* 22 */
"",	/* 23 */
"",	/* 24 */
"",	/* 25 */
"",	/* 26 */
"",	/* 27 */
"",	/* 28 */
"",	/* 29 */
""},	/* 30 */

/* Illusionist Level 6 */
{"Conjure Animals",	/* 1 */
"Demi-Shadow Magic",	/* 2 */
"Mass Suggestion",	/* 3 */
"Permanent Illusion",	/* 4 */
"Programmed Illusion",	/* 5 */
"Shades",	/* 6 */
"True Sight",	/* 7 */
"Veil",	/* 8 */
"",	/* 9 */
"",	/* 10 */
"",	/* 11 */
"",	/* 12 */
"",	/* 13 */
"",	/* 14 */
"",	/* 15 */
"",	/* 16 */
"",	/* 17 */
"",	/* 18 */
"",	/* 19 */
"",	/* 20 */
"",	/* 21 */
"",	/* 22 */
"",	/* 23 */
"",	/* 24 */
"",	/* 25 */
"",	/* 26 */
"",	/* 27 */
"",	/* 28 */
"",	/* 29 */
""},	/* 30 */

/* Illusionist Level 7 */
{"Alter Reality",	/* 1 */
"Astral Spell",	/* 2 */
"Prismatic Spray",	/* 3 */
"Prismatic Wall",	/* 4 */
"Vision",	/* 5 */
"First Level Magic-User Spells",	/* 6 */
"",	/* 7 */
"",	/* 8 */
"",	/* 9 */
"",	/* 10 */
"",	/* 11 */
"",	/* 12 */
"",	/* 13 */
"",	/* 14 */
"",	/* 15 */
"",	/* 16 */
"",	/* 17 */
"",	/* 18 */
"",	/* 19 */
"",	/* 20 */
"",	/* 21 */
"",	/* 22 */
"",	/* 23 */
"",	/* 24 */
"",	/* 25 */
"",	/* 26 */
"",	/* 27 */
"",	/* 28 */
"",	/* 29 */
""},	/* 30 */

/* Illusionist Level 8 */
{"",	/* 1 */
"",	/* 2 */
"",	/* 3 */
"",	/* 4 */
"",	/* 5 */
"",	/* 6 */
"",	/* 7 */
"",	/* 8 */
"",	/* 9 */
"",	/* 10 */
"",	/* 11 */
"",	/* 12 */
"",	/* 13 */
"",	/* 14 */
"",	/* 15 */
"",	/* 16 */
"",	/* 17 */
"",	/* 18 */
"",	/* 19 */
"",	/* 20 */
"",	/* 21 */
"",	/* 22 */
"",	/* 23 */
"",	/* 24 */
"",	/* 25 */
"",	/* 26 */
"",	/* 27 */
"",	/* 28 */
"",	/* 29 */
""},	/* 30 */

/* Illusionist Level 9 */
{"",	/* 1 */
"",	/* 2 */
"",	/* 3 */
"",	/* 4 */
"",	/* 5 */
"",	/* 6 */
"",	/* 7 */
"",	/* 8 */
"",	/* 9 */
"",	/* 10 */
"",	/* 11 */
"",	/* 12 */
"",	/* 13 */
"",	/* 14 */
"",	/* 15 */
"",	/* 16 */
"",	/* 17 */
"",	/* 18 */
"",	/* 19 */
"",	/* 20 */
"",	/* 21 */
"",	/* 22 */
"",	/* 23 */
"",	/* 24 */
"",	/* 25 */
"",	/* 26 */
"",	/* 27 */
"",	/* 28 */
"",	/* 29 */
""},	/* 30 */

}  /* end Illusionist Spell List */
}; /* end Spell List */

	/* Classes have differing chances of finding spells for spell books */
static int chance_to_find_spells[2] = {50,40};

	/* chance to learn spells, indexed by INT-1 */
static int chance_to_learn_spells[25] = {
        0,0,0,0,0,0,0,0,35,45,45,45,55,55,65,65,75,85,95,96,97,98,99,100,100};
/* INT: 1,2,3,4,5,6,7,8, 9,10,11,12,13,14,15,16,17,18,19,20,21,22,23, 24, 25 */

	/* max number of spells per spell level, indexed by INT-1 */
static int max_spells[25] = {
        0,0,0,0,0,0,0,0,6, 7, 7, 7, 9, 9,11,11,14,18,99,99,99,99,99,99,99};
/* INT: 1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25 */

	/* min number of spells per spell level, indexed by INT-1 */
        /* THIS IS NOT USED IN THE CURRENT VERSION OF THE PROGRAM. */
static int min_spells[25] = {
        0,0,0,0,0,0,0,0,4, 5, 5, 5, 6, 6, 7, 7, 8, 9,10,11,12,13,15,16,17};
/* INT: 1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25 */

	/* maximum spell level, indexed by character level (18 max) */
static int max_spell_level[2][18] = {
/* Level      1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18+ */
/* MU    */ { 1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6, 6, 7, 7, 8, 8, 9}, 
/* ILSN  */ { 1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6, 6, 7, 7, 8, 8, 9}};

	/* initial MU spells: 0=offensive, 1=defensive, 2=misc */
static int initial_mu_spells[3][10] = {
/* Off   */ { 2, 3, 7,11,15,16,21,24,25,25}, /* Kludge: (choice)=Sleep */
/* Def   */ { 1, 5, 9,12,14,20,23,26,29,23}, /* Kludge: (choice)=Shield */
/* Misc  */ { 4, 6, 8,10,13,17,18,28,30, 2}};/* Kludge: (choice)=Det Magic */

/* Note to future developers: If you change the spell_list table, you'll
 * need to change the initial_mu_spells table, too. */

/* Kludge justification: These are (arguably) the best spells of the
 * categories.  It seems logical that characters would choose these. 
 * See table, page 39 DMG. */

/* Spell book rolling logic:
 *
 * A spell book starts off with an initial list of first-level spells.
 * For MU's (including multi-class and Rangers), this list is determined
 * by a table in the DMG (p. 39).  For Illusionists, this initial list is
 * determined by three random first-level spells (no duplication).  After
 * the initial list of spells is determined the spell caster receives a
 * new spell from his mentor every time he trains up to the next level.
 * In addition to this automatic spell gain (which does not require an
 * intelligence check), this program assumes that the spell caster has a
 * chance to find extra spells while adventuring.  (This is checked each
 * level.)  If he finds a spell, he has a like chance to find another
 * spell.  This repeats until he either blows his chance for this
 * character level or he finds all the spells of this spell level.
 * 
 * Confused yet?  Just wait. */

/************************************************************************
* die_roll() - simple function to roll one die of the given size
************************************************************************/

int die_roll(size)
int size;
{
   return ((rand() % size) + 1);
}

/************************************************************************
* all_known() - yes or no: does he know all spells of this level?
*
************************************************************************/

int
all_known(clas, intel, sp_lvl)
int clas; /* 0=MU, 1=ILSN */
int intel;
int sp_lvl;
{
  int there_are_blanks,count,i;

  there_are_blanks = count = 0;

  /* Logic: If all spells are either known or beyond end of spell list...*/

  for(i=0; i<30; i++)
    if(strlen(spell_list[clas][sp_lvl-1][i]) == 0) /* If null spell name */
      break;
    else
      if(already_in_book[sp_lvl-1][i] == 0)
        there_are_blanks++;
      else
        if(already_in_book[sp_lvl-1][i] == 1)
          count++;
        /* Don't count missed spells (already_in_book == 2) */

  if((!there_are_blanks) || (count >= max_spells[intel-1]))
    return(1);
  else
    return(0);
}; /* end proc all_known */

/************************************************************************
* add_spell_proc() - This proc is invoked through add_a_spell macro.
*                    return(0) on success, (1) on failure to add to book.
*                    Can fail if spell already known or if spell beyond
*                    end of spell_list for this spell level.
************************************************************************/

#define add_a_spell(lz,sz) add_spell_proc(clas,intel,lz,sz,outfile)
#define MIN(a,b) ((a<b) ? a : b)

int
add_spell_proc(clas, intel, sp_lvl, sp, outfile)
int clas; /* 0=MU, 1=ILSN */
int intel, sp_lvl, sp;
FILE *outfile;
{
#ifdef DEBUG
  fprintf(stderr,"Attempting to add %s...\n",spell_list[clas][sp_lvl-1][sp-1]);
#endif
  if(all_known(clas, intel, sp_lvl))
    return(0); /* pretend we were successful */
  else
    if(already_in_book[sp_lvl-1][sp-1]) 
      return(1); /* failed to add */
    else
      if(strlen(spell_list[clas][sp_lvl-1][sp-1]) == 0)
        return(1); /* failed to add */
      else {

#ifdef DEBUG
        fprintf(stderr," Success!\n");
#endif
        already_in_book[sp_lvl-1][sp-1] = 1;
        return(0); /* Success! */
      }

}; /* end proc add_spell_proc */

/************************************************************************
* spell_book() - print out character's spell book
*
************************************************************************/

void
spell_book(intel, clas, lvl, outfile)
int intel, clas, lvl;
FILE *outfile;
{
  int i,j;
  int l;
  int find_another;
  int sp_lvl, sp;
  int first_this_level;

  printf("Spell book for an %s, Level: %i, Intelligence: %i\n",
    clas?"Illusionist":"MU", lvl,intel);
  for (i=0; i<9; i++)
    for (j=0; j<30; j++)
      already_in_book[i][j] = 0;

  /* Initial spell book calculation */

  switch(clas) {
    case MU:   add_a_spell( 1, ReadMagic);
               for(i=0; i<3; i++)
                 add_a_spell( 1, initial_mu_spells[i][die_roll(10)-1]);
                 /* Ignore return value; must always succeed. */
               break;

    case ILSN: for(i=0; i<3; i++)
                 while(add_a_spell( 1, die_roll(12)));
                 /* repeat until we get a spell we don't already know */
               break;
  } /* end switch */;

   /* And now, for each character level beyond first... */

  for (l=2; l<=lvl; l++) {

    /* Spells learned from mentor... */

#ifdef DEBUG
    fprintf(stderr,"Spell learned from mentor at level %i...\n",l);
#endif

    while(add_a_spell(max_spell_level[clas][MIN(l,18)-1],die_roll(30)))
      ;

    /*  Spells found while adventuring... */

#ifdef DEBUG
    fprintf(stderr,"Spells learned while adventuring at level %i...\n",l);
#endif

    find_another = 1;
    while(find_another)
      if(chance_to_find_spells[clas] < die_roll(100)) 
        find_another = 0;
      else {
        /* Determine level of spell found */
        sp_lvl = die_roll(max_spell_level[clas]
                                         [MIN(lvl,18)-1]);

#ifdef DEBUG
        fprintf(stderr,"Found a level %i spell: ",sp_lvl);
#endif

        if(all_known(clas, intel, sp_lvl)) {
          find_another = 0;
#ifdef DEBUG
          fprintf(stderr,"BUT we already know all of those!\n\n");
#endif
          }
        else {
          while(already_in_book[sp_lvl-1][(sp=die_roll(30))-1] || 
                (strlen(spell_list[clas][sp_lvl-1][sp-1]) == 0))
            ; /* (null statement) keep rolling 'til we get a good one */

#ifdef DEBUG
          fprintf(stderr,"%s\n",spell_list[clas][sp_lvl-1][sp-1]);
#endif

          if(chance_to_learn_spells[intel-1] < die_roll(100)) {
            already_in_book[sp_lvl-1][sp-1] = 2;

#ifdef DEBUG
            fprintf(stderr,",  but blew intelligence role.\n\n");
#endif

            }
            /* Note: This is a little screwy.  If he blows his Int roll,
             * we set the already_in_book array to indicate this char
             * can't know that spell.  If I stuck exactly by the PHB
             * rules, I'd have to go figure the MINIMUM spells/level,
             * based on Int.  For ease of programming, I'm ignoring this.
             * Ignoring the minimums can hurt high level Illusionists. */
          else 
            add_a_spell(sp_lvl,sp); 
            /* ignore return value, since we already know we have a winner */

        } /* end if all_known... */
      } /* end if chance_to_find... */

  } /* end for each level above first */

  /* Now, print spell book */

#ifdef DEBUG
  fprintf(stderr,"\nPrinting spell book...\n\n");
#endif

  for (sp_lvl=1; sp_lvl<= 9; sp_lvl++) {
    first_this_level = 1;

    for(sp=1; sp <= 30; sp++) {
      if(already_in_book[sp_lvl-1][sp-1]) {
        if(first_this_level)
          fprintf(outfile," %i: ",sp_lvl); /* no newline */
        else
          fprintf(outfile,"    "); /* no newline */
        first_this_level = 0;

        if(already_in_book[sp_lvl-1][sp-1] == 2)
          fprintf(outfile,"         Missed: "); /* no newline */

        fprintf(outfile," %s\n",spell_list[clas][sp_lvl-1][sp-1]);
        } /* end if already_in_book */
      } /* end for sp */
    } /* end for sp_lvl */
  fprintf(outfile,"\n");

}; /* end proc spell_book */

main(argc,argv)
int argc;
char **argv;
{
  int clas;
  int level, intel;

  if((argc != 4) && (argc != 5)) {
    fprintf(stderr,"Generates spell books for AD&D version 1 characters.\n");
    fprintf(stderr,"\n");
    fprintf(stderr," usage: %s class level intelligence [chance]\n",argv[0]);
    fprintf(stderr,"   where class is either 'mu' or 'il'\n");
    fprintf(stderr,"         level is numeric level of spell caster\n");
    fprintf(stderr,"         intelligence is numeric 3-25\n");
    fprintf(stderr,"         chance is [optional] %c chance to find spells\n",
             '%');
    fprintf(stderr,"           (default 50%c for MU, 40%c for Illusionists)\n",
             '%','%');
    fprintf(stderr,"\n  Examples: %s mu 10 18\n",argv[0]);
    fprintf(stderr,"            %s ill 12 17 65\n",argv[0]);
    exit(0);
  }

  if(*argv[1] == 'i') /* Illusionist? */
    clas = 1;
  else
    clas = 0;         /* Default = MU */

  level = atoi(argv[2]);
  intel = atoi(argv[3]);

  if(argc==5)
    chance_to_find_spells[clas] = atoi(argv[4]);

  /* seed random generator, may require system dependant command if
   * time(0) does not work, or might need to use a scanf to get seed */
  srand(time(0));

  spell_book(intel, clas, level, stdout);

} /* end main */
