/*	AD&D version 2 spell book generation,
 *        written May, 1992, by Ken Jenks, 
 *        kjenks@gothamcity.jsc.nasa.gov
 *
 *	This program generates spell books for AD&D 2 Wizards.
 *
 *	Compile and run with no command line options to see 'usage.'
 */

#include <ctype.h>
#include <math.h>
#include <stdio.h>

/* This is where individual DM's will want to change the order in
 * which first-level spells are granted.  Just move the spells up
 * and down in the list.  Note that the first two are special. */

static int prioritized_initial_spells[46] = { /* only first 45 used */

/* Read Magic */ 34,  /* Special: Always granted */
/* Detect Magic */ 13,  /* Special: Always granted */

/* These are the best first-level spells: */

/* Phantasmal Force */ 32,
/* Sleep */ 37,
/* Shield */ 35,
/* Charm Person */ 8,
/* Color Spray */ 10,
/* Magic Missile */ 27,

/* These are so-so first level spells: */

/* Enlarge */ 15,
/* Feather Fall */ 17,
/* Light */ 26,
/* Unseen Servant */ 42,
/* Burning Hands */ 5,
/* Hypnotism */ 23,
/* Protection from Evil */ 33,

/* These are pretty crummy first-level spells: */

/* Audible Glamer */ 4,
/* Comprehend Languages */ 11,
/* Find Familiar */ 18,
/* Tenser's Floating Disc */ 41,
/* Shocking Grasp */ 36,
/* Spider Climb */ 38,

/* These are the worst of the first-level spells, in no particular order: */

/* Affect Normal Fires */ 1,
/* Alarm */ 2,
/* Armor */ 3,
/* Cantrip */ 6,
/* Change Self */ 7,
/* Chill Touch */ 9,
/* Dancing Lights */ 12,
/* Detect Undead */ 14,
/* Erase */ 16,
/* Friends */ 19,
/* Gaze Reflection */ 20,
/* Grease */ 21,
/* Hold Portal */ 22,
/* Identify */ 24,
/* Jump */ 25,
/* Mending */ 28,
/* Message */ 29,
/* Mount */ 30,
/* Nystul's Magic Aura */ 31,
/* Spook */ 39,
/* Taunt */ 40,
/* Ventriloquism */ 43,
/* Wall of Fog */ 44,
/* Wizard Mark */ 45,

0};

	/* Global array used to determine if char already has a spell */
static int already_in_book[9][45];

static char *spell_list[9][45] = {

/* First Level Wizard Spells */
{"Affect Normal Fires",	/* 1 */
"Alarm",	/* 2 */
"Armor",	/* 3 */
"Audible Glamer",	/* 4 */
"Burning Hands",	/* 5 */
"Cantrip",	/* 6 */
"Change Self",	/* 7 */
"Charm Person",	/* 8 */
"Chill Touch",	/* 9 */
"Color Spray",	/* 10 */
"Comprehend Languages",	/* 11 */
"Dancing Lights",	/* 12 */
"Detect Magic",	/* 13 */
"Detect Undead",	/* 14 */
"Enlarge",	/* 15 */
"Erase",	/* 16 */
"Feather Fall",	/* 17 */
"Find Familiar",	/* 18 */
"Friends",	/* 19 */
"Gaze Reflection",	/* 20 */
"Grease",	/* 21 */
"Hold Portal",	/* 22 */
"Hypnotism",	/* 23 */
"Identify",	/* 24 */
"Jump",	/* 25 */
"Light",	/* 26 */
"Magic Missile",	/* 27 */
"Mending",	/* 28 */
"Message",	/* 29 */
"Mount",	/* 30 */
"Nystul's Magic Aura",	/* 31 */
"Phantasmal Force",	/* 32 */
"Protection from Evil",	/* 33 */
"Read Magic",	/* 34 */
"Shield",	/* 35 */
"Shocking Grasp",	/* 36 */
"Sleep",	/* 37 */
"Spider Climb",	/* 38 */
"Spook",	/* 39 */
"Taunt",	/* 40 */
"Tenser's Floating Disc",	/* 41 */
"Unseen Servant",	/* 42 */
"Ventriloquism",	/* 43 */
"Wall of Fog",	/* 44 */
"Wizard Mark"}, /* 45 */

/* Spell Level 2 */
{"Alter Self",	/* 1 */
"Bind",	/* 2 */
"Blindness",	/* 3 */
"Blur",	/* 4 */
"Continual Light",	/* 5 */
"Darkness, 15' Radius",	/* 6 */
"Deafness",	/* 7 */
"Deeppockets",	/* 8 */
"Detect Evil",	/* 9 */
"Detect Invisibility",	/* 10 */
"ESP",	/* 11 */
"Flaming Sphere",	/* 12 */
"Fog Cloud",	/* 13 */
"Fool's Gold",	/* 14 */
"Forget",	/* 15 */
"Glitterdust",	/* 16 */
"Hypnotic Pattern",	/* 17 */
"Improved Phantasmal Force",	/* 18 */
"Invisibility",	/* 19 */
"Irritation",	/* 20 */
"Knock",	/* 21 */
"Know Alignment",	/* 22 */
"Leomund's Trap",	/* 23 */
"Levitate",	/* 24 */
"Locate Object",	/* 25 */
"Magic Mouth",	/* 26 */
"Melf's Acid Arrow",	/* 27 */
"Mirror Image",	/* 28 */
"Misdirection",	/* 29 */
"Protection from Cantrips",	/* 30 */
"Pyrotechnics",	/* 31 */
"Ray of Enfeeblement",	/* 32 */
"Rope Trick",	/* 33 */
"Scare",	/* 34 */
"Shatter",	/* 35 */
"Spectral Hand",	/* 36 */
"Stinking Cloud",	/* 37 */
"Strength",	/* 38 */
"Summon Swarm",	/* 39 */
"Tasha's Unconrollable Hideous Laughter",	/* 40 */
"Web",	/* 41 */
"Whispering Wind",	/* 42 */
"Wizard Lock",	/* 43 */
"",	/* 44 */
""},	/* 45 */

/* Spell Level 3 */
{"Blink",
"Clairaudience",
"Clairvoyance",
"Delude",
"Dispel Magic",
"Explosive Runes",
"Feign Death",
"Fireball",
"Flame Arrow",
"Fly",
"Gust of Wind",
"Haste",
"Hold Person",
"Hold Undead",
"Illusionary Script",
"Infravision",
"Invisibility 10' Radius",
"Item",
"Leomund's Tiny Hut",
"Lightning Bolt",
"Melf's Minute Meteors",
"Monster Summoning I",
"Non-Detection",
"Phantom Steed",
"Protection from Evil 10' Radius",
"Protection from Normal Missiles",
"Secret Page",
"Sepia Snake Sigil",
"Slow",
"Spectral Force",
"Suggestion",
"Tongues",
"Vampiric Touch",
"Water Breathing",
"Wind Wall",
"Wraithform",
"",
"",
"",
"",
"",
"",
"",
"",
""},

/* Spell Level 4 */
{"Charm Monster",
"Confusion",
"Contagion",
"Detect Scrying",
"Dig",
"Dimension Door",
"Emotion",
"Enchanted Weapon",
"Enervation",
"Evard's Black Tentacles",
"Extension I",
"Fear",
"Fire Charm",
"Fire Shield",
"Fire Trap",
"Fumble",
"Hallucinatory Terrain",
"Ice Storm",
"Illusionary Wall",
"Improved Invisibility",
"Leomund's Secure Shelter",
"Magic Mirror",
"Massmorph",
"Minor Creation",
"Minor Globe of Invulnerability",
"Monster Summoning II",
"Otiluke's Resilient Sphere",
"Phantasmal Killer",
"Plant Growth",
"Polymorph Other",
"Polymorph Self",
"Rainbow Pattern",
"Rary's Mnemonic Enhancer",
"Remove Curse",
"Shadow Monsters",
"Shout",
"Solid Fog",
"Stoneskin",
"Vacancy",
"Wall of Fire",
"Wall of Ice",
"Wizard Eye",
"",
"",	/* 44 */
""},	/* 45 */

/* Spell Level 5 */
{"Advanced Illusion",
"Airy Water",
"Animal Growth",
"Animate Dead",
"Avoidance",
"Bigby's Interposing Hand",
"Chaos",
"Cloudkill",
"Cone of Cold",
"Conjure Elemental",
"Contact Other Plane",
"Demi-Shadow Monsters",
"Dismissal",
"Distance Distortion",
"Domination",
"Extension II",
"Fabricate",
"False Vision",
"Feeblemind",
"Hold Monster",
"Leomund's Lamentable Belaborment",
"Leomund's Secret Chest",
"Magic Jar",
"Major Creation",
"Monster Summoning III",
"Mordenkainen's Faithful Hound",
"Passwall",
"Seeming",
"Sending",
"Stone Shape",
"Summon Shadow",
"Telekinesis",
"Teleport",
"Transmute Rock to Mud",
"Wall of Force",
"Wall of Iron",
"Wall of Stone",
"",	/* 41 */
"",	/* 42 */
"",	/* 43 */
"",	/* 44 */
""},	/* 45 */

/* Spell Level 6 */
{"Anti-Magic Shell",
"Bigby's Forceful Hand",
"Chain Lightning",
"Conjure Animals",
"Contingency",
"Control Weather",
"Death Fog",
"Death Spell",
"Demi-Shadow Magic",
"Disintigrate",
"Enchant an Item",
"Ensnarement",
"Extension III",
"Eyebite",
"Geas",
"Glassee",
"Globe of Invulnerability",
"Guards and Wards",
"Invisible Stalker",
"Legend Lore",
"Lower Water",
"Mass Suggestion",
"Mirage Arcane",
"Mislead",
"Monster Summoning IV",
"Mordenkainen's Lucubration",
"Move Earth",
"Otiluke's Freezing Sphere",
"Part Water",
"Permanent Illusion",
"Programmed Illusion",
"Project Image",
"Reincaration",
"Repulsion",
"Shades",
"Stone to Flesh",
"Tenser's Transformation",
"Transmute Water to Dust",
"True Seeing",
"Veil",
"",	/* 41 */
"",	/* 42 */
"",	/* 43 */
"",	/* 44 */
""},	/* 45 */

/* Spell Level 7 */
{"Banishment",
"Bigby's Grasing Hand",
"Charm Plants",
"Control Undead",
"Delayed Blast Fireball",
"Drawmij's Instant Summons",
"Duo-Dimension",
"Finger of Death",
"Forcecage",
"Limited Wish",
"Mass Invisibility",
"Monster Summoning V",
"Mordenkainen's Magnigicant Mansion",
"Mordenkainen's Sword",
"Phase Door",
"Power Word, Stun",
"Prismatic Spray",
"Reverse Gravity",
"Sequester",
"Shadow Walk",
"Similicrum",
"Spell Turning",
"Statue",
"Teleport Without Error",
"Vanish",
"Vision",
"",	/* 27 */
"",	/* 28 */
"",	/* 29 */
"",	/* 30 */
"",	/* 31 */
"",	/* 32 */
"",	/* 33 */
"",	/* 34 */
"",	/* 35 */
"",	/* 36 */
"",	/* 37 */
"",	/* 38 */
"",	/* 39 */
"",	/* 40 */
"",	/* 41 */
"",	/* 42 */
"",	/* 43 */
"",	/* 44 */
""},	/* 45 */

/* Spell Level 8 */
{"Antipathy-Sympathy",
"Bigby's Clenched Fist",
"Binding",
"Clone",
"Demand",
"Glassteel",
"Incendiary Cloud",
"Mass Charm",
"Maze",
"Mind Blank",
"Monster Summoning VI",
"Otiluke's Telekinetic Sphere",
"Otto's Irresistable Dance",
"Permanency",
"Polymorph Any Object",
"Power Word, Blind",
"Prismatic Wall",
"Screen",
"Serten's Spell Immunity",
"Sink",
"Symbol",
"Trap The Soul",
"",	/* 23 */
"",	/* 24 */
"",	/* 25 */
"",	/* 26 */
"",	/* 27 */
"",	/* 28 */
"",	/* 29 */
"",	/* 30 */
"",	/* 31 */
"",	/* 32 */
"",	/* 33 */
"",	/* 34 */
"",	/* 35 */
"",	/* 36 */
"",	/* 37 */
"",	/* 38 */
"",	/* 39 */
"",	/* 40 */
"",	/* 41 */
"",	/* 42 */
"",	/* 43 */
"",	/* 44 */
""},	/* 45 */

/* Spell Level 9 */
{"Astral Spell",
"Bigby's Crushing Hand",
"Crystalbrittle",
"Energy Drain",
"Foresight",
"Gate",
"Imprisonment",
"Meteor Swarm",
"Monster Summoning VII",
"Mordenkainen's Disjunction",
"Power Word, Kill",
"Prismatic Sphere",
"Shape Change",
"Succor",
"Temporal Stasis",
"Time Stop",
"Weird",
"Wish",
"",	/* 19 */
"",	/* 20 */
"",	/* 21 */
"",	/* 22 */
"",	/* 23 */
"",	/* 24 */
"",	/* 25 */
"",	/* 26 */
"",	/* 27 */
"",	/* 28 */
"",	/* 29 */
"",	/* 30 */
"",	/* 31 */
"",	/* 32 */
"",	/* 33 */
"",	/* 34 */
"",	/* 35 */
"",	/* 36 */
"",	/* 37 */
"",	/* 38 */
"",	/* 39 */
"",	/* 40 */
"",	/* 41 */
"",	/* 42 */
"",	/* 43 */
"",	/* 44 */
""}	/* 45 */

}; /* end Spell List */

	/* chance to learn spells, indexed by INT-1 */
static int chance_to_learn_spells[25] = {
        0,0,0,0,0,0,0,0,35,45,45,45,55,55,65,65,75,85,95,96,97,98,99,100,100};
/* INT: 1,2,3,4,5,6,7,8, 9,10,11,12,13,14,15,16,17,18,19,20,21,22,23, 24, 25 */

	/* max number of spells per spell level, indexed by INT-1 */
static int max_spells[25] = {
        0,0,0,0,0,0,0,0,6, 7, 7, 7, 9, 9,11,11,14,18,99,99,99,99,99,99,99};
/* INT: 1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25 */

	/* maximum spell level, indexed by character level (18 max) */
static int max_spell_level[18] = {
/* Level      1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18+ */
              1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 5, 6, 6, 7, 7, 8, 8, 9}; 

/* Spell book rolling logic:
 *
 * A spell book starts off with an initial list of first-level spells.
 * This list is determined by the "Player Choice" method described
 * in the DMG2 (p. 41).  I have "prioritized" the spells into a list,
 * with the top of the list being most desirable, and the bottom of the 
 * list being the least desirable spells.  Lots of initial spells.  After
 * the initial list of spells is determined the spell caster receives a
 * new spell from his mentor every time he trains up to the next level.
 * In addition to this automatic spell gain (which does not require an
 * intelligence check), this program assumes that the spell caster has a
 * chance to find extra spells while adventuring.  (This is checked each
 * level.)  If he finds a spell, he has a like chance to find another
 * spell.  This repeats until he either blows his chance for this
 * character level or he finds all the spells of this spell level.
 * 
 * Confused yet?  Just wait. */

/************************************************************************
* die_roll() - simple function to roll one die of the given size
************************************************************************/

int die_roll(size)
int size;
{
   return ((rand() % size) + 1);
}

/************************************************************************
* all_known() - yes or no: does he know all spells of this level?
*
************************************************************************/

int
all_known(sp_lvl, intel)
int sp_lvl;
int intel;
{
  int there_are_blanks,count,i;

  there_are_blanks = count = 0;

  /* Logic: If all spells are either known or beyond end of spell list...*/

  for(i=0; i<45; i++)
    if(strlen(spell_list[sp_lvl-1][i]) == 0) /* If null spell name */
      break;
    else
      if(already_in_book[sp_lvl-1][i] == 0)
        there_are_blanks++;
      else
        if(already_in_book[sp_lvl-1][i] == 1)
          count++;
        /* Don't count missed spells (already_in_book == 2) */

  if((!there_are_blanks) || (count >= max_spells[intel-1]))
    return(1);
  else
    return(0);
}; /* end proc all_known */

/************************************************************************
* add_spell_proc() - This proc is invoked through add_a_spell macro.
*                    return(0) on success, (1) on failure to add to book.
*                    Can fail if spell already known or if spell beyond
*                    end of spell_list for this spell level.
************************************************************************/

#define add_a_spell(lz,sz) add_spell_proc(intel,lz,sz,outfile)
#define MIN(a,b) ((a<b) ? a : b)

int
add_spell_proc(intel, sp_lvl, sp, outfile)
int intel, sp_lvl, sp;
FILE *outfile;
{
  if(strlen(spell_list[sp_lvl-1][sp-1]) == 0)
    return(1); /* failed to add */
  else {
#ifdef DEBUG
    fprintf(stderr,"Trying to add %s...\n", spell_list[sp_lvl-1][sp-1]);
#endif

    if(all_known(sp_lvl,intel)) {
#ifdef DEBUG
      fprintf(stderr," All level %i spells known!\n",sp_lvl);
#endif
      return(0); /* pretend we were successful */
    }
    else
      if(already_in_book[sp_lvl-1][sp-1]) {

#ifdef DEBUG
        fprintf(stderr," Already in book.\n");
#endif
        return(1); /* failed to add */
      }
      else {
#ifdef DEBUG
        fprintf(stderr," Success!\n");
#endif
        already_in_book[sp_lvl-1][sp-1] = 1;
        return(0); /* Success! */
      }
  }
}; /* end proc add_spell_proc */

/************************************************************************
* spell_book() - print out character's spell book
*
************************************************************************/

void
spell_book(intel, lvl, chance_to_find_spells, outfile)
int intel, lvl, chance_to_find_spells;
FILE *outfile;
{
  int i,j;
  int l;
  int find_another;
  int sp_lvl, sp;
  int first_this_level;

  fprintf(outfile,"Spell book for a Wizard, Level: %i, Intelligence: %i\n",
           lvl,intel);

  for (i=0; i<9; i++)
    for (j=0; j<40; j++)
      already_in_book[i][j] = 0;

    /* Initial spell book calculation */

  for (i=0; i<45; i++)

    /* He's guaranteed to know Read Magic (i=0) and Detect Magic (i=1) */

    if((i<=1) || (chance_to_learn_spells[intel-1] >= die_roll(100)))
      add_a_spell( 1, prioritized_initial_spells[i]);
    else
     /* mark spell as unknowable */
      already_in_book[1-1][prioritized_initial_spells[i]-1] = 2;

   /* And now, for each character level beyond first... */

  for (l=2; l<=lvl; l++) {

    /* Spells learned from mentor... */

    while(add_a_spell(max_spell_level[MIN(l,18)-1],die_roll(45)))
      ;

    /*  Spells found while adventuring... */

    find_another = 1;
    while(find_another)
      if(chance_to_find_spells < die_roll(100)) 
        find_another = 0;
      else {
        /* Determine level of spell found */
        sp_lvl = die_roll(max_spell_level[MIN(lvl,18)-1]);
        if(all_known(sp_lvl,intel))
          find_another = 0;
        else {
          while(already_in_book[sp_lvl-1][(sp=die_roll(45))-1] || 
                (strlen(spell_list[sp_lvl-1][sp-1]) == 0))
            ; /* (null statement) keep rolling 'til we get a good one */

          if(chance_to_learn_spells[intel-1] < die_roll(100))
            already_in_book[sp_lvl-1][sp-1] = 2;

          else 
            add_a_spell(sp_lvl,sp); 
            /* ignore return value, since we already know we have a winner */

        } /* end if all_known... */
      } /* end if chance_to_find... */

  } /* end for each level above first */

  /* Now, print spell book */
#ifdef DEBUG
  for (sp_lvl=1; sp_lvl<= 9; sp_lvl++) {
    for(sp=1; sp <= 45; sp++)
      fprintf(stderr,"%i", already_in_book[sp_lvl-1][sp-1]);
    fprintf(stderr,"\n");
  }
  fprintf(stderr,"\n");
#endif

  for (sp_lvl=1; sp_lvl<= 9; sp_lvl++) {
    first_this_level = 1;
    for(sp=1; sp <= 45; sp++) {
      if(already_in_book[sp_lvl-1][sp-1]) {
        if(first_this_level)
          fprintf(outfile," %i: ",sp_lvl); /* no newline */
        else
          fprintf(outfile,"    "); /* no newline */
        first_this_level = 0;

        if(already_in_book[sp_lvl-1][sp-1] == 2)
          fprintf(outfile,"         Missed: "); /* no newline */

        fprintf(outfile," %s\n",spell_list[sp_lvl-1][sp-1]);
        } /* end if already_in_book */
      } /* end for sp */
    } /* end for sp_lvl */
  fprintf(outfile,"\n");

}; /* end proc spell_book */

main(argc,argv)
int argc;
char **argv;
{
  int level, intel, chance_to_find_spells;

  if((argc != 3) && (argc != 4)) {
    fprintf(stderr,"Generates spell books for AD&D version 2 Wizards.\n");
    fprintf(stderr,"\n");
    fprintf(stderr," usage: %s level intelligence [chance]\n",argv[0]);
    fprintf(stderr,"   where level is numeric level of Wizard\n");
    fprintf(stderr,"         intelligence is numeric 3-25\n");
    fprintf(stderr,"         chance is [optional] %c chance to find spells\n",
             '%');
    fprintf(stderr,"           (default 50%c)\n", '%');
    fprintf(stderr,"\n  Examples: %s 10 18\n",argv[0]);
    fprintf(stderr,"            %s 12 17 65\n",argv[0]);
    exit(0);
  }

  level = atoi(argv[1]);
  intel = atoi(argv[2]);

  if(argc==4)
    chance_to_find_spells = atoi(argv[3]);
  else
    chance_to_find_spells = 50;

  /* seed random generator, may require system dependant command if
   * time(0) does not work, or might need to use a scanf to get seed */
  srand(time(0));

  spell_book(intel, level, chance_to_find_spells, stdout);

} /* end main spells.phb2.c */
