/*
 * API to read/write sectors in a disk image.
 */

#ifndef IMAGE_H
#define IMAGE_H 1

#ifndef WINAPI
#include <windows.h>
#endif

/*
** magic numbers for ATR disks
*/
#define ATR_MAGIC1            0x96
#define ATR_MAGIC2            0x02

/*
** ATR header.
*/
typedef struct tag_ATRHeader
{
     unsigned char  nMagic1;
     unsigned char  nMagic2;
     unsigned char  nSecCountLo;
     unsigned char  nSecCountHi;
     unsigned char  nSecSizeLo;
     unsigned char  nSecSizeHi;
     unsigned char  nHugeSecCountLo;
     unsigned char  nHugeSecCountHi;
     unsigned char  nReserved[8];
} ATRHeader;

/*
** error codes written in nResult field.
*/
typedef enum {
     IMG_XFD             = 0,      /* XFD Image Detected */
     IMG_ATR             = 1,      /* ATR Image Detected */
     IMG_BAD_MAGIC       = -1,     /* File does not have a NICKATARI Signature */
     IMG_FILE_NOT_FOUND  = -2,     /* File can not be opened */
     IMG_OUT_OF_RANGE    = -3,     /* Sector out of range */
     IMG_DISK_ERROR      = -4,     /* Disk error occured */
     IMG_WRITE_PROTECT   = -5      /* Denied: Write protect error */
} ImgError;

/*
** structure to get information about a .XFD/.ATR image disk.
*/
typedef struct tag_ImgInfo {
     ImgError  nResult;            /* Operation status */
     WORD      wDensity;           /* Verified bytes-per-sector in image */
     WORD      wSectors;           /* Verified number of sectors in image */
     BYTE      cWriteProtect;      /* TRUE if the image is write-protected */
} ImgInfo;

typedef ImgInfo FAR * LPImgInfo;

/*
** structure to read/write a sector to a .XFD/.ATR image disk.
*/
typedef struct tag_ImgRWPacket {
     LPCSTR    lpszFileName;       /* Null terminated C string of a .XFD/.ATR image disk */
     WORD      wSectorNumber;      /* Sector number to read or write */
     BYTE      cSectorData[256];   /* Area for transfer of data, read/write */
     WORD      wSectorSize;        /* Size of sector to read or bytes written */
     ImgError  nResult;            /* Operation status */
} ImgRWPacket;

typedef ImgRWPacket FAR * LPImgRWPacket;

/*
** prototypes.
*/
void ImageGetInfo(LPCSTR pFilename, LPImgInfo pInfo);
void ImageReadSector(LPImgRWPacket pSector);
void ImageWriteSector(LPImgRWPacket pSector);
ImgError ImageError(HWND hWnd, ImgError err);

#endif
