/****************************************************************************
*
*  ATTENTION!!!
*
*  THIS FILE HAS BEEN MODIFIED!!! IT IS NOT PART OF THE OFFICAL
*  POV-RAY 2.2 DISTRIBUTION!!!
*
*  THIS FILE IS PART OF "FASTER THAN POV-RAY" (VERSION 1.1),
*  A SPED-UP VERSION OF POV-RAY 2.2. USE AT YOUR OWN RISK!!!!!!
*
*  New files: addon0.c, addon1.c, addon2.c, addon3.c, addon.h
*
*  The additional modules were written by Dieter Bayer.
*
*  Send comments, suggestions, bugs to:
*
*  dieter@cip.e-technik.uni-erlangen.de
*
*  If I have enough time I will try to fix any bugs.
*
*  All changed/added lines are enclosed in #ifdef DB_CODE ... #endif
*
*  The new/changed modules speed-up ray-tracing in two ways:
*
*   - Objects are projected onto the viewing plane a priori. Thus
*     the number of ray/object intersection tests is reduced for
*     primary rays.
*
*   - A light buffer is used for every spotlight. Each object is
*     projected a priori onto the six sides of a cube enclosing
*     the light source. Thus the number of ray/object intersection
*     tests is reduced for shadow rays.
*
*  The vista projection of qaudrics was taken from:
*
*    A. Hashimoto, T. Akimoto, K. Mase, and Y. Suenaga, "Vista
*    Ray-Tracing: High Speed Ray Tracing Using Perspective
*    Projection Image", New Advances in Computer Graphics,
*    Proceedings of CG International '89, R. A. Earnshaw,
*    B. Wyvill (Eds.), Springer, ...
*
*  The idea for the light buffer was taken from:
*
*    E. Haines and D. Greenberg, "The Light Buffer: A Shadow-
*    Testing Accelerator", IEEE CG&A, Vol. 6, No. 9, Sept. 1986, pp. 6-16
*
*****************************************************************************/

/****************************************************************************
*  addon1.c
*
*  This module was written by Dieter Bayer.
*
*  This module contains functions used only for the vista buffer.
*
*  01.03.1994 : Creation
*
*  29.04.1994 : Version 2.0
*
******************************************************************************/

#include <time.h>
#include "frame.h"
#include "vector.h"
#include "povproto.h"
#include "addon.h"

#ifdef DB_CODE

#define NUMBER_OF_TYPES      18
#define TYPE_BICUBIC_PATCH    0
#define TYPE_BLOB             1
#define TYPE_BOX              2
#define TYPE_CONE             3
#define TYPE_CSG_INTERSECTION 4
#define TYPE_CSG_MERGE        5
#define TYPE_CSG_UNION        6
#define TYPE_DISC             7
#define TYPE_ELLIPSOID        8
#define TYPE_HEIGHT_FIELD     9
#define TYPE_LIGHT_SOURCE    10
#define TYPE_PLANE           11
#define TYPE_POLY            12
#define TYPE_QUADRIC         13
#define TYPE_SMOOTH_TRIANGLE 14
#define TYPE_SPHERE          15
#define TYPE_TRIANGLE        16
#define TYPE_UNKNOWN         17

#define NUMBER_OF_FLAGS     9
#define FLAG_SUM            0
#define FLAG_INFINITE       1
#define FLAG_SCREEN_FILLING 2
#define FLAG_USED_IN_CSG    3
#define FLAG_INVISIBLE      4
#define FLAG_BOUNDED        5
#define FLAG_CLIPPED        6
#define FLAG_BOUND_OBJECT   7
#define FLAG_CLIP_OBJECT    8



/* external variables */

extern long Number_Of_Rays;
extern FRAME Frame;
extern int Trace_Level;
extern int Use_Slabs;
extern DBL Max_Trace_Level;
extern time_t tstart, tstop;
extern unsigned int Options;
extern OBJECT *Root_Object;

extern METHODS Bicubic_Patch_Methods;
extern METHODS Blob_Methods;
extern METHODS Box_Methods;
extern METHODS Cone_Methods;
extern METHODS Csg_Height_Field_Methods;
extern METHODS CSG_Intersection_Methods;
extern METHODS CSG_Merge_Methods;
extern METHODS CSG_Union_Methods;
extern METHODS Disc_Methods;
extern METHODS Ellipsoid_Methods;
extern METHODS Height_Field_Methods;
extern METHODS Light_Source_Methods;
extern METHODS Plane_Methods;
extern METHODS Poly_Methods;
extern METHODS Quadric_Methods;
extern METHODS Smooth_Triangle_Methods;
extern METHODS Sphere_Methods;
extern METHODS Triangle_Methods;



/* non-static variables */

VISTA_TREE_NODE *Root_Vista;

/* table containing 2**bit_number */

unsigned short int BIT_NUMBER[BUNCHING_FACTOR] = {0x01, 0x02, 0x04, 0x08};
unsigned short int BIT_NUMBER_INVERTED[BUNCHING_FACTOR] = {0xfe, 0xfd, 0xfb, 0xf7};



/* static variables */

static DBL distance;
static MATRIX WC2VC, WC2VCinv;
static VECTOR O, U, V, W;

/* planes for 3d-clipping */

static VECTOR VIEW_VX1 = {-0.8944271910, 0.0, -0.4472135955};
static VECTOR VIEW_VX2 = { 0.8944271910, 0.0, -0.4472135955};
static VECTOR VIEW_VY1 = {0.0, -0.8944271910, -0.4472135955};
static VECTOR VIEW_VY2 = {0.0,  0.8944271910, -0.4472135955};
static DBL VIEW_DX1 = 0.4472135955;
static DBL VIEW_DX2 = 0.4472135955;
static DBL VIEW_DY1 = 0.4472135955;
static DBL VIEW_DY2 = 0.4472135955;



/* static functions */

static void Intersect_Vista_Tree PARAMS((RAY *Ray, VISTA_TREE_NODE *Vista, int x, int y, INTERSECTION *Best, int *Found));

static void Project_square PARAMS((VISTA_PROJECT *Project, VECTOR P1, VECTOR P2, VECTOR P3, VECTOR P4, int *visible));
static void Project_triangle PARAMS((VISTA_PROJECT *Project, VECTOR P1, VECTOR P2, VECTOR P3, int *visible));

static void Project_Bbox_Box PARAMS((VISTA_PROJECT *Project, VECTOR *Points, int *visible));
static void Project_Bounds PARAMS((VISTA_PROJECT *Project, BBOX *Bounds, int *visible));

static void Get_Projection PARAMS((OBJECT *Object, VISTA_PROJECT *Project, int infinite));

static void Project_Object PARAMS((OBJECT *Object, VISTA_PROJECT *Project));

static void Project_Bicubic_Patch PARAMS((VISTA_PROJECT *Project, OBJECT *Object));
static void Project_Box PARAMS((VISTA_PROJECT *Project, OBJECT *Object, int *visible));
static void Project_Height_Field PARAMS((VISTA_PROJECT *Project, OBJECT *Object, int *visible));
static void Project_Sphere PARAMS((VISTA_PROJECT *Project, OBJECT *Object));
static void Project_Triangle PARAMS((VISTA_PROJECT *Project, OBJECT *Object, int *visible));
static void Project_Smooth_Triangle PARAMS((VISTA_PROJECT *Project, OBJECT *Object, int *visible));

static void Get_Ellipse_Projection PARAMS((VISTA_PROJECT *Project, DBL a20, DBL a02, DBL a11, DBL a10, DBL a01, DBL a00));

static void Transform_Point PARAMS((VECTOR *P));

static void Project_Bounding_Slab PARAMS((VISTA_PROJECT *Project, VISTA_TREE_NODE **Vista, OBJECT *Object, size_t *Mem_Vista_Buffer));




/****************************************************************************
   Trace a primary ray using the vista tree.
 ****************************************************************************/

void Trace_Primary_Ray (Ray, Colour, x, y)
RAY *Ray;
COLOUR *Colour;
int x, y;
{
  INTERSECTION Best;
  int Found;

  COOPERATE

  Number_Of_Rays++;

  if (Trace_Level > (int)Max_Trace_Level)
    return;

  Colour->Red = Colour->Green = Colour->Blue = 0.0;

  Found = FALSE;

  Best.Depth = BOUND_HUGE;

  /* What objects does this ray intersect? */

  if ((x >= Root_Vista->Project.x1) && (x <= Root_Vista->Project.x2) &&
      (y >= Root_Vista->Project.y1) && (y <= Root_Vista->Project.y2))
  {
    Intersect_Vista_Tree(Ray, Root_Vista, x, y, &Best, &Found);
  }

  if (Found)
  {
    Determine_Apparent_Colour (&Best, Colour, Ray);
  }
  else
  {
    if (Frame.Fog_Distance > 0.0)
    {
      *Colour = Frame.Fog_Colour;
    }
    else
    {
      *Colour = Frame.Background_Colour;
    }
  }
}




/****************************************************************************
   Descend the vista tree. 
   (I don't know what's faster? A recursion or a queue???)
 ****************************************************************************/
 
static void Intersect_Vista_Tree(Ray, Vista, x, y, Best, Found)
RAY *Ray;
VISTA_TREE_NODE *Vista;
int x, y;
INTERSECTION *Best;
int *Found;
{
  unsigned short int i;
  INTERSECTION New;
  VISTA_PROJECT Project;
  VISTA_TREE_LEAF *Leaf;
  
  for (i = 0; i < Vista->Entries; i++)
  {
    if (Vista->Leaf & BIT_NUMBER[i])
    {
      Leaf = (VISTA_TREE_LEAF *)Vista->Entry[i];
  
      Project = Leaf->Project;
  
      if ((y >= Project.y1) && (y <= Project.y2) &&
          (x >= Project.x1) && (x <= Project.x2))
      {
        if (Intersection(&New, Leaf->Object, Ray))
  	{
  	  if (New.Depth < Best->Depth)
  	  {
  	    *Best = New;
  	    *Found = TRUE;
  	  }
  	}
      }
    }
    else
    {
      Project = Vista->Entry[i]->Project;
  
      if ((y >= Project.y1) && (y <= Project.y2) &&
  	  (x >= Project.x1) && (x <= Project.x2))
      {
  	Intersect_Vista_Tree(Ray, Vista->Entry[i], x, y, Best, Found);
      }
    }
  }
}



/****************************************************************************
   Project a triangle onto the screen.
 ****************************************************************************/

static void Project_triangle (Project, P1, P2, P3, visible)
VISTA_PROJECT *Project;
VECTOR P1, P2, P3;
int *visible;
{
  VECTOR Points[MAX_CLIP_POINTS];
  int i, number;
  int x, y;

  if ((P1.z <= 0.0) && (P2.z <= 0.0) && (P3.z <= 0.0))
    return;

  Points[0] = P1;
  Points[1] = P2;
  Points[2] = P3;

  number = 3;

  Clip_Polygon(Points, &number, &VIEW_VX1, &VIEW_VX2, &VIEW_VY1, &VIEW_VY2,
				 VIEW_DX1,  VIEW_DX2,  VIEW_DY1,  VIEW_DY2);

  if (!number)
    return;

  for (i = 0; i < number; i++)
  {
    if (Points[i].z == -1.0)
    {
      Points[i].x = Points[i].y = 0.0;
    }
    else
    {
      Points[i].x = Points[i].x / (1.0 + Points[i].z);
      Points[i].y = Points[i].y / (1.0 + Points[i].z);
    }

    x = Frame.Screen_Width/2  + (int)(Frame.Screen_Width  * Points[i].x);
    y = Frame.Screen_Height/2 - (int)(Frame.Screen_Height * Points[i].y);

    if (x < Project->x1) Project->x1 = x;
    if (x > Project->x2) Project->x2 = x;
    if (y < Project->y1) Project->y1 = y;
    if (y > Project->y2) Project->y2 = y;
  }

  *visible = TRUE;
}



/****************************************************************************
   Project a square onto the screen.
 ****************************************************************************/

static void Project_square(Project, P1, P2, P3, P4, visible)
VISTA_PROJECT *Project;
VECTOR P1, P2, P3, P4;
int *visible;
{
  VECTOR Points[MAX_CLIP_POINTS];
  int i, number;
  int x, y;

  if ((P1.z <= 0.0) && (P2.z <= 0.0) && (P3.z <= 0.0) && (P4.z <= 0.0))
    return;

  Points[0] = P1;
  Points[1] = P2;
  Points[2] = P3;
  Points[3] = P4;

  number = 4;

  Clip_Polygon(Points, &number, &VIEW_VX1, &VIEW_VX2, &VIEW_VY1, &VIEW_VY2,
				 VIEW_DX1,  VIEW_DX2,  VIEW_DY1,  VIEW_DY2);

  if (!number)
    return;

  for (i = 0; i < number; i++)
  {
    if (Points[i].z == -1.0)
    {
      Points[i].x = Points[i].y = 0.0;
    }
    else
    {
      Points[i].x = Points[i].x / (1.0 + Points[i].z);
      Points[i].y = Points[i].y / (1.0 + Points[i].z);
    }

    x = Frame.Screen_Width/2  + (int)(Frame.Screen_Width  * Points[i].x);
    y = Frame.Screen_Height/2 - (int)(Frame.Screen_Height * Points[i].y);

    if (x < Project->x1) Project->x1 = x;
    if (x > Project->x2) Project->x2 = x;
    if (y < Project->y1) Project->y1 = y;
    if (y > Project->y2) Project->y2 = y;
  }

  *visible = TRUE;
}




/****************************************************************************
   Project a box onto the screen.
 ****************************************************************************/

static void Project_Bbox_Box(Project, Points, visible)
VISTA_PROJECT *Project;
VECTOR *Points;
int *visible;
{
  int vis;
  VISTA_PROJECT New;

  New.x1 = MAX_VB_ENTRY;
  New.x2 = MIN_VB_ENTRY;
  New.y1 = MAX_VB_ENTRY;
  New.y2 = MIN_VB_ENTRY;

  vis = FALSE;

  Project_square(&New, Points[0], Points[1], Points[3], Points[2], &vis);
  Project_square(&New, Points[4], Points[5], Points[7], Points[6], &vis);
  Project_square(&New, Points[0], Points[1], Points[5], Points[4], &vis);
  Project_square(&New, Points[2], Points[3], Points[7], Points[6], &vis);
  Project_square(&New, Points[1], Points[3], Points[7], Points[5], &vis);
  Project_square(&New, Points[0], Points[2], Points[6], Points[4], &vis);

  if (vis)
  {
    if (New.x1 > Project->x1) Project->x1 = New.x1;
    if (New.x2 < Project->x2) Project->x2 = New.x2;
    if (New.y1 > Project->y1) Project->y1 = New.y1;
    if (New.y2 < Project->y2) Project->y2 = New.y2;
    *visible = TRUE;
  }
}



/****************************************************************************
   Project a bounding box onto the screen.
 ****************************************************************************/

static void Project_Bounds(Project, Bounds, visible)
VISTA_PROJECT *Project;
BBOX *Bounds;
int *visible;
{
  int i;
  VECTOR P[8];

  for (i = 0; i<8; i++)
  {
    P[i].x = ((i & 1) ? Bounds->Max.x : Bounds->Min.x);
    P[i].y = ((i & 2) ? Bounds->Max.y : Bounds->Min.y);
    P[i].z = ((i & 4) ? Bounds->Max.z : Bounds->Min.z);

    Transform_Point (&P[i]);
  }

  Project_Bbox_Box(Project, &P[0], visible);
}



/****************************************************************************
 ****************************************************************************/

static void Project_Bicubic_Patch(Project, Object)
VISTA_PROJECT *Project;
OBJECT *Object;
{
  BICUBIC_PATCH *patch;
  DBL x, y, z, r, a20, a02, a11, a10, a01, a00;
  MATRIX A;

  patch = (BICUBIC_PATCH *)Object;

  /* set up 4x4 quadric matrix A */

  x = patch->Bounding_Sphere_Center.x;
  y = patch->Bounding_Sphere_Center.y;
  z = patch->Bounding_Sphere_Center.z;

  r = 1.0 / (patch->Bounding_Sphere_Radius * patch->Bounding_Sphere_Radius);

  A[0][0] = r;    A[0][1] = 0.0;  A[0][2] = 0.0;  A[0][3] = -x*r;
  A[1][0] = 0.0;  A[1][1] = r;    A[1][2] = 0.0;  A[1][3] = -y*r;
  A[2][0] = 0.0;  A[2][1] = 0.0;  A[2][2] = r;    A[2][3] = -z*r;
  A[3][0] = -x*r; A[3][1] = -y*r; A[3][2] = -z*r; A[3][3] = r*(x*x+y*y+z*z) - 1.0;

  Project_Vista(&A, NULL, &a20, &a02, &a11, &a10, &a01, &a00);

  /* get vista's bounding rectangle */

  Get_Ellipse_Projection(Project, a20, a02, a11, a10, a01, a00);
}



/****************************************************************************
 ****************************************************************************/

static void Project_Box(Project, Object, visible)
VISTA_PROJECT *Project;
OBJECT *Object;
int *visible;
{
  int i;
  VECTOR P[8];
  BOX *box;

  box = (BOX *)Object;

  for (i = 0; i<8; i++)
  {
    P[i] = box->bounds[0];

    if (i & 1) P[i].x = box->bounds[1].x;
    if (i & 2) P[i].y = box->bounds[1].y;
    if (i & 4) P[i].z = box->bounds[1].z;

    if (box->Trans != NULL)
      MTransPoint(&P[i], &P[i], box->Trans);

    Transform_Point (&P[i]);
  }

  Project_Bbox_Box(Project, &P[0], visible);
}



/****************************************************************************
 ****************************************************************************/

static void Project_Height_Field(Project, Object, visible)
VISTA_PROJECT *Project;
OBJECT *Object;
int *visible;
{
  int i;
  VECTOR P[8];
  HEIGHT_FIELD *hfield;

  hfield = (HEIGHT_FIELD *)Object;

  for (i = 0; i<8; i++)
  {
    P[i] = hfield->bounding_box->bounds[0];

    if (i & 1) P[i].x = hfield->bounding_box->bounds[1].x;
    if (i & 2) P[i].y = hfield->bounding_box->bounds[1].y;
    if (i & 4) P[i].z = hfield->bounding_box->bounds[1].z;

    if (hfield->Trans != NULL)
      MTransPoint(&P[i], &P[i], hfield->Trans);

    Transform_Point (&P[i]);
  }

  Project_Bbox_Box(Project, &P[0], visible);
}



/****************************************************************************
 ****************************************************************************/

static void Project_Triangle(Project, Object, visible)
VISTA_PROJECT *Project;
OBJECT *Object;
int *visible;
{
  int i, vis;
  VECTOR P[3];
  VISTA_PROJECT New;

  New.x1 = MAX_VB_ENTRY;
  New.x2 = MIN_VB_ENTRY;
  New.y1 = MAX_VB_ENTRY;
  New.y2 = MIN_VB_ENTRY;

  P[0] = ((TRIANGLE *)Object)->P1;
  P[1] = ((TRIANGLE *)Object)->P2;
  P[2] = ((TRIANGLE *)Object)->P3;

  for (i = 0; i < 3; i++)
  {
    Transform_Point(&P[i]);
  }

  vis = FALSE;

  Project_triangle(&New, P[0], P[1], P[2], &vis);

  if (vis)
  {
    if (New.x1 > Project->x1) Project->x1 = New.x1;
    if (New.x2 < Project->x2) Project->x2 = New.x2;
    if (New.y1 > Project->y1) Project->y1 = New.y1;
    if (New.y2 < Project->y2) Project->y2 = New.y2;
    *visible = TRUE;
  }
}



/****************************************************************************
 ****************************************************************************/

static void Project_Smooth_Triangle(Project, Object, visible)
VISTA_PROJECT *Project;
OBJECT *Object;
int *visible;
{
  int i, vis;
  VECTOR P[3];
  VISTA_PROJECT New;

  New.x1 = MAX_VB_ENTRY;
  New.x2 = MIN_VB_ENTRY;
  New.y1 = MAX_VB_ENTRY;
  New.y2 = MIN_VB_ENTRY;

  P[0] = ((SMOOTH_TRIANGLE *)Object)->P1;
  P[1] = ((SMOOTH_TRIANGLE *)Object)->P2;
  P[2] = ((SMOOTH_TRIANGLE *)Object)->P3;

  for (i = 0; i < 3; i++)
  {
    Transform_Point(&P[i]);
  }

  vis = FALSE;

  Project_triangle(&New, P[0], P[1], P[2], &vis);

  if (vis)
  {
    if (New.x1 > Project->x1) Project->x1 = New.x1;
    if (New.x2 < Project->x2) Project->x2 = New.x2;
    if (New.y1 > Project->y1) Project->y1 = New.y1;
    if (New.y2 < Project->y2) Project->y2 = New.y2;
    *visible = TRUE;
  }
}



/****************************************************************************
 ****************************************************************************/

static void Project_Sphere(Project, Object)
VISTA_PROJECT *Project;
OBJECT *Object;
{
  SPHERE *sphere;
  DBL x, y, z, r, a20, a02, a11, a10, a01, a00;
  MATRIX A;

  sphere = (SPHERE *)Object;

  /* set up 4x4 quadric matrix A0 */

  x = sphere->Center.x;
  y = sphere->Center.y;
  z = sphere->Center.z;

  r = 1.0/sphere->Radius_Squared;

  A[0][0] = r;    A[0][1] = 0.0;  A[0][2] = 0.0;  A[0][3] = -x*r;
  A[1][0] = 0.0;  A[1][1] = r;    A[1][2] = 0.0;  A[1][3] = -y*r;
  A[2][0] = 0.0;  A[2][1] = 0.0;  A[2][2] = r;    A[2][3] = -z*r;
  A[3][0] = -x*r; A[3][1] = -y*r; A[3][2] = -z*r; A[3][3] = r*(x*x+y*y+z*z) - 1.0;

  Project_Vista(&A, sphere->Trans, &a20, &a02, &a11, &a10, &a01, &a00);

  /* get vista's bounding rectangle */

  Get_Ellipse_Projection(Project, a20, a02, a11, a10, a01, a00);
}



/****************************************************************************
   Get the bounding rectangle of an ellipse.
   Quadratic curve: a20*x*x + a02*y*y + a11*x*y + a10*x + a01*y + a00 = 0
 ****************************************************************************/

static void Get_Ellipse_Projection(Project, a20, a02, a11, a10, a01, a00)
VISTA_PROJECT *Project;
DBL a20, a02, a11, a10, a01, a00;
{
  int x1i, x2i, y1i, y2i;
  DBL x1, y1, x2, y2;
  DBL a, b, c, d, k, k1, k2, k3, k4;

  x1 = y1 = -0.5;
  x2 = y2 = +0.5;

  k1 = a11/(-2.0*a20);
  k2 = a10/(-2.0*a20);

  k3 = a11/(-2.0*a02);
  k4 = a01/(-2.0*a02);

  a = a20 + a02*k3*k3 + a11*k3;

  b = a10 + 2.0*a02*k3*k4 + a01*k3 + a11*k4;

  c = a02*k4*k4 + a01*k4 + a00;

  d = b*b - 4.0*a*c;

  if (d > 0.0)
  {
    a = 2.0*a;

    d = sqrt(d);

    x1 = (-b+d)/a;
    x2 = (-b-d)/a;

    if (x1>x2)
    {
      k = x1; x1 = x2; x2 = k;
    }

    a = a02 + a20*k1*k1 + a11*k1;

    b = a01 + 2.0*a20*k1*k2 + a10*k1 + a11*k2;

    c = a20*k2*k2 + a10*k2 + a00;

    d = b*b - 4.0*a*c;

    if (d > 0.0)
    {
      a = 2.0*a;

      d = sqrt(d);

      y1 = (-b+d)/a;
      y2 = (-b-d)/a;

      if (y1>y2)
      {
	k = y1; y1 = y2; y2 = k;
      }
    }
  }

  if (x1 < -0.5) x1 = -0.5;
  if (x2 > +0.5) x2 = +0.5;
  if (y1 < -0.5) y1 = -0.5;
  if (y2 > +0.5) y2 = +0.5;

  x1i = Frame.Screen_Width/2 + (int)(Frame.Screen_Width * x1) - 1;
  x2i = Frame.Screen_Width/2 + (int)(Frame.Screen_Width * x2) + 1;

  y1i = Frame.Screen_Height/2 - (int)(Frame.Screen_Height * y2) - 1;
  y2i = Frame.Screen_Height/2 - (int)(Frame.Screen_Height * y1) + 1;

  if (x1i > Project->x1) Project->x1 = x1i;
  if (x2i < Project->x2) Project->x2 = x2i;

  if (y1i > Project->y1) Project->y1 = y1i;
  if (y2i < Project->y2) Project->y2 = y2i;
}



/****************************************************************************
   Get the vista of a quadric object.
 ****************************************************************************/

void Project_Vista(A0, Trans, a20, a02, a11, a10, a01, a00)
MATRIX *A0;
TRANSFORM *Trans;
DBL *a20, *a02, *a11, *a10, *a01, *a00;
{
  int i, j;
  MATRIX Tinv, A1, A2, B, Transposed_Matrix;
  DBL k1, k2, k3;

  /* apply object transformations */

  if (Trans == NULL)
  {
    for (i=0;i<4;i++)
    {
      for (j=0;j<4;j++)
      {
	A1[i][j] = (*A0)[i][j];
      }
    }
  }
  else
  {
    MTranspose(&Transposed_Matrix, &Trans->inverse);
    MTimes(&A1, &Trans->inverse, A0);
    MTimes(&A1, &A1, &Transposed_Matrix);
  }

  /* transform into viewing system */

  MTranspose(&Transposed_Matrix, &WC2VCinv);
  MTimes(&A2, &Transposed_Matrix, &A1);
  MTimes(&A2, &A2, &WC2VCinv);

  /* calculate quadrics vista */

  MIdentity(&Tinv);

  Tinv[3][2] = -1.0;

  MTranspose(&Transposed_Matrix, &Tinv);

  MTimes(&B, &Transposed_Matrix, &A2);
  MTimes(&B, &B, &Tinv);

  k1 = (B[0][2]+B[2][0])/(-2.0*B[2][2]);

  k2 = (B[1][2]+B[2][1])/(-2.0*B[2][2]);

  k3 = (B[2][3]+B[3][2])/(-2.0*B[2][2]);

  *a20 = B[0][0] + k1*(B[0][2]+B[2][0]) + k1*k1*B[2][2];

  *a02 = B[1][1] + k2*(B[1][2]+B[2][1]) + k2*k2*B[2][2];

  *a10 = B[0][3]+B[3][0] + k1*(B[2][3]+B[3][2]) + k3*(B[0][2]+B[2][0]) + 2.0*k1*k3*B[2][2];

  *a01 = B[1][3]+B[3][1] + k2*(B[2][3]+B[3][2]) + k3*(B[1][2]+B[2][1]) + 2.0*k2*k3*B[2][2];

  *a11 = B[0][1]+B[1][0] + k1*(B[1][2]+B[2][1]) + k2*(B[0][2]+B[2][0]) + 2.0*k1*k2*B[2][2];

  *a00 = B[3][3] + k3*(B[2][3]+B[3][2]) + k3*k3*B[2][2];
}



/****************************************************************************
   Transform a point from the world coordinate system to the viewer's
   coordinate system.
 ****************************************************************************/

static void Transform_Point(P)
VECTOR *P;
{
  DBL x,y,z;

  x = P->x - O.x;
  y = P->y - O.y;
  z = P->z - O.z;

  P->x = U.x * x + U.y * y + U.z * z;
  P->y = V.x * x + V.y * y + V.z * z;
  P->z = W.x * x + W.y * y + W.z * z;
}



/****************************************************************************
   Init the matrices and vectors used to transform a point from
   the world coordinate system to the viewer's coordinate system.
 ****************************************************************************/

void Init_View_Coordinates()
{
  DBL k, up_length, right_length;
  MATRIX A, B;

  U = Frame.Camera->Right;
  V = Frame.Camera->Up;
  W = Frame.Camera->Direction;

  VAdd (O, Frame.Camera->Location, Frame.Camera->Direction);

  VNormalize(U,U);
  VNormalize(V,V);
  VNormalize(W,W);

  VDot(k, U,V);
  if (fabs(k) > EPSILON)
    Fatal_Error("Camera: Right-vector not perpendicular to Up-vector.\n");

  VDot(k, U,W);
  if (fabs(k) > EPSILON)
    Fatal_Error("Camera: Right-vector not perpendicular to Direction-vector.\n");

  VDot(k, V,W);
  if (fabs(k) > EPSILON)
    Fatal_Error("Camera: Up-vector not perpendicular to Direction-vector.\n");

  VLength (distance, Frame.Camera->Direction);

  VLength (up_length, Frame.Camera->Up);
  VLength (right_length, Frame.Camera->Right);

  VScaleEq (U, 1.0/right_length);
  VScaleEq (V, 1.0/up_length);
  VScaleEq (W, 1.0/distance);

  A[0][0] = U.x; A[0][1] = U.y; A[0][2] = U.z; A[0][3] = 0.0;
  A[1][0] = V.x; A[1][1] = V.y; A[1][2] = V.z; A[1][3] = 0.0;
  A[2][0] = W.x; A[2][1] = W.y; A[2][2] = W.z; A[2][3] = 0.0;
  A[3][0] = 0.0; A[3][1] = 0.0; A[3][2] = 0.0; A[3][3] = 1.0;

  B[0][0] = 1.0; B[0][1] = 0.0; B[0][2] = 0.0; B[0][3] = -O.x;
  B[1][0] = 0.0; B[1][1] = 1.0; B[1][2] = 0.0; B[1][3] = -O.y;
  B[2][0] = 0.0; B[2][1] = 0.0; B[2][2] = 1.0; B[2][3] = -O.z;
  B[3][0] = 0.0; B[3][1] = 0.0; B[3][2] = 0.0; B[3][3] = 1.0;

  MTimes(&WC2VC, &A, &B);
  MInvers(&WC2VCinv, &WC2VC);
}



/****************************************************************************
   Get the projection of a single object, i. e. the smalles rectangle
   enclosing the object's image on the screen.
 ****************************************************************************/

static void Get_Projection(Object, Project, infinite)
OBJECT *Object;
VISTA_PROJECT *Project;
int infinite;
{
  int visible;
  METHODS *Methods;

  visible = FALSE;

  Methods = Object->Methods;

  /* if the object is infinite, there's no sense of projecting */
  
  if (!infinite)
  {
    if ((Methods == &Box_Methods) ||
	(Methods == &Smooth_Triangle_Methods) ||
	(Methods == &Triangle_Methods) ||
	(Methods == &Csg_Height_Field_Methods) ||
	(Methods == &Height_Field_Methods))
    {
      if (Methods == &Box_Methods)
	Project_Box(Project, Object, &visible);

      if (Methods == &Csg_Height_Field_Methods)
	Project_Height_Field(Project, Object, &visible);

      if (Methods == &Height_Field_Methods)
	Project_Height_Field(Project, Object, &visible);

      if (Methods == &Smooth_Triangle_Methods)
	Project_Smooth_Triangle(Project, Object, &visible);

      if (Methods == &Triangle_Methods)
	Project_Triangle(Project, Object, &visible);
    }
    else
    {
      Project_Bounds(Project, &Object->Bounds, &visible);

      if (Object->Methods == &Bicubic_Patch_Methods)
	Project_Bicubic_Patch(Project, Object);

      if (Object->Methods == &Ellipsoid_Methods)
	Project_Sphere(Project, Object);

      if (Object->Methods == &Sphere_Methods)
	Project_Sphere(Project, Object);
    }
  }
  
  /* we don't want invisible objects to become visible due to
     an oversized bouding object! */

  if (visible & (Object->Bound != NULL))
  {
    if (Object->Bound->Methods == &Box_Methods)
      Project_Box(Project, Object->Bound, &visible);

    if (Object->Bound->Methods == &Ellipsoid_Methods)
      Project_Sphere(Project, Object->Bound);

    if (Object->Bound->Methods == &Sphere_Methods)
      Project_Sphere(Project, Object->Bound);
  }

  if (visible)
  {
    if (Options & ANTIALIAS)
    {
      /* increase the rectangle to make sure that nothing will be missed.
         for anti-aliased images decrease the lower borders. */

      Project->x1 = max (0,                     Project->x1 - 2);
      Project->x2 = min (Frame.Screen_Width-1,  Project->x2 + 1);
      Project->y1 = max (-1,                    Project->y1 - 2);
      Project->y2 = min (Frame.Screen_Height-1, Project->y2 + 1);
    }
    else
    {
      /* increase the rectangle to make sure that nothing will be missed. */

      Project->x1 = max (0,                     Project->x1 - 1);
      Project->x2 = min (Frame.Screen_Width-1,  Project->x2 + 1);
      Project->y1 = max (0,                     Project->y1 - 1);
      Project->y2 = min (Frame.Screen_Height-1, Project->y2 + 1);
    }
  }
  else
  {
    if (!infinite)
    {
      /* object is invisible (the camera can't see it) */

      Project->x1 = Project->y1 = MAX_VB_ENTRY;
      Project->x2 = Project->y2 = MIN_VB_ENTRY;
    }
  }
}



/****************************************************************************
   Get the projection of an object.
 ****************************************************************************/

static void Project_Object(Object, Project)
OBJECT *Object;
VISTA_PROJECT *Project;
{
  int infinite;
  DBL Volume;

  /* init project fields, assuming the object is visible! */

  Project->x1 = Project->y1 = MIN_VB_ENTRY;
  Project->x2 = Project->y2 = MAX_VB_ENTRY;

  BOUNDS_VOLUME(Volume, Object->Bounds);

  infinite = (Volume > INFINITE_VOLUME);

  Get_Projection(Object, Project, infinite);

  Print_Point(POINT_MOD);
}



/****************************************************************************
   Project the bounding slabs onto the screen.
 ****************************************************************************/

static void Project_Bounding_Slab(Project, Vista, Object, Mem_Vista_Buffer)
VISTA_PROJECT *Project;
VISTA_TREE_NODE **Vista;
OBJECT *Object;
size_t *Mem_Vista_Buffer;
{
  unsigned short int i;
  size_t size;
  COMPOSITE *Comp;
  VISTA_PROJECT New, Temp;
  VISTA_TREE_LEAF *Leaf;

  if (Object->Type & BOUNDING_OBJECT)
  {
    /* allocate memory for new node in the vista tree (never freed!)  */

    size = sizeof(VISTA_TREE_NODE);

    if ((*Vista = (VISTA_TREE_NODE *)malloc(size)) == NULL)
      Fatal_MAError("Vista Tree Entry");

    *Mem_Vista_Buffer += size;

    Comp = (COMPOSITE *)Object;

    /* init new node */

    (*Vista)->Leaf = 0;

    (*Vista)->Entries = Comp->Entries;

    for (i = 0; i < BUNCHING_FACTOR; i++)
    {
      (*Vista)->Entry[i] = NULL;
    }

    New.x1 = New.y1 = MAX_VB_ENTRY;
    New.x2 = New.y2 = MIN_VB_ENTRY;

    /* project children and get the enclosing rectangle */

    for (i = 0; i < Comp->Entries; i++)
    {
      Project_Bounding_Slab(&Temp, &(*Vista)->Entry[i], Comp->Objects[i], Mem_Vista_Buffer);

      New.x1 = min(New.x1, Temp.x1);
      New.x2 = max(New.x2, Temp.x2);
      New.y1 = min(New.y1, Temp.y1);
      New.y2 = max(New.y2, Temp.y2);

      if (!(Comp->Objects[i]->Type & BOUNDING_OBJECT))
      {
	(*Vista)->Leaf |= BIT_NUMBER[i];
      }
    }

    *Project = New;

    (*Vista)->Project = New;
  }
  else
  {
    /* allocate memory for new leaf in the vista tree (never freed!)  */

    size = sizeof(VISTA_TREE_LEAF);

    if ((*Vista = (VISTA_TREE_NODE *)malloc(size)) == NULL)
      Fatal_MAError("Vista Tree Entry");

    *Mem_Vista_Buffer += size;

    /* init new leaf */

    Leaf = (VISTA_TREE_LEAF *)(*Vista);

    Leaf->Object = Object;

    /* get object's projection */

    Project_Object(Object, Project);

    Leaf->Project = *Project;
  }
}



/****************************************************************************
   Build the vista tree (2d representation of bounding slabs in image space)
 ****************************************************************************/

void Build_Vista_Tree(Mem_Vista_Buffer)
size_t *Mem_Vista_Buffer;
{
  VISTA_PROJECT Project;

  Root_Vista = NULL;

  if (Use_Slabs)
  {
    fprintf(stderr, "Building vista buffer");

    Begin_Point();

    Project_Bounding_Slab(&Project, &Root_Vista, Root_Object, Mem_Vista_Buffer);

    End_Point();
  }
}



#endif

