/****************************************************************************
*
*  ATTENTION!!!
*
*  THIS FILE HAS BEEN MODIFIED!!! IT IS NOT PART OF THE OFFICAL
*  POV-RAY 2.2 DISTRIBUTION!!!
*
*  THIS FILE IS PART OF "FASTER THAN POV-RAY" (VERSION 1.1),
*  A SPED-UP VERSION OF POV-RAY 2.2. USE AT YOUR OWN RISK!!!!!!
*
*  New files: addon0.c, addon1.c, addon2.c, addon3.c, addon.h
*
*  The additional modules were written by Dieter Bayer.
*
*  Send comments, suggestions, bugs to:
*
*  dieter@cip.e-technik.uni-erlangen.de
*
*  If I have enough time I will try to fix any bugs.
*
*  All changed/added lines are enclosed in #ifdef DB_CODE ... #endif
*
*  The new/changed modules speed-up ray-tracing in two ways:
*
*   - Objects are projected onto the viewing plane a priori. Thus
*     the number of ray/object intersection tests is reduced for
*     primary rays.
*
*   - A light buffer is used for every spotlight. Each object is
*     projected a priori onto the six sides of a cube enclosing
*     the light source. Thus the number of ray/object intersection
*     tests is reduced for shadow rays.
*
*  The vista projection of qaudrics was taken from:
*
*    A. Hashimoto, T. Akimoto, K. Mase, and Y. Suenaga, "Vista
*    Ray-Tracing: High Speed Ray Tracing Using Perspective
*    Projection Image", New Advances in Computer Graphics,
*    Proceedings of CG International '89, R. A. Earnshaw,
*    B. Wyvill (Eds.), Springer, ...
*
*  The idea for the light buffer was taken from:
*
*    E. Haines and D. Greenberg, "The Light Buffer: A Shadow-
*    Testing Accelerator", IEEE CG&A, Vol. 6, No. 9, Sept. 1986, pp. 6-16
*
*****************************************************************************/

/****************************************************************************
*  addon2.c
*
*  This module was written by Dieter Bayer.
*
*  This module contains functions used only for the light buffer.
*
*  01.03.1994 : Creation
*
*  29.04.1994 : Creation
*
******************************************************************************/

#include "frame.h"
#include "vector.h"
#include "povproto.h"
#include "addon.h"

#ifdef DB_CODE



/* external variables */

extern FRAME Frame;
extern unsigned long Quality_Flags;
extern unsigned int Options, Extended_Options;

extern METHODS Bicubic_Patch_Methods;
extern METHODS Blob_Methods;
extern METHODS Box_Methods;
extern METHODS Cone_Methods;
extern METHODS Csg_Height_Field_Methods;
extern METHODS CSG_Intersection_Methods;
extern METHODS CSG_Merge_Methods;
extern METHODS CSG_Union_Methods;
extern METHODS Disc_Methods;
extern METHODS Ellipsoid_Methods;
extern METHODS Height_Field_Methods;
extern METHODS Light_Source_Methods;
extern METHODS Plane_Methods;
extern METHODS Poly_Methods;
extern METHODS Quadric_Methods;
extern METHODS Smooth_Triangle_Methods;
extern METHODS Sphere_Methods;
extern METHODS Triangle_Methods;

extern OBJECT *Root_Object;

extern unsigned short int BIT_NUMBER[BUNCHING_FACTOR];


/* static variabls */

static VECTOR VIEW_VX1 = {-0.7071067812, 0.0, -0.7071067812};
static VECTOR VIEW_VX2 = { 0.7071067812, 0.0, -0.7071067812};
static VECTOR VIEW_VY1 = {0.0, -0.7071067812, -0.7071067812};
static VECTOR VIEW_VY2 = {0.0,  0.7071067812, -0.7071067812};
static DBL VIEW_DX1 = 0.0;
static DBL VIEW_DX2 = 0.0;
static DBL VIEW_DY1 = 0.0;
static DBL VIEW_DY2 = 0.0;

static unsigned short int BIT_MASK_1[4] = { 0x00, 0x01, 0x03, 0x07 };
static unsigned short int BIT_MASK_2[4] = { 0xfe, 0xfc, 0xf8, 0xf0 };



/* static functions */

static void Calc_Points PARAMS((OBJECT *Object, int *Number, VECTOR *Points, VECTOR *Origin));

static void Project_Square PARAMS((LIGHT_PROJECT *Project, VECTOR P1, VECTOR P2, VECTOR P3, VECTOR P4, int *visible));
static void Project_Triangle PARAMS((LIGHT_PROJECT *Project, VECTOR *Points, int *visible));
static void Project_Bbox_Box PARAMS((LIGHT_PROJECT *Project, VECTOR *Points, int *visible));
static void Project_Object PARAMS((LIGHT_PROJECT *Project, OBJECT *Object, int Axis, VECTOR *Origin));

static void Project_Bounding_Slab PARAMS((int Axis, VECTOR *Origin, LIGHT_PROJECT *Project, LIGHT_TREE_NODE **Entry, OBJECT *Object, size_t *Mem_Light_Buffer));

static void Remove_Invisible_Entries PARAMS((LIGHT_TREE_NODE **Entry, long *removed, size_t *Mem_Light_Buffer));


/****************************************************************************
 ****************************************************************************/

static void Calc_Points(Object, Number, Points, Origin)
OBJECT *Object;
int *Number;
VECTOR *Points, *Origin;
{
  int i;

  if ((Object->Methods == &Triangle_Methods) ||
      (Object->Methods == &Smooth_Triangle_Methods))
  {
    if (Object->Methods == &Triangle_Methods)
    {
      *Number = 3;
      Points[0] = ((TRIANGLE *)Object)->P1;
      Points[1] = ((TRIANGLE *)Object)->P2;
      Points[2] = ((TRIANGLE *)Object)->P3;
    }
    if (Object->Methods == &Smooth_Triangle_Methods)
    {
      *Number = 3;
      Points[0] = ((SMOOTH_TRIANGLE *)Object)->P1;
      Points[1] = ((SMOOTH_TRIANGLE *)Object)->P2;
      Points[2] = ((SMOOTH_TRIANGLE *)Object)->P3;
    }
  }
  else
  {
    *Number = 8;

    for (i = 0; i < *Number; i++)
    {
      Points[i].x = ((i & 1) ? Object->Bounds.Max.x : Object->Bounds.Min.x);
      Points[i].y = ((i & 2) ? Object->Bounds.Max.y : Object->Bounds.Min.y);
      Points[i].z = ((i & 4) ? Object->Bounds.Max.z : Object->Bounds.Min.z);
    }
  }

  for (i = 0; i < *Number; i++)
  {
    Points[i].x -= Origin->x;
    Points[i].y -= Origin->y;
    Points[i].z -= Origin->z;
  }
}



/****************************************************************************
 ****************************************************************************/

static void Project_Square(Project, P1, P2, P3, P4, visible)
LIGHT_PROJECT *Project;
VECTOR P1, P2, P3, P4;
int *visible;
{
  VECTOR Points[MAX_CLIP_POINTS];
  int i, anzahl;
  int x, y;

  if ((P1.z <= 0.0) && (P2.z <= 0.0) && (P3.z <= 0.0) && (P4.z <= 0.0))
    return;

  Points[0] = P1;
  Points[1] = P2;
  Points[2] = P3;
  Points[3] = P4;

  anzahl = 4;

  Clip_Polygon(Points, &anzahl, &VIEW_VX1, &VIEW_VX2, &VIEW_VY1, &VIEW_VY2,
				 VIEW_DX1,  VIEW_DX2,  VIEW_DY1,  VIEW_DY2);

  if (!anzahl)
    return;

  for (i = 0; i < anzahl; i++)
  {
    if (fabs(Points[i].z) < EPSILON)
    {
      Points[i].x = Points[i].y = 0.0;
    }
    else
    {
      Points[i].x = Points[i].x / Points[i].z;
      Points[i].y = Points[i].y / Points[i].z;
      if (fabs(Points[i].x) < EPSILON) Points[i].x = 0.0;
      if (fabs(Points[i].y) < EPSILON) Points[i].y = 0.0;
    }

    x = (int)(MAX_LB_ENTRY * Points[i].x);
    y = (int)(MAX_LB_ENTRY * Points[i].y);

    if (x < Project->x1) Project->x1 = x;
    if (x > Project->x2) Project->x2 = x;
    if (y < Project->y1) Project->y1 = y;
    if (y > Project->y2) Project->y2 = y;
  }

  *visible = TRUE;
}




/****************************************************************************
 ****************************************************************************/

static void Project_Triangle(Project, Old, visible)
LIGHT_PROJECT *Project;
VECTOR *Old;
int *visible;
{
  VECTOR Points[MAX_CLIP_POINTS];
  int i, anzahl;
  int x, y;

  if ((Old[0].z <= 0.0) && (Old[1].z <= 0.0) && (Old[2].z <= 0.0))
    return;

  Points[0] = Old[0];
  Points[1] = Old[1];
  Points[2] = Old[2];

  anzahl = 3;

  Clip_Polygon(Points, &anzahl, &VIEW_VX1, &VIEW_VX2, &VIEW_VY1, &VIEW_VY2,
				 VIEW_DX1,  VIEW_DX2,  VIEW_DY1,  VIEW_DY2);

  if (!anzahl)
    return;

  for (i = 0; i < anzahl; i++)
  {
    if (fabs(Points[i].z) < EPSILON)
    {
      Points[i].x = Points[i].y = 0.0;
    }
    else
    {
      Points[i].x = Points[i].x / Points[i].z;
      Points[i].y = Points[i].y / Points[i].z;
      if (fabs(Points[i].x) < EPSILON) Points[i].x = 0.0;
      if (fabs(Points[i].y) < EPSILON) Points[i].y = 0.0;
    }

    x = (int)(MAX_LB_ENTRY * Points[i].x);
    y = (int)(MAX_LB_ENTRY * Points[i].y);

    if (x < Project->x1) Project->x1 = x;
    if (x > Project->x2) Project->x2 = x;
    if (y < Project->y1) Project->y1 = y;
    if (y > Project->y2) Project->y2 = y;
  }

  *visible = TRUE;
}




/****************************************************************************
 ****************************************************************************/

static void Project_Bbox_Box(Project, Points, visible)
LIGHT_PROJECT *Project;
VECTOR *Points;
int *visible;
{
  Project_Square(Project, Points[0], Points[1], Points[3], Points[2], visible);
  Project_Square(Project, Points[4], Points[5], Points[7], Points[6], visible);
  Project_Square(Project, Points[0], Points[1], Points[5], Points[4], visible);
  Project_Square(Project, Points[2], Points[3], Points[7], Points[6], visible);
  Project_Square(Project, Points[1], Points[3], Points[7], Points[5], visible);
  Project_Square(Project, Points[0], Points[2], Points[6], Points[4], visible);
}



/****************************************************************************
 ****************************************************************************/

static void Project_Object(Project, Object, Axis, Origin)
LIGHT_PROJECT *Project;
OBJECT *Object;
int Axis;
VECTOR *Origin;
{
  int i, visible, Number;
  DBL Direction, Volume;
  VECTOR Points[8], New_Points[8];

  /* do not project infinite objects (always visible!) */

  BOUNDS_VOLUME(Volume, Object->Bounds);

  if (Volume > INFINITE_VOLUME)
  {
    Project->x1 = Project->y1 = MIN_LB_ENTRY;
    Project->x2 = Project->y2 = MAX_LB_ENTRY;
    return;
  }

  Project->x1 = Project->y1 = MAX_LB_ENTRY;
  Project->x2 = Project->y2 = MIN_LB_ENTRY;

  Calc_Points(Object, &Number, &Points[0], Origin);

  if ((Axis == XaxisP) || (Axis == YaxisP) || (Axis == ZaxisP))
  {
    Direction = +1.0;
  }
  else
  {
    Direction = -1.0;
  }

  /* modify points so that the new z-direction is the projection axis */
  
  switch (Axis)
  {
    case XaxisP :
    case XaxisM :

      for (i = 0; i < Number; i++)
      {
	New_Points[i].x = Points[i].y;
	New_Points[i].y = Points[i].z;
	New_Points[i].z = Points[i].x * Direction;
      }
      break;

    case YaxisP :
    case YaxisM :

      for (i = 0; i < Number; i++)
      {
	New_Points[i].x = Points[i].x;
	New_Points[i].y = Points[i].z;
	New_Points[i].z = Points[i].y * Direction;
      }
      break;

    case ZaxisP :
    case ZaxisM :

      for (i = 0; i < Number; i++)
      {
	New_Points[i].x = Points[i].x;
	New_Points[i].y = Points[i].y;
	New_Points[i].z = Points[i].z * Direction;
      }
      break;
  }

  visible = FALSE;

  if (Number == 3)
  {
    Project_Triangle(Project, &New_Points[0], &visible);
  }
  else
  {
    Project_Bbox_Box(Project, &New_Points[0], &visible);
  }

  if (!visible)
  {
    /* object is invisible */
    
    Project->x1 = Project->y1 = MAX_LB_ENTRY;
    Project->x2 = Project->y2 = MIN_LB_ENTRY;
  }
  else
  {
    /* we don't want to miss something */
    
    Project->x1--;
    Project->x2++;
    Project->y1--;
    Project->y2++;
  }

  Print_Point(6 * POINT_MOD);
}



/****************************************************************************
   Project the bounding slabs onto the screen.
 ****************************************************************************/

void Project_Bounding_Slab(Axis, Origin, Project, Entry, Object, Mem_Light_Buffer)
int Axis;
VECTOR *Origin;
LIGHT_PROJECT *Project;
LIGHT_TREE_NODE **Entry;
OBJECT *Object;
size_t *Mem_Light_Buffer;
{
  unsigned short int i;
  size_t size;
  COMPOSITE *Comp;
  LIGHT_PROJECT New, Temp;
  LIGHT_TREE_LEAF *Leaf;

  if (Object->Type & BOUNDING_OBJECT)
  {
    /* allocate memory for new node in the light tree (never freed!)  */

    size = sizeof(LIGHT_TREE_NODE);

    if ((*Entry = (LIGHT_TREE_NODE *)malloc(size)) == NULL)
      Fatal_MAError("Light Tree Entry");

    *Mem_Light_Buffer += size;

    Comp = (COMPOSITE *)Object;

    /* init new node */

    (*Entry)->Leaf = 0;
    
    (*Entry)->Entries = Comp->Entries;

    for (i = 0; i < BUNCHING_FACTOR; i++)
      (*Entry)->Entry[i] = NULL;

    New.x1 = New.y1 = MAX_VB_ENTRY;
    New.x2 = New.y2 = MIN_VB_ENTRY;

    /* project all entries of the bounding slabs */

    for (i = 0; i < Comp->Entries; i++)
    {
      Project_Bounding_Slab(Axis, Origin, &Temp, &(*Entry)->Entry[i], Comp->Objects[i], Mem_Light_Buffer);

      New.x1 = min(New.x1, Temp.x1);
      New.x2 = max(New.x2, Temp.x2);
      New.y1 = min(New.y1, Temp.y1);
      New.y2 = max(New.y2, Temp.y2);

      if (!(Comp->Objects[i]->Type & BOUNDING_OBJECT))
      {
	(*Entry)->Leaf |= BIT_NUMBER[i];
      }
    }

    *Project = New;

    (*Entry)->Project = New;
  }
  else
  {
    /* project object onto light source */

    Project_Object(Project, Object, Axis, Origin);

    /* is the object visible? */

    if ((Project->x1 <= Project->x2) && (Project->y1 <= Project->y2))
    {
      /* allocate memory for new leaf in the light tree (never freed!)  */

      size = sizeof(LIGHT_TREE_LEAF);

      if ((*Entry = (LIGHT_TREE_NODE *)malloc(size)) == NULL)
	Fatal_MAError("Light Tree Entry");

      *Mem_Light_Buffer += size;

      /* init new leaf */

      Leaf = (LIGHT_TREE_LEAF *)(*Entry);

      Leaf->Object = Object;

      Leaf->Project = *Project;
    }
  }
}



/****************************************************************************
 ****************************************************************************/

static void Remove_Invisible_Entries(Entry, removed, Mem_Light_Buffer)
LIGHT_TREE_NODE **Entry;
long *removed;
size_t *Mem_Light_Buffer;
{
  unsigned short int i, j;
  LIGHT_TREE_NODE *Temp;

  for (i = 0; i < (*Entry)->Entries; i++)
  {
    if (!(*Entry)->Leaf & BIT_NUMBER[i])
    {
      if ((*Entry)->Entry[i] != NULL)
      {
	Remove_Invisible_Entries(&(*Entry)->Entry[i], removed, Mem_Light_Buffer);
      }

      if ((*Entry)->Entry[i] == NULL)
      {
	(*Entry)->Entries--;

	for (j = i; j < (*Entry)->Entries; j++)
	{
	  (*Entry)->Entry[j] = (*Entry)->Entry[j+1];
	}

	(*Entry)->Entry[(*Entry)->Entries] = NULL;

	(*Entry)->Leaf = (((*Entry)->Leaf & BIT_MASK_2[i]) >> 1) |
			  ((*Entry)->Leaf & BIT_MASK_1[i]);

	i--;
      }
      else
      {
	if ((*Entry)->Entry[i]->Entries == 1)
	{
	  *Mem_Light_Buffer -= sizeof(LIGHT_TREE_NODE);

	  (*Entry)->Leaf |= ((*Entry)->Entry[i]->Leaf & BIT_NUMBER[i]);

	  Temp = (*Entry)->Entry[i];

	  (*Entry)->Entry[i] = (*Entry)->Entry[i]->Entry[0];

	  free(Temp);
	}
      }
    }
  }

  if ((*Entry)->Entries == 0)
  {
    (*removed)++;

    *Mem_Light_Buffer -= sizeof(LIGHT_TREE_NODE);

    free(*Entry);

    *Entry = NULL;
  }
}



/****************************************************************************
 ****************************************************************************/

void Build_Light_Buffers(Mem_Light_Buffer)
size_t *Mem_Light_Buffer;
{
  int i, Axis;
  long removed;
  LIGHT_PROJECT Project;
  LIGHT_SOURCE *Light;

  *Mem_Light_Buffer = 0;

  /* shadows are not calculated -> return */

  if (!(Quality_Flags & Q_SHADOW))
    return;

  /* we don't want a light buffer -> return */

  if (!(Extended_Options & USE_LIGHT_BUFFER))
    return;

  /* build the light buffer for all point(!) light sources */

  i = removed = 0;

  for (Light = Frame.Light_Sources; Light != NULL; Light = Light->Next_Light_Source)
  {
    if (!Light->Area_Light)
    {
      fprintf(stderr, "Building light buffer %ld", ++i);

      Begin_Point();

      /* project bounding slabs on all six sides */

      for (Axis = 0; Axis < 6; Axis++)
      {
	Light->Light_Buffer[Axis] = NULL;

	Project_Bounding_Slab(Axis, &Light->Center, &Project,
	  &Light->Light_Buffer[Axis], Root_Object, Mem_Light_Buffer);

	Remove_Invisible_Entries(&Light->Light_Buffer[Axis], &removed, Mem_Light_Buffer);
      }

      End_Point();
    }
  }
}

#endif

