/****************************************************************************
*
*  ATTENTION!!!
*
*  THIS FILE HAS BEEN MODIFIED!!! IT IS NOT PART OF THE OFFICAL
*  POV-RAY 2.2 DISTRIBUTION!!!
*
*  THIS FILE IS PART OF "FASTER THAN POV-RAY" (VERSION 1.1),
*  A SPED-UP VERSION OF POV-RAY 2.2. USE AT YOUR OWN RISK!!!!!!
*
*  New files: addon0.c, addon1.c, addon2.c, addon3.c, addon.h
*
*  The additional modules were written by Dieter Bayer.
*
*  Send comments, suggestions, bugs to:
*
*  dieter@cip.e-technik.uni-erlangen.de
*
*  If I have enough time I will try to fix any bugs.
*
*  All changed/added lines are enclosed in #ifdef DB_CODE ... #endif
*
*  The new/changed modules speed-up ray-tracing in two ways:
*
*   - Objects are projected onto the viewing plane a priori. Thus
*     the number of ray/object intersection tests is reduced for
*     primary rays.
*
*   - A light buffer is used for every spotlight. Each object is
*     projected a priori onto the six sides of a cube enclosing
*     the light source. Thus the number of ray/object intersection
*     tests is reduced for shadow rays.
*
*  The vista projection of qaudrics was taken from:
*
*    A. Hashimoto, T. Akimoto, K. Mase, and Y. Suenaga, "Vista
*    Ray-Tracing: High Speed Ray Tracing Using Perspective
*    Projection Image", New Advances in Computer Graphics,
*    Proceedings of CG International '89, R. A. Earnshaw,
*    B. Wyvill (Eds.), Springer, ...
*
*  The idea for the light buffer was taken from:
*
*    E. Haines and D. Greenberg, "The Light Buffer: A Shadow-
*    Testing Accelerator", IEEE CG&A, Vol. 6, No. 9, Sept. 1986, pp. 6-16
*
*****************************************************************************/

/****************************************************************************
*                spheres.c
*
*  This module implements the sphere primitive.
*
*  from Persistence of Vision Raytracer
*  Copyright 1993 Persistence of Vision Team
*---------------------------------------------------------------------------
*  NOTICE: This source code file is provided so that users may experiment
*  with enhancements to POV-Ray and to port the software to platforms other 
*  than those supported by the POV-Ray Team.  There are strict rules under
*  which you are permitted to use this file.  The rules are in the file
*  named POVLEGAL.DOC which should be distributed with this file. If 
*  POVLEGAL.DOC is not available or for more info please contact the POV-Ray
*  Team Coordinator by leaving a message in CompuServe's Graphics Developer's
*  Forum.  The latest version of POV-Ray may be found there as well.
*
* This program is based on the popular DKB raytracer version 2.12.
* DKBTrace was originally written by David K. Buck.
* DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
*
*****************************************************************************/

#include "frame.h"
#include "vector.h"
#include "povproto.h"

#ifndef Sphere_Tolerance
#define Sphere_Tolerance 1.0e-8
#endif

METHODS Sphere_Methods =
  {
  All_Sphere_Intersections,
  Inside_Sphere, Sphere_Normal,
  Copy_Sphere,
  Translate_Sphere, Rotate_Sphere,
  Scale_Sphere, Transform_Sphere, Invert_Sphere,
  Destroy_Sphere
};

METHODS Ellipsoid_Methods =
  {
  All_Ellipsoid_Intersections,
  Inside_Ellipsoid, Ellipsoid_Normal,
  Copy_Sphere,
  Translate_Sphere, Rotate_Sphere,
  Scale_Sphere, Transform_Sphere, Invert_Sphere,
  Destroy_Sphere
};

extern RAY *CM_Ray;
extern long Ray_Sphere_Tests, Ray_Sphere_Tests_Succeeded;

#ifdef DB_CODE
/******************************************************************************
 ******************************************************************************/
int All_Sphere_Intersections (Object, Ray, Depth_Stack)
OBJECT *Object;
RAY *Ray;
ISTACK *Depth_Stack;
{
  DBL Depth1, Depth2;
  VECTOR IPoint;
  register int Intersection_Found;

  Intersection_Found = FALSE;

  if (Intersect_Sphere (Ray, (SPHERE*) Object, &Depth1, &Depth2))
  {
    IPoint.x = Ray->Initial.x + Depth1 * Ray->Direction.x;
    IPoint.y = Ray->Initial.y + Depth1 * Ray->Direction.y;
    IPoint.z = Ray->Initial.z + Depth1 * Ray->Direction.z;

    if (Point_In_Clip (&IPoint, Object->Clip))
    {
      push_entry(Depth1,IPoint,Object,Depth_Stack);
      Intersection_Found = TRUE;
    }

    if (Depth2 != Depth1)
    {
      IPoint.x = Ray->Initial.x + Depth2 * Ray->Direction.x;
      IPoint.y = Ray->Initial.y + Depth2 * Ray->Direction.y;
      IPoint.z = Ray->Initial.z + Depth2 * Ray->Direction.z;

      if (Point_In_Clip (&IPoint, Object->Clip))
      {
	push_entry(Depth2,IPoint,Object,Depth_Stack);
	Intersection_Found = TRUE;
      }
    }
  }
  return (Intersection_Found);
}
#else
int All_Sphere_Intersections (Object, Ray, Depth_Stack)
OBJECT *Object;
RAY *Ray;
ISTACK *Depth_Stack;
  {
  DBL Depth1, Depth2;
  VECTOR IPoint;
  register int Intersection_Found;

  Intersection_Found = FALSE;

  if (Intersect_Sphere (Ray, (SPHERE*) Object, &Depth1, &Depth2))
    {
    VScale (IPoint, Ray->Direction, Depth1);
    VAddEq (IPoint, Ray->Initial);

    if (Point_In_Clip (&IPoint, Object->Clip))
      {
      push_entry(Depth1,IPoint,Object,Depth_Stack);
      Intersection_Found = TRUE;
      }

    if (Depth2 != Depth1)
      {
      VScale (IPoint, Ray->Direction, Depth2);
      VAddEq (IPoint, Ray->Initial);

      if (Point_In_Clip (&IPoint, Object->Clip))
	{
	push_entry(Depth2,IPoint,Object,Depth_Stack);
	Intersection_Found = TRUE;
	}
      }
    }
  return (Intersection_Found);
  }
#endif

#ifdef DB_CODE
/******************************************************************************
   using a cache saves 9 MUL and 9 ADD for all primary rays but the first.
 ******************************************************************************/
int All_Ellipsoid_Intersections (Object, Ray, Depth_Stack)
OBJECT *Object;
RAY *Ray;
ISTACK *Depth_Stack;
{
  DBL Depth1, Depth2, len;
  VECTOR IPoint, dv;
  register int Intersection_Found;
  RAY New_Ray;

  /* Transform the ray into the sphere's space */
  MInvTransDirection(&New_Ray.Direction, &Ray->Direction, ((SPHERE *)Object)->Trans);

  if (Ray == CM_Ray)
  {
    if (((SPHERE *)Object)->CMCached2)
    {
      New_Ray.Initial = ((SPHERE *)Object)->CMInitial;
    }
    else
    {
      MInvTransPoint(&New_Ray.Initial, &Ray->Initial, ((SPHERE *)Object)->Trans);
      ((SPHERE *)Object)->CMInitial = New_Ray.Initial;
      ((SPHERE *)Object)->CMCached2 = TRUE;
    }
  }
  else
  {
    MInvTransPoint(&New_Ray.Initial, &Ray->Initial, ((SPHERE *)Object)->Trans);
  }

  VDot(len, New_Ray.Direction, New_Ray.Direction);
  if (len == 0.0)
    return 0;
  len = 1.0 / sqrt(len);
  VScaleEq(New_Ray.Direction, len);

  Intersection_Found = FALSE;

  if (Intersect_Sphere (&New_Ray, ((SPHERE *)Object), &Depth1, &Depth2))
  {
    IPoint.x = New_Ray.Initial.x + Depth1 * New_Ray.Direction.x;
    IPoint.y = New_Ray.Initial.y + Depth1 * New_Ray.Direction.y;
    IPoint.z = New_Ray.Initial.z + Depth1 * New_Ray.Direction.z;

    if (((SPHERE *)Object)->Trans != NULL)
      MTransPoint(&IPoint, &IPoint, ((SPHERE *)Object)->Trans);

    VSub(dv, IPoint, Ray->Initial);
    VLength(len, dv);

    if (Point_In_Clip (&IPoint, Object->Clip))
    {
      push_entry(len,IPoint,Object,Depth_Stack);
      Intersection_Found = TRUE;
    }

    if (Depth2 != Depth1)
    {
      IPoint.x = New_Ray.Initial.x + Depth2 * New_Ray.Direction.x;
      IPoint.y = New_Ray.Initial.y + Depth2 * New_Ray.Direction.y;
      IPoint.z = New_Ray.Initial.z + Depth2 * New_Ray.Direction.z;

      if (((SPHERE *)Object)->Trans != NULL)
	MTransPoint(&IPoint, &IPoint, ((SPHERE *)Object)->Trans);

      VSub(dv, IPoint, Ray->Initial);
      VLength(len, dv);

      if (Point_In_Clip (&IPoint, Object->Clip))
      {
	push_entry(len,IPoint,Object,Depth_Stack);
	Intersection_Found = TRUE;
      }
    }
  }
  return (Intersection_Found);
}
#else
int All_Ellipsoid_Intersections (Object, Ray, Depth_Stack)
OBJECT *Object;
RAY *Ray;
ISTACK *Depth_Stack;
  {
  DBL Depth1, Depth2, len;
  VECTOR IPoint, dv;
  register int Intersection_Found;
  RAY New_Ray;

  /* Transform the ray into the sphere's space */
  MInvTransPoint(&New_Ray.Initial, &Ray->Initial, ((SPHERE *)Object)->Trans);
  MInvTransDirection(&New_Ray.Direction, &Ray->Direction, ((SPHERE *)Object)->Trans);

  VDot(len, New_Ray.Direction, New_Ray.Direction);
  if (len == 0.0)
    return 0;
  len = 1.0 / sqrt(len);
  VScaleEq(New_Ray.Direction, len);

  Intersection_Found = FALSE;

  if (Intersect_Sphere (&New_Ray, (SPHERE*) Object, &Depth1, &Depth2))
    {
    VScale (IPoint, New_Ray.Direction, Depth1);
    VAddEq (IPoint, New_Ray.Initial);
    if (((SPHERE *)Object)->Trans != NULL)
      MTransPoint(&IPoint, &IPoint, ((SPHERE *)Object)->Trans);

    VSub(dv, IPoint, Ray->Initial);
    VLength(len, dv);

    if (Point_In_Clip (&IPoint, Object->Clip))
      {
      push_entry(len,IPoint,Object,Depth_Stack);
      Intersection_Found = TRUE;
      }

    if (Depth2 != Depth1)
      {
      VScale (IPoint, New_Ray.Direction, Depth2);
      VAddEq (IPoint, New_Ray.Initial);
      if (((SPHERE *)Object)->Trans != NULL)
        MTransPoint(&IPoint, &IPoint, ((SPHERE *)Object)->Trans);

      VSub(dv, IPoint, Ray->Initial);
      VLength(len, dv);

      if (Point_In_Clip (&IPoint, Object->Clip))
        {
        push_entry(len,IPoint,Object,Depth_Stack);
        Intersection_Found = TRUE;
        }
      }
    }
  return (Intersection_Found);
  }
#endif

int Intersect_Sphere (Ray, Sphere, Depth1, Depth2)
RAY *Ray;
SPHERE *Sphere;
DBL *Depth1, *Depth2;
  {
  VECTOR Origin_To_Center;
  DBL OCSquared, t_Closest_Approach, Half_Chord, t_Half_Chord_Squared;
  short inside;

  Ray_Sphere_Tests++;
  if (Ray == CM_Ray) 
    {
    if (!Sphere->CMCached) 
      {
      VSub (Sphere->CMOtoC, Sphere->Center, Ray->Initial);
      VDot (Sphere->CMOCSquared, Sphere->CMOtoC, Sphere->CMOtoC);
      Sphere->CMinside = (Sphere->CMOCSquared < Sphere->Radius_Squared);
      Sphere->CMCached = TRUE;
      }
    VDot (t_Closest_Approach, Sphere->CMOtoC, Ray->Direction);
    if (!Sphere->CMinside && (t_Closest_Approach < Sphere_Tolerance))
      return (FALSE);      
    t_Half_Chord_Squared = Sphere->Radius_Squared - Sphere->CMOCSquared +
    (t_Closest_Approach * t_Closest_Approach);
    }
  else 
    {
    VSub (Origin_To_Center, Sphere->Center, Ray->Initial);
    VDot (OCSquared, Origin_To_Center, Origin_To_Center);
    inside = (OCSquared < Sphere->Radius_Squared);
    VDot (t_Closest_Approach, Origin_To_Center, Ray->Direction);
    if (!inside && (t_Closest_Approach < Sphere_Tolerance))
      return (FALSE);

    t_Half_Chord_Squared = Sphere->Radius_Squared - OCSquared +
    (t_Closest_Approach * t_Closest_Approach);
    }

  if (t_Half_Chord_Squared < Sphere_Tolerance)
    return (FALSE);

  Half_Chord = sqrt (t_Half_Chord_Squared);
  *Depth1 = t_Closest_Approach + Half_Chord;
  *Depth2 = t_Closest_Approach - Half_Chord;

  if ((*Depth1 < Sphere_Tolerance) || (*Depth1 > Max_Distance))
    if ((*Depth2 < Sphere_Tolerance) || (*Depth2 > Max_Distance))
      return (FALSE);
    else
      *Depth1 = *Depth2;
  else
    if ((*Depth2 < Sphere_Tolerance) || (*Depth2 > Max_Distance))
      *Depth2 = *Depth1;

  Ray_Sphere_Tests_Succeeded++;
  return (TRUE);
  }

int Inside_Sphere (IPoint, Object)
VECTOR *IPoint;
OBJECT *Object;
  {
  VECTOR Origin_To_Center;
  DBL OCSquared;

  VSub (Origin_To_Center, ((SPHERE *)Object)->Center, *IPoint);
  VDot (OCSquared, Origin_To_Center, Origin_To_Center);

  if (((SPHERE *)Object)->Inverted)
    return (OCSquared - ((SPHERE *)Object)->Radius_Squared > Sphere_Tolerance);
  else
    return (OCSquared - ((SPHERE *)Object)->Radius_Squared < Sphere_Tolerance);
  }

int Inside_Ellipsoid (IPoint, Object)
VECTOR *IPoint;
OBJECT *Object;
  {
  VECTOR Origin_To_Center;
  DBL OCSquared;
  VECTOR New_Point;

  /* Transform the point into the sphere's space */
  MInvTransPoint(&New_Point, IPoint, ((SPHERE *)Object)->Trans);
  VSub (Origin_To_Center, ((SPHERE *)Object)->Center, New_Point);
  VDot (OCSquared, Origin_To_Center, Origin_To_Center);

  if (((SPHERE *)Object)->Inverted)
    return (OCSquared - ((SPHERE *)Object)->Radius_Squared > Sphere_Tolerance);
  else
    return (OCSquared - ((SPHERE *)Object)->Radius_Squared < Sphere_Tolerance);
  }

#ifdef DB_CODE
void Sphere_Normal (Result, Object, Intersection)
OBJECT *Object;
VECTOR *Result;
INTERSECTION *Intersection;
{
  Result->x = (Intersection->IPoint.x - ((SPHERE *)Object)->Center.x) *
	      ((SPHERE *)Object)->Inverse_Radius;
  Result->y = (Intersection->IPoint.y - ((SPHERE *)Object)->Center.y) *
	      ((SPHERE *)Object)->Inverse_Radius;
  Result->z = (Intersection->IPoint.z - ((SPHERE *)Object)->Center.z) *
	      ((SPHERE *)Object)->Inverse_Radius;
}
#else
void Sphere_Normal (Result, Object, IPoint)
OBJECT *Object;
VECTOR *Result, *IPoint;
  {
  VSub (*Result, *IPoint, ((SPHERE *)Object)->Center);
  VScaleEq (*Result, ((SPHERE *)Object)->Inverse_Radius);
  }
#endif

#ifdef DB_CODE
void Ellipsoid_Normal (Result, Object, Intersection)
OBJECT *Object;
VECTOR *Result;
INTERSECTION *Intersection;
{
  VECTOR New_Point;

  /* Transform the point into the sphere's space */
  MInvTransPoint(&New_Point, &Intersection->IPoint, ((SPHERE *)Object)->Trans);

  Result->x = (New_Point.x - ((SPHERE *)Object)->Center.x) *
	      ((SPHERE *)Object)->Inverse_Radius;
  Result->y = (New_Point.y - ((SPHERE *)Object)->Center.y) *
	      ((SPHERE *)Object)->Inverse_Radius;
  Result->z = (New_Point.z - ((SPHERE *)Object)->Center.z) *
	      ((SPHERE *)Object)->Inverse_Radius;

  MTransNormal(Result, Result, ((SPHERE *)Object)->Trans);
  VNormalize(*Result, *Result);
}
#else
void Ellipsoid_Normal (Result, Object, IPoint)
OBJECT *Object;
VECTOR *Result, *IPoint;
  {
  VECTOR New_Point;

  /* Transform the point into the sphere's space */
  MInvTransPoint(&New_Point, IPoint, ((SPHERE *)Object)->Trans);

  VSub (*Result, New_Point, ((SPHERE *)Object)->Center);
  VScaleEq (*Result, ((SPHERE *)Object)->Inverse_Radius);

  MTransNormal(Result, Result, ((SPHERE *)Object)->Trans);
  VNormalize(*Result, *Result);
  }
#endif

void *Copy_Sphere (Object)
OBJECT *Object;
  {
  SPHERE *New;

  New = Create_Sphere ();
  *New = *((SPHERE *) Object);

  New->Trans = Copy_Transform(((SPHERE *)Object)->Trans);
  return (New);
  }

void Translate_Sphere (Object, Vector)
OBJECT *Object;
VECTOR *Vector;
  {
  TRANSFORM Trans;

  if (((SPHERE *)Object)->Trans == NULL)
    {
    VAddEq (((SPHERE *) Object)->Center, *Vector);
#ifdef DB_CODE
    VAddEq(Object->Bounds.Min, *Vector);
    VAddEq(Object->Bounds.Max, *Vector);
#else
    VAddEq(Object->Bounds.Lower_Left, *Vector);
#endif
    }
  else
    {
    Compute_Translation_Transform(&Trans, Vector);
    Transform_Sphere(Object, &Trans);
    }
  }

void Rotate_Sphere (Object, Vector)
OBJECT *Object;
VECTOR *Vector;
  {
  TRANSFORM Trans;
  SPHERE *Sphere = (SPHERE *) Object;

  Compute_Rotation_Transform (&Trans, Vector);

  if (Sphere->Trans == NULL)
    {
    MTransPoint(&Sphere->Center, &Sphere->Center, &Trans);
#ifdef DB_CODE
    Make_Vector(&Sphere->Bounds.Min,
      Sphere->Center.x - Sphere->Radius,
      Sphere->Center.y - Sphere->Radius,
      Sphere->Center.z - Sphere->Radius);
    Make_Vector(&Sphere->Bounds.Max,
      Sphere->Center.x + Sphere->Radius,
      Sphere->Center.y + Sphere->Radius,
      Sphere->Center.z + Sphere->Radius);
#else
    Make_Vector(&Sphere->Bounds.Lower_Left,
      Sphere->Center.x - Sphere->Radius,
      Sphere->Center.y - Sphere->Radius,
      Sphere->Center.z - Sphere->Radius);
    Make_Vector(&Sphere->Bounds.Lengths,
      2.0 * Sphere->Radius,
      2.0 * Sphere->Radius,
      2.0 * Sphere->Radius);
#endif
    }
  else
    {
    Transform_Sphere (Object, &Trans);
    }
  }

void Scale_Sphere (Object, Vector)
OBJECT *Object;
VECTOR *Vector;
  {
  SPHERE *Sphere = (SPHERE *) Object;
  TRANSFORM Trans;

  if ((Vector->x != Vector->y) || (Vector->x != Vector->z))
    if (Sphere->Trans == NULL)
      {
      Sphere->Methods = &Ellipsoid_Methods;
      Sphere->Trans = Create_Transform();
      }

  if (Sphere->Trans == NULL)
    {
    VScaleEq (Sphere->Center, Vector->x);
    Sphere->Radius *= Vector->x;
    Sphere->Radius_Squared = Sphere->Radius * Sphere->Radius;
    Sphere->Inverse_Radius = 1.0 / Sphere->Radius;
#ifdef DB_CODE
    Make_Vector(&Sphere->Bounds.Min,
      Sphere->Center.x - Sphere->Radius,
      Sphere->Center.y - Sphere->Radius,
      Sphere->Center.z - Sphere->Radius);
    Make_Vector(&Sphere->Bounds.Max,
      Sphere->Center.x + Sphere->Radius,
      Sphere->Center.y + Sphere->Radius,
      Sphere->Center.z + Sphere->Radius);
#else
    Make_Vector(&Sphere->Bounds.Lower_Left,
      Sphere->Center.x - Sphere->Radius,
      Sphere->Center.y - Sphere->Radius,
      Sphere->Center.z - Sphere->Radius);
    Make_Vector(&Sphere->Bounds.Lengths,
      2.0 * Sphere->Radius,
      2.0 * Sphere->Radius,
      2.0 * Sphere->Radius);
#endif
    }
  else
    {
    Compute_Scaling_Transform(&Trans, Vector);
    Transform_Sphere(Object, &Trans);
    }
  }

void Invert_Sphere (Object)
OBJECT *Object;
  {
  ((SPHERE *) Object)->Inverted ^= TRUE;
  }

SPHERE *Create_Sphere ()
  {
  SPHERE *New;

  if ((New = (SPHERE *) malloc (sizeof (SPHERE))) == NULL)
    MAError ("sphere");

  INIT_OBJECT_FIELDS(New, SPHERE_OBJECT, &Sphere_Methods)
    Make_Vector (&(New->Center), 0.0, 0.0, 0.0);
  New->Radius = 1.0;
  New->Radius_Squared = 1.0;
  New->Inverse_Radius = 1.0;
  New->CMCached = FALSE;
  New->Trans = NULL;
  New->Inverted = FALSE;
#ifdef DB_CODE
  New->CMCached2 = FALSE;
#endif
  /* CMOtoC, CMOtoCSquared and CMinside are only valid when CMCached */
  return (New);
  }

void Transform_Sphere (Object, Trans)
OBJECT *Object;
TRANSFORM *Trans;
  {
  SPHERE *Sphere = (SPHERE *)Object;

  if (Sphere->Trans == NULL)
    {
    Sphere->Methods = &Ellipsoid_Methods;
    Sphere->Trans = Create_Transform();
    }

  Compose_Transforms(Sphere->Trans, Trans);

#ifdef DB_CODE
    Make_Vector(&Sphere->Bounds.Min,
      Sphere->Center.x - Sphere->Radius,
      Sphere->Center.y - Sphere->Radius,
      Sphere->Center.z - Sphere->Radius);
    Make_Vector(&Sphere->Bounds.Max,
      Sphere->Center.x + Sphere->Radius,
      Sphere->Center.y + Sphere->Radius,
      Sphere->Center.z + Sphere->Radius);
#else
  Make_Vector(&Sphere->Bounds.Lower_Left,
    Sphere->Center.x - Sphere->Radius,
    Sphere->Center.y - Sphere->Radius,
    Sphere->Center.z - Sphere->Radius);
  Make_Vector(&Sphere->Bounds.Lengths,
    2.0 * Sphere->Radius,
    2.0 * Sphere->Radius,
    2.0 * Sphere->Radius);
#endif

  recompute_bbox(&Object->Bounds, Sphere->Trans);
  }

void Destroy_Sphere (Object)
OBJECT *Object;
  {
  Destroy_Transform(((SPHERE *)Object)->Trans);
  free (Object);
  }
