// Altirra Acid800 test suite
// Copyright (C) 2010 Avery Lee, All Rights Reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.

///////////////////////////////////////////////////////////////////////////////

// This program generates the test data for the cpu_insn.s test.

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string>

typedef unsigned char uint8;
typedef unsigned int uint32;

enum {
	kModeInvalid,
	kModeImplied,		// implied
	kModeRel,			// beq rel-offset
	kModeImm,			// lda #$00
	kModeZp,			// lda $00
	kModeZpX,			// lda $00,x
	kModeZpY,			// lda $00,y
	kModeAbs,			// lda $0100
	kModeAbsX,			// lda $0100,x
	kModeAbsY,			// lda $0100,y
	kModeIndA,			// jmp ($0100)
	kModeIndX,			// lda ($00,x)
	kModeIndY,			// lda ($00),y
};

enum {
	kOpcodebad,
	kOpcodeADC,
	kOpcodeAND,
	kOpcodeASL,
	kOpcodeBCC,
	kOpcodeBCS,
	kOpcodeBEQ,
	kOpcodeBIT,
	kOpcodeBMI,
	kOpcodeBNE,
	kOpcodeBPL,
	kOpcodeBRK,
	kOpcodeBVC,
	kOpcodeBVS,
	kOpcodeCLC,
	kOpcodeCLD,
	kOpcodeCLI,
	kOpcodeCLV,
	kOpcodeCMP,
	kOpcodeCPX,
	kOpcodeCPY,
	kOpcodeDEC,
	kOpcodeDEX,
	kOpcodeDEY,
	kOpcodeEOR,
	kOpcodeINC,
	kOpcodeINX,
	kOpcodeINY,
	kOpcodeJMP,
	kOpcodeJSR,
	kOpcodeLDA,
	kOpcodeLDX,
	kOpcodeLDY,
	kOpcodeLSR,
	kOpcodeNOP,
	kOpcodeORA,
	kOpcodePHA,
	kOpcodePHP,
	kOpcodePLA,
	kOpcodePLP,
	kOpcodeROL,
	kOpcodeROR,
	kOpcodeRTI,
	kOpcodeRTS,
	kOpcodeSBC,
	kOpcodeSEC,
	kOpcodeSED,
	kOpcodeSEI,
	kOpcodeSTA,
	kOpcodeSTX,
	kOpcodeSTY,
	kOpcodeTAX,
	kOpcodeTAY,
	kOpcodeTSX,
	kOpcodeTXA,
	kOpcodeTXS,
	kOpcodeTYA,
};

static const char *kOpcodes[]={
	"bad",
	"ADC",
	"AND",
	"ASL",
	"BCC",
	"BCS",
	"BEQ",
	"BIT",
	"BMI",
	"BNE",
	"BPL",
	"BRK",
	"BVC",
	"BVS",
	"CLC",
	"CLD",
	"CLI",
	"CLV",
	"CMP",
	"CPX",
	"CPY",
	"DEC",
	"DEX",
	"DEY",
	"EOR",
	"INC",
	"INX",
	"INY",
	"JMP",
	"JSR",
	"LDA",
	"LDX",
	"LDY",
	"LSR",
	"NOP",
	"ORA",
	"PHA",
	"PHP",
	"PLA",
	"PLP",
	"ROL",
	"ROR",
	"RTI",
	"RTS",
	"SBC",
	"SEC",
	"SED",
	"SEI",
	"STA",
	"STX",
	"STY",
	"TAX",
	"TAY",
	"TSX",
	"TXA",
	"TXS",
	"TYA",
};

#define xx(op) { kModeInvalid, 0 }
#define Ip(op) { kModeImplied, kOpcode##op }
#define Re(op) { kModeRel, kOpcode##op }
#define Im(op) { kModeImm, kOpcode##op }
#define Zp(op) { kModeZp, kOpcode##op }
#define Zx(op) { kModeZpX, kOpcode##op }
#define Zy(op) { kModeZpY, kOpcode##op }
#define Ab(op) { kModeAbs, kOpcode##op }
#define Ax(op) { kModeAbsX, kOpcode##op }
#define Ay(op) { kModeAbsY, kOpcode##op }
#define Ia(op) { kModeIndA, kOpcode##op }
#define Il(op) { kModeIndAL, kOpcode##op }
#define Ix(op) { kModeIndX, kOpcode##op }
#define Iy(op) { kModeIndY, kOpcode##op }

const uint8 kModeTbl_6502[256][2]={
	//			   0,       1,       2,       3,       4,       5,       6,       7,       8,       9,       A,       B,       C,       D,       E,       F
	/* 00 */	Im(BRK), Ix(ORA), xx(bad), xx(bad), xx(bad), Zp(ORA), Zp(ASL), xx(bad), Ip(PHP), Im(ORA), Ip(ASL), xx(bad), xx(bad), Ab(ORA), Ab(ASL), xx(bad), 
	/* 10 */	Re(BPL), Iy(ORA), xx(bad), xx(bad), xx(bad), Zx(ORA), Zx(ASL), xx(bad), Ip(CLC), Ay(ORA), xx(bad), xx(bad), xx(bad), Ax(ORA), Ax(ASL), xx(bad), 
	/* 20 */	Ab(JSR), Ix(AND), xx(bad), xx(bad), Zp(BIT), Zp(AND), Zp(ROL), xx(bad), Ip(PLP), Im(AND), Ip(ROL), xx(bad), Ab(BIT), Ab(AND), Ab(ROL), xx(bad), 
	/* 30 */	Re(BMI), Iy(AND), xx(bad), xx(bad), xx(bad), Zx(AND), Zx(ROL), xx(bad), Ip(SEC), Ay(AND), xx(bad), xx(bad), xx(bad), Ax(AND), Ax(ROL), xx(bad), 
	/* 40 */	Ip(RTI), Ix(EOR), xx(bad), xx(bad), xx(bad), Zp(EOR), Zp(LSR), xx(bad), Ip(PHA), Im(EOR), Ip(LSR), xx(bad), Ab(JMP), Ab(EOR), Ab(LSR), xx(bad), 
	/* 50 */	Re(BVC), Iy(EOR), xx(bad), xx(bad), xx(bad), Zx(EOR), Zx(LSR), xx(bad), Ip(CLI), Ay(EOR), xx(bad), xx(bad), xx(bad), Ax(EOR), Ax(LSR), xx(bad), 
	/* 60 */	Ip(RTS), Ix(ADC), xx(bad), xx(bad), xx(bad), Zp(ADC), Zp(ROR), xx(bad), Ip(PLA), Im(ADC), Ip(ROR), xx(bad), Ia(JMP), Ab(ADC), Ab(ROR), xx(bad), 
	/* 70 */	Re(BVS), Iy(ADC), xx(bad), xx(bad), xx(bad), Zx(ADC), Zx(ROR), xx(bad), Ip(SEI), Ay(ADC), xx(bad), xx(bad), xx(bad), Ax(ADC), Ax(ROR), xx(bad), 
	/* 80 */	xx(bad), Ix(STA), xx(bad), xx(bad), Zp(STY), Zp(STA), Zp(STX), xx(bad), Ip(DEY), xx(bad), Ip(TXA), xx(bad), Ab(STY), Ab(STA), Ab(STX), xx(bad), 
	/* 90 */	Re(BCC), Iy(STA), xx(bad), xx(bad), Zx(STY), Zx(STA), Zy(STX), xx(bad), Ip(TYA), Ay(STA), Ip(TXS), xx(bad), xx(bad), Ax(STA), xx(bad), xx(bad), 
	/* A0 */	Im(LDY), Ix(LDA), Im(LDX), xx(bad), Zp(LDY), Zp(LDA), Zp(LDX), xx(bad), Ip(TAY), Im(LDA), Ip(TAX), xx(bad), Ab(LDY), Ab(LDA), Ab(LDX), xx(bad), 
	/* B0 */	Re(BCS), Iy(LDA), xx(bad), xx(bad), Zx(LDY), Zx(LDA), Zy(LDX), xx(bad), Ip(CLV), Ay(LDA), Ip(TSX), xx(bad), Ax(LDY), Ax(LDA), Ay(LDX), xx(bad), 
	/* C0 */	Im(CPY), Ix(CMP), xx(bad), xx(bad), Zp(CPY), Zp(CMP), Zp(DEC), xx(bad), Ip(INY), Im(CMP), Ip(DEX), xx(bad), Ab(CPY), Ab(CMP), Ab(DEC), xx(bad), 
	/* D0 */	Re(BNE), Iy(CMP), xx(bad), xx(bad), xx(bad), Zx(CMP), Zx(DEC), xx(bad), Ip(CLD), Ay(CMP), xx(bad), xx(bad), xx(bad), Ax(CMP), Ax(DEC), xx(bad), 
	/* E0 */	Im(CPX), Ix(SBC), xx(bad), xx(bad), Zp(CPX), Zp(SBC), Zp(INC), xx(bad), Ip(INX), Im(SBC), Ip(NOP), xx(bad), Ab(CPX), Ab(SBC), Ab(INC), xx(bad), 
	/* F0 */	Re(BEQ), Iy(SBC), xx(bad), xx(bad), xx(bad), Zx(SBC), Zx(INC), xx(bad), Ip(SED), Ay(SBC), xx(bad), xx(bad), xx(bad), Ax(SBC), Ax(INC), xx(bad),
};

struct Processor {
	uint8 mA;
	uint8 mX;
	uint8 mY;
	uint8 mP;
	uint8 mS;
	uint8 mM;

	void SetNZ(uint8 v) {
		mP &= 0x7d;

		mP |= (v & 0x80);

		if (!v)
			mP |= 0x02;
	}

	void Execute(uint8 mnem, uint8 mode, uint8 data) {

		if (mode != kModeImm)
			data = mM;

		switch(mnem) {
			case kOpcodeADC:
				// We assume decimal mode is disabled here.
				{
					uint32 carry7 = (mA & 0x7f) + (data & 0x7f) + (mP & 0x01);
					uint32 result = carry7 + (mA & 0x80) + (data & 0x80);

					mP &= 0x3c;

					if (result & 0x80)
						mP |= 0x80;

					if (result >= 0x100)
						mP |= 0x01;

					if (!(result & 0xff))
						mP |= 0x02;

					mP |= ((result >> 2) ^ (carry7 >> 1)) & 0x40;

					mA = (uint8)result;
				}
				break;

			case kOpcodeAND:
				mA &= data;
				SetNZ(mA);
				break;

			case kOpcodeASL:
				if (mode == kModeImplied) {
					mP &= 0x7c;

					mP += mA >> 7;
					mA += mA;
					mP += mA & 0x80;
					if (!mA)
						mP += 2;
				} else {
					mP &= 0x7c;

					mP += data >> 7;
					data += data;
					mP += data & 0x80;
					if (!data)
						mP += 2;
				}
				break;

			case kOpcodeBIT:
				mP = (mP & 0x3d) + (data & 0xc0) + (mA & data ? 0x00 : 0x02);
				break;

			case kOpcodeCLC:
				mP &= 0xfe;
				break;

			case kOpcodeCLD:
				mP &= 0xf7;
				break;

			case kOpcodeCLI:
				mP &= 0xfb;
				break;

			case kOpcodeCLV:
				mP &= 0xbf;
				break;

			case kOpcodeCMP:
				{
					uint8 d = ~data;
					uint32 result = mA + d + 1;

					mP &= 0x7c;

					if (result & 0x80)
						mP |= 0x80;

					if (result >= 0x100)
						mP |= 0x01;

					if (!(result & 0xff))
						mP |= 0x02;
				}
				break;

			case kOpcodeCPX:
				{
					uint8 d = ~data;
					uint32 result = mX + d + 1;

					mP &= 0x7c;

					if (result & 0x80)
						mP |= 0x80;

					if (result >= 0x100)
						mP |= 0x01;

					if (!(result & 0xff))
						mP |= 0x02;
				}
				break;

			case kOpcodeCPY:
				{
					uint8 d = ~data;
					uint32 result = mY + d + 1;

					mP &= 0x7c;

					if (result & 0x80)
						mP |= 0x80;

					if (result >= 0x100)
						mP |= 0x01;

					if (!(result & 0xff))
						mP |= 0x02;
				}
				break;

			case kOpcodeDEC:
				--data;
				SetNZ(data);
				break;

			case kOpcodeDEX:
				--mX;
				SetNZ(mX);
				break;

			case kOpcodeDEY:
				--mY;
				SetNZ(mY);
				break;

			case kOpcodeEOR:
				mA ^= data;
				SetNZ(mA);
				break;

			case kOpcodeINC:
				++data;
				SetNZ(data);
				break;

			case kOpcodeINX:
				++mX;
				SetNZ(mX);
				break;

			case kOpcodeINY:
				++mY;
				SetNZ(mY);
				break;

			case kOpcodeLDA:
				mA = data;
				SetNZ(mA);
				break;

			case kOpcodeLDX:
				mX = data;
				SetNZ(mX);
				break;

			case kOpcodeLDY:
				mY = data;
				SetNZ(mY);
				break;

			case kOpcodeLSR:
				if (mode == kModeImplied) {
					mP &= 0x7c;

					mP += mA & 0x01;
					mA >>= 1;
					mP += mA & 0x80;
					if (!mA)
						mP += 2;
				} else {
					mP &= 0x7c;

					mP += data & 0x01;
					data >>= 1;
					mP += data & 0x80;
					if (!data)
						mP += 2;
				}
				break;

			case kOpcodeNOP:
				break;

			case kOpcodeORA:
				mA |= data;
				SetNZ(mA);
				break;

			case kOpcodeROL:
				if (mode == kModeImplied) {
					uint8 c = mP & 1;
					mP &= 0x7c;

					mP += mA >> 7;
					mA = c + (mA << 1);
					mP += mA & 0x80;
					if (!mA)
						mP += 2;
				} else {
					uint8 c = mP & 1;
					mP &= 0x7c;

					mP += data >> 7;
					data = c + (data << 1);
					mP += data & 0x80;
					if (!data)
						mP += 2;
				}
				break;

			case kOpcodeROR:
				if (mode == kModeImplied) {
					uint8 c = mP << 7;
					mP &= 0x7c;

					mP += mA & 0x01;
					mA = (mA >> 1) + c;
					mP += mA & 0x80;
					if (!mA)
						mP += 2;
				} else {
					uint8 c = mP << 7;
					mP &= 0x7c;

					mP += data & 0x01;
					data = (data >> 1) + c;
					mP += data & 0x80;
					if (!data)
						mP += 2;
				}
				break;

			case kOpcodeSBC:
				{
					uint32 carry7 = (mA & 0x7f) + (~data & 0x7f) + (mP & 0x01);
					uint32 result = carry7 + (mA & 0x80) + (~data & 0x80);

					mP &= 0x3c;

					if (result & 0x80)
						mP |= 0x80;

					if (result >= 0x100)
						mP |= 0x01;

					if (!(result & 0xff))
						mP |= 0x02;

					mP |= ((result >> 2) ^ (carry7 >> 1)) & 0x40;

					mA = (uint8)result;
				}
				break;

			case kOpcodeSEC:
				mP |= 0x01;
				break;

			case kOpcodeSED:
				mP |= 0x08;
				break;

			case kOpcodeSEI:
				mP |= 0x04;
				break;

			case kOpcodeSTA:
				data = mA;
				break;

			case kOpcodeSTX:
				data = mX;
				break;

			case kOpcodeSTY:
				data = mY;
				break;

			case kOpcodeTAX:
				mX = mA;
				SetNZ(mX);
				break;

			case kOpcodeTAY:
				mY = mA;
				SetNZ(mY);
				break;

			case kOpcodeTXA:
				mA = mX;
				SetNZ(mA);
				break;

			case kOpcodeTYA:
				mA = mY;
				SetNZ(mA);
				break;
		}

		if (mode != kModeImm)
			mM = data;
	}
};

unsigned seed = 1;

int detrand() {
	int v = (seed >> 16) & 0x7fff;

	seed = seed * 214013 + 2531011;

	return v;
}

// This LCG RNG is tracked by the 6502 code and its values are not stored.
unsigned seed2 = 0x12345678;
int detrand2() {
	int v = seed2 & 0xff;

	seed2 >>= 8;

	// $A3000000
	seed2 ^= (v << 24);
	seed2 ^= (v << 22);
	seed2 ^= (v << 18);
	seed2 ^= (v << 17);

	return v;
}

std::string buffer;
void appendf(const char *format, ...) {
	char buf[256];
	va_list val;

	va_start(val, format);
	vsprintf(buf, format, val);
	va_end(val);

	buffer += buf;
}

int main() {
	Processor pr;

	uint8 lastop1 = 0x60;
	uint8 lastop2 = 0x60;
	int lastcontrol = -1;

	for(uint32 i=0; i<256; ++i) {
		const uint8 op = i;

		const uint8 mode = kModeTbl_6502[i][0];
		const uint8 mnem = kModeTbl_6502[i][1];

		if (!mode)
			continue;

		// skip branch, return, and stack instructions
		switch(mnem) {
			case kOpcodeBCC:
			case kOpcodeBCS:
			case kOpcodeBPL:
			case kOpcodeBMI:
			case kOpcodeBVC:
			case kOpcodeBVS:
			case kOpcodeBEQ:
			case kOpcodeBNE:
			case kOpcodeRTS:
			case kOpcodeRTI:
			case kOpcodeJMP:
			case kOpcodeJSR:
			case kOpcodeBRK:
			case kOpcodeTSX:
			case kOpcodeTXS:
			case kOpcodePHA:
			case kOpcodePHP:
			case kOpcodePLA:
			case kOpcodePLP:
				continue;
		}

		printf("\n");

		const char *name = kOpcodes[mnem];
		switch(mode) {
			case kModeImplied:
				printf("\t\t; $%02X  %s\n", op, name);
				break;
			case kModeImm:
				printf("\t\t; $%02X  %s #imm\n", op, name);
				break;
			case kModeZp:
				printf("\t\t; $%02X  %s zp\n", op, name);
				break;
			case kModeZpX:
				printf("\t\t; $%02X  %s zp,x\n", op, name);
				break;
			case kModeZpY:
				printf("\t\t; $%02X  %s zp,y\n", op, name);
				break;
			case kModeAbs:
				printf("\t\t; $%02X  %s abs\n", op, name);
				break;
			case kModeAbsX:
				printf("\t\t; $%02X  %s abs,x\n", op, name);
				break;
			case kModeAbsY:
				printf("\t\t; $%02X  %s abs,y\n", op, name);
				break;
			case kModeIndX:
				printf("\t\t; $%02X  %s (zp,x)\n", op, name);
				break;
			case kModeIndY:
				printf("\t\t; $%02X  %s (zp),y\n", op, name);
				break;
		}

		int controlmask = 0;
		for(int j=0; j<8; ++j) {
			pr.mA = (uint8)detrand2();
			pr.mY = (uint8)detrand();

			// Force decimal flag off for SBC/ADC; we test that elsewhere.
			pr.mP = ((uint8)detrand2() | 0x30) & 0xf7;

			pr.mM = (uint8)detrand2();
			uint8 immdata = (uint8)detrand();

			if (mode == kModeIndY)
				pr.mY &= 1;
			else if (mode == kModeAbsX)
				pr.mY |= 0x80;
			else if (mode == kModeAbsY)
				pr.mY &= 0x7F;

			pr.mX = ~pr.mY;

			Processor pr1(pr);
			pr1.Execute(mnem, mode, immdata);

			// Compute compression byte.
			//
			//	D7: Instruction changed; opcode byte included
			//	D6: First operand byte changed
			//	D5: Second operand byte changed
			//	D4: A result is different
			//	D3: X result is different
			//	D2: Y result is different
			//	D1: P result is different
			//	D0: M result is different
			//
			// Note that we are hardcoding a few variables here:
			//	
			// D5 is at $CD.
			// A1 is at $C2.
			// A2 is at $C4.

			uint8 compbyte = (j ? 0x00 : 0x80);

			if (!j) {
				lastop1 = 0x60;
				lastop2 = 0x60;
			}

			uint8 op1 = lastop1;
			uint8 op2 = lastop2;

			enum {
				ADDR_D5 = 0xCD,
				ADDR_A1 = 0xC2,
				ADDR_A2 = 0xC4
			};

			switch(mode) {
				case kModeImplied:
					break;
				case kModeImm:
					op1 = immdata;
					break;
				case kModeZp:
					op1 = ADDR_D5;
					break;
				case kModeZpX:
					op1 = ADDR_D5 - pr.mX;
					break;
				case kModeZpY:
					op1 = ADDR_D5 - pr.mY;
					break;
				case kModeAbs:
					op1 = ADDR_D5;
					op2 = 0;
					break;

				// We used to test whether the absolute modes wrapped around 64K. This is no
				// longer allowed as it breaks on 65C816s.

				case kModeAbsX:
					op1 = ADDR_D5 - pr.mX;
					op2 = 0;
					break;
				case kModeAbsY:
					op1 = ADDR_D5 - pr.mY;
					op2 = 0;
					break;
				case kModeIndX:
					op1 = ADDR_A1 - pr.mX;
					break;
				case kModeIndY:
					op1 = pr.mY ? ADDR_A2 : ADDR_A1;
					break;
			}

			if (op1 != lastop1)
				compbyte |= 0x40;

			if (op2 != lastop2)
				compbyte |= 0x20;

			if (pr.mA != pr1.mA)
				compbyte |= 0x10;

			if (pr.mX != pr1.mX)
				compbyte |= 0x08;

			if (pr.mY != pr1.mY)
				compbyte |= 0x04;

			if (pr.mP != pr1.mP)
				compbyte |= 0x02;

			if (pr.mM != pr1.mM)
				compbyte |= 0x01;

			lastop1 = op1;
			lastop2 = op2;

			// check if we need to emit a different control byte
			controlmask += controlmask;
			if (compbyte != lastcontrol) {
				++controlmask;
				lastcontrol = compbyte;
			}

			appendf("\t\tdta\t");
			if (controlmask & 1)
				appendf("$%02X,", compbyte);

			if (compbyte & 0x80) appendf("$%02X,", i);
			if (compbyte & 0x40) appendf("$%02X,", op1);
			if (compbyte & 0x20) appendf("$%02X,", op2);

			// A, P, and M bytes are implicit.
			appendf("$%02X", pr.mY);

			if (compbyte & 0x10) appendf(",$%02X", pr1.mA);
			if (compbyte & 0x08) appendf(",$%02X", pr1.mX);
			if (compbyte & 0x04) appendf(",$%02X", pr1.mY);
			if (compbyte & 0x02) appendf(",$%02X", pr1.mP);
			if (compbyte & 0x01) appendf(",$%02X", pr1.mM);
			appendf(" ;Input: A=$%02X Y=$%02X P=$%02X M=$%02X  Output: A=$%02X X=$%02X Y=$%02X P=$%02X M=$%02X\n"
				, pr.mA, pr.mY, pr.mP, pr.mM
				, pr1.mA, pr1.mX, pr1.mY, pr1.mP, pr1.mM
				);
		}

		printf("\t\tdta\t$%02X\n", controlmask);
		fputs(buffer.c_str(), stdout);
		buffer.clear();
	}
}
