#include <stdafx.h>
#include "blackbox.h"
#include "memorymanager.h"
#include "firmwaremanager.h"
#include "devicemanager.h"
#include "irqcontroller.h"
#include "idedisk.h"
#include "scsidisk.h"

ATBlackBoxEmulator::ATBlackBoxEmulator()
	: mPBIBANK(0)
	, mRAMPAGE(0)
	, mActiveIRQs(0)
	, mpFwMan(nullptr)
	, mpUIRenderer(nullptr)
	, mpIRQController(nullptr)
	, mIRQBit(0)
	, mpScheduler(nullptr)
	, mpMemMan(nullptr)
	, mpMemLayerPBI(nullptr)
	, mpMemLayerRAM(nullptr)
	, mpMemLayerFirmware(nullptr)

{
	mpEventsReleaseButton[0] = nullptr;
	mpEventsReleaseButton[1] = nullptr;

	mVIA.SetPortAInput(0xff);
	mVIA.SetPortBInput(0xff);
	mVIA.SetPortOutputFn(OnVIAOutputChanged, this);
	mPIA.AllocOutput(OnPIAOutputChanged, this, 0xFFFFFFFFUL);
	mSCSIBus.SetBusMonitor(this);
}

ATBlackBoxEmulator::~ATBlackBoxEmulator() {
}

void *ATBlackBoxEmulator::AsInterface(uint32 iid) {
	switch(iid) {
		case IATDeviceMemMap::kTypeID: return static_cast<IATDeviceMemMap *>(this);
		case IATDeviceFirmware::kTypeID: return static_cast<IATDeviceFirmware *>(this);
		case IATDeviceIRQSource::kTypeID: return static_cast<IATDeviceIRQSource *>(this);
		case IATDeviceScheduling::kTypeID: return static_cast<IATDeviceScheduling *>(this);
		case IATDeviceButtons::kTypeID: return static_cast<IATDeviceButtons *>(this);
		case IATDeviceParent::kTypeID: return static_cast<IATDeviceParent *>(this);
		case IATDeviceIndicators::kTypeID: return static_cast<IATDeviceIndicators *>(this);
	}

	return nullptr;
}

void ATBlackBoxEmulator::GetDeviceInfo(ATDeviceInfo& info) {
	info.mTag = "blackbox";
	info.mName = L"BlackBox";
}

void ATBlackBoxEmulator::GetSettings(ATPropertySet& settings) {
}

void ATBlackBoxEmulator::Shutdown() {
	mSCSIBus.Shutdown();

	while(!mSCSIDisks.empty()) {
		const SCSIDiskEntry& ent = mSCSIDisks.back();

		ent.mpDevice->Release();
		ent.mpDisk->Release();

		mSCSIDisks.pop_back();
	}

	if (mpIRQController) {
		if (mIRQBit) {
			mpIRQController->FreeIRQ(mIRQBit);
			mIRQBit = 0;
		}

		mpIRQController = nullptr;
	}

	if (mpScheduler) {
		mpScheduler->UnsetEvent(mpEventsReleaseButton[1]);
		mpScheduler->UnsetEvent(mpEventsReleaseButton[0]);

		mpScheduler = nullptr;
	}

	if (mpMemLayerFirmware) {
		mpMemMan->DeleteLayer(mpMemLayerFirmware);
		mpMemLayerFirmware = nullptr;
	}

	if (mpMemLayerRAM) {
		mpMemMan->DeleteLayer(mpMemLayerRAM);
		mpMemLayerRAM = NULL;
	}

	if (mpMemLayerPBI) {
		mpMemMan->DeleteLayer(mpMemLayerPBI);
		mpMemLayerPBI = NULL;
	}

	mVIA.Shutdown();

	mpUIRenderer = nullptr;
}

void ATBlackBoxEmulator::WarmReset() {
	mPBIBANK = 0xFF;
	mRAMPAGE = 0xFF;

	SetPBIBANK(0);
	SetRAMPAGE(0);

	mPIA.Reset();
	mVIA.Reset();

	mActiveIRQs = 0;
	mpIRQController->Negate(mIRQBit, false);

	mpScheduler->UnsetEvent(mpEventsReleaseButton[0]);
	mpScheduler->UnsetEvent(mpEventsReleaseButton[1]);
}

void ATBlackBoxEmulator::ColdReset() {
	memset(mRAM, 0xFF, sizeof mRAM);

	WarmReset();
}

void ATBlackBoxEmulator::InitMemMap(ATMemoryManager *memmap) {
	mpMemMan = memmap;

	ATMemoryHandlerTable handlers={};
	handlers.mpDebugReadHandler = OnDebugRead;
	handlers.mpReadHandler = OnRead;
	handlers.mpWriteHandler = OnWrite;
	handlers.mpThis = this;
	handlers.mbPassAnticReads = true;
	handlers.mbPassReads = true;
	handlers.mbPassWrites = true;

	mpMemLayerPBI = mpMemMan->CreateLayer(kATMemoryPri_PBISelect+1, handlers, 0xD1, 0x01);
	mpMemMan->SetLayerName(mpMemLayerPBI, "BlackBox PBI");
	mpMemMan->EnableLayer(mpMemLayerPBI, true);

	mpMemLayerRAM = mpMemMan->CreateLayer(kATMemoryPri_PBI, mRAM, 0xD6, 0x01, false);
	mpMemMan->SetLayerName(mpMemLayerRAM, "BlackBox RAM");
	mpMemMan->EnableLayer(mpMemLayerRAM, true);

	mpMemLayerFirmware = mpMemMan->CreateLayer(kATMemoryPri_PBI, mFirmware, 0xD8, 0x08, true);
	mpMemMan->SetLayerName(mpMemLayerFirmware, "BlackBox ROM");
	mpMemMan->EnableLayer(mpMemLayerFirmware, true);
}

void ATBlackBoxEmulator::InitFirmware(ATFirmwareManager *fwman) {
	mpFwMan = fwman;

	memset(mFirmware, 0xFF, sizeof mFirmware);
	fwman->LoadFirmware(fwman->GetCompatibleFirmware(kATFirmwareType_BlackBox), mFirmware, 0, sizeof mFirmware);
}

void ATBlackBoxEmulator::InitIRQSource(ATIRQController *irqc) {
	mpIRQController = irqc;
}

void ATBlackBoxEmulator::InitScheduling(ATScheduler *sch) {
	mpScheduler = sch;

	mVIA.Init(sch);
}

void ATBlackBoxEmulator::InitIndicators(IATUIRenderer *r) {
	mpUIRenderer = r;
}

void ATBlackBoxEmulator::ActivateButton(uint32 idx) {
	if (idx == 0) {
		mActiveIRQs |= 0x04;
		mpIRQController->Assert(mIRQBit, false);
		mpScheduler->SetEvent(170000, this, 1, mpEventsReleaseButton[0]);
	} else if (idx == 1) {
		mActiveIRQs |= 0x08;
		mpIRQController->Assert(mIRQBit, false);
		mpScheduler->SetEvent(170000, this, 2, mpEventsReleaseButton[1]);
	}
}

void ATBlackBoxEmulator::GetChildDevices(vdfastvector<IATDevice *>& devs) {
	for(auto it = mSCSIDisks.begin(), itEnd = mSCSIDisks.end();
		it != itEnd;
		++it)
	{
		const SCSIDiskEntry& ent = *it;

		devs.push_back(vdpoly_cast<IATDevice *>(ent.mpDisk));
	}
}

void ATBlackBoxEmulator::AddChildDevice(IATDevice *dev) {
	IATIDEDisk *disk = vdpoly_cast<IATIDEDisk *>(dev);

	if (disk) {
		VDASSERT(vdpoly_cast<IATDevice *>(disk));

		vdrefptr<IATSCSIDiskDevice> dev;
		ATCreateSCSIDiskDevice(disk, ~dev);

		SCSIDiskEntry entry = { dev, disk };
		mSCSIDisks.push_back(entry);
		dev->AddRef();
		disk->AddRef();

		dev->SetUIRenderer(mpUIRenderer);

		mSCSIBus.AttachDevice(0, dev);
	}
}

void ATBlackBoxEmulator::RemoveChildDevice(IATDevice *dev) {
	IATIDEDisk *disk = vdpoly_cast<IATIDEDisk *>(dev);

	if (!disk)
		return;

	for(auto it = mSCSIDisks.begin(), itEnd = mSCSIDisks.end();
		it != itEnd;
		++it)
	{
		const SCSIDiskEntry& ent = *it;

		if (ent.mpDisk == disk) {
			mSCSIBus.DetachDevice(ent.mpDevice);
			ent.mpDisk->Release();
			ent.mpDevice->Release();
			mSCSIDisks.erase(it);
		}
	}
}

void ATBlackBoxEmulator::OnSCSIControlStateChanged(uint32 state) {
	uint8 portb = 0xFF;

	if (state & kATSCSICtrlState_IO)
		portb &= ~0x01;

	if (state & kATSCSICtrlState_CD)
		portb &= ~0x02;

	if (state & kATSCSICtrlState_BSY)
		portb &= ~0x40;

	if (state & kATSCSICtrlState_REQ)
		portb &= ~0x80;

	mVIA.SetPortAInput(state & 0xff);
	mVIA.SetPortBInput(portb);
	mVIA.SetCA1Input(!(state & kATSCSICtrlState_REQ));
}

void ATBlackBoxEmulator::OnScheduledEvent(uint32 id) {
	if (id == 1) {
		mpEventsReleaseButton[0] = nullptr;
		mActiveIRQs &= ~0x04;
		if (!mActiveIRQs)
			mpIRQController->Negate(mIRQBit, false);
	} else if (id == 2) {
		mpEventsReleaseButton[1] = nullptr;
		mActiveIRQs &= ~0x08;
		if (!mActiveIRQs)
			mpIRQController->Negate(mIRQBit, false);
	}
}

sint32 ATBlackBoxEmulator::OnDebugRead(void *thisptr0, uint32 addr) {
	const auto thisptr = (ATBlackBoxEmulator *)thisptr0;

	if ((addr - 0xD170) < 0x10)
		return thisptr->mVIA.DebugReadByte(addr);
	
	return OnRead(thisptr0, addr);
}

sint32 ATBlackBoxEmulator::OnRead(void *thisptr0, uint32 addr) {
	const auto thisptr = (ATBlackBoxEmulator *)thisptr0;

	if ((addr - 0xD170) < 0x10)
		return thisptr->mVIA.ReadByte(addr);

	if ((addr - 0xD180) < 0x40)
		return thisptr->mPIA.ReadByte(addr);

	if ((addr - 0xD1C0) < 0x40)
		return thisptr->mActiveIRQs ? 0x02 | thisptr->mActiveIRQs : 0x00;

	return -1;
}

bool ATBlackBoxEmulator::OnWrite(void *thisptr0, uint32 addr, uint8 value) {
	const auto thisptr = (ATBlackBoxEmulator *)thisptr0;

	if ((addr - 0xD170) < 0x10) {
		thisptr->mVIA.WriteByte(addr, value);
		return false;
	}

	if ((addr - 0xD180) < 0x40) {
		thisptr->mPIA.WriteByte(addr, value);
		return false;
	}

	if ((addr - 0xD1C0) < 0x40) {	// PBIBANK/PDVS
		thisptr->SetPBIBANK(value);
		return false;
	}

	return false;
}

void ATBlackBoxEmulator::OnPIAOutputChanged(void *thisptr0, uint32 val) {
	const auto thisptr = (ATBlackBoxEmulator *)thisptr0;

	// PORTA -> RAMPAGE
	thisptr->SetRAMPAGE(val & 0xff);
}

void ATBlackBoxEmulator::OnVIAOutputChanged(void *thisptr0, uint32 val) {
	const auto thisptr = (ATBlackBoxEmulator *)thisptr0;
	uint32 newState = val & 0xff;

	if (!(val & 0x400))
		newState |= kATSCSICtrlState_SEL;

	if (!(val & 0x800))
		newState |= kATSCSICtrlState_RST;

	if (!(val & kATVIAOutputBit_CA2))
		newState |= kATSCSICtrlState_ACK;

	thisptr->mSCSIBus.SetControl(0, newState, kATSCSICtrlState_SEL | kATSCSICtrlState_RST | kATSCSICtrlState_ACK | 0xFF);
}

void ATBlackBoxEmulator::SetPBIBANK(uint8 value) {
	value &= 15;

	if (mPBIBANK == value)
		return;

	mPBIBANK = value;

	if (value) {
		mpMemMan->SetLayerMemory(mpMemLayerFirmware, mFirmware + ((uint32)((value - 1) & 7) << 11));
		mpMemMan->EnableLayer(mpMemLayerFirmware, true);
	} else {
		mpMemMan->EnableLayer(mpMemLayerFirmware, false);
	}
}

void ATBlackBoxEmulator::SetRAMPAGE(uint8 value) {
	if (mRAMPAGE == value)
		return;

	mRAMPAGE = value;

	mpMemMan->SetLayerMemory(mpMemLayerRAM, mRAM + ((uint32)value << 8));
}

/////////////////////////////////////////////////////////////////

void ATCreateDeviceBlackBoxEmulator(const ATPropertySet& pset, IATDevice **dev) {
	*dev = new ATBlackBoxEmulator;
	(*dev)->AddRef();
}

void ATRegisterDeviceBlackBox(ATDeviceManager& dev) {
	dev.AddDeviceFactory("blackbox", ATCreateDeviceBlackBoxEmulator);
}
