//	Altirra - Atari 800/800XL/5200 emulator
//	Copyright (C) 2009-2014 Avery Lee
//
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <stdafx.h>
#include <vd2/system/refcount.h>
#include <vd2/system/unknown.h>
#include <at/atcore/device.h>
#include "devicemanager.h"
#include "simulator.h"

ATDeviceManager::ATDeviceManager()
	: mpSim(nullptr)
{
}

ATDeviceManager::~ATDeviceManager() {
}

void ATDeviceManager::Init(ATSimulator& sim) {
	mpSim = &sim;
}

IATDevice *ATDeviceManager::AddDevice(const char *tag, const ATPropertySet& pset, bool child) {
	vdrefptr<IATDevice> dev2;
	IATDevice *dev = GetDeviceByTag(tag);

	if (!dev) {

		for(auto it = mDeviceFactories.begin(), itEnd = mDeviceFactories.end();
			it != itEnd;
			++it)
		{
			if (!strcmp(it->mpTag, tag)) {
				it->mpCreate(pset, ~dev2);
				break;
			}
		}

		if (dev2) {
			AddDevice(dev2, child);
			dev = dev2;
		}
	}

	return dev;
}

void ATDeviceManager::AddDevice(IATDevice *dev, bool child) {
	if (auto devmm = vdpoly_cast<IATDeviceMemMap *>(dev))
		devmm->InitMemMap(mpSim->GetMemoryManager());

	if (auto devfw = vdpoly_cast<IATDeviceFirmware *>(dev))
		devfw->InitFirmware(mpSim->GetFirmwareManager());

	if (auto devirq = vdpoly_cast<IATDeviceIRQSource *>(dev))
		devirq->InitIRQSource(mpSim->GetIRQController());

	if (auto devsch = vdpoly_cast<IATDeviceScheduling *>(dev))
		devsch->InitScheduling(mpSim->GetScheduler());

	if (auto devin = vdpoly_cast<IATDeviceIndicators *>(dev))
		devin->InitIndicators(mpSim->GetUIRenderer());

	DeviceEntry ent = { dev, child };
	mDevices.push_back(ent);
	dev->AddRef();

	dev->ColdReset();
}

void ATDeviceManager::RemoveDevice(const char *tag) {
	if (IATDevice *dev = GetDeviceByTag(tag))
		RemoveDevice(dev);
}

void ATDeviceManager::RemoveDevice(IATDevice *dev) {
	for(auto it = mDevices.begin(), itEnd = mDevices.end();
		it != itEnd;
		++it)
	{
		if (it->mpDevice == dev) {
			if (it->mbChild) {
				// scan all possible parents
				for(auto it2 = mDevices.begin(), it2End = mDevices.end();
					it2 != it2End;
					++it2)
				{
					IATDeviceParent *parent = vdpoly_cast<IATDeviceParent *>(it2->mpDevice);

					if (parent)
						parent->RemoveChildDevice(dev);
				}
			}

			mDevices.erase(it);
			dev->Shutdown();
			dev->Release();
			break;
		}
	}
}

void ATDeviceManager::RemoveAllDevices() {
	while(!mDevices.empty())
		RemoveDevice(mDevices.front().mpDevice);
}

uint32 ATDeviceManager::GetDeviceCount() const {
	return mDevices.size();
}

IATDevice *ATDeviceManager::GetDeviceByTag(const char *tag) const {
	ATDeviceInfo info;

	for(auto it = mDevices.begin(), itEnd = mDevices.end();
		it != itEnd;
		++it)
	{
		IATDevice *dev = it->mpDevice;

		dev->GetDeviceInfo(info);
		if (info.mTag == tag)
			return dev;
	}

	return nullptr;
}

IATDevice *ATDeviceManager::GetDeviceByIndex(uint32 i) const {
	return i < mDevices.size() ? mDevices[i].mpDevice : nullptr;
}

ATDeviceConfigureFn ATDeviceManager::GetDeviceConfigureFn(const char *tag) const {
	for(auto it = mDeviceConfigurers.begin(), itEnd = mDeviceConfigurers.end();
		it != itEnd;
		++it)
	{
		if (!strcmp(tag, it->mpTag))
			return it->mpConfigure;
	}

	return nullptr;
}

void ATDeviceManager::AddDeviceFactory(const char *tag, ATDeviceFactoryFn factory) {
	auto& fac = mDeviceFactories.push_back();

	fac.mpTag = tag;
	fac.mpCreate = factory;
}

void ATDeviceManager::AddDeviceConfigurer(const char *tag, ATDeviceConfigureFn configurer) {
	auto& fac = mDeviceConfigurers.push_back();

	fac.mpTag = tag;
	fac.mpConfigure = configurer;
}

void ATDeviceManager::MarkAndSweep(IATDevice *const *pExcludedDevs, size_t numExcludedDevs, vdfastvector<IATDevice *>& garbage) {
	vdhashset<IATDevice *> devSet;

	for(auto it = mDevices.begin(), itEnd = mDevices.end();
		it != itEnd;
		++it)
	{
		if (it->mbChild)
			devSet.insert(it->mpDevice);
	}

	for(auto it = mDevices.begin(), itEnd = mDevices.end();
		it != itEnd;
		++it)
	{
		if (!it->mbChild)
			Mark(it->mpDevice, pExcludedDevs, numExcludedDevs, devSet);
	}

	for(size_t i=0; i<numExcludedDevs; ++i) {
		devSet.erase(pExcludedDevs[i]);
	}

	for(auto it = devSet.begin(), itEnd = devSet.end();
		it != itEnd;
		++it)
	{
		garbage.push_back(*it);
	}
}

void ATDeviceManager::Mark(IATDevice *dev, IATDevice *const *pExcludedDevs, size_t numExcludedDevs, vdhashset<IATDevice *>& devSet) {
	for(size_t i=0; i<numExcludedDevs; ++i) {
		if (dev == pExcludedDevs[i])
			return;
	}

	auto *parent = vdpoly_cast<IATDeviceParent *>(dev);
	if (parent) {
		vdfastvector<IATDevice *> children;
		parent->GetChildDevices(children);

		while(!children.empty()) {
			IATDevice *child = children.back();

			auto it = devSet.find(dev);
			if (it != devSet.end()) {
				devSet.erase(it);
				Mark(child, pExcludedDevs, numExcludedDevs, devSet);
			}

			children.pop_back();
		}
	}
}
