//	Altirra - Atari 800/800XL/5200 emulator
//	Copyright (C) 2009-2014 Avery Lee
//
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include "stdafx.h"
#include <windows.h>
#include <vd2/system/error.h>
#include <vd2/system/file.h>
#include <vd2/system/filesys.h>
#include <vd2/system/math.h>
#include <vd2/Dita/services.h>
#include <at/atui/dialog.h>
#include "resource.h"
#include "oshelper.h"
#include "firmwaremanager.h"
#include "firmwaredetect.h"
#include "uimenu.h"
#include "uimenulist.h"

void ATUIScanForFirmware(VDGUIHandle hParent, ATFirmwareManager& fwmgr);

///////////////////////////////////////////////////////////////////////////

namespace {
	struct FirmwareItem : public vdrefcounted<IVDUITreeViewVirtualItem> {
		FirmwareItem(uint64 id, ATFirmwareType type, bool category, const wchar_t *text, const wchar_t *path)
			: mId(id)
			, mType(type)
			, mbCategory(category)
			, mText(text)
			, mPath(path)
			, mFlags(0)
		{
		}

		void *AsInterface(uint32 iid) {
			return NULL;
		}

		void GetText(VDStringW& s) const;

		const uint64 mId;
		const ATFirmwareType mType;
		VDUIProxyTreeViewControl::NodeRef mNode;
		VDStringW mText;
		VDStringW mPath;
		uint32 mFlags;
		bool mbCategory;
	};

	void FirmwareItem::GetText(VDStringW& s) const {
		if (mbCategory)
			s = mText;
		else
			s.sprintf(L"%ls (%ls)", mText.c_str(), mId < kATFirmwareId_Custom ? L"internal" : VDFileSplitPath(mPath.c_str()));
	}

	struct FirmwareItemComparer : public IVDUITreeViewVirtualItemComparer {
		virtual int Compare(IVDUITreeViewVirtualItem& x, IVDUITreeViewVirtualItem& y) const {
			return static_cast<const FirmwareItem&>(x).mText.comparei(static_cast<const FirmwareItem&>(y).mText);
		}
	};

	VDStringW BrowseForFirmware(VDDialogFrameW32 *parent) {
		return VDGetLoadFileName('ROMI', (VDGUIHandle)parent->GetWindowHandle(), L"Browse for ROM image", L"ROM image (*.rom)\0*.rom\0All files\0*.*\0", NULL);
	}
}

///////////////////////////////////////////////////////////////////////////

class ATUIDialogEditFirmwareSettings : public VDDialogFrameW32 {
public:
	ATUIDialogEditFirmwareSettings(FirmwareItem& item);

protected:
	bool OnLoaded();
	void OnDataExchange(bool write);

	FirmwareItem& mItem;
	VDUIProxyListView mOptionsView;
	uint32 mFlagCount;
};

ATUIDialogEditFirmwareSettings::ATUIDialogEditFirmwareSettings(FirmwareItem& item)
	: VDDialogFrameW32(IDD_FIRMWARE_EDIT)
	, mItem(item)
	, mFlagCount(0)
{
}

bool ATUIDialogEditFirmwareSettings::OnLoaded() {
	AddProxy(&mOptionsView, IDC_OPTIONS);

	mOptionsView.SetFullRowSelectEnabled(true);
	mOptionsView.SetItemCheckboxesEnabled(true);

	switch(mItem.mType) {
		case kATFirmwareType_KernelXL:
		case kATFirmwareType_KernelXEGS:
			mFlagCount = 1;
			mOptionsView.InsertItem(-1, L"OPTION key inverted (hold to enable BASIC)");
			break;

		default:
			mOptionsView.SetEnabled(false);
			break;
	}

	VDDialogFrameW32::OnLoaded();
	SetFocusToControl(IDC_NAME);
	return false;
}

void ATUIDialogEditFirmwareSettings::OnDataExchange(bool write) {
	ExchangeControlValueString(write, IDC_NAME, mItem.mText);

	if (write) {
		if (mFlagCount)
			mItem.mFlags = mOptionsView.IsItemChecked(0) ? 1 : 0;
		else
			mItem.mFlags = 0;
	} else {
		SetControlText(IDC_PATH, mItem.mPath.c_str());

		if (mFlagCount)
			mOptionsView.SetItemChecked(0, (mItem.mFlags & 1) != 0);
	}
}

///////////////////////////////////////////////////////////////////////////

class ATUIDialogFirmware : public VDDialogFrameW32 {
public:
	ATUIDialogFirmware(ATFirmwareManager& sim);

protected:
	bool OnLoaded();
	void OnDestroy();
	void OnDataExchange(bool write);
	bool OnCommand(uint32 id, uint32 extcode);
	void Add();
	void Remove();
	void EditSettings();
	void UpdateFirmwareItem(const FirmwareItem& item);

	void OnSelChanged(VDUIProxyTreeViewControl *sender, int idx);
	void OnItemDoubleClicked(VDUIProxyTreeViewControl *sender, bool *handled);
	void OnItemGetDisplayAttributes(VDUIProxyTreeViewControl *sender, VDUIProxyTreeViewControl::GetDispAttrEvent *event);
	void OnBeginEdit(VDUIProxyTreeViewControl *sender, VDUIProxyTreeViewControl::BeginEditEvent *event);
	void OnEndEdit(VDUIProxyTreeViewControl *sender, VDUIProxyTreeViewControl::EndEditEvent *event);

	ATFirmwareManager& mFwManager;

	vdrefptr<FirmwareItem> mpCategories[kATFirmwareTypeCount];

	VDUIProxyTreeViewControl mTreeView;

	VDDelegate mDelSelChanged;
	VDDelegate mDelDblClk;
	VDDelegate mDelItemGetDispAttr;
	VDDelegate mDelBeginEdit;
	VDDelegate mDelEndEdit;
};

ATUIDialogFirmware::ATUIDialogFirmware(ATFirmwareManager& fw)
	: VDDialogFrameW32(IDD_FIRMWARE)
	, mFwManager(fw)
{
}

bool ATUIDialogFirmware::OnLoaded() {
	AddProxy(&mTreeView, IDC_TREE);

	OnDataExchange(false);

	mTreeView.OnItemSelectionChanged() += mDelSelChanged.Bind(this, &ATUIDialogFirmware::OnSelChanged);
	mTreeView.OnItemDoubleClicked() += mDelDblClk.Bind(this, &ATUIDialogFirmware::OnItemDoubleClicked);
	mTreeView.OnItemGetDisplayAttributes() += mDelItemGetDispAttr.Bind(this, &ATUIDialogFirmware::OnItemGetDisplayAttributes);
	mTreeView.OnItemBeginEdit() += mDelBeginEdit.Bind(this, &ATUIDialogFirmware::OnBeginEdit);
	mTreeView.OnItemEndEdit() += mDelEndEdit.Bind(this, &ATUIDialogFirmware::OnEndEdit);

	SetFocusToControl(IDC_LIST);
	return true;
}

void ATUIDialogFirmware::OnDestroy() {
	mTreeView.Clear();

	VDDialogFrameW32::OnDestroy();
}

void ATUIDialogFirmware::OnDataExchange(bool write) {
	if (write) {
	} else {
		mTreeView.SetRedraw(false);
		mTreeView.Clear();

		struct Category {
			ATFirmwareType mType;
			const wchar_t *mpName;
		} kCategories[]={
			{ kATFirmwareType_Kernel800_OSA, L"400/800 Kernel ROMs (OS-A compatible)" },
			{ kATFirmwareType_Kernel800_OSB, L"400/800 Kernel ROMs (OS-B compatible)" },
			{ kATFirmwareType_KernelXL, L"XL/XE Kernel ROMs" },
			{ kATFirmwareType_KernelXEGS, L"XEGS Kernel ROMs" },
			{ kATFirmwareType_Kernel1200XL, L"1200XL Kernel ROMs" },
			{ kATFirmwareType_Kernel5200, L"5200 Kernel ROMs" },
			{ kATFirmwareType_Basic, L"Internal BASIC ROMs (XL/XE/XEGS)" },
			{ kATFirmwareType_U1MB, L"Ultimate1MB ROMs" },
			{ kATFirmwareType_MyIDE, L"MyIDE ROMs" },
			{ kATFirmwareType_MyIDE2, L"MyIDE-II ROMs" },
			{ kATFirmwareType_SIDE, L"SIDE ROMs" },
			{ kATFirmwareType_SIDE2, L"SIDE 2 ROMs" },
			{ kATFirmwareType_KMKJZIDE, L"KMK/JZ IDE ROMs" },
			{ kATFirmwareType_KMKJZIDE2, L"KMK/JZ IDE 2 (IDEPlus) main ROMs" },
			{ kATFirmwareType_KMKJZIDE2_SDX, L"KMK/JZ IDE 2 (IDEPlus) SDX ROMs" },
			{ kATFirmwareType_BlackBox, L"BlackBox ROMs" },
		};

		for(size_t i=0; i<vdcountof(kCategories); ++i) {
			const ATFirmwareType type = kCategories[i].mType;

			mpCategories[type] = new FirmwareItem(0, type, true, kCategories[i].mpName, L"");
			mpCategories[type]->mNode = mTreeView.AddVirtualItem(mTreeView.kNodeRoot, mTreeView.kNodeLast, mpCategories[type]);
		}

		typedef vdvector<ATFirmwareInfo> Firmwares;
		Firmwares firmwares;
		mFwManager.GetFirmwareList(firmwares);

		for(Firmwares::const_iterator it(firmwares.begin()), itEnd(firmwares.end());
			it != itEnd;
			++it)
		{
			if (!it->mbVisible)
				continue;

			if (mpCategories[it->mType]) {
				vdrefptr<FirmwareItem> item(new FirmwareItem(it->mId, it->mType, false, it->mName.c_str(), it->mPath.c_str()));
				item->mFlags = it->mFlags;
				item->mNode = mTreeView.AddVirtualItem(mpCategories[it->mType]->mNode, mTreeView.kNodeLast, item);
			}
		}

		for(size_t i=0; i<vdcountof(mpCategories); ++i) {
			if (mpCategories[i])
				mTreeView.SortChildren(mpCategories[i]->mNode, FirmwareItemComparer());
		}

		mTreeView.SetRedraw(true);
	}
}

bool ATUIDialogFirmware::OnCommand(uint32 id, uint32 extcode) {
	switch(id) {
		case IDC_ADD:
			Add();
			break;

		case IDC_REMOVE:
			Remove();
			break;

		case IDC_SETTINGS:
			EditSettings();
			break;

		case IDC_SCAN:
			ATUIScanForFirmware((VDGUIHandle)mhdlg, mFwManager);
			OnDataExchange(false);
			break;

	}

	return false;
}

void ATUIDialogFirmware::Add() {
	FirmwareItem *item = static_cast<FirmwareItem *>(mTreeView.GetSelectedVirtualItem());
	if (!item)
		return;

	ATFirmwareType type = item->mType;

	const VDStringW& path = BrowseForFirmware(this);

	if (path.empty())
		return;

	const uint64 id = ATGetFirmwareIdFromPath(path.c_str());

	vdrefptr<FirmwareItem> newItem(new FirmwareItem(id, type, false, VDFileSplitExtLeft(VDFileSplitPathRight(path)).c_str(), path.c_str()));

	// try to autodetect it
	try {
		VDFile f(path.c_str());

		sint64 size = f.size();

		if (ATFirmwareAutodetectCheckSize(size)) {
			uint32 size32 = (uint32)size;
			vdblock<char> buf(size32);

			f.read(buf.data(), size32);

			ATFirmwareInfo info;
			if (ATFirmwareAutodetect(buf.data(), size32, info)) {
				newItem->mText = info.mName;
				newItem->mFlags = info.mFlags;
			}
		}
	} catch(const MyError&) {
	}

	ATUIDialogEditFirmwareSettings dlg2(*newItem);
	if (dlg2.ShowDialog(this)) {
		newItem->mNode = mTreeView.AddVirtualItem(mpCategories[type]->mNode, mTreeView.kNodeLast, newItem);
		if (newItem->mNode) {
			UpdateFirmwareItem(*newItem);

			mTreeView.RefreshNode(newItem->mNode);
			mTreeView.SortChildren(mpCategories[type]->mNode, FirmwareItemComparer());
			mTreeView.MakeNodeVisible(newItem->mNode);
			mTreeView.SelectNode(newItem->mNode);
		}
	}
}

void ATUIDialogFirmware::Remove() {
	vdrefptr<FirmwareItem> item(static_cast<FirmwareItem *>(mTreeView.GetSelectedVirtualItem()));
	if (!item)
		return;

	if (item->mId < kATFirmwareId_Custom)
		return;

	mTreeView.DeleteItem(item->mNode);
	mFwManager.RemoveFirmware(item->mId);
}

void ATUIDialogFirmware::EditSettings() {
	vdrefptr<FirmwareItem> item(static_cast<FirmwareItem *>(mTreeView.GetSelectedVirtualItem()));
	if (!item)
		return;

	if (item->mId < kATFirmwareId_Custom)
		return;

	ATUIDialogEditFirmwareSettings dlg2(*item);
	VDStringW name(item->mText);
	if (dlg2.ShowDialog(this)) {
		mTreeView.RefreshNode(item->mNode);

		if (name.comparei(item->mText)) {
			mTreeView.SortChildren(mpCategories[item->mType]->mNode, FirmwareItemComparer());
			mTreeView.MakeNodeVisible(item->mNode);
			mTreeView.SelectNode(item->mNode);
		}

		UpdateFirmwareItem(*item);
	}
}

void ATUIDialogFirmware::UpdateFirmwareItem(const FirmwareItem& item) {
	ATFirmwareInfo info;
	info.mId = item.mId;
	info.mName = item.mText;
	info.mPath = item.mPath;
	info.mType = item.mType;
	info.mFlags = item.mFlags;
	mFwManager.AddFirmware(info);
}

void ATUIDialogFirmware::OnSelChanged(VDUIProxyTreeViewControl *sender, int idx) {
	FirmwareItem *pItem = static_cast<FirmwareItem *>(sender->GetSelectedVirtualItem());

	if (pItem) {
		EnableControl(IDC_ADD, true);
		EnableControl(IDC_REMOVE, !pItem->mbCategory && pItem->mId >= kATFirmwareId_Custom);
		EnableControl(IDC_SETTINGS, !pItem->mbCategory && pItem->mId >= kATFirmwareId_Custom);
	} else {
		EnableControl(IDC_ADD, false);
		EnableControl(IDC_REMOVE, false);
		EnableControl(IDC_SETTINGS, false);
	}
}

void ATUIDialogFirmware::OnItemDoubleClicked(VDUIProxyTreeViewControl *sender, bool *handled) {
	FirmwareItem *pItem = static_cast<FirmwareItem *>(sender->GetSelectedVirtualItem());

	if (pItem && pItem->mId >= kATFirmwareId_Custom) {
		EditSettings();
		*handled = true;
	}
}

void ATUIDialogFirmware::OnItemGetDisplayAttributes(VDUIProxyTreeViewControl *sender, VDUIProxyTreeViewControl::GetDispAttrEvent *event) {
	FirmwareItem *pItem = static_cast<FirmwareItem *>(event->mpItem);

	event->mbIsBold = pItem->mbCategory;
}

void ATUIDialogFirmware::OnBeginEdit(VDUIProxyTreeViewControl *sender, VDUIProxyTreeViewControl::BeginEditEvent *event) {
	FirmwareItem *pItem = static_cast<FirmwareItem *>(event->mpItem);

	if (!pItem || pItem->mId < kATFirmwareId_Custom)
		event->mbAllowEdit = false;
	else {
		event->mbOverrideText = true;
		event->mOverrideText = pItem->mText;
	}
}

void ATUIDialogFirmware::OnEndEdit(VDUIProxyTreeViewControl *sender, VDUIProxyTreeViewControl::EndEditEvent *event) {
	FirmwareItem *pItem = static_cast<FirmwareItem *>(event->mpItem);

	if (pItem && pItem->mId >= kATFirmwareId_Custom && event->mpNewText)
		pItem->mText = event->mpNewText;
}

void ATUIShowDialogFirmware(VDGUIHandle hParent, ATFirmwareManager& fw) {
	ATUIDialogFirmware dlg(fw);

	dlg.ShowDialog(hParent);
}
