//	Altirra - Atari 800/800XL/5200 emulator
//	Copyright (C) 2009-2012 Avery Lee
//
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#ifndef f_AT_SIOMANAGER_H
#define f_AT_SIOMANAGER_H

#include <at/atcore/devicesio.h>
#include "pokey.h"
#include "scheduler.h"

class ATCPUEmulator;
class ATCPUEmulatorMemory;
class ATSimulator;
class IATUIRenderer;
struct ATCPUHookNode;
class ATPokeyEmulator;

struct ATSIORequest {
	uint8	mDevice;
	uint8	mCommand;
	uint8	mMode;
	uint8	mTimeout;
	uint16	mAddress;
	uint16	mLength;
	uint16	mSector;
	uint8	mAUX[2];
};

class ATSIOManager final : public IATPokeySIODevice, public IATDeviceSIOManager, public IATSchedulerCallback {
	ATSIOManager(const ATSIOManager&) = delete;
	ATSIOManager& operator=(const ATSIOManager&) = delete;
public:
	ATSIOManager();
	~ATSIOManager();

	void Init(ATCPUEmulator *cpu, ATSimulator *sim);
	void Shutdown();

	void ColdReset();

	void ReinitHooks();
	void UninitHooks();

	bool TryAccelRequest(const ATSIORequest& req, bool isDSKINV);

public:
	virtual void PokeyAttachDevice(ATPokeyEmulator *pokey) override;
	virtual void PokeyWriteSIO(uint8 c, bool command, uint32 cyclesPerBit) override;
	virtual void PokeyBeginCommand() override;
	virtual void PokeyEndCommand() override;
	virtual void PokeySerInReady() override;

public:
	virtual void AddDevice(IATDeviceSIO *dev) override;
	virtual void RemoveDevice(IATDeviceSIO *dev) override;
	virtual void BeginCommand() override;
	virtual void SendData(const void *data, uint32 len, bool addChecksum) override;
	virtual void SendACK() override;
	virtual void SendNAK() override;
	virtual void SendComplete() override;
	virtual void SendError() override;
	virtual void ReceiveData(uint32 len) override;
	virtual void Delay(uint32 ticks) override;
	virtual void InsertFence(uint32 id) override;
	virtual void EndCommand() override;

public:
	virtual void OnScheduledEvent(uint32 id) override;

private:
	enum {
		kEventId_Delay = 1,
		kEventId_Send
	};

	uint8 OnHookDSKINV(uint16);
	uint8 OnHookSIOV(uint16);

	void ExecuteNextStep();
	void ShiftTransmitBuffer();

	ATCPUEmulator *mpCPU;
	ATCPUEmulatorMemory *mpMemory;
	ATSimulator *mpSim;
	IATUIRenderer *mpUIRenderer;
	ATScheduler *mpScheduler;
	ATPokeyEmulator *mpPokey;

	ATCPUHookNode *mpSIOVHook;
	ATCPUHookNode *mpDSKINVHook;

	uint32	mTransferLevel;		// Write pointer for accumulating send data.
	uint32	mTransferStart;		// Starting offset for current transfer.
	uint32	mTransferIndex;		// Next byte to send/receive for current transfer.
	uint32	mTransferEnd;		// Stopping offset for current transfer.
	uint32	mTransferCyclesPerBit;
	uint32	mTransferCyclesPerByte;
	bool	mbTransferSend;
	bool	mbCommandMode;
	uint8	mPollCount;
	uint32	mAccelBufferAddress;
	const ATDeviceSIORequest *mpAccelRequest;
	uint8	*mpAccelStatus;
	ATEvent *mpTransferEvent;
	IATDeviceSIO *mpActiveDevice;

	vdfastvector<IATDeviceSIO *> mSIODevices;

	enum StepType {
		kStepType_None,
		kStepType_Delay,
		kStepType_Send,
		kStepType_Receive,
		kStepType_Fence,
		kStepType_EndCommand
	};

	struct Step {
		StepType mType;
		union {
			uint32 mTransferLength;
			uint32 mFenceId;
			uint32 mDelayTicks;
		};

		uint32 mTransferCyclesPerBit;
		uint32 mTransferCyclesPerByte;
	};

	Step mCurrentStep;

	vdfastdeque<Step> mStepQueue;

	uint8 mTransferBuffer[65536];
};

#endif	// f_AT_SIOMANAGER_H
