//	Altirra - Atari 800/800XL/5200 emulator
//	Copyright (C) 2009-2014 Avery Lee
//
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include "stdafx.h"
#include <initializer_list>
#include <vd2/system/vdalloc.h>
#include <vd2/VDDisplay/textrenderer.h>
#include "uicontainer.h"
#include "uimanager.h"
#include "uilabel.h"
#include "uibutton.h"
#include "uisettingswindow.h"

///////////////////////////////////////////////////////////////////////////

void *ATUISetting::AsInterface(uint32 id) {
	if (id == ATUISetting::kTypeID)
		return static_cast<ATUISetting *>(this);

	return nullptr;
}

bool ATUISetting::IsDeferred() const {
	return false;
}

void ATUISetting::Read() {}
void ATUISetting::Write() {}

///////////////////////////////////////////////////////////////////////////

ATUIBoolSetting::ATUIBoolSetting(const wchar_t *name)
	: ATUISetting(name)
{
}

void *ATUIBoolSetting::AsInterface(uint32 id) {
	if (id == ATUIBoolSetting::kTypeID)
		return static_cast<ATUIBoolSetting *>(this);

	return ATUISetting::AsInterface(id);
}

void ATUIBoolSetting::Read() {
	if (mpGetter)
		mValue = mpGetter();
}

void ATUIBoolSetting::Write() {
	if (mpSetter)
		mpSetter(mValue);
}

void ATUIBoolSetting::SetGetter(const vdfunction<bool()>& fn) { mpGetter = fn; }
void ATUIBoolSetting::SetSetter(const vdfunction<void(bool)>& fn) { mpSetter = fn; }
void ATUIBoolSetting::SetImmediateSetter(const vdfunction<void(bool)>& fn) { mpImmediateSetter = fn; }

void ATUIBoolSetting::SetValue(bool value) {
	if (mValue == value)
		return;

	mValue = value;

	if (mpImmediateSetter)
		mpImmediateSetter(mValue);
}

///////////////////////////////////////////////////////////////////////////

ATUIEnumSetting::ATUIEnumSetting(const wchar_t *name, const ATUIEnumValue *values, uint32 n)
	: ATUISetting(name)
	, mValues(values, values + n)
	, mValueIndex(0)
{
}

ATUIEnumSetting::ATUIEnumSetting(const wchar_t *name, std::initializer_list<ATUIEnumValue> il)
	: ATUISetting(name)
	, mValues(il.begin(), il.end())
	, mValueIndex(0)
{
}

void *ATUIEnumSetting::AsInterface(uint32 id) {
	if (id == ATUIEnumSetting::kTypeID)
		return static_cast<ATUIEnumSetting *>(this);

	return ATUISetting::AsInterface(id);
}

bool ATUIEnumSetting::IsDeferred() const {
	return mpSetter;
}

void ATUIEnumSetting::Read() {
	if (mpGetter) {
		sint32 value = mpGetter();

		sint32 index = 0;
		for(const auto& entry : mValues) {
			if (entry.mValue == value) {
				mValueIndex = index;
				break;
			}

			++index;
		}
	}
}

void ATUIEnumSetting::Write() {
	if ((uint32)mValueIndex < mValues.size()) {
		if (mpSetter)
			mpSetter(mValues[mValueIndex].mValue);
	}
}

void ATUIEnumSetting::SetGetter(const vdfunction<sint32()>& fn) { mpGetter = fn; }
void ATUIEnumSetting::SetSetter(const vdfunction<void(sint32)>& fn) { mpSetter = fn; }
void ATUIEnumSetting::SetImmediateSetter(const vdfunction<void(sint32)>& fn) { mpImmediateSetter = fn; }

void ATUIEnumSetting::SetValue(sint32 value) {
	if (mValueIndex == value)
		return;

	mValueIndex = value;

	if ((uint32)mValueIndex < mValues.size()) {
		if (mpImmediateSetter)
			mpImmediateSetter(mValues[value].mValue);
	}
}

/////////////////////////////////////////////////////////////////////////

void *ATUISubScreenSetting::AsInterface(uint32 id) {
	if (id == ATUISubScreenSetting::kTypeID)
		return static_cast<ATUISubScreenSetting *>(this);

	return ATUISetting::AsInterface(id);
}

void ATUISubScreenSetting::BuildScreen(IATUISettingsScreen **screen) {
	mpBuilder(screen);
}

/////////////////////////////////////////////////////////////////////////

void *ATUIActionSetting::AsInterface(uint32 id) {
	if (id == ATUIActionSetting::kTypeID)
		return static_cast<ATUIActionSetting *>(this);

	return ATUISetting::AsInterface(id);
}

bool ATUIActionSetting::Activate() {
	return mpAction();
}

vdrefptr<ATUIFutureWithResult<bool>> ATUIActionSetting::ActivateAsync() {
	return mpAsyncAction();
}

/////////////////////////////////////////////////////////////////////////

class ATUISettingWindow final : public ATUIContainer {
public:
	enum {
		kActionActivate = kActionCustom,
		kActionTurnOn,
		kActionTurnOff,
	};

	ATUISettingWindow(ATUISetting *setting);

	void SetSelectedIndex(int index);

	void SetOnSubScreen(const vdfunction<void(ATUISubScreenSetting *)>& fn) {
		mpOnSubScreen = fn;
	}

	void SetOnAction(const vdfunction<void(ATUIActionSetting *)>& fn) {
		mpOnAction = fn;
	}

public:
	void OnCreate() override;
	void OnSize() override;
	void Paint(IVDDisplayRenderer&r, sint32 w, sint32 h) override;
	void OnMouseDownL(sint32 x, sint32 y) override;
	void OnMouseUpL(sint32 x, sint32 y) override;
	void OnMouseLeave() override;
	void OnActionStart(uint32 id) override;

private:
	void OnPrev(ATUIButton *);
	void OnNext(ATUIButton *);
	void UpdateLabels();

	vdrefptr<IVDDisplayFont> mpFont;
	vdrefptr<ATUILabel> mpLabelName;
	vdrefptr<ATUILabel> mpLabelSetting;
	vdrefptr<ATUIButton> mpButtonPrev;
	vdrefptr<ATUIButton> mpButtonNext;

	ATUISetting *mpSetting;
	ATUIBoolSetting *mpBoolSetting;
	ATUIEnumSetting *mpEnumSetting;
	ATUIActionSetting *mpActionSetting;
	ATUISubScreenSetting *mpSubScreenSetting;

	sint32 mSelectedIndex;
	sint32 mMinVal;
	sint32 mMaxVal;
	bool mbDirty;
	bool mbMouseActivatePending;

	vdfunction<void(ATUISubScreenSetting *)> mpOnSubScreen;
	vdfunction<void(ATUIActionSetting *)> mpOnAction;
};

ATUISettingWindow::ATUISettingWindow(ATUISetting *setting)
	: mpSetting(setting)
	, mpBoolSetting(vdpoly_cast<ATUIBoolSetting *>(setting))
	, mpEnumSetting(vdpoly_cast<ATUIEnumSetting *>(setting))
	, mpActionSetting(vdpoly_cast<ATUIActionSetting *>(setting))
	, mpSubScreenSetting(vdpoly_cast<ATUISubScreenSetting *>(setting))
	, mbDirty(false)
	, mbMouseActivatePending(false)
{
	SetAlphaFillColor(0);

	mSelectedIndex = 0;
	mMinVal = 0;
	mMaxVal = 0;

	if (mpEnumSetting)
		mMaxVal = (sint32)mpEnumSetting->GetValueCount() - 1;

	if (mpBoolSetting)
		mMaxVal = 1;
}

void ATUISettingWindow::SetSelectedIndex(int index) {
	if (index < mMinVal)
		index = mMinVal;

	if (index > mMaxVal)
		index = mMaxVal;

	if (mSelectedIndex == index)
		return;

	mSelectedIndex = index;

	if (mpEnumSetting)
		mpEnumSetting->SetValue(index);
	else if (mpBoolSetting)
		mpBoolSetting->SetValue(index != 0);

	if (mpSetting->IsDeferred() && !mbDirty) {
		mbDirty = true;

		if (mpLabelName)
			mpLabelName->SetTextColor(0xFF0000);

		if (mpLabelSetting)
			mpLabelSetting->SetTextColor(0xFF0000);
	}

	UpdateLabels();
}

void ATUISettingWindow::OnCreate() {
	mpFont = mpManager->GetThemeFont(kATUIThemeFont_Default);

	mpLabelName = new ATUILabel;
	mpLabelName->SetHitTransparent(true);
	mpLabelName->SetAlphaFillColor(0);
	mpLabelName->SetTextColor(0xFFFFFF);
	mpLabelName->SetText(mpSetting->GetName());

	AddChild(mpLabelName);

	if (mpEnumSetting || mpBoolSetting) {
		mpButtonPrev = new ATUIButton;
		mpButtonPrev->SetFrameEnabled(false);
		mpButtonPrev->SetTextColor(0xFFFFFF);
		mpButtonPrev->SetStockImage(kATUIStockImageIdx_ButtonLeft);
		mpButtonPrev->OnActivatedEvent() = ATBINDCALLBACK(this, &ATUISettingWindow::OnPrev);
		AddChild(mpButtonPrev);
	}

	if (mpEnumSetting || mpBoolSetting) {
		mpLabelSetting = new ATUILabel;
		mpLabelSetting->SetHitTransparent(true);
		mpLabelSetting->SetAlphaFillColor(0);
		mpLabelSetting->SetTextColor(0xFFFFFF);
		mpLabelSetting->SetTextAlign(ATUILabel::kAlignCenter);
		AddChild(mpLabelSetting);
	}

	if (mpEnumSetting || mpSubScreenSetting || mpBoolSetting) {
		mpButtonNext = new ATUIButton;
		mpButtonNext->SetTextColor(0xFFFFFF);
		mpButtonNext->SetFrameEnabled(false);
		mpButtonNext->SetStockImage(kATUIStockImageIdx_ButtonRight);
		mpButtonNext->OnActivatedEvent() = ATBINDCALLBACK(this, &ATUISettingWindow::OnNext);
		AddChild(mpButtonNext);
	}

	OnSize();

	if (mpButtonPrev) {
		BindAction(kATUIVK_Left, ATUIButton::kActionActivate, 0, mpButtonPrev->GetInstanceId());
		BindAction(kATUIVK_UILeft, ATUIButton::kActionActivate, 0, mpButtonPrev->GetInstanceId());
	}

	if (mpButtonNext) {
		BindAction(kATUIVK_Right, ATUIButton::kActionActivate, 0, mpButtonNext->GetInstanceId());
		BindAction(kATUIVK_UIRight, ATUIButton::kActionActivate, 0, mpButtonNext->GetInstanceId());
	}

	BindAction(kATUIVK_UIAccept, kActionActivate);

	if (mpBoolSetting) {
		BindAction(kATUIVK_Left, kActionTurnOff);
		BindAction(kATUIVK_UILeft, kActionTurnOff);
		BindAction(kATUIVK_Right, kActionTurnOn);
		BindAction(kATUIVK_UIRight, kActionTurnOn);
	}

	mSelectedIndex = 0;

	if (mpEnumSetting)
		mSelectedIndex = mpEnumSetting->GetValue();
	else if (mpBoolSetting)
		mSelectedIndex = mpBoolSetting->GetValue();

	UpdateLabels();
}

void ATUISettingWindow::OnSize() {
	const vdsize32& sz = GetArea().size();

	if (mpLabelName)
		mpLabelName->SetArea(vdrect32(0, 0, sz.w/2, sz.h));

	if (mpButtonPrev)
		mpButtonPrev->SetArea(vdrect32(sz.w/2, 0, sz.w/2 + sz.h, sz.h));

	if (mpLabelSetting)
		mpLabelSetting->SetArea(vdrect32(sz.w/2 + sz.h, 0, sz.w - sz.h, sz.h));

	if (mpButtonNext)
		mpButtonNext->SetArea(vdrect32(sz.w - sz.h, 0, sz.w, sz.h));
}

void ATUISettingWindow::Paint(IVDDisplayRenderer& r, sint32 w, sint32 h) {
	ATUIContainer::Paint(r, w, h);
}

void ATUISettingWindow::OnMouseDownL(sint32 x, sint32 y) {
	mbMouseActivatePending = true;
}

void ATUISettingWindow::OnMouseUpL(sint32 x, sint32 y) {
	if (mbMouseActivatePending) {
		if (mpBoolSetting || mpSubScreenSetting || mpActionSetting || mbDirty)
			OnActionStart(kActionActivate);

		mbMouseActivatePending = false;
	}
}

void ATUISettingWindow::OnMouseLeave() {
	mbMouseActivatePending = false;
}

void ATUISettingWindow::OnActionStart(uint32 id) {
	switch(id) {
		case kActionActivate:
			if (mpSetting->IsDeferred() && mbDirty) {
				mbDirty = false;

				mpSetting->Write();

				if (mpLabelName)
					mpLabelName->SetTextColor(0xFFFFFF);

				if (mpLabelSetting)
					mpLabelSetting->SetTextColor(0xFFFFFF);
			} else if (mpBoolSetting)
				SetSelectedIndex(mSelectedIndex ^ 1);
			else if (mpSubScreenSetting || mpActionSetting)
				OnNext(nullptr);
			break;

		case kActionTurnOn:
			SetSelectedIndex(1);
			break;

		case kActionTurnOff:
			SetSelectedIndex(0);
			break;
	}

	ATUIContainer::OnActionStart(id);
}

void ATUISettingWindow::OnPrev(ATUIButton *) {
	if (mpBoolSetting)
		SetSelectedIndex(mSelectedIndex ^ 1);
	else
		SetSelectedIndex(mSelectedIndex - 1);
}

void ATUISettingWindow::OnNext(ATUIButton *) {
	if (mpActionSetting) {
		if (mpOnAction)
			mpOnAction(mpActionSetting);
	} else if (mpSubScreenSetting) {
		if (mpOnSubScreen)
			mpOnSubScreen(mpSubScreenSetting);
	} else if (mpBoolSetting) {
		SetSelectedIndex(mSelectedIndex ^ 1);
	} else {
		SetSelectedIndex(mSelectedIndex + 1);
	}
}

void ATUISettingWindow::UpdateLabels() {
	if (mpLabelSetting) {
		if (mpEnumSetting)
			mpLabelSetting->SetText(mpEnumSetting->GetValueName(mSelectedIndex));
		else if (mpBoolSetting)
			mpLabelSetting->SetText(mSelectedIndex ? L"On" : L"Off");
		else
			mpLabelSetting->SetTextF(L"%d", mSelectedIndex);
	}

	if (mpButtonPrev && !mpBoolSetting)
		mpButtonPrev->SetVisible(mSelectedIndex > mMinVal);

	if (mpButtonNext && mpEnumSetting)
		mpButtonNext->SetVisible(mSelectedIndex < mMaxVal);
}

///////////////////////////////////////////////////////////////////////////

ATUISettingsWindow::ATUISettingsWindow() {
	SetAlphaFillColor(0xE0202020);
	SetCursorImage(kATUICursorImage_Arrow);

	BindAction(kATUIVK_Up, kActionUp);
	BindAction(kATUIVK_UIUp, kActionUp);
	BindAction(kATUIVK_Down, kActionDown);
	BindAction(kATUIVK_UIDown, kActionDown);
	BindAction(kATUIVK_UIReject, kActionCancel);
	BindAction(kATUIVK_Escape, kActionCancel);
}

void ATUISettingsWindow::SetSettingsScreen(IATUISettingsScreen *screen) {
	if (mpCurrentScreen == screen)
		return;

	// tear down current screen and settings
	DestroyScreen();

	mpCurrentScreen = screen;

	BuildScreen();
}

void ATUISettingsWindow::AddSetting(ATUISetting *setting) {
	mSettings.push_back(setting);
}

void ATUISettingsWindow::SetOnDestroy(const vdfunction<void()>& fn) {
	mpOnDestroy = fn;
}

void ATUISettingsWindow::SetSelectedIndex(int index) {
	if (index < 0)
		index = -1;
	else if ((unsigned)index >= mSettings.size())
		index = mSettings.size() - 1;

	if (mSelectedIndex == index)
		return;

	mSelectedIndex = index;

	if (mpSelectionFill) {
		if (index < 0) {
			mpSelectionFill->SetVisible(false);
		} else {
			mpSelectionFill->SetVisible(true);
			mpSelectionFill->SetArea(vdrect32(0, index * mRowHeight, GetArea().width(), index * mRowHeight + mRowHeight));
		}
	}

	if (index < 0)
		Focus();
	else
		mSettingWindows[index]->Focus();
}

void ATUISettingsWindow::OnCreate() {
	VDDisplayFontMetrics metrics;
	mpManager->GetThemeFont(kATUIThemeFont_Default)->GetMetrics(metrics);
	mRowHeight = metrics.mAscent + metrics.mDescent + 4;

	// if we currently have a screen, build it now
	BuildScreen();

	mpSelectionFill = new ATUILabel;
	mpSelectionFill->SetFillColor(0xFF0034D0);
	AddChild(mpSelectionFill);
	SendToBack(mpSelectionFill);

	mSelectedIndex = -1;
	SetSelectedIndex(0);

	mpManager->AddTrackingWindow(this);
}

void ATUISettingsWindow::OnDestroy() {
	mpPendingResult.clear();

	SetSettingsScreen(nullptr);

	while(!mScreenStack.empty()) {
		mScreenStack.back().mpScreen->Release();
		mScreenStack.pop_back();
	}

	if (mpOnDestroy)
		mpOnDestroy();

	ATUIContainer::OnDestroy();
}

void ATUISettingsWindow::OnSetFocus() {
	if ((uint32)mSelectedIndex < mSettingWindows.size())
		mSettingWindows[mSelectedIndex]->Focus();
}

void ATUISettingsWindow::OnMouseDownL(sint32 x, sint32 y) {
	Focus();
}

void ATUISettingsWindow::OnActionStart(uint32 trid) {
	OnActionRepeat(trid);
}

void ATUISettingsWindow::OnActionRepeat(uint32 trid) {
	switch(trid) {
		case kActionUp:
			if (mSelectedIndex > 0)
				SetSelectedIndex(mSelectedIndex - 1);
			break;

		case kActionDown:
			SetSelectedIndex(mSelectedIndex + 1);
			break;

		case kActionCancel:
			if (mScreenStack.empty()) {
				Destroy();
			} else {
				auto stackedScreen = mScreenStack.back();
				mScreenStack.pop_back();

				SetSettingsScreen(stackedScreen.mpScreen);
				SetSelectedIndex(stackedScreen.mSelIndex);
				stackedScreen.mpScreen->Release();
			}
			break;
	}
}

void ATUISettingsWindow::OnActionStop(uint32 trid) {
}

void ATUISettingsWindow::OnTrackCursorChanges(ATUIWidget *w) {
	sint32 index = -1;

	if (w && w != this) {
		vdpoint32 pt;

		if (TranslateScreenPtToClientPt(w->TranslateClientPtToScreenPt(vdpoint32(0, 0)), pt)) {
			index = pt.y / mRowHeight;

			if ((uint32)index >= mSettingWindows.size())
				index = -1;
		}
	}

	SetSelectedIndex(index);
}

void ATUISettingsWindow::DestroyScreen() {
	while(!mSettings.empty()) {
		auto *p = mSettings.back();
		mSettings.pop_back();

		delete p;
	}

	while(!mSettingWindows.empty()) {
		auto *p = mSettingWindows.back();
		mSettingWindows.pop_back();

		RemoveChild(p);

		p->Release();
	}
}

void ATUISettingsWindow::BuildScreen() {
	if (!mpManager || !mpCurrentScreen)
		return;

	DestroyScreen();

	mpCurrentScreen->BuildSettings(this);

	int y = 0;
	for(ATUISetting *s : mSettings) {
		s->Read();

		vdrefptr<ATUISettingWindow> w(new ATUISettingWindow(s));

		w->SetOnSubScreen([this](ATUISubScreenSetting *s) { this->OnSubScreenActivated(s); });
		w->SetOnAction([this](ATUIActionSetting *s) { this->OnAction(s); });
	
		w->SetArea(vdrect32(0, y, GetArea().width(), y+mRowHeight));
		y += mRowHeight;
		AddChild(w);

		mSettingWindows.push_back(w);
		w.release();
	}

	mSelectedIndex = -1;
	SetSelectedIndex(0);
}

void ATUISettingsWindow::OnSubScreenActivated(ATUISubScreenSetting *s) {
	vdrefptr<IATUISettingsScreen> newScreen;

	s->BuildScreen(~newScreen);

	if (mpCurrentScreen) {
		auto& stackedEntry = mScreenStack.push_back();
		stackedEntry.mpScreen = mpCurrentScreen.release();
		stackedEntry.mSelIndex = mSelectedIndex;
	}

	SetSettingsScreen(newScreen);
}

void ATUISettingsWindow::OnAction(ATUIActionSetting *s) {
	if (s->IsAsync()) {
		auto p = vdmakerefptr(this);

		ATUIPushStep([p]() { p->OnAsyncActionCompleted(); });
		mpPendingResult = s->ActivateAsync();
		ATUIPushStep(mpPendingResult->GetStep());
	} else {
		if (s->Activate())
			Destroy();
	}
}

void ATUISettingsWindow::OnAsyncActionCompleted() {
	if (mpPendingResult) {
		bool result = mpPendingResult->GetResult();
		mpPendingResult.clear();

		if (result)
			Destroy();
	}
}

///////////////////////////////////////////////////////////////////////////

void ATCreateUISettingsWindow(ATUISettingsWindow **pp) {
	ATUISettingsWindow *p = new ATUISettingsWindow;
	p->AddRef();
	*pp = p;
}
