#include "stdafx.h"
#include <vd2/system/cpuaccel.h>
#include <vd2/system/math.h>
#include <vd2/system/win32/intrin.h>
#include "audiofilters.h"

extern "C" VDALIGN(16) const float gVDCaptureAudioResamplingKernel[32][8] = {
	{+0x0000,+0x0000,+0x0000,+0x4000,+0x0000,+0x0000,+0x0000,+0x0000 },
	{-0x000a,+0x0052,-0x0179,+0x3fe2,+0x019f,-0x005b,+0x000c,+0x0000 },
	{-0x0013,+0x009c,-0x02cc,+0x3f86,+0x0362,-0x00c0,+0x001a,+0x0000 },
	{-0x001a,+0x00dc,-0x03f9,+0x3eef,+0x054a,-0x012c,+0x002b,+0x0000 },
	{-0x001f,+0x0113,-0x0500,+0x3e1d,+0x0753,-0x01a0,+0x003d,+0x0000 },
	{-0x0023,+0x0141,-0x05e1,+0x3d12,+0x097c,-0x021a,+0x0050,-0x0001 },
	{-0x0026,+0x0166,-0x069e,+0x3bd0,+0x0bc4,-0x029a,+0x0066,-0x0001 },
	{-0x0027,+0x0182,-0x0738,+0x3a5a,+0x0e27,-0x031f,+0x007d,-0x0002 },
	{-0x0028,+0x0197,-0x07b0,+0x38b2,+0x10a2,-0x03a7,+0x0096,-0x0003 },
	{-0x0027,+0x01a5,-0x0807,+0x36dc,+0x1333,-0x0430,+0x00af,-0x0005 },
	{-0x0026,+0x01ab,-0x083f,+0x34db,+0x15d5,-0x04ba,+0x00ca,-0x0007 },
	{-0x0024,+0x01ac,-0x085b,+0x32b3,+0x1886,-0x0541,+0x00e5,-0x0008 },
	{-0x0022,+0x01a6,-0x085d,+0x3068,+0x1b40,-0x05c6,+0x0101,-0x000b },
	{-0x001f,+0x019c,-0x0846,+0x2dfe,+0x1e00,-0x0644,+0x011c,-0x000d },
	{-0x001c,+0x018e,-0x0819,+0x2b7a,+0x20c1,-0x06bb,+0x0136,-0x0010 },
	{-0x0019,+0x017c,-0x07d9,+0x28e1,+0x2380,-0x0727,+0x014f,-0x0013 },
	{-0x0016,+0x0167,-0x0788,+0x2637,+0x2637,-0x0788,+0x0167,-0x0016 },
	{-0x0013,+0x014f,-0x0727,+0x2380,+0x28e1,-0x07d9,+0x017c,-0x0019 },
	{-0x0010,+0x0136,-0x06bb,+0x20c1,+0x2b7a,-0x0819,+0x018e,-0x001c },
	{-0x000d,+0x011c,-0x0644,+0x1e00,+0x2dfe,-0x0846,+0x019c,-0x001f },
	{-0x000b,+0x0101,-0x05c6,+0x1b40,+0x3068,-0x085d,+0x01a6,-0x0022 },
	{-0x0008,+0x00e5,-0x0541,+0x1886,+0x32b3,-0x085b,+0x01ac,-0x0024 },
	{-0x0007,+0x00ca,-0x04ba,+0x15d5,+0x34db,-0x083f,+0x01ab,-0x0026 },
	{-0x0005,+0x00af,-0x0430,+0x1333,+0x36dc,-0x0807,+0x01a5,-0x0027 },
	{-0x0003,+0x0096,-0x03a7,+0x10a2,+0x38b2,-0x07b0,+0x0197,-0x0028 },
	{-0x0002,+0x007d,-0x031f,+0x0e27,+0x3a5a,-0x0738,+0x0182,-0x0027 },
	{-0x0001,+0x0066,-0x029a,+0x0bc4,+0x3bd0,-0x069e,+0x0166,-0x0026 },
	{-0x0001,+0x0050,-0x021a,+0x097c,+0x3d12,-0x05e1,+0x0141,-0x0023 },
	{+0x0000,+0x003d,-0x01a0,+0x0753,+0x3e1d,-0x0500,+0x0113,-0x001f },
	{+0x0000,+0x002b,-0x012c,+0x054a,+0x3eef,-0x03f9,+0x00dc,-0x001a },
	{+0x0000,+0x001a,-0x00c0,+0x0362,+0x3f86,-0x02cc,+0x009c,-0x0013 },
	{+0x0000,+0x000c,-0x005b,+0x019f,+0x3fe2,-0x0179,+0x0052,-0x000a },
};

extern "C" VDALIGN(16) const sint16 gATAudioResamplingKernel63To44[65][64] = {
{9,10,-86,-74,249,266,-486,-680,761,1444,-971,-2691,940,4547,-384,-7089,-1087,10317,4006,-14122,-9069,18286,17314,-22484,-956,823,1674,-919,-3214,983,10339,15375,10339,983,-3214,-919,1674,823,-956,-702,17314,18286,-9069,-14122,4006,10317,-1087,-7089,-384,4547,940,-2691,-971,1444,761,-680,-486,266,249,-74,-86,10,9,0},
{9,10,-84,-77,244,273,-473,-691,734,1458,-921,-2702,855,4547,-251,-7059,-1281,10230,4272,-13935,-9412,17937,17728,-21880,-970,791,1687,-863,-3218,860,10198,15374,10478,1108,-3209,-975,1659,855,-941,-720,16888,18627,-8718,-14303,3736,10400,-891,-7114,-518,4545,1025,-2677,-1021,1430,788,-669,-499,259,254,-71,-88,9,10,0},
{9,11,-83,-80,238,279,-460,-701,707,1471,-871,-2713,771,4545,-119,-7026,-1474,10138,4535,-13741,-9748,17581,18130,-21267,-984,759,1699,-806,-3219,738,10057,15369,10616,1234,-3202,-1031,1644,886,-925,-739,16451,18960,-8361,-14477,3462,10477,-693,-7137,-652,4540,1110,-2662,-1071,1415,814,-658,-512,252,259,-68,-89,8,10,0},
{8,12,-81,-82,233,285,-447,-711,680,1482,-820,-2721,686,4540,11,-6989,-1664,10040,4793,-13541,-10077,17217,18521,-20647,-997,727,1711,-750,-3219,618,9915,15363,10753,1361,-3194,-1088,1627,917,-909,-757,16002,19285,-7998,-14644,3184,10549,-494,-7155,-787,4532,1196,-2646,-1120,1398,841,-646,-524,245,264,-65,-91,7,10,0},
{8,12,-79,-85,227,291,-434,-720,652,1493,-770,-2729,602,4533,142,-6949,-1853,9938,5048,-13335,-10399,16846,18899,-20019,-1009,694,1721,-694,-3217,499,9771,15354,10888,1490,-3184,-1144,1610,948,-892,-774,15542,19601,-7628,-14805,2904,10615,-293,-7170,-922,4522,1281,-2628,-1170,1381,867,-633,-536,238,269,-61,-93,7,11,0},
{8,13,-77,-87,222,296,-421,-729,625,1503,-720,-2734,518,4523,271,-6905,-2039,9831,5298,-13123,-10714,16468,19264,-19385,-1021,661,1731,-638,-3213,382,9626,15341,11023,1620,-3171,-1200,1592,979,-875,-792,15071,19908,-7252,-14958,2620,10676,-90,-7180,-1058,4509,1366,-2608,-1219,1363,893,-620,-549,230,274,-58,-94,6,11,0},
{7,13,-75,-90,216,302,-407,-737,597,1512,-669,-2738,434,4511,400,-6858,-2223,9720,5544,-12905,-11020,16083,19617,-18743,-1033,629,1739,-582,-3208,267,9481,15326,11156,1752,-3157,-1256,1572,1009,-857,-809,14589,20207,-6870,-15104,2333,10732,112,-7187,-1194,4494,1450,-2587,-1268,1344,919,-606,-561,222,279,-54,-96,5,11,0},
{7,14,-74,-92,211,307,-394,-744,570,1520,-619,-2741,351,4496,527,-6807,-2404,9604,5785,-12681,-11320,15692,19958,-18095,-1043,596,1746,-527,-3201,153,9335,15308,11287,1885,-3141,-1311,1552,1039,-839,-826,14096,20496,-6482,-15242,2043,10783,317,-7190,-1330,4476,1535,-2564,-1317,1324,944,-592,-572,214,284,-51,-97,4,12,0},
{7,14,-72,-94,205,311,-380,-752,542,1527,-569,-2742,268,4480,653,-6753,-2583,9483,6022,-12452,-11611,15294,20286,-17442,-1053,563,1753,-471,-3192,40,9187,15288,11417,2019,-3123,-1367,1530,1069,-820,-842,13593,20775,-6088,-15374,1750,10827,523,-7190,-1467,4456,1620,-2539,-1365,1303,969,-578,-584,206,288,-47,-99,3,12,0},
{6,15,-70,-97,199,316,-366,-758,514,1533,-519,-2741,186,4460,779,-6696,-2760,9358,6255,-12218,-11894,14891,20601,-16783,-1063,529,1758,-416,-3182,-69,9040,15264,11546,2155,-3104,-1422,1508,1098,-800,-858,13079,21045,-5688,-15497,1455,10867,730,-7185,-1604,4433,1704,-2513,-1413,1281,994,-563,-595,197,293,-43,-100,2,12,0},
{6,15,-68,-99,194,320,-353,-764,486,1539,-469,-2739,104,4439,902,-6635,-2934,9229,6482,-11978,-12170,14482,20904,-16119,-1071,496,1763,-361,-3170,-178,8891,15238,11672,2291,-3082,-1477,1485,1127,-780,-874,12556,21306,-5284,-15613,1157,10900,938,-7176,-1740,4408,1788,-2485,-1461,1258,1018,-547,-607,188,297,-39,-102,1,13,0},
{6,16,-66,-100,188,324,-339,-770,459,1543,-419,-2736,23,4415,1025,-6572,-3105,9096,6705,-11733,-12437,14067,21193,-15450,-1080,463,1766,-306,-3157,-285,8742,15210,11798,2429,-3059,-1532,1460,1155,-759,-889,12022,21556,-4874,-15721,857,10928,1147,-7164,-1877,4379,1871,-2456,-1508,1235,1043,-531,-617,179,301,-35,-103,0,13,0},
{6,16,-64,-102,182,328,-325,-775,431,1546,-370,-2731,-57,4389,1146,-6505,-3274,8958,6923,-11483,-12696,13648,21469,-14777,-1087,430,1768,-252,-3142,-390,8591,15179,11921,2568,-3033,-1586,1435,1183,-738,-904,11479,21796,-4459,-15822,554,10950,1356,-7147,-2013,4349,1954,-2425,-1554,1210,1066,-515,-628,170,305,-31,-104,0,13,0},
{5,16,-63,-104,176,332,-311,-780,403,1549,-320,-2724,-137,4361,1266,-6435,-3440,8817,7136,-11229,-12947,13223,21733,-14099,-1094,397,1770,-197,-3125,-494,8440,15143,12043,2708,-3005,-1640,1409,1211,-716,-919,10927,22025,-4039,-15914,249,10966,1566,-7126,-2150,4315,2037,-2393,-1601,1185,1090,-498,-639,161,309,-27,-106,-1,14,0},
{5,17,-61,-106,170,335,-297,-784,375,1551,-271,-2716,-217,4331,1384,-6362,-3602,8671,7343,-10970,-13189,12794,21983,-13419,-1100,363,1771,-144,-3107,-596,8289,15107,12164,2849,-2976,-1694,1382,1238,-694,-933,10366,22244,-3615,-15998,-56,10977,1776,-7102,-2286,4280,2119,-2359,-1646,1159,1113,-480,-649,151,313,-23,-107,-2,14,0},
{5,17,-59,-107,164,338,-283,-788,348,1552,-222,-2707,-295,4298,1500,-6286,-3762,8522,7546,-10706,-13423,12360,22219,-12735,-1106,330,1770,-90,-3088,-696,8138,15069,12283,2991,-2945,-1747,1353,1264,-671,-947,9795,22453,-3186,-16074,-365,10981,1987,-7073,-2421,4241,2201,-2323,-1691,1131,1135,-463,-659,141,317,-18,-108,-4,14,0},
{5,17,-57,-109,158,341,-270,-791,320,1552,-173,-2696,-373,4264,1615,-6207,-3919,8369,7743,-10438,-13649,11922,22443,-12048,-1111,297,1769,-37,-3067,-794,7986,15026,12401,3134,-2912,-1800,1324,1290,-648,-960,9217,22650,-2753,-16142,-675,10980,2198,-7040,-2557,4200,2282,-2286,-1736,1103,1158,-444,-669,131,321,-14,-109,-5,14,0},
{4,17,-55,-110,153,344,-256,-794,293,1551,-125,-2684,-451,4227,1729,-6125,-4073,8213,7935,-10167,-13866,11480,22653,-11359,-1115,264,1767,14,-3044,-890,7833,14981,12515,3278,-2876,-1852,1294,1316,-624,-973,8629,22836,-2316,-16201,-986,10972,2409,-7004,-2692,4156,2362,-2247,-1780,1074,1179,-426,-678,121,324,-10,-110,-6,15,0},
{4,18,-53,-111,147,346,-242,-796,265,1549,-77,-2671,-527,4189,1840,-6040,-4224,8053,8121,-9891,-14074,11034,22850,-10667,-1119,231,1763,66,-3021,-985,7680,14934,12630,3423,-2839,-1904,1263,1341,-599,-985,8034,23011,-1876,-16252,-1299,10959,2620,-6963,-2826,4110,2442,-2207,-1823,1045,1201,-407,-687,110,327,-5,-111,-7,15,0},
{4,18,-51,-112,141,348,-228,-798,238,1547,-29,-2656,-602,4148,1950,-5953,-4371,7889,8302,-9612,-14273,10586,23033,-9975,-1122,198,1759,118,-2996,-1078,7527,14885,12742,3568,-2800,-1955,1231,1365,-575,-997,7431,23175,-1432,-16295,-1613,10939,2831,-6918,-2959,4061,2520,-2166,-1866,1014,1221,-387,-696,99,331,0,-112,-9,15,0},
{4,18,-50,-113,135,350,-214,-799,211,1543,17,-2640,-677,4105,2058,-5863,-4515,7723,8478,-9329,-14463,10133,23203,-9280,-1124,165,1754,169,-2969,-1168,7373,14832,12851,3715,-2759,-2006,1198,1389,-549,-1009,6820,23327,-984,-16329,-1927,10914,3042,-6869,-3092,4010,2598,-2122,-1908,983,1242,-367,-704,89,334,4,-113,-10,16,0},
{3,18,-48,-114,129,352,-201,-800,184,1539,64,-2622,-751,4061,2165,-5771,-4656,7553,8648,-9042,-14645,9679,23360,-8585,-1126,132,1748,220,-2941,-1257,7219,14777,12958,3862,-2715,-2056,1165,1413,-524,-1020,6202,23467,-533,-16354,-2243,10882,3252,-6816,-3224,3956,2676,-2078,-1950,951,1262,-347,-713,78,336,8,-114,-11,16,0},
{3,18,-46,-115,123,353,-187,-801,157,1534,110,-2603,-823,4014,2269,-5676,-4793,7380,8812,-8753,-14818,9221,23503,-7890,-1128,100,1742,270,-2912,-1344,7065,14719,13064,4010,-2670,-2105,1130,1435,-498,-1030,5578,23595,-80,-16371,-2559,10845,3462,-6759,-3355,3900,2752,-2031,-1990,918,1281,-326,-721,66,339,13,-115,-13,16,0},
{3,18,-44,-116,117,355,-173,-801,130,1528,156,-2583,-895,3966,2371,-5578,-4927,7204,8970,-8460,-14981,8761,23633,-7194,-1128,67,1734,319,-2882,-1429,6911,14659,13168,4159,-2623,-2154,1094,1458,-471,-1040,4946,23712,375,-16379,-2876,10801,3671,-6698,-3486,3841,2827,-1984,-2030,884,1300,-305,-728,55,342,18,-116,-14,16,1},
{3,18,-42,-117,111,356,-160,-800,104,1522,202,-2562,-966,3916,2472,-5478,-5057,7026,9123,-8165,-15136,8299,23749,-6498,-1128,35,1725,368,-2851,-1513,6757,14597,13270,4309,-2574,-2202,1058,1479,-444,-1050,4308,23816,833,-16378,-3193,10751,3880,-6633,-3615,3779,2902,-1934,-2069,849,1318,-283,-735,43,344,23,-116,-16,17,1},
{3,18,-40,-117,106,357,-146,-799,78,1514,247,-2539,-1035,3864,2570,-5376,-5184,6844,9270,-7867,-15282,7835,23852,-5803,-1128,3,1716,417,-2818,-1594,6602,14531,13368,4459,-2522,-2249,1021,1500,-416,-1059,3664,23908,1294,-16368,-3510,10695,4088,-6564,-3743,3715,2975,-1884,-2107,814,1335,-262,-742,32,346,28,-117,-17,17,1},
{2,18,-39,-118,100,357,-133,-798,52,1506,291,-2516,-1104,3810,2667,-5272,-5307,6661,9411,-7566,-15418,7370,23941,-5109,-1126,-28,1705,465,-2784,-1673,6447,14463,13466,4609,-2469,-2295,983,1520,-389,-1067,3015,23988,1757,-16349,-3827,10633,4295,-6491,-3870,3649,3047,-1832,-2144,778,1352,-239,-749,20,348,34,-117,-19,17,1},
{2,18,-37,-118,94,358,-119,-796,26,1497,335,-2491,-1171,3755,2761,-5165,-5427,6474,9546,-7263,-15546,6903,24018,-4416,-1125,-60,1694,512,-2749,-1750,6293,14393,13562,4760,-2414,-2341,944,1540,-360,-1075,2360,24055,2221,-16321,-4144,10565,4501,-6413,-3995,3580,3118,-1778,-2181,741,1369,-217,-755,8,350,39,-118,-20,17,1},
{2,18,-35,-118,88,358,-106,-794,0,1488,378,-2465,-1238,3698,2853,-5056,-5543,6286,9675,-6958,-15665,6436,24080,-3725,-1122,-91,1682,558,-2713,-1826,6139,14320,13656,4912,-2356,-2386,904,1559,-332,-1083,1699,24109,2687,-16285,-4461,10490,4705,-6332,-4120,3508,3188,-1724,-2216,703,1385,-194,-761,-4,352,44,-118,-22,17,1},
{2,18,-33,-119,83,358,-93,-791,-24,1477,421,-2437,-1303,3639,2943,-4945,-5655,6095,9798,-6651,-15774,5967,24130,-3036,-1119,-122,1670,604,-2675,-1899,5984,14245,13746,5064,-2297,-2430,863,1577,-303,-1090,1034,24151,3154,-16239,-4777,10410,4908,-6247,-4243,3435,3256,-1667,-2251,665,1400,-170,-766,-16,353,49,-118,-23,18,1},
{2,18,-32,-119,77,357,-80,-788,-49,1466,463,-2409,-1367,3579,3030,-4833,-5763,5902,9915,-6342,-15874,5498,24166,-2350,-1116,-153,1656,649,-2637,-1970,5831,14167,13835,5217,-2236,-2473,822,1594,-273,-1096,365,24180,3622,-16184,-5092,10323,5110,-6158,-4364,3358,3324,-1610,-2285,626,1414,-147,-772,-28,354,55,-119,-25,18,1},
{2,18,-30,-119,71,357,-67,-784,-74,1454,505,-2379,-1429,3517,3116,-4718,-5868,5706,10026,-6031,-15965,5029,24189,-1666,-1112,-184,1642,693,-2598,-2040,5677,14089,13922,5370,-2172,-2516,780,1611,-244,-1102,-308,24196,4090,-16120,-5406,10230,5311,-6065,-4484,3280,3390,-1551,-2317,586,1428,-123,-776,-41,355,60,-119,-27,18,1},
{1,18,-28,-119,66,356,-54,-781,-99,1442,546,-2349,-1491,3454,3199,-4602,-5968,5510,10131,-5719,-16047,4560,24199,-985,-1107,-214,1627,737,-2557,-2107,5523,14006,14006,5523,-2107,-2557,737,1627,-214,-1107,-985,24199,4560,-16047,-5719,10131,5510,-5968,-4602,3199,3454,-1491,-2349,546,1442,-99,-781,-54,356,66,-119,-28,18,1},
{1,18,-27,-119,60,355,-41,-776,-123,1428,586,-2317,-1551,3390,3280,-4484,-6065,5311,10230,-5406,-16120,4090,24196,-308,-1102,-244,1611,780,-2516,-2172,5370,13921,14088,5677,-2040,-2597,693,1642,-183,-1112,-1666,24189,5029,-15965,-6031,10026,5706,-5868,-4718,3116,3517,-1429,-2379,505,1454,-74,-784,-67,357,71,-119,-30,18,2},
{1,18,-25,-119,55,354,-28,-772,-147,1414,626,-2285,-1610,3324,3358,-4364,-6158,5110,10323,-5092,-16184,3622,24180,365,-1096,-273,1594,822,-2474,-2236,5217,13835,14168,5831,-1970,-2637,649,1656,-153,-1116,-2350,24166,5498,-15874,-6342,9915,5902,-5763,-4833,3030,3579,-1367,-2409,463,1466,-49,-788,-80,357,77,-119,-32,18,2},
{1,18,-23,-118,49,353,-16,-766,-170,1400,665,-2251,-1667,3256,3435,-4243,-6247,4908,10410,-4777,-16239,3154,24151,1034,-1090,-303,1577,863,-2430,-2297,5064,13746,14244,5985,-1899,-2675,604,1670,-122,-1119,-3036,24130,5967,-15774,-6651,9798,6095,-5655,-4945,2943,3639,-1303,-2437,421,1477,-24,-791,-93,358,83,-119,-33,18,2},
{1,17,-22,-118,44,352,-4,-761,-194,1385,703,-2216,-1724,3188,3508,-4120,-6332,4705,10490,-4461,-16285,2687,24109,1699,-1083,-332,1558,904,-2386,-2357,4912,13656,14322,6139,-1826,-2713,558,1682,-91,-1122,-3725,24080,6436,-15665,-6958,9675,6286,-5543,-5056,2853,3698,-1238,-2465,378,1488,0,-794,-106,358,88,-118,-35,18,2},
{1,17,-20,-118,39,350,8,-755,-217,1369,741,-2181,-1778,3118,3580,-3995,-6413,4501,10565,-4144,-16321,2221,24055,2360,-1075,-360,1540,944,-2341,-2414,4760,13562,14393,6293,-1750,-2749,512,1694,-60,-1125,-4416,24017,6903,-15546,-7263,9546,6474,-5427,-5165,2761,3755,-1171,-2491,335,1497,26,-796,-119,358,94,-118,-37,18,2},
{1,17,-19,-117,34,348,20,-749,-239,1352,778,-2144,-1832,3047,3649,-3870,-6491,4295,10633,-3827,-16349,1757,23988,3015,-1067,-389,1520,983,-2296,-2469,4609,13467,14463,6448,-1673,-2784,464,1705,-28,-1126,-5109,23941,7370,-15418,-7566,9411,6661,-5307,-5272,2667,3810,-1104,-2516,291,1506,52,-798,-133,357,100,-118,-39,18,2},
{1,17,-17,-117,28,346,32,-742,-262,1335,814,-2107,-1884,2975,3715,-3743,-6564,4088,10695,-3510,-16368,1294,23908,3664,-1059,-416,1500,1021,-2249,-2522,4459,13369,14530,6602,-1594,-2818,417,1716,3,-1128,-5803,23852,7835,-15282,-7867,9270,6844,-5184,-5376,2570,3864,-1035,-2539,247,1514,78,-799,-146,357,106,-117,-40,18,3},
{1,17,-16,-116,23,344,43,-735,-283,1318,849,-2069,-1934,2902,3779,-3615,-6633,3880,10751,-3193,-16378,833,23816,4308,-1050,-444,1479,1058,-2202,-2574,4309,13270,14597,6757,-1513,-2851,368,1725,35,-1128,-6498,23749,8299,-15136,-8165,9123,7026,-5057,-5478,2472,3916,-966,-2562,202,1522,104,-800,-160,356,111,-117,-42,18,3},
{1,16,-14,-116,18,342,55,-728,-305,1300,884,-2030,-1984,2827,3841,-3486,-6698,3671,10801,-2876,-16379,375,23712,4946,-1040,-471,1458,1094,-2154,-2623,4159,13168,14659,6911,-1429,-2882,319,1734,67,-1128,-7194,23633,8761,-14981,-8460,8970,7204,-4927,-5578,2371,3966,-895,-2583,156,1528,130,-801,-173,355,117,-116,-44,18,3},
{0,16,-13,-115,13,339,66,-721,-326,1281,918,-1990,-2031,2752,3900,-3355,-6759,3462,10845,-2559,-16371,-80,23595,5578,-1030,-498,1435,1130,-2105,-2670,4010,13064,14718,7065,-1344,-2912,270,1742,100,-1127,-7890,23503,9221,-14818,-8753,8812,7380,-4793,-5676,2269,4014,-823,-2603,110,1534,157,-801,-187,353,123,-115,-46,18,3},
{0,16,-11,-114,8,336,78,-713,-347,1262,951,-1950,-2078,2676,3956,-3224,-6816,3252,10882,-2243,-16354,-533,23467,6202,-1020,-524,1413,1165,-2056,-2716,3862,12959,14777,7219,-1257,-2941,220,1748,132,-1126,-8585,23360,9679,-14645,-9042,8648,7553,-4656,-5771,2165,4061,-751,-2622,64,1539,184,-800,-201,352,129,-114,-48,18,3},
{0,16,-10,-113,4,334,89,-704,-367,1242,983,-1908,-2122,2598,4010,-3092,-6869,3042,10914,-1927,-16329,-984,23327,6820,-1009,-549,1389,1198,-2006,-2759,3715,12851,14832,7373,-1168,-2969,169,1754,165,-1124,-9280,23203,10133,-14463,-9329,8478,7723,-4515,-5863,2058,4105,-677,-2640,17,1543,211,-799,-214,350,135,-113,-50,18,4},
{0,15,-9,-112,0,331,99,-696,-387,1221,1014,-1866,-2166,2520,4061,-2959,-6918,2831,10939,-1613,-16295,-1432,23175,7431,-997,-575,1365,1231,-1955,-2800,3568,12742,14885,7527,-1078,-2996,118,1759,198,-1122,-9974,23033,10586,-14273,-9612,8302,7889,-4371,-5953,1950,4148,-602,-2656,-29,1547,238,-798,-228,348,141,-112,-51,18,4},
{0,15,-7,-111,-5,327,110,-687,-407,1201,1045,-1823,-2207,2442,4110,-2826,-6963,2620,10959,-1299,-16252,-1876,23011,8034,-985,-599,1341,1263,-1904,-2839,3423,12629,14935,7680,-985,-3021,66,1763,231,-1119,-10667,22850,11034,-14074,-9891,8121,8053,-4224,-6040,1840,4189,-527,-2671,-77,1549,265,-796,-242,346,147,-111,-53,18,4},
{0,15,-6,-110,-10,324,121,-678,-426,1179,1074,-1780,-2247,2362,4156,-2692,-7004,2409,10972,-986,-16201,-2316,22836,8629,-973,-624,1316,1294,-1852,-2876,3278,12515,14981,7833,-890,-3044,14,1767,264,-1115,-11359,22653,11480,-13866,-10167,7935,8213,-4073,-6125,1729,4227,-451,-2684,-125,1551,293,-794,-256,344,153,-110,-55,17,4},
{0,14,-5,-109,-14,321,131,-669,-444,1158,1103,-1736,-2286,2282,4200,-2557,-7040,2198,10980,-675,-16142,-2753,22650,9217,-960,-648,1290,1324,-1800,-2912,3134,12401,15026,7986,-794,-3067,-37,1769,297,-1111,-12048,22443,11922,-13649,-10438,7743,8369,-3919,-6207,1615,4264,-373,-2696,-173,1552,320,-791,-270,341,158,-109,-57,17,5},
{0,14,-4,-108,-18,317,141,-659,-463,1135,1131,-1691,-2323,2201,4241,-2421,-7073,1987,10981,-365,-16074,-3186,22453,9795,-947,-671,1264,1354,-1747,-2945,2991,12283,15068,8138,-696,-3088,-90,1770,330,-1106,-12735,22219,12360,-13423,-10706,7546,8522,-3762,-6286,1500,4298,-295,-2707,-222,1552,348,-788,-283,338,164,-107,-59,17,5},
{0,14,-2,-107,-23,313,151,-649,-480,1113,1159,-1646,-2359,2119,4280,-2286,-7102,1776,10977,-56,-15998,-3615,22244,10366,-933,-694,1238,1382,-1694,-2976,2849,12164,15107,8289,-596,-3107,-144,1771,363,-1100,-13419,21983,12794,-13189,-10970,7343,8671,-3602,-6362,1384,4331,-217,-2716,-271,1551,375,-784,-297,335,170,-106,-61,17,5},
{0,14,-1,-106,-27,309,161,-639,-498,1090,1185,-1601,-2393,2037,4315,-2150,-7126,1566,10966,249,-15914,-4039,22025,10927,-919,-716,1211,1409,-1640,-3006,2708,12043,15144,8440,-494,-3125,-197,1770,397,-1094,-14099,21733,13223,-12947,-11229,7136,8817,-3440,-6435,1266,4361,-137,-2724,-320,1549,403,-780,-311,332,176,-104,-63,16,5},
{0,13,0,-104,-31,305,170,-628,-515,1066,1210,-1554,-2425,1954,4349,-2013,-7147,1356,10950,554,-15822,-4459,21796,11479,-904,-738,1183,1435,-1586,-3033,2568,11922,15178,8591,-390,-3142,-252,1768,430,-1087,-14777,21469,13648,-12696,-11483,6923,8958,-3274,-6505,1146,4389,-57,-2731,-370,1546,431,-775,-325,328,182,-102,-64,16,6},
{0,13,0,-103,-35,301,179,-617,-531,1043,1235,-1508,-2456,1871,4379,-1877,-7164,1147,10928,857,-15721,-4874,21556,12022,-889,-759,1155,1460,-1532,-3059,2429,11798,15209,8741,-285,-3156,-306,1766,463,-1079,-15450,21193,14067,-12437,-11733,6705,9096,-3105,-6572,1025,4415,23,-2736,-419,1543,459,-770,-339,324,188,-100,-66,16,6},
{0,13,1,-102,-39,297,188,-607,-547,1018,1258,-1461,-2485,1788,4408,-1740,-7176,938,10900,1157,-15613,-5284,21306,12556,-874,-780,1127,1485,-1477,-3082,2291,11672,15238,8891,-178,-3170,-361,1763,496,-1071,-16119,20904,14482,-12170,-11978,6482,9229,-2934,-6635,902,4439,104,-2739,-469,1539,486,-764,-353,320,194,-99,-68,15,6},
{0,12,2,-100,-43,293,197,-595,-563,994,1281,-1413,-2513,1704,4433,-1604,-7185,730,10867,1455,-15497,-5688,21045,13079,-858,-800,1098,1508,-1422,-3104,2155,11545,15265,9040,-69,-3182,-416,1758,529,-1063,-16783,20601,14891,-11894,-12218,6255,9358,-2760,-6696,779,4460,186,-2741,-519,1533,514,-758,-366,316,199,-97,-70,15,6},
{0,12,3,-99,-47,288,206,-584,-578,969,1303,-1365,-2539,1620,4456,-1467,-7190,523,10827,1750,-15374,-6088,20775,13593,-842,-820,1069,1530,-1367,-3123,2019,11417,15288,9187,40,-3192,-471,1753,563,-1053,-17442,20286,15294,-11611,-12452,6022,9483,-2583,-6753,653,4480,268,-2742,-569,1527,542,-752,-380,311,205,-94,-72,14,7},
{0,12,4,-97,-51,284,214,-572,-592,944,1324,-1317,-2564,1535,4476,-1330,-7190,317,10783,2043,-15242,-6482,20496,14096,-826,-839,1039,1552,-1311,-3141,1885,11287,15308,9335,153,-3201,-527,1746,596,-1043,-18095,19958,15692,-11319,-12681,5785,9604,-2404,-6807,527,4496,351,-2741,-619,1520,570,-744,-394,307,211,-92,-74,14,7},
{0,11,5,-96,-54,279,222,-561,-606,919,1344,-1268,-2587,1450,4494,-1194,-7187,112,10732,2333,-15104,-6870,20207,14589,-809,-857,1009,1572,-1256,-3157,1752,11155,15327,9481,267,-3208,-582,1739,629,-1033,-18743,19617,16083,-11020,-12905,5544,9720,-2223,-6858,400,4511,434,-2738,-669,1512,597,-737,-407,302,216,-90,-75,13,7},
{0,11,6,-94,-58,274,230,-549,-620,893,1363,-1219,-2608,1366,4509,-1058,-7180,-90,10676,2620,-14958,-7252,19908,15071,-792,-875,979,1592,-1200,-3171,1620,11022,15342,9626,382,-3213,-638,1731,661,-1021,-19385,19264,16468,-10713,-13123,5298,9831,-2039,-6905,271,4523,518,-2734,-720,1503,625,-729,-421,296,222,-87,-77,13,8},
{0,11,7,-93,-61,269,238,-536,-633,867,1381,-1170,-2628,1281,4522,-922,-7170,-293,10615,2904,-14805,-7628,19601,15542,-774,-892,948,1610,-1144,-3184,1490,10888,15354,9771,499,-3217,-694,1721,694,-1009,-20019,18899,16846,-10399,-13335,5048,9938,-1853,-6949,142,4533,602,-2729,-770,1493,652,-720,-434,291,227,-85,-79,12,8},
{0,10,7,-91,-65,264,245,-524,-646,841,1398,-1120,-2646,1196,4532,-787,-7155,-494,10549,3184,-14644,-7998,19285,16002,-757,-909,917,1627,-1088,-3194,1361,10753,15364,9915,618,-3219,-750,1711,726,-997,-20647,18521,17217,-10077,-13541,4793,10040,-1664,-6989,11,4540,686,-2721,-820,1482,680,-711,-447,285,233,-82,-81,12,8},
{0,10,8,-89,-68,259,252,-512,-658,814,1415,-1071,-2662,1110,4540,-652,-7137,-693,10477,3462,-14477,-8361,18960,16451,-739,-925,886,1644,-1031,-3202,1234,10615,15370,10057,738,-3219,-806,1699,759,-984,-21267,18130,17581,-9748,-13741,4535,10138,-1474,-7026,-119,4545,771,-2713,-871,1471,707,-701,-460,279,238,-80,-83,11,9},
{0,10,9,-88,-71,254,259,-499,-669,788,1430,-1021,-2677,1025,4545,-518,-7114,-891,10400,3736,-14303,-8718,18627,16888,-720,-941,855,1659,-975,-3209,1108,10478,15374,10198,860,-3218,-863,1687,791,-970,-21880,17728,17937,-9412,-13935,4272,10230,-1281,-7059,-251,4547,855,-2702,-921,1458,734,-691,-473,273,244,-77,-84,10,9},
{0,9,10,-86,-74,249,266,-486,-680,761,1444,-971,-2691,940,4547,-384,-7089,-1087,10317,4006,-14122,-9069,18286,17314,-702,-956,823,1674,-919,-3214,983,10338,15376,10339,983,-3214,-919,1674,823,-956,-22484,17314,18286,-9069,-14122,4006,10317,-1087,-7089,-384,4547,940,-2691,-971,1444,761,-680,-486,266,249,-74,-86,10,9},
};

void ATFilterNormalizeKernel(float *v, size_t n, float scale) {
	float sum = 0;

	for(size_t i=0; i<n; ++i)
		sum += v[i];

	scale /= sum;

	for(size_t i=0; i<n; ++i)
		v[i] *= scale;
}

uint64 ATFilterResampleMono(sint16 *d, const float *s, uint32 count, uint64 accum, sint64 inc) {
	do {
		const float *s2 = s + (accum >> 32);
		const float *f = gVDCaptureAudioResamplingKernel[(uint32)accum >> 27];

		accum += inc;

		float v = s2[0]*f[0]
				+ s2[1]*f[1]
				+ s2[2]*f[2]
				+ s2[3]*f[3]
				+ s2[4]*f[4]
				+ s2[5]*f[5]
				+ s2[6]*f[6]
				+ s2[7]*f[7];

		*d++ = VDClampedRoundFixedToInt16Fast(v * (1.0f / 16384.0f));
	} while(--count);

	return accum;
}

uint64 ATFilterResampleMonoToStereo(sint16 *d, const float *s, uint32 count, uint64 accum, sint64 inc) {
	do {
		const float *s2 = s + (accum >> 32);
		const float *f = gVDCaptureAudioResamplingKernel[(uint32)accum >> 27];

		accum += inc;

		float v = s2[0]*f[0]
				+ s2[1]*f[1]
				+ s2[2]*f[2]
				+ s2[3]*f[3]
				+ s2[4]*f[4]
				+ s2[5]*f[5]
				+ s2[6]*f[6]
				+ s2[7]*f[7];

		d[0] = d[1] = VDClampedRoundFixedToInt16Fast(v * (1.0f / 16384.0f));
		d += 2;
	} while(--count);

	return accum;
}

uint64 ATFilterResampleStereo(sint16 *d, const float *s1, const float *s2, uint32 count, uint64 accum, sint64 inc) {
	do {
		const float *r1 = s1 + (accum >> 32);
		const float *r2 = s2 + (accum >> 32);
		const float *f = gVDCaptureAudioResamplingKernel[(uint32)accum >> 27];

		accum += inc;

		float a = r1[0]*f[0]
				+ r1[1]*f[1]
				+ r1[2]*f[2]
				+ r1[3]*f[3]
				+ r1[4]*f[4]
				+ r1[5]*f[5]
				+ r1[6]*f[6]
				+ r1[7]*f[7];

		float b = r2[0]*f[0]
				+ r2[1]*f[1]
				+ r2[2]*f[2]
				+ r2[3]*f[3]
				+ r2[4]*f[4]
				+ r2[5]*f[5]
				+ r2[6]*f[6]
				+ r2[7]*f[7];

		d[0] = VDClampedRoundFixedToInt16Fast(a * (1.0f / 16384.0f));
		d[1] = VDClampedRoundFixedToInt16Fast(b * (1.0f / 16384.0f));
		d += 2;
	} while(--count);

	return accum;
}

void ATFilterComputeSymmetricFIR_8_32F_Scalar(float *dst, const float *src, size_t n, const float *kernel) {
	const float k0 = kernel[0];
	const float k1 = kernel[1];
	const float k2 = kernel[2];
	const float k3 = kernel[3];
	const float k4 = kernel[4];
	const float k5 = kernel[5];
	const float k6 = kernel[6];
	const float k7 = kernel[7];

	do {
		float v = src[7] * k0
				+ (src[ 6] + src[ 8]) * k1
				+ (src[ 5] + src[ 9]) * k2
				+ (src[ 4] + src[10]) * k3
				+ (src[ 3] + src[11]) * k4
				+ (src[ 2] + src[12]) * k5
				+ (src[ 1] + src[13]) * k6
				+ (src[ 0] + src[14]) * k7;

		++src;

		*dst++ = v;
	} while(--n);
}

#if defined(VD_CPU_X86) || defined(VD_CPU_AMD64)
void ATFilterComputeSymmetricFIR_8_32F_SSE(float *dst, const float *src, size_t n, const float *kernel) {
	__m128 zero = _mm_setzero_ps();
	__m128 x0 = zero;
	__m128 x1 = zero;
	__m128 x2 = zero;
	__m128 x3 = zero;
	__m128 f0;
	__m128 f1;
	__m128 f2;
	__m128 f3;

	// init filter
	__m128 k0 = _mm_loadu_ps(kernel + 0);
	__m128 k1 = _mm_loadu_ps(kernel + 4);

	f0 = _mm_shuffle_ps(k1, k1, _MM_SHUFFLE(0, 1, 2, 3));
	f1 = _mm_shuffle_ps(k0, k0, _MM_SHUFFLE(0, 1, 2, 3));
	f2 = _mm_move_ss(k0, k1);
	f2 = _mm_shuffle_ps(f2, f2, _MM_SHUFFLE(0, 3, 2, 1));
	f3 = _mm_move_ss(k1, zero);
	f3 = _mm_shuffle_ps(f3, f3, _MM_SHUFFLE(0, 3, 2, 1));

	// prime
	for(int i=0; i<14; ++i) {
		x0 = _mm_move_ss(x0, x1);
		x0 = _mm_shuffle_ps(x0, x0, _MM_SHUFFLE(0, 3, 2, 1));
		x1 = _mm_move_ss(x1, x2);
		x1 = _mm_shuffle_ps(x1, x1, _MM_SHUFFLE(0, 3, 2, 1));
		x2 = _mm_move_ss(x2, x3);
		x2 = _mm_shuffle_ps(x2, x2, _MM_SHUFFLE(0, 3, 2, 1));
		x3 = _mm_move_ss(x3, zero);
		x3 = _mm_shuffle_ps(x3, x3, _MM_SHUFFLE(0, 3, 2, 1));

		__m128 s = _mm_load1_ps(src++);
		x0 = _mm_add_ps(x0, _mm_mul_ps(f0, s));
		x1 = _mm_add_ps(x1, _mm_mul_ps(f1, s));
		x2 = _mm_add_ps(x2, _mm_mul_ps(f2, s));
		x3 = _mm_add_ps(x3, _mm_mul_ps(f3, s));
	}

	// pipeline
	do {
		x0 = _mm_move_ss(x0, x1);
		x0 = _mm_shuffle_ps(x0, x0, _MM_SHUFFLE(0, 3, 2, 1));
		x1 = _mm_move_ss(x1, x2);
		x1 = _mm_shuffle_ps(x1, x1, _MM_SHUFFLE(0, 3, 2, 1));
		x2 = _mm_move_ss(x2, x3);
		x2 = _mm_shuffle_ps(x2, x2, _MM_SHUFFLE(0, 3, 2, 1));
		x3 = _mm_move_ss(x3, zero);
		x3 = _mm_shuffle_ps(x3, x3, _MM_SHUFFLE(0, 3, 2, 1));

		__m128 s = _mm_load1_ps(src++);
		x0 = _mm_add_ps(x0, _mm_mul_ps(f0, s));
		x1 = _mm_add_ps(x1, _mm_mul_ps(f1, s));
		x2 = _mm_add_ps(x2, _mm_mul_ps(f2, s));
		x3 = _mm_add_ps(x3, _mm_mul_ps(f3, s));

		_mm_store_ss(dst++, x0);
	} while(--n);
}
#endif

#if defined(VD_CPU_X86) && defined(VD_COMPILER_MSVC)
void __declspec(naked) __cdecl ATFilterComputeSymmetricFIR_8_32F_SSE_asm(float *dst, const float *src, size_t n, const float *kernel) {
	__asm {
	mov	edx, esp
	and	esp, -16
	sub	esp, 80
	mov	[esp+64], edx

	mov	eax, [edx+16]
	mov	ecx, [edx+8]

	xorps	xmm7, xmm7

	movups	xmm0, [eax+16]
	movups	xmm1, [eax]
	movaps	xmm2, xmm1
	movaps	xmm3, xmm0
	shufps	xmm0, xmm0, 1bh
	shufps	xmm1, xmm1, 1bh
	movss	xmm2, xmm3
	shufps	xmm2, xmm2, 39h
	movss	xmm3, xmm7
	shufps	xmm3, xmm3, 39h
	movaps	[esp], xmm0
	movaps	[esp+16], xmm1
	movaps	[esp+32], xmm2
	movaps	[esp+48], xmm3

	mov	eax, 14
	xorps	xmm0, xmm0
	xorps	xmm1, xmm1
	xorps	xmm2, xmm2
	xorps	xmm3, xmm3
ploop:
	movss	xmm0, xmm1
	shufps	xmm0, xmm0, 39h
	movss	xmm1, xmm2
	shufps	xmm1, xmm1, 39h
	movss	xmm2, xmm3
	shufps	xmm2, xmm2, 39h
	movss	xmm3, xmm7
	shufps	xmm3, xmm3, 39h

	movss	xmm6, [ecx]
	add		ecx, 4
	shufps	xmm6, xmm6, 0

	movaps	xmm4, xmm6
	mulps	xmm4, [esp+0]
	addps	xmm0, xmm4

	movaps	xmm5, xmm6
	mulps	xmm5, [esp+16]
	addps	xmm1, xmm5

	movaps	xmm4, xmm6
	mulps	xmm4, [esp+32]
	addps	xmm2, xmm4

	mulps	xmm6, [esp+48]
	addps	xmm3, xmm6

	dec		eax
	jne		ploop
	
	mov	eax, [edx+12]
	mov	edx, [edx+4]
xloop:
	movss	xmm0, xmm1
	shufps	xmm0, xmm0, 39h
	movss	xmm1, xmm2
	shufps	xmm1, xmm1, 39h
	movss	xmm2, xmm3
	shufps	xmm2, xmm2, 39h
	movss	xmm3, xmm7
	shufps	xmm3, xmm3, 39h

	movss	xmm6, [ecx]
	add		ecx, 4
	shufps	xmm6, xmm6, 0

	movaps	xmm4, xmm6
	mulps	xmm4, [esp+0]
	addps	xmm0, xmm4

	movaps	xmm5, xmm6
	mulps	xmm5, [esp+16]
	addps	xmm1, xmm5

	movaps	xmm4, xmm6
	mulps	xmm4, [esp+32]
	addps	xmm2, xmm4

	mulps	xmm6, [esp+48]
	addps	xmm3, xmm6

	movss	[edx], xmm0
	add	edx, 4

	dec	eax
	jne	xloop

	mov	esp, [esp+64]
	ret
	}
}
#endif

void ATFilterComputeSymmetricFIR_8_32F(float *dst, const float *src, size_t n, const float *kernel) {
#if defined(VD_CPU_X86) && defined(VD_COMPILER_MSVC)
	if (SSE_enabled) {
		ATFilterComputeSymmetricFIR_8_32F_SSE_asm(dst, src, n, kernel);
		return;
	}
#endif

	ATFilterComputeSymmetricFIR_8_32F_Scalar(dst, src, n, kernel);
}

void ATFilterComputeSymmetricFIR_8_32F(float *dst, size_t n, const float *kernel) {
#if defined(VD_CPU_X86) && defined(VD_COMPILER_MSVC)
	if (SSE_enabled) {
		ATFilterComputeSymmetricFIR_8_32F_SSE_asm(dst, dst, n, kernel);
		return;
	}

	ATFilterComputeSymmetricFIR_8_32F_Scalar(dst, dst, n, kernel);
#elif defined(VD_CPU_X64)
	ATFilterComputeSymmetricFIR_8_32F_SSE(dst, dst, n, kernel);
#else
	ATFilterComputeSymmetricFIR_8_32F_Scalar(dst, dst, n, kernel);
#endif
}

///////////////////////////////////////////////////////////////////////////

ATAudioFilter::ATAudioFilter()
	: mHiPassAccum(0)
{
	SetScale(1.0f);
	SetActiveMode(true);

	// Set up a FIR low pass filter, Blackman window, 15KHz cutoff (63920Hz sampling rate)
	const float fc = 15000.0f / 63920.0f;
	float sum = 0.5f;
	mLoPassCoeffs[0] = 1.0f;

	for(int i=1; i<kFilterOverlap; ++i) {
		float x = (float)i * nsVDMath::kfPi;
		float y = x / 128.0f * 2.0f;
		float w = 0.42f + 0.5f * cosf(y) + 0.08f * cosf(y+y);

		float f = sinf(2.0f * x * fc) / (2.0f * x * fc) * w;

		mLoPassCoeffs[i] = f;

		sum += f;
	}

	float scale = 0.5f / sum;

	for(int i=0; i<kFilterOverlap; ++i)
		mLoPassCoeffs[i] *= scale;
}

float ATAudioFilter::GetScale() const {
	return mRawScale;
}

void ATAudioFilter::SetScale(float scale) {
	// We accumulate 28 cycles worth of output per sample, and each output can
	// be from 0-60 (4*15). We ignore the speaker and cassette inputs in order
	// to get a little more range.
	mRawScale = scale;
	mScale = scale * (1.0f / (60.0f * 28.0f));
}

void ATAudioFilter::SetActiveMode(bool active) {
	if (active)
		mHiCoeff = 0.003f;
	else
		mHiCoeff = 0.0001f;
}

void ATAudioFilter::PreFilter(float * VDRESTRICT dst, uint32 count) {
	const float scale = mScale;
	const float hiCoeff = mHiCoeff;
	float hiAccum = mHiPassAccum;

	do {
		float v0 = *dst;
		float v1 = v0 - hiAccum;
		hiAccum += v1 * hiCoeff;

		*dst++ = v1 * scale;
	} while(--count);

	// prevent denormals
	if (fabsf(hiAccum) < 1e-20f)
		hiAccum = 0;

	mHiPassAccum = hiAccum;
}

void ATAudioFilter::Filter(float *dst, const float *src, uint32 count) {
	ATFilterComputeSymmetricFIR_8_32F(dst, src, count, mLoPassCoeffs);
}

void ATAudioFilter::Filter(float *dst, uint32 count) {
	ATFilterComputeSymmetricFIR_8_32F(dst, count, mLoPassCoeffs);
}
