//	Altirra - Atari 800/800XL/5200 emulator
//	Copyright (C) 2009-2016 Avery Lee
//
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#ifndef f_AT_DISKDRIVEFULL_H
#define f_AT_DISKDRIVEFULL_H

#include <vd2/system/function.h>
#include <vd2/system/refcount.h>
#include <vd2/system/vdstl.h>
#include <at/atcore/devicediskdrive.h>
#include <at/atcore/deviceimpl.h>
#include <at/atcore/deviceserial.h>
#include <at/atcore/devicesio.h>
#include <at/atcpu/co65802.h>
#include <at/atcpu/breakpoints.h>
#include <at/atcpu/history.h>
#include <at/atdebugger/target.h>
#include <at/atcore/scheduler.h>
#include "riot.h"
#include "fdc.h"
#include "disk.h"

class ATIRQController;

class ATDeviceDiskDriveFull final : public ATDevice
	, public IATDeviceScheduling
	, public IATDeviceFirmware
	, public IATDeviceDiskDrive
	, public IATDeviceSIO
	, public IATDeviceAudioOutput
	, public IATDeviceDebugTarget
	, public IATDebugTarget
	, public IATDebugTargetHistory
	, public IATDebugTargetBreakpoints
	, public IATDebugTargetExecutionControl
	, public IATCPUBreakpointHandler
	, public IATSchedulerCallback
	, public IATDeviceRawSIO
	, public IATDiskInterfaceClient
{
public:
	ATDeviceDiskDriveFull(bool happy810);
	~ATDeviceDiskDriveFull();

	void *AsInterface(uint32 iid) override;

	void GetDeviceInfo(ATDeviceInfo& info) override;
	void Init() override;
	void Shutdown() override;
	void WarmReset() override;
	void ColdReset() override;

public:
	void InitScheduling(ATScheduler *sch, ATScheduler *slowsch) override;

public:		// IATDeviceFirmware
	void InitFirmware(ATFirmwareManager *fwman) override;
	bool ReloadFirmware() override;
	const wchar_t *GetWritableFirmwareDesc(uint32 idx) const override;
	bool IsWritableFirmwareDirty(uint32 idx) const override;
	void SaveWritableFirmware(uint32 idx, IVDStream& stream) override;

public:		// IATDeviceDiskDrive
	void InitDiskDrive(IATDiskDriveManager *ddm) override;

public:		// IATDeviceSIO
	void InitSIO(IATDeviceSIOManager *mgr) override;
	CmdResponse OnSerialBeginCommand(const ATDeviceSIOCommand& cmd) override;
	void OnSerialAbortCommand() override;
	void OnSerialReceiveComplete(uint32 id, const void *data, uint32 len, bool checksumOK) override;
	void OnSerialFence(uint32 id) override; 
	CmdResponse OnSerialAccelCommand(const ATDeviceSIORequest& request) override;

public:		// IATDeviceAudioOutput
	void InitAudioOutput(IATAudioOutput *output, ATAudioSyncMixer *syncmixer) override;

public:	// IATDeviceDebugTarget
	IATDebugTarget *GetDebugTarget(uint32 index) override;

public:	// IATDebugTarget
	const char *GetName() override;
	ATDebugDisasmMode GetDisasmMode() override;

	void GetExecState(ATCPUExecState& state) override;
	void SetExecState(const ATCPUExecState& state) override;

	sint32 GetTimeSkew() override;

	uint8 ReadByte(uint32 address) override;
	void ReadMemory(uint32 address, void *dst, uint32 n) override;

	uint8 DebugReadByte(uint32 address) override;
	void DebugReadMemory(uint32 address, void *dst, uint32 n) override;

	void WriteByte(uint32 address, uint8 value) override;
	void WriteMemory(uint32 address, const void *src, uint32 n) override;

public:	// IATDebugTargetHistory
	bool GetHistoryEnabled() const override;
	void SetHistoryEnabled(bool enable) override;

	std::pair<uint32, uint32> GetHistoryRange() const override;
	uint32 ExtractHistory(const ATCPUHistoryEntry **hparray, uint32 start, uint32 n) const override;
	uint32 ConvertRawTimestamp(uint32 rawTimestamp) const override;

public:	// IATDebugTargetBreakpoints
	virtual void SetBreakpointHandler(IATCPUBreakpointHandler *handler) override;

	virtual void ClearBreakpoint(uint16 pc) override;
	virtual void SetBreakpoint(uint16 pc) override;

public:	// IATDebugTargetExecutionControl
	void Break() override;
	bool StepInto(const vdfunction<void(bool)>& fn) override;
	bool StepOver(const vdfunction<void(bool)>& fn) override;
	bool StepOut(const vdfunction<void(bool)>& fn) override;
	void StepUpdate() override;
	void RunUntilSynced() override;

public:	// IATCPUBreakpointHandler
	bool CheckBreakpoint(uint32 pc) override;

public:	// IATSchedulerCallback
	void OnScheduledEvent(uint32 id) override;

public:	// IATDeviceRawSIO
	void OnCommandStateChanged(bool asserted) override;
	void OnMotorStateChanged(bool asserted) override;
	void OnReceiveByte(uint8 c, bool command, uint32 cyclesPerBit) override;
	void OnSendReady() override;

public:	// IATDiskInterfaceClient
	void OnDiskChanged() override;
	void OnWriteModeChanged() override;
	void OnTimingModeChanged() override;
	void OnAudioModeChanged() override;

protected:
	void CancelStep();

	void Sync();
	void AccumSubCycles();

	void BeginTransmit();
	void QueueNextTransmit();

	void OnRIOTRegisterWrite(uint32 addr, uint8 val);

	void PlayStepSound();
	void UpdateRotationStatus();
	void UpdateWriteProtectStatus();

	enum {
		kEventId_Run = 1,
		kEventId_Transmit,
		kEventId_DriveReceiveBit,
		kEventId_DriveTransmitBit
	};

	ATScheduler *mpScheduler = nullptr;
	ATScheduler *mpSlowScheduler = nullptr;
	ATEvent *mpRunEvent = nullptr;
	ATEvent *mpTransmitEvent = nullptr;
	ATEvent *mpEventDriveReceiveBit = nullptr;
	ATEvent *mpEventDriveTransmitBit = nullptr;
	IATDeviceSIOManager *mpSIOMgr = nullptr;
	IATDiskDriveManager *mpDiskDriveManager = nullptr;
	ATDiskInterface *mpDiskInterface = nullptr;

	ATFirmwareManager *mpFwMgr = nullptr;
	ATAudioSyncMixer *mpAudioSyncMixer = nullptr;

	uint32 mReceiveShiftRegister = 0;
	uint32 mReceiveTimingAccum = 0;
	uint32 mReceiveTimingStep = 0;

	uint32 mTransmitResetCounter = 0;
	uint32 mTransmitCyclesPerBit = 0;
	uint32 mTransmitTimingAccum = 0;
	uint32 mTransmitTimingStep = 0;
	uint32 mTransmitShiftRegister = 0;

	uint32 mCurrentTrack = 0;

	uint32 mLastSync = 0;
	uint32 mLastSyncDriveTime = 0;
	uint32 mSubCycleAccum = 0;
	uint8 *mpCoProcWinBase = nullptr;

	const bool mbHappy810;
	bool mbSoundsEnabled = false;
	uint32 mRotationSoundId = 0;
	uint32 mLastStepSoundTime = 0;
	uint32 mLastStepPhase = 0;

	ATCoProcReadMemNode mReadNodeFDCRAM {};
	ATCoProcReadMemNode mReadNodeRIOTRAM {};
	ATCoProcReadMemNode mReadNodeRIOTRegisters {};
	ATCoProcWriteMemNode mWriteNodeFDCRAM {};
	ATCoProcWriteMemNode mWriteNodeRIOTRAM {};
	ATCoProcWriteMemNode mWriteNodeRIOTRegisters {};

	ATScheduler mDriveScheduler;
	ATCoProc65802 mCoProc;

	ATFDCEmulator mFDC;
	ATRIOT6532Emulator mRIOT;

	vdfastvector<ATCPUHistoryEntry> mHistory;

	struct QueuedTransmit {
		uint32 mTime;
		uint32 mCyclesPerBit;
		uint8 mByte;
		bool mbFramingError;
	};

	vdfastdeque<QueuedTransmit> mTransmitQueue;

	VDALIGN(4) uint8 mRAM[0xD00] = {};
	VDALIGN(4) uint8 mROM[0xC00] = {};
	VDALIGN(4) uint8 mDummyWrite[0x100] = {};
	
	vdfunction<void(bool)> mpStepHandler = {};
	bool mbStepOut = false;
	uint32 mStepStartSubCycle = 0;
	uint16 mStepOutS = 0;
	uint32 mBreakpointCount = 0;
	IATCPUBreakpointHandler *mpBreakpointHandler = nullptr;
	bool mBreakpointMap[0x10000] = {};
	bool mStepBreakpointMap[0x10000];
};

#endif
