/**
 * Copyright (C) 2014 <a href="http://www.wudsn.com" target="_top">Peter Dell</a>
 *
 * This file is part of ROM Checker.
 * 
 * ROM Checker is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * ROM Checker is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with ROM Checker.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.wudsn.tools.atariroms.model;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.zip.CRC32;

import com.wudsn.tools.atariroms.DataTypes;
import com.wudsn.tools.base.common.ByteArrayUtility;
import com.wudsn.tools.base.common.HexUtility;
import com.wudsn.tools.base.repository.Attribute;

public final class ROM {
    public static final class Attributes {
	private Attributes() {
	}

	public static final Attribute FPP_CHECK_SUM = new Attribute("fppCheckSum", DataTypes.ROM_FPP_CheckSum);
	public static final Attribute OS1_CHECK_SUM = new Attribute("os1CheckSum", DataTypes.ROM_OS1_CheckSum);
	public static final Attribute OS2_CHECK_SUM = new Attribute("os2CheckSum", DataTypes.ROM_OS2_CheckSum);
    }

    public static final int MAX_SIZE = 0x10000;

    private static final int UNKNOWN = 0;
    private static final int OS_AB = 1;
    private static final int OS_XL = 2;

    private static final int FPP_OFFSET = 0x07fe;
    private static final int OS1_OFFSET1 = 0x0c0f;
    private static final int OS1_OFFSET2 = 0x0c1f;
    private static final int OS2_OFFSET = 0x27f8;

    private byte[] content;
    private int type;

    public ROM(byte[] content) {
	if (content == null) {
	    throw new IllegalArgumentException("Parameter 'content' must not be null.");
	}
	this.content = content;
	switch (content.length) {
	case 0x2800:
	    type = OS_AB;
	    break;
	case 0x4000:
	    type = OS_XL;
	    break;
	default:
	    type = UNKNOWN;
	}
    }

    public int getSize() {
	return content.length;
    }

    public int getFPP() {
	if (type != OS_AB) {
	    return -1;
	}
	return getWord(FPP_OFFSET);
    }

    public int getOS1() {
	if (type != OS_AB) {
	    return -1;
	}
	return getByte(OS1_OFFSET1) + (getByte(OS1_OFFSET2) << 8);
    }

    public int getOS2() {
	if (type != OS_AB) {
	    return -1;
	}
	return getWord(OS2_OFFSET);
    }

    public String getCRC32() {
	CRC32 crc32 = new CRC32();
	crc32.update(content);
	return "0x" + HexUtility.getLongValueHexString(crc32.getValue(), 8).toLowerCase();
    }

    public String getMD5() {
	MessageDigest md5;
	try {
	    md5 = MessageDigest.getInstance("MD5");
	} catch (NoSuchAlgorithmException ex) {
	    throw new RuntimeException(ex);
	}

	md5.update(content);
	return ByteArrayUtility.toHexString(md5.digest()).toLowerCase();
    }

    private int getWord(int offset) {
	return getByte(offset) + (getByte(offset + 1) << 8);
    }

    public int getByte(int offset) {
	return (content[offset] & 0xff);
    }

    public boolean contentEquals(ROM other) {
	return Arrays.equals(this.content, other.content);
    }

    @Override
    public String toString() {
	return "FPP=" + toHexBytes(getFPP()) + " OS1=" + toHexBytes(getOS1()) + " OS2=" + toHexBytes(getOS2())
		+ " MD5=" + getMD5() + " CRC-32=" + getCRC32();
    }

    /**
     * Converts a file offset into a ROM address.
     * 
     * @param offset
     *            The file offset or <code>-1</code> if the offset is not valid.
     * @return The ROM address or or <code>-1</code> if the offset is not valid.
     */
    public int toAddress(int offset) {
	switch (type) {
	case UNKNOWN:
	    return -1;
	case OS_AB:
	    if (offset >= 0 && offset < 0x2800) {
		return 0xd800 + offset;
	    }
	    break;
	case OS_XL:
	    if (offset >= 0 && offset < 0x4000) {
		if (offset >= 0x1000 && offset < 0x1800) {
		    return 0x4000 + offset;
		}
		return 0xc000 + offset;
	    }
	    break;

	default:
	    throw new RuntimeException();
	}

	return -1;
    }

    public static String toHexBytes(int value) {
	if (value < 0) {
	    return "";
	}
	int low = value & 0xff;
	int high = (value >>> 8) & 0xff;
	return toHexByte(low) + "/" + toHexByte(high);
    }

    private static String toHexByte(int value) {
	value = value & 0xff;
	if (value < 10) {
	    return "$0" + Integer.toHexString(value);
	}
	return "$" + Integer.toHexString(value);
    }

    public ROM createFixture() {
	if (type != OS_AB) {
	    return this;
	}

	byte[] newContent;

	newContent = new byte[content.length];
	System.arraycopy(content, 0, newContent, 0, content.length);

	int fpp;
	int os1;
	int os2;

	/* Checksum of FPP ROM. */
	fpp = 0;
	for (int i = 0x0000; i < 0x07fe; i++) {
	    fpp += content[i] & 0xff;
	}
	fpp &= 0xffff;
	newContent[FPP_OFFSET] = (byte) (fpp & 0xff);
	newContent[FPP_OFFSET + 1] = (byte) (fpp >>> 8);

	/* Checksum of first 4K ROM. */
	os1 = 0;
	for (int i = 0x0800; i < 0x0c0f; i++) {
	    os1 += content[i] & 0xff;
	}
	for (int i = 0x0c10; i < 0x0c1f; i++) {
	    os1 += content[i] & 0xff;
	}
	for (int i = 0x0c20; i < 0x1800; i++) {
	    os1 += content[i] & 0xff;
	}
	os1 &= 0xffff;
	newContent[OS1_OFFSET1] = (byte) (os1 & 0xff);
	newContent[OS1_OFFSET2] = (byte) (os1 >>> 8);

	/* Checksum of second 4K ROM. */
	os2 = 0;
	for (int i = 0x1800; i < 0x27f8; i++) {
	    os2 += content[i] & 0xff;
	}
	for (int i = 0x27fa; i < 0x2800; i++) {
	    os2 += content[i] & 0xff;
	}
	os2 &= 0xffff;
	newContent[OS2_OFFSET] = (byte) (os2 & 0xff);
	newContent[OS2_OFFSET + 1] = (byte) (os2 >>> 8);

	return new ROM(newContent);

    }
}