/**
 * Copyright (C) 2014 <a href="http://www.wudsn.com" target="_top">Peter Dell</a>
 *
 * This file is part of ROM Checker.
 * 
 * ROM Checker is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * ROM Checker is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with ROM Checker.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.wudsn.tools.atariroms.model;

import java.util.HashMap;
import java.util.Map;

import com.wudsn.tools.atariroms.DataTypes;
import com.wudsn.tools.base.repository.Attribute;

public final class ROMVersion {

    public static final ROMVersion UNKNOWN;
    public static final ROMVersion ATARI_OS_A_NTSC;
    public static final ROMVersion ATARI_OS_A_PAL;
    public static final ROMVersion ATARI_OS_B_NTSC_LINBUG;
    public static final ROMVersion ATARI_OS_B_NTSC;
    public static final ROMVersion ATARI_OS_B_NTSC_PCXFORMER;
    public static final ROMVersion ATARI_OS_B_PAL;

    private static Map<String, ROMVersion> crc32Map;

    static {
	UNKNOWN = new ROMVersion(0, "Unknown", "Unknown", "Unknown", "Unknown", "Unknown", "Unknown");

	ATARI_OS_A_NTSC = new ROMVersion(0x2800, "Atari OS", "OS-A", "NTSC", "1979", "Production ROM",
		"CO12499A (low-OS, NTSC), CO14599A (high-OS NTSC), CO12399B (FPP)");
	ATARI_OS_A_PAL = new ROMVersion(0x2800, "Atari OS", "OS-A", "PAL", "1979", "Production ROM",
		"CO15199  (low-OS, NTSC), CO15299 (high-OS NTSC), CO12399B (FPP)");
	ATARI_OS_B_NTSC_LINBUG = new ROMVersion(0x2800, "Atari OS", "OS-B", "NTSC", "1981",
		"Production ROM, LINBUG Version with incorrect checksums",
		"C012499B (low-OS, NTSC), CO14599B (high-OS, NTSC), CO12399B (FPP)");
	ATARI_OS_B_NTSC = new ROMVersion(0x2800, "Atari OS", "OS-B", "NTSC", "1981", "Production ROM", "Unknown");
	ATARI_OS_B_NTSC_PCXFORMER = new ROMVersion(0x2800, "Atari OS", "OS-B", "NTSC", "1996", "PCXFormer hack", "n/a");
	ATARI_OS_B_PAL = new ROMVersion(0x2800, "Atari OS", "OS-B", "PAL", "1981", "Tomasz Krasuski", "Unknown");

	crc32Map = new HashMap<String, ROMVersion>();
	crc32Map.put("0xc1b3bb02", ATARI_OS_A_NTSC);
	crc32Map.put("0x72b3fed4", ATARI_OS_A_PAL);
	crc32Map.put("0x0e86d61d", ATARI_OS_B_NTSC_LINBUG);
	crc32Map.put("0xf28bc97d", ATARI_OS_B_NTSC);
	crc32Map.put("0x3e28a1fe", ATARI_OS_B_NTSC_PCXFORMER);
	crc32Map.put("0x0c913dfc", ATARI_OS_B_PAL);
    }

    public static final class Attributes {
	private Attributes() {
	}

	public static final Attribute TYPE = new Attribute("type", DataTypes.Version_Type);
	public static final Attribute REVISION = new Attribute("revision", DataTypes.Version_Revision);
	public static final Attribute NORM = new Attribute("norm", DataTypes.Version_Norm);
	public static final Attribute YEAR = new Attribute("year", DataTypes.Version_Year);
	public static final Attribute COMMENT = new Attribute("comment", DataTypes.Version_Comment);
	public static final Attribute PARTS = new Attribute("parts", DataTypes.Version_Parts);
    }

    private final int fileSize;
    private final String type;
    private final String revision;
    private final String norm;
    private final String year;
    private final String comment;
    private final String parts;

    public ROMVersion(int fileSize, String type, String revision, String norm, String year, String comment, String parts) {
	if (type == null) {
	    throw new IllegalArgumentException("Parameter 'type' must not be null.");
	}
	if (revision == null) {
	    throw new IllegalArgumentException("Parameter 'revision' must not be null.");
	}
	if (norm == null) {
	    throw new IllegalArgumentException("Parameter 'norm' must not be null.");
	}
	if (year == null) {
	    throw new IllegalArgumentException("Parameter 'year' must not be null.");
	}
	if (comment == null) {
	    throw new IllegalArgumentException("Parameter 'comment' must not be null.");
	}
	if (parts == null) {
	    throw new IllegalArgumentException("Parameter 'parts' must not be null.");
	}
	this.fileSize = fileSize;
	this.type = type;
	this.revision = revision;
	this.norm = norm;
	this.year = year;
	this.comment = comment;
	this.parts = parts;
    }

    public int getFileSize(){
	return fileSize;
    }
    
    public String getType() {
	return type;
    }

    public String getRevision() {
	return revision;
    }

    public String getNorm() {
	return norm;
    }

    public String getYear() {
	return year;
    }

    public String getComment() {
	return comment;
    }

    public String getParts() {
	return parts;
    }

    @Override
    public String toString() {
	return type + "/" + revision + "/" + norm + "/" + year + "/" + comment + "/" + parts;
    }

    public static ROMVersion getByCRC32(String crc32) {
	if (crc32 == null) {
	    throw new IllegalArgumentException("Parameter 'crc32' must not be null.");
	}
	ROMVersion result = crc32Map.get(crc32);
	if (result == null) {
	    result = UNKNOWN;
	}
	return result;
    }
}