/**
 * Copyright (C) 2014 <a href="http://www.wudsn.com" target="_top">Peter Dell</a>
 *
 * This file is part of ROM Checker.
 * 
 * ROM Checker is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * 
 * ROM Checker is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with ROM Checker.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.wudsn.tools.atariroms.ui;

import java.awt.Font;

import javax.swing.table.DefaultTableCellRenderer;

import com.wudsn.tools.atariroms.Preferences;
import com.wudsn.tools.atariroms.model.ROM;
import com.wudsn.tools.atariroms.model.ROMVersion;
import com.wudsn.tools.atariroms.model.Workbook;
import com.wudsn.tools.atariroms.model.WorkbookEntry;
import com.wudsn.tools.atariroms.model.WorkbookEntry.Attributes;
import com.wudsn.tools.base.gui.AttributeTableModel;
import com.wudsn.tools.base.gui.AttributeTablePanel;
import com.wudsn.tools.base.repository.Attribute;

@SuppressWarnings("serial")
public final class WorkbookEntriesPanel extends AttributeTablePanel {

    public final static class TableModel extends AttributeTableModel {

	public final class Columns {
	    public final static int ID = 0;
	    public final static int FOLDER_PATH = 1;
	    public final static int FILE_NAME = 2;
	    public final static int MD5 = 3;
	    public final static int CRC32 = 4;

	    public final static int FPP_CHECK_SUM = 5;
	    public final static int OS1_CHECK_SUM = 6;
	    public final static int OS2_CHECK_SUM = 7;

	    public final static int TYPE = 8;
	    public final static int REVISION = 9;
	    public final static int NORM = 10;
	    public final static int YEAR = 11;
	    public final static int COMMENT = 12;
	    public final static int PARTS = 13;
	    public final static int MESSAGE = 14;

	}

	private Workbook workbook;

	public TableModel(final Workbook workbook) {
	    if (workbook == null) {
		throw new IllegalArgumentException("Parameter 'workbook' must not be null.");
	    }

	    this.workbook = workbook;

	    final Font monoSpacedFont = new Font(Font.MONOSPACED, Font.PLAIN, 12);
	    DefaultTableCellRenderer cellRenderer = new DefaultTableCellRenderer() {
		@Override
		protected void setValue(Object value) {
		    setFont(monoSpacedFont);
		    super.setValue(value);
		}
	    };

	    addColumn(Attributes.ID, Column.VISIBLE);
	    addColumn(Attributes.FOLDER_PATH, Column.VISIBLE);
	    addColumn(Attributes.FILE_NAME, Column.VISIBLE);
	    addColumn(Attributes.MD5, Column.VISIBLE, cellRenderer, null);
	    addColumn(Attributes.CRC32, Column.VISIBLE, cellRenderer, null);

	    addColumn(ROM.Attributes.FPP_CHECK_SUM, Column.VISIBLE, cellRenderer, null);
	    addColumn(ROM.Attributes.OS1_CHECK_SUM, Column.VISIBLE, cellRenderer, null);
	    addColumn(ROM.Attributes.OS2_CHECK_SUM, Column.VISIBLE, cellRenderer, null);

	    addColumn(ROMVersion.Attributes.TYPE, Column.VISIBLE);
	    addColumn(ROMVersion.Attributes.REVISION, Column.VISIBLE);
	    addColumn(ROMVersion.Attributes.NORM, Column.VISIBLE);
	    addColumn(ROMVersion.Attributes.YEAR, Column.VISIBLE);
	    addColumn(ROMVersion.Attributes.COMMENT, Column.VISIBLE);
	    addColumn(ROMVersion.Attributes.PARTS, Column.VISIBLE);
	    addColumn(Attributes.MESSAGE, Column.VISIBLE);

	}

	@Override
	public Object getValueAt(int row, int column) {
	    WorkbookEntry entry = workbook.getEntry(row);
	    ROM rom = entry.getROM();
	    ROMVersion romVersion = entry.getROMVersion();
	    if (romVersion == null) {
		romVersion = ROMVersion.UNKNOWN;
	    }
	    switch (column) {
	    case Columns.ID:
		return Integer.valueOf(table.convertRowIndexToView(row) + 1);
	    case Columns.FOLDER_PATH:
		return entry.getFolderPath();
	    case Columns.FILE_NAME:
		return entry.getFileName();
	    case Columns.MD5:
		return entry.getMD5();
	    case Columns.CRC32:
		return entry.getCRC32();

	    case Columns.FPP_CHECK_SUM:
		return rom != null ? ROM.toHexBytes(rom.getFPP()) : "";
	    case Columns.OS1_CHECK_SUM:
		return rom != null ? ROM.toHexBytes(rom.getOS1()) : "";
	    case Columns.OS2_CHECK_SUM:
		return rom != null ? ROM.toHexBytes(rom.getOS2()) : "";

	    case Columns.TYPE:
		return romVersion.getType();
	    case Columns.REVISION:
		return romVersion.getRevision();
	    case Columns.NORM:
		return romVersion.getNorm();
	    case Columns.YEAR:
		return romVersion.getYear();
	    case Columns.COMMENT:
		return romVersion.getComment();
	    case Columns.PARTS:
		return romVersion.getParts();
	    case Columns.MESSAGE:
		return entry.getMessage();

	    default:
		throw new IllegalArgumentException("Invalid column " + column + ".");
	    }

	}

	@Override
	public boolean isCellEditable(int row, int column) {
	    return false;
	}

	@Override
	public int getRowCount() {
	    return workbook.getEntryCount();
	}
    }

    private Workbook workbook;

    public WorkbookEntriesPanel(Preferences preferences, Workbook workbook) {
	super(new TableModel(workbook), preferences, "workbookEntriesTable");

	this.workbook = workbook;
	workbook.setEntriesTableModel((AttributeTableModel)getTable().getModel());
    }

    /**
     * Sets the selected workbook entry, if it exists.
     * 
     * @param workbookEntry
     *            The workbook entry, not <code>null</code>.
     * 
     * @param attribute
     *            The attribute, not <code>null</code>.
     */
    public void setSelectedWorkbookEntry(WorkbookEntry workbookEntry, Attribute attribute) {
	if (workbookEntry == null) {
	    throw new IllegalArgumentException("Parameter 'workbookEntry' must not be null.");
	}
	int modelRowIndex = workbook.getUnmodifiableEntriesList().indexOf(workbookEntry);

	table.selectCell(modelRowIndex, attribute);
    }
}
