;===============================================
;Superchain loader (Super Turbo, Czechoslovakia)
;===============================================

;Binary load emulation
;Binary file being loaded is required to have a RUN segment

;This loader loads sequence of turbo files. Almost each turbo
;file represents one segment from the binary file.

;Information for the loader is stored in the NAME
;part of super turbo header

;Super turbo header
;Offset Meaning
;00     Always 183
;01     File type
;02-21    File name, specially interpreted
;   02    R - RUN  segment
;         I - INIT segment
;         D - DATA segment
;         X - DATA segment, followed by RUN. Loader performs JMP to RUN address  
;         Y - DATA Segment, followed by INIT. Loader emulates JSR(RUN address)
;         K - RUNINIT Segment
;   03-08 Informative number of segment
;   09    Sequential number of segment     
;   10-21 Blank
;22-23  LOAD address
;24-25  LENGTH of file
;26-27  RUN address (Used only for X and Y segment types)
;28     Checksum

;Note: Inverse letter at offset 2 indicates last file
;      X and Y segment types were designed to save pilot tones

;If error occurs, 4 tape beeps are emitted
;If file is not loaded in sequence, 1 tape beep is emitted imediattely after
;loading the super turbo header



.INCLUDE s_equates.asm


            *= 1800


L049F       jsr SHOWTITLE  ;Call subroutine that shows program title


CHAIN       lda #52        ;Switch on tape motor
            sta PACTL
            sta PBCTL

;----------------------------------------------
;Header decoding
;----------------------------------------------
                           
L04B1       jsr L0634           ;Call cleanup subroutine
            jsr L0645           ;Call speed detection subroutine
            jsr L05FC           ;Call subroutine that enables interrupts

L04BA       lda BUFRFL
            cmp #180            ;Determine Turbo 2000/Super Turbo
            bcc L04B1           ;Turbo 2000 not supported, return back


            lda #<HEADBFEN      ;Setup buffer
            sta BFENLO          
            lda #<HEADBUF
            sta BUFRLO
            lda #>HEADBUF
            sta BUFRHI
            lda #>HEADBFEN
            sta BFENHI

            lda #183            ;First byte in the block is 183
            jsr L059C           ;Call Super turbo block decoding subroutine
            bcc L04B1           ;If error occured, return back

ISMAEL      clc                 ;Expected segment is read ?
            lda HEADBUF+10    
            cmp EXPECTED      
            beq SETMBUF         ;Segment match, jump to main block decoding

BADSECT     jsr L05FC           ;Call subroutine that enables interrupts
            lda #255            ;One tape beep
            sta 764
            lda #1            
            jsr 65020
            jmp CHAIN           ;Return back

;----------------------------------------------
;Main block decoding
;----------------------------------------------
SETMBUF     lda HEADBUF+11+10   ;Setup buffer
            sta BUFRLO
            clc
            adc HEADBUF+13+10
            sta BFENLO
            lda HEADBUF+12+10
            sta BUFRHI
            adc HEADBUF+14+10
            sta BFENHI

                                
            lda #237            ;First byte in the block is 237
            jsr L059C           ;Call Super turbo block decoding subroutine
            bcs L04FB           ;If no error, jump to handle loaded block
     
            jsr L05FC           ;Call subroutine that enables interrupts
            lda #255            ;4 tape beeps
            sta 764
            lda #4
            jsr 65020
            jmp CHAIN           ;Return back

;----------------------------------------------
;Handling loaded block
;----------------------------------------------

L04FB       lda #60             ;Switch off tape motor
            sta PACTL
            sta PBCTL

            lda HEADBUF+1       ;Check the segment type
            and #127            ;Get rid of MSB

            cmp #82             ;Is it RUN ?
            beq NEXTCHAIN
            cmp #68             ;Is it DATA ?
            beq NEXTCHAIN 
            cmp #73             ;Is it INIT ?
            beq DOINIT
            cmp #88             ;Is it DATA+RUN ?
            beq DATARUN
            cmp #89             ;Is it DATA+INIT ?
            beq DATAINIT

RUNINIT    jmp DOINIT           ;RUN+INIT 


DATARUN    lda HEADBUF+15+10    ;DATA+RUN, store RUN address to 736,737
           sta 736
           lda HEADBUF+16+10
           sta 737
           jmp NEXTCHAIN

DATAINIT   lda HEADBUF+15+10    ;DATA+INIT,store INIT address to 738,739 
           sta 738              ;and perform init
           lda HEADBUF+16+10
           sta 739

DOINIT     jsr TRICK            ;Emulating JSR(738)
           jmp NEXTCHAIN        ;Go to load next file
TRICK      jmp (738)


NEXTCHAIN   lda HEADBUF+1       ;Last segment ?
            and #128
            beq NXC             ;Not last segment, continue

            jmp (736)           ;RUN loaded program

NXC         inc EXPECTED        ;Increase number of expected segment
            jmp CHAIN           ;Continue



;==============================================
;Block decoding subroutine
;Block is placed from: BUFRLO+256*BUFRHI
;                to:   (BFENLO+256*BFENHI)-1
;First byte of the block should be in
;acumulator register before subroutine
;is called        
;==============================================

L059C       sta LTEMP
            jsr L0634           ;Call subroutine for clearing status variables. Disable interrupts.
            php

L05A2       jsr L0692           ;Measure length of pulse
L05A5       ldy RECVDN          ;Count from MED
            jsr L0697           ;Wait for edge
            bcc L05A2           ;If Y wrapped, repeat
            cpy XMTDON
            bcs L05A5           ;If Y>HIGH repeat
            jsr L0697           ;Wait for edge
            bcc L05FB           ;If Y wrapped, end block decoding
            ldy BUFRFL
            jmp L05D6

L05BA       plp                 ;Store data to memory
            bne L05C5
            lda LTEMP
            eor NOCKSM
            bne L05FC
            beq L05D1
L05C5       ldy #0
            lda NOCKSM
            sta (BUFRLO),Y
            inc BUFRLO
            bne L05D1
            inc BUFRHI
L05D1       ldy BUFRFL
            iny
            iny
            php

L05D6       lda #1
            sta NOCKSM
L05DA       jsr L0692          ;Measure length of pulse
            bcc L05FB          ;If pulse is too long, end block decoding
            cpy RECVDN         ;Compate to MED
            rol NOCKSM         ;Rotate NOCKSM with carry, length >MED "1", length <MED "0"
            ldy BUFRFL
            bcc L05DA          ;If bit 7 was 0, repeat
            lda CHKSUM         ;Update checksum
            eor NOCKSM
            sta CHKSUM

            lda BUFRLO         ;Write number to buffer
            cmp BFENLO
            lda BUFRHI
            sbc BFENHI
            bcc L05BA
            lda #0
            cmp CHKSUM
L05FB       pla


L05FC       lda #64            ;Enable interrupts
            sta NMIEN
            sta IRQEN
            rts



;----------------------------------------------
;Turbo 2000 setup
;Because Superchain expects only super turbo,
;this part is omitted
;----------------------------------------------


;-----------------------------------------------
;Clear status variables, disable interrupts
;-----------------------------------------------
L0634       ldy #0
            sty STATUS
            sty CHKSUM
            sty NMIEN
            sty DMACLT
            sty IRQEN
            clc
            rts

;-----------------------------------------------
;ST speed detection
;-----------------------------------------------
L0645       jsr L0692           ;Measure length of pulse
            bcc L0645           ;If no CF, repeat
            lda #3
            sta LTEMP+1

L064E       ldy #0
            sty FREQ            ;{{

            ldx #3
L0654       jsr L0692           ;Measure length of 3 pulses {
            bcc L0645           ;No CF, => return to  L0645
            dex
            bne L0654           ;}
            tya                 ;A now holds number of measuring loop passes
            lsr A               ;A=A/2
            pha
            eor #255            ;A = 255-A
            sta BUFRFL          ;LO
            pla
            lsr A
            pha
            eor #255
            sta RECVDN          ;MED
            pla
            lsr A
            eor #255
            sta XMTDON          ;HI
            ldy #0
            jsr L0697           ;Wait for edge
            bcc L0645
L0677       ldy BUFRFL          ;Star from LO
            jsr L0692           ;Measure length of pulse
            bcc L0645
            cpy RECVDN          ;Compare with MED
            bcc L0645           ;If Y<MED, measure speed again
            inc FREQ
            bne L0677           ;256 pulses must have appropriate length
            dec LTEMP+1
            bpl L064E           ;Repeat 4 times }}

            ldy #0
            jsr L0697           ;Wait for edge
            bcc L0645           ;No CF, measure speed again
            rts


;-----------------------------------------------
;Measure length of pulse
;-----------------------------------------------
L0692       jsr L0697           ;Call wait for edge
            bcc L06B4           ;No CF, return

;-----------------------------------------------
;Wait for edge + colour effect
;-----------------------------------------------
L0697       lda RANDOM
            and #228
            ora STATUS
            lsr A
            and LTEMP+1
            sta COLBK

L06A4       iny                 ;Measuring loop
            beq L06B4
            lda SKSTAT
            and #16
            cmp STATUS
            beq L06A4
            sta STATUS
            sec
            rts
L06B4       clc
            rts



EXPECTED    .byte 0            ;Expected segment number


;-----------------------------------------------
;Space for strings
;-----------------------------------------------
HEADBUF     .byte 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
HEADBFEN    .byte 0,0
TITLE       .byte "TS SUPERCHAIN",155

;-----------------------------------------------
;Subroutine that shows title
;-----------------------------------------------
SHOWTITLE   ldx #<TITLE
            ldy #>TITLE
            jsr $C642
            lda #1
            sta 764
            jsr 65020

            lda #0            ;Reset coldstart flag
            sta 580        
            lda #1            ;Indicate disk boot ok
            sta 9
            rts
