#include <iostream>
#include <iomanip>
#include <string>
#include <cstring>
#include <cstdlib>

#include "options.h"

using namespace std;
using namespace xex2cas;

Options::Options() {

    /*Defaults - mode*/
    operationMode = MODE_BINARY;

    /*Defaults - common*/
    overwriteOutputs = false;
    useShorterLeader = false;
    useFasterTransferSpeed = false;
    useEmptyFUJIChunk = false;

    /*Defaults - BINARY*/
    loaderType = LDR_MODERN_STDBLOAD2;
    elongateInitIRGs = 0;
    programName = "";
    companyName = "";
    useLongerIRGs = false;

    /*Defaults - BOOTBASIC*/
    basicInitializerType = BINIT_LAUNCHBAS;
    
    /*Defaults - PLAIN*/
    useLongIRGs = false;
    eofTrickType = EOF_TRICK_NONE;

    /*Input and output files*/
    inputFile = "";
    outputFile = "";

}

void Options::displayUsage() {

    cout << "Usage:" << endl;
    cout << "xex2cas [mode] [option1] [option2] ... infile.ext [outfile.cas]" << endl;
    cout << "Modes: " << endl;
    cout << " -binary    Conversion of Atari DOS 2 binary file (default)" << endl;
    cout << " -bootbasic Conversion of Atari BASIC file to a form bootable from tape" << endl;
    cout << " -plain     Conversion of a plain file" << endl;
    
    cout << "Common options:" << endl;
    cout << " -r          Overwrite output file" << endl;
    cout << " -s          Use a shorter leader (12 s)" << endl;
    cout << " -f          Faster transfer speed (720 bd)" << endl;
    cout << " -e          Generate empty FUJI tape image chunk" << endl;
    cout << " -h,-help,-? Display usage instructions" << endl;
    
    cout << "Options for the BINARY mode:" << endl ;
    cout << " -l<n>       Binary loader selection:" << endl;
    cout << "             n=0 No binary loader, n=1 STDBLOAD 2a (default), " << endl;
    cout << "             n=2 XL/XE Exclamation mark, n=3 Original Exclamation mark, " << endl;
    cout << "             n=4 L.K. Avalon loader, n=5 Fancy loader, n=6 SIECOD loader." << endl;
    cout << " -i<n>       Elongate IRGs after INIT segments to n seconds (0 to 99)" << endl;
    cout << " -np'<name>' Set program name displayed by the binary loader " << endl;
    cout << " -nc'<name>' Set company name displayed by the binary loader " << endl;
    cout << " -g          Generate longer (350 ms) IRGs" << endl;
    
    cout << "Options for the BOOTBASIC mode:" << endl;
    cout << " -bi<n>      BASIC initializer. n=0 LAUNCHBAS (default), n=1 BAS2CAS" << endl;
    
    cout << "Options for the PLAIN mode:" << endl;
    cout << " -gl         Long (3000 ms) IRGs " << endl;
    cout << " -et<n>      Trick with last record:" << endl;
    cout << "             n=0 None (default), n=1 Data in EOF record, n=2 Last full record" << endl;

}

int Options::parse(int argc, char* argv[]) {

    /*If there were no options, bail out*/
    if (argc == 1) {
        cout << "No command line arguments specified" << endl;
        displayUsage();
        return OPTIONS_HELP;
    }

    /*Syntax flags - Common*/
    bool specOverWriteOutputs = false;
    bool specUseShortLeader = false;
    bool specUseFasterTransferSpeed = false;
    bool specUseEmptyFUJIChunk = false;
    bool specDisplayUsage = false;

    /*Syntax flags - BINARY*/
    bool specLoaderType = false;
    bool specElongateInitIRGs = false;
    bool specProgramName = false;
    bool specCompanyName = false;
    bool specUseLongerIRGs = false;
    
    /*Syntax flags - BOOTBASIC*/
    bool specBasicInitializerType = false;
    
    /*Syntax flags - PLAIN*/
    bool specEofTrickType = false;
    bool specUseLongIRGs = false;

    /*Syntax flags - Input and output file*/
    bool specInputFile = false;
    bool specOutputFile = false;

    /*Processing flags and fields*/
    string rawOpt;
    bool isSyntaxCorrect = true;

    /*Determine operation mode and index of the firts option*/
    rawOpt = argv[1];
    int optStart = 1;

    if (rawOpt == "-bootbasic") {
        operationMode = MODE_BOOTBASIC;
        optStart = 2;
    } else if (rawOpt == "-plain") {
        operationMode = MODE_PLAIN;
        optStart = 2;
    } else if (rawOpt == "-binary") {
        operationMode = MODE_BINARY;
        optStart = 2;
    }
    

    /*Process options*/
    for (int i = optStart; i < argc; i++) {

        rawOpt = argv[i];

        /*Handle options that start with hyphen*/
        if (rawOpt.find("-") == 0) {

            /*Options must precede input and output files*/
            if (specInputFile == true || specOutputFile == true) {
                isSyntaxCorrect = false;
                break;
            }

            /*Common options*/

            /*Overwrite outputs*/
            if (rawOpt == "-r") {
                if (specOverWriteOutputs == true) {
                    isSyntaxCorrect = false;
                    break;
                }
                overwriteOutputs = true;
                specOverWriteOutputs = true;
                continue;
            }

            /*Shorter leader*/
            if (rawOpt == "-s") {
                if (specUseShortLeader == true) {
                    isSyntaxCorrect = false;
                    break;
                }
                useShorterLeader = true;
                specUseShortLeader = true;
                continue;
            }

            /*Faster transfer speed*/
            if (rawOpt == "-f") {
                if (specUseFasterTransferSpeed == true) {
                    isSyntaxCorrect = false;
                    break;
                }
                useFasterTransferSpeed = true;
                specUseFasterTransferSpeed = true;
                continue;
            }

            /*Empty FUJI tape image chunk*/
            if (rawOpt == "-e") {
                if (specUseEmptyFUJIChunk == true) {
                    isSyntaxCorrect = false;
                    break;
                }
                useEmptyFUJIChunk = true;
                specUseEmptyFUJIChunk = true;
                continue;
            }

            /*Help always terminates processing*/
            if (rawOpt == "-h" || rawOpt == "-help" || rawOpt == "-?") {
                displayUsage();
                specDisplayUsage = true;
                return OPTIONS_HELP;
            }


            /*BINARY mode options*/
            if (operationMode == MODE_BINARY) {
                /*Process loader type*/
                if (rawOpt.find("-l") == 0) {

                    /*Check repetition*/
                    if (specLoaderType == true) {
                        isSyntaxCorrect = false;
                        break;
                    }

                    /*Check length*/
                    if (rawOpt.length() != 3) {
                        isSyntaxCorrect = false;
                        break;
                    }

                    specLoaderType = true;

                    /*Check loader specification*/
                    if (rawOpt[2] == '0') {
                        loaderType = LDR_NONE;
                        continue;
                    }
                    if (rawOpt[2] == '1') {
                        loaderType = LDR_MODERN_STDBLOAD2;
                        continue;
                    }
                    if (rawOpt[2] == '2') {
                        loaderType = LDR_XL_FIXED_EXMA;
                        continue;
                    }
                    if (rawOpt[2] == '3') {
                        loaderType = LDR_ORIGINAL_EXMA;
                        continue;
                    }
                    if (rawOpt[2] == '4') {
                        loaderType = LDR_LKAVALON_BL;
                        continue;
                    }
                    if (rawOpt[2] == '5') {
                        loaderType = LDR_FANCY;
                        continue;
                    }
                    if (rawOpt[2] == '6') {
                        loaderType = LDR_SIECOD;
                        continue;
                    }

                    /*Invalid loader type specification*/
                    isSyntaxCorrect = false;
                    break;
                }

                /*Process elongated IRGs after INIT segments*/
                if (rawOpt.find("-i") == 0) {

                    /*Check repetition*/
                    if (specElongateInitIRGs == true) {
                        isSyntaxCorrect = false;
                        break;
                    }

                    /*Check length*/
                    int optLen = rawOpt.length();
                    if (optLen < 3 || optLen > 4) {
                        isSyntaxCorrect = false;
                        break;
                    }

                    /*Check value*/
                    string digits = rawOpt.substr(2, optLen - 2);
                    char* iptr;
                    const char* cstr = digits.c_str();
                    long int value = strtol(cstr, &iptr, 10);
                    if (value < 0 || value > 99 || iptr != cstr + digits.length()) {
                        isSyntaxCorrect = false;
                        break;
                    }

                    /*Value is OK*/
                    elongateInitIRGs = value;
                    specElongateInitIRGs = true;
                    continue;

                }

                if (rawOpt.find("-nc") == 0 || rawOpt.find("-np") == 0) {

                    /*Prepare pointers*/
                    bool *specPtr;
                    string *namePtr;
                    int maxLen;

                    /*Determine if program name or company name*/
                    if (rawOpt.find("-nc") == 0) {
                        specPtr = &specCompanyName;
                        namePtr = &companyName;
                        maxLen = 20;
                    } else {
                        specPtr = &specProgramName;
                        namePtr = &programName;
                        maxLen = 34;
                    }

                    /*Check repetition*/
                    if (*specPtr == true) {
                        isSyntaxCorrect = false;
                        break;
                    }
                    /*Check if it has proper length*/
                    if (rawOpt.length() < 5) {
                        isSyntaxCorrect = false;
                        break;
                    }
                    /*Check if it starts with single quote*/
                    if (rawOpt[3] != '\'') {
                        isSyntaxCorrect = false;
                        break;
                    }
                    /*Check if it ends with single quote*/
                    if (rawOpt[rawOpt.length() - 1] != '\'') {
                        isSyntaxCorrect = false;
                        break;
                    }
                    /*Check maximum length*/
                    *namePtr = rawOpt.substr(4, rawOpt.length() - 5);
                    if ((*namePtr).length() > maxLen) {
                        isSyntaxCorrect = false;
                        break;
                    }
                    *specPtr = true;
                    continue;
                }

                if (rawOpt == "-g") {
                    if (specUseLongerIRGs == true) {
                        isSyntaxCorrect = false;
                        break;
                    }
                    useLongerIRGs = true;
                    specUseLongerIRGs = true;
                    continue;
                }


            } /*Binary mode*/
            
            /*BOOTBASIC mode options*/
            if (operationMode == MODE_BOOTBASIC) {
                
                /*Process initializer type*/
                if (rawOpt.find("-bi") == 0) {

                    /*Check repetition*/
                    if (specBasicInitializerType == true) {
                        isSyntaxCorrect = false;
                        break;
                    }

                    /*Check length*/
                    if (rawOpt.length() != 4) {
                        isSyntaxCorrect = false;
                        break;
                    }

                    specLoaderType = true;

                    /*Check loader specification*/
                    if (rawOpt[3] == '0') {
                        basicInitializerType = BINIT_LAUNCHBAS;
                        continue;
                    }
                    if (rawOpt[3] == '1') {
                        basicInitializerType = BINIT_BAS2CAS;
                        continue;
                    }

                    /*Invalid loader type specification*/
                    isSyntaxCorrect = false;
                    break;
                }

                
            }
            
            /*PLAIN mode options*/
            if (operationMode == MODE_PLAIN) {
                
                /*Long IRGs*/
                if (rawOpt == "-gl") {
                    if (specUseLongIRGs == true) {
                        isSyntaxCorrect = false;
                        break;
                    }
                    useLongIRGs = true;
                    specUseLongIRGs = true;
                    continue;
                }
                
                /*EOF trick type*/
                if (rawOpt.find("-et") ==0 && rawOpt.length()==4) {
                    if (specEofTrickType == true) {
                        isSyntaxCorrect = false;
                        break;
                    }
                    specEofTrickType = true;
                    
                    if (rawOpt[3]=='0') {
                        eofTrickType = EOF_TRICK_NONE;
                        continue;
                    }
                    
                    if (rawOpt[3]=='1') {
                        eofTrickType = EOF_TRICK_DATA_IN_EOF;
                        continue;
                    }
                    
                    if (rawOpt[3]=='2') {
                        eofTrickType = EOF_TRICK_LAST_FULL;
                        continue;
                    }
                    
                    isSyntaxCorrect = false;
                    break;
                }
                
            } /*Plain*/
            
            /*Unknown argument has been specified*/
            isSyntaxCorrect = false;
            break;
        }
        
        /*Input and output files*/
        else {

            /*Not more than 2 files*/
            if (specInputFile == true && specOutputFile == true) {
                isSyntaxCorrect = false;
                break;
            }

            /*Get input tape image*/
            if (specInputFile == false) {
                inputFile = rawOpt;
                specInputFile = true;
            } else {
                outputFile = rawOpt;
                specOutputFile = true;
            }

            /*No wildcards*/
            if (rawOpt.find("*") != rawOpt.npos || rawOpt.find("?") != rawOpt.npos) {
                isSyntaxCorrect = false;
                break;
            }

            continue;

        }


    }

    /*How the syntax processing ended*/
    if (isSyntaxCorrect == false) {
        cout << "Error: Invalid command line arguments specified" << endl;
        cout << "       Argument in error: " << rawOpt << endl;
        displayUsage();
        return OPTIONS_ERROR;
    } else {
        return OPTIONS_OK;
    }

}
