/* XEX2CAS 2.4
 * 
 * XEX2CAS 2.4 is a standalone command line utility that converts files to
 * standard tape records.
 * 
 * I have placed this work in the Public Domain, thereby relinquishing
 * all copyrights. Everyone is free to use, modify, republish, sell or give away
 * this work without prior consent from anybody.
 * 
 * Michael Kalouš (BAKTRA Software)
 * zylon@post.cz
 */

#include <iostream> 
#include <fstream>  
#include <cstring>
#include <cstdlib>
#include <vector>
#include <sstream>
#include <iomanip>
#include <string>

#include "tapeimagewriter.h"
#include "dos2binary.h"
#include "segment.h"
#include "options.h"
#include "loaders.h"

using namespace std;
using namespace xex2cas;

string createNameForLoader(string fn,int maxLength);
string ascii2Internal(string &s,bool toColor1,bool toColor2);
int convertBinaryFile(unsigned char *inFileBuffer,int inFileLength, ofstream &casFile,Options &opts);
int convertPlainFile(unsigned char *inFileBuffer,int inFileLength, ofstream &casFile,Options &opts);
int convertBootBasic(unsigned char *inFileBuffer,int inFileLength, ofstream &casFile,Options &opts);

/* Mainline code*/
int _CRT_glob = 0;

int main(int argc, char *argv[]) {

    /*Display program title*/
    cout << "XEX2CAS 2.4 - Convert Atari DOS 2 Binary File to standard tape records" << endl;

    /*Parse options*/
    Options opts;
    int parseResult = opts.parse(argc, argv);

    /*In case of bad syntax or help requisition, terminate processing*/
    if (parseResult == opts.OPTIONS_ERROR) {
        return -1;
    }
    if (parseResult == opts.OPTIONS_HELP) {
        return 0;
    }

    /*Try to open the input file*/
    ifstream inFile;
    inFile.open(opts.inputFile.c_str(), ios_base::binary);
    if (inFile.fail() == true) {
        cout << "Error: Unable to open the input file " << opts.inputFile << endl;
        return -1;
    }
    /*Determine size of the input file*/
    inFile.seekg(0, ios_base::end);
    long inFileLength = inFile.tellg();

    /*If the file is bigger than 4 MB, do not process it*/
    if (inFileLength > 4 * 1024 * 1024) {
        cout << "Error: Input file size exceeds 4 MB" << endl;
        return -1;
    }
    inFile.seekg(0, ios_base::beg);

    /*Construct name of the output file if needed*/
    string outFileName;
    if (opts.outputFile == "") {
        /*Check for last period*/
        int periodPos = opts.inputFile.find_last_of('.');

        /*If no period found, just append the .cas extension*/
        if (periodPos == opts.inputFile.npos) {
            outFileName = opts.inputFile + ".cas";
        } else {
            outFileName = (opts.inputFile.substr(0, periodPos)) + ".cas";
        }
    } else {
        outFileName = opts.outputFile;
    }

    /*Display input and output files*/
    cout << opts.inputFile << " >> ";
    cout << outFileName << endl;
     
    /*Check existence of the output file*/
    ifstream casFilePeek;
    casFilePeek.open(outFileName.c_str(), ios_base::binary);
    if (casFilePeek.fail() == false) {
        if (opts.overwriteOutputs == true) {
            cout << "Warning: Overwriting output file" << endl;
        } else {
            cout << "Error: Output file already exists" << endl;
            return -1;
        }
        casFilePeek.close();
    }

    /*Try to open the output file*/
    ofstream casFile;
    casFile.open(outFileName.c_str(), ios_base::binary);
    if (casFile.fail() == true) {
        cout << "Error: Unable to open output file " << outFileName << endl;
        return -1;
    }

    /*Read input file to memory*/
    unsigned char* inFileBuffer = new unsigned char[inFileLength];
    for (int i = 0; i < inFileLength; i++) {
        inFileBuffer[i] = inFile.get();
    }

    /*Check if there were read errors*/
    if (inFile.fail()) {
        cout << "Error: Failed to read from the input file" << endl;
        return -1;
    }

    /*If there were no errors, close the input file*/
    inFile.close();
    
    
    /*Process file according to the operation mode*/
    int conversionResult = 0;
    switch (opts.operationMode) {
        case Options::MODE_BINARY : {
            conversionResult = convertBinaryFile(inFileBuffer,inFileLength,casFile,opts);
            break;
        }
        case Options::MODE_BOOTBASIC : {
            conversionResult = convertBootBasic(inFileBuffer,inFileLength,casFile,opts);
            break;
        }
        case Options::MODE_PLAIN : {
            conversionResult = convertPlainFile(inFileBuffer,inFileLength,casFile,opts);
            break;
        }
    }
    
    /*Check status of the output file*/
    if (casFile.bad() == true || casFile.fail() == true) {
        cout << "Error: Failed to write to the output file" << endl;
        return -1;
    }

    casFile.close();
    cout << "Done" << endl;

    return 0;
}

/*Convert Atari DOS 2 binary file*/
int convertBinaryFile(unsigned char *inFileBuffer,int inFileLength, ofstream &casFile,Options &opts) {

    vector<int> emptyInitRBAs;
    vector<int> d2bInitRBAs = emptyInitRBAs;

    /*Check if the input file is a DOS 2 binary file*/
    DOS2BinaryFile d2b;
    int parseResult = d2b.parse(inFileBuffer, inFileLength);

    switch (parseResult) {
        case (DOS2BinaryFile::D2BF_ALIEN):
        {
            cout << "Warning: Input file is not a binary file" << endl;
            break;
        }
        case (DOS2BinaryFile::D2BF_CORRUPT):
        {
            cout << "Warning: Input binary file is corrupt" << endl;
            cout << d2b.getParseError() << endl;
            cout << "Listing valid segments:" << endl;
            d2b.dump();
            d2bInitRBAs = d2b.getInitRBAs();
            break;
        }
        default:
        {
            d2bInitRBAs = d2b.getInitRBAs();
        }
    }
    

    /*Write tape image*/
    TapeImageWriter writer;

    /*Prepare FUJI tape image chunk data*/
    string fujiChunkData = (opts.useEmptyFUJIChunk == true) ?
            string("") : string("File generated by XEX2CAS 2.0");
    
    int minimumInitIRGDuration=500;
    bool siecodFormat = false;
    int baseIRGDuration = opts.useLongerIRGs?350:250; 
    
    /*Write loader*/
    if (opts.loaderType != opts.LDR_NONE) {

        unsigned char* loaderData;
        int loaderDataLen;
        bool loaderUseDataInEOFTrick = false;
        bool loaderForceLastFullBlock = false;
        

        switch (opts.loaderType) {
            case Options::LDR_ORIGINAL_EXMA:
            {
                loaderData = exma_original;
                loaderDataLen = 564;
                break;
            }
            case Options::LDR_XL_FIXED_EXMA:
            {
                loaderData = exma_xlxe;
                loaderDataLen = 564;
                break;
            }
            case Options::LDR_LKAVALON_BL:
            {
                loaderData = lkavalon_bl;
                loaderDataLen = 256;
                loaderUseDataInEOFTrick = true;
                loaderForceLastFullBlock = true;
                minimumInitIRGDuration=850;
                break;
            }
            case Options::LDR_MODERN_STDBLOAD2:
            {
                loaderData = stdbload_2a;
                loaderDataLen = 481;
                loaderUseDataInEOFTrick = true;
                /*Clear file name*/
                memset(loaderData + (0x09A4 - 0x07E8), ' ', 34);
                
                /*Determine file name if needed*/
                string fn = opts.programName;
                if (fn == "") fn = createNameForLoader(opts.inputFile,34);
                /*File name is guaranteed to be up to 34 characters long*/
                memcpy(loaderData + (0x09A4 - 0x07E8), fn.c_str(), fn.length());
                break;
            }
            case Options::LDR_FANCY: {
                loaderData = fancy_loader;
                loaderDataLen = 638;
                
                /*Clear file name and company name*/
                memset(loaderData + 597,0,20);
                memset(loaderData + 617,0,20);
                
                /*Prepare program and company name*/
                string fn = opts.programName;
                string cn = opts.companyName;
                if (fn=="") {
                    fn=createNameForLoader(opts.inputFile,20);
                }
                else {
                    if (fn.length()>20) fn=fn.substr(0,20);
                }
                /*Place program and company name to the loader skeleton*/
                memcpy(loaderData + 617,ascii2Internal(fn,false,true).c_str(),20);
                memcpy(loaderData + 597,ascii2Internal(cn,true,false).c_str(),20);
                break;
            }
            
            case Options::LDR_SIECOD: {
                baseIRGDuration = 80;
                writer.writeRawData(casFile,siecod_loader,2615);
                siecodFormat=true;
                if (d2b.hasInitSegment()==true) {
                    cout << "Warning: Binary file has INIT segment(s) not supported by the SIECOD loader" << endl;
                }
                break;
            }
        }

        /*Call tape image writer to output the selected loader. The SIECOD
          loader is written differently*/
        if (opts.loaderType != Options::LDR_SIECOD) {
        writer.writeFile(
                casFile,
                loaderData,
                loaderDataLen,
                loaderUseDataInEOFTrick,
                loaderForceLastFullBlock,
                true,
                opts.useFasterTransferSpeed,
                250,
                opts.useShorterLeader,
                emptyInitRBAs,
                0,
                0,
                fujiChunkData,
                false
                );
        }
    }

    /*Write the file*/
    writer.writeFile(
            casFile,
            inFileBuffer,
            inFileLength,
            false,
            false,
            (opts.loaderType==Options::LDR_NONE)?true:false,
            opts.useFasterTransferSpeed,
            baseIRGDuration,
            opts.useShorterLeader,
            d2bInitRBAs,
            opts.elongateInitIRGs,
            minimumInitIRGDuration,
            fujiChunkData,
            siecodFormat
            );
    
}

/*Convert plain file*/
int convertPlainFile(unsigned char *inFileBuffer,int inFileLength, ofstream &casFile,Options &opts) {
    
    vector<int> emptyInitRBAs;
    
    /*Prepare FUJI tape image chunk data*/
    string fujiChunkData = (opts.useEmptyFUJIChunk == true) ?
            string("") : string("File generated by XEX2CAS 2.4");
    
    
    bool dataInEof = false;
    bool lastFull = false;
    
    /*Prepare EOF trick, if any*/
    switch (opts.eofTrickType) {
        case Options::EOF_TRICK_DATA_IN_EOF : {
            dataInEof = true;
            break;
        }
        case Options::EOF_TRICK_LAST_FULL: {
            lastFull = true;
            break;
        }
    }
    
    /*Write the file*/
    TapeImageWriter writer;
    writer.writeFile(
            casFile,
            inFileBuffer,
            inFileLength,
            dataInEof,
            lastFull,
            true,
            opts.useFasterTransferSpeed,
            opts.useLongIRGs?3000:250,
            opts.useShorterLeader,
            emptyInitRBAs,
            0,
            0,
            fujiChunkData,
            false
            );

}

/*Convert tokenized Atari BASIC to a bootable form*/
int convertBootBasic(unsigned char *inFileBuffer,int inFileLength, ofstream &casFile,Options &opts) {
    
    vector<int> emptyInitRBAs;
    
    /*Prepare FUJI tape image chunk data*/
    string fujiChunkData = (opts.useEmptyFUJIChunk == true) ?
            string("") : string("File generated by XEX2CAS 2.4");
    
    TapeImageWriter writer;
    
    unsigned char* initializerData;
    int initializerDataLen;
    
    if (opts.basicInitializerType == Options::BINIT_LAUNCHBAS) {
        initializerData = basic_init_launchbas;
        initializerDataLen = 222;
    }
    else {
        initializerData = basic_init_bas2cas;
        initializerDataLen = 257;
    }
    
    
    /*Write the Atari BASIC initializer*/
    writer.writeFile(
            casFile,
            initializerData,
            initializerDataLen,
            false,
            false,
            true,
            opts.useFasterTransferSpeed,
            250,
            opts.useShorterLeader,
            emptyInitRBAs,
            0,
            0,
            fujiChunkData,
            false
            );
    
    /*Write the file*/
    writer.writeFile(
            casFile,
            inFileBuffer,
            inFileLength,
            false,
            false,
            false,
            opts.useFasterTransferSpeed,
            250,
            opts.useShorterLeader,
            emptyInitRBAs,
            0,
            0,
            fujiChunkData,
            false
            );
}

string createNameForLoader(string fn,int maxLen) {

    /*Determine directory/folder separator*/
#if defined(_WIN32) 
    const char separator = '\\';
#elif defined(_WIN64)    
    const char separator = '\\';
#else
    const char separator = '/';
#endif

    /*Check the position of the last separator in the file name*/
    int sepPos = fn.find_last_of(separator);

    /*If there is no separator, then use the first 34 characters*/
    if (sepPos == fn.npos) {
        return fn.length() <= maxLen ? fn : fn.substr(0, maxLen);
    }

    /*Otherwise, use the first 34 characters of what's behind the separator,
     *if there is anything behind, of course
     */
    if (sepPos == fn.length() - 1) return "";

    fn = fn.substr(sepPos + 1);
    return fn.length() <= maxLen ? fn : fn.substr(0, maxLen);
}

/*Convert ASCII string to the internal code, center in 20 columns*/
string ascii2Internal(string &s,bool toColor1,bool toColor2) {
    
    string retString = "";
    int numSpaces = (20-s.length())/2;
    
    unsigned char blank = (unsigned char)0;
    for (int i=0;i<numSpaces;i++) {
        retString+=blank;
    }
    
    int l = s.length();
    unsigned char c;
    unsigned char ic;
    
    for (int i=0;i<l;i++) {
        c = s[i];
        if (c>127) c-=128;
        
        if (c>=32 && c<96) {
            ic = c-32;
        }
        else if (c<32) {
            ic = c+64;
        }
        else {
            ic = c;
        }
        
        if (toColor1 && ic>64) {
            ic-=64;
        }
        if (toColor2 && ic<64) {
            ic+=64;
        }
        retString+=ic;
    }
    
    int rightPad = 20-retString.length();
    for (int i=0;i<rightPad;i++) {
        retString += blank;
    }
    
    return retString;
}