#include <iostream>
#include <iomanip>
#include <string>
#include <cstring>
#include <cstdlib>

#include "options.h"

using namespace std;
using namespace xex2cas;

Options::Options() {
    doNotPrependLoader = false;
    useShorterLeader = false;
    useFasterTransferSpeed = false;
    useLongerIRGs = false;
    useEmptyFUJIChunk = false;
    initIRGs = 0;
    overwriteOutputs = false;
    loaderType = LDR_MODERN_EXMA;
    filename = "";
    inputFile = "";
    outputFile = "";
}

void Options::displayUsage() {

    cout << "Usage:" << endl;
    cout << "xex2cas [option1] [option2] ... infile.xex [outfile.cas]" << endl;
    cout << "Options: " << endl;
    cout << "-b          Do not prepend any binary loader" << endl;
    cout << "-s          Use a shorter leader (12 s)" << endl;
    cout << "-f          Faster transfer speed (720 bd)" << endl;
    cout << "-g          Generate longer (350 ms) inter-record-gaps (IRGs)" << endl;
    cout << "-i<n>       Elongate IRGs after INIT segments to n seconds (0 to 99)" << endl;
    cout << "-r          Overwrite output file" << endl;
    cout << "-n'Name'    Set name displayed by the binary loader (modern binary loader only)" << endl;
    cout << "            The name can be up to 34 characters long" << endl;
    cout << "-l<n>       Binary loader selection" << endl;
    cout << "            n=0 Use modern XL/XE only binary loader (default)" << endl;
    cout << "            n=1 Use ! loader updated for XL/XE computers" << endl;
    cout << "            n=2 Use original ! loader" << endl;
    cout << "-e          Generate empty FUJI tape image chunk" << endl;
    cout << "-h,-help,-? Display usage instructions" << endl << endl;
}

int Options::parse(int argc, char* argv[]) {

    /*If there were no options, bail out*/
    if (argc == 1) {
        cout << "No command line arguments specified" << endl;
        displayUsage();
        return OPTIONS_HELP;
    }

    bool specDoNotPrependLoader = false;
    bool specUseShortLeader = false;
    bool specUseFasterTransferSpeed = false;
    bool specUseLongerIRGs = false;
    bool specUseEmptyFUJIChunk = false;
    bool specInitIRGs = false;
    bool specOverWriteOutputs = false;
    bool specLoaderType = false;
    bool specFilename = false;
    bool specDisplayUsage = false;

    bool isSyntaxCorrect = true;
    bool specInputFile = false;
    bool specOutputFile = false;

    string rawOpt;

    for (int i = 1; i < argc; i++) {

        rawOpt = argv[i];

        /*Handle options that start with hyphen*/
        if (rawOpt.find("-") == 0) {

            /*Options must precede input and output files*/
            if (specInputFile == true || specOutputFile == true) {
                isSyntaxCorrect = false;
                break;
            }

            if (rawOpt == "-b") {
                if (specDoNotPrependLoader == true) {
                    isSyntaxCorrect = false;
                    break;
                }
                doNotPrependLoader = true;
                specDoNotPrependLoader = true;
                continue;
            }

            if (rawOpt == "-s") {
                if (specUseShortLeader == true) {
                    isSyntaxCorrect = false;
                    break;
                }
                useShorterLeader = true;
                specUseShortLeader = true;
                continue;
            }
            if (rawOpt == "-f") {
                if (specUseFasterTransferSpeed == true) {
                    isSyntaxCorrect = false;
                    break;
                }
                useFasterTransferSpeed = true;
                specUseFasterTransferSpeed = true;
                continue;
            }
            if (rawOpt == "-g") {
                if (specUseLongerIRGs == true) {
                    isSyntaxCorrect = false;
                    break;
                }
                useLongerIRGs = true;
                specUseLongerIRGs = true;
                continue;
            }

            if (rawOpt == "-r") {
                if (specOverWriteOutputs == true) {
                    isSyntaxCorrect = false;
                    break;
                }
                overwriteOutputs = true;
                specOverWriteOutputs = true;
                continue;
            }
            
            if (rawOpt == "-e") {
                if (specUseEmptyFUJIChunk == true) {
                    isSyntaxCorrect=false;
                    break;
                }
                useEmptyFUJIChunk = true;
                specUseEmptyFUJIChunk = true;
                continue;
            }

            /*Help always terminates processing*/
            if (rawOpt == "-h" || rawOpt == "-help" || rawOpt == "-?") {
                displayUsage();
                specDisplayUsage = true;
                return OPTIONS_HELP;
            }

            /*Process loader type*/
            if (rawOpt.find("-l") == 0) {

                /*Check repetition*/
                if (specLoaderType == true) {
                    isSyntaxCorrect = false;
                    break;
                }

                /*Check length*/
                if (rawOpt.length() != 3) {
                    isSyntaxCorrect = false;
                    break;
                }

                specLoaderType = true;

                /*Check loader specification*/
                if (rawOpt[2] == '0') {
                    loaderType = LDR_MODERN_EXMA;
                    continue;
                }
                if (rawOpt[2] == '1') {
                    loaderType = LDR_XL_FIXED_EXMA;
                    continue;
                }
                if (rawOpt[2] == '2') {
                    loaderType = LDR_ORIGINAL_EXMA;
                    continue;
                }

                /*Invalid loader type specification*/
                isSyntaxCorrect = false;
                break;
            }

            /*Process displayed name*/
            if (rawOpt.find("-n") == 0) {
                /*Check repetition*/
                if (specFilename == true) {
                    isSyntaxCorrect = false;
                    break;
                }
                /*Check if it has proper length*/
                if (rawOpt.length() < 4) {
                    isSyntaxCorrect = false;
                    break;
                }
                /*Check if it starts with single quote*/
                if (rawOpt[2] != '\'') {
                    isSyntaxCorrect = false;
                    break;
                }
                /*Check if it ends with single quote*/
                if (rawOpt[rawOpt.length() - 1] != '\'') {
                    isSyntaxCorrect = false;
                    break;
                }
                filename = rawOpt.substr(3, rawOpt.length() - 4);
                if (filename.length() > 34) {
                    isSyntaxCorrect = false;
                    break;
                }

                specFilename = true;
                continue;
            }

            /*Process elongated IRGs after INIT segments*/
            if (rawOpt.find("-i") == 0) {

                /*Check repetition*/
                if (specInitIRGs == true) {
                    isSyntaxCorrect = false;
                    break;
                }

                /*Check length*/
                int optLen = rawOpt.length();
                if (optLen < 3 || optLen > 4) {
                    isSyntaxCorrect = false;
                    break;
                }

                /*Check value*/
                string digits = rawOpt.substr(2, optLen - 2);
                char* iptr;
                const char* cstr = digits.c_str();
                long int value = strtol(cstr,&iptr,10);
                if (value < 0 || value > 99 || iptr!=cstr+digits.length()) {
                    isSyntaxCorrect = false;
                    break;
                }

                /*Value is OK*/
                initIRGs = value;
                specInitIRGs = true;
                continue;

            }

            /*Unknown argument has been specified*/
            isSyntaxCorrect = false;
            break;
        }            /*Input and output files*/
        else {

            /*Not more than 2 files*/
            if (specInputFile == true && specOutputFile == true) {
                isSyntaxCorrect = false;
                break;
            }

            /*Get input tape image*/
            if (specInputFile == false) {
                inputFile = rawOpt;
                specInputFile = true;
            } else {
                outputFile = rawOpt;
                specOutputFile = true;
            }

            /*No wildcards*/
            if (rawOpt.find("*") != rawOpt.npos || rawOpt.find("?") != rawOpt.npos) {
                isSyntaxCorrect = false;
                break;
            }

            continue;

        }


    }

    /*How the syntax processing ended*/
    if (isSyntaxCorrect == false) {
        cout << "Error: Invalid command line arguments specified" << endl;
        cout << "       Argument in error: " << rawOpt << endl << endl;
        displayUsage();
        return OPTIONS_ERROR;
    } else {
        return OPTIONS_OK;
    }

}

void Options::dump() {

    cout << "Options in effect:" << endl;
    cout << "Do not prepend binary loader: " << (doNotPrependLoader ? "YES" : "NO") << endl;
    cout << "Use shorter leader: " << (useShorterLeader ? "YES" : "NO") << endl;
    cout << "Use faster transfer speed: " << (useFasterTransferSpeed ? "YES" : "NO") << endl;
    cout << "Use longer IRGs: " << (useLongerIRGs ? "YES" : "NO") << endl;
    cout << "Overwrite output files: " << (overwriteOutputs ? "YES" : "NO") << endl;
    cout << "Generate empty FUJI tape image chunk: " << (useEmptyFUJIChunk ? "YES" : "NO") << endl;

    cout << "Loader type: " << loaderType << endl;
    cout << "INIT IRGs: " << initIRGs << endl;
    cout << "File name: " << filename << endl;
    cout << endl;
    cout << "Input file: " << inputFile << endl;
    cout << "Output file: " << outputFile << endl;
}

