#include "tapeimagewriter.h"
#include <cstdlib>
#include <cstring>

using namespace xex2cas;
using namespace std;

TapeImageWriter::TapeImageWriter() {

}

int TapeImageWriter::writeFile(
        ofstream &os,
        unsigned char* data,
        int dataLen,
        bool trailingEOF,
        bool first,
        bool fast,
        bool longerIRGs,
        bool shortLeader,
        vector<int> initRBAs,
        int initIRGDuration,
        string fujiChunkText
        ) {

    if (first == true) {

        /*FUJI chunk*/
        os.put('F');
        os.put('U');
        os.put('J');
        os.put('I');
        
        /*Ensure that text of the FUJI chunk is up to 255 characters long*/
        const string fujiChunkEffectiveText = (fujiChunkText.length()<256) ? fujiChunkText : fujiChunkText.substr(0,255);
        
        /*Length*/
        os.put(fujiChunkEffectiveText.length());
        os.put(0);

        /*Aux*/
        os.put(0x00);
        os.put(0x00);

        /*Text of the FUJI chunk*/
        for (int i = 0; i < fujiChunkEffectiveText.length(); i++) {
            os.put(fujiChunkEffectiveText[i]);
        }

        /*BAUD chunk*/
        unsigned int baudRate = (fast == true ? 720 : 600);
        os.put('b');
        os.put('a');
        os.put('u');
        os.put('d');

        os.put(0x00);
        os.put(0x00);

        os.put(baudRate % 256);
        os.put(baudRate / 256);
    }

    /* Generate the records*/
    int irg;
    int dataIndex = 0;

    unsigned char record[132];
    bool firstRecord = true;
    vector<int>::iterator initIterator = initRBAs.begin();

    while (true) {

        /*Determie what record to generate
         * 0xFC - Full
         * 0xFA - Partial
         * 0xFE - EOF
         */
        int recordType;

        int bytesToGo = dataLen - dataIndex;

        /*More than 128 bytes left - Always full Record*/
        if (bytesToGo > 128) {
            recordType = 0xFC;
        }
        
        /*Exactly 128 bytes left - Full or EOF*/
        else if (bytesToGo == 128) {
            if (trailingEOF == true) {
                recordType = 0xFE;
            } else {
                recordType = 0xFC;
            }
        }
        
        /*No data left - EOF*/
        else if (bytesToGo == 0) {
            recordType = 0xFE;
        }
        
        /*Less than 128 bytes left - Partial or EOF*/
        else if (bytesToGo < 128) {
            if (trailingEOF == true) {
                recordType = 0xFE;
            } else {
                recordType = 0xFA;
            }
        }

        /*Determine IRG duration for all records but the first one*/
        if (firstRecord == false) {
            
            /*Begin with base IRG duration*/
            irg = (longerIRGs == true ? 350 : 250);

            /* Elongate IRG when the record contains INIT segment(s)*/
            bool recordHasINITSegment=false;
            while (initIterator != initRBAs.end() && (*initIterator) < dataIndex) {
                irg += (1000 * initIRGDuration);
                initIterator++;
                recordHasINITSegment=true;
            }
            /* Ensure that even though there is no elongation specified, the IRG is at least 500 ms. 
               The data recorder may need some time to get its motor to full speed*/
            if (recordHasINITSegment==true && irg<500) {
                irg=500;
            }
            
        /*Generate leader for the first record*/    
        } else {
            irg = (shortLeader == true ? 12000 : 20000);
        }

        /*Generate fields of a data tape image chunk*/
        os.put('d');
        os.put('a');
        os.put('t');
        os.put('a');

        /*Length of the data chunk*/
        os.put(132 % 256);
        os.put(132 / 256);

        /*Inter-record-gap (IRG)*/
        os.put(irg % 256);
        os.put(irg / 256);

        /*Record prefix - synchronization bytes and type of the record*/
        record[0] = 0x55;
        record[1] = 0x55;
        record[2] = recordType;

        /*Record data - begin with all zeros*/
        memset(record + 3, 0x00, 128);

        /*Place valid bytes*/
        int numValidBytes = dataLen - dataIndex;
        if (numValidBytes > 128) numValidBytes = 128;
        for (int i = 0; i < numValidBytes; i++) {
            record[3 + i] = data[dataIndex];
            dataIndex++;
        }

        /*Place number of valid bytes to the block if needed*/
        if (numValidBytes < 128 && numValidBytes > 0) {
            record[3 + 127] = numValidBytes;
        }

        /*Calculate checksum*/
        unsigned int chsum = 0;
        for (int i = 0; i < 131; i++) {
            chsum += record[i];
            if (chsum > 255) {
                chsum &= 0xFF;
                chsum++;
            }
        }
        record[131] = chsum;

        /*Output record*/
        for (int i = 0; i < 132; i++) {
            os.put(record[i]);
        }

        firstRecord = false;

        /*Are we done?*/
        if (recordType == 0xFE) break;
    }
    
    return 0;

}
