/* XEX2CAS 2.0
 * 
 * CAS File generator by Michael Kalouš (BAKTRA Software)
 * Original version by Gienek plug@poczta.fm
 * 
 * This program converts Atari DOS 2 binary files to standard (FSK) tape records
 * 
 * The author has placed this work in the Public Domain, thereby relinquishing
 * all copyrights. Everyone is free to use, modify, republish, sell or give away
 * this work without prior consent from anybody. 
 * 
 * Change log:
 * 2014-12-21 Complete rewrite from scratch. Fixed problem with extra bytes in
 *            the last record. Added more options and loaders.
 * 
 * 2003-09-30 Gienek. Initial version
 */

#include <iostream> 
#include <fstream>  
#include <cstring>
#include <cstdlib>
#include <vector>
#include <sstream>
#include <iomanip>
#include <string>

#include "tapeimagewriter.h"
#include "dos2binary.h"
#include "segment.h"
#include "options.h"

using namespace std;
using namespace xex2cas;

string createNameForModernLoader(string fn);

/* Binary loader - Modern STDBLOAD 2 */
unsigned char stdbload_2[456] = {
0x00,0x03,0xe8,0x07,0x06,0x09,0xa9,0x3c,0x8d,0x02,0xd3,0xa2,0xff,0x9a,0xa2,0x80,0xbd,0xff,0x03,0x9d,
0x67,0x09,0xca,0xd0,0xf7,0x20,0x66,0x09,0xa9,0x01,0x8d,0x0b,0x09,0x8d,0x0c,0x09,0x20,0x16,0x09,0x20,
0x21,0x09,0xad,0x53,0x03,0x10,0x06,0x20,0x43,0x09,0x4c,0x04,0x08,0xa9,0x06,0x8d,0xe2,0x02,0xa9,0x09,
0x8d,0xe3,0x02,0xa9,0x07,0x8d,0x54,0x03,0xa9,0x09,0x8d,0x55,0x03,0xa9,0x02,0x8d,0x58,0x03,0xa9,0x00,
0x8d,0x59,0x03,0x20,0xde,0x08,0xa9,0xff,0xcd,0x07,0x09,0xd0,0x0d,0xcd,0x08,0x09,0xd0,0x08,0xa9,0x00,
0x8d,0x0c,0x09,0x4c,0x27,0x08,0xa9,0x09,0x8d,0x54,0x03,0xa9,0x09,0x8d,0x55,0x03,0xa9,0x02,0x8d,0x58,
0x03,0xa9,0x00,0x8d,0x59,0x03,0x20,0xde,0x08,0xad,0x0c,0x09,0xf0,0x06,0x20,0x4e,0x09,0x4c,0x04,0x08,
0xad,0x0b,0x09,0xf0,0x11,0xa9,0x00,0x8d,0x0b,0x09,0xad,0x07,0x09,0x8d,0xe0,0x02,0xad,0x08,0x09,0x8d,
0xe1,0x02,0x38,0xad,0x09,0x09,0xed,0x07,0x09,0x8d,0x58,0x03,0xb0,0x03,0xce,0x0a,0x09,0x38,0xad,0x0a,
0x09,0xed,0x08,0x09,0x8d,0x59,0x03,0x18,0xad,0x58,0x03,0x69,0x01,0x8d,0x58,0x03,0x90,0x03,0xee,0x59,
0x03,0xad,0x07,0x09,0x8d,0x54,0x03,0xad,0x08,0x09,0x8d,0x55,0x03,0x20,0xde,0x08,0xad,0xe2,0x02,0xc9,
0x06,0xd0,0x07,0xad,0xe3,0x02,0xc9,0x09,0xf0,0x0d,0xa9,0x3c,0x8d,0x02,0xd3,0x20,0x03,0x09,0xa9,0x34,
0x8d,0x02,0xd3,0x4c,0x1d,0x08,0xa2,0x10,0xa9,0x07,0x8d,0x52,0x03,0x20,0x56,0xe4,0xad,0x53,0x03,0x30,
0x01,0x60,0xc9,0x88,0xd0,0x09,0xa2,0xff,0x9a,0x20,0x16,0x09,0x6c,0xe0,0x02,0x20,0x43,0x09,0x68,0x68,
0x4c,0x04,0x08,0x6c,0xe2,0x02,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x43,0x3a,0x9b,0x7d,0x42,0x9b,0x7d,
0x45,0x9b,0xa2,0x10,0xa9,0x0c,0x8d,0x52,0x03,0x20,0x56,0xe4,0x60,0xa2,0x10,0xa9,0x03,0x8d,0xfc,0x02,
0x8d,0x52,0x03,0xa9,0x04,0x8d,0x5a,0x03,0xa9,0x80,0x8d,0x5b,0x03,0xa9,0x0d,0x8d,0x54,0x03,0xa9,0x09,
0x8d,0x55,0x03,0x20,0x56,0xe4,0x60,0xa2,0x13,0xa0,0x09,0x20,0x42,0xc6,0x20,0x59,0x09,0x60,0xa2,0x13,
0xa0,0x09,0x20,0x42,0xc6,0x20,0x59,0x09,0x60,0xa9,0xff,0x8d,0xfc,0x02,0xad,0xfc,0x02,0xc9,0xff,0xf0,
0xf9,0x60,0xa9,0x00,0x8d,0x44,0x02,0xa9,0x01,0x85,0x09,0x20,0x81,0x09,0xa9,0x81,0x85,0x0c,0xa9,0x09,
0x85,0x0d,0xa2,0x8a,0xa0,0x09,0x20,0x42,0xc6,0xa9,0x04,0x85,0x0a,0xa9,0x08,0x85,0x0b,0x60,0x7d,0x20,
0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,
0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0x20,0xa1,0x9b,
};


/* Binary loader - Original Exclamation mark (!) loader */
unsigned char exma_original[564] = {
0x20,0x05,0x00,0x07,0x20,0x07,0x20,0xf4,0x08,0x4c,0xed,0xf2,0x56,0xff,0x55,0x57,0x52,0x03,0x40,0x2f,
0xef,0x17,0x2f,0x07,0x37,0x2f,0x0a,0x56,0xfe,0x72,0x03,0x40,0x52,0x03,0x40,0x0f,0xf9,0xe7,0x97,0x97,
0xb3,0xaf,0xf8,0x56,0xfe,0x72,0x0f,0xfd,0x56,0xf8,0x72,0xad,0xff,0x56,0xf5,0x72,0xaa,0xff,0x72,0xab,
0xff,0x5d,0x01,0x5f,0xf7,0xdf,0x7a,0x0c,0xdf,0x1d,0x09,0x36,0xe4,0x2f,0x06,0x97,0x97,0xb3,0xa4,0xf8,
0x56,0xa4,0x72,0xf5,0xff,0x56,0xf8,0x72,0xf4,0xff,0x15,0x56,0xfd,0x72,0xad,0xff,0x56,0xff,0x72,0x0f,
0xfd,0x56,0x82,0xdf,0x5b,0x09,0xdf,0x1a,0xf7,0x5d,0xef,0x56,0xfc,0x62,0xbd,0xfc,0x56,0x04,0x62,0xbb,
0xfc,0x56,0xf7,0x62,0xba,0xfc,0x56,0xfb,0x62,0xb5,0xfc,0x56,0x7f,0x62,0xb4,0xfc,0x56,0xf3,0x72,0x03,
0xfd,0xdf,0xa9,0x1b,0x56,0xff,0x72,0xc9,0xf6,0x5d,0xef,0x56,0xcb,0x62,0xbb,0xfc,0x56,0xf6,0x62,0xba,
0xfc,0x56,0xfd,0x62,0xb7,0xfc,0x56,0xff,0x62,0xb6,0xfc,0x56,0xf8,0x62,0xbd,0xfc,0xdf,0xa9,0x1b,0xcf,
0xda,0x5d,0xef,0x56,0xcf,0x62,0xbb,0xfc,0x56,0xf6,0x62,0xba,0xfc,0x56,0xfb,0x62,0xb7,0xfc,0x56,0xff,
0x62,0xb6,0xfc,0xdf,0xa9,0x1b,0xef,0xe6,0x3f,0x77,0x2f,0xf9,0xdf,0x1a,0xf7,0xb3,0x82,0xf7,0x67,0xb7,
0xdf,0x1a,0xf7,0x97,0x57,0x0f,0xf9,0xdf,0x7e,0x0c,0xb3,0x82,0xf7,0x52,0xc9,0xf6,0x2f,0xf0,0x52,0xcf,
0xf6,0x72,0x1f,0xfd,0x52,0xce,0xf6,0x72,0x1e,0xfd,0x31,0xc9,0xf6,0x5d,0xef,0x52,0xcf,0xf6,0x62,0xbb,
0xfc,0xb7,0x52,0xce,0xf6,0x62,0xba,0xfc,0x57,0x97,0x37,0x2f,0xe0,0x57,0x37,0x2f,0xe4,0x52,0xcd,0xf6,
0x72,0xcf,0xf6,0x52,0xcc,0xf6,0x72,0xce,0xf6,0x56,0xcd,0x62,0xbb,0xfc,0x56,0xf6,0x62,0xba,0xfc,0x56,
0xfd,0xb3,0x3c,0xf8,0x52,0xcd,0xf6,0xc7,0x12,0xcf,0xf6,0x62,0xb7,0xfc,0x52,0xcc,0xf6,0x12,0xce,0xf6,
0x62,0xb6,0xfc,0x52,0xce,0xf6,0x01,0xb7,0xfc,0x2f,0xfc,0x01,0xb6,0xfc,0x5d,0xef,0x56,0x0c,0x72,0x1d,
0xfd,0x56,0xf7,0x72,0x1c,0xfd,0xdf,0xa9,0x1b,0xef,0xfc,0xb3,0x25,0xf8,0x52,0x1d,0xfd,0x72,0xf5,0xff,
0x52,0x1c,0xfd,0x72,0xf4,0xff,0xdf,0x6d,0xf7,0xdf,0x85,0xf7,0xdf,0x30,0xf7,0xb3,0x4a,0xf8,0x93,0x1d,
0xfd,0x52,0x1f,0xfd,0x72,0xf5,0xff,0x52,0x1e,0xfd,0x72,0xf4,0xff,0xdf,0x6d,0xf7,0xdf,0x30,0xf7,0x93,
0x1f,0xfd,0x56,0xef,0x72,0xfd,0xff,0x56,0x14,0x72,0xfc,0xff,0x60,0xa9,0x0c,0x8d,0xd0,0x00,0xa9,0x07,
0x8d,0xd1,0x00,0xa0,0x00,0xb1,0xd0,0x49,0xff,0x91,0xd0,0xee,0xd0,0x00,0xd0,0x03,0xee,0xd1,0x00,0xad,
0xd1,0x00,0xc9,0x08,0xd0,0xe9,0xad,0xd0,0x00,0xc9,0x9c,0xd0,0xe2,0xa9,0x00,0x8d,0xd0,0x00,0x8d,0xd1,
0x00,0xea,0x60,0xa9,0x3c,0x8d,0x02,0xd3,0x60,0xa9,0x20,0x8d,0x09,0x07,0xa9,0x9d,0x8d,0x0a,0x07,0xa9,
0x08,0x8d,0x0b,0x07,0x60,0xa2,0x10,0xa9,0x0c,0x20,0x56,0xe4,0xa2,0x20,0xa9,0x0c,0x20,0x56,0xe4,0x60,
0x20,0xcf,0x08,0x20,0xd5,0x08,0x60,0x43,0x3a,0x9b,0xfd,0xe3,0xe1,0xf2,0xf4,0xf2,0xe9,0xe4,0xe7,0xe5,
0xa0,0xe9,0xee,0xf3,0xf4,0xe1,0xec,0xec,0xe5,0xe4,0x1d,0x9c,0xf4,0xf9,0xf0,0xe5,0xa0,0xbc,0xc5,0xd3,
0xc3,0xbe,0xa0,0xf4,0xef,0xa0,0xe3,0xef,0xee,0xf4,0xe9,0xf5,0xe5,0xa0,0xe2,0xef,0xef,0xf4,0xfd,0x9b,
0x00,0x00,0x00,0x00,
};

/*Binary loader - Exclamation mark (!) loader updated for XL/XE computers*/
unsigned char exma_xlxe[564] = {
0x20,0x05,0x00,0x07,0x20,0x07,0x20,0xf4,0x08,0x4c,0xa7,0xc5,0x56,0xff,0x55,0x57,0x52,0x03,0x40,0x2f,
0xef,0x17,0x2f,0x07,0x37,0x2f,0x0a,0x56,0xfe,0x72,0x03,0x40,0x52,0x03,0x40,0x0f,0xf9,0xe7,0x97,0x97,
0xb3,0xaf,0xf8,0x56,0xfe,0x72,0x0f,0xfd,0x56,0xf8,0x72,0xad,0xff,0x56,0xf5,0x72,0xaa,0xff,0x72,0xab,
0xff,0x5d,0x01,0x5f,0xf7,0xdf,0x7a,0x0c,0xdf,0x1d,0x09,0x36,0xe4,0x2f,0x06,0x97,0x97,0xb3,0xa4,0xf8,
0x56,0xa4,0x72,0xf5,0xff,0x56,0xf8,0x72,0xf4,0xff,0x15,0x56,0xfd,0x72,0xad,0xff,0x56,0xff,0x72,0x0f,
0xfd,0x56,0x82,0xdf,0x5b,0x09,0xdf,0x1a,0xf7,0x5d,0xef,0x56,0xfc,0x62,0xbd,0xfc,0x56,0x04,0x62,0xbb,
0xfc,0x56,0xf7,0x62,0xba,0xfc,0x56,0xfb,0x62,0xb5,0xfc,0x56,0x7f,0x62,0xb4,0xfc,0x56,0xf3,0x72,0x03,
0xfd,0xdf,0xa9,0x1b,0x56,0xff,0x72,0xc9,0xf6,0x5d,0xef,0x56,0xcb,0x62,0xbb,0xfc,0x56,0xf6,0x62,0xba,
0xfc,0x56,0xfd,0x62,0xb7,0xfc,0x56,0xff,0x62,0xb6,0xfc,0x56,0xf8,0x62,0xbd,0xfc,0xdf,0xa9,0x1b,0xcf,
0xda,0x5d,0xef,0x56,0xcf,0x62,0xbb,0xfc,0x56,0xf6,0x62,0xba,0xfc,0x56,0xfb,0x62,0xb7,0xfc,0x56,0xff,
0x62,0xb6,0xfc,0xdf,0xa9,0x1b,0xef,0xe6,0x3f,0x77,0x2f,0xf9,0xdf,0x1a,0xf7,0xb3,0x82,0xf7,0x67,0xb7,
0xdf,0x1a,0xf7,0x97,0x57,0x0f,0xf9,0xdf,0x7e,0x0c,0xb3,0x82,0xf7,0x52,0xc9,0xf6,0x2f,0xf0,0x52,0xcf,
0xf6,0x72,0x1f,0xfd,0x52,0xce,0xf6,0x72,0x1e,0xfd,0x31,0xc9,0xf6,0x5d,0xef,0x52,0xcf,0xf6,0x62,0xbb,
0xfc,0xb7,0x52,0xce,0xf6,0x62,0xba,0xfc,0x57,0x97,0x37,0x2f,0xe0,0x57,0x37,0x2f,0xe4,0x52,0xcd,0xf6,
0x72,0xcf,0xf6,0x52,0xcc,0xf6,0x72,0xce,0xf6,0x56,0xcd,0x62,0xbb,0xfc,0x56,0xf6,0x62,0xba,0xfc,0x56,
0xfd,0xb3,0x3c,0xf8,0x52,0xcd,0xf6,0xc7,0x12,0xcf,0xf6,0x62,0xb7,0xfc,0x52,0xcc,0xf6,0x12,0xce,0xf6,
0x62,0xb6,0xfc,0x52,0xce,0xf6,0x01,0xb7,0xfc,0x2f,0xfc,0x01,0xb6,0xfc,0x5d,0xef,0x56,0x0c,0x72,0x1d,
0xfd,0x56,0xf7,0x72,0x1c,0xfd,0xdf,0xa9,0x1b,0xef,0xfc,0xb3,0x25,0xf8,0x52,0x1d,0xfd,0x72,0xf5,0xff,
0x52,0x1c,0xfd,0x72,0xf4,0xff,0xdf,0x6d,0xf7,0xdf,0x85,0xf7,0xdf,0x30,0xf7,0xb3,0x4a,0xf8,0x93,0x1d,
0xfd,0x52,0x1f,0xfd,0x72,0xf5,0xff,0x52,0x1e,0xfd,0x72,0xf4,0xff,0xdf,0x6d,0xf7,0xdf,0x30,0xf7,0x93,
0x1f,0xfd,0x56,0xef,0x72,0xfd,0xff,0x56,0x14,0x72,0xfc,0xff,0x60,0xa9,0x0c,0x8d,0xd0,0x00,0xa9,0x07,
0x8d,0xd1,0x00,0xa0,0x00,0xb1,0xd0,0x49,0xff,0x91,0xd0,0xee,0xd0,0x00,0xd0,0x03,0xee,0xd1,0x00,0xad,
0xd1,0x00,0xc9,0x08,0xd0,0xe9,0xad,0xd0,0x00,0xc9,0x9c,0xd0,0xe2,0xa9,0x00,0x8d,0xd0,0x00,0x8d,0xd1,
0x00,0xea,0x60,0xa9,0x3c,0x8d,0x02,0xd3,0x60,0xa9,0x20,0x8d,0x09,0x07,0xa9,0x9d,0x8d,0x0a,0x07,0xa9,
0x08,0x8d,0x0b,0x07,0x60,0xa2,0x10,0xa9,0x0c,0x20,0x56,0xe4,0xa2,0x20,0xa9,0x0c,0x20,0x56,0xe4,0x60,
0x20,0xcf,0x08,0x20,0xd5,0x08,0x60,0x43,0x3a,0x9b,0xfd,0xe3,0xe1,0xf2,0xf4,0xf2,0xe9,0xe4,0xe7,0xe5,
0xa0,0xe9,0xee,0xf3,0xf4,0xe1,0xec,0xec,0xe5,0xe4,0x1d,0x9c,0xf4,0xf9,0xf0,0xe5,0xa0,0xbc,0xc5,0xd3,
0xc3,0xbe,0xa0,0xf4,0xef,0xa0,0xe3,0xef,0xee,0xf4,0xe9,0xf5,0xe5,0xa0,0xe2,0xef,0xef,0xf4,0xfd,0x9b,
0x00,0x00,0x00,0x00,
};


string createNameForModernLoader(string fn) {

    /*Determine directory/folder separator*/    
#if defined(_WIN32) 
    const char separator ='\\';
#elif defined(_WIN64)    
    const char separator ='\\';
#else
    const char separator ='/';
#endif
    
    /*Check the position of the last separator in the file name*/
    int sepPos = fn.find_last_of(separator);
    
    /*If there is no separator, then use the first 34 characters*/
    if (sepPos == fn.npos) {
        return fn.length() <= 34 ? fn : fn.substr(0, 34);
    }
    
    /*Otherwise, use the first 34 characters of what's behind the separator,
     *if there is anything behind, of course
     */
    if (sepPos==fn.length()-1) return "";
    
    fn=fn.substr(sepPos+1);
    return fn.length() <= 34 ? fn : fn.substr(0, 34);
}

/* Mainline code*/
int _CRT_glob = 0;

int main(int argc, char *argv[]) {

    /*Display program title*/
    cout << "XEX2CAS 2.0 - Convert Atari DOS 2 Binary File to standard tape records" << endl;

    /*Parse options*/
    Options opts;
    int parseResult = opts.parse(argc, argv);

    /*In case of bad syntax or help requisition, terminate processing*/
    if (parseResult == opts.OPTIONS_ERROR) {
        return -1;
    }
    if (parseResult == opts.OPTIONS_HELP) {
        return 0;
    }
    
    /*Try to open the input file*/
    ifstream xexFile;
    xexFile.open(opts.inputFile.c_str(), ios_base::binary);
    if (xexFile.fail() == true) {
        cout << "Error: Unable to open the input file " << opts.inputFile << endl;
        return -1;
    }
    /*Determine size of the input file*/
    xexFile.seekg(0, ios_base::end);
    long xexFileLength = xexFile.tellg();

    /*If the file is bigger than 4 MB, do not process it*/
    if (xexFileLength > 4 * 1024 * 1024) {
        cout << "Error: Input file size exceeds 4 MB" << endl;
        return -1;
    }
    xexFile.seekg(0, ios_base::beg);

    /*Construct name of the output file if needed*/
    string outFileName;
    if (opts.outputFile == "") {
        /*Check for last dot*/
        int dotPos = opts.inputFile.find_last_of('.');

        /*If no dot, just append the .cas extension*/
        if (dotPos == opts.inputFile.npos) {
            outFileName = opts.inputFile + ".cas";
        } else {
            outFileName = (opts.inputFile.substr(0, dotPos)) + ".cas";
        }
    } else {
        outFileName = opts.outputFile;
    }

    /*Display input and output files*/
    cout << opts.inputFile << " >> ";
    cout << outFileName << endl;

    /*Check existence of the output file*/
    ifstream casFilePeek;
    casFilePeek.open(outFileName.c_str(), ios_base::binary);
    if (casFilePeek.fail() == false) {
        if (opts.overwriteOutputs == true) {
            cout << "Warning: Overwriting output file" << endl;
        } else {
            cout << "Error: Output file already exists" << endl;
            return -1;
        }
        casFilePeek.close();
    }

    /*Try to open the output file*/
    ofstream casFile;
    casFile.open(outFileName.c_str(), ios_base::binary);
    if (casFile.fail() == true) {
        cout << "Error: Unable to open output file " << outFileName << endl;
        return -1;
    }

    /*Read input file to memory*/
    unsigned char* xexBuffer = new unsigned char[xexFileLength];
    for (int i = 0; i < xexFileLength; i++) {
        xexBuffer[i] = xexFile.get();
    }
    
    /*Check if there were read errors*/
    if (xexFile.fail()) {
        cout << "Error: Failed to read from the input file" << endl;
        return -1;
    }
    
    /*If there were no errors, close the input file*/
    xexFile.close();

    /*Check if the input file is a DOS 2 binary file*/
    DOS2BinaryFile d2b;
    parseResult = d2b.parse(xexBuffer, xexFileLength);

    switch (parseResult) {
        case (DOS2BinaryFile::D2BF_ALIEN):
        {
            cout << "Warning: Input file is not a binary file" << endl;
            break;
        }
        case (DOS2BinaryFile::D2BF_CORRUPT):
        {
            cout << "Warning: Input binary file is corrupt" << endl;
            cout << d2b.getParseError() << endl;
            cout << "Listing valid segments:" << endl;
            d2b.dump();
            break;
        }
    }

    /*Write tape image*/
    TapeImageWriter writer;

    /*Write loader*/
    if (opts.doNotPrependLoader == false) {

        vector<int> emptyInitRBAs;
        
        unsigned char* loaderData;
        int loaderDataLen;
	bool trailingEOF=false;
        
	switch(opts.loaderType) {
	  case Options::LDR_ORIGINAL_EXMA: {
            loaderData = exma_original;
            loaderDataLen=564;
	    break;
	  }
	  case Options::LDR_XL_FIXED_EXMA: {
            loaderData = exma_xlxe;
            loaderDataLen=564;
	    break;
	  }
	  case Options::LDR_MODERN_EXMA: {
	    loaderData = stdbload_2;
	    loaderDataLen=456;
	    trailingEOF=true;
            /*Clear file name*/
	    memset(loaderData+(0x098B-0x07E8),' ',34);
            
            /*Determine file name if needed*/
            string fn = opts.filename;
            if (fn=="") fn=createNameForModernLoader(opts.inputFile);
            /*File name is guaranteed to be up to 34 characters long*/
            memcpy(loaderData+(0x098B-0x07E8),fn.c_str(),fn.length());
	    break;
	  }
	}

        /*Call tape image writer to output the selected loader*/
        writer.writeFile(
                casFile,
                loaderData,
                loaderDataLen,
                trailingEOF,
                true,
                opts.useFasterTransferSpeed,
                opts.useLongerIRGs,
                opts.useShorterLeader,
                emptyInitRBAs,
                0,
                (opts.useEmptyFUJIChunk==true)? string("") : string("File generated by XEX2CAS 2.0")
                );
    }

    /*Write the binary file*/
    bool isBinaryFileFirst = opts.doNotPrependLoader;

    writer.writeFile(
            casFile,
            xexBuffer,
            xexFileLength,
            false,
            isBinaryFileFirst,
            opts.useFasterTransferSpeed,
            opts.useLongerIRGs,
            opts.useShorterLeader,
            d2b.getInitRBAs(),
            opts.initIRGs,
            ""
            );
    
    /*Check status of the output file*/
    if (casFile.bad()==true || casFile.fail()==true) {
        cout << "Error: Failed to write to the output file" << endl;
        return -1;
    }
    
    casFile.close();
    cout << "Done" << endl;

    return 0;
}

