// a8rawconv - A8 raw disk conversion utility
// Copyright (C) 2014 Avery Lee
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

#include "stdafx.h"

template<class T>
class for_each_iterator {
public:
	for_each_iterator(T& c) : mContainer(c) {}

	template<class Fn>
	void operator>>=(const Fn& fn) {
		std::for_each(mContainer.begin(), mContainer.end(), fn);
	}

private:
	T& mContainer;
};

template<class T>
for_each_iterator<T> get_for_each(T& container) {
	return for_each_iterator<T>(container);
}

#define FOR_EACH(var, container) get_for_each((container)) >>= [&](decltype(*container.begin())& var)

uint32_t read_u32(const uint8_t *p) {
	return ((uint32_t)p[0])
		+ ((uint32_t)p[1] << 8)
		+ ((uint32_t)p[2] << 16)
		+ ((uint32_t)p[3] << 24);
}

void write_u8(uint8_t v, FILE *f) {
	putc(v, f);
}

void write_u16(uint16_t v, FILE *f) {
	putc(v & 0xff, f);
	putc(v >> 8, f);
}

void write_u32(uint32_t v, FILE *f) {
	putc(v & 0xff, f);
	putc((v >> 8) & 0xff, f);
	putc((v >> 16) & 0xff, f);
	putc((v >> 24) & 0xff, f);
}

void write_pad(int n, FILE *f) {
	while(n--)
		putc(0, f);
}

void fatal(const char *msg) {
	puts(msg);
	exit(10);
}

void fatalf(const char *msg, ...) {
	va_list val;
	va_start(val, msg);
	vprintf(msg, val);
	va_end(val);
	exit(10);
}

struct SectorInfo {
	float mPosition;
	int mIndex;
	int mWeakOffset;
	uint32_t mSectorSize;
	bool mbMFM;
	uint8_t mAddressMark;
	uint16_t mRecordedCRC;
	uint16_t mComputedCRC;
	uint8_t mData[1024];

	uint32_t ComputeContentHash() const;
	bool HasSameContents(const SectorInfo& other) const;
};

uint32_t SectorInfo::ComputeContentHash() const {
	uint32_t hash = mbMFM;

	hash += mAddressMark;
	hash += mSectorSize;
	hash += mComputedCRC;
	hash += (uint32_t)mRecordedCRC << 16;
	hash += mSectorSize;

	for(uint32_t i=0; i<mSectorSize; i+=4) {
		hash += *(const uint32_t *)&mData[i];
		hash = (hash >> 1) + (hash << 31);
	}

	return hash;
}

bool SectorInfo::HasSameContents(const SectorInfo& other) const {
	if (mbMFM != other.mbMFM)
		return false;

	if (mAddressMark != other.mAddressMark)
		return false;

	if (mSectorSize != other.mSectorSize)
		return false;

	if (mComputedCRC != other.mComputedCRC)
		return false;

	if (mRecordedCRC != other.mRecordedCRC)
		return false;

	if (memcmp(mData, other.mData, mSectorSize))
		return false;

	return true;
}

struct TrackInfo {
	std::vector<SectorInfo> mSectors;
};

TrackInfo g_tracks[40];

std::string g_inputPath;
int g_inputPathCountPos;
int g_inputPathCountWidth;

std::string g_outputPath;
bool g_showLayout;
bool g_dumpBadSectors;
int g_verbosity;
float g_clockPeriodAdjust = 1.0f;

enum InputFormat {
	kInputFormat_Auto,
	kInputFormat_KryoFluxStream,
	kInputFormat_SuperCardProStream
} g_inputFormat = kInputFormat_Auto;

enum OutputFormat {
	kOutputFormat_Auto,
	kOutputFormat_ATX,
	kOutputFormat_ATR
} g_outputFormat = kOutputFormat_Auto;

///////////////////////////////////////////////////////////////////////////

uint16_t ComputeCRC(const uint8_t *buf, uint32_t len) {
	// x^16 + x^12 + x^5 + 1
	uint16_t crc = 0xFFFF;

	for(uint32_t i=0; i<len; ++i) {
		uint8_t c = buf[i];

		crc ^= (uint16_t)c << 8;

		for(int j=0; j<8; ++j) {
			uint16_t xorval = (crc & 0x8000) ? 0x1021 : 0;

			crc += crc;

			crc ^= xorval;
		}
	}

	return crc;
}

void fatal_read() {
	fatalf("Unable to read from input file: %s.\n", g_inputPath.c_str());
}

///////////////////////////////////////////////////////////////////////////

class SectorParser {
public:
	SectorParser();

	void Init(int track, const std::vector<uint32_t> *indexTimes, float samplesPerCell);

	bool Parse(uint32_t stream_time, uint8_t clock_bits, uint8_t data_bits);

protected:
	int mTrack;
	int mSector;
	int mSectorSize;
	int mReadPhase;
	int mBitPhase;
	float mRotPos;
	float mSamplesPerCell;

	uint8_t mBuf[1024 + 4];
	uint8_t mClockBuf[1024 + 4];
	uint32_t mStreamTimes[1024 + 4];

	const std::vector<uint32_t> *mpIndexTimes;
};

SectorParser::SectorParser()
	: mReadPhase(0)
	, mBitPhase(0)
{
}

void SectorParser::Init(int track, const std::vector<uint32_t> *indexTimes, float samplesPerCell) {
	mTrack = track;
	mpIndexTimes = indexTimes;
	mSamplesPerCell = samplesPerCell;
}

bool SectorParser::Parse(uint32_t stream_time, uint8_t clock_bits, uint8_t data_bits) {
	if (mReadPhase < 6) {
		if (++mBitPhase == 16) {
			mBitPhase = 0;

			if (clock_bits != 0xFF)
				return false;

			mBuf[++mReadPhase] = data_bits;

			if (mReadPhase == 6) {
				if (mBuf[1] != mTrack) {
					//printf("Track number mismatch!\n");
					return false;
				}

				// Byte 3 of the address mark is the side indicator, which is supposed to be
				// zero for the 1771. It appears not to validate it, because Rescue on Fractalus
				// has garbage in that field.
#if 0
				if (mBuf[2] != 0) {
					printf("Zero #1 failed! (got %02X)\n", mBuf[2]);
					return false;
				}
#endif

				if (mBuf[3] < 1 || mBuf[3] > 18) {
					printf("Invalid sector number\n");
					return false;
				}

				if (mBuf[4] > 3) {
					printf("Bad sector size!\n");
					return false;
				}

				mBuf[0] = 0xFE;
				const uint16_t computedCRC = ComputeCRC(mBuf, 5);
				const uint16_t recordedCRC = ((uint16_t)mBuf[5] << 8) + mBuf[6];

				if (computedCRC != recordedCRC) {
					printf("CRC failure on sector header: %04X != %04X\n", computedCRC, recordedCRC);
					return false;
				}

				mSector = mBuf[3];
				mSectorSize = 128 << mBuf[4];

				int vsn_time = stream_time;

				// find the nearest index mark
				auto it_index = std::upper_bound(mpIndexTimes->begin(), mpIndexTimes->end(), (uint32_t)vsn_time + 1);

				if (it_index == mpIndexTimes->begin()) {
					if (g_verbosity >= 2)
						printf("Skipping track %d, sector %d before first index mark\n", mTrack, mSector);
					return false;
				}

				if (it_index == mpIndexTimes->end()) {
					if (g_verbosity >= 2)
						printf("Skipping track %d, sector %d after last index mark\n", mTrack, mSector);
					return false;
				}

				int vsn_offset = vsn_time - *--it_index;

				mRotPos = (float)vsn_offset / (float)(it_index[1] - it_index[0]);

				if (mRotPos >= 1.0f)
					mRotPos -= 1.0f;

				if (g_verbosity >= 2)
					printf("Found track %d, sector %d at position %4.2f\n", mTrack, mSector, mRotPos);
			}
		}
	} else if (mReadPhase == 6) {
		// There must be at least a full zero byte before the DAM is recognized.
		if (clock_bits == 0xFF && data_bits == 0x00) {
			mBitPhase = 0;
		} else {
			++mBitPhase;
			
			if (clock_bits == 0xC7 && mBitPhase == 16) {
				// another IDAM detected before DAM -- terminate
				if (data_bits == 0xFE)
					return false;

				if (data_bits == 0xF8 || data_bits == 0xF9 || data_bits == 0xFA || data_bits == 0xFB) {
					//printf("DAM detected (%02X)\n", data_bits);

					mReadPhase = 7;
					mBitPhase = 0;
					mBuf[0] = data_bits;
					mClockBuf[0] = clock_bits;
					mStreamTimes[0] = stream_time;
				}
			}
		}
	} else {
		if (++mBitPhase == 16) {
			if (clock_bits != 0xFF) {
				if (g_verbosity > 1)
					printf("Bad data clock: %02X\n", clock_bits);
			}

//			printf("Data; %02X\n", ~data_bits);
			mBuf[mReadPhase - 6] = data_bits;
			mClockBuf[mReadPhase - 6] = clock_bits;
			mStreamTimes[mReadPhase - 6] = stream_time;

			mBitPhase = 0;
			if (++mReadPhase >= mSectorSize + 3 + 6) {
				// check if the CRC is correct
				// x^16 + x^12 + x^5 + 1
				uint16_t crc = ComputeCRC(mBuf, mSectorSize + 1);

				const uint16_t recordedCRC = ((uint16_t)mBuf[mSectorSize + 1] << 8) + (uint8_t)mBuf[mSectorSize + 2];

				//printf("Read sector %d: CRCs %04X, %04X\n", mSector, crc, recordedCRC);

				// add new sector entry
				auto& tracksecs = g_tracks[mTrack].mSectors;
				tracksecs.push_back(SectorInfo());
				SectorInfo& newsec = tracksecs.back();

				for(int i=0; i<mSectorSize; ++i)
					newsec.mData[i] = ~mBuf[i+1];

				newsec.mIndex = mSector;
				newsec.mPosition = mRotPos;
				newsec.mAddressMark = mBuf[0];
				newsec.mRecordedCRC = recordedCRC;
				newsec.mComputedCRC = crc;
				newsec.mSectorSize = mSectorSize;
				newsec.mWeakOffset = -1;
				newsec.mbMFM = false;

				if (g_verbosity >= 1 || (crc != recordedCRC && g_dumpBadSectors))
					printf("Decoded FM track %d, sector %d with %u bytes, DAM %02X, recorded CRC %04X (computed %04X)\n",
						mTrack,
						mSector,
						mSectorSize,
						mBuf[0],
						recordedCRC,
						crc);
			
				if (g_dumpBadSectors && crc != recordedCRC) {
					printf("  Index Clk Data Cells\n");
					for(int i=0; i<mSectorSize + 1; ++i) {
						printf("  %4d  %02X | %02X (%02X,%02X %02X %02X %02X %02X %02X %02X) | %+6.1f%s\n"
							, i - 1
							, mClockBuf[i]
							, mBuf[i]
							, 0xFF ^ mBuf[i]
							, 0xFF ^ (((mBuf[i] << 1) + (mBuf[i+1]>>7)) & 0xFF)
							, 0xFF ^ (((mBuf[i] << 2) + (mBuf[i+1]>>6)) & 0xFF)
							, 0xFF ^ (((mBuf[i] << 3) + (mBuf[i+1]>>5)) & 0xFF)
							, 0xFF ^ (((mBuf[i] << 4) + (mBuf[i+1]>>4)) & 0xFF)
							, 0xFF ^ (((mBuf[i] << 5) + (mBuf[i+1]>>3)) & 0xFF)
							, 0xFF ^ (((mBuf[i] << 6) + (mBuf[i+1]>>2)) & 0xFF)
							, 0xFF ^ (((mBuf[i] << 7) + (mBuf[i+1]>>1)) & 0xFF)
							, i ? (float)(mStreamTimes[i] - mStreamTimes[i-1]) / mSamplesPerCell : 0
							, i && mClockBuf[i] != 0xFF ? " <!>" : ""
							);
					}
				}

				return false;
			}
		}
	}

	return true;
}

///////////////////////////////////////////////////////////////////////////

class SectorParserMFM {
public:
	SectorParserMFM();

	void Init(int track, const std::vector<uint32_t> *indexTimes, float samplesPerCell);

	bool Parse(uint32_t stream_time, uint8_t clock_bits, uint8_t data_bits);

protected:
	int mTrack;
	int mSector;
	int mSectorSize;
	int mReadPhase;
	int mBitPhase;
	float mRotPos;
	float mSamplesPerCell;

	uint8_t mBuf[1024 + 3];

	const std::vector<uint32_t> *mpIndexTimes;
};

SectorParserMFM::SectorParserMFM()
	: mReadPhase(0)
	, mBitPhase(0)
{
}

void SectorParserMFM::Init(int track, const std::vector<uint32_t> *indexTimes, float samplesPerCell) {
	mTrack = track;
	mpIndexTimes = indexTimes;
	mSamplesPerCell = samplesPerCell;
}

bool SectorParserMFM::Parse(uint32_t stream_time, uint8_t clock_bits, uint8_t data_bits) {
	if (mReadPhase < 7) {
		if (++mBitPhase == 16) {
			mBitPhase = 0;

//			if (clock_bits != 0xFF)
//				return false;

			mBuf[mReadPhase+3] = data_bits;
			++mReadPhase;

			if (mReadPhase == 7) {
				if (mBuf[3] != 0xFE)
					return false;

				if (mBuf[4] != mTrack) {
					printf("Track number mismatch! %02X != %02X\n", mBuf[4], mTrack);
					return false;
				}

				if (mBuf[5] != 0) {
					printf("Zero #1 failed!\n");
					return false;
				}

				if (mBuf[6] < 1 || mBuf[6] > 26) {
					printf("Invalid sector number %d\n", mBuf[6]);
					return false;
				}

				if (mBuf[7] >= 4) {
					printf("Sector size failed! %02X\n", mBuf[7]);
					return false;
				}

				mSectorSize = 128 << mBuf[7];

				mBuf[0] = 0xA1;
				mBuf[1] = 0xA1;
				mBuf[2] = 0xA1;
				const uint16_t computedCRC = ComputeCRC(mBuf, 8);
				const uint16_t recordedCRC = ((uint16_t)mBuf[8] << 8) + mBuf[9];

				if (computedCRC != recordedCRC) {
					printf("CRC failure on sector header: %04X != %04X\n", computedCRC, recordedCRC);
					return false;
				}

				mSector = mBuf[6];

				int vsn_time = stream_time;

				// find the nearest index mark
				auto it_index = std::upper_bound(mpIndexTimes->begin(), mpIndexTimes->end(), (uint32_t)vsn_time + 1);

				if (it_index == mpIndexTimes->begin()) {
					if (g_verbosity >= 2)
						printf("Skipping track %d, sector %d before first index mark\n", mTrack, mSector);
					return false;
				}

				if (it_index == mpIndexTimes->end()) {
					if (g_verbosity >= 2)
						printf("Skipping track %d, sector %d after last index mark\n", mTrack, mSector);
					return false;
				}

				int vsn_offset = vsn_time - *--it_index;

				mRotPos = (float)vsn_offset / (float)(it_index[1] - it_index[0]);

				if (mRotPos >= 1.0f)
					mRotPos -= 1.0f;

				if (g_verbosity >= 2)
					printf("Found track %d, sector %d at position %4.2f\n", mTrack, mSector, mRotPos);
			}
		}
	} else if (mReadPhase == 7) {
//		if (clock_bits == 0x0A && data_bits != 0xA1)
		if ((clock_bits & 0x7F) == 0x0A && data_bits == 0xA1) {
			++mReadPhase;
		}
	} else if (mReadPhase == 8 || mReadPhase == 9) {
		if ((clock_bits & 0x7F) == 0x0A) {
			if (data_bits != 0xA1) {
				mReadPhase = 7;
				return true;
			}

			++mReadPhase;
			mBitPhase = 0;
			mBuf[0] = data_bits;
			mBuf[1] = data_bits;
			mBuf[2] = data_bits;
		}
	} else if (mReadPhase == 10) {
		if (++mBitPhase == 16) {
			if (clock_bits == 0x0A && data_bits == 0xA1) {
				mBitPhase = 0;
			} else {
				if (data_bits == 0xF8 || data_bits == 0xF9 || data_bits == 0xFA || data_bits == 0xFB) {
					//printf("DAM detected (%02X)\n", data_bits);
					++mReadPhase;
					mBitPhase = 0;
					mBuf[3] = data_bits;
				} else
					return false;
			}
		}
	} else {
		if (++mBitPhase == 16) {
//			printf("Data; %02X\n", ~data_bits);
			mBuf[mReadPhase - 7] = data_bits;

			mBitPhase = 0;
			if (++mReadPhase >= 7 + mSectorSize + 6) {
				// check if the CRC is correct
				// x^16 + x^12 + x^5 + 1
				uint16_t crc = ComputeCRC(mBuf, mSectorSize + 4);

				const uint16_t recordedCRC = ((uint16_t)mBuf[mSectorSize + 4] << 8) + (uint8_t)mBuf[mSectorSize + 5];

				//printf("Read sector %d: CRCs %04X, %04X\n", mSector, crc, recordedCRC);

				// add new sector entry
				auto& tracksecs = g_tracks[mTrack].mSectors;
				tracksecs.push_back(SectorInfo());
				SectorInfo& newsec = tracksecs.back();

				for(int i=0; i<mSectorSize; ++i)
					newsec.mData[i] = ~mBuf[i+4];

				newsec.mIndex = mSector;
				newsec.mPosition = mRotPos;
				newsec.mAddressMark = mBuf[3];
				newsec.mRecordedCRC = recordedCRC;
				newsec.mComputedCRC = crc;
				newsec.mSectorSize = mSectorSize;
				newsec.mbMFM = true;
				newsec.mWeakOffset = -1;

				if (g_verbosity >= 1)
					printf("Decoded MFM track %d, sector %d with %u bytes, DAM %02X, recorded CRC %04X (computed %04X)\n",
						mTrack,
						mSector,
						mSectorSize,
						mBuf[3],
						recordedCRC,
						crc);

				return false;
			}
		}
	}

	return true;
}

///////////////////////////////////////////////////////////////////////////

struct RawTrack {
	int mTrack;
	float mSamplesPerRev;

	std::vector<uint32_t> mTransitions;
	std::vector<uint32_t> mIndexTimes;
};

void process_track_fm(const RawTrack& rawTrack);
void process_track_mfm(const RawTrack& rawTrack);

class MemStream {
public:
	MemStream(const uint8_t *s, const uint8_t *t) : mpSrc(s), mpSrcEnd(t), mpSrcBegin(s) {}

	int get() {
		return mpSrc != mpSrcEnd ? *mpSrc++ : -1;
	}

	int pos() const { return mpSrc - mpSrcBegin; }

private:
	const uint8_t *mpSrc;
	const uint8_t *const mpSrcEnd;
	const uint8_t *const mpSrcBegin;
};

///////////////////////////////////////////////////////////////////////////

// KryoFlux system constants
const double mck = 18432000.0 * 73.0 / 14.0 / 2.0;
const double sck = mck / 2.0;
const double ick = mck / 16.0;

void read_kryoflux_track(RawTrack& rawTrack, int track) {
	std::vector<uint32_t> indexmarks;
	std::vector<uint32_t> indexstreamposes;

	typedef std::pair<uint32_t, uint32_t> streamtime_t;
	std::vector<streamtime_t> streamtimes;

	char buf[64];
	sprintf(buf, "%0*u", g_inputPathCountWidth, track * 2);

	std::string track_filename(g_inputPath);
	track_filename.replace(g_inputPathCountPos, g_inputPathCountWidth, buf);
	
	if (g_verbosity >= 1)
		printf("Reading KryoFlux track stream: %s\n", track_filename.c_str());

	FILE *f = fopen(track_filename.c_str(), "rb");
	if (!f)
		fatalf("Unable to open input track stream: %s.\n", track_filename.c_str());

	fseek(f, 0, SEEK_END);
	long len = ftell(f);
	if (len > 500*1024*1024 || len != size_t(len))
		fatalf("Stream too long: %ld bytes\n", len);

	std::vector<uint8_t> rawstream((size_t)len);
	if (fseek(f, 0, SEEK_SET) || 1 != fread(rawstream.data(), (size_t)len, 1, f))
		fatalf("Error reading from stream: %s\n", track_filename.c_str());
	fclose(f);

	// getc() is painfully slow in VS2010
	MemStream ms(rawstream.data(), rawstream.data() + rawstream.size());

	int t = 0;
	for(;;) {
		streamtimes.push_back(streamtime_t(ms.pos(), t));

		int c = ms.get();

		if (c < 0)
			break;

//		printf("%d | %d %d\n", t, ftell(f)-1, c);

		if (c < 8) {
			int delay = c << 8;

			c = ms.get();
			if (c < 0)
				fatal("Incomplete cell");

			delay += c;

			t += delay;
			rawTrack.mTransitions.push_back(t);
		} else if (c == 8) {
		} else if (c == 9) {
			c = ms.get();
			if (c < 0)
				fatal("Incomplete Nop2");
		} else if (c == 10) {
			for(int i=0; i<2; ++i) {
				c = ms.get();
				if (c < 0)
					fatal("Incomplete Nop3");
			}
		} else if (c == 11) {
			t += 0x10000;
		} else if (c == 12) {
			c = ms.get();
			if (c < 0)
				fatal("Incomplete Value16");
			t += c << 8;

			c = ms.get();
			if (c < 0)
				fatal("Incomplete Value16");
			t += c;
			rawTrack.mTransitions.push_back(t);
		} else if (c == 13) {
			int oobPos = ms.pos();
			int oobType = ms.get();
			if (oobType < 0)
				fatal("Incomplete OOB block");

			// end is a special block with no length
			if (oobType == 13)
				break;

			c = ms.get();
			if (c < 0)
				fatal("Incomplete OOB block");

			int oobLen = c;

			c = ms.get();
			if (c < 0)
				fatal("Incomplete OOB block");

			oobLen += c << 8;

			//printf("Received %u bytes of OOB data (type %u) (offset %u)\n", oobLen, oobType, ftell(f));

			std::vector<uint8_t> oobData(oobLen);

			for(int i=0; i<oobLen; ++i) {
				c = ms.get();
				if (c < 0)
					fatal("Incomplete OOB block");

				oobData[i] = (uint8_t)c;
			}

			if (oobType == 2) {
				if (oobLen != 12)
					fatal("Invalid index mark OOB block");

				const uint32_t raw_index_time = read_u32(&oobData[8]);
				const uint32_t raw_streampos = read_u32(&oobData[0]);
				const uint32_t raw_timer = read_u32(&oobData[4]);

				indexmarks.push_back(raw_index_time);
				rawTrack.mIndexTimes.push_back(raw_timer);
				indexstreamposes.push_back(raw_streampos);

				if (g_verbosity >= 2)
					printf("Index mark: oobpos=%u, streampos=%u, timer=%u, systime=%u\n", oobPos, read_u32(&oobData[0]), raw_timer, raw_index_time);
			}
		} else {
			t += c;
			rawTrack.mTransitions.push_back(t);
		}
	}

	// go back and just all of the index mark times
	for(size_t i=0, n=rawTrack.mIndexTimes.size(); i<n; ++i) {
		auto it = std::lower_bound(streamtimes.begin(), streamtimes.end(), streamtime_t(indexstreamposes[i], 0),
			[](const streamtime_t& x, const streamtime_t& y) { return x.first < y.first; });

		if (it == streamtimes.end())
			fatal("Unable to match index mark against stream position.");

		rawTrack.mIndexTimes[i] += it->second;
	}

	//printf("%u transitions read\n", (unsigned)transitions.size());

	if (indexmarks.size() < 2) {
		fatal("Less than two index marks read -- need at least one full disk revolution.");
	}

	// compute disk RPM
	double icks_per_rev = (double)(int32_t)(indexmarks.back() - indexmarks.front()) / (double)(indexmarks.size() - 1);
	double rpm = ick / icks_per_rev * 60.0;
	double samples_per_rev = icks_per_rev * sck / ick;

	if (g_verbosity >= 1) {
		printf("  %u index marks found\n", (unsigned)indexmarks.size());
		printf("  Rotational rate: %.2f RPM (%.0f samples per revolution)\n", rpm, samples_per_rev);
	}

	rawTrack.mTrack = track;
	rawTrack.mSamplesPerRev = (float)samples_per_rev;
}

void read_kryoflux() {
	for(int i=0; i<40; ++i) {
		RawTrack rawTrack;
		read_kryoflux_track(rawTrack, i);
		process_track_fm(rawTrack);
		process_track_mfm(rawTrack);
	}
}

//////////////////////////////////////////////////////////////////////////

struct SCPFileHeader {
	uint8_t		mSignature[3];
	uint8_t		mVersion;
	uint8_t		mDiskType;
	uint8_t		mNumRevs;
	uint8_t		mStartTrack;
	uint8_t		mEndTrack;
	uint8_t		mFlags;
	uint8_t		mBitCellEncoding;
	uint8_t		mSides;
	uint8_t		mReserved;
	uint32_t	mChecksum;
	uint32_t	mTrackOffsets[166];
};

static_assert(sizeof(SCPFileHeader) == 0x2A8, "");

struct SCPTrackRevolution {
	uint32_t	mTimeDuration;
	uint32_t	mDataLength;
	uint32_t	mDataOffset;
};

struct SCPTrackHeader {
	uint8_t		mSignature[3];
	uint8_t		mTrackNumber;
};

void read_supercardpro() {
	printf("Reading SuperCard Pro image: %s\n", g_inputPath.c_str());

	FILE *fi = fopen(g_inputPath.c_str(), "rb");
	if (!fi)
		fatalf("Unable to open input file: %s.\n", g_inputPath.c_str());

	SCPFileHeader fileHeader = {0};
	if (1 != fread(&fileHeader, sizeof(fileHeader), 1, fi))
		fatal_read();

	if (memcmp(fileHeader.mSignature, "SCP", 3))
		fatalf("Input file does not start with correct SCP image signature.\n");

	const int track_step = ((fileHeader.mFlags & 0x02) ? 2 : 1) * (fileHeader.mSides > 1 ? 2 : 1);

	std::vector<uint16_t> data_buf;
	for(int i=0; i<40; ++i) {
		const int track = i * track_step * 2;

		const uint32_t track_offset = fileHeader.mTrackOffsets[track];
		if (!track_offset)
			continue;

		if (fseek(fi, track_offset, SEEK_SET))
			fatal_read();

		SCPTrackHeader track_hdr = {};
		if (1 != fread(&track_hdr, sizeof(track_hdr), 1, fi))
			fatal_read();

		if (memcmp(track_hdr.mSignature, "TRK", 3))
			fatalf("SCP raw track %d has broken header at %08x with incorrect signature.", track, track_offset);

		std::vector<SCPTrackRevolution> revs(fileHeader.mNumRevs);
		if (1 != fread(revs.data(), sizeof(SCPTrackRevolution)*fileHeader.mNumRevs, 1, fi))
			fatal_read();

		// initialize raw track parameters
		RawTrack raw_track;
		raw_track.mIndexTimes.push_back(0);

		// compute average revolution time
		uint32_t total_rev_time = 0;
		FOR_EACH(rev, revs) {
			total_rev_time += rev.mTimeDuration;
			raw_track.mIndexTimes.push_back(total_rev_time);
		};

		raw_track.mSamplesPerRev = (float)total_rev_time / (float)fileHeader.mNumRevs;
		raw_track.mTrack = i;

		if (g_verbosity >= 1)
			printf("Track %d: %.2f RPM\n", i, 60.0 / (raw_track.mSamplesPerRev * 0.000000025));

		// parse out flux transitions for each rev
		uint32_t time = 0;

		FOR_EACH(rev, revs) {
			data_buf.resize(rev.mDataLength);

			if (fseek(fi, track_offset + rev.mDataOffset, SEEK_SET) < 0
				|| 1 != fread(data_buf.data(), rev.mDataLength * 2, 1, fi))
				fatal_read();

			FOR_EACH(offset_be, data_buf) {
				// need to byte-swap
				uint32_t offset = ((offset_be << 8) & 0xff00) + ((offset_be >> 8) & 0xff);

				if (offset) {
					time += offset;
					raw_track.mTransitions.push_back(time);
				} else
					time += 0x10000;
			};
		};

		// process the track
		process_track_fm(raw_track);
		process_track_mfm(raw_track);
	}

	fclose(fi);
};

//////////////////////////////////////////////////////////////////////////

void process_track_fm(const RawTrack& rawTrack) {
	// Atari disk timing produces 250,000 clocks per second at 288 RPM. We must compute the
	// effective sample rate given the actual disk rate.
	const double cells_per_rev = 250000.0 / (288.0 / 60.0);
	double scks_per_cell = rawTrack.mSamplesPerRev / cells_per_rev * g_clockPeriodAdjust;

	//printf("%.2f samples per cell\n", scks_per_cell);

	const uint32_t *samp = rawTrack.mTransitions.data();
	size_t samps_left = rawTrack.mTransitions.size() - 1;
	int time_basis = 0;
	int time_left = 0;

	int cell_len = (int)(scks_per_cell + 0.5);
	int cell_range = cell_len / 3;
	int cell_timer = 0;

	uint8_t last_track = 0;
	uint8_t last_sector = 0;
	int write_vsn = -1;
	float write_vsn_pos = 0;

	uint8_t shift_even = 0;
	uint8_t shift_odd = 0;

	std::vector<SectorParser> sectorParsers;
	uint8_t spew_data[16];
	int spew_index = 0;
	uint32_t spew_last_time = samp[0];

	for(;;) {
		while (time_left <= 0) {
			if (!samps_left)
				goto done;

			time_left += samp[1] - samp[0];
			time_basis = samp[1];
			++samp;
			--samps_left;
		}

		//printf("next_trans = %d, cell_timer = %d, %d transitions left\n", time_left, cell_timer, samps_left);

		// if the shift register is empty, restart shift timing at next transition
		if (!(shift_even | shift_odd)) {
			time_left = 0;
			cell_timer = cell_len;
			shift_even = 0;
			shift_odd = 1;
		} else {
			// compare time to next transition against cell length
			int trans_delta = time_left - cell_timer;

			if (trans_delta < -cell_range) {
				// ignore the transition
				cell_timer -= time_left;
				continue;
			}

			std::swap(shift_even, shift_odd);
			shift_odd += shift_odd;
			
			if (trans_delta <= cell_range) {
				cell_timer = cell_len;

				// we have a transition in range -- clock in a 1 bit
				if (trans_delta < -5)
					cell_timer -= 3;
				else if (trans_delta < -3)
					cell_timer -= 2;
				else if (trans_delta < 1)
					--cell_timer;
				else if (trans_delta > 1)
					++cell_timer;
				else if (trans_delta > 3)
					cell_timer += 2;
				else if (trans_delta > 5)
					cell_timer += 3;

				shift_odd++;
				time_left = 0;
			} else {
				// we don't have a transition in range -- clock in a 0 bit
				time_left -= cell_timer;
				cell_timer = cell_len;
			}

			if (g_verbosity >= 3) {
				spew_data[spew_index] = shift_odd;
				if (++spew_index == 16) {
					spew_index = 0;
					printf("%02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X | %.2f\n"
						, spew_data[0]
						, spew_data[1]
						, spew_data[2]
						, spew_data[3]
						, spew_data[4]
						, spew_data[5]
						, spew_data[6]
						, spew_data[7]
						, spew_data[8]
						, spew_data[9]
						, spew_data[10]
						, spew_data[11]
						, spew_data[12]
						, spew_data[13]
						, spew_data[14]
						, spew_data[15]
						, (double)(time_basis - time_left - spew_last_time) / scks_per_cell
						);

					spew_last_time = time_basis - time_left;
				}
			}
			
			const uint32_t vsn_time = time_basis - time_left;
			for(auto it = sectorParsers.begin(); it != sectorParsers.end();) {
				if (it->Parse(vsn_time, shift_even, shift_odd))
					++it;
				else
					it = sectorParsers.erase(it);
			}

			if (shift_even == 0xC7 && shift_odd == 0xFE) {
				sectorParsers.push_back(SectorParser());
				sectorParsers.back().Init(rawTrack.mTrack, &rawTrack.mIndexTimes, (float)scks_per_cell);
			}
		}
	}

done:
	;
}

void process_track_mfm(const RawTrack& rawTrack) {
	const double cells_per_rev = 500000.0 / (288.0 / 60.0);
	double scks_per_cell = rawTrack.mSamplesPerRev / cells_per_rev * g_clockPeriodAdjust;

	//printf("%.2f samples per cell\n", scks_per_cell);

	const uint32_t *samp = rawTrack.mTransitions.data();
	size_t samps_left = rawTrack.mTransitions.size() - 1;
	int time_basis = 0;
	int time_left = 0;

	int cell_len = (int)(scks_per_cell + 0.5);
	int cell_range = cell_len / 2;
	int cell_timer = 0;

	uint8_t last_track = 0;
	uint8_t last_sector = 0;
	int write_vsn = -1;
	float write_vsn_pos = 0;

	uint8_t shift_even = 0;
	uint8_t shift_odd = 0;

	std::vector<SectorParserMFM> sectorParsers;

	int state = 0;
	for(;;) {
		while (time_left <= 0) {
			if (!samps_left)
				goto done;

			time_left += samp[1] - samp[0];
			time_basis = samp[1];
			++samp;
			--samps_left;
		}

//		printf("next_trans = %d, cell_timer = %d, %d transitions left\n", time_left, cell_timer, samps_left);

		// if the shift register is empty, restart shift timing at next transition
		if (!(shift_even | shift_odd)) {
			time_left = 0;
			cell_timer = cell_len;
			shift_even = 0;
			shift_odd = 1;
		} else {
			// compare time to next transition against cell length
			int trans_delta = time_left - cell_timer;

			if (trans_delta < -cell_range) {
				// ignore the transition
				cell_timer -= time_left;
				continue;
			}

			std::swap(shift_even, shift_odd);
			shift_odd += shift_odd;
			
			if (trans_delta <= cell_range) {
				cell_timer = cell_len;

				// we have a transition in range -- clock in a 1 bit
				if (trans_delta < -5)
					cell_timer -= 3;
				else if (trans_delta < -3)
					cell_timer -= 2;
				else if (trans_delta < 1)
					--cell_timer;
				else if (trans_delta > 1)
					++cell_timer;
				else if (trans_delta > 3)
					cell_timer += 2;
				else if (trans_delta > 5)
					cell_timer += 3;

				shift_odd++;
				time_left = 0;
			} else {
				// we don't have a transition in range -- clock in a 0 bit
				time_left -= cell_timer;
				cell_timer = cell_len;
			}

			const uint32_t vsn_time = time_basis - time_left;
			for(auto it = sectorParsers.begin(); it != sectorParsers.end();) {
				if (it->Parse(vsn_time, shift_even, shift_odd))
					++it;
				else
					it = sectorParsers.erase(it);
			}

			// The IDAM is 0xA1 with a missing clock pulse:
			//
			// data		 0 0 0 0 0 0 0 0 1 0 1 0 0 0 0 1
			// clock	1 1 1 1 1 1 1 1 0 0 0 0 1>0<1 0

			if (state == 0) {
				if (shift_even == 0x0A && shift_odd == 0xA1)
					++state;
			} else if (state == 16) {
				if (shift_even == 0x0A && shift_odd == 0xA1)
					++state;
				else
					state = 0;
			} else if (state == 32) {
				if (shift_even == 0x0A && shift_odd == 0xA1) {
					sectorParsers.push_back(SectorParserMFM());
					sectorParsers.back().Init(rawTrack.mTrack, &rawTrack.mIndexTimes, (float)scks_per_cell);
				}

				state = 0;
			} else {
				++state;
			}
		}
	}

done:
	;
}

void sift_sectors(int track, std::vector<SectorInfo *>& secptrs) {
	std::vector<SectorInfo *> newsecptrs;

	// gather sectors from track
	secptrs.clear();
	FOR_EACH(sector, g_tracks[track].mSectors) {
		secptrs.push_back(&sector);
	};

	// sort sectors by index
	std::sort(secptrs.begin(), secptrs.end(),
		[](const SectorInfo *x, const SectorInfo *y) -> bool {
			return x->mIndex < y->mIndex;
		}
	);

	// extract out one group at a time
	std::vector<SectorInfo *> secgroup;

	while(!secptrs.empty()) {
		const int sector = secptrs.front()->mIndex;

		secgroup.clear();
		for(size_t idx = 0; idx < secptrs.size(); ) {
			if (secptrs[idx]->mIndex == sector) {
				secgroup.push_back(secptrs[idx]);
				secptrs[idx] = secptrs.back();
				secptrs.pop_back();
			} else {
				++idx;
			}
		}

		// sort sectors in group by angular position
		std::sort(secgroup.begin(), secgroup.end(),
			[](const SectorInfo *x, const SectorInfo *y) -> bool {
				return x->mPosition < y->mPosition;
			}
		);

		// check if we have any sectors which passed CRC check; if so, remove all that didn't
		bool crcOK = true;

		int n1 = secgroup.size();

		if (std::find_if(secgroup.begin(), secgroup.end(), [](const SectorInfo *x) { return x->mRecordedCRC == x->mComputedCRC; }) != secgroup.end()) {
			secgroup.erase(std::remove_if(secgroup.begin(), secgroup.end(),
				[](const SectorInfo *x) { return x->mRecordedCRC != x->mComputedCRC; }), secgroup.end());
		} else {
			crcOK = false;
		}

		int n2 = secgroup.size();

		if (n1 != n2)
			printf("WARNING: %u/%u bad sector reads discarded for track %d, sector %d\n", n1-n2, n2, track, sector);

		// fish out subgroups from remaining sectors
		auto it1 = secgroup.begin();
		int sector_count = 0;

		while(it1 != secgroup.end()) {
			float position0 = (*it1)->mPosition;
			float poserr_sum = 0;
			auto it2 = it1 + 1;

			bool mismatch = false;
			while(it2 != secgroup.end()) {
				// stop if sector angle is more than 5% off
				float poserr = (*it2)->mPosition - position0;

				if (poserr > 0.5f)
					poserr -= 1.0f;

				if (fabsf(poserr) > 0.05f)
					break;

				poserr_sum += poserr;

				if (!(*it1)->HasSameContents(**it2))
					mismatch = true;

				++it2;
			}

			SectorInfo *best_sector = *it1;

			// compute average angular position
			position0 += poserr_sum / (float)(it2 - it1);
			position0 -= floorf(position0);

			// check if we had more than one sector in this position that we kept
			if (it2 - it1 > 1 && mismatch) {
				// Alright, we have multiple sectors in the same place with different contents. Let's see
				// if we can narrow this down. Compute hashes of all the sectors, then find the most
				// popular sector.
				struct HashedSectorRef {
					SectorInfo *mpSector;
					uint32_t mHash;
				};

				struct HashedSectorPred {
					bool operator()(const HashedSectorRef& x, const HashedSectorRef& y) const {
						return x.mHash == y.mHash && x.mpSector->HasSameContents(*y.mpSector);
					}

					size_t operator()(const HashedSectorRef& x) const {
						return x.mHash;
					}
				};

				std::unordered_map<HashedSectorRef, uint32_t, HashedSectorPred, HashedSectorPred> hashedSectors;

				for(auto it = it1; it != it2; ++it) {
					HashedSectorRef hsref = { *it, (*it)->ComputeContentHash() };
					++hashedSectors[hsref];
				}

				// find the most popular; count is small, so we'll just do this linearly
				auto best_ref = hashedSectors.begin();
				for(auto it = hashedSectors.begin(), itEnd = hashedSectors.end(); it != itEnd; ++it) {
					if (it->second > best_ref->second)
						best_ref = it;
				}

				best_sector = best_ref->first.mpSector;

				// check if we had a sector duplicated more than once
				if (best_ref->second > 1) {
					printf(
						"WARNING: Multiple sectors found at track %d, sector %d at the same position\n"
						"         %.2f but different %s data. Keeping the most popular one.\n",
						track,
						sector,
						position0,
						crcOK ? "good" : "bad");
				} else {
					if (crcOK) {
						printf(
							"WARNING: Multiple different sectors found at track %d, sector %d at the same\n"
							"         position %.2f but different good data. Keeping one of them.\n",
							track,
							sector,
							position0);
					} else {
						// compute how much of the sector is in common
						uint32_t max_match = best_sector->mSectorSize;

						for(auto it = it1; it != it2; ++it) {
							if ((*it) == best_sector)
								continue;

							for(uint32_t i=0; i<max_match; ++i) {
								if ((*it)->mData[i] != best_sector->mData[i]) {
									max_match = i;
									break;
								}
							}
						}

						best_sector->mWeakOffset = max_match;

						printf(
							"WARNING: Multiple sectors found at track %d, sector %d at the same position\n"
							"         %.2f but different bad data. Encoding weak sector at offset %d.\n",
							track,
							sector,
							position0,
							max_match);
					}
				}
			} else {
				if (!crcOK) {
					printf("WARNING: CRC error detected for track %d, sector %d.\n", track, sector);
				}
			}

			// adjust sector position to center
			best_sector->mPosition = position0;

			newsecptrs.push_back(best_sector);
			++sector_count;

			it1 = it2;
		}

		if (sector_count > 1)
			printf("WARNING: %u phantom sector%s found for track %d, sector %d.\n", sector_count - 1, sector_count > 2 ? "s" : "", track, sector);
	}

	secptrs.swap(newsecptrs);

	// resort sectors by angular position
	std::sort(secptrs.begin(), secptrs.end(),
		[](const SectorInfo *x, const SectorInfo *y) -> bool {
			return x->mPosition < y->mPosition;
		}
	);
}

///////////////////////////////////////////////////////////////////////////

void banner() {
	puts("A8 raw disk conversion utility v0.3");
	puts("Copyright (C) 2014 Avery Lee, All Rights Reserved.");
	puts("Licensed under GNU General Public License, version 2 or later.");
	puts("");
}

void exit_usage() {
	puts("Usage: a8rawconv [options] input output");
	puts("");
	puts("Options:");
	puts("  -b    Dump detailed contents of bad sectors");
	puts("  -if   Set input format:");
	puts("          auto       Determine by input name extension");
	puts("          kryoflux   Read as KryoFlux raw stream (specify track00.0.raw)");
	puts("          scp        Read as SuperCard Pro stream");
	puts("  -l    Show track/sector layout map");
	puts("  -of   Set output format:");
	puts("          auto       Determine by output name extension");
	puts("          atr        Use ATR file format");
	puts("          atx        Use ATX file format");
	puts("  -p    Adjust clock period by percentage (50-200)");
	puts("          -p 50      Use 50% of normal period (2x clock)");
	puts("  -v    Verbose output");
	puts("  -vv   Very verbose output");
	puts("  -vvv  Extremely verbose output");

	exit(1);
}

void exit_argerr() {
	puts("Use -h or -? for help.");
	exit(1);
}

void parse_args(int argc, char **argv) {
	bool allow_switches = true;

	banner();

	if (argc) {
		--argc;
		++argv;
	}

	if (!argc) {
		exit_usage();
	}

	while(argc--) {
		const char *arg = *argv++;

		if (allow_switches && *arg == '-') {
			const char *sw = arg+1;

			if (!strcmp(sw, "-")) {
				allow_switches = false;
			} else if (!strcmp(sw, "h") || !strcmp(sw, "?")) {
				exit_usage();
			} else if (!strcmp(sw, "b")) {
				g_dumpBadSectors = true;
			} else if (!strcmp(sw, "l")) {
				g_showLayout = true;
			} else if (!strcmp(sw, "if")) {
				if (!argc--) {
					printf("Missing argument for -if switch.\n");
					exit_argerr();
				}

				arg = *argv++;
				if (!strcmp(arg, "auto"))
					g_inputFormat = kInputFormat_Auto;
				else if (!strcmp(arg, "kryoflux"))
					g_inputFormat = kInputFormat_KryoFluxStream;
				else {
					printf("Unsupported input format type: %s.\n", arg);
					exit_argerr();
				}
			} else if (!strcmp(sw, "of")) {
				if (!argc--) {
					printf("Missing argument for -of switch.\n");
					exit_argerr();
				}

				arg = *argv++;
				if (!strcmp(arg, "auto"))
					g_outputFormat = kOutputFormat_Auto;
				else if (!strcmp(arg, "atx"))
					g_outputFormat = kOutputFormat_ATX;
				else if (!strcmp(arg, "atr"))
					g_outputFormat = kOutputFormat_ATR;
				else {
					printf("Unsupported output format type: %s.\n", arg);
					exit_argerr();
				}
			} else if (!strcmp(sw, "p")) {
				if (!argc--) {
					printf("Missing argument for -if switch.\n");
					exit_argerr();
				}

				arg = *argv++;

				char dummy;
				float period;
				if (1 != sscanf(arg, "%g%c", &period, &dummy)
					|| !(period >= 50.0f && period <= 200.0f))
				{
					printf("Invalid period adjustment: %s\n", arg);
					exit_argerr();
				}

				g_clockPeriodAdjust = period / 100.0f;
			} else if (!strcmp(sw, "v")) {
				g_verbosity = 1;
			} else if (!strcmp(sw, "vv")) {
				g_verbosity = 2;
			} else if (!strcmp(sw, "vvv")) {
				g_verbosity = 3;
			} else {
				printf("Unknown switch: %s\n", arg);
				exit_argerr();
			}
		} else {
			if (g_inputPath.empty()) {
				if (!*arg) {
					printf("Invalid input path.\n");
					exit_argerr();
				}

				g_inputPath = arg;
			} else if (g_outputPath.empty()) {
				if (!*arg) {
					printf("Invalid output path.\n");
					exit_argerr();
				}

				g_outputPath = arg;
			} else {
				printf("Extraneous argument: %s\n", arg);
				exit_argerr();
			}
		}
	}

	if (g_inputPath.empty()) {
		printf("Missing input path.\n");
		exit_argerr();
	}

	if (g_outputPath.empty()) {
		printf("Missing output path.\n");
		exit_argerr();
	}

	if (g_inputFormat == kInputFormat_Auto) {
		const char *extptr = strrchr(g_inputPath.c_str(), '.');

		if (extptr) {
			std::string ext(extptr+1);

			std::transform(ext.begin(), ext.end(), ext.begin(), [](char c) { return tolower((unsigned char)c); });

			if (ext == "raw")
				g_inputFormat = kInputFormat_KryoFluxStream;
			else if (ext == "scp")
				g_inputFormat = kInputFormat_SuperCardProStream;
		}

		if (g_inputFormat == kInputFormat_Auto) {
			printf("Unable to determine input format from input path: %s. Use -if to override input format.\n", g_inputPath.c_str());
			exit_usage();
		}
	}

	if (g_inputFormat == kInputFormat_KryoFluxStream) {
		// attempt to identify track counter in KryoFlux stream filename
		const char *fn = g_inputPath.c_str();
		const char *s = strrchr(fn, '.');

		if (s) {
			while(s != fn && s[-1] >= '0' && s[-1] <= '9')
				--s;

			if (s != fn && s[-1] == '.') {
				--s;

				g_inputPathCountWidth = 0;
				while(s != fn && s[-1] == '0') {
					++g_inputPathCountWidth;
					--s;
				}

				if (s != fn && (s[-1] < '0' || s[-1] > '9'))
					g_inputPathCountPos = s - fn;
			}
		}

		if (!g_inputPathCountPos || !g_inputPathCountWidth || g_inputPathCountWidth > 10) {
			printf("Unable to determine filename pattern for KryoFlux raw track streams. Expected pattern: track00.0.raw.\n");
			exit_usage();
		}
	}

	if (g_outputFormat == kOutputFormat_Auto) {
		const char *extptr = strrchr(g_outputPath.c_str(), '.');

		if (extptr) {
			std::string ext(extptr+1);

			std::transform(ext.begin(), ext.end(), ext.begin(), [](char c) { return tolower((unsigned char)c); });

			if (ext == "atr") {
				g_outputFormat = kOutputFormat_ATR;
			} else if (ext == "atx") {
				g_outputFormat = kOutputFormat_ATX;
			}
		}

		if (g_outputFormat == kOutputFormat_Auto) {
			printf("Unable to determine output format from output path: %s. Use -of to override output format.\n", g_outputPath.c_str());
			exit_usage();
		}
	}
}

void show_layout() {
	char trackbuf[73];

	// write tracks
	for(int i=0; i<40; ++i) {
		TrackInfo& track_info = g_tracks[i];
		const int num_raw_secs = (int)track_info.mSectors.size();

		// sort sectors by angular position
		std::vector<SectorInfo *> secptrs(num_raw_secs);
		for(int j=0; j<num_raw_secs; ++j)
			secptrs[j] = &track_info.mSectors[j];

		sift_sectors(i, secptrs);

		const int num_secs = (int)secptrs.size();

		memset(trackbuf, ' ', 68);
		memset(trackbuf+68, 0, 5);

		FOR_EACH(sec_ptr, secptrs) {
			int xpos = (unsigned)(sec_ptr->mPosition * 68);
			if (sec_ptr->mIndex >= 10)
				trackbuf[xpos++] = '0' + sec_ptr->mIndex / 10;

			trackbuf[xpos++] = '0' + sec_ptr->mIndex % 10;
		};

		printf("%2d (%2d) | %s\n", i, (int)secptrs.size(), trackbuf);
	}
}

void write_atr(FILE *fo) {
	uint32_t sector_size = 128;
	uint32_t sectors_per_track = 18;

	// scan track 0 to see if we have DD format (256 byte sectors) or ED format
	// (128 byte sectors, >18 sectors/track)
	FOR_EACH(si, g_tracks[0].mSectors) {
		if (si.mSectorSize >= 256)
			sector_size = 256;

		if (si.mIndex > 18)
			sectors_per_track = 26;
	};

	if (sector_size >= 256)
		printf("Writing double density ATR file: %s\n", g_outputPath.c_str());
	else if (sectors_per_track > 18)
		printf("Writing enhanced density ATR file: %s\n", g_outputPath.c_str());
	else
		printf("Writing single density ATR file: %s\n", g_outputPath.c_str());

	const uint32_t total_bytes = 720 * sector_size;
	const uint32_t total_paras = total_bytes >> 4;

	// write header
	write_u8(0x96, fo);
	write_u8(0x02, fo);
	write_u16((uint16_t)total_paras, fo);
	write_u16(128, fo);
	write_u16((uint16_t)(total_paras >> 16), fo);
	write_pad(8, fo);

	// write tracks
	char secbuf[256];

	for(int i=0; i<40; ++i) {
		TrackInfo& track_info = g_tracks[i];
		const int num_raw_secs = (int)track_info.mSectors.size();

		// sort and sift out usable sectors from track
		std::vector<SectorInfo *> secptrs;
		sift_sectors(i, secptrs);

		// iterate over sectors
		const SectorInfo *secptrs2[26] = { 0 };

		FOR_EACH(secptr, secptrs) {
			if (secptr->mIndex >= 1 && secptr->mIndex <= (int)sectors_per_track) {
				if (secptrs2[secptr->mIndex - 1])
					printf("WARNING: Variable sectors not supported by ATR format. Discarding duplicate physical sector for track %d, sector %d.\n", i, secptr->mIndex);
				else
					secptrs2[secptr->mIndex - 1] = secptr;
			}
		};

		// write out sectors
		for(uint32_t j=0; j<sectors_per_track; ++j) {
			const SectorInfo *sec = secptrs2[j];

			if (!sec) {
				printf("WARNING: Missing sectors not supported by ATR format. Writing out null data for track %d, sector %d.\n", i, j+1);
				memset(secbuf, 0, sizeof secbuf);
			} else {
				memcpy(secbuf, sec->mData, sector_size);

				if (sec->mSectorSize != sector_size)
					printf("WARNING: Variable sector size not supported by ATR format. Writing out truncated data for track %d, sector %d.\n", i, j+1);
				else if (sec->mRecordedCRC != sec->mComputedCRC)
					printf("WARNING: CRC error encoding not supported by ATR format. Ignoring CRC error for track %d, sector %d.\n", i, j+1);
				else if (sec->mAddressMark != 0xFB)
					printf("WARNING: Deleted sector encoding not supported by ATR format. Ignoring error for track %d, sector %d.\n", i, j+1);
				else if (sec->mWeakOffset >= 0)
					printf("WARNING: Weak sector encoding not supported by ATR format. Ignoring error for track %d, sector %d.\n", i, j+1);
			}

			fwrite(secbuf, sector_size, 1, fo);
		}
	}
}

void write_atx(FILE *fo) {
	printf("Writing ATX file: %s\n", g_outputPath.c_str());

	// write header
	fwrite("AT8X", 4, 1, fo);
	write_u16(1, fo);
	write_u16(1, fo);
	write_u16(2, fo);
	write_u16(0, fo);
	write_pad(16, fo);
	write_u32(48, fo);
	write_u32(0, fo);
	write_pad(12, fo);

	// write tracks
	int phantom_sectors = 0;
	int missing_sectors = 0;
	int error_sectors = 0;

	for(int i=0; i<40; ++i) {
		TrackInfo& track_info = g_tracks[i];
		const int num_raw_secs = (int)track_info.mSectors.size();

		// sort and sift out usable sectors from track
		std::vector<SectorInfo *> secptrs;
		sift_sectors(i, secptrs);

		const int num_secs = (int)secptrs.size();

		bool sector_map[18] = {0};
		int weak_count = 0;
		FOR_EACH(sec_ptr, secptrs) {
			if (sec_ptr->mWeakOffset >= 0)
				++weak_count;

			if (sec_ptr->mIndex >= 1 && sec_ptr->mIndex <= 18) {
				if (sector_map[sec_ptr->mIndex - 1])
					++phantom_sectors;
				else
					sector_map[sec_ptr->mIndex - 1] = true;
			}
		};

		// write track header
		write_u32(32 + 8 + 8 + (128 + 8)*num_secs + 8*weak_count + 8, fo);
		write_u16(0, fo);
		write_u16(0, fo);
		write_u8(i, fo);
		write_u8(0, fo);
		write_u16(num_secs, fo);
		write_pad(5, fo);
		write_u8(0, fo);
		write_pad(2, fo);
		write_u32(32, fo);
		write_pad(8, fo);

		// write sector list header
		write_u32(8 + 8 * num_secs, fo);
		write_u8(1, fo);
		write_pad(3, fo);

		// write out sector headers

		uint32_t data_offset = 32 + 8 + 8*num_secs + 8;
//		for(const auto *sec_ptr : secptrs) {
		FOR_EACH(sec_ptr, secptrs) {
			write_u8(sec_ptr->mIndex, fo);

			uint8_t fdcStatus = 0;

			fdcStatus += (~sec_ptr->mAddressMark & 3) << 5;

			// set CRC error flag if CRC didn't match
			if (sec_ptr->mComputedCRC != sec_ptr->mRecordedCRC)
				fdcStatus += 0x08;

			// set CRC error, lost data, and DRQ flags if sector is long
			if (sec_ptr->mSectorSize != 128) {
				printf("WARNING: Long sector of %u bytes found for track %d, sector %d.\n", sec_ptr->mSectorSize, i, sec_ptr->mIndex);
				fdcStatus += 0x0e;
			}

			if (fdcStatus)
				++error_sectors;

			write_u8(fdcStatus, fo);
			write_u16((uint16_t)((int)(sec_ptr->mPosition * 26042) % 26042), fo);
			write_u32(data_offset, fo);
			data_offset += 128;
		};

		// write out sector data
		write_u32(128 * num_secs + 8, fo);
		write_u8(0, fo);
		write_pad(3, fo);

//		for(const auto *sec_ptr : secptrs) {
		FOR_EACH(sec_ptr, secptrs) {
			fwrite(sec_ptr->mData, 128, 1, fo);
		};

		// write out weak chunk info
		for(int j=0; j<num_secs; ++j) {
			if (secptrs[j]->mWeakOffset >= 0) {
				write_u32(8, fo);
				write_u8(0x10, fo);
				write_u8((uint8_t)j, fo);
				write_u16((uint16_t)secptrs[j]->mWeakOffset, fo);
			}
		};

		// write end of track chunks
		write_u32(0, fo);
		write_u32(0, fo);

		// report any missing sectors
		if (std::find(std::begin(sector_map), std::end(sector_map), true) == std::end(sector_map)) {
			printf("WARNING: No sectors found for track %d -- possibly unformatted.\n", i);
			missing_sectors += 18;
		} else {
			for(int j=0; j<18; ++j) {
				if (!sector_map[j]) {
					printf("WARNING: Missing track %d, sector %d.\n", i, j + 1);
					++missing_sectors;
				}
			}
		}
	}

	// back-patch size
	uint32_t total_size = ftell(fo);
	fseek(fo, 32, SEEK_SET);
	write_u32(total_size, fo);

	printf("%d missing sector%s, %d phantom sector%s, %d sector%s with errors\n"
		, missing_sectors, missing_sectors == 1 ? "" : "s"
		, phantom_sectors, phantom_sectors == 1 ? "" : "s"
		, error_sectors, error_sectors == 1 ? "" : "s");
}

int main(int argc, char **argv) {
	parse_args(argc, argv);

	switch(g_inputFormat) {
		case kInputFormat_KryoFluxStream:
			read_kryoflux();
			break;

		case kInputFormat_SuperCardProStream:
			read_supercardpro();
			break;
	}

	if (g_showLayout)
		show_layout();

	FILE *fo = fopen(g_outputPath.c_str(), "wb");
	if (!fo)
		fatalf("Unable to open output file: %s.\n", g_outputPath.c_str());

	if (g_outputFormat == kOutputFormat_ATX)
		write_atx(fo);
	else
		write_atr(fo);

	fclose(fo);

	return 0;
}
