// a8rawconv - A8 raw disk conversion utility
// Copyright (C) 2014 Avery Lee
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

#include "stdafx.h"
#include "encode.h"

template<class T>
class for_each_iterator {
public:
	for_each_iterator(T& c) : mContainer(c) {}

	template<class Fn>
	void operator>>=(const Fn& fn) {
		std::for_each(mContainer.begin(), mContainer.end(), fn);
	}

private:
	T& mContainer;
};

template<class T>
for_each_iterator<T> get_for_each(T& container) {
	return for_each_iterator<T>(container);
}

#define FOR_EACH(var, container) get_for_each((container)) >>= [&](decltype(*container.begin())& var)

DiskInfo g_disk;

int g_inputPathCountPos;
int g_inputPathCountWidth;

std::string g_outputPath;
bool g_showLayout;
bool g_encoding_fm = true;
bool g_encoding_mfm = true;
bool g_encoding_gcr = false;
float g_clockPeriodAdjust = 1.0f;
int g_trackSelect = -1;
int g_trackCount = 40;
int g_trackStep = 2;

enum InputFormat {
	kInputFormat_Auto,
	kInputFormat_KryoFluxStream,
	kInputFormat_SuperCardProStream,
	kInputFormat_SuperCardProDirect,
	kInputFormat_ATR,
	kInputFormat_ATX
} g_inputFormat = kInputFormat_Auto;

enum OutputFormat {
	kOutputFormat_Auto,
	kOutputFormat_ATX,
	kOutputFormat_ATR,
	kOutputFormat_SCP,
	kOutputFormat_SCPDirect
} g_outputFormat = kOutputFormat_Auto;

///////////////////////////////////////////////////////////////////////////

void process_track_fm(const RawTrack& rawTrack);
void process_track_mfm(const RawTrack& rawTrack);
void process_track_gcr(const RawTrack& rawTrack);

void process_track(const RawTrack& rawTrack) {
	if (g_encoding_fm)
		process_track_fm(rawTrack);
	
	if (g_encoding_mfm)
		process_track_mfm(rawTrack);

	if (g_encoding_gcr)
		process_track_gcr(rawTrack);
}

//////////////////////////////////////////////////////////////////////////

void process_track_fm(const RawTrack& rawTrack) {
	if (rawTrack.mTransitions.empty())
		return;

	// Atari disk timing produces 250,000 clocks per second at 288 RPM. We must compute the
	// effective sample rate given the actual disk rate.
	const double cells_per_rev = 250000.0 / (288.0 / 60.0);
	double scks_per_cell = rawTrack.mSamplesPerRev / cells_per_rev * g_clockPeriodAdjust;

	//printf("%.2f samples per cell\n", scks_per_cell);

	const uint32_t *samp = rawTrack.mTransitions.data();
	size_t samps_left = rawTrack.mTransitions.size() - 1;
	int time_basis = 0;
	int time_left = 0;

	int cell_len = (int)(scks_per_cell + 0.5);
	int cell_range = cell_len / 3;
	int cell_timer = 0;
	int cell_fine_adjust = 0;

	uint8_t shift_even = 0;
	uint8_t shift_odd = 0;

	std::vector<SectorParser> sectorParsers;
	uint8_t spew_data[16];
	int spew_index = 0;
	uint32_t spew_last_time = samp[0];

	for(;;) {
		while (time_left <= 0) {
			if (!samps_left)
				goto done;

			int delta = samp[1] - samp[0];

			if (g_verbosity >= 4)
				printf(" %02X %02X | %3d | %d\n", shift_even, shift_odd, delta, samp[0]);

			time_left += delta;
			time_basis = samp[1];
			++samp;
			--samps_left;
		}

		//printf("next_trans = %d, cell_timer = %d, %d transitions left\n", time_left, cell_timer, samps_left);

		// if the shift register is empty, restart shift timing at next transition
		if (!(shift_even | shift_odd)) {
			time_left = 0;
			cell_timer = cell_len;
			shift_even = 0;
			shift_odd = 1;
		} else {
			// compare time to next transition against cell length
			int trans_delta = time_left - cell_timer;

			if (trans_delta < -cell_range) {
				if (g_verbosity >= 4)
					printf(" %02X %02X | delta = %+3d | ignore\n", shift_even, shift_odd, trans_delta);
				// ignore the transition
				cell_timer -= time_left;
				continue;
			}

			std::swap(shift_even, shift_odd);
			shift_odd += shift_odd;
			
			if (trans_delta <= cell_range) {
				++shift_odd;

				if (g_verbosity >= 4)
					printf(" %02X %02X | delta = %+3d | 1\n", shift_even, shift_odd, trans_delta);

				// we have a transition in range -- clock in a 1 bit
#if 1
				cell_timer = cell_len;
				//cell_timer += cell_fine_adjust / 256;
				time_left = 0;

#if 1
				if (trans_delta < -5)
					cell_timer -= 3;
				else if (trans_delta < -3)
					cell_timer -= 2;
				else if (trans_delta < 1)
					--cell_timer;
				else if (trans_delta > 1)
					++cell_timer;
				else if (trans_delta > 3)
					cell_timer += 2;
				else if (trans_delta > 5)
					cell_timer += 3;
#endif

#if 0
				cell_fine_adjust += trans_delta;
				if (cell_fine_adjust < -16384)
					cell_fine_adjust = -16384;
				else if (cell_fine_adjust > 16384)
					cell_fine_adjust = 16384;
#endif
#elif 0
				cell_timer -= time_left;
				time_left = 0;
				cell_timer += cell_len + (cell_fine_adjust / 256);

				cell_timer += trans_delta / 3;
				cell_fine_adjust += trans_delta;
				if (cell_fine_adjust < -16384)
					cell_fine_adjust = -16384;
				else if (cell_fine_adjust > 16384)
					cell_fine_adjust = 16384;
#else
				cell_timer = cell_len + (cell_fine_adjust / 256);
				time_left = 0;
#endif

			} else {
				if (g_verbosity >= 4)
					printf(" %02X %02X | delta = %+3d | 0\n", shift_even, shift_odd, trans_delta);

				// we don't have a transition in range -- clock in a 0 bit
				time_left -= cell_timer;
				cell_timer = cell_len + (cell_fine_adjust / 256);
			}

			if (g_verbosity >= 3) {
				spew_data[spew_index] = shift_odd;
				if (++spew_index == 16) {
					spew_index = 0;
					printf("%02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X | %.2f\n"
						, spew_data[0]
						, spew_data[1]
						, spew_data[2]
						, spew_data[3]
						, spew_data[4]
						, spew_data[5]
						, spew_data[6]
						, spew_data[7]
						, spew_data[8]
						, spew_data[9]
						, spew_data[10]
						, spew_data[11]
						, spew_data[12]
						, spew_data[13]
						, spew_data[14]
						, spew_data[15]
						, (double)(time_basis - time_left - spew_last_time) / scks_per_cell
						);

					spew_last_time = time_basis - time_left;
				}
			}
			
			const uint32_t vsn_time = time_basis - time_left;
			for(auto it = sectorParsers.begin(); it != sectorParsers.end();) {
				if (it->Parse(vsn_time, shift_even, shift_odd))
					++it;
				else
					it = sectorParsers.erase(it);
			}

			if (shift_even == 0xC7 && shift_odd == 0xFE) {
				sectorParsers.push_back(SectorParser());
				sectorParsers.back().Init(rawTrack.mTrack, &rawTrack.mIndexTimes, (float)scks_per_cell, &g_disk.mTracks[rawTrack.mTrack]);
			}
		}
	}

done:
	;
}

void process_track_mfm(const RawTrack& rawTrack) {
	if (rawTrack.mTransitions.empty())
		return;

	const double cells_per_rev = 500000.0 / (288.0 / 60.0);
	double scks_per_cell = rawTrack.mSamplesPerRev / cells_per_rev * g_clockPeriodAdjust;

	//printf("%.2f samples per cell\n", scks_per_cell);

	const uint32_t *samp = rawTrack.mTransitions.data();
	size_t samps_left = rawTrack.mTransitions.size() - 1;
	int time_basis = 0;
	int time_left = 0;

	int cell_len = (int)(scks_per_cell + 0.5);
	int cell_range = cell_len / 2;
	int cell_timer = 0;

	uint8_t shift_even = 0;
	uint8_t shift_odd = 0;

	std::vector<SectorParserMFM> sectorParsers;
	uint8_t spew_data[16];
	int spew_index = 0;
	uint32_t spew_last_time = samp[0];

	int state = 0;
	for(;;) {
		while (time_left <= 0) {
			if (!samps_left)
				goto done;

			time_left += samp[1] - samp[0];
			time_basis = samp[1];
			++samp;
			--samps_left;
		}

//		printf("next_trans = %d, cell_timer = %d, %d transitions left\n", time_left, cell_timer, samps_left);

		// if the shift register is empty, restart shift timing at next transition
		if (!(shift_even | shift_odd)) {
			time_left = 0;
			cell_timer = cell_len;
			shift_even = 0;
			shift_odd = 1;
		} else {
			// compare time to next transition against cell length
			int trans_delta = time_left - cell_timer;

			if (trans_delta < -cell_range) {
				// ignore the transition
				cell_timer -= time_left;
				continue;
			}

			std::swap(shift_even, shift_odd);
			shift_odd += shift_odd;
			
			if (trans_delta <= cell_range) {
				cell_timer = cell_len;

				// we have a transition in range -- clock in a 1 bit
				if (trans_delta < -5)
					cell_timer -= 3;
				else if (trans_delta < -3)
					cell_timer -= 2;
				else if (trans_delta < 1)
					--cell_timer;
				else if (trans_delta > 1)
					++cell_timer;
				else if (trans_delta > 3)
					cell_timer += 2;
				else if (trans_delta > 5)
					cell_timer += 3;

				shift_odd++;
				time_left = 0;
			} else {
				// we don't have a transition in range -- clock in a 0 bit
				time_left -= cell_timer;
				cell_timer = cell_len;
			}

			if (g_verbosity >= 3) {
				spew_data[spew_index] = shift_odd;
				if (++spew_index == 16) {
					spew_index = 0;
					printf("%02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X | %.2f\n"
						, spew_data[0]
						, spew_data[1]
						, spew_data[2]
						, spew_data[3]
						, spew_data[4]
						, spew_data[5]
						, spew_data[6]
						, spew_data[7]
						, spew_data[8]
						, spew_data[9]
						, spew_data[10]
						, spew_data[11]
						, spew_data[12]
						, spew_data[13]
						, spew_data[14]
						, spew_data[15]
						, (double)(time_basis - time_left - spew_last_time) / scks_per_cell
						);

					spew_last_time = time_basis - time_left;
				}
			}

			const uint32_t vsn_time = time_basis - time_left;
			for(auto it = sectorParsers.begin(); it != sectorParsers.end();) {
				if (it->Parse(vsn_time, shift_even, shift_odd))
					++it;
				else
					it = sectorParsers.erase(it);
			}

			// The IDAM is 0xA1 with a missing clock pulse:
			//
			// data		 0 0 0 0 0 0 0 0 1 0 1 0 0 0 0 1
			// clock	1 1 1 1 1 1 1 1 0 0 0 0 1>0<1 0

			if (state == 0) {
				if (shift_even == 0x0A && shift_odd == 0xA1)
					++state;
			} else if (state == 16) {
				if (shift_even == 0x0A && shift_odd == 0xA1)
					++state;
				else
					state = 0;
			} else if (state == 32) {
				if (shift_even == 0x0A && shift_odd == 0xA1) {
					sectorParsers.push_back(SectorParserMFM());
					sectorParsers.back().Init(rawTrack.mTrack, &rawTrack.mIndexTimes, (float)scks_per_cell, &g_disk.mTracks[rawTrack.mTrack]);
				}

				state = 0;
			} else {
				++state;
			}
		}
	}

done:
	;
}

static const uint8_t kGCR6Decoder[256]={
#define IL 255
	IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,
	IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,
	IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,
	IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,
	IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,
	IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,
	IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,
	IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,
	IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,

	// $90
	IL,IL,IL,IL,IL,IL, 0, 1,IL,IL, 2, 3,IL, 4, 5, 6,

	// $A0
	IL,IL,IL,IL,IL,IL, 7, 8,IL,IL, 8, 9,10,11,12,13,

	// $B0
	IL,IL,14,15,16,17,18,19,IL,20,21,22,23,24,25,26,

	// $C0
	IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,IL,27,IL,28,29,30,

	// $D0
	IL,IL,IL,31,IL,IL,32,33,IL,34,35,36,37,38,39,40,

	// $E0
	IL,IL,IL,IL,IL,41,42,43,IL,44,45,46,47,48,49,50,

	// $F0
	IL,IL,51,52,53,54,55,56,IL,57,58,59,60,61,62,63,
#undef IL
};

void process_track_gcr(const RawTrack& rawTrack) {
	double rpm = 590.0;

	if (rawTrack.mTrack < 16)
		rpm = 394.0;
	else if (rawTrack.mTrack < 32)
		rpm = 429.0;
	else if (rawTrack.mTrack < 48)
		rpm = 472.0;
	else if (rawTrack.mTrack < 64)
		rpm = 525.0;

	const double cells_per_rev = 500000.0 / (rpm / 60.0);
	double scks_per_cell = rawTrack.mSamplesPerRev / cells_per_rev * g_clockPeriodAdjust;

	//printf("%.2f samples per cell\n", scks_per_cell);

	const uint32_t *samp = rawTrack.mTransitions.data();
	size_t samps_left = rawTrack.mTransitions.size() - 1;
	int time_left = 0;

	int cell_len = (int)(scks_per_cell + 0.5);
	int cell_range = cell_len / 2;
	int cell_timer = 0;

	uint8_t shifter = 0;

	int bit_state = 0;
	int byte_state = 0;

	int sector_headers = 0;
	int data_sectors = 0;
	int good_sectors = 0;

	uint8_t buf[704];
	uint8_t decbuf[528];

	for(;;) {
		while (time_left <= 0) {
			if (!samps_left)
				goto done;

			//printf("%d\n", samp[1] - samp[0]);
			time_left += samp[1] - samp[0];
			++samp;
			--samps_left;
		}

		// if the shift register is empty, restart shift timing at next transition
		if (!shifter) {
			time_left = 0;
			cell_timer = cell_len;
			shifter = 1;
		} else {
			// compare time to next transition against cell length
			int trans_delta = time_left - cell_timer;

			if (0 && trans_delta < -cell_range) {
				// ignore the transition
				cell_timer -= time_left;
				time_left = 0;
				continue;
			}

			shifter += shifter;
			
			if (trans_delta <= cell_range) {
				cell_timer = cell_len;
				time_left = 0;

				// we have a transition in range -- clock in a 1 bit
				if (trans_delta < -5)
					cell_timer -= 3;
				else if (trans_delta < -3)
					cell_timer -= 2;
				else if (trans_delta < 1)
					--cell_timer;
				else if (trans_delta > 1)
					++cell_timer;
				else if (trans_delta > 3)
					cell_timer += 2;
				else if (trans_delta > 5)
					cell_timer += 3;

				shifter++;
			} else {
				// we don't have a transition in range -- clock in a 0 bit
				time_left -= cell_timer;
				cell_timer = cell_len;
			}

			// advance bit machine state
			if (bit_state == 0) {
				if (shifter & 0x80) {
					bit_state = 1;

//					printf("%4u  %02X\n", byte_state, shifter);

					// okay, we have a byte... advance the byte state machine.
					if (byte_state == 0) {			// waiting for FF
						if (shifter == 0xFF)
							byte_state = 1;
					} else if (byte_state == 1) {	// waiting for D5 in address/data mark
						if (shifter == 0xD5)
							byte_state = 2;
						else if (shifter != 0xFF)
							byte_state = 0;
					} else if (byte_state == 2) {	// waiting for AA in address/data mark
						if (shifter == 0xAA)
							byte_state = 3;
						else if (shifter == 0xFF)
							byte_state = 1;
						else
							byte_state = 0;
					} else if (byte_state == 3) {	// waiting for 96 for address mark or AD for data mark
						if (shifter == 0x96)
							byte_state = 10;
						else if (shifter == 0xAD)
							byte_state = 1000;
						else if (shifter == 0xFF)
							byte_state = 1;
						else
							byte_state = 0;
					} else if (byte_state >= 10 && byte_state < 15) {
						// found D5 AA 96 for address mark - read track, sector, side,
						// format, checksum bytes
						buf[byte_state - 10] = shifter;

						if (++byte_state == 15) {
							uint8_t checksum = 0;

							for(int i=0; i<5; ++i) {
								decbuf[i] = kGCR6Decoder[buf[i]];
								checksum ^= decbuf[i];
							}

							if (!checksum) {
								printf("Sector header %02X %02X %02X %02X %02X\n", decbuf[0], decbuf[1], decbuf[2], decbuf[3], decbuf[4]);
							}

							++sector_headers;
							byte_state = 0;
						}
					} else if (byte_state >= 1000 && byte_state < 1704) {
						buf[byte_state - 1000] = shifter;

						if (++byte_state == 1704) {
							// decode first 522 of 524 data bytes
							uint8_t checksumA = 0;
							uint8_t checksumB = 0;
							uint8_t checksumC = 0;
							uint8_t carry = 0;
							uint32_t invalid = 0;

							for(int i=0; i<175; ++i) {
								const uint8_t x0 = kGCR6Decoder[buf[i*4+0+1]];
								const uint8_t x1 = kGCR6Decoder[buf[i*4+1+1]];
								const uint8_t x2 = kGCR6Decoder[buf[i*4+2+1]];
								const uint8_t x3 = kGCR6Decoder[buf[i*4+3+1]];

								invalid += (x0 >> 7);
								invalid += (x1 >> 7);
								invalid += (x2 >> 7);
								invalid += (x3 >> 7);

								checksumC = (checksumC << 1) + (checksumC >> 7);

								uint8_t y0 = x1 + ((x0 << 2) & 0xc0);
								y0 ^= checksumC;

								uint32_t tmpSumA = (uint32_t)checksumA + y0 + (checksumC & 1);
								checksumA = (uint8_t)tmpSumA;
								carry = (uint8_t)(tmpSumA >> 8);

								uint8_t y1 = x2 + ((x0 << 4) & 0xc0);
								y1 ^= checksumA;

								uint32_t tmpSumB = (uint32_t)checksumB + y1 + carry;
								checksumB = (uint8_t)tmpSumB;
								carry = (uint8_t)(tmpSumB >> 8);

								decbuf[i*3+0] = y0;
								decbuf[i*3+1] = y1;

								if (i<174) {		// @&*(@$
									uint8_t y2 = x3 + ((x0 << 6) & 0xc0);
									y2 ^= checksumB;

									uint32_t tmpSumC = (uint32_t)checksumC + y2 + carry;
									checksumC = (uint8_t)tmpSumC;
									carry = (uint8_t)(tmpSumC >> 8);
									decbuf[i*3+2] = y2;
								}
							}

							const uint8_t z0 = kGCR6Decoder[buf[175*4+0]];
							const uint8_t z1 = kGCR6Decoder[buf[175*4+1]];
							const uint8_t z2 = kGCR6Decoder[buf[175*4+2]];
							const uint8_t z3 = kGCR6Decoder[buf[175*4+3]];
							invalid += (z0 >> 7);
							invalid += (z1 >> 7);
							invalid += (z2 >> 7);
							invalid += (z3 >> 7);

							uint8_t decCheckA = z1 + ((z0 << 2) & 0xc0);
							uint8_t decCheckB = z2 + ((z0 << 4) & 0xc0);
							uint8_t decCheckC = z3 + ((z0 << 6) & 0xc0);

							if (invalid)
								printf("%u invalid GCR bytes encountered\n", invalid);

							bool checksumOK = (checksumA == decCheckA && checksumB == decCheckB && checksumC == decCheckC);

							printf("checksums: %02X %02X %02X vs. %02X %02X %02X (%s)\n"
								, checksumA
								, checksumB
								, checksumC
								, decCheckA
								, decCheckB
								, decCheckC
								, checksumOK
									? "good" : "BAD"
								);

							++data_sectors;

							if (checksumOK)
								++good_sectors;

							byte_state = 1;
						}
					}
				}
			} else {
				++bit_state;

				if (bit_state == 8)
					bit_state = 0;
			}
		}
	}

done:
	;

	printf("%d sector headers decoded\n", sector_headers);
	printf("%d data sectors decoded\n", data_sectors);
	printf("%d good sectors decoded\n", good_sectors);
}

///////////////////////////////////////////////////////////////////////////

void banner() {
	puts("A8 raw disk conversion utility v0.6");
	puts("Copyright (C) 2014-2015 Avery Lee, All Rights Reserved.");
	puts("Licensed under GNU General Public License, version 2 or later.");
	puts("");
}

void exit_usage() {
	puts("Usage: a8rawconv [options] input output");
	puts("");
	puts("Options:");
	puts("  -b    Dump detailed contents of bad sectors");
	puts("  -d    Decoding mode");
	puts("          auto       Try both FM and MFM");
	puts("          fm         FM only");
	puts("          mfm        MFM only");
	puts("          gcr        GCR only");
	puts("  -if   Set input format:");
	puts("          auto       Determine by input name extension");
	puts("          atr        Read as ATR disk image");
	puts("          atx        Read as ATX disk image");
	puts("          kryoflux   Read as KryoFlux raw stream (specify track00.0.raw)");
	puts("          scp        Read as SuperCard Pro stream");
	puts("  -l    Show track/sector layout map");
	puts("  -of   Set output format:");
	puts("          auto       Determine by output name extension");
	puts("          atr        Use ATR disk image format");
	puts("          atx        Use ATX disk image format");
	puts("          scp        Use SuperCard Pro stream format");
	puts("  -p    Adjust clock period by percentage (50-200)");
	puts("          -p 50      Use 50% of normal period (2x clock)");
	puts("  -t    Restrict processing to single track");
	puts("          -t 4       Process only track 4");
	puts("  -v    Verbose output");
	puts("  -vv   Very verbose output: sector-level debugging");
	puts("  -vvv  Extremely verbose output: bit level debugging");
	puts("  -vvvv Ridiculously verbose output: flux level debugging");
	puts("");
	puts("Direct SCP read/write is possible using the following path:");
	puts("    scp[drive]:(48tpi|96tpi)[:override_port_path]");
	puts("");
	puts("Example: scp0:96tpi to use a 96 TPI drive 0");

	exit(1);
}

void exit_argerr() {
	puts("Use -h or -? for help.");
	exit(1);
}

void parse_args(int argc, char **argv) {
	bool allow_switches = true;

	banner();

	if (argc) {
		--argc;
		++argv;
	}

	if (!argc) {
		exit_usage();
	}

	while(argc--) {
		const char *arg = *argv++;

		if (allow_switches && *arg == '-') {
			const char *sw = arg+1;

			if (!strcmp(sw, "-")) {
				allow_switches = false;
			} else if (!strcmp(sw, "h") || !strcmp(sw, "?")) {
				exit_usage();
			} else if (!strcmp(sw, "b")) {
				g_dumpBadSectors = true;
			} else if (!strcmp(sw, "l")) {
				g_showLayout = true;
			} else if (!strcmp(sw, "d")) {
				if (!argc--) {
					printf("Missing argument for -d switch.\n");
					exit_argerr();
				}

				arg = *argv++;
				if (!strcmp(arg, "auto")) {
					g_encoding_fm = true;
					g_encoding_mfm = true;
					g_encoding_gcr = false;
				} else if (!strcmp(arg, "fm")) {
					g_encoding_fm = true;
					g_encoding_mfm = false;
					g_encoding_gcr = false;
				} else if (!strcmp(arg, "mfm")) {
					g_encoding_fm = false;
					g_encoding_mfm = true;
					g_encoding_gcr = false;
				} else if (!strcmp(arg, "gcr")) {
					g_encoding_fm = false;
					g_encoding_mfm = false;
					g_encoding_gcr = true;
					g_trackCount = 80;
					g_trackStep = 1;
				} else {
					printf("Unsupported decoding mode: %s.\n", arg);
					exit_argerr();
				}
			} else if (!strcmp(sw, "if")) {
				if (!argc--) {
					printf("Missing argument for -if switch.\n");
					exit_argerr();
				}

				arg = *argv++;
				if (!strcmp(arg, "auto"))
					g_inputFormat = kInputFormat_Auto;
				else if (!strcmp(arg, "kryoflux"))
					g_inputFormat = kInputFormat_KryoFluxStream;
				else if (!strcmp(arg, "scp"))
					g_inputFormat = kInputFormat_SuperCardProStream;
				else if (!strcmp(arg, "atr"))
					g_inputFormat = kInputFormat_ATR;
				else if (!strcmp(arg, "atx"))
					g_inputFormat = kInputFormat_ATX;
				else {
					printf("Unsupported input format type: %s.\n", arg);
					exit_argerr();
				}
			} else if (!strcmp(sw, "of")) {
				if (!argc--) {
					printf("Missing argument for -of switch.\n");
					exit_argerr();
				}

				arg = *argv++;
				if (!strcmp(arg, "auto"))
					g_outputFormat = kOutputFormat_Auto;
				else if (!strcmp(arg, "atx"))
					g_outputFormat = kOutputFormat_ATX;
				else if (!strcmp(arg, "atr"))
					g_outputFormat = kOutputFormat_ATR;
				else if (!strcmp(arg, "scp"))
					g_outputFormat = kOutputFormat_SCP;
				else {
					printf("Unsupported output format type: %s.\n", arg);
					exit_argerr();
				}
			} else if (!strcmp(sw, "p")) {
				if (!argc--) {
					printf("Missing argument for -if switch.\n");
					exit_argerr();
				}

				arg = *argv++;

				char dummy;
				float period;
				if (1 != sscanf(arg, "%g%c", &period, &dummy)
					|| !(period >= 50.0f && period <= 200.0f))
				{
					printf("Invalid period adjustment: %s\n", arg);
					exit_argerr();
				}

				g_clockPeriodAdjust = period / 100.0f;
			} else if (!strcmp(sw, "v")) {
				g_verbosity = 1;
			} else if (!strcmp(sw, "vv")) {
				g_verbosity = 2;
			} else if (!strcmp(sw, "vvv")) {
				g_verbosity = 3;
			} else if (!strcmp(sw, "vvvv")) {
				g_verbosity = 4;
			} else if (!strcmp(sw, "t")) {
				if (!argc--) {
					printf("Missing argument for -t switch.\n");
					exit_argerr();
				}

				arg = *argv++;

				char dummy;
				unsigned track;
				if (1 != sscanf(arg, "%u%c", &track, &dummy) || track > 79)
				{
					printf("Invalid track number: %s\n", arg);
					exit_argerr();
				}

				g_trackSelect = track;
			} else {
				printf("Unknown switch: %s\n", arg);
				exit_argerr();
			}
		} else {
			if (g_inputPath.empty()) {
				if (!*arg) {
					printf("Invalid input path.\n");
					exit_argerr();
				}

				g_inputPath = arg;
			} else if (g_outputPath.empty()) {
				if (!*arg) {
					printf("Invalid output path.\n");
					exit_argerr();
				}

				g_outputPath = arg;
			} else {
				printf("Extraneous argument: %s\n", arg);
				exit_argerr();
			}
		}
	}

	if (g_inputPath.empty()) {
		printf("Missing input path.\n");
		exit_argerr();
	}

	if (g_outputPath.empty()) {
		printf("Missing output path.\n");
		exit_argerr();
	}

	if (g_inputFormat == kInputFormat_Auto) {
		if (g_inputPath.compare(0, 5, "scp0:") == 0
			|| g_inputPath.compare(0, 5, "scp1:") == 0)
		{
			g_inputFormat = kInputFormat_SuperCardProDirect;
		} else {
			const char *extptr = strrchr(g_inputPath.c_str(), '.');

			if (extptr) {
				std::string ext(extptr+1);

				std::transform(ext.begin(), ext.end(), ext.begin(), [](char c) { return tolower((unsigned char)c); });

				if (ext == "raw")
					g_inputFormat = kInputFormat_KryoFluxStream;
				else if (ext == "scp")
					g_inputFormat = kInputFormat_SuperCardProStream;
				else if (ext == "atr")
					g_inputFormat = kInputFormat_ATR;
				else if (ext == "atx")
					g_inputFormat = kInputFormat_ATX;
			}
		}

		if (g_inputFormat == kInputFormat_Auto) {
			printf("Unable to determine input format from input path: %s. Use -if to override input format.\n", g_inputPath.c_str());
			exit_usage();
		}
	}

	if (g_inputFormat == kInputFormat_KryoFluxStream) {
		// attempt to identify track counter in KryoFlux stream filename
		const char *fn = g_inputPath.c_str();
		const char *s = strrchr(fn, '.');

		if (s) {
			while(s != fn && s[-1] >= '0' && s[-1] <= '9')
				--s;

			if (s != fn && s[-1] == '.') {
				--s;

				g_inputPathCountWidth = 0;
				while(s != fn && s[-1] == '0') {
					++g_inputPathCountWidth;
					--s;
				}

				if (s != fn && (s[-1] < '0' || s[-1] > '9'))
					g_inputPathCountPos = s - fn;
			}
		}

		if (!g_inputPathCountPos || !g_inputPathCountWidth || g_inputPathCountWidth > 10) {
			printf("Unable to determine filename pattern for KryoFlux raw track streams. Expected pattern: track00.0.raw.\n");
			exit_usage();
		}
	}

	if (g_outputFormat == kOutputFormat_Auto) {
		if (g_outputPath.compare(0, 5, "scp0:") == 0
			|| g_outputPath.compare(0, 5, "scp1:") == 0)
		{
			g_outputFormat = kOutputFormat_SCPDirect;
		} else {
			const char *extptr = strrchr(g_outputPath.c_str(), '.');

			if (extptr) {
				std::string ext(extptr+1);

				std::transform(ext.begin(), ext.end(), ext.begin(), [](char c) { return tolower((unsigned char)c); });

				if (ext == "atr") {
					g_outputFormat = kOutputFormat_ATR;
				} else if (ext == "atx") {
					g_outputFormat = kOutputFormat_ATX;
				} else if (ext == "scp") {
					g_outputFormat = kOutputFormat_SCP;
				}
			}
		}

		if (g_outputFormat == kOutputFormat_Auto) {
			printf("Unable to determine output format from output path: %s. Use -of to override output format.\n", g_outputPath.c_str());
			exit_usage();
		}
	}
}

void show_layout() {
	char trackbuf[73];

	DiskInfo tempDisk(g_disk);

	// write tracks
	for(int i=0; i<40; ++i) {
		if (g_trackSelect >= 0 && g_trackSelect != i)
			continue;

		TrackInfo& track_info = tempDisk.mTracks[i];
		const int num_raw_secs = (int)track_info.mSectors.size();

		// sort sectors by angular position
		std::vector<SectorInfo *> secptrs(num_raw_secs);
		for(int j=0; j<num_raw_secs; ++j)
			secptrs[j] = &track_info.mSectors[j];

		sift_sectors(track_info, i, secptrs);

		memset(trackbuf, ' ', 68);
		memset(trackbuf+68, 0, 5);

		FOR_EACH(sec_ptr, secptrs) {
			int xpos = (unsigned)(sec_ptr->mPosition * 68);
			if (sec_ptr->mIndex >= 10)
				trackbuf[xpos++] = '0' + sec_ptr->mIndex / 10;

			trackbuf[xpos++] = '0' + sec_ptr->mIndex % 10;
		};

		printf("%2d (%2d) | %s\n", i, (int)secptrs.size(), trackbuf);
	}
}

int main(int argc, char **argv) {
	parse_args(argc, argv);

	bool src_raw = false;
	bool dst_raw = false;

	switch(g_outputFormat) {
		case kOutputFormat_SCP:
		case kOutputFormat_SCPDirect:
			dst_raw = true;
			break;
	}

	RawDisk raw_disk;

	switch(g_inputFormat) {
		case kInputFormat_KryoFluxStream:
			kf_read(raw_disk, g_trackCount, g_trackStep, g_inputPath.c_str(), g_inputPathCountPos, g_inputPathCountWidth, g_trackSelect);
			src_raw = true;
			break;

		case kInputFormat_SuperCardProStream:
			scp_read(raw_disk, g_inputPath.c_str(), g_trackSelect);
			src_raw = true;
			break;

		case kInputFormat_SuperCardProDirect:
			scp_direct_read(raw_disk, g_inputPath.c_str(), g_trackSelect);
			src_raw = true;
			break;

		case kInputFormat_ATR:
			read_atr(g_disk, g_inputPath.c_str(), g_trackSelect);
			break;

		case kInputFormat_ATX:
			read_atx(g_disk, g_inputPath.c_str(), g_trackSelect);
			break;
	}

	if (src_raw && !dst_raw) {
		if (g_trackSelect >= 0)
			process_track(raw_disk.mTracks[g_trackSelect]);
		else {
			for(int i=0; i<40; ++i)
				process_track(raw_disk.mTracks[i]);
		}
	} else if (!src_raw && dst_raw) {
		encode_disk(raw_disk, g_disk, g_clockPeriodAdjust);
	}

	if (g_showLayout && (!src_raw || !dst_raw))
		show_layout();

	switch(g_outputFormat) {
		case kOutputFormat_ATX:
			write_atx(g_outputPath.c_str(), g_disk, g_trackSelect);
			break;

		case kOutputFormat_ATR:
			write_atr(g_outputPath.c_str(), g_disk, g_trackSelect);
			break;

		case kOutputFormat_SCP:
			scp_write(raw_disk, g_outputPath.c_str(), g_trackSelect);
			break;

		case kOutputFormat_SCPDirect:
			scp_direct_write(raw_disk, g_outputPath.c_str(), g_trackSelect);
			break;
	}

	return 0;
}
