#include "stdafx.h"

SectorParser::SectorParser()
	: mReadPhase(0)
	, mBitPhase(0)
{
}

void SectorParser::Init(int track, const std::vector<uint32_t> *indexTimes, float samplesPerCell, TrackInfo *dstTrack) {
	mTrack = track;
	mpIndexTimes = indexTimes;
	mSamplesPerCell = samplesPerCell;
	mpDstTrack = dstTrack;
}

bool SectorParser::Parse(uint32_t stream_time, uint8_t clock_bits, uint8_t data_bits) {
	if (mReadPhase < 6) {
		if (++mBitPhase == 16) {
			mBitPhase = 0;

			if (clock_bits != 0xFF)
				return false;

			mBuf[++mReadPhase] = data_bits;

			if (mReadPhase == 6) {
				if (mBuf[1] != mTrack) {
					//printf("Track number mismatch!\n");
					return false;
				}

				// Byte 3 of the address mark is the side indicator, which is supposed to be
				// zero for the 1771. It appears not to validate it, because Rescue on Fractalus
				// has garbage in that field.
#if 0
				if (mBuf[2] != 0) {
					printf("Zero #1 failed! (got %02X)\n", mBuf[2]);
					return false;
				}
#endif

				if (mBuf[3] < 1 || mBuf[3] > 18) {
					printf("Invalid sector number\n");
					return false;
				}

				if (mBuf[4] > 3) {
					printf("Bad sector size!\n");
					return false;
				}

				mBuf[0] = 0xFE;
				const uint16_t computedCRC = ComputeCRC(mBuf, 5);
				const uint16_t recordedCRC = ((uint16_t)mBuf[5] << 8) + mBuf[6];

				if (computedCRC != recordedCRC) {
					printf("CRC failure on sector header: %04X != %04X\n", computedCRC, recordedCRC);
					return false;
				}

				mSector = mBuf[3];
				mSectorSize = 128 << mBuf[4];

				int vsn_time = stream_time;

				// find the nearest index mark
				auto it_index = std::upper_bound(mpIndexTimes->begin(), mpIndexTimes->end(), (uint32_t)vsn_time + 1);

				if (it_index == mpIndexTimes->begin()) {
					if (g_verbosity >= 2)
						printf("Skipping track %d, sector %d before first index mark\n", mTrack, mSector);
					return false;
				}

				if (it_index == mpIndexTimes->end()) {
					if (g_verbosity >= 2)
						printf("Skipping track %d, sector %d after last index mark\n", mTrack, mSector);
					return false;
				}

				int vsn_offset = vsn_time - *--it_index;

				mRotPos = (float)vsn_offset / (float)(it_index[1] - it_index[0]);

				if (mRotPos >= 1.0f)
					mRotPos -= 1.0f;

				if (g_verbosity >= 2)
					printf("Found track %d, sector %d at position %4.2f\n", mTrack, mSector, mRotPos);
			}
		}
	} else if (mReadPhase == 6) {
		// There must be at least a full zero byte before the DAM is recognized.
		if (clock_bits == 0xFF && data_bits == 0x00) {
			mBitPhase = 0;
		} else {
			++mBitPhase;
			
			if (clock_bits == 0xC7 && mBitPhase == 16) {
				// another IDAM detected before DAM -- terminate
				if (data_bits == 0xFE)
					return false;

				if (data_bits == 0xF8 || data_bits == 0xF9 || data_bits == 0xFA || data_bits == 0xFB) {
					//printf("DAM detected (%02X)\n", data_bits);

					mReadPhase = 7;
					mBitPhase = 0;
					mBuf[0] = data_bits;
					mClockBuf[0] = clock_bits;
					mStreamTimes[0] = stream_time;
				}
			}
		}
	} else {
		if (++mBitPhase == 16) {
			if (clock_bits != 0xFF) {
				if (g_verbosity > 1)
					printf("Bad data clock: %02X\n", clock_bits);
			}

//			printf("Data; %02X\n", ~data_bits);
			mBuf[mReadPhase - 6] = data_bits;
			mClockBuf[mReadPhase - 6] = clock_bits;
			mStreamTimes[mReadPhase - 6] = stream_time;

			mBitPhase = 0;
			if (++mReadPhase >= mSectorSize + 3 + 6) {
				// check if the CRC is correct
				// x^16 + x^12 + x^5 + 1
				uint16_t crc = ComputeCRC(mBuf, mSectorSize + 1);

				const uint16_t recordedCRC = ((uint16_t)mBuf[mSectorSize + 1] << 8) + (uint8_t)mBuf[mSectorSize + 2];

				//printf("Read sector %d: CRCs %04X, %04X\n", mSector, crc, recordedCRC);

				// add new sector entry
				auto& tracksecs = mpDstTrack->mSectors;
				tracksecs.push_back(SectorInfo());
				SectorInfo& newsec = tracksecs.back();

				for(int i=0; i<mSectorSize; ++i)
					newsec.mData[i] = ~mBuf[i+1];

				newsec.mIndex = mSector;
				newsec.mPosition = mRotPos;
				newsec.mAddressMark = mBuf[0];
				newsec.mRecordedCRC = recordedCRC;
				newsec.mComputedCRC = crc;
				newsec.mSectorSize = mSectorSize;
				newsec.mWeakOffset = -1;
				newsec.mbMFM = false;

				if (g_verbosity >= 1 || (crc != recordedCRC && g_dumpBadSectors))
					printf("Decoded FM track %d, sector %d with %u bytes, DAM %02X, recorded CRC %04X (computed %04X)\n",
						mTrack,
						mSector,
						mSectorSize,
						mBuf[0],
						recordedCRC,
						crc);
			
				if (g_dumpBadSectors && crc != recordedCRC) {
					printf("  Index Clk Data Cells\n");
					for(int i=0; i<mSectorSize + 1; ++i) {
						printf("  %4d  %02X | %02X (%02X,%02X %02X %02X %02X %02X %02X %02X) | %+6.1f%s\n"
							, i - 1
							, mClockBuf[i]
							, mBuf[i]
							, 0xFF ^ mBuf[i]
							, 0xFF ^ (((mBuf[i] << 1) + (mBuf[i+1]>>7)) & 0xFF)
							, 0xFF ^ (((mBuf[i] << 2) + (mBuf[i+1]>>6)) & 0xFF)
							, 0xFF ^ (((mBuf[i] << 3) + (mBuf[i+1]>>5)) & 0xFF)
							, 0xFF ^ (((mBuf[i] << 4) + (mBuf[i+1]>>4)) & 0xFF)
							, 0xFF ^ (((mBuf[i] << 5) + (mBuf[i+1]>>3)) & 0xFF)
							, 0xFF ^ (((mBuf[i] << 6) + (mBuf[i+1]>>2)) & 0xFF)
							, 0xFF ^ (((mBuf[i] << 7) + (mBuf[i+1]>>1)) & 0xFF)
							, i ? (float)(mStreamTimes[i] - mStreamTimes[i-1]) / mSamplesPerCell : 0
							, i && mClockBuf[i] != 0xFF ? " <!>" : ""
							);
					}
				}

				return false;
			}
		}
	}

	return true;
}

///////////////////////////////////////////////////////////////////////////

SectorParserMFM::SectorParserMFM()
	: mReadPhase(0)
	, mBitPhase(0)
{
}

void SectorParserMFM::Init(int track, const std::vector<uint32_t> *indexTimes, float samplesPerCell, TrackInfo *dstTrack) {
	mTrack = track;
	mpIndexTimes = indexTimes;
	mSamplesPerCell = samplesPerCell;
	mpDstTrack = dstTrack;
}

bool SectorParserMFM::Parse(uint32_t stream_time, uint8_t clock_bits, uint8_t data_bits) {
	if (mReadPhase < 7) {
		if (++mBitPhase == 16) {
			mBitPhase = 0;

//			if (clock_bits != 0xFF)
//				return false;

			mBuf[mReadPhase+3] = data_bits;
			++mReadPhase;

			if (mReadPhase == 7) {
				if (mBuf[3] != 0xFE)
					return false;

				if (mBuf[4] != mTrack) {
					printf("Track number mismatch! %02X != %02X\n", mBuf[4], mTrack);
					return false;
				}

				if (mBuf[5] != 0) {
					printf("Zero #1 failed!\n");
					return false;
				}

				if (mBuf[6] < 1 || mBuf[6] > 26) {
					printf("Invalid sector number %d\n", mBuf[6]);
					return false;
				}

				if (mBuf[7] >= 4) {
					printf("Sector size failed! %02X\n", mBuf[7]);
					return false;
				}

				mSectorSize = 128 << mBuf[7];

				mBuf[0] = 0xA1;
				mBuf[1] = 0xA1;
				mBuf[2] = 0xA1;
				const uint16_t computedCRC = ComputeCRC(mBuf, 8);
				const uint16_t recordedCRC = ((uint16_t)mBuf[8] << 8) + mBuf[9];

				if (computedCRC != recordedCRC) {
					printf("CRC failure on sector header: %04X != %04X\n", computedCRC, recordedCRC);
					return false;
				}

				mSector = mBuf[6];

				int vsn_time = stream_time;

				// find the nearest index mark
				auto it_index = std::upper_bound(mpIndexTimes->begin(), mpIndexTimes->end(), (uint32_t)vsn_time + 1);

				if (it_index == mpIndexTimes->begin()) {
					if (g_verbosity >= 2)
						printf("Skipping track %d, sector %d before first index mark\n", mTrack, mSector);
					return false;
				}

				if (it_index == mpIndexTimes->end()) {
					if (g_verbosity >= 2)
						printf("Skipping track %d, sector %d after last index mark\n", mTrack, mSector);
					return false;
				}

				int vsn_offset = vsn_time - *--it_index;

				mRotPos = (float)vsn_offset / (float)(it_index[1] - it_index[0]);

				if (mRotPos >= 1.0f)
					mRotPos -= 1.0f;

				if (g_verbosity >= 2)
					printf("Found track %d, sector %d at position %4.2f\n", mTrack, mSector, mRotPos);
			}
		}
	} else if (mReadPhase == 7) {
//		if (clock_bits == 0x0A && data_bits != 0xA1)
		if ((clock_bits & 0x7F) == 0x0A && data_bits == 0xA1) {
			++mReadPhase;
		}
	} else if (mReadPhase == 8 || mReadPhase == 9) {
		if ((clock_bits & 0x7F) == 0x0A) {
			if (data_bits != 0xA1) {
				mReadPhase = 7;
				return true;
			}

			++mReadPhase;
			mBitPhase = 0;
			mBuf[0] = data_bits;
			mBuf[1] = data_bits;
			mBuf[2] = data_bits;
		}
	} else if (mReadPhase == 10) {
		if (++mBitPhase == 16) {
			if (clock_bits == 0x0A && data_bits == 0xA1) {
				mBitPhase = 0;
			} else {
				if (data_bits == 0xF8 || data_bits == 0xF9 || data_bits == 0xFA || data_bits == 0xFB) {
					//printf("DAM detected (%02X)\n", data_bits);
					++mReadPhase;
					mBitPhase = 0;
					mBuf[3] = data_bits;
				} else
					return false;
			}
		}
	} else {
		if (++mBitPhase == 16) {
//			printf("Data; %02X\n", ~data_bits);
			mBuf[mReadPhase - 7] = data_bits;

			mBitPhase = 0;
			if (++mReadPhase >= 7 + mSectorSize + 6) {
				// check if the CRC is correct
				// x^16 + x^12 + x^5 + 1
				uint16_t crc = ComputeCRC(mBuf, mSectorSize + 4);

				const uint16_t recordedCRC = ((uint16_t)mBuf[mSectorSize + 4] << 8) + (uint8_t)mBuf[mSectorSize + 5];

				//printf("Read sector %d: CRCs %04X, %04X\n", mSector, crc, recordedCRC);

				// add new sector entry
				auto& tracksecs = mpDstTrack->mSectors;
				tracksecs.push_back(SectorInfo());
				SectorInfo& newsec = tracksecs.back();

				for(int i=0; i<mSectorSize; ++i)
					newsec.mData[i] = ~mBuf[i+4];

				newsec.mIndex = mSector;
				newsec.mPosition = mRotPos;
				newsec.mAddressMark = mBuf[3];
				newsec.mRecordedCRC = recordedCRC;
				newsec.mComputedCRC = crc;
				newsec.mSectorSize = mSectorSize;
				newsec.mbMFM = true;
				newsec.mWeakOffset = -1;

				if (g_verbosity >= 1)
					printf("Decoded MFM track %d, sector %d with %u bytes, DAM %02X, recorded CRC %04X (computed %04X)\n",
						mTrack,
						mSector,
						mSectorSize,
						mBuf[3],
						recordedCRC,
						crc);

				return false;
			}
		}
	}

	return true;
}
