#include "stdafx.h"
#include "serial.h"

void scp_init(const char *path) {
	serial_open(path);

	// attempt to resync the SCP
	printf("Initializing SCP device.\n");

	static const uint8_t infocmd[] = { 0x00, 0x00, 0x00, 0xD0, 0x00, 0x1A };

	serial_write(infocmd, sizeof infocmd);

	uint8_t resp[2];
	for(;;) {
		serial_read(resp, 1);

		if (resp[0] == 0x4F)
			break;
	}

	serial_read(resp, 2);
	printf("SCP version info: hardware version %u.%u, firmware version %u.%u\n", resp[0] >> 4, resp[0] & 15, resp[1] >> 4, resp[1] & 15);
}

void scp_shutdown() {
	serial_close();
}

bool scp_read_status() {
	uint8_t buf[2];
	serial_read(buf, 2);

	if (buf[1] != 0x4F) {
		fatalf("SCP command failed with status: %02X %02X\n", buf[0], buf[1]);
	}

	return true;
}

uint8_t scp_compute_checksum(const uint8_t *src, size_t len) {
	uint8_t chk = 0x4A;

	while(len--)
		chk += *src++;

	return chk;
}

bool scp_send_command(const void *buf, uint32_t len) {
	char buf2[32];

	if (len > 31)
		fatal("SCP command too long");

	// we have to write the entire command all in one go, or the SCP device
	// issues a USB timeout error

	memcpy(buf2, buf, len);
	buf2[len] = scp_compute_checksum((const uint8_t *)buf, len);
	serial_write(buf2, len+1);

	return scp_read_status();
}

bool scp_select_drive(bool driveB, bool select) {
	uint8_t selcmd[2];
	selcmd[0] = (select ? 0x80 : 0x82) + (driveB ? 1 : 0);
	selcmd[1] = 0;
	return scp_send_command(selcmd, 2);
}

bool scp_select_side(bool side2) {
	uint8_t selcmd[3];
	selcmd[0] = 0x8D;
	selcmd[1] = 0x01;
	selcmd[2] = side2 ? 1 : 0;

	return scp_send_command(selcmd, 3);
}

bool scp_select_density(bool hidensity) {
	uint8_t selcmd[3];
	selcmd[0] = 0x8C;
	selcmd[1] = 0x01;
	selcmd[2] = hidensity ? 1 : 0;

	return scp_send_command(selcmd, 3);
}

bool scp_motor(bool drive, bool enabled) {
	uint8_t cmd[2];
	cmd[0] = (enabled ? 0x84 : 0x86) + (drive ? 0x01 : 0x00);
	cmd[1] = 0;

	return scp_send_command(cmd, 2);
}

bool scp_seek0() {
	uint8_t cmd[2];
	cmd[0] = 0x88;
	cmd[1] = 0x00;

	return scp_send_command(cmd, 2);
}

bool scp_seek(int track) {
	uint8_t cmd[3];
	cmd[0] = 0x89;
	cmd[1] = 0x01;
	cmd[2] = track;

	return scp_send_command(cmd, 3);
}

bool scp_mem_read(void *data, uint32_t offset, uint32_t len) {
	while(len > 0) {
		uint32_t tc = len > 0xF000 ? 0xF000 : len;

		uint8_t cmd[11];
		cmd[0] = 0xA9;
		cmd[1] = 0x08;
		cmd[2] = (uint8_t)(offset >> 24);
		cmd[3] = (uint8_t)(offset >> 16);
		cmd[4] = (uint8_t)(offset >>  8);
		cmd[5] = (uint8_t)(offset >>  0);
		cmd[6] = (uint8_t)(tc >> 24);
		cmd[7] = (uint8_t)(tc >> 16);
		cmd[8] = (uint8_t)(tc >>  8);
		cmd[9] = (uint8_t)(tc >>  0);
		cmd[10] = scp_compute_checksum(cmd, 10);
		serial_write(cmd, 11);
		serial_read(data, tc);

		bool r = scp_read_status();
		if (!r)
			return false;

		data = (char *)data + tc;
		len -= tc;
		offset += tc;
	}

	return true;
}

bool scp_mem_write(const void *data, uint32_t offset, uint32_t len) {
	while(len > 0) {
		uint32_t tc = len > 0xF000 ? 0xF000 : len;

		uint8_t cmd[11];
		cmd[0] = 0xAA;
		cmd[1] = 0x08;
		cmd[2] = (uint8_t)(offset >> 24);
		cmd[3] = (uint8_t)(offset >> 16);
		cmd[4] = (uint8_t)(offset >>  8);
		cmd[5] = (uint8_t)(offset >>  0);
		cmd[6] = (uint8_t)(tc >> 24);
		cmd[7] = (uint8_t)(tc >> 16);
		cmd[8] = (uint8_t)(tc >>  8);
		cmd[9] = (uint8_t)(tc >>  0);
		cmd[10] = 0x4A + cmd[0] + cmd[1] + cmd[2] + cmd[3] + cmd[4] + cmd[5] + cmd[6] + cmd[7] + cmd[8] + cmd[9];
		serial_write(cmd, 11);
		serial_write(data, tc);

		bool r = scp_read_status();
		if (!r)
			return false;

		data = (const char *)data + tc;
		len -= tc;
		offset += tc;
	}

	return true;
}

bool scp_erase(bool rpm360) {
	uint8_t tmpbuf[204];
	memset(tmpbuf, 0, sizeof tmpbuf);
	scp_mem_write(tmpbuf, 0, 204);

	uint8_t cmd[8];
	cmd[0] = 0xA2;
	cmd[1] = 0x05;
	cmd[2] = 0;
	cmd[3] = 0;
	cmd[4] = 0;
	cmd[5] = 102;
	cmd[6] = rpm360 ? 0x0D : 0x05;
	cmd[7] = 0x4A + cmd[0] + cmd[1] + cmd[2] + cmd[3] + cmd[4] + cmd[5] + cmd[6];
	serial_write(cmd, 8);
	return scp_read_status();
}

bool scp_track_read(bool rpm360, uint8_t revs) {
	uint8_t cmd[4];
	cmd[0] = 0xA0;
	cmd[1] = 0x02;
	cmd[2] = revs;
	cmd[3] = rpm360 ? 0x09 : 0x01;
	return scp_send_command(cmd, 4);
}

bool scp_track_getreadinfo(uint32_t data[10]) {
	uint8_t buf[40];
	buf[0] = 0xA1;
	buf[1] = 0x00;
	buf[2] = 0xEB;
	serial_write(buf, 3);
	bool status = scp_read_status();
	serial_read(buf, 40);

	for(int i=0; i<10; ++i) {
		data[i] = ((uint32_t)buf[i*4+0] << 24)
			+ ((uint32_t)buf[i*4+1] << 16)
			+ ((uint32_t)buf[i*4+2] <<  8)
			+ ((uint32_t)buf[i*4+3] <<  0);
	}

	return status;
}

bool scp_track_write(bool rpm360, uint32_t bitCellCount) {
	uint8_t cmd[7];
	cmd[0] = 0xA2;
	cmd[1] = 0x05;
	cmd[2] = (uint8_t)(bitCellCount >> 24);
	cmd[3] = (uint8_t)(bitCellCount >> 16);
	cmd[4] = (uint8_t)(bitCellCount >>  8);
	cmd[5] = (uint8_t)(bitCellCount >>  0);
	cmd[6] = rpm360 ? 0x0D : 0x05;
	return scp_send_command(cmd, 7);
}
