#include "stdafx.h"

void write_apple2_dsk(const char *path, DiskInfo& disk, int track) {
	uint32_t sector_size = 256;
	uint32_t sectors_per_track = 16;

	printf("Writing Apple II disk image (DOS 3.3 ordering): %s\n", path);

	FILE *fo = fopen(path, "wb");
	if (!fo)
		fatalf("Unable to open output file: %s.\n", path);

	// write tracks
	char secbuf[256];
	bool seenMissingWarning = false;
	uint32_t missingSectors = 0;
	uint32_t badSectors = 0;

	for(int i=0; i<35; ++i) {
		TrackInfo& track_info = disk.mTracks[i];

		// sort and sift out usable sectors from track
		std::vector<SectorInfo *> secptrs;
		sift_sectors(track_info, i, secptrs);

		// iterate over sectors
		const SectorInfo *secptrs2[16] = { 0 };

		for(auto it = secptrs.begin(), itEnd = secptrs.end();
			it != itEnd;
			++it)
		{
			const SectorInfo *secptr = *it;

			if (secptr->mIndex >= 0 && secptr->mIndex < (int)sectors_per_track) {
				if (secptrs2[secptr->mIndex])
					printf("WARNING: Variable sectors not supported by DSK/DO format. Discarding duplicate physical sector for track %d, sector %d.\n", i, secptr->mIndex);
				else
					secptrs2[secptr->mIndex] = secptr;
			}
		};

		// write out sectors
		static const int kLogicalToPhysical[16]={
			0, 13, 11, 9, 7, 5, 3, 1, 14, 12, 10, 8, 6, 4, 2, 15
		};

		uint32_t missingSectorMask = 0;

		for(uint32_t j=0; j<sectors_per_track; ++j) {
			int physec = kLogicalToPhysical[j];
			const SectorInfo *sec = secptrs2[physec];

			if (!sec) {
				if (track < 0 || track == i) {
					++missingSectors;
					missingSectorMask |= (1 << physec);
				}

				memset(secbuf, 0, sizeof secbuf);
			} else {
				memcpy(secbuf, sec->mData, sector_size);

				if (sec->mSectorSize != sector_size) {
					printf("WARNING: Variable sector size not supported by DSK format. Writing out truncated data for track %d, sector %d.\n", i, physec+1);
					++badSectors;
				} else if (sec->mRecordedCRC != sec->mComputedCRC) {
					printf("WARNING: CRC error encoding not supported by DSK format. Ignoring CRC error for track %d, sector %d.\n", i, physec+1);
					++badSectors;
				} else if (sec->mWeakOffset >= 0) {
					printf("WARNING: Weak sector encoding not supported by DSK format. Ignoring error for track %d, sector %d.\n", i, physec+1);
					++badSectors;
				}
			}

			fwrite(secbuf, sector_size, 1, fo);
		}

		if (missingSectorMask) {
			if (!seenMissingWarning) {
				seenMissingWarning = true;
				printf("WARNING: Missing sectors not supported by DSK/DO format. Writing out null data.\n");
			}

			if (missingSectorMask == (1 << sectors_per_track) - 1)
				printf("WARNING: No sectors found on track %u.\n", i);
			else {
				printf("WARNING: Track %u: missing sectors:", i);

				for(uint32_t j=0; j<sectors_per_track; ++j) {
					if (missingSectorMask & (1 << j))
						printf(" %u", j+1);
				}

				printf("\n");
			}
		}
	}

	fclose(fo);

	printf("%d missing sector%s, %d sector%s with errors\n"
		, missingSectors, missingSectors == 1 ? "" : "s"
		, badSectors, badSectors == 1 ? "" : "s");
}

void write_apple2_nib(const char *path, const DiskInfo& disk, int track) {
	printf("Writing Apple II nibble disk image: %s\n", path);

	if (disk.mTracks[0].mGCRData.size() < 0x1A00)
		fatalf("No GCR data present. Apple II GCR decoding must be used for nibble output.\n");

	FILE *fo = fopen(path, "wb");
	if (!fo)
		fatalf("Unable to open output file: %s.\n", path);

	// write tracks
	char nibbuf[0x1A00];

	for(int i=0; i<35; ++i) {
		const auto& track_data = disk.mTracks[i];
		const uint8_t *p = track_data.mGCRData.data();
		int track_offset = 0;
		int track_len = (int)track_data.mGCRData.size();
		int max_offset = track_len - 0x1A00;

		if (max_offset <= 0) {
			if (max_offset < 0)
				printf("WARNING: Track %u is short (<$1A00 bytes) and will be padded.\n", track);

			max_offset = 0;
			memcpy(nibbuf, p, track_len);
			memset(nibbuf + track_len, 0, 0x1A00 - track_len);
		} else {
			int best_len = 0;
			int sync_count = 0;


			for(int i=0; i<=max_offset; ++i) {
				if (p[i] == 0xFF)
					++sync_count;
				else {
					if (sync_count > best_len) {
						best_len = sync_count;
						track_offset = i - sync_count/2;
					}

					sync_count = 0;
				}
			}

			if (sync_count > best_len) {
				best_len = sync_count;
				track_offset = max_offset - sync_count/2;
			}

			memcpy(nibbuf, p + track_offset, 0x1A00);
		}

		fwrite(nibbuf, 0x1A00, 1, fo);
	}

	fclose(fo);
}
