/*

--------------------------------------------------------------------------------------------

                                                                  July 23, 2009

      Pokey Glider 2.0

--------------------------------------------------------------------------------------------

    8-bit ATARI computer code written in C which uses the POKEY sound chip to
  implement a gliding synthesizer.

author:
Bart Jaszcz
bpj1138@yahoo.com

*/

#include <ctype.h>
#include <conio.h>
#include <string.h>
#include <stdio.h>

// to be or not to be
#define TRUE 1
#define FALSE 0

// shorten unsigned names
typedef unsigned char		byte;
typedef unsigned int		uint;
typedef unsigned long		ulong;

// program screen
char *text[24] = {
	"",
		"    P o k e y     ver   Keyboard Notes ",
		"   G l i d e r    2.0    s d   g h j",
		" _____________________  z x c v b n m",
		"",
		" | CLOCK       |  -0-",
		" | VOLUME      |  q-w",
		" | ATTACK      |  e-r",
		" | RELESE      |  t-y",
		" |  STEPS      |  u-i",
		" |  SPEED      |  o-p",
		" | CH-MOD      |  -/=",
		" | DETUNE      |  k-l",
		" | FILTER      |  -f-",
		" | OCTAVE      |  1-9",
		" |  ASYNC      |  -a-",
		" | CH-MIX      |  ;-+",
		" | LENGTH      |  /-*",
		" |  TRANS      |  ,-.",
		" |  TEMPO      |  <->",
		" |  STATE      |  C-#",
		" |  NOISE      |  ESC",
		"                       |MODE| |NOTE|  |",
};

// help screen
char *help[16] = {
		"                ",
		"   User Help:   ",
		" play/stop  RET ",
		"  del note  SPC ",
		" read/write TAB ",
		"   clear   BKSP ",
		"                ",
		" Control Keys:  ", 
		"   load  C-l    ",
		"   save  C-s    ",
		"   copy  C-c    ",
		"  paste  C-v    ",
		"   exit  C-q    ",
		"                ",
		"(c) Bart Jaszcz ",
		"                "
};

// screen related defines
#define VARCOL 10            // variable display column
#define SEQCOL 23            // sequencer column
#define SEQROW 5             // sequencer row
#define MODCOL 6             // sequencer mode display column offset

// musical note names
char *note_name[13] = {
	"C ", "C#", "D ", "D#", "E ", "F ", "F#", "G ", "G#", "A ", "A#", "B ", "  "
};

// other musical defines
#define SILENCE 12
#define INVALID 13

// must deal with negative numbers using bias
#define DETUNE_BIAS 48
#define TRANS_BIAS 11

// muscial note frequency table (octaves 1-9)
uint note_hz[9][12] = {
	{ 32, 34, 36, 38, 41, 43, 46, 48, 51, 55, 58, 61 },
	{ 65, 69, 73, 77, 82, 87, 92, 97, 103, 110, 116, 123 },
	{ 130, 138, 146, 155, 164, 174, 184, 195, 207, 220, 233, 246 },
	{ 261, 277, 293, 311, 329, 349, 369, 391, 415, 440, 466, 493 },
	{ 523, 554, 587, 622, 659, 698, 739, 783, 830, 880, 932, 987 },
	{ 1046, 1108, 1174, 1244, 1318, 1396, 1479, 1567, 1661, 1760, 1864, 1975 },
	{ 2093, 2217, 2349, 2489, 2637, 2793, 2959, 3135, 3322, 3520, 3729, 3951 },
	{ 4186, 4434, 4698, 4978, 5274, 5587, 5919, 6271, 6644, 7040, 7458, 7902 },
	{ 8372, 8869, 9397, 9956, 10548, 11175, 11839, 12543, 13289, 14080, 14917, 15804 }
};

// attract mode timer (that's a screen saver for the non-Atarians)
#define ATRACT ((byte*)0x4d)

// disable ATARI OS keyboard handler flag location
#define KEYDIS ((byte*)0x26d)

// pokey keyboard registers (POKEY also handles IO with keyboard & serial port)
#define KBCODE ((byte*) 0xd209)              // last key read
#define KBREST ((byte*) 0xd20a)              // reset keyboard
#define SKSTAT ((byte*) 0xd20f)              // keyboard status register

// pokey keyboard flags
#define KB_OVERRUN (1<<5)
#define KB_PRESSED (1<<2)

// pokey audio registers
#define AUDCTL ((byte*)0xd208)        // master control register
#define AUD ((byte*)0xd200)      // start of two 16 bit frequency registers

// extra audio control register flags
#define JOIN_CH (3<<3)
#define HIGHPASS_2_4 (1<<1)

// random number generator (Yet another POKEY function!)
#define RANDOM ((byte*)0xd20a)

// channel defines
#define CH1 0
#define CH2 1

// clock configurations
#define CLK_17_17 0
#define CLK_17_64 1
#define CLK_64_17 2
#define CLK_64_64 3
#define CLK_15_15 4
#define CLK_17_15 5
#define CLK_15_17 6

// max values
#define MAX_VOL 8
#define MAX_STEPS 64
#define MAX_ATT 100
#define MAX_REL 200
#define MAX_SPEED 16
#define KEY_RATE 63
#define MAX_LEN 128
#define BASE_TEMPO 3
#define MAX_TEMPO 250
#define MAX_STATES 11

// filename editing
#define NAMROW SEQROW+5
#define NAMCOL SEQCOL+2
#define MAX_CHARS 11

// filename status
#define FILE_ABORT 0
#define DRIVE_SPEC_ERROR 1
#define NULL_NAME 2
#define NAME_TOO_LONG 3
#define NAME_OK 4

// filename edit signals
#define ENTER_CHAR 1
#define ERASE_CHAR 2

// shift and control key constants
#define SHIFT 64
#define CTRL 128

// display control character constant
#define DISP_CTRL 192

// clipboard index in state array
#define CLPBRD MAX_STATES-1

// define synth state variables (so we can shadow them in global space)
#define STATE_VARS                     \
byte used;                             \
byte tempo;                            \
byte octave;                           \
byte clock_select;                     \
byte clock_flags;                      \
ulong clock[2];                        \
byte volume;                           \
byte level;                            \
byte attack;                           \
byte release;                          \
byte mode[2];                          \
byte async;                            \
byte trans;                            \
byte detune;                           \
byte steps;                            \
byte speed;                            \
byte mix;                              \
byte extra_flags;                      \
byte noise;                            \
byte size;                             \
byte pattern[MAX_LEN];                 

// define the structure containing the synth state variables above
typedef struct {
    STATE_VARS
}
state_t;

// allocate one synth state in global space (current state)
STATE_VARS

// allocate synth state array (stored states)
state_t state[MAX_STATES];

// allocate additional state variables
byte select;                           // state array selector
byte position;                         // current position in pattern
byte seq;                              // sequencer mode
byte len;                              // current pattern length
byte play;                             // if sequencer is turned on
byte rec;                              // if writinig to pattern
byte key_timer;                        // keyboard debounce timer
byte prev_len;                         // previous pattern length
ulong interation;                      // synth interation counter
byte note_on;                          // if a note is being played
byte name_file;                        // if getting filename from user
byte save_flag;                        // if saving file
byte wait_key;                         // wait for user to press a key
byte run;                              // exit flag
char name[MAX_CHARS+1];                // user specified filename
char magic[] = "PG20";                 // pokey glider 2.0 file magic
char ext[] = "PG2";                    // pokey glider 2.0 file extension

// prototype our functions
byte get_note(byte);
void set_octave(byte);
void set_clock(byte);
void set_audctl(void);
void set_volume(byte);
void set_attack(byte);
void set_release(byte);
void set_mode(byte,byte);
void set_async(byte);
void set_detune(byte);
void set_trans(byte);
void set_steps(byte);
void set_speed(byte);
void set_filter(byte);
void set_mix(byte);
void set_length(byte);
void set_tempo(byte);
void set_noise(byte);
void redraw_seq(void);
void set_seq(void);
void set_cursor(byte);
void init_state(void);
void select_state(byte);
void display_help(void);
void clear_display(void);
byte save(void);
byte load(void);
byte check_name(void);
char __fastcall__ name_char(byte);

// let the fun begin
void main(void)
{
	byte status, key, key_on;          // keyboard state
	byte note;                         // musical note
	byte start_note;                   // request note to be played
	uint freq1, dest1;                 // frequency, destination frequency (osc 1)
	uint freq2, dest2;                 // frequency, destination frequency (osc 2)
	uint step1, step2;                 // frequency step
	byte sign1, sign2;                 // frequency step sign
	byte m, l;                         // mode, level
	byte i, j;                         // integer
	byte p;           				   // new cursor position
	byte n;   						   // note
	byte step_timer, attack_timer, release_timer, speed_timer; // timers for synth

	// show display screen
	clrscr();
	for (i = 0; i < 24; i++)
		cputsxy(0,i,text[i]);
	// draw lines on screen
	chlinexy(1,0,38);
	chlinexy(1,23,38);
	cvlinexy(0,1,22);
	cvlinexy(39,1,22);
	cvlinexy(22,1,22);
	chlinexy(23,4,16);
	chlinexy(23,21,16);
	chlinexy(2,4,13);
	chlinexy(2,22,13);
	// draw line intersection chars
	cputcxy(0,0,DISP_CTRL+'Q');
	cputcxy(39,0,DISP_CTRL+'E');
	cputcxy(0,23,DISP_CTRL+'Z');
	cputcxy(39,23,DISP_CTRL+'C');
	cputcxy(22,0,DISP_CTRL+'W');
	cputcxy(22,23,DISP_CTRL+'X');
	// disable keyboard handler
	*KEYDIS = 0xff;
	// init synth state
	run = TRUE;                      // init exit flag
	start_note = FALSE;              // don't start note
	note_on = FALSE;                 // note off
	key_on = FALSE;                  // key off
	key = 0;                         // init key scancode
	note = INVALID;                  // init note to be played to none
	level = 0;                       // current volume level
	interation = 0;                  // init interation counter
	key_timer = 0;                   // init key rate timer
	position = 0;                    // init position within pattern
	extra_flags = JOIN_CH;           // use two channels
	position = (MAX_LEN - 1);	     // init position
	// init user dialog
	name[0] = NULL;
	name_file = FALSE;
	save_flag = FALSE;
	wait_key = FALSE;
	// mark all states as unused
	for (i = 0; i < MAX_STATES; i++)
		state[i].used = FALSE;
	// force space fill
	prev_len = MAX_LEN;
	// select and init first state
	select = MAX_STATES;           // signal there was no previous state
	select_state(0);
	// turn sequencer on, write to pattern
	play = TRUE;
	rec = TRUE;
	set_seq();
	// init ouput
	freq1 = (uint) ((clock[CH1] / note_hz[octave][0]) >> 1);
	freq2 = (uint) ((clock[CH2] / note_hz[octave][0]) >> 1);
	// synth loop
	while (run)
	{
		// read keyboard
		status = *SKSTAT;                  // read keyboard status register
		if (status & KB_OVERRUN)             // reset keyboard if overrun detected
			*KBREST = TRUE;
		if (!(status & KB_PRESSED))  // if key is being pressed (status bit == 0)
		{
			if (key_on)               // was a key on before?
			{
				key = *KBCODE;          // yes, read 2nd key code
				n = get_note(key);         // get note from key
				if (n != INVALID) {
					// if valid note
					if (n != note) {
						// if new note
						note = n;
						start_note = TRUE;
					}
				}
				else
					key_on = FALSE;
			}
			else         // no, all keys were off..
			{
				key = *KBCODE;            // read key code
				note = get_note(key);         // get note from key
				if (note != INVALID)         // if valid note
				{
					start_note = TRUE;
					key_on = TRUE;       // set key on flag
				}
				else
					key_on = FALSE;
			}
		}
		else              // key is not being pressed
			if (key_on)          // if flag is set
				key_on = FALSE;     // clear flag
		// if writing to pattern and valid note
		if (rec && key_on && (note != INVALID)) {
			pattern[position] = note;
			set_cursor(position);
		}
		// don't start silent notes
		if (start_note && (note == SILENCE))
		{
			start_note = FALSE;
			key_on = FALSE;
		}
		// if playing pattern
		if (play) {
			// update cursor position
			p = (byte) ((interation / (BASE_TEMPO + MAX_TEMPO - tempo)) &
						((MAX_LEN-1) >> size)
						);
			n = pattern[p];  // note from pattern
			if (p != position)
			{
				set_cursor(p);
				if (n != SILENCE && !key_on)
				{
					note = n;
					start_note = TRUE;
				}
			}
			if (n != SILENCE)
				if (!key_on)
					key_on = TRUE;
		}
		// does a note need to be started?
		if (start_note)
		{
			// compute transposed note
			j = octave * 12 + note + trans;
			// check bounds of transposed note
			if (j < TRANS_BIAS)
				j = 0;
			else
				j -= TRANS_BIAS;
			n = j % 12;                      // save transposed note n
			// set osc 1 destination
			dest1 = (uint) ((clock[CH1] / note_hz[j/12][n]) >> 1);
			// detune second osc
			i = j + detune;
			// check bounds of detuned note
			if (i < DETUNE_BIAS)
				i = 0;
			else
				i -= DETUNE_BIAS;
			dest2 = (uint) ((clock[CH2] / note_hz[i/12][i%12]) >> 1);
			// set steps
			if (dest1 >= freq1) {
				step1 = (dest1 - freq1) / steps;
				sign1 = TRUE;
			}
			else {
				step1 = (freq1 - dest1) / steps;
				sign1 = FALSE;
			}
			if (dest2 >= freq2) {
				step2 = (dest2 - freq2) / steps;
				sign2 = TRUE;
			}
			else {
				step2 = (freq2 - dest2) / steps;
				sign2 = FALSE;
			}
			// set timers
		    step_timer = steps;
			attack_timer = attack;
			release_timer = release;
			speed_timer = speed;
			// print out note being started including transposition
			cputsxy(36,22,note_name[n]);          // retrive transposed note n
			// acknowledge note was started
			start_note = FALSE;
			note_on = TRUE;
		}
		// handle note glide and level attack/release
		if (note_on)
		{
			if (step_timer && !speed_timer--)  // wait for speed timer to expire
			{
				speed_timer = speed;           // set timer again
				--step_timer;                  // decrement step timer
				// CH1
				if (sign1)   // positive step?
					freq1 += step1;            // yes, increase frequency
				else
					freq1 -= step1;            // no, decrease frequency
				if (sign2)   // CH2
					freq2 += step2;
				else
					freq2 -= step2;
			}
			if (key_on)  // note attack
			{
				if (level < volume) {
					// if haven't reached full volume level yet
					if (!attack_timer--) {
						attack_timer = attack;           // set timer again
						++level;
					}
				}
			}
			else  // release note
			{
				if (!release_timer--) {
					release_timer = release;           // set timer again
					// decrease vol every Nth interation
					if (level > 0)                 // N == release
						--level;
					else
					{
						// note released
						note_on = FALSE;
						cputsxy(36,22,"  ");         // clear note display
					}
				}
			}
		}
		else               // force zero level when note is off
			level = 0;
		// set audio level and distorion mode for each channel
		m = mode[0] << 5;            // shift mode through volume bits
		l = m | ((level * mix) / MAX_VOL);                  // set volume bits
		AUD[1] = m;
		AUD[3] = l;
		m = mode[1] << 5;            // shift mode through volume bits
		l = m | ((level * (MAX_VOL - mix)) / MAX_VOL);       // set volume bits
		AUD[5] = m;
		AUD[7] = l;
		// add frequency noise
		n = 0xff >> (8 - noise);
		i = *RANDOM & n;
		j = *RANDOM & n;
		dest1 = freq1 + i;
		dest2 = freq2 + j;
		// output frequency to POKEY
		AUD[0] = (byte) (dest1 & 0xff);
		AUD[2] = (byte) (dest1 >> 8);
		AUD[4] = (byte) (dest2 & 0xff);
		AUD[6] = (byte) (dest2 >> 8);
		// if not zero, decrement key timer
		if (key_timer)
			--key_timer;
		// sleep for short random interval (this will strech glide/attack/release)
		i = *RANDOM >> (async + 2);
		while (i--)		// reset attract mode timer while in sleep loop
			*ATRACT = 0;
		// next synth interation
		++interation;
	}
	// enable keyboard handler
	*KEYDIS = 0;
}

void init_state(void)     // init one pattern / settings
{
	int i;      // note index
	for (i = 0; i < MAX_LEN;i++)   // silence pattern
		pattern[i] = SILENCE;
	set_length(3);                 // shift max length right by 3
	set_tempo(3*MAX_TEMPO/4);      // 3/4 of max tempo
	set_octave(2);                 // octave 3 if counting from 1
	set_volume(8);                 // max volume level = 8 (max)
	set_attack(3);                 // set attack = 3 (steps)
	set_release(31);               // set release = 31 (steps)
	set_mode(7, 7);                // set pure tone for both channels
	set_async(4);                  // set random tempo change
	set_detune(DETUNE_BIAS);       // set second osc offset
	set_steps(3);                  // glide steps
	set_speed(1);                  // glide speed
	set_filter(FALSE);             // filter off
	set_clock(CLK_17_17);          // set clock to both channels at 1.79 MHz
	set_mix(MAX_VOL/2);            // set 1/2-1/2 mix level
	set_trans(TRANS_BIAS);         // no transpose
	set_noise(5);                  // set moderate frequency noise
	gotoxy(VARCOL,20);             // update state index display
	cprintf("%4d", select+1);      // count from zero
	used = TRUE;                   // mark state as used
}

void select_state(byte index)  // synth state selection
{
	byte *m;               // clock mode
	state_t *s;            // state pointer (destination)
	if (index == select)   // user selected the current state
		return;
	// kill sound kill note
	AUD[3] = 0;
	AUD[7] = 0;
	note_on = FALSE;
	// copy current state to state array only if select is valid!
	if (select < MAX_STATES)
	{
		s = &state[select];    // point at current state in state array
		memcpy(s, &used, sizeof(state_t));  // save current state
	}
	// point at requested state in state array
	s = &state[index];
	if (!s->used)          // if unused state needs init
		init_state();                 // do the init in global space
	else   // copy stored state to current state
	{                                     // instead of memcpy
		set_length(s->size);            // use the 'set' methods
		set_tempo(s->tempo);              // which will update the display
		set_octave(s->octave);            // as well as set the state
		set_volume(s->volume);
		set_attack(s->attack);
		set_release(s->release);
		set_async(s->async);
		set_detune(s->detune);
		set_steps(s->steps);
		set_speed(s->speed);
		set_filter(s->extra_flags & HIGHPASS_2_4);
		set_clock(s->clock_select);
		set_mix(s->mix);
		set_trans(s->trans);
		set_noise(s->noise);
		m = s->mode;
		set_mode(m[0], m[1]);
		memcpy(pattern, s->pattern, MAX_LEN);
	}
	if (index != CLPBRD)      // if not dealing with clipboard
	{
		select = index;       // set new state index
		gotoxy(VARCOL,20);    // update state index display (count from zero)
		cprintf("%4d", select+1);
	}
	redraw_seq();             // draw new pattern
}

/*

key map (one octave)
                                                                             
   0   1   2   3   4   5   6   7   8   9   10  11          <--  note #       
   |   |   |   |   |   |   |   |   |   |   |   |                             
   C   C#  D   D#  E   F   F#  G   G#  A   A#  B           <--  note name    
   |   |   |   |   |   |   |   |   |   |   |   |                             
   z   s   x   d   c   v   g   b   h   n   j   m           <--  key          
   |   |   |   |   |   |   |   |   |   |   |   |                             
   23  62  22  58  18  16  61  21  57  35  1   37          <--  key scancode 
                                                                             

*/

byte get_note(byte key)
{
	char c;                              // character
	byte i;                              // counter
	byte x, y;                           // 2D cursor position

	// since the scancodes aren't in any particular order, it's best to construct
	// a jump table for each key which returns musical note # for a specific key
	// or INVALID if the user is pressing a key that doesn't have an note assigned
	//
	//	gotoxy(0,0);
	//	cprintf("%d   ", (int) key);
	
	if (wait_key)    // dismiss file/help dialog
	{
		if (!key_timer)        // wait for debounce
		{
			key_timer = KEY_RATE;        // set debounce
			wait_key = FALSE;            // clear wait for key flag
			prev_len = MAX_LEN;          // force sequencer update
			if (name_file)     // if dismissing file operation dialog
			{
				name_file = FALSE;       // clear file operation flag
				if (save_flag)           // if file was saved
				{
					prev_len = MAX_LEN;          // force sequencer update
					redraw_seq();
				}
				else                     // if file was loaded
				{
					select = MAX_STATES; // don't copy current state to state array
					select_state(0);     // select state zero
				}
			}
			else   // otherwise it was the help screen
				redraw_seq();
			name[0] = NULL;          // always clear file name
		}
		return INVALID;         // not a note
	}

	if (name_file)    // handle filename dialog
	{
		if (!key_timer)                // debounce
		{
			c = (char) toupper((int) name_char(key)); // convert scancode to char
			if (c != NULL)         // if allowed character
			{
				i = strlen(name);  // get length of name
				if (c == ENTER_CHAR)        // user finished entering name
				{
					gotoxy(NAMCOL+i, NAMROW);   // erase cursor
					revers(FALSE);
					cputc(' ');
					// check filename
					i = check_name();
					if (i == NAME_OK)         // if valid filename
					{
						gotoxy(NAMCOL+2, NAMROW+3);
						if (save_flag)            // print out operation
							cputs("WRITING");
						else
							cputs("READING");
						revers(TRUE);
						gotoxy(NAMCOL, NAMROW+5); // go to progress bar location
						if (save_flag)    // call file operation
							i = save();
						else
							i = load();
						revers(FALSE);
						gotoxy(NAMCOL+2, NAMROW+7);
						if (i)               // print out result
							cputs("SUCCESS");
						else
						{
							cputs("FAILURE");
							name_file = FALSE;     // cancel file operation
							save_flag = FALSE;
						}
					}
					else          // error in filename
					{
						gotoxy(NAMCOL, NAMROW+3);
						switch (i)              // print out error
						{
						case FILE_ABORT:
							cputs("  ABORTED");
							break;
						case DRIVE_SPEC_ERROR:
							cputs(" BAD DRIVE");
							break;
						case NULL_NAME:
							cputs("MISSING NAME");
							break;
						case NAME_TOO_LONG:
							cputs("SHORTEN NAME");
							break;
						}
						name_file = FALSE;     // cancel file operation
					}
					wait_key = TRUE;          // wait for user
					key_timer = KEY_RATE;
					return INVALID;            // not a note!
				}
				if (c == ERASE_CHAR)    // bksp
				{
					if (i != 0)           // if not zero chars!
						name[--i] = NULL;       // erase last char
				}
				else                      // allowed name char
				{
					if (i < MAX_CHARS)       // if less than max letters in name
					{
						name[i++] = c;          // append typed key
						name[i] = NULL;
					}
				}
				gotoxy(NAMCOL, NAMROW);           // goto filename location
				revers(FALSE);              // draw filename
				cputs(name);
				revers(TRUE);          // draw cursor
				cputc(' ');
				if (i < MAX_CHARS)       // if name not full
				{
					revers(FALSE);          // add blank space at the end
					cputc(' ');
				}
			}
			key_timer = KEY_RATE;            // set key timer
		}
		return INVALID;               // not a note
	}
		
	// sequencer control
	switch (key) {

	case CTRL+47: // ctrl-q                            quit program
		run = FALSE;
		break;
		
		// handle clipboard
	case CTRL+18: // ctrl-c                      copy current state to clipboard
		memcpy(&state[CLPBRD], &used, sizeof(state_t));
		break;
	case CTRL+16: // ctrl-v                     paste clipboard to current state
		if (state[CLPBRD].used)           // don't paste uninitialized clipboard
			select_state(CLPBRD);
		break;

		// handle file operations
	case CTRL+62: // ctrl-s                               save file
	case CTRL+0:  // ctrl-l                               load file
		play = FALSE;               // stop sequencer
		set_seq();
		name_file = TRUE;             // get filename
		clear_display();
		gotoxy(NAMCOL+1, NAMROW-2);
		if (key == CTRL+0) {
			save_flag = FALSE;             // save operation
			cputs("LOAD FILE");
		}
		else {
			save_flag = TRUE;             // save operation
			cputs("SAVE FILE");
		}	
		// draw filename box
		chlinexy(NAMCOL,NAMROW-1,MAX_CHARS+1);
		chlinexy(NAMCOL,NAMROW+1,MAX_CHARS+1);
		cvlinexy(NAMCOL-1,NAMROW,1);
		cvlinexy(NAMCOL+MAX_CHARS+1,NAMROW,1);
		cputcxy(NAMCOL-1,NAMROW-1,DISP_CTRL+'Q');
		cputcxy(NAMCOL+MAX_CHARS+1,NAMROW-1,DISP_CTRL+'E');
		cputcxy(NAMCOL-1,NAMROW+1,DISP_CTRL+'Z');
		cputcxy(NAMCOL+MAX_CHARS+1,NAMROW+1,DISP_CTRL+'C');
		revers(TRUE);                // draw cursor
		gotoxy(NAMCOL, NAMROW);
		cputc(' ');
		revers(FALSE);
		break;

	case 17:    // HELP
		// handle user help 
		play = FALSE;               // otherwise display help
		set_seq();
		display_help();
		wait_key = TRUE;            // wait for key immediately
		break;
	case 12:  // RETURN
		if (!key_timer) {             // play/stop
			play = !play;
			set_seq();
		}
		break;
	case 44:  // TAB
		if (!key_timer)
		{
			rec = !rec;           // read/write to pattern
			set_seq();
		}
		break;

		// clear pattern
	case 52:  // BKSPC
		for (i = 0; i < MAX_LEN; i++)         // clear pattern
			pattern[i] = SILENCE;
		redraw_seq();
		break;

		// handle filter
	case 56:  // f
		if (!key_timer)
			set_filter(extra_flags & HIGHPASS_2_4 ? FALSE : TRUE);
		break;

		// handle random shift
	case 63: // a
		if (!key_timer)
			set_async((async + 4) % 5);
		break;

		// handle distortion mode changes
	case 14: // -
		if (!key_timer)
			set_mode((mode[0] + 1) % 8, mode[1]);
		break;
	case 15: // =
		if (!key_timer)
			set_mode(mode[0], (mode[1] + 1) % 8);
		break;

		// handle detune
	case 5: // k
		if (!key_timer)
			if (detune != 0)
				set_detune(detune - 1);
		break;
	case 0: // l
		if (!key_timer)
			if (detune < DETUNE_BIAS*2)
				set_detune(detune + 1);
		break;

		// handle clock changes
	case 50: // 0
		if (!key_timer)
			set_clock(++clock_select % 7);
		break;

		// handle volume changes
	case 47: // q
		if (!key_timer)
			if (volume > 1)
				set_volume(volume - 1);
		break;
	case 46: // w
		if (!key_timer)
			if (volume < MAX_VOL)
				set_volume(volume + 1);
		break;

		// handle attack changes
	case 42: // e
		if (!key_timer)
			if (attack > 1)
				set_attack(attack - 1);
		break;
	case 40: // r
		if (!key_timer)
			if (attack < MAX_ATT)
				set_attack(attack + 1);
		break;

		// handle release changes
	case 45: // t
		if (!key_timer)
			if (release > 1)
				set_release(release - 1);
		break;
	case 43: // y
		if (!key_timer)
			if (release < MAX_REL)
				set_release(release + 1);
		break;

		// handle step changes
	case 11: // u
		if (!key_timer)
			if (steps > 1)
				set_steps(steps - 1);
		break;
	case 13: // i
		if (!key_timer)
			if (steps < MAX_STEPS)
				set_steps(steps + 1);
		break;

		// handle speed changes
	case 10: // p
		if (!key_timer)
			if (speed > 1)
				set_speed(speed - 1);
		break;
	case 8: // o
		if (!key_timer)
			if (speed < MAX_SPEED)
				set_speed(speed + 1);
		break;

		// handle mix changes
	case 6:  // +
		if (!key_timer)
			if (mix != 0)
				set_mix(mix - 1);
		break;
	case 2:  // ;
		if (!key_timer)
			if (mix < MAX_VOL)
				set_mix(mix + 1);
		break;

		// handle transpose changes
	case 32: // ,
		if (!key_timer)
			if (trans != 0)
				set_trans(trans - 1);
		break;
	case 34: // .
		if (!key_timer)
			if (trans < TRANS_BIAS*2)
				set_trans(trans + 1);
		break;

		// handle length changes
	case 7: // *
		if (!key_timer)
			if (size > 0)  // these are shift values
				set_length(size - 1);
		break;
	case 38: // /
		if (!key_timer)
			if (size < 4)
				set_length(size + 1);
		break;

		// handle tempo changes
	case 54: // <
		if (!key_timer)
			if (tempo > 1)
				set_tempo(tempo - 1);
		break;
	case 55: // >
		if (!key_timer)
			if (tempo < MAX_TEMPO)
				set_tempo(tempo + 1);
		break;

		// handle octave changes
	case 31: // 1
		set_octave(0);
		break;
	case 30: // 2
		set_octave(1);
		break;
	case 26: // 3
		set_octave(2);
		break;
	case 24: // 4
		set_octave(3);
		break;
	case 29: // 5
		set_octave(4);
		break;
	case 27: // 6
		set_octave(5);
		break;
	case 51: // 7
		set_octave(6);
		break;
	case 53: // 8
		set_octave(7);
		break;
	case 48: // 9
		set_octave(8);
		break;

		// handle state changes
	case CTRL+31: // 1
		select_state(0);
		break;
	case CTRL+30: // 2
		select_state(1);
		break;
	case CTRL+26: // 3
		select_state(2);
		break;
	case CTRL+24: // 4
		select_state(3);
		break;
	case CTRL+29: // 5
		select_state(4);
		break;
	case CTRL+27: // 6
		select_state(5);
		break;
	case CTRL+51: // 7
		select_state(6);
		break;
	case CTRL+53: // 8
		select_state(7);
		break;
	case CTRL+48: // 9
		select_state(8);
		break;
	case CTRL+50: // 0
		select_state(9);
		break;

		// handle noise changes
	case 28:       // ESC
		if (!key_timer)
		{
			set_noise((noise + 1) % 9);		
			key_timer = KEY_RATE;
		}
		break;

		// handle arrow keys
	case CTRL+14: // CTRL + '-' (Up)
		if (!key_timer)
		{
			x = position & 0x7;
			y = position >> 3;
			y = (y + (len >> 3) - 1) % (len >> 3);
			set_cursor((y << 3) + x);
			key_timer = KEY_RATE;
		}
		break;
	case CTRL+15: // CTRL + '=' (Down)
		if (!key_timer)
		{
			x = position & 0x7;
			y = position >> 3;
			y = (y + 1) % (len >> 3);
			set_cursor((y << 3) + x);
			key_timer = KEY_RATE;
		}
		break;
	case CTRL+6:  // CTRL + '+' (Left)
		if (!key_timer)
		{
			x = position & 0x7;
			y = position >> 3;
			set_cursor((y << 3) + ((x + (8 - 1)) & 0x7));
			key_timer = KEY_RATE;
		}
		break;
	case CTRL+7: // CTRL + '*' (Right)
		if (!key_timer)
		{
			x = position & 0xf;
			y = position >> 3;
			set_cursor((y << 3) + ((x + 1) & 0x7));
			key_timer = KEY_RATE;
		}
		break;

		
		// handle notes
	case 23:                // z
		return 0;
	case 62:                // s
		return 1;
	case 22:                // x
		return 2;
	case 58:                // d
		return 3;
	case 18:                // c
		return 4;
	case 16:                // v
		return 5;
	case 61:                // g
		return 6;
	case 21:                // b
		return 7;
	case 57:                // h
		return 8;
	case 35:                // n
		return 9;
	case 1:                 // j
		return 10;
	case 37:                // m
		return 11;
	case 33:  // space
		return SILENCE;
	}
	return INVALID;       // not a note
}

void set_octave(byte o)    // set current octave
{
	octave = o;
	cputcxy(VARCOL+3,14,'1' + (char) o);
}

void set_clock(byte c)  // select clock with configuration c
{
	clock_select = c;  // save configuration
	switch (c)                   // set clock configuration
	{
	case CLK_17_17:
		clock[CH1] = 1789773L;        // 1.79 MHz
		clock[CH2] = 1789773L;        // 1.79 MHz
		clock_flags = 3 << 5;
		cputsxy(VARCOL+1,5,"7/7");
		break;
	case CLK_17_64:
		clock[CH1] = 1789773L;        // 1.79 MHz
		clock[CH2] = 63920L;          // 64 KHz
		clock_flags = 2 << 5;
		cputsxy(VARCOL+1,5,"7/6");
		break;
	case CLK_64_17:
		clock[CH1] = 63920L;          // 64 KHz
		clock[CH2] = 1789773L;        // 1.79 MHz
		clock_flags = 1 << 5;
		cputsxy(VARCOL+1,5,"6/7");
		break;
	case CLK_64_64:
		clock[CH1] = 63920L;          // 64 KHz
		clock[CH2] = 63920L;          // 64 KHz
		clock_flags = 0;
		cputsxy(VARCOL+1,5,"6/6");
		break;
	case CLK_15_15:
		clock[CH1] = 15700L;          // 15 KHz
		clock[CH2] = 15700L;          // 15 KHz
		clock_flags = 1;
		cputsxy(VARCOL+1,5,"5/5");
		break;
	case CLK_17_15:
		clock[CH1] = 1789773L;        // 1.79 MHz
		clock[CH2] = 15700L;          // 15 KHz
		clock_flags = (2 << 5) | 1;
		cputsxy(VARCOL+1,5,"7/5");
		break;
	case CLK_15_17:
		clock[CH1] = 15700L;          // 15 KHz
		clock[CH2] = 1789773L;        // 1.79 MHz
		clock_flags = (1 << 5) | 1;
		cputsxy(VARCOL+1,5,"5/7");
		break;
	}
	set_audctl();
	key_timer = KEY_RATE;
}

void set_audctl(void)    // set pokey audio control
{
	byte c;
	// set pokey audio control register
	c = clock_flags | extra_flags;
	*AUDCTL = c;
}

void set_volume(byte v)
{
	volume = v;
	gotoxy(VARCOL,6);
	cprintf("%4d", (int) v);
	key_timer = KEY_RATE;
}

void set_attack(byte a)
{
	attack = a;                     // level attack
	gotoxy(VARCOL,7);
	cprintf("%4d", a);
	key_timer = KEY_RATE >> 2;
}

void set_release(byte r)
{
	release = r;                    // level release
	gotoxy(VARCOL,8);
	cprintf("%4d", r);
	key_timer = KEY_RATE >> 2;
}

void set_filter(byte b)             // set filter bit
{
	if (b)         // filter on
	{
		extra_flags |= HIGHPASS_2_4;
		cputsxy(VARCOL+1,13," on");
	}
	else        // filter off
	{
		extra_flags &= ~HIGHPASS_2_4;
		cputsxy(VARCOL+1,13,"off");
	}
	set_audctl();
	key_timer = KEY_RATE;
}

void set_mode(byte m1, byte m2)      // set oscillator mode
{
	mode[0] = m1;
	mode[1] = m2;
	gotoxy(VARCOL,11);
	cprintf(" %d/%d", m1, m2);
	key_timer = KEY_RATE;
}

void set_async(byte r)                // set random pause
{
	async = r;
	cputcxy(VARCOL+3,15,'0' + (5-r));
	key_timer = KEY_RATE;
}

void set_detune(char d)               // set oscillator 2 detune
{
	detune = d;
	gotoxy(VARCOL,12);
	cprintf("%4d", (int) d - DETUNE_BIAS);
	key_timer = KEY_RATE >> 1;
}

void set_trans(byte t)              // transpose
{
	trans = t;
	gotoxy(VARCOL,18);
	cprintf("%4d", (int) t - TRANS_BIAS);
	key_timer = KEY_RATE >> 1;
}

void set_tempo(byte t)            // not a real tempo!
{
	tempo = t;
	gotoxy(VARCOL,19);
	cprintf("%4d", t);
	key_timer = KEY_RATE >> 2;
}

void set_noise(byte n)                  // set frequency noise
{
	noise = n;
	gotoxy(VARCOL,21);
	cprintf("%4d", (int) n);
	key_timer = KEY_RATE >> 1;
}

void set_steps(byte s)               // set glide steps
{
	steps = s;
	gotoxy(VARCOL,9);
	cprintf("%4d", (int) s);
	key_timer = KEY_RATE >> 1;
}

void set_speed(byte s)              // set glide speed
{
	speed = s;
	gotoxy(VARCOL,10);
	cprintf("%4d", MAX_SPEED-s);
	key_timer = KEY_RATE >> 1;
}

void set_mix(byte m)                 // set channel mix
{
	mix = m;
	gotoxy(VARCOL+1,16);
	cprintf("%d/%d", m, MAX_VOL-m);
	key_timer = KEY_RATE;
}

void set_length(byte l)         // l == shift count not actual length!
{
	size = l;          // number of bits to right shift max pattern length
	gotoxy(VARCOL,17);
	len = MAX_LEN >> l;      // shift max value by shift count
	cprintf("%4d", len);         // report this in bytes (notes)
	if (len > prev_len)   // copy first half to second half if length increased
		memcpy(&pattern[prev_len], pattern, prev_len);
	redraw_seq();
	prev_len = len;
	key_timer = KEY_RATE;
}

void set_seq(void)
{
	// sequencer mode
	char c;
	if (play)       // sequencer is running
	{
		if (rec)                // if user input writes to pattern
			c = '`';                    // filled circle (record)
		else
			c = '>';           // play sign (user input ignored)
	}
	else         // sequencer is stopped
	{
		if (rec)  // pattern editor mode--user can edit pattern interactively
			c = '#';
		else  // piano mode--synth still plays user input but it's not stored
			c = 'X';
	}
	cputcxy(SEQCOL+MODCOL,22,c);
	key_timer = KEY_RATE << 1;
}

void redraw_seq(void)                  // redraw sequencer display
{
	char i, j, k, l;
	byte index;  // note index in pattern
	if (position >= len)    // correct position
		position = 0;
	l = k = len >> 3;  // number of rows in note pattern
	j = SEQROW;        // start at first sequencer row
	index = 0;         // init note index
	while (l--)        // draw pattern rows
	{
		for (i = SEQCOL; i < SEQCOL+16; i+=2)  // for each note in a row
			cputsxy(i, j, note_name[pattern[index++]]);
		++j;     // next row
	}
	// get number of unused rows
	l = 16 - k;
	if (l > 0)         // if there are unused rows
	{
		// subtract one row for division line
		--l;
		// draw division line below pattern notes
		chlinexy(SEQCOL,j++,16);
		// fill unused space
		if (len < prev_len) // only if new pattern is shorter than old pattern!
		{
			i = 16 - (prev_len >> 3); // number of unused rows in previous pattern
			if (i > 0)   // if there were any unused rows in previous pattern..
				l -= i - 1;    // subtract them from total minus the division line
			while (l--)             // fill unused rows
				cputsxy(SEQCOL, j++, "................");
		}
	}
	set_cursor(position);
}

void set_cursor(byte p)         // update play cursor
{
	byte x, y;     // 2D position
	// erase previous position (unless it's the same as new position)
	if ((position != p) && (position < len))  // check if it's even in bounds
	{
		x = SEQCOL + ((position & 0x7) << 1);
		y = SEQROW + (position >> 3);
		cputsxy(x,y,note_name[pattern[position]]);
	}
	// highlight new position
	revers(TRUE);
	x = SEQCOL + ((p & 0x7) << 1);
	y = SEQROW + (p >> 3);
	cputsxy(x,y,note_name[pattern[p]]);
	revers(FALSE);
	// set new position
	position = p;
}

void display_help(void)               // display help screen
{
	char i, j;
	j = SEQROW;
	for (i = 0; i < 16; i++)
		cputsxy(SEQCOL, j++, help[i]);
}

void clear_display(void)                 // clear pattern area
{
	char i, j;                           // used as the save/load dialog box
	j = SEQROW;
	for (i = 0; i < 16; i++)
		cputsxy(SEQCOL, j++, "                ");
}

byte check_name(void)                              // check filename correctness
{ 
	char n;           // drive number
	byte l;              // length of string
	char *s;                 // string pointer
	if (strlen(name) == 0)      // zero length name.. abort!
		return FILE_ABORT;
	if ((s = strstr(name, ":")) != NULL)      // if name contains drive spec
	{
		if ((uint) s - (uint) name != 2)        // drive spec must be 2 chars
			return DRIVE_SPEC_ERROR;        // error in drive spec
		n = name[1];      // get drive number
		switch (name[0])         // check drive letter
		{
		case 'D':                          // floppy
			if (n < '1' || n > '8')
				return DRIVE_SPEC_ERROR;        // error in drive #
			break;
		case 'H':                          // HD
			if (n < '1' || n > '4')
				return DRIVE_SPEC_ERROR;        // error in drive #
			break;
		default:
			return DRIVE_SPEC_ERROR;        // unknown drive
		}
		++s;     // skip ':'
	}
	else            // no drive specified
		s = name;
	l = strlen(s);            // get length of name without drive spec
	if (l < 1)
		return NULL_NAME;         // error in filename
	if (l > 8)
		return NAME_TOO_LONG;         // error in filename
	// passed all checks
	return NAME_OK;
}

byte save(void)                                       // save file
{
	char n[MAX_CHARS+4+1];           // filename chars + extension + null char
	FILE *f;                         // file object
	state_t *s;                      // state pointer
	byte l;                          // total length of state
	byte i;                          // state index
	sprintf(n, "%s.%s", name, ext);  // add extension
	f = fopen(n, "w");   // open for writing
	if (f != NULL)       // show progress if file opened successfully
		cputc(' ');
	else                 // otherwise return failure
		return FALSE;
	if (fwrite(magic, 1, 4, f) == 4)           // write magic
		cputc(' ');
	else                 // otherwise return failure
		return FALSE;
	// update current state in state array
	s = &state[select];    // point at current state in state array
	memcpy(s, &used, sizeof(state_t));  // copy current state
	// save each state
	for (i = 0; i < MAX_STATES-1; i++)           // minus one clipboard
	{
		s = &state[i];           // point at state
		if (s->used)             // if state was used
		{
			// compute total length of state
			l = sizeof(state_t) - (MAX_LEN - (MAX_LEN >> s->size));
			if (fwrite(&l, 1, 1, f) != 1) // write length of state
				return FALSE;
			if (fwrite(s, 1, l, f) != l)  // write state
				return FALSE;
		}
		else       // unused state
		{
			l = 0;        // set length to zero (empty state)
			if (fwrite(&l, 1, 1, f) != 1)     // write zero length
				return FALSE;
		}
		cputc(' ');          // show progress
	}
	fclose(f);
	return TRUE;             // finally return success
}

byte load(void)                                       // load file
{
	char n[MAX_CHARS+4+1];           // filename chars + extension + null char
	char m[5];                       // read magic
	FILE *f;                         // file object
	state_t *s;                      // state pointer
	byte l;                          // total length of state
	byte i;                          // state index
	sprintf(n, "%s.%s", name, ext);  // add extension
	f = fopen(n, "r");   // open for reading
	if (f != NULL)    // show progress if file opened successfully
		cputc(' ');
	else                 // otherwise return failure
		return FALSE;
	m[4] = NULL;             // terminate read magic string
	if ((fread(m, 1, 4, f) == 4) && (strcmp(magic, m) == 0))  // check magic
		cputc(' ');             // show progress
	else                
		return FALSE; // otherwise return failure
	// load each state
	for (i = 0; i < MAX_STATES-1; i++)           // minus one clipboard
	{
		s = &state[i];               // point at state
		if (fread(&l, 1, 1, f) != 1) // read length of state
			return FALSE;
		if (l != 0) {                // if state not empty
			memset(s->pattern, SILENCE, MAX_LEN);  // clear pattern before loading
			if (fread(s, 1, l, f) != l)   // load state
				return FALSE;
		}
		else             // otherwise mark state as unused
			s->used = FALSE;
		cputc(' ');          // show progress
	}
	fclose(f);
	return TRUE;             // finally return success
}

char __fastcall__ name_char(byte key)        // allowed chars in filename
{
	switch (key)    // need this mess if we're going to read the keyboard ourselves
	{
	case 12:              // RETURN
		return ENTER_CHAR;
	case 52:              // BACKSPACE 
		return ERASE_CHAR;
	case 66:              // DRIVE
		return ':';
	case 56:  // f
		return 'f';
	case 63: // a
		return 'a';
	case 5: // k
		return 'k';
	case 0: // l
		return 'l';
	case 50: // 0
		return '0';
	case 47: // q
		return 'q';
	case 46: // w
		return 'w';
	case 42: // e
		return 'e';
	case 40: // r
		return 'r';
	case 45: // t
		return 't';
	case 43: // y
		return 'y';
	case 11: // u
		return 'u';
	case 13: // i
		return 'i';
	case 10: // p
		return 'p';
	case 8: // o
		return 'o';
	case 31: // 1
		return '1';
	case 30: // 2
		return '2';
	case 26: // 3
		return '3';
	case 24: // 4
		return '4';
	case 29: // 5
		return '5';
	case 27: // 6
		return '6';
	case 51: // 7
		return '7';
	case 53: // 8
		return '8';
	case 48: // 9
		return '9';
	case 23: // z
		return 'z';
	case 62: // s
		return 's';
	case 22: // x
		return 'x';
	case 58:  // d
		return 'd';
	case 18:                // c
		return 'c';
	case 16:                // v
		return 'v';
	case 61:                // g
		return 'g';
	case 21:                // b
		return 'b';
	case 57:                // h
		return 'h';
	case 35:                // n
		return 'n';
	case 1:                 // j
		return 'j';
	case 37:                // m
		return 'm';
	}
	return NULL;            // invalid filename character
}

