; Altirra BASIC - I/O module
; Copyright (C) 2014 Avery Lee, All Rights Reserved.
;
; Copying and distribution of this file, with or without modification,
; are permitted in any medium without royalty provided the copyright
; notice and this notice are preserved.  This file is offered as-is,
; without any warranty.

;==========================================================================
; Print a message from the message database.
;
; Entry:
;	X = LSB of message pointer
;
.proc IoPrintMessage
		stx		inbuff
loop:
		ldx		inbuff
		lda		msg_base&$ff00,x
		beq		xit
		jsr		IoPutCharAndInc
		jmp		loop
xit:
		rts
.endp

;==========================================================================
IoPrintInt:
		jsr		fp_ifp
IoPrintNumber:
		jsr		fp_fasc
.proc printStringINBUFF
loop:
		ldy		#0
		lda		(inbuff),y
		pha
		and		#$7f
		jsr		IoPutCharAndInc
		pla
		bpl		loop
		rts
.endp

;==========================================================================
.proc IoConvNumToHex
		php
		jsr		fp_ldbufa
		jsr		fp_fpi
		ldy		#0
		plp
		lda		fr0+1
		bcc		force_16bit
		beq		print_8bit
force_16bit:
		jsr		print_digit_pair
print_8bit:
		lda		fr0
print_digit_pair:
		pha
		lsr
		lsr
		lsr
		lsr
		jsr		print_digit
		pla
		and		#$0f
print_digit:
		cmp		#10
		scc:adc	#6
		adc		#$30
		sta		(inbuff),y
		iny
		rts
.endp

;==========================================================================
.proc IoPutCharAndInc
		inw		inbuff
		jmp		putchar
.endp

;==========================================================================
IoPutCharDirectX = putchar.direct_with_x
IoPutCharDirect = putchar.direct

IoPutNewline:
		lda		#$9b
		dta		{bit $0100}
IoPutSpace:
		lda		#' '
.proc putchar
		dec		ioPrintCol
		bne		not_tabstop
		mvx		ptabw ioPrintCol
not_tabstop:
direct:
		ldx		iocbidx
direct_with_x:
		jsr		dispatch
		tya

.def :ioCheck = *
		bpl		done
dispatch_error:
		sty		errno
		jmp		errorDispatch
		
dispatch:
		jsr		CommBeginMessage
		stx		COMM_OFFSET+comm_putchar_iocb
		sta		COMM_OFFSET+comm_putchar_char
		mwa		#comm_HandlePutchar COMM_OFFSET+comm_hostrout
		jsr		CommSendMessage
		lda		COMM_OFFSET+comm_buffer+1
		ldy		COMM_OFFSET+comm_buffer+2
		phy
		pha
		jsr		CommAckReply
		pla
		ply
done:
		rts
.endp

;==========================================================================
ioChecked = IoDoCmd._check_entry2
IoDoCmdX = IoDoCmd._with_x
.proc IoDoCmd
		ldx		iocbidx
_with_x:
		sta		iccmd,x
_check_entry2:
		jsr		ciov816
		jmp		ioCheck
.endp

;==========================================================================
; Issue I/O call with a filename.
;
; Entry:
;	A = command to run
;	fr0 = Pointer to string info (ptr/len)
;	iocbidx = IOCB to use
;
; ICBAL/ICBAH is automatically filled in by this fn. Because BASIC strings
; are not terminated, this routine temporarily overwrites the end of the
; string with an EOL, issues the CIO call, and then restores that byte.
; The string is limited to 255 characters.
;
; I/O errors are checked after calling CIO and the error handler is issued
; if one occurs.
;
IoDoOpenReadWithFilename:
		lda		#4
IoDoOpenWithFilename:
		ldx		iocbidx
		sta		icax1,x
		lda		#CIOCmdOpen
.proc IoDoWithFilename
		;stash command
		ldx		iocbidx
		sta		iccmd,x
						
		;move pointer to ICBAL/H
		mwa		fr0 icbal,x
		
		;call CIO
		jsr		IoTerminateString
		jsr		ciov816
		jsr		IoUnterminateString
		
		;now we can check for errors and exit
		jmp		ioCheck
.endp

;==========================================================================
IoSetupIOCB7:
		ldx		#$70
		stx		iocbidx
IoCloseX = IoClose.with_IOCB_X
.proc IoClose
		ldx		iocbidx
with_IOCB_X:
		lda		#CIOCmdClose
.def :IoTryCmdX = *
		sta		iccmd,x
		jmp		ciov816
.endp

;==========================================================================
; Replace the byte after a string with an EOL terminator.
;
; Entry:
;	FR0 = string pointer
;	FR0+2 = string length (16-bit)

; Registers:
;	A, Y modified; X preserved
;
; Exit:
;	INBUFF = string pointer
;
; This is needed anywhere where a substring needs to be passed to a module
; that expects a terminated string, such as the math pack or CIO. This
; will temporarily munge the byte _after_ the string, which can be a
; following program token, the first byte of another string or array, or
; even the runtime stack. Therefore, the following byte MUST be restored
; ASAP.
;
; The length of the string is limited to 255 characters.
;
.proc IoTerminateString
		;compute termination offset		
		ldy		fr0+2
		lda		fr0+3
		seq:ldy	#$ff
		sty		ioTermOff
		
		;copy term address
		mwa		fr0 inbuff
		
		;save existing byte
		lda		(inbuff),y
		sta		ioTermSave
		
		;stomp it with an EOL
		lda		#$9b
		sta		(inbuff),y
		rts
.endp

;==========================================================================
; Entry:
;	INBUFF = string pointer
;
; Registers:
;	Y, P preserved
;
.proc IoUnterminateString
		php
		tya
		pha
		ldy		ioTermOff
		lda		ioTermSave
		sta		(inbuff),y
		pla
		tay
		plp
		rts
.endp

;==========================================================================
; Open the cassette (C:) device or any other stock device.
;
; Entry (IoOpenCassette):
;	None
;
; Entry (IoOpenStockDeviceIOCB7):
;	A = AUX1 mode
;	Y = Low byte of device name address in constant page
;
; Entry (IoOpenStockDevice):
;	A = AUX1 mode
;	X = IOCB #
;	Y = Low byte of device name address in constant page
;
.proc IoOpenCassette
		ldy		#<devname_c
.def :IoOpenStockDeviceIOCB7 = *
		ldx		#$70
		stx		iocbidx
.def :IoOpenStockDevice = *
		sta		icax1,x
		tya
		sta		icbal,x
		mva		#>devname_c icbah,x
		lda		#CIOCmdOpen
		jmp		IoDoCmdX
.endp

;==========================================================================
.proc IoSetupReadLine
		;we are using some pretty bad hacks here:
		;- GET RECORD and >LBUFF are $05
		;- <LBUFF is $80
		mva		#CIOCmdGetRecord iccmd,x
		sta		icbah,x
		lda		#$80
		sta		icbal,x
		asl
		sta		icblh,x
		lda		#$ff
		sta		icbll,x
		rts
.endp


;==========================================================================
.proc ciov816
		sep		#$30

		;save IOCB
		phx

		;own the comm buffer
		jsr		CommBeginMessage

		;check if this is a big read, >512 bytes -- if so, we must use
		;BigIO
		plx
		lda		iccmd,x
		sub		#4
		cmp		#8
		bcs		not_readwrite

		lda		icblh,x
		cmp		#2
		scc:jmp		requires_bigio
not_readwrite:

		;copy the request to the comm buffer
		stx		COMM_OFFSET+comm_buffer
		ldy		#0
copy_loop:
		mva		iccmd,x COMM_OFFSET+comm_buffer+1,y
		inx
		iny
		cpy		#14
		bne		copy_loop

		;set copy limit -- 8 for read/write, 10 for open or special to include ICAX1/2
		lda		COMM_OFFSET+comm_buffer+1
		ldy		#8
		cmp		#CIOCmdOpen
		beq		need_icax12
		cmp		#CIOCmdSpecial
		bcc		need_not_icax12
need_icax12:
		ldy		#10
need_not_icax12:
		sty		COMM_OFFSET+comm_ciov_copylimit

		;check if we are doing a write or open -- if so, copy the data into the
		;aperture as well
		cmp		#CIOCmdOpen
		bne		not_open

		;copy filename into aperture, until (and including) EOL
		ldx		COMM_OFFSET+comm_buffer
		rep		#$10
		ldy		icbal,x
		ldx.w		#COMM_OFFSET+comm_buffer+$10
strcpy_loop:
		lda		0,y
		sta		0,x
		iny
		inx
		cmp		#$9b
		bne		strcpy_loop
		bra		after_strcpy

not_open:
		and		#$fc
		cmp		#$08
		bne		not_write

		;copy buffer into aperture
		rep		#$30

		ldx		COMM_OFFSET+comm_buffer+1+[icbal-iccmd]
		ldy.w	#COMM_OFFSET+comm_buffer+$10
		sty		COMM_OFFSET+comm_buffer+1+[icbal-iccmd]
		lda		COMM_OFFSET+comm_buffer+1+[icbll-iccmd]
		dec
		mvn		0,0

after_strcpy:
		sep		#$30

not_write:
		;set the handler address
		mwa		#comm_HandleCIOV COMM_OFFSET+comm_hostrout

		;issue the request
		jsr		CommSendMessage

		;if we are doing a read, copy the data back from the aperture
		lda		COMM_OFFSET+comm_buffer+1
		and		#$fc
		cmp		#$04
		bne		not_read

		ldx		COMM_OFFSET+comm_buffer

		rep		#$30
		lda		COMM_OFFSET+comm_buffer+1+[icbll-iccmd]
		sta		icbll,x
		beq		no_copy_back
		ldy		icbal,x
		ldx.w	#COMM_OFFSET+comm_buffer+$10
		dec
		mvn		0,0

no_copy_back:
		sep		#$30

not_read:
ack_and_exit:
		;load status and exit -- must do this BEFORE we exit
		ldx		COMM_OFFSET+comm_buffer
		ldy		COMM_OFFSET+comm_buffer+1+[icsta-iccmd]

		phy
		phx

		;ACK the reply
		jsr		CommAckReply

		plx
		ply
		rts

requires_bigio:
		;Okay, we have to do a trampoline I/O. This is going to suck:
		;
		; - The low page is used for the buffer so we can do 8K I/Os, since
		;	the standard 512b aperture is too small for burst I/O. This means
		;	that we have to jump out of here since we're executing in the
		;	banking region (doh!).
		;
		; - The I/O may be bigger than 8K, in which case we need to split the
		;	I/O request. Yay....
		;
		; Request block:
		;	+0		in: IOCB offset
		;	+1		out: status
		;	+2		in: command (0 = end)
		;	+3,4	in/out: I/O length

		mwa		#comm_HandleBigIO COMM_OFFSET+comm_hostrout
		jsr		CommSendMessage

		stx		COMM_OFFSET+comm_buffer
		lda		iccmd,x
		sta		COMM_OFFSET+comm_buffer+2

		rep		#$10
		ldy		icbal,x
		sty		ioPtr
		ldy		icbll,x
		sty		ioLeft
		stz		icbll,x
		stz		icbll+1,x

bigio_loop:
		;determine amount to read/write
		ldy.w	#$2000
		cpy		ioLeft
		scc:ldy	ioLeft
		sty		ioLen
		sty		COMM_OFFSET+comm_buffer+3

		;check whether this is a write request
		lda		iccmd,x
		bit		#$08
		beq		bigio_not_write

		;it's a write request -- copy into aperture
		rep		#$30
		lda		ioLen
		ldx		ioPtr
		ldy.w	#$e000
		jsr		ApertureCopy

bigio_not_write:

		;do the I/O
		sep		#$30
		jsr		CommSendMessage

		;check whether this is a read request
		ldx		COMM_OFFSET+comm_buffer
		lda		iccmd,x
		bit		#$04
		beq		bigio_not_read

		;it's a read request -- copy out of aperture
		rep		#$30
		phx
		lda		ioLen
		ldx.w	#$e000
		ldy		ioPtr
		jsr		ApertureCopy
		plx

bigio_not_read:
		;update status
		lda		COMM_OFFSET+comm_buffer+1
		sta		icsta,x
		tay

		;update result length
		rep		#$31
		lda		icbll,x
		adc		COMM_OFFSET+comm_buffer+3
		sta		icbll,x

		;exit loop if we got an error
		tya
		bmi		bigio_done

		;update pointer
		lda		ioPtr
		adc		COMM_OFFSET+comm_buffer+3
		sta		ioPtr

		;update size left
		lda		ioLeft
		sec
		sbc		COMM_OFFSET+comm_buffer+3
		sta		ioLeft

		rep		#$10
		bne		bigio_loop

bigio_done:
		;send ACK with command 0 to terminate
		sep		#$30
		lda		#0
		sta		COMM_OFFSET+comm_buffer+2

		;load IOCB/status and exit
		jmp		ack_and_exit
.endp
