; Altirra BASIC - Memory handling module
; Copyright (C) 2014 Avery Lee, All Rights Reserved.
;
; Copying and distribution of this file, with or without modification,
; are permitted in any medium without royalty provided the copyright
; notice and this notice are preserved.  This file is offered as-is,
; without any warranty.

;==========================================================================
; Input:
;	Y:A		Total bytes required
;	X		ZP offset of first pointer to offset
;	A0		Insertion point
;
; Errors:
;	Error 2 if out of memory (yes, this may invoke TRAP!)
;
.proc expandTable
		;##TRACE "Expanding table: $%04x bytes required, table offset $%02x (%y:%y) [$%04x:$%04x], insert pt=$%04x" y*256+a x x-2 x dw(x-2) dw(x) dw(a0)
		sta		a2
		sty		a2+1
		phx

		rep		#$30

		;Check how much space is available and throw error 2 if we're
		;out. Note that we're one byte off here -- this is because
		;FRE(0) has this bug, and we don't want to have that value go
		;negative.
		lda		memtop
		sec
		sbc		memtop2
		cmp		a2
		bcc		out_of_memory
memory_ok:

		;compute number of bytes to copy
		;##ASSERT dw(a0) <= dw(memtop2)
		lda		memtop2
		sec
		sbc		a0
		sta		a3
		
		;top of src = memtop2		
		;top of dst = memtop2 + N
		clc
		lda		memtop2
		sta		a1
		adc		a2
		sta		a0
		
		sep		#$30

		jsr		copyDescending
		jmp		contractTable.adjust_table_pointers

out_of_memory:
		jmp		errorNoMemory
.endp

;==========================================================================
; Input:
;	A1	end of source range
;	A0	end of destination range
;	A3	bytes to copy
;
; Modified:
;	A0, A1
;
; Preserved:
;	A2
.proc copyDescending
		;##TRACE "Copy descending src=$%04x-$%04x, dst=$%04x-$%04x (len=$%04x)" dw(a0)-dw(a3) dw(a0) dw(a1)-dw(a3) dw(a1) dw(a3)
		;##ASSERT dw(a3) <= dw(a0) and dw(a3) <= dw(a1)
		rep		#$30

		lda		a3
		beq		no_bytes
		dec
		ldx		a1
		dex
		ldy		a0
		dey
		mvp		0,0

no_bytes:
		sep		#$30
		rts
.endp

;==========================================================================
;
; Input:
;	Y:A		Total bytes required (negative)
;	X		ZP offset of first pointer to offset
;	A0		Deletion point
.proc contractTable
		;##TRACE "Memory: Contracting table %y by %u bytes at $%04X" x-2 -(y*256+a)&$ffff dw(a0)
		sta		a2
		sty		a2+1
		ora		a2+1
		beq		nothing_to_do
		phx
		
		rep		#$30

		;compute source position
		lda		a0
		sub		a2
		sta		a1
		
		;compute bytes to copy	
		lda		memtop2
		sub		a1
		sta		a3

		sep		#$30
		jsr		copyAscending

adjust_table_pointers:
		pla
		tax

.def :MemAdjustTablePtrs = *

		ldy		#a2
offset_loop:
		jsr		IntAdd
		inx
		inx
		cpx		#memtop2+2
		bne		offset_loop

.def :MemAdjustAPPMHI = *			;NOTE: Must not modify X or CLR/NEW will break.
		;update OS APPMHI from our memory top
		lda		memtop2
		sta		appmhi
		lda		memtop2+1
		sta		appmhi+1

nothing_to_do:
		rts
.endp

;==========================================================================
.proc IntAdd
		rep		#$20

		lda		0,x
		add		0,y
		sta		0,x

		sep		#$20
		rts
.endp

;==========================================================================
; Input:
;	A1	source start
;	A0	destination
;	A3	bytes to copy
;
; Modified:
;	A0, A1
;
; Preserved:
;	A2
.proc copyAscending
		;##TRACE "Copy ascending src=$%04X, dst=$%04X (len=$%04X)" dw(a1) dw(a0) dw(a3)
		rep		#$30

		lda		a3
		beq		no_bytes

		dec
		ldx		a1
		ldy		a0
		mvn		0,0

no_bytes:
		sep		#$30
		rts
.endp
