/*
 * Copyright (c) 2010-2013 A8CAS developers (see AUTHORS)
 *
 * This file is part of the A8CAS project which allows to manipulate tape
 * images for Atari 8-bit computers.
 *
 * A8CAS is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * A8CAS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along
 * with A8CAS; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.
 */
#ifndef A8CAS_H
#define A8CAS_H

#include <stdio.h>

#ifdef __cplusplus
extern "C" {
#endif

#if defined(_WIN32) && !defined(_XBOX)
 #define A8CAS_API __declspec(dllexport)
#elif defined(HAVE_VISIBILITY_PRAGMA)
 #define A8CAS_API __attribute__((visibility("default")))
#else
 #define A8CAS_API
#endif

/* Error codes returned by A8CAS functions.
   Some errors below are "fatal" - getting one of them means that
   reading/writing from/to a file will not possible until A8CAS_seek() is
   called. The file should still be closed anyway (using A8CAS_close(). */
typedef enum A8CAS_errnum {
	A8CAS_ERR_NONE,                 /* No error */
	A8CAS_ERR_UNSUPPORTED,          /* Function not available (f.e. for the current format) */
	A8CAS_ERR_EOF,                  /* End of file reached */
	A8CAS_ERR_NOMEM,                /* fatal: malloc() failed */
	A8CAS_ERR_INVALID,              /* Invalid value of a parameter */
	A8CAS_ERR_BADFILE,              /* fatal: File format not recognised (or maybe file is broken) */
	A8CAS_ERR_NOMATCH,              /* wrong format when opening file for read, or wrong moment to call a function */
	A8CAS_ERR_FOPEN,                /* fatal: fopen() failed, check errno */
	A8CAS_ERR_FCLOSE,               /* fatal: fclose() failed, check errno */
	A8CAS_ERR_FSEEK,                /* fatal: fseek() failed, check errno */
	A8CAS_ERR_FTELL,                /* fatal: ftell() failed, check errno */
	A8CAS_ERR_FREAD,                /* fatal: fread() failed, check errno */
	A8CAS_ERR_FWRITE,               /* fatal: fwrite()/fprintf()/fflush() failed, check errno */
	A8CAS_ERR_FTRUNCATE,            /* fatal: ftruncate() failed, check errno. */
	A8CAS_ERR_SNDFILE,              /* fatal: error in a libsndfile function */
	A8CAS_ERR_LIMIT,                /* fatal: internal or file format's limit exceeded while reading/writing */
	A8CAS_ERR_SAMPLERATE            /* fatal: Too low sample rate for sound file reading */
} A8CAS_errnum;

/* Formats recognised by A8CAS_open */
typedef enum A8CAS_format {
	A8CAS_FORMAT_ANY = 0,
	A8CAS_FORMAT_CAS,
	A8CAS_FORMAT_HEX,
	A8CAS_FORMAT_FSK,
	A8CAS_FORMAT_SNDFILE,
	A8CAS_FORMAT_RAW
} A8CAS_format;

/* Type of the value containing the block number */
typedef unsigned int A8CAS_block_count_t;

/* TODO delete after read_samples_short is removed! */
typedef unsigned int A8CAS_count_t;

typedef struct A8CAS_signal {
	unsigned int rate; /* length / rate is the length of signal (in seconds) */
	unsigned int length; /* length of signal */
	char signal; /* logical 1 (Mark) or 0 (space) */
} A8CAS_signal;

typedef enum A8CAS_open_mode {
	A8CAS_READ = 0,
	A8CAS_WRITE,
	A8CAS_RDWR,
	A8CAS_WRRD
} A8CAS_open_mode;

typedef struct A8CAS_info {
	A8CAS_open_mode mode; /* read / write / both */
	/* Samplerate for the generated audio file (only A8CAS_FORMAT_SNDFILE).
	   If opening for write, should be given by user. Otherwise, it is set 
	   by A8CAS_open(). */
	unsigned int samplerate;
	/* File's description. Used in CAS, HEX and FSK formats. When opening
	   a file for reading, this field should be left NULL. On return from
	   A8CAS_open it contains the descritpion read from file, and it will
	   be deallocated on a call to A8CAS_close.
	   When opening a file for writing, this field may be NULL (no
	   description) or point to a description that will be written to
	   file. The user is responsible for deallocating the buffer, and it
	   can be done immediately after A8CAS_open. */
	char const *description;
	A8CAS_format format;
	A8CAS_errnum errnum;
} A8CAS_info;

typedef struct A8CAS_FILE A8CAS_FILE;

A8CAS_API A8CAS_errnum A8CAS_error(A8CAS_FILE *file);

/* Returns a pointer to an A8CAS_FILE object.
   On error returns NULL and sets info->errnum:
   - A8CAS_ERR_NOMEM - memory could not be allocated;
   - A8CAS_ERR_INVALID - invalid value for INFO.MODE or INFO.FORMAT,
     or INFO.FORMAT set to A8CAS_FORMAT_ANY when opening for write or WRRD;
   - A8CAS_ERR_BADFILE - the opened file has invalid format;
   - A8CAS_ERR_NOMATCH - when opening for read or RDWR, the opened file has
     valid format, but it does not match INFO,FORMAT;
   - and possibly other values.
   */
A8CAS_API A8CAS_FILE* A8CAS_open(char const *path, A8CAS_info *info);

/* Returns 0 on success, else returns error number.
   Invoking on FILE not previously opened yields undefined result. */
A8CAS_API int A8CAS_close(A8CAS_FILE *file);

/* Returns 0 on success, else call A8CAS_error() to get an error code:
   - A8CAS_ERR_EOF - end of file reached;
   - A8CAS_ERR_BADFILE - file format is invalid;
   - other errors possible.
   Invoking on FILE not previously opened yields undefined result. */
A8CAS_API int A8CAS_read(A8CAS_FILE *file, A8CAS_signal *sig);

/* Returns 0 on success, else call A8CAS_error() to get an error code:
   - A8CAS_ERR_BADFILE - file's format does not allow to store the stream of
     signals that was written to this point. Happens only in the raw format,
     which can store only standard SIO records;
   - other errors possible.
   Invoking on FILE not previously opened yields undefined result. */
A8CAS_API int A8CAS_write(A8CAS_FILE *file, A8CAS_signal const *sig);

/* Reads at most SIZE bytes of data from cassette; stores them in BYTES.
   Returns number of bytes written. If 0 is returned, A8CAS_error() should be
   checked. If there was no error (A8CAS_ERR_NONE) then it means that a
   non-byte signal was encountered and next time the function
   A8CAS_read_signals() should be called instead. */
A8CAS_API unsigned int A8CAS_read_bytes(A8CAS_FILE *file, unsigned char *bytes, unsigned int size, unsigned int *baudrate);

/* Reads at most SIZE signals from cassette; stores them in SIGS. Returns
   number of signals written. If 0 is returned, A8CAS_error() should be
   checked. If there was no error (A8CAS_ERR_NONE) then it means that a
   full byte was encountered and next time the function A8CAS_read_bytes()
   should be called instead. */
A8CAS_API unsigned int A8CAS_read_signals(A8CAS_FILE *file, A8CAS_signal *sigs, unsigned int size);

/* Writes a SIZE number of BYTES to a tape with the given BAUDRATE. Returns 0
   on success, non-zero otherwise. */
A8CAS_API int A8CAS_write_bytes(A8CAS_FILE *file, unsigned char const *bytes, unsigned int size, unsigned int baudrate);

/* Writes out all buffeed data. Any CAS chunks being written are immediately
 * ended. Returns 0 on success, non-zero otherwise. */
A8CAS_API int A8CAS_flush(A8CAS_FILE *file);

/* ---- Audio playback functions ---- */
A8CAS_API void A8CAS_set_audio(A8CAS_FILE *file, unsigned int samplerate, unsigned int bits);

/* Reads next signal SIG from FILE. Stores a number in *NUM_SAMPLES; the value
   is number of samples of an audio track that should be read before the next
   call to A8CAS_read_with_audio.
   Returns 0 on success; otherwise returns non-zero and sets A8CAS_error(). */
A8CAS_API int A8CAS_read_with_audio(A8CAS_FILE *file, A8CAS_signal *sig, unsigned int *num_samples);

/* Should be called after A8CAS_read_with_audio(). Fills the audio buffer,
   SAMPLES, with a specified number of samples, NUM_SAMPLES. It may actually
   write fewer samples than requested - so the function returns number of
   samples really written. */
A8CAS_API int A8CAS_read_audio(A8CAS_FILE *file, void *samples, unsigned int num_samples);

A8CAS_API int A8CAS_write_audio(A8CAS_FILE *file, A8CAS_signal const *sig, void *samples);

/* ---- Block-navigation functions ---- */

/* Returns current block/sample number (counted from 0). */
A8CAS_API unsigned long A8CAS_tell(A8CAS_FILE *file);

/* Returns number of blocks/samples. */
A8CAS_API unsigned long A8CAS_size(A8CAS_FILE *file);

/* Seeks the file to a block/sample given in POSITION. It can be greater or
   equal than the size returned by A8CAS_size() - in that case,
   the tape will be rewinded to the end.
   Returns 0 on success, otherwise non-zero and sets A8CAS_error(). */
A8CAS_API int A8CAS_seek(A8CAS_FILE *file, unsigned long position);

enum {
	A8CAS_SEEK_UNIT_BLOCKS,
	A8CAS_SEEK_UNIT_SAMPLES
};

A8CAS_API int A8CAS_seek_unit(A8CAS_FILE *file);

/* ---- Functions for adjusting encoding/decoding options and parameters ---- */

/* Several different parameters can be adjusted when reading or writing an
   A8CAS_FILE. A parameter is identified by variable TYPE of type
   A8CAS_param. Different parameter values have different types; the interface
   functions (A8CAS_set/get_param) operate on void pointers to overcome that.
   To set a new value of a parameter, A8CAS_set_param must be used. VALUE must
   point to a new parameter's value. Type of the value depends on parameter's
   type.
   To get a parameter's value, A8CAS_get_param must be used. After return, the
   memory pointed to by VALUE will contain the parameter's value.
   If the particular file does not support a given paramaeter's type, the
   functions will return non-zero and sets A8CAS_error() to
   A8CAS_ERR_UNSUPPORTED. */
typedef enum A8CAS_param {
	A8CAS_PARAM_PWM_DECODE, /* Value type: int */
	A8CAS_PARAM_PWM_TOLERANCE, /* Value type: unsigned short int */
	A8CAS_PARAM_BLOCK_HEADER_DEVIATION, /* Value type: double */
	A8CAS_PARAM_STD_BLOCK_HEADER_DEVIATION = A8CAS_PARAM_BLOCK_HEADER_DEVIATION, /* Deprecated */
	A8CAS_PARAM_BIT_DEVIATION, /* Value type: double */
	A8CAS_PARAM_STOP_BIT_DEVIATION, /* Value type: double */
	A8CAS_PARAM_BIT_TIMESHIFT, /* Value type: double */
	A8CAS_PARAM_BLOCK_HEADER_LENGTH, /* Value type: unsigned int */
	A8CAS_PARAM_NONSTD_BLOCK_HEADER_LENGTH = A8CAS_PARAM_BLOCK_HEADER_LENGTH, /* Deprecated */
	A8CAS_PARAM_NONSTD_BLOCK_HEADER_DEVIATION, /* Deprecated */
	A8CAS_PARAM_BAUDRATE_DEVIATION, /* Value type: double */
	A8CAS_PARAM_SILENCE_LEVEL, /* Value type: float */
	A8CAS_PARAM_CROSSTALK_VOLUME /* Value type: float */
} A8CAS_param;

/* Returns 0 on success, else call A8CAS_error() to get an error code:
   - A8CAS_ERR_UNSUPPORTED - parameter type not supported for this file;
   - A8CAS_ERR_INVALID - parameter value not valid (e.g. out of bounds);
   - A8CAS_ERR_NOMATCH - parameter type and value valid, but cannot change
     the parameter at this moment. */
A8CAS_API int A8CAS_set_param(A8CAS_FILE *file, A8CAS_param type, void const *value);

/* Returns 0 on success, else call A8CAS_error() to get an error code:
   - A8CAS_ERR_UNSUPPORTED - parameter type not supported for this file. */
A8CAS_API int A8CAS_get_param(A8CAS_FILE *file, A8CAS_param type, void *value);

/* ---- Functions for turbo systems ---- */
/* This function is deprecated; use instead
   A8CAS_set_param(file, A8CAS_PARAM_PWM_DECODE, &on)
   Switches read decoding between standard Frequency Shift Keying, and
   Pulse Width Modulation used by European turbo modifications.
   Returns 0 on success, non-zero otherwise. */
A8CAS_API int A8CAS_switch_PWM_decoding(A8CAS_FILE *file, int on);

/* This function is deprecated; use instead
   A8CAS_set_param(file, A8CAS_PARAM_PWM_TOLERANCE, &tolerance) */
A8CAS_API int A8CAS_set_PWM_tolerance(A8CAS_FILE *file, unsigned short int tolerance);

/* This function is deprecated; use instead
   A8CAS_get_param(file, A8CAS_PARAM_PWM_TOLERANCE, &tolerance) */
A8CAS_API int A8CAS_PWM_tolerance(A8CAS_FILE *file, unsigned short int *tolerance);

/* ---- Functions for debugging ---- */
A8CAS_API void A8CAS_set_log_stream(A8CAS_FILE *file, FILE *stream);

/* Returns the library's version - the least significant 16 bits are the minor
   number, the rest is the major number. */
extern A8CAS_API int const A8CAS_version;

#ifdef __cplusplus
}
#endif

#endif /* #ifndef A8CAS_H */
