{
  Program    : Effectus - Atari MADS cross-assembler/parser for Action! language

  Unit file  : core.pas
  Description: Build-in Action! core behaviour
                           
  Author: Bostjan Gorisek, Slovenia
  
  Program compiled with Free Pascal 2.6.4
  Reference: http://www.freepascal.org/
  
  This is cource code generator for MADS Assembler, which then generates
  executable files from parser of Action! language for 8-bit Atari home computers.
  
  References:
  http://mads.atari8.info/
  http://gury.atari8.info/effectus/ 
  
  This is open source and freeware project!  
}
unit Core;
{$ifdef FPC}{$mode objfpc}{$h+}{$endif}

interface

uses
  sysutils, Classes, StrUtils;

procedure sc_Proc;
procedure sc_Return;
procedure sc_Include;
procedure sc_Var;
procedure sc_Var2;
procedure sc_Var3(xType, xVar, xVarType : String);
procedure sc_Array;
procedure sc_VarExpr;
procedure sc_Data;
procedure sc_ML_data;
procedure sc_ProcTrack;
procedure sc_For;
procedure sc_od;
procedure sc_fi;
procedure sc_else;
function FuncCheck: Boolean;
function FuncCheckTrue(stmt : String) : Boolean;
procedure sc_Define;
procedure GenLoop(Flag : Boolean);
procedure Cond(Stmt : String);
procedure sc_do;
procedure sc_Exit;
procedure sc_ML;

implementation

uses
  Decl, Common, Routines;

{
  Procedure name : sc_Proc
  Description    : Handles Action! PROC and FUNC definitions
  Parameters     : None
}
procedure sc_Proc;
var
  ParamList : TStringList;
  n1, n, j : LongInt;
  FuncName, FuncParams, VarType, VarX, Buffer, MemAddr : String;
begin
  n1 := System.Pos('PROC ', UpperCase(TextBuf[CurLine]));
  n := System.Pos('FUNC ', UpperCase(TextBuf[CurLine]));

  if ((n1 > 0) or (n > 0)) then
  begin
    if System.Pos('"', TextBuf[CurLine]) > 0 then Exit;
    
    if n1 > 0 then
      Split(UpperCase(TextBuf[CurLine]), 'PROC', [])
    else
      Split(UpperCase(TextBuf[CurLine]), 'FUNC', []);

    FuncName := StrBuf[1];    
    Split(FuncName, '(', []);
    FuncName := StrBuf[0];
    FuncParams := Extract(StrBuf[1], ')', 1);    
    
    // Check for routine vectored to some memory address
    MemAddr := Extract(FuncName, '=', 2);
    FuncName := Extract(FuncName, '=', 1);
    
    n := System.Pos('[', TextBuf[CurLine]);
    n1 := System.Pos('PROC ', UpperCase(TextBuf[CurLine]));
    j := System.Pos('FUNC ', UpperCase(TextBuf[CurLine]));
    
    // Example: PROC SCROLL =$F7F7 ()
    if (FuncName <> MemAddr) and (n < 1) then
    begin
      flags := flags + [sMemAddr];
    end;
    
    // Example: PROC Pokest2=*()[$A9$60$8D$02C6$0$60]
    if (n > 0) and ((n1 > 0) or (j > 0)) then
    begin
      flags := flags + [sProcAsm];
    end;

    // Check for machine language mnemonics
    if System.Pos('=*', FuncName) > 0 then
    begin
      FuncName := Copy(FuncName, 1, Length(FuncName) - 2);      
      CntML := MemCnt;
      Inc(ProcML_cnt);
      ProcML[ProcML_cnt].Name := FuncName;
      ProcML[ProcML_cnt].ProcType := 0;      
      ProcML[ProcML_cnt].Code := ' .he';
      ProcML_start := True;
      PrgVar.SB := _SB_PROC_ML;
    end;    

    if lIncludeX and not lInclude then
    begin
      if (LowerCase(Copy(ProcBuf[ProcCount2 - 1], 5, Length(ProcBuf[ProcCount2 - 1]) - 4)) = LowerCase(FuncName)) then
      begin
        CodeBuf.Add(FuncName + _REFF);
        lMainProc := True;
        Exit;
      end;
    end;
    
    // Routine has parameters
    //
    if Length(FuncParams) > 0 then
    begin        
      ParamList := TStringList.Create;
      Buffer := '';
      StrBuf2.Clear;
      FuncParams := StringReplace(FuncParams, ', ', ',', [rfReplaceAll]);
      Split(FuncParams, ' ', []);      

      for n := 0 to StrBuf.Count - 1 do
      begin      
        StrBuf2.Add(StrBuf[n]);
      end;
      
      for n := 0 to StrBuf2.Count - 1 do
      begin
        j := VarTypes.IndexOf(StrBuf2[n]);
        
        if j >= 0 then
        begin        
          case j of
            0, 1 : VarX := 'byte';
            2, 3 : VarX := 'word';
          end;
          
          VarType := VarTypes[j];
          Buffer := Buffer + ' .' + VarX + ' ';
        end else begin
          if  System.Pos(',', StrBuf2[n]) > 0 then
          begin
            Split(StrBuf2[n], ',', []);
            
            for n1 := 0 to StrBuf.Count - 1 do
            begin
              Inc(GVarCnt);
              GVar[GVarCnt].VarType  := SetType(VarType);
              GVar[GVarCnt].OrigType := '';
              GVar[GVarCnt].VarName  := StrBuf[n1];
              GVar[GVarCnt].Location := FuncName;
              GVar[GVarCnt].Value    := '';
              GVar[GVarCnt].Dim      := 0;
              GVar[GVarCnt].ML_type  := VarX;
              GVar[GVarCnt].Scope    := 'L';
              
              ParamList.Add(StrBuf[n1]);
              
              if (sMemAddr in Flags) or (sProcAsm in Flags) then
              begin
                case n1 of
                  0: Buffer := Buffer + 'a';
                  1: Buffer := Buffer + ', x';
                  2: Buffer := Buffer + ', y';
                end;
              end else begin
                if n1 = 0 then
                  Buffer := Buffer + StrBuf[n1] + _EFF
                else
                  Buffer := Buffer + ', ' + StrBuf[n1] + _EFF;
              end;
            end;          
          end else begin
            Inc(GVarCnt);
            GVar[GVarCnt].VarType  := SetType(VarType);
            GVar[GVarCnt].OrigType := '';
            GVar[GVarCnt].VarName  := StrBuf2[n];
            GVar[GVarCnt].Location := FuncName;
            GVar[GVarCnt].Value    := '';
            GVar[GVarCnt].Dim      := 0;
            GVar[GVarCnt].ML_type  := VarX;
            GVar[GVarCnt].Scope    := 'L';
            
            ParamList.Add(StrBuf2[n]);
            
            if (sMemAddr in Flags) or (sProcAsm in Flags) then
              Buffer := Buffer + 'a'
            else
              Buffer := Buffer + StrBuf2[n] + _EFF;
          end;
        end;
      end;

      if (sMemAddr in Flags) or (sProcAsm in Flags) then
        CodeBuf.Add(FuncName + _REFF + ' .proc (' + LowerCase(Buffer) + ') .reg')
      else
        CodeBuf.Add(FuncName + _REFF + ' .proc (' + LowerCase(Buffer) + ') .var');

      if not lInclude
         and (LowerCase(Copy(ProcBuf[ProcCount], 5, Length(ProcBuf[ProcCount]) - 4)) = LowerCase(FuncName)) then
      begin
        lMainProc := True
      end else begin
        lMainProc := False;
      end;      

      n := 0;
      
      if not (sMemAddr in Flags) and not (sProcAsm in Flags) then
      begin
        for j := 1 to GVarCnt do
        begin        
          //for n1 := 0 to ParamList.Count - 1 do          
          //begin
            if (ParamList.IndexOf(GVar[j].VarName) >= 0) and (GVar[j].Scope = 'L') then
            //if (UpperCase(GVar[j].VarName) = UpperCase(ParamList[n1])) and (GVar[j].Scope = 'L') then
            begin
              Inc(n);
              CodeBuf.Add(' .var ' + GVar[j].VarName + _EFF + ' .' + GVar[j].ML_type);
            
              (*
              if PrgVar.SB <> _SB_PROC_ML then
              begin
                if System.Pos(GVar[j].VarType, 'T1T2') > 0 then  // BYTE, CHAR
                  CodeBuf.Add(' mva b_param' + IntToStr(n) + ' ' + GVar[j].VarName + _EFF)
                else
                  CodeBuf.Add(' mwa w_param' + IntToStr(n) + ' ' + GVar[j].VarName + _EFF);
              end;
              *)
            end;
          //end;
        end;
      end;
      
      ParamList.Free;
    
    // Routine has no parameters
    //  
    end else begin
      lMainProc := False;           
      
      if not lInclude then
      begin
        if UpperCase(Copy(ProcBuf[ProcCount], 5, Length(ProcBuf[ProcCount]) - 4)) = UpperCase(FuncName) then
        begin
          CodeBuf.Add(FuncName + _REFF);
          lMainProc := True;
        end else begin
          if PrgVar.SB <> _SB_PROC_ML then
            CodeBuf.Add(FuncName + _REFF + ' .proc');
        end;
      end else begin      
        if PrgVar.SB <> _SB_PROC_ML then
          CodeBuf.Add(FuncName + _REFF + ' .proc');
      end;    
    end;
    
    if sMemAddr in flags then
    begin
      flags := flags - [sMemAddr];
      CodeBuf.Add(' jsr ' + MemAddr);      
      CodeBuf.Add(' rts');
      CodeBuf.Add('');
      CodeBuf.Add(' .endp');
    end;              
  end;  
end;

{
  Procedure name : sc_Return
  Description    : Handles Action! RETURN statement
  Parameters     : None
}
procedure sc_Return;
var
  n : Integer;
begin
  if (System.Pos('RETURN', UpperCase(TextBuf[CurLine])) > 0) then  // or (sMemAddr in flags) then
  begin
    if (System.Pos('(', UpperCase(TextBuf[CurLine])) > 0) and (System.Pos(')', UpperCase(TextBuf[CurLine])) > 0) then
    begin
      for n := 1 to GVarCnt do
      begin      
        if System.Pos(UpperCase(GVar[n].VarName), UpperCase(TextBuf[CurLine])) > 0 then
        begin
          //if not (sProcAsm in Flags) then
          if not ProcML_start then
            CodeBuf.Add(' mwa ' + GVar[n].VarName + _EFF + ' STORE1');
                    
          Break;
        end;
      end;
    end;     

    (*
    if sMemAddr in flags then
    begin
      CodeBuf.Add(' jsr ' + MemAddrCarry);
      flags := flags - [sMemAddr];
    end;
    *)

    if not lMainProc then
    begin
      CodeBuf.Add(' rts');
      CodeBuf.Add('');
      CodeBuf.Add(' .endp');
      //CodeBuf.Add('');
    end;    
  end;
end;

{
  Function name : FuncCheck
  Description   : Checks for Action! reserved standard function names in statements
  Parameters    : None
  Returns       : Returns True if no function name was found, otherwise False is returned 
}
function FuncCheck : Boolean;
var
  n : Byte;
begin
  Result := True;
  
  for n := 0 to FuncList.Count - 1 do
  begin
    if System.Pos(UpperCase(FuncList[n]) + '(', UpperCase(TextBuf[CurLine])) > 0 then
    begin
      Result := False;
      Break;
    end;
  end;
end;

{
  Function name : FuncCheck
  Description   : Checks for Action! reserved standard function names in statements
  Parameters    : None
  Returns       : Returns True if function name was found, otherwise False is returned 
}
function FuncCheckTrue(stmt : String) : Boolean;
var
  n : Byte;
begin
  Result := False;
  stmt := Trim(stmt);
  
  for n := 0 to FuncList.Count - 1 do
  begin
    if System.Pos(UpperCase(FuncList[n]) + '(', UpperCase(stmt)) > 0 then
    begin
      Result := True;
      Break;
    end;
  end;
end;

{
  Procedure name : sc_Var
  Description    : Handles Action! variable declarations
  Parameters     : None
}
procedure sc_Var;
var
  VarPos, n, j, k : LongInt;
  MemAddr, Buffer, VarType, VarX, Str1, Str2 : String;
  EquPos : Byte;
begin
  if (System.Pos('[', TextBuf[CurLine]) > 0) and (System.Pos('TYPE ', UpperCase(TextBuf[CurLine])) > 0) then
  begin
    Exit;
  end;
  
  if not boolType then
  begin
    for j := 0 to VarTypes.Count - 1 do
    begin
      VarPos := System.Pos(UpperCase(VarTypes[j]) + ' ', UpperCase(TextBuf[CurLine]));
      
      if (VarPos > 0) and VarDeclCheck(TextBuf[CurLine]) then
      begin      
        Buffer := Trim(TextBuf[CurLine]);
        VarType := UpperCase(Extract(Buffer, ' ', 1));
        
        if VarType = VarTypes[j] then
        begin              
          if (VarType = 'CARD') or (VarType = 'INT') then
            VarType := 'word'
          else if VarType = 'CHAR' then
            VarType := 'byte';
          
          VarX := LowerCase(VarType);
        end;
          
        VarType := Extract(Buffer, ' ', 2);
        Split(VarType, ',', []);            
        
        for n := 0 to StrBuf.Count - 1 do
        begin
          MemAddr := '';
          
          if System.Pos('=', StrBuf[n]) < 1 then
            Str2 := '-1'
          else begin          
            Buffer := Strip(StrBuf[n], ' ');
            Str1 := Extract(Buffer, '=', 1);
            Str2 := Extract(Buffer, '=', 2);
            k := System.Pos('[', Str2);
            
            if k < 1 then
              MemAddr := Str2
            else begin
              Str2 := Copy(Str2, k + 1, Length(Str2) - k - 1);
              PrgVar.SB := _SB_ARRAY_SET;
            end;                    
            
            StrBuf[n] := Str1;        
          end;
           
          if System.Pos('"', StrBuf[n]) < 1 then
          begin
            Inc(GVarCnt);
            GVar[GVarCnt].VarType    := SetType(VarTypes[j]);
            GVar[GVarCnt].ParentType := SetType(VarTypes[j]);
            GVar[GVarCnt].OrigType   := SetType(VarTypes[j]);        
            GVar[GVarCnt].VarName    := StrBuf[n];
            GVar[GVarCnt].Location   := '';
            GVar[GVarCnt].Value      := MemAddr;
            GVar[GVarCnt].Dim        := 0;
            GVar[GVarCnt].InitValue  := StrToInt(Str2);
            GVar[GVarCnt].ML_type    := VarX;
          end;
        end;
          
        Break;
      end;
    end;
    
    for j := 1 to GVarCnt2 do
    begin
      if System.Pos(GVar2[j].VarType, 'T5T6T7T8T10') > 0 then Continue;  // ARRAY, POINTER

      VarPos := System.Pos(UpperCase(GVar2[j].VarName), UpperCase(TextBuf[CurLine]));
      if (VarPos > 0) and VarDeclCheck(TextBuf[CurLine]) then
      begin      
        Buffer := Trim(TextBuf[CurLine]);
        VarType := Extract(Buffer, ' ', 1);
        VarType := UpperCase(VarType);
              
        if UpperCase(VarType) = UpperCase(GVar2[j].VarName) then
        begin
          VarX := VarType;
          
          if (VarType = 'CARD') or (VarType = 'INT') then
            VarX := 'word'
          else if (VarType = 'BYTE') or (VarType = 'CHAR') then
            VarX := 'byte';
        end;
           
        VarType := Extract(Buffer, ' ', 2);
        Split(VarType, ',', []);
        MemAddr := '';
        
        for n := 0 to StrBuf.Count - 1 do
        begin
          if System.Pos('"', StrBuf[n]) < 1 then
          begin              
            Inc(GVarCnt);
            GVar[GVarCnt].VarType    := SetType(VarType);
            GVar[GVarCnt].ParentType := SetType(GVar2[j].VarType);
            GVar[GVarCnt].OrigType   := SetType(GVar2[j].VarName);        
            GVar[GVarCnt].VarName    := StrBuf[n];
            GVar[GVarCnt].Location   := '';
            GVar[GVarCnt].Value      := MemAddr;
            GVar[GVarCnt].Dim        := 0;
            GVar[GVarCnt].ML_type    := VarX;
            
            SData[Cnt] := GVar[GVarCnt].VarName + _EFF + ' ' + VarX + _EFF;
            Inc(Cnt);
          end;
        end;
        
        Break;
      end;
    end;
  
  // TYPE declaration variables
  //
  end else begin
    for j := 0 to VarTypes.Count - 1 do
    //for j := 1 to _VAR_TYPES do
    begin
      VarPos := System.Pos(UpperCase(VarTypes[j]) + ' ', UpperCase(TextBuf[CurLine]));      
      
      if (VarPos > 0) and VarDeclCheck(TextBuf[CurLine]) then
      begin      
        Buffer := Trim(TextBuf[CurLine]);
        VarType := Extract(Buffer, ' ', 1);
        
        if UpperCase(VarType) = VarTypes[j] then
        begin
          if (UpperCase(VarType) = 'CHAR') or (UpperCase(VarType) = 'BYTE') then
            VarType := 'byte'
          else if (UpperCase(VarType) = 'CARD') or (UpperCase(VarType) = 'INT') then
            VarType := 'word';                      
        end;
        
        Buffer := Extract(Buffer, ' ', 2);
        Split(Buffer, ',', []);
        
        for n := 0 to StrBuf.Count - 1 do
        begin
          EquPos := System.Pos('=', StrBuf[n]);
          MemAddr := '';
          
          if EquPos > 0 then 
          begin
            MemAddr := Copy(StrBuf[n], EquPos + 1, Length(StrBuf[n]) - EquPos);                        
            StrBuf[n] := Copy(StrBuf[n], 1, EquPos - 1);
          end;
          
          if VarType = 'byte' then
            Inc(TypeMemCnt, 1)
          else
            Inc(TypeMemCnt, 2);
          
          if System.Pos(']', StrBuf[n]) > 0 then
          begin
            StrBuf[n] := Copy(StrBuf[n], 1, Length(StrBuf[n]) - 1);            
            boolType := False;
            PrgVar.SB := _SB_ARRAY_SET;
          end;
          
          Inc(GVarCnt);
          GVar[GVarCnt].VarType    := SetType(VarTypes[j]);
          GVar[GVarCnt].ParentType := SetType(GVar2[GVarCnt2].VarName);
          GVar[GVarCnt].OrigType   := SetType(VarTypes[j]);
          GVar[GVarCnt].VarName    := StrBuf[n];
          GVar[GVarCnt].Location   := IntToStr(GVarCnt2);
          GVar[GVarCnt].Value      := MemAddr;
          GVar[GVarCnt].Dim        := 0;
          GVar[GVarCnt].ML_type := VarType;
        end;
        
        Break;
      end;
    end;
    
    if (System.Pos(']', TextBuf[CurLine]) > 0) and boolType then
      boolType := False;
  end;
end;

{
  Procedure name : sc_Var2
  Description    : Handles Action! ARRAY, TYPE and POINTER variable declarations
  Parameters     : None
}
procedure sc_Var2;
var
  VarPos, n, n4, j : LongInt;
  CharPos : String[1];
  Buffer, VarType, VarX, Str1, Str2, xType, xVar : String;
  boolFirstSpace : Boolean;
begin
  // TYPE variable
  //
  if (System.Pos('TYPE ', UpperCase(TextBuf[CurLine])) > 0) then
  begin
    if System.Pos('"', TextBuf[CurLine]) > 0 then Exit;
    
    CodeBuf.Add('; Handling TYPE variables');
    CodeBuf.Add(' .var struct_ptr_var .word');
    CodeBuf.Add('');
       
    boolType := True;
    TypeMemCnt := 0;
    Buffer := Trim(TextBuf[CurLine]);
    Str1 := Extract(Buffer, '=', 1);
    VarType := Extract(Str1, ' ', 2);
    Str2 := Extract(Buffer, '=', 2);
    Split(Str1, ' ', []);
    Str2 := Extract(Str2, '[', 2);
    PrgVar.SB := _SB_TYPE;
    
    Inc(GVarCnt2);
    
    // TYPE members, delimited with ,
    //    
    if System.Pos(',', Str2) > 0 then
    begin
      Split(Str2, ',', []);
      
      for n := 0 to StrBuf.Count - 1 do
      begin
        if n <> 0 then
          xVar := StrBuf[n]
        else begin
          Str1 := Extract(StrBuf[n], ' ', 1);
          xType := Str1;
          Str2 := Extract(StrBuf[n], ' ', 2);
          xVar := Str2;
        end;
        
        sc_var3(UpperCase(xType), xVar, VarType);
      end;

    // TYPE lone member
    end else begin
      Str1 := Extract(Str2, ' ', 1);
      Str2 := Extract(Str2, ' ', 2);
      sc_var3(UpperCase(Str1), Str2, VarType);
    end;
    
    GVar2[GVarCnt2].VarType  := SetType('type');
    GVar2[GVarCnt2].ParentType := SetType('type');
    GVar2[GVarCnt2].OrigType := SetType('type');
    GVar2[GVarCnt2].VarName  := VarType;
    GVar2[GVarCnt2].Location := '';
    GVar2[GVarCnt2].Value    := '';
    GVar2[GVarCnt2].Dim      := 0;
    GVar2[GVarCnt2].ML_type := '';
    
    Exit;
  end;

  Buffer := Strip(UpperCase(TextBuf[CurLine]), ' ');
  if (System.Pos('PRINT(', Buffer) > 0) or (System.Pos('PRINTE(', Buffer) > 0) then
  begin
    Exit;
  end;

  boolPtr := False;

  for j := 0 to VarTypes2.Count - 1 do
  //for j := 1 to _VAR_TYPES2 do
  begin    
    VarPos := System.Pos(UpperCase(VarTypes2[j]) + ' ', UpperCase(TextBuf[CurLine]));
    
    if System.Pos('ARRAY', UpperCase(VarTypes2[j])) > 0 then
    begin
      PrgVar.SB := _SB_ARRAY_SET;
      boolArray := True;
    end else if System.Pos(' POINTER ', UpperCase(VarTypes2[j])) > 0 then
    begin
      boolPtr := True;
      boolFirstSpace := True;
      VarPos := System.Pos(UpperCase(VarTypes2[j]), UpperCase(TextBuf[CurLine]));
    end;
    
    if (VarPos > 0) then
    begin
      Buffer := Trim(TextBuf[CurLine]);
      VarType := ''; VarX := '';
      
      // CARD ARRAY values=[1 2 3 4 5 6 7]
      if (System.Pos(' ARRAY ', UpperCase(Buffer)) > 0)
         and (System.Pos('=', Buffer) > 0) and (System.Pos('[', Buffer) > 0) then
      begin
        Buffer := Extract(Buffer, '=', 1);
        PrgVar.SB := _SB_ARRAY_SET;
      end;
      
      // BYTE ARRAY str1="Text"
      if (System.Pos(' ARRAY ', UpperCase(Buffer)) > 0) and (System.Pos('=', Buffer) > 0)
         and (System.Pos('"', Buffer) > 0) then
      begin
        VarType := ExtractText(Buffer, '"', '"');
        Split(UpperCase(TextBuf[CurLine]), 'ARRAY', []);
        Split(StrBuf[1], '=', []);
        
        Inc(GVarCnt2);
        GVar2[GVarCnt2].VarType  := 'T5';
        GVar2[GVarCnt2].ParentType := 'T5';
        GVar2[GVarCnt2].OrigType := 'T5';
        GVar2[GVarCnt2].VarName  := StrBuf[0];
        GVar2[GVarCnt2].Location := 'T5';
        GVar2[GVarCnt2].Value := VarType;
        GVar2[GVarCnt2].ML_type := '';
        GVar2[GVarCnt2].Dim := 0;  // Length(VarType);
        
        // BYTE ARRAY str1(4)="Text"
        if (System.Pos('(', Buffer) > 0) and (System.Pos(')', Buffer) > 0) then
        begin
          //n6 := StrToInt(ExtractText(Buffer, '(', ')'));
          VarType := Extract(StrBuf[0], '(', 1);
          GVar2[GVarCnt2].VarName := VarType;
          GVar2[GVarCnt2].Dim := StrToInt(ExtractText(Buffer, '(', ')'));
        end;
      
      end else begin      
        for n := 1 to Length(Buffer) do
        begin
          CharPos := Copy(Buffer, n, 1);
          VarType := VarType + CharPos;
          
          // ARRAY          
          if not boolPtr then
          begin                     
            if (CharPos = ',') or (n = Length(Buffer)) then
            begin            
              if n = Length(Buffer) then
                VarType := Trim(Copy(VarType, 1, Length(VarType)))
              else
                VarType := Trim(Copy(VarType, 1, Length(VarType) - 1));
              
              // Extract variable and its dimension
              if System.Pos('(', VarType) < 1 then
                n4 := 0
              else begin
                n4 := StrToInt(ExtractText(VarType, '(', ')'));
                VarType := Extract(VarType, '(', 1);
                PrgVar.SB := _SB_ARRAY;
              end;              
              
              // It is ARRAY set
              if System.Pos('[', TextBuf[CurLine]) > 0 then
                PrgVar.SB := _SB_ARRAY_SET;
              
              // Set variable parameters
              Inc(GVarCnt2);
              GVar2[GVarCnt2].VarType  := SetType(VarX);
              GVar2[GVarCnt2].ParentType := SetType(VarTypes2[j]);
              GVar2[GVarCnt2].OrigType := SetType(VarTypes2[j]);
              GVar2[GVarCnt2].VarName  := VarType;
              
              if PrgVar.SB = _SB_ARRAY_SET then
                GVar2[GVarCnt2].Location := 'SET'
              else
                GVar2[GVarCnt2].Location := '';
              
              GVar2[GVarCnt2].ML_type := VarX;
              GVar2[GVarCnt2].Value := '';
              GVar2[GVarCnt2].Dim := n4;
              RecPtrVar.ArrayDim := n4;
              VarType := '';      
            end else if UpperCase(VarType) = UpperCase(VarTypes2[j]) then
            begin        
              VarX := LowerCase(VarType);
              VarType := '';
            end;
            
          // POINTER
          //
          end else begin
            if (CharPos = ' ') and boolFirstSpace then
            begin
              boolFirstSpace := False;
              Inc(GVarCnt2);
              GVar2[GVarCnt2].OrigType := SetType(Copy(VarType, 1, Length(VarType) - 1));
              VarType := '';
            end;
            
            if (CharPos = ' ') and not boolFirstSpace then
            begin
              GVar2[GVarCnt2].VarType := SetType(Copy(VarType, 1, Length(VarType) - 1));
              GVar2[GVarCnt2].ParentType := SetType(Copy(VarType, 1, Length(VarType) - 1));
              VarType := '';
            end;
            
            if (n = Length(Buffer)) then
            begin
              if System.Pos(GVar2[GVarCnt2].OrigType, 'T1T2') > 0 then  // BYTE, CHAR            
                VarX := 'byte'
              else if System.Pos(GVar2[GVarCnt2].OrigType, 'T3T4') > 0 then  // INT, CARD
                VarX := 'word'
              else
                VarX := GVar2[GVarCnt2].OrigType;
                                     
              VarType := Trim(Copy(VarType, 1, Length(VarType)));
              GVar2[GVarCnt2].VarName  := VarType;
              GVar2[GVarCnt2].Location := '';
              GVar2[GVarCnt2].Value    := IntToStr(PtrCnt);
              GVar2[GVarCnt2].Dim      := 0;
              GVar2[GVarCnt2].ML_type := VarType;
              Inc(PtrCnt, 2);
              
              Inc(GVarCnt);
              GVar[GVarCnt].VarType  := SetType(VarX);
              GVar[GVarCnt].ParentType := SetType('POINTER');
              GVar[GVarCnt].OrigType := SetType(VarType);
              GVar[GVarCnt].VarName  := VarType + '_ptr';
              GVar[GVarCnt].Location := '';
              GVar[GVarCnt].Value    := '';
              GVar[GVarCnt].Dim      := 0;
              GVar[GVarCnt].ML_type := VarX;
              
              VarType := '';        
            end;
          end;
        end;   
      end;                      
    end;               
  end;
end;

{
  Procedure name : sc_Var3
  Description    : Handles Action! variable declarations
  Parameters     : xType
                   xVar
                   xVarType
}
procedure sc_Var3(xType, xVar, xVarType : String);
var
  VarPos, j : LongInt;
  MemAddr, VarType : String;
  EquPos : Byte;
begin
  for j := 0 to VarTypes.Count - 1 do
  begin
    VarPos := System.Pos(UpperCase(VarTypes[j]) + ' ', UpperCase(xType + ' ' + xVar));
    
    if (VarPos > 0) and VarDeclCheck(UpperCase(xType + ' ' + xVar)) then
    begin
      VarType := xType;
      
      if UpperCase(VarType) = UpperCase(VarTypes[j]) then
      begin
        if (VarType = 'BYTE') or (VarType = 'CHAR') then
          VarType := 'byte'
        else if (VarType = 'INT') or (VarType = 'CARD') then
          VarType := 'word';
      end;
      
      EquPos := System.Pos('=', xVar);
      MemAddr := '';
      
      if EquPos > 0 then 
      begin
        MemAddr := Copy(xVar, EquPos + 1, Length(xVar) - EquPos);
        xVar := Copy(xVar, 1, EquPos - 1);
      end;
      
      if System.Pos(']', xVar) > 0 then
      begin
        xVar := Copy(xVar, 1, Length(xVar) - 1);
        boolType := False;
      end;
      
      if VarType = 'byte' then
        Inc(TypeMemCnt, 1)
      else
        Inc(TypeMemCnt, 2);
      
      Inc(GVarCnt);
      GVar[GVarCnt].VarType    := SetType(VarTypes[j]);
      GVar[GVarCnt].ParentType := SetType(xVarType);
      GVar[GVarCnt].OrigType   := SetType(VarTypes[j]);
      GVar[GVarCnt].VarName    := xVar;
      GVar[GVarCnt].Location   := IntToStr(GVarCnt2);
      GVar[GVarCnt].Value      := MemAddr;
      GVar[GVarCnt].Dim        := 0;
      GVar[GVarCnt].ML_type    := VarType;      
            
      Break;
    end;
  end;
  
  if (System.Pos(']', xType + ' ' + xVar) > 0) and boolType then
  begin
    boolType := False;
  end;
end;

{
  Procedure name : sc_Array
  Description    : Handles Action! ARRAY set variable declarations
  Parameters     : None
}
procedure sc_Array;
var
  Buffer : String;
  n : Integer;
begin
  if (System.Pos('[', TextBuf[CurLine]) > 0) then  //and (PrgVar.SB = _SB_NULL) then
  begin
    if System.Pos('ARRAY ', UpperCase(TextBuf[CurLine])) > 0 then
    begin
      PrgVar.SB := _SB_ARRAY_SET;
      ArraySet_start := True;
    end;
  end;

  if ArraySet_start then
  begin   
    // Array elements in [] block
    if (System.Pos('[', TextBuf[CurLine]) > 0) and (System.Pos(']', TextBuf[CurLine]) > 0) then
    begin
      Buffer := ExtractText(TextBuf[CurLine], '[', ']');      
      ArraySet_start := False;
      PrgVar.SB := _SB_NULL;
    
    // Start of array
    end else if System.Pos('[', TextBuf[CurLine]) > 0 then    
    begin
      Split(TextBuf[CurLine], '[', []);
      Buffer := StrBuf[1];
    
    // End of array
    end else begin    
      Buffer := TextBuf[CurLine];
      if System.Pos(']', Buffer) > 0 then
      begin
        Buffer := Replace(Buffer, ']', ' ');
        ArraySet_start := False;
        PrgVar.SB := _SB_NULL;
      end;  
    end;    
    
    Buffer := Trim(Buffer);    
    Split(Buffer, ' ', []);
    
    Buffer := '';

    // Loop through elements of the array
    //
    for n := 0 to StrBuf.Count - 1 do
    begin
      // Blank elements deprecated
      if Trim(StrBuf[n]) = '' then Continue;
      
      // Get element from the array
      if n = 0 then
        Buffer := StrBuf[n]
      else
        Buffer := Buffer + ', ' + StrBuf[n];
    end;

    Buffer := Buffer + ', ';
        
    // Store array elements
    if StrBuf[0] <> '' then
      GVar2[GVarCnt2].Value := GVar2[GVarCnt2].Value + Buffer;
  end;
end;

{
  Procedure name : sc_VarExpr
  Description    : Handles Action! variable expressions
  Parameters     : None
}
procedure sc_VarExpr;
var
  Str1, Str2, Str3, Str4, VarName, TempBuf, TempBuf2 : String;
  j, n, r, n5, n1, p : Integer;
  arrOper : TStringList;  
begin
  if Pos('=', TextBuf[CurLine]) < 1 then Exit;
      
  for n := 1 to GVarCnt do
  begin
    Str1 := Strip(TextBuf[CurLine], ' ');
    n5 := System.Pos('==', Str1);
        
    if n5 > 0 then
    begin      
      Str2 := Copy(Str1, n5 + 1, Length(Str1) - (n5 + 2));
      Str1 := Extract(Str1, '==', 1);
    end else if (System.Pos('=-', Str1) > 0) then begin      
      str2 := 'M=-' + Extract(Str1, '=-', 2);
      str1 := Extract(Str1, '=-', 1);
    end else begin
      Split(Str1, '=', []);
      Str1 := StrBuf[0];
      Str2 := StrBuf[1];
    end;       
        
    if (UpperCase(GVar[n].VarName) = UpperCase(Str1)) and FuncCheck and 
       ExprCheck(UpperCase(TextBuf[CurLine])) then
    begin      
      TempBuf := ''; TempBuf2 := '';
      
      for p := 1 to GVarCnt2 do
      begin
        // Handle ARRAY variables
        if System.Pos(UpperCase(GVar2[p].VarName) + '(', UpperCase(Str2)) > 0 then
        begin
          Str2 := Replace(Str2, '(', '[');
          Str2 := Replace(Str2, ')', ']');
          GVar2[p].Value := Str2;
          TempBuf := GVar2[p].VarType;
          TempBuf2 := GVar2[p].VarName;
          break;
        end;
      end;    
                   
      if (System.Pos('M=-', Str2) > 0) then
      begin
        lMath := True;
        //CodeBuf.Add(' mwa #65535 STORE1');
        CodeBuf.Add(' sub16 #65535 ' + GVar[n].VarName + _EFF);
        CodeBuf.Add(' mwa STORE1 ' + GVar[n].VarName + _EFF);
        CodeBuf.Add(' inw ' + GVar[n].VarName + _EFF);
      end else begin
        if (System.Pos('=+', UpperCase(Str2)) > 0) or (System.Pos('=-', UpperCase(Str2)) > 0) or (System.Pos('=*', UpperCase(Str2)) > 0)
           or(System.Pos('=/', UpperCase(Str2)) > 0) or (System.Pos('=!', UpperCase(Str2)) > 0) or (System.Pos('=XOR', UpperCase(Str2)) > 0)
           or (System.Pos('=LSH', UpperCase(Str2)) > 0) or (System.Pos('=RSH', UpperCase(Str2)) > 0)
           or (System.Pos('=&', UpperCase(Str2)) > 0) or (System.Pos('=%', UpperCase(Str2)) > 0) then
        begin          
          Delete(Str2, 1, 1);
          Str2 := Str1 + Str2;
          MathExpr(GVar[n].VarType, Str1, Str2, 1, n);
        end else begin          
          if System.Pos('(', TextBuf[CurLine]) > 0 then
          begin
            //j = data(i)
            Str4 := Extract(Str2, '[', 1);
            Str3 := Extract(Str2, '[', 2);
            Str3 := Copy(Str3, 1, Length(Str3) - 1);
            
            if IsNumber(Str3[1]) then
            begin
              MathExpr(GVar[n].VarType, Str1, Str2, 1, n);
            end else begin
              if sFor in flags then
              //if ForFlag then
              begin
                CodeBuf.Add(' ldy array_index_' + TempBuf2 + _EFF);
                
                if System.Pos(TempBuf, 'T5T6') > 0 then  // BYTE ARRAY or CHAR ARRAY
                  CodeBuf.Add(' mva ' + Str4 + _EFF + ',y ' + GVar[n].VarName + _EFF)
                else begin
                  CodeBuf.Add(' mwa ' + Str4 + _EFF + ',y ' + GVar[n].VarName + _EFF);
                  CodeBuf.Add(' inc array_index_' + TempBuf2 + _EFF);
                end;
              end;
            end;
          end else begin
            //
            // Expression manipulation process
                                                          
            // Check and index operators
            //
            arrOper := TStringList.Create;
    
            try
              for r := 0 to Length(Str2) - 1 do
              begin
                if Str2[r] = '+' then
                  arrOper.Add('1')
                else if Str2[r] = '-' then
                  arrOper.Add('2')
                else if Str2[r] = '*' then
                  arrOper.Add('3')
                else if Str2[r] = '/' then
                  arrOper.Add('4');
              end;
              
              // Generate expression output                       
              //
              if arrOper.Count = 0 then
              begin
                MathExpr(GVar[n].VarType, Str1, Str2, 1, n);
                Exit;
              end;
  
              SplitEx2(Str2, '+', '-', '*', '/');
                                                                          
              for r := 0 to arrOper.Count - 1 do
              begin
                if r = 0 then
                begin
                  if arrOper[r] = '1' then
                    Str2 := strbuf2[0] + '+' + strbuf2[1]
                  else if arrOper[r] = '2' then
                    Str2 := strbuf2[0] + '-' + strbuf2[1]                  
                  else if arrOper[r] = '3' then
                    Str2 := strbuf2[0] + '*' + strbuf2[1]                  
                  else if arrOper[r] = '4' then
                    Str2 := strbuf2[0] + '/' + strbuf2[1]                  
                end else begin
                  if arrOper[r] = '1' then
                    Str2 := Str1 + '+' + strbuf2[r + 1]
                  else if arrOper[r] = '2' then
                    Str2 := Str1 + '-' + strbuf2[r + 1]
                  else if arrOper[r] = '3' then
                    Str2 := Str1 + '*' + strbuf2[r + 1]
                  else if arrOper[r] = '4' then
                    Str2 := Str1 + '/' + strbuf2[r + 1]
                end;
                        
                MathExpr(GVar[n].VarType, Str1, Str2, 1, n);
              end;
              
            finally
              if Assigned(arrOper) then FreeAndNil(arrOper);
            end;
          end;
        end;
      end;

    // TYPE variable expression
    //
    end else if (System.Pos('.' + UpperCase(GVar[n].VarName), UpperCase(Str1)) > 0) and FuncCheck and
      ExprCheck(UpperCase(TextBuf[CurLine])) then
    begin
      if System.Pos('"', TextBuf[CurLine]) > 0 then Exit;
            
      for j := 1 to GVarCnt do
      begin            
        if GVar[n].ParentType = GVar[j].OrigType then
        begin
          CodeBuf.Add(mvwa(GVar[n].VarType) + AsmStrNum(Str2) + ' ' + GVar[j].VarName + _EFF + '.' + GVar[n].VarName + _EFF);
          Exit;
        end;
      end;

    // TYPE variable expression
    //
    end else if (System.Pos('.' + UpperCase(GVar[n].VarName), UpperCase(Str2)) > 0) and FuncCheck and
      ExprCheck(UpperCase(TextBuf[CurLine])) then
    begin
      if System.Pos('"', TextBuf[CurLine]) > 0 then Exit;
            
      for j := 1 to GVarCnt do
      begin            
        if GVar[n].ParentType = GVar[j].OrigType then
        begin
          CodeBuf.Add(mvwa(GVar[n].VarType) + GVar[j].VarName + _EFF + '.' + GVar[n].VarName + _EFF + ' ' + AsmStrNum(Str1));          
          Exit;
        end;
      end;
    end;    
  end;

  for n := 1 to GVarCnt2 do
  begin
    Str1 := TextBuf[CurLine];
    Split(TextBuf[CurLine], '=', []);
    Str1 := StrBuf[0];
    Str2 := StrBuf[1];
    VarName := Str1;

    if System.Pos(UpperCase(GVar2[n].VarName) + '=@', UpperCase(TextBuf[CurLine])) > 0 then
    begin      
      if FuncCheck and ExprCheck(UpperCase(TextBuf[CurLine])) then
      begin
        for p := 1 to GVarCnt do
        begin
          if System.Pos('@' + UpperCase(GVar[p].VarName), Str2) > 0 then
          begin          
            CodeBuf.Add(' lda #<' + GVar[p].VarName + _EFF);
            CodeBuf.Add(' sta $c0');
            CodeBuf.Add(' lda #>' + GVar[p].VarName + _EFF);
            CodeBuf.Add(' sta $c1');            
            CodeBuf.Add('');
            
            if System.Pos(GVar2[n].VarType, 'T10') > 0 then  // POINTER
            //if UpperCase(Trim(GVar2[n].VarType)) = 'POINTER' then
            begin            
              for r := 1 to GVarCnt do
              begin                
                // POINTER
                if (System.Pos(GVar[r].ParentType, 'T10') > 0) and (UpperCase(GVar[r].OrigType) = UpperCase(GVar2[n].VarName)) then
                //if (GVar[r].ParentType = 'POINTER') and (GVar[r].OrigType = GVar2[n].VarName) then
                begin              
                  CodeBuf.Add(' mwa $c0 ' + GVar[r].VarName + _EFF);
                  CodeBuf.Add('');
                  Break;
                end;
              end;            
            end;
            
            Break;
          end;
        end;
      end;
      
      Exit;
      
    end else if System.Pos(UpperCase(GVar2[n].VarName) + '^=', UpperCase(TextBuf[CurLine])) > 0 then
    begin
      if FuncCheck and ExprCheck(UpperCase(TextBuf[CurLine])) then
      begin
        if System.Pos(GVar2[n].VarType, 'T10') > 0 then  // POINTER
        //if UpperCase(Trim(GVar2[n].VarType)) = 'POINTER' then
        begin            
          for r := 1 to GVarCnt do
          begin
            // POINTER
            if (System.Pos(GVar[r].ParentType, 'T10') > 0)
               and (UpperCase(GVar[r].OrigType) = UpperCase(GVar2[n].VarName)) then
            //if (GVar[r].ParentType = 'POINTER') and (GVar[r].OrigType = GVar2[n].VarName) then
            begin              
              CodeBuf.Add(' mwa #' + Str2 + ' ' + GVar[r].VarName + _EFF);
              CodeBuf.Add(' lda ' + GVar[r].VarName + _EFF);
              CodeBuf.Add(' ldy #0');
              CodeBuf.Add(' sta ($c0),y');
              Break;
            end;
          end;            
        end;        
      end;
    
    // Handling TYPE, ARRAY and POINTER variables
    //      
    end else if System.Pos(UpperCase(GVar2[n].VarName), UpperCase(Str1)) > 0 then
    begin
      // Handling TYPE record ARRAY POINTER variables
      // ENTRY=DATA+(0*SIZE)
      if (System.Pos('(', Str1) < 1) and (System.Pos('.', Str1) < 1)
         and (System.Pos(GVar2[n].VarType, 'T10') > 0) then  // POINTER
      begin
        for j := 1 to GVarCnt2 do
        begin
          if (System.Pos(UpperCase(GVar2[j].VarName), UpperCase(Str2)) > 0)
             and (System.Pos(GVar2[j].OrigType, 'T5T6T7T8') > 0)  // ARRAY
             and (System.Pos(GVar2[n].VarType, 'T10') > 0) then   // POINTER
          begin
            Str3 := Str2;
            Str4 := Extract(Str3, '+', 1);
            Str3 := Extract(Str3, '+', 2);
            
            if Str3 <> '' then
            begin
              RecPtrVar.Name := GVar2[n].VarName;
              RecPtrVar.Dim := StrToInt(Str3) div TypeMemCnt;
              RecPtrVar.Flag := True;
              
              for n1 := 1 to GVarCnt2 do
              begin
                if (UpperCase(GVar2[n1].VarName) = UpperCase(Str4))
                   and (PtrData <> GVar2[n].VarName + '_sv') then
                begin                                  
                  n5 := (GVar2[n1].Dim div TypeMemCnt) - 1;
                  TypeMemDim := n5;
                  PtrData := GVar2[n].VarName + '_sv';
                  SData[Cnt] := PtrData + _EFF + ' dta ' + GVar2[n].OrigType + _EFF + ' [' + IntToStr(n5) + ']';                    
                  Inc(Cnt);
                  
                  Break;              
                end;
              end;                        
            end;
          end
        end
      
      // Handling ARRAY variables, not POINTERS
      //
      end else if (System.Pos('(', Str1) > 0) and (System.Pos(GVar2[n].VarType, 'T10') < 1) then
      begin
        VarName := Replace(VarName, '(', '[');
        VarName := Replace(VarName, ')', ']');
        GVar2[n].Value := VarName;       
        Str3 := Extract(VarName, '[', 2);
        Str4 := Extract(VarName, '[', 1);
        Str4 := Str4 + _EFF + '[' + Str3;
        Str3 := Copy(Str3, 1, Length(Str3) - 1);
        
        if IsNumber(Str3[1]) then
        begin          
          if System.Pos('"', Str2) < 1 then
            MathExpr(GVar2[n].VarType, Str4, Str2, 2, n)
          else begin
            PrgVar.Pointer := CurLine;
            Str2 := ExtractText(Str2, '"', '"');
            MathExpr(GVar2[n].VarType, Str4, Str2, 3, n);
          end
        end else begin
          if sFor in flags then
          //if ForFlag then
          begin
            CodeBuf.Add(' ldy array_index_' + GVar2[n].VarName + _EFF);
            
            if System.Pos(GVar2[n].VarType, 'T5T6') > 0 then  // BYTE and CHAR ARRAY
            begin                           
              CodeBuf.Add(' lda ' + AsmStrNum(Str2));
              CodeBuf.Add(' sta ' + GVar2[n].VarName + _EFF + ',y');
              CodeBuf.Add(' lda #0');
              CodeBuf.Add(' sta ' + GVar2[n].VarName + _EFF + '+1,y');              
            end else begin
              CodeBuf.Add(' mwa ' + AsmStrNum(Str2) + ' ' + GVar2[n].VarName + _EFF + ',y');
            end;            
          end;        
        end;

      // Handling TYPE record ARRAY POINTER variables
      //
      end else if (System.Pos(UpperCase(GVar2[n].VarName) + '.', UpperCase(Str1)) > 0)
                  and (System.Pos(GVar2[n].VarType, 'T10') > 0) then
      begin
        Split(Str1, '.', []);
        Str3 := AsmStrNum(Str2);
        
        for r := 1 to GVarCnt do
        begin
          if UpperCase(GVar[r].VarName) = UpperCase(StrBuf[1]) then
          begin
            if System.Pos('"', TextBuf[CurLine]) > 0 then Break;
            
            CodeBuf.Add(mvwa(GVar[r].VarType) + Str3 + ' ' + PtrData + _EFF + '[' + IntToStr(RecPtrVar.Dim) + '].' + StrBuf[1] + _EFF);
            Break;
          end;
        end;
      end      
    end;    
  end;
end;

{
  Procedure name : sc_Define
  Description    : Handles Action! DEFINE declaration statement
  Parameters     : None
}
procedure sc_Define;
var
  n1 : LongInt;
  Buffer : String;
  StrBufX : TStringList;
begin
  if (System.Pos('DEFINE ', UpperCase(TextBuf[CurLine])) > 0) then
  begin
    Buffer := ExtractText(TextBuf[CurLine], '"', '"');
    if (Buffer <> '') and (System.Pos('DEFINE', UpperCase(Buffer)) > 0) then
    begin
      Exit;
    end;
    
    Split(Trim(UpperCase(TextBuf[CurLine])), 'DEFINE', []);
    Buffer := StrBuf[1];
    Split(Buffer, ',', []);
    
    if StrBuf.Count = 1 then
    begin
      Buffer := Strip(StrBuf[0], ' ');
      Split(Buffer, '=', []);
      Buffer := ExtractText(StrBuf[1], '"', '"');
      
      if (StrToInt(Buffer) >= 0) and (StrToInt(Buffer) <= 255) then
        CodeBuf.Add(' .var ' + StrBuf[0] + _EFF + '=' + Buffer + ' .byte')
      else
        CodeBuf.Add(' .var ' + StrBuf[0] + _EFF + '=' + Buffer + ' .word');
    end else begin
      StrBufX := TStringList.Create;
      
      for n1 := 0 to StrBuf.Count - 1 do
        StrBufX.Add(StrBuf[n1]);
      
      for n1 := 0 to StrBufX.Count - 1 do
      begin
        Buffer := Strip(StrBufX[n1], ' ');
        Split(Buffer, '="', []);
        Buffer := Copy(StrBuf[1], 1, Length(StrBuf[1]) - 1);
        
        if (StrToInt(Buffer) >= 0) and (StrToInt(Buffer) <= 255) then
          CodeBuf.Add(' .var ' + StrBuf[0] + _EFF + '=' + Buffer + ' .byte')
        else
          CodeBuf.Add(' .var ' + StrBuf[0] + _EFF + '=' + Buffer + ' .word');
      end;
      
      StrBufX.Free;
    end;
  end;
end;

{
  Procedure name : sc_Include
  Description    : Handles Action! INCLUDE directive
  Parameters     : None
}
procedure sc_Include;
var
  Temp : String;
begin
  if System.Pos('INCLUDE "', UpperCase(TextBuf[CurLine])) > 0 then
  begin
    lInclude := True;
    lIncludeX := True;
    Temp := ExtractText(TextBuf[CurLine], '"', '"');
    
    if System.Pos(PathDelim, Temp) < 1 then
    //if (System.Pos('\', Temp) < 1) and (System.Pos('/', Temp) < 1) then
    begin
      if ExtractFileDir(meditEff_src_filename) <> '' then
        Temp := ExtractFileDir(meditEff_src_filename) + PathDelim + Temp;
    end;
    
    Inc(Icl);
    ASM_icl[icl] := Temp;
    Temp := StringReplace(ExtractFilename(Temp), '.EFF', '.asm', [rfIgnoreCase]);
    WriteLn(fASM, ' icl ' + AnsiQuotedStr(Temp, ''''));
  end;
end;

{
  Procedure name : sc_Data
  Description    : Mads words (labels) list
  Parameters     : None
}
procedure sc_Data;
var
  i : Byte;
begin
  i := 1;
  while SData[i] <> '' do
  begin
    WriteLn(fASM, SData[i]);
    Inc(i);
  end;
end;

{
  Procedure name : sc_ML_data
  Description    : Mads words (labels) list
  Parameters     : None
}
procedure sc_ML_data;
var
  i : Byte;
begin
  i := 1;
  while ProcML_data[i] <> '' do
  begin
    WriteLn(fASM, ProcML_data[i]);
    WriteLn(fASM, ProcML[i].Code);
    Inc(i);
  end;
end;

{
  Function name : CheckProcML
  Description   : Mads words (labels) list
  Parameters    : Proc
  Returns       :
}
function CheckProcML(Proc : String) : Boolean;
var
  bFound: Boolean = False;
  k : Integer;
begin  
  for k := 1 to ProcML_cnt do
  begin
    if UpperCase(ProcML[k].Name) = UpperCase(Copy(Proc, 5, Length(Proc) - 4)) then
    begin
      bFound := True;
      Break;
    end;
  end;
  
  result := bFound;            
end;

{
  Function name : GetMLAddress
  Description   : Mads words (labels) list
  Parameters    : Proc
  Returns       :
}
function GetMLAddress(Proc : String) : String;
var
  MLAddr: String = '';
  k: Integer;
begin  
  for k := 1 to ProcML_cnt do
  begin
    if UpperCase(ProcML[k].Name) = UpperCase(Copy(Proc, 5, Length(Proc) - 4)) then
    begin
      MLAddr := ProcML[k].Address;
      Break;
    end;
  end;
  
  result := MLAddr;            
end;
 
{
  Procedure name : sc_ProcTrack
  Description    : Handles Action! procedures (PROC) and functions (FUNC)
  Parameters     : None
}
procedure sc_ProcTrack;
var
  n, n1, j : Integer;
  Buffer, param, Proc : String;  
  Params : Array[1..21] of String;
  ParamInc : Integer;
begin
  if (System.Pos('PROC ', UpperCase(TextBuf[CurLine])) > 0)
     or (System.Pos('FUNC ', UpperCase(TextBuf[CurLine])) > 0) then
  begin
    Exit;
  end;
  
  //TextBuf[CurLine] := Strip(TextBuf[CurLine], ' ');
   
  //v.0.1.5 
    
  for n := 0 to ProcCount do
  begin
    Proc := Copy(ProcBuf[n], 5, Length(ProcBuf[n]) - 4);
    
    if System.Pos(UpperCase(Proc) + '(', UpperCase(TextBuf[CurLine])) > 0 then
    begin
      Buffer := ExtractText(TextBuf[CurLine], '(', ')');
      Split(Buffer, ',', []);
      
      param := buffer;
      ParamInc := StrBuf.Count;
      
      if Buffer = '' then ParamInc := 0;
      
      buffer := '';
                         
      if ParamInc > 0 then                         
      //if StrBuf.Count > 0 then      
      begin
        ParamInc := 0;
        for j := 1 to 21 do Params[j] := '';

        for j := 0 to StrBuf.Count - 1 do
        begin
          Inc(ParamInc);
          Params[ParamInc] := AsmStrNum(StrBuf[j]);
        end;
        
        // Check the parameters and fill appropriate global variables with them depending on the parameter type
                
        for j := 1 to ParamInc do
        begin          
          for n1 := 1 to GVarCnt do
          begin                                
            if UpperCase(GVar[n1].VarName + _EFF) = UpperCase(Params[j]) then
            begin
              if System.Pos(GVar[n1].VarType, 'T1T2') > 0 then
              begin            
                if CheckProcML(ProcBuf[n]) then
                begin
                  if j = 1 then
                    CodeBuf.Add(' lda ' + Params[j])
                  else
                    CodeBuf.Add(' ldx ' + Params[j]);
                end;
                // else begin          
                //  CodeBuf.Add(' mva ' + Params[j] + ' b_param' + IntToStr(j));
                //end;
              
                if j = 1 then
                  //Buffer := 'b_param1'
                  Buffer := Params[j]
                else
                  //Buffer := Buffer + ', b_param' + IntToStr(j);
                  Buffer := Buffer + ', ' + Params[j];
              end else begin
                if CheckProcML(ProcBuf[n]) then
                begin
                  if System.Pos('#', Params[j]) > 0 then
                  begin                
                    if j = 1 then
                      CodeBuf.Add(' lda ' + Params[j])
                    else
                      CodeBuf.Add(' ldx ' + Params[j]);
                  end;
                end;  // else begin          
                //  CodeBuf.Add(' mwa ' + Params[j] + ' w_param' + IntToStr(j));
                //end;
                                                                                                                   
                if j = 1 then
                  //Buffer := 'w_param1'
                  Buffer := Params[j]
                else
                  //Buffer := Buffer + ', w_param' + IntToStr(j);
                  Buffer := Buffer + ', ' + Params[j];
              end;
            end;
          end;
                                         
          if Pos('#', Params[j]) > 0 then          
          begin
            if StrToInt(Extract(Params[j], '#', 2)) < 256 then
            begin
              if CheckProcML(ProcBuf[n]) then
              begin
                if j = 1 then
                  CodeBuf.Add(' lda ' + Params[j])
                else
                  CodeBuf.Add(' ldx ' + Params[j]);
              end else begin
                //CodeBuf.Add(' mva ' + Params[j] + ' b_param' + IntToStr(j));
              
                if j = 1 then
                  Buffer := Params[j]
                  //Buffer := 'b_param1'
                else
                  Buffer := Buffer + ', ' + Params[j];
                  //Buffer := Buffer + ', b_param' + IntToStr(j);
              end;              
            end else begin
              //CodeBuf.Add(' mwa ' + Params[j] + ' w_param' + IntToStr(j));
              
              if j = 1 then
                Buffer := Params[j]
                //Buffer := 'w_param1'
              else
                Buffer := Buffer + ', ' + Params[j];
                //Buffer := Buffer + ', w_param' + IntToStr(j);
            end;              
          end;
        end;

        // End block: Check the parameters and fill variables
        //                
        if CheckProcML(ProcBuf[n]) then
          CodeBuf.Add(' jsr ' + GetMLAddress(ProcBuf[n]))
        else begin
          CodeBuf.Add(' ' + Proc  + _REFF + ' ' + Buffer);
        end;
      
      end else begin
        if CheckProcML(ProcBuf[n]) then
        begin
            if (param <> '') then
            begin
              if IsNumber(param[1]) then
              begin         
                CodeBuf.Add(' lda #' + param);
              end;
            end;
            
            CodeBuf.Add(' jsr ' + GetMLAddress(ProcBuf[n]));
        end else begin
          CodeBuf.Add(' ' + Proc + _REFF);
        end;
      end;
      
      if System.Pos('=', TextBuf[CurLine]) > 0 then
      begin
        n1 := System.Pos('=', TextBuf[CurLine]) - 1;
        Buffer := Trim(Copy(TextBuf[CurLine], 1, n1));

        if CheckProcML(ProcBuf[n]) then
          CodeBuf.Add(' mwa $A0 STORE1');
                
        CodeBuf.Add(' mwa STORE1 ' + Buffer + _EFF);
      end;
            
      Break;
    end;
  end;
end;

{
  Procedure name : sc_od
  Description    : Handles Action! OD statement for ending loop commands
  Parameters     : None
}
procedure sc_od;
var
  n1, n2 : Byte;
  Buffer : String;
  boolByte : Boolean = True;
begin
  if System.Pos('OD', UpperCase(TextBuf[CurLine])) > 0 then 
  begin
    Buffer := Trim(UpperCase(TextBuf[CurLine]));
    
    if Copy(Buffer, 1, 2) = 'OD' then
    begin
      if sWhile in flags then
      //if WhileFlag then 
      begin
        for n2 := 1 to GVarCnt2 do
        begin
          if System.Pos(GVar2[n2].VarType, 'T5T6T7T8') > 0 then  // ARRAY
          //if System.Pos(' ARRAY', UpperCase(GVar2[n2].VarType)) > 0 then
            if boolArray then CodeBuf.Add(' inc array_index_' + GVar2[n2].VarName + _EFF);
        end;                           
        
        CodeBuf.Add(' #end');
        CodeBuf.Add('jump_from_while_' + IntToStr(WhileIndex));
        
        flags := flags - [sWhile];
        //WhileFlag := False;
        
      end else if sFor in flags (*if ForFlag*) then begin
        for n1 := 1 to GVarCnt do
        begin
          if UpperCase(ForVar1) = UpperCase(GVar[n1].VarName) then
          begin
            if System.Pos(GVar[n1].VarType, 'T1T2') > 0 then  // BYTE, CHAR
            begin
              for n2 := 1 to StrToInt(ForVar4) do
                CodeBuf.Add(' inc ' + ForVar1 + _EFF);
            end else begin
              boolByte := False;
              
              for n2 := 1 to StrToInt(ForVar4) do
                CodeBuf.Add(' inw ' + ForVar1 + _EFF);
            end;
            
            if boolArray then
            begin
              for n2 := 1 to GVarCnt2 do
              begin
                if System.Pos(GVar2[n2].VarType, 'T5T6T7T8') > 0 then  // ARRAY
                  CodeBuf.Add(' inc array_index_' + GVar2[n2].VarName + _EFF);
              end;              
            end;
            
            Break;
          end;      
        end;      
        
        if boolByte then
        begin
          CodeBuf.Add(' ldx ' + ForVar1 + _EFF);        
          CodeBuf.Add(' cpx ' + AsmStrNum(ForVar3));
        end else begin      
          CodeBuf.Add(' lda ' + ForVar1 + _EFF);
          CodeBuf.Add(' cmp STORE3');
          CodeBuf.Add(' lda ' + ForVar1 + _EFF + '+1');
          CodeBuf.Add(' sbc STORE3+1');
        end;
        
        CodeBuf.Add(' jcc ' + ForLabels[ForCnt]);
        
        // Jump from FOR loop in case of EXIT command
        CodeBuf.Add('jump_from_' + ForLabels[ForCnt]);
        
        flags := flags - [sFor];
        //ForFlag := False;
      end else if sUntil in flags then
      //end else if UntilFlag then
      begin
         CodeBuf.Add('loop_jump' + IntToStr(LoopIndex));
         //CodeBuf.Add('jump_from_until_' + IntToStr(UntilIndex));
         flags := flags - [sUntil];
         //UntilFlag := False;
      end else begin
        CodeBuf.Add(' ldx #0');
        CodeBuf.Add(' cpx loop_var');
        CodeBuf.Add(' jcc loop' + IntToStr(LoopIndex));
        CodeBuf.Add('loop_jump' + IntToStr(LoopIndex))
      end;
    end;
  end;
end;

{
  Procedure name : Cond
  Description    : Handles Action! conditional statements
  Parameters     : Stmt - WHILE, IF, ELSEIF
}
procedure Cond(Stmt : String);
var
  n, n1, n2, n3, c1, c2 : Integer;
  IFExpr, Number, Buffer, Buffer2 : String;
  arrOper : Array[0..9] of String[4];
begin
  Stmt := UpperCase(Stmt);
  
  for n := 0 to 9 do arrOper[n] := '';
  
  n1 := System.Pos(Stmt + ' ', UpperCase(TextBuf[CurLine]));
  if (n1 > 0) then
  begin
    if System.Pos('"', TextBuf[CurLine]) > 0 then Exit;
    if Stmt = 'WHILE' then
      flags := flags + [sWhile];
      //WhileFlag := True;
    
    IFExpr := Trim(LowerCase(TextBuf[CurLine]));
    IFExpr := Extract(IFExpr, Stmt, 2);

    if System.Pos('IF', UpperCase(TextBuf[CurLine])) > 0 then
    begin    
      IFExpr := Extract(IFExpr, 'then', 1);
    end;
    
    // Parse IF and WHILE conditional OR and AND expressions
        
    IFExpr := StringReplace(IFExpr,' and ','&', [rfReplaceAll]);
    IFExpr := StringReplace(IFExpr,' or ','|', [rfReplaceAll]);
    IFExpr := StringReplace(IFExpr,'#','<>', [rfReplaceAll]);
    IFExpr := Strip(IFExpr, ' ');
    IFExpr := Strip(IFExpr, '(');
    IFExpr := Strip(IFExpr, ')');
    c1 := System.Pos('&', IFExpr);
    c2 := System.Pos('|', IFExpr);        
    StrBuf.Clear;
    
    if c1 + c2 = 0 then
    begin
      StrBuf.Add('dummy');  // Dummy value
    end else begin
      // If AND before OR
      if (c1 < c2) and (c1 > 0) then
      //if (c1sum < c2sum) and (c2sum > 0) and (c1 > 0) then
      begin
        arrOper[0] := '.and';
        arrOper[1] := '.or';
        StrBuf.Add(ExtractWord(1, IFExpr, ['&']));
        StrBuf.Add(ExtractWord(2, IFExpr, ['&', '|']));
        StrBuf.Add(ExtractWord(2, IFExpr, ['|']));
      
      // If OR before AND
      end else if (c2 < c1) and (c2 > 0) then
      //end else if (c2sum < c1sum) and (c1sum > 0) and (c2 > 0) then
      begin
        arrOper[0] := '.or';
        arrOper[1] := '.and';
        StrBuf.Add(ExtractWord(1, IFExpr, ['|']));
        StrBuf.Add(ExtractWord(2, IFExpr, ['|', '&']));
        StrBuf.Add(ExtractWord(2, IFExpr, ['&']));
      
      // If AND exists and OR doesn't
      end else if (c1 > 0) and (c2 < 1) then
      //end else if (c1sum > 0) and (c2sum < 1) then
      begin
        IFExpr := StringReplace(IFExpr,'&',' and ',[rfReplaceAll]);
        Split(IFExpr, ' and ', []);

        for n3 := 0 to StrBuf.Count - 1 do
        begin
          arrOper[n3] := '.and';
        end;
        
      // If OR exists and AND doesn't
      end else if (c1 < 1) and (c2 > 0) then
      //end else if (c1sum < 1) and (c2sum > 0) then
      begin
        IFExpr := StringReplace(IFExpr,'|',' or ',[rfReplaceAll]);
        Split(IFExpr, ' or ', []);

        for n3 := 0 to StrBuf.Count - 1 do
        begin
          arrOper[n3] := '.or';
        end;          
      end;
    end;              
      
    Buffer := '';
    
    for n3 := 0 to StrBuf.Count - 1 do
    begin
      for n2 := 1 to 20 do
      begin
        if StrBuf.Count = 1 then
        begin
          n := System.Pos(Operators[n2], IFExpr);
          n1 := Length(Operators[n2]);
        
          if n > 0 then
          begin
            Number := Extract(IFExpr, Operators[n2], 2);
            //Number := Copy(IFExpr, n + n1, Length(IFExpr) - (n + n1) + 1);
            Number := AsmStrNum(Number);            
            Buffer2 := Extract(IFExpr, Operators[n2], 1);
            //Buffer2 := Copy(IFExpr, 1, Length(IFExpr) - (Length(IFExpr) - (n + n1) + 1) - n1);
            //Buffer2 := Extract(IFExpr, '=', 1) + _EFF;
            IFExpr := Copy(IFExpr, 1, n + n1 - 1) + Number;
            //IFExpr := Buffer2 + '=' + Number; 
          
            for n1 := 1 to GVarCnt do
            begin
              if LowerCase(GVar[n1].VarName) = LowerCase(Buffer2) then
              begin
                if System.Pos(GVar[n1].VarType, 'T1T2') > 0 then  // BYTE, CHAR
                  Buffer := ' .byte '  // + Buffer2 + _EFF + Operators[n2] + Number
                else
                  Buffer := ' .word ';  // + Buffer2 + _EFF + Operators[n2] + Number;
                  
                Buffer := Buffer + Buffer2 + _EFF + Operators[n2] + Number;
              
                Break;
              end;
            end;          
                   
            Break;
          end;
                  
        end else begin
          if System.Pos(Operators[n2], StrBuf[n3]) > 0 then
          begin
            Number := AsmStrNum(Extract(StrBuf[n3], Operators[n2][1], 2));
            Buffer2 := Extract(StrBuf[n3], Operators[n2][1], 1);
          
            for n1 := 1 to GVarCnt do
            begin
              if LowerCase(GVar[n1].VarName) = LowerCase(Buffer2) then
              begin
                if System.Pos(GVar[n1].VarType, 'T1T2') > 0 then  // BYTE, CHAR
                  Buffer := Buffer + ' .byte '  // + Buffer2 + _EFF + Operators[n2] + Number
                else
                  Buffer := Buffer + ' .word ';  // + Buffer2 + _EFF + Operators[n2] + Number;
                  
                Buffer := Buffer + Buffer2 + _EFF + Operators[n2] + Number;
                                                                           
                Break;
              end;
            end;
          
            if n3 < StrBuf.Count - 1 then
              Buffer := Buffer + ' ' + arrOper[n3];
          
            Break;
          end;
        end;
      end;
    end;

    if (Stmt = 'IF') and (System.Pos('ELSEIF', UpperCase(TextBuf[CurLine])) < 1) then
    begin
      ifElseIndex := CurLine;    
      
      if sElseIf in flags then
        CodeBuf.Add(' mva #0 else_flag');
      
      CodeBuf.Add(' #if ' + Buffer);
      
      if sElseIf in flags then
        CodeBuf.Add(' mva #1 else_flag');
                
    end else if Stmt = 'ELSEIF' then begin
      CodeBuf.Add(' #end');
      CodeBuf.Add(' #if ' + Buffer);

      if sElseIf in flags then
        CodeBuf.Add(' mva #1 else_flag');

    end else if Stmt = 'WHILE' then begin
      for n2 := 1 to GVarCnt2 do
      begin
        if System.Pos(GVar2[n2].VarType, 'T5T6T7T8') > 0 then  // ARRAY
          CodeBuf.Add(' mva #0 array_index_' + GVar2[n2].VarName + _EFF);
      end;
      
      WhileIndex := CurLine;
      CodeBuf.Add(' #while ' + Buffer);      
    end;
  end;
end;

{
  Procedure name : sc_else
  Description    : Handles Action! ELSE conditional statements
  Parameters     : None
}
procedure sc_else;
begin
  if ((System.Pos('ELSE', UpperCase(TextBuf[CurLine])) > 0)
       and (System.Pos('ELSEIF', UpperCase(TextBuf[CurLine])) < 1)) then
    if System.Pos('"', TextBuf[CurLine]) < 1 then
    begin    
      boolElse := True;  // ELSE statement was found
      CodeBuf.Add(' #else');
      
      if sElseIf in flags then
      //if flagElse then
      begin      
        CodeBuf.Add(' #if .byte else_flag=#1');
        CodeBuf.Add(' jmp from_else' + IntToStr(ifElseIndex));
        CodeBuf.Add(' #end');
      end;
    end;
end;

{
  Procedure name : sc_fi
  Description    : Handles Action! FI conditional closing directive
  Parameters     : None
}
procedure sc_fi;
var
  Buffer : String;  
begin
  if ((System.Pos('FI', UpperCase(TextBuf[CurLine])) > 0)
      and (System.Pos('FILL', UpperCase(TextBuf[CurLine])) < 1)) then
  begin
    Buffer := Strip(UpperCase(TextBuf[CurLine]), ' ');
    
    if Copy(Buffer, 1, 2) = 'FI' then
    begin
      CodeBuf.Add(' #end');
      
      if boolElse then  // ELSE statement was found
        CodeBuf.Add('from_else' + IntToStr(ifElseIndex));
        
      boolElse := False;  // ELSE statement not initialized again
    end;
  end;  
end;

{
  Procedure name : sc_until
  Description    : Handles Action! UNTIL conditional statements
  Parameters     : None
}
procedure sc_until;
var
  Buffer : String;
  n, n2 : Integer;
begin  
  if (System.Pos('UNTIL ', UpperCase(TextBuf[CurLine])) > 0) then
  begin
    if System.Pos('"', TextBuf[CurLine]) > 0 then Exit;
    
    flags := flags + [sUntil];
    //UntilFlag := True;
    Buffer := Extract(UpperCase(TextBuf[CurLine]), 'UNTIL ', 2);    
    
    for n2 := 1 to 20 do
    begin
      n := System.Pos(Operators[n2], Buffer);
        
      if n > 0 then
      begin
        Split(Buffer, Operators[n2], []);
        CodeBuf.Add(' lda ' + AsmStrNum(StrBuf[0]));
        
        if Operators[n2] = '>' then
        begin          
          CodeBuf.Add(' ldx ' + AsmStrNum(StrBuf[1]));
          CodeBuf.Add(' inx');
          CodeBuf.Add(' stx STORE1');
          CodeBuf.Add(' cmp STORE1');
          //CodeBuf.Add(' bcs jump_from_until_' + IntToStr(UntilIndex));
          CodeBuf.Add(' bcs loop_jump' + IntToStr(LoopIndex));
        end else if Operators[n2] = '<' then begin
          //CodeBuf.Add(' lda ' + AsmStrNum(StrBuf[0]));
          CodeBuf.Add(' cmp ' + AsmStrNum(StrBuf[1]));
          //CodeBuf.Add(' bcc jump_from_until_' + IntToStr(UntilIndex));
          CodeBuf.Add(' bcc loop_jump' + IntToStr(LoopIndex));
        end else begin
          //CodeBuf.Add(' lda ' + AsmStrNum(StrBuf[0]));
          CodeBuf.Add(' cmp ' + AsmStrNum(StrBuf[1]));
          //CodeBuf.Add(' beq jump_from_until_' + IntToStr(UntilIndex));
          CodeBuf.Add(' beq loop_jump' + IntToStr(LoopIndex));
        end;
                                           
        Break;
      end;        
    end;

    CodeBuf.Add(' jmp LabelUntil' + IntToStr(UntilIndex));
  end;
end;

{
  Procedure name : sc_For
  Description    : Handles Action! FOR looping statement
  Parameters     : None

  Examples:
    FOR n = 1 to 25600
    FOR i = 0 to 30 STEP 1
}
procedure sc_For;
var
  n1, n2 : LongInt;
begin
  n1 := System.Pos('FOR ', UpperCase(TextBuf[CurLine]));
  if (n1 > 0) then
  begin
    if System.Pos('"', TextBuf[CurLine]) > 0 then Exit;
  
    ForVar1 := Extract(TextBuf[CurLine], '=', 1);
    ForVar1 := Extract(UpperCase(ForVar1), 'FOR', 2);
    ForVar2 := Extract(TextBuf[CurLine], '=', 2);
    Split(UpperCase(ForVar2), 'TO', []);
    ForVar2 := StrBuf[0];
    ForVar3 := StrBuf[1];
    Split(UpperCase(ForVar3), 'STEP', []);
    
    if StrBuf.Count <= 1 then
      ForVar4 := '1'
    else begin
      ForVar3 := StrBuf[0];
      ForVar4 := StrBuf[1];
    end;

    if IsNumber(ForVar3[1]) then
    //if (((ForVar3[1] > Chr(47)) and (ForVar3[1] < Chr(58))) or (ForVar3[1] = '$')) then
      ForVar3 := IntToStr(StrToInt(ForVar3) + 1)
    else
      CodeBuf.Add(' inc ' + ForVar3 + _EFF);
    
    for n1 := 1 to GVarCnt do
    begin
      if UpperCase(ForVar1) = UpperCase(GVar[n1].VarName) then
      begin
        ForVar := GVar[n1].VarName;               
        CodeBuf.Add(mvwa(GVar[n1].VarType) + AsmStrNum(ForVar2) + ' ' + AsmStrNum(ForVar1));
        
        if boolArray then
        begin
          if IsNumber(ForVar2[1]) then        
          //if (((ForVar2[1] > Chr(47)) and (ForVar2[1] < Chr(58))) or (ForVar2[1] = '$')) then
          begin
            for n2 := 1 to GVarCnt2 do
            begin
              if System.Pos(GVar2[n2].VarType, 'T5T6T7T8') > 0 then  // ARRAY
              //if System.Pos(' ARRAY', UpperCase(GVar2[n2].VarType)) > 0 then
                CodeBuf.Add(' mva #' + IntToStr(StrToInt(ForVar2) * 2) + ' array_index_' + GVar2[n2].VarName + _EFF);
            end;
          end;
        end;            
        
        //ForFlag := True;
        flags := flags + [sFor];
        Inc(ForCnt);
        ForLabels[ForCnt] := 'for_loop' + IntToStr(ForCnt);
                
        if System.Pos(GVar[n1].VarType, 'T3T4') > 0 then                
          CodeBuf.Add(' mwa ' + AsmStrNum(ForVar3) + ' STORE3');
            
        CodeBuf.Add(ForLabels[ForCnt]);
        
        Break;
      end;
    end;
  end;
end;

{
  Procedure name : GenLoop
  Description    : Core routine for processing Action! source code listing
  Parameters     : Flag
}
procedure GenLoop(Flag : Boolean);
var
  i : LongInt;    
begin
  boolElse := False;  // ELSE statement not yet initialized

  for i := 1 to CR_LF do
  begin  
    CurLine := i;

    // Check for comments
    if System.Pos(';', TextBuf[i]) > 0 then
    begin
      CodeBuf.Add('; ' + Extract(TextBuf[i], ';', 2));
      TextBuf[i] := Extract(TextBuf[i], ';', 1);
    end;            

    // Check for comment (;) as the first character on the line
    if Copy(Trim(TextBuf[i]), 1, 1) = ';' then
    begin
      CodeBuf.Add(Copy(TextBuf[i], 1, Length(TextBuf[i]) - Length(LineEnding)));
      Continue;
    end;

    sc_Proc;
    sc_Return;
    
    // Parse machine code opcodes
    sc_ML;

    // Action! statements, assignments, directives
    sc_VarExpr;    
    sc_ProcTrack;
    Cond('WHILE');
    sc_until;    
    sc_for;
    sc_od;
    sc_do;
    Cond('IF');
    Cond('ELSEIF');
    sc_fi;
    sc_else;
    sc_Exit;    

    sc_Print(True);
    sc_Print(False);
    sc_PrintF;
    sc_Locate;
    
    sc_Command('Color', '3');
    sc_Command('Device', '3');

    sc_Command('Graphics', '1');
    sc_Command('Poke', '01');
    sc_Command('PokeC', '01');
    sc_Command('Position', '11');
    sc_Command('Plot', '11');
    sc_Command('DrawTo', '11');
    sc_Command('Fill', '11');
    sc_Command('SetColor', '111');
    sc_Command('Sound', '1111');
    
    sc_Command('SetBlock', '011');
    sc_Command('Zero', '01');
    sc_Command('MoveBlock', '001');
        
    sc_Command('Put', '1');
    sc_PutE;
    sc_PrintX('PrintB', False);
    sc_PrintX('PrintBE', True);
    sc_PrintX('PrintI', False);
    sc_PrintX('PrintIE', True);
    sc_PrintX('PrintC', False);
    sc_PrintX('PrintCE', True);

    sc_InputS;
    sc_SCopy;
    sc_SCopyS;
    sc_SAssign;
    sc_StrNum('StrB');
    sc_StrNum('StrC');
    sc_StrNum('StrI');

    sc_Open;
    sc_Close;
    sc_PrintDX(True);
    sc_PrintDX(False);
    sc_PutDX('PutD');
    sc_PutDX('PutDE');
    sc_Point;
    sc_Note;
    sc_GetD;
    sc_InputSMD('INPUTSD', '#255', '21');
    sc_InputSMD('INPUTMD', '0', '211');
    sc_PrintXD('PrintBD', False);
    sc_PrintXD('PrintBDE', True);
    sc_PrintXD('PrintID', False);
    sc_PrintXD('PrintIDE', True);
    sc_PrintXD('PrintCD', False);
    sc_PrintXD('PrintCDE', True);

    sc_SndRst;
    
    sc_Func1('Peek', '0');
    sc_Func1('PeekC', '0');
    sc_Func1('Stick', '1');
    sc_Func1('Strig', '1');
    sc_Func1('Paddle', '1');
    sc_Func1('Ptrig', '1');
    sc_Func1('Rand', '1');

    sc_Func2('ValB', '1');
    sc_Func2('ValI', '1');
    sc_Func2('ValC', '1');

    sc_XIO;

    if TextBuf[i] = LineEnding then CodeBuf.Add('');
  end;
end;

{
  Procedure name : sc_do
  Description    : Handles Action! looping statements by initializing all necessary variables
  Parameters     : None
}
procedure sc_do;
var
  Buffer: String;
begin
  // Check to see if this is infinite loop (DO OD)
  // If this is the case, then no labels are defined
  if System.Pos(' OD', UpperCase(TextBuf[CurLine])) > 0 then
  begin
    Exit;
  end; 

  if System.Pos('DO', UpperCase(TextBuf[CurLine])) > 0 then 
  begin
    Buffer := Strip(UpperCase(TextBuf[CurLine]), ' ');

    if (Copy(Buffer, 1, 2) = 'DO') and not (sFor in flags) (*not ForFlag*) and not (sWhile in flags)(*WhileFlag*) and not (sUntil in flags) then  // UntilFlag
    begin
      CodeBuf.Add(' mva #10 loop_var');
      LoopIndex := CurLine;
      CodeBuf.Add('loop' + IntToStr(LoopIndex));
      UntilIndex := CurLine;
      CodeBuf.Add('LabelUntil' + IntToStr(UntilIndex));
    end;
  end;
end;

{
  Procedure name : sc_Exit
  Description    : Handles Action! EXIT command
  Parameters     : None
}
procedure sc_Exit;
var
  Buffer : String;
begin
  if System.Pos('EXIT', UpperCase(TextBuf[CurLine])) > 0 then
  begin
    Buffer := Strip(UpperCase(TextBuf[CurLine]), ' ');
    
    if (Copy(Buffer, 1, 4) = 'EXIT') then
    begin
      if sFor in flags then
      //if ForFlag then
        CodeBuf.Add(' jmp jump_from_' + ForLabels[ForCnt])
      else if sWhile in flags(*WhileFlag*) then
        CodeBuf.Add(' jmp jump_from_while_' + IntToStr(WhileIndex))
//      else if UntilFlag then
//        CodeBuf.Add(' jmp jump_from_until_' + IntToStr(UntilIndex))
      else
        CodeBuf.Add(' jmp loop_jump' + IntToStr(LoopIndex));
    end;
  end;
end;

{
  Procedure name : sc_ML
  Description    : Checks for PROC or FUNC block consisting of machine language mnemonics
                   embeded between [ and ]
  Parameters     : None
}
procedure sc_ML;
var
  Buffer : String;
  n : Integer;
begin
  // PROC / FUNC machine code call
  if ProcML_start then
  begin
    // Machine code [] block
    //
    if (System.Pos('[', TextBuf[CurLine]) > 0) and (System.Pos(']', TextBuf[CurLine]) > 0) then
    begin
      Buffer := ExtractText(TextBuf[CurLine], '[', ']');      
      ProcML_start := False;
      PrgVar.SB := _SB_NULL;
    
    // Start of machine code
    end else if System.Pos('[', TextBuf[CurLine]) > 0 then        
    begin
      Split(TextBuf[CurLine], '[', []);
      Buffer := StrBuf[1];
    
    // End of machine code
    //
    end else begin      
      Buffer := TextBuf[CurLine];
      
      if System.Pos(']', Buffer) > 0 then
      begin
        Buffer := Replace(Buffer, ']', ' ');
        ProcML_start := False;
        PrgVar.SB := _SB_NULL;
      end;              
    end;
    
    Buffer := Trim(Buffer);    
    Split(Buffer, ' ', []);
        
    // Machine code opcodes without spaces (just opcodes supperceded with $)
    //
    if StrBuf.Count <= 1 then
    begin
      Split(Buffer, '$', []);
      Buffer := '';
      
      for n := 0 to StrBuf.Count - 1 do
      begin      
        if Length(StrBuf[n]) > 2 then
          StrBuf[n] := Copy(StrBuf[n], 3, 2) + ' ' + Copy(StrBuf[n], 1, 2);
               
        Inc(MemCnt, Length(StrBuf[n]) div 2);        
        Buffer := Buffer + ' ' + StrBuf[n];
      end;
    
    // Machine code opcodes separated with spaces (decimal and hexadecimal format allowed)
    //
    end else begin
      Buffer := '';
      
      for n := 0 to StrBuf.Count - 1 do
      begin
        if Pos('$', StrBuf[n]) < 1 then
          StrBuf[n] := Dec2Numb(StrToInt(StrBuf[n]), 4, 16);                
        
        StrBuf[n] := Trim(Replace(StrBuf[n], '$', ' '));                        
        
        if Length(StrBuf[n]) > 2 then
          StrBuf[n] := Copy(StrBuf[n], 3, 2) + ' ' + Copy(StrBuf[n], 1, 2);
        
        Inc(MemCnt, Length(StrBuf[n]) div 2);
        Buffer := Buffer + ' ' + StrBuf[n];                
      end;
    end;       
    
    // Write current machine code
    ProcML[ProcML_cnt].Code := ProcML[ProcML_cnt].Code + Buffer;       
    
    // Write machine code memory address
    if not ProcML_start then
    begin
      Buffer := Dec2Numb(CntML, 4, 16);
      ProcML[ProcML_cnt].Address := '$' + Buffer;
      ProcML_data[ProcML_cnt] := ' org $' + Buffer;
      CntML := 0;
    end;

  // Machine code elsewhere in the program
  //
  end else if (System.Pos('[', TextBuf[CurLine]) > 0) then
  begin    
    ML_start := True;
    
    // v.0.1.5
    // Check for non-inline assembler code starting with character '['
    //
    if not (sProcAsm in Flags) then
    begin  
      for n := 0 to VarTypes.Count - 1 do
      begin
        if System.Pos(VarTypes[n], UpperCase(TextBuf[CurLine])) > 0 then
        begin
          ML_start := False;
          Break;
        end;
      end;  
      
      for n := 0 to VarTypes2.Count - 1 do
      //for n := 1 to _VAR_TYPES2 do
      begin
        if System.Pos(VarTypes2[n], UpperCase(TextBuf[CurLine])) > 0 then
        begin
          ML_start := False;
          Break;
        end;
      end;
    end;
    
    // No such situation, so continue...
    CntML := MemCnt;
    Inc(ProcML_cnt);
    PrgVar.SB := _SB_ML;
    ProcML[ProcML_cnt].Code := ' .he';    
  end;
  
  if ML_start then
  begin
    if (System.Pos('[', TextBuf[CurLine]) > 0) and (System.Pos(']', TextBuf[CurLine]) > 0) then
    begin
      Buffer := ExtractText(TextBuf[CurLine], '[', ']');
      PrgVar.SB := _SB_NULL;
      ML_start := False;
    end else if System.Pos('[', TextBuf[CurLine]) > 0 then
    begin
      Split(TextBuf[CurLine], '[', []);
      Buffer := StrBuf[1];
    end else begin      
      Buffer := TextBuf[CurLine];
      if System.Pos(']', Buffer) > 0 then
      begin
        Buffer := Replace(Buffer, ']', ' ');
        ML_start := False;
        PrgVar.SB := _SB_NULL;
      end;  
    end;          
    
    Buffer := Trim(Buffer);    
    Split(Buffer, ' ', []);        
    
    if StrBuf.Count <= 1 then
    begin
      Split(Buffer, '$', []);
      Buffer := '';
      
      for n := 0 to StrBuf.Count - 1 do
      begin      
        if Length(StrBuf[n]) > 2 then
          StrBuf[n] := Copy(StrBuf[n], 3, 2) + ' ' + Copy(StrBuf[n], 1, 2);
                
        Inc(MemCnt, Length(StrBuf[n]) div 2);
        
        Buffer := Buffer + ' ' + StrBuf[n];
      end;
    
    end else begin    
      Buffer := '';
      
      for n := 0 to StrBuf.Count - 1 do
      begin
        if Pos('$', StrBuf[n]) < 1 then
          StrBuf[n] := Dec2Numb(StrToInt(StrBuf[n]), 4, 16);      
        
        StrBuf[n] := Trim(Replace(StrBuf[n], '$', ' '));
        
        if Length(StrBuf[n]) > 2 then
          StrBuf[n] := Copy(StrBuf[n], 3, 2) + ' ' + Copy(StrBuf[n], 1, 2);
        
        Inc(MemCnt, Length(StrBuf[n]) div 2);
        Buffer := Buffer + ' ' + StrBuf[n];        
      end;
    end;
    
    ProcML[ProcML_cnt].Code := ProcML[ProcML_cnt].Code + Buffer;
      
    // v0.1.4  
    if not ML_start then
    begin
      Buffer := Dec2Numb(CntML, 4, 16);
      //ProcML[ProcML_cnt].Address := '$' + Buffer;
      //ProcML_data[ProcML_cnt] := ' org $' + Buffer;
      CntML := 0;
      //CodeBuf.Add(' jsr ' + ProcML[ProcML_cnt].Address);
      
      CodeBuf.Add(ProcML[ProcML_cnt].Code);                        
    end;
    
    // v0.1.5
    if (sProcAsm in Flags) and (PrgVar.SB = _SB_NULL) then
    begin
      CodeBuf.Add(' rts');
      CodeBuf.Add('');
      CodeBuf.Add(' .endp');
      CodeBuf.Add('');
      flags := flags - [sProcAsm];
    end;        
  end;
end;

end.
