{ 
  Program    : Effectus - Atari MADS cross-assembler/parser for Action! language

  Unit file  : common.pas
  Description: Common library
  
  Author: Bostjan Gorisek, Slovenia

  Program compiled with Free Pascal 2.6.4
  Reference: http://www.freepascal.org/
  
  This is cource code generator for MADS Assembler, which then generates
  executable files from parser of Action! language for 8-bit Atari home computers.
  
  References:
  http://mads.atari8.info/
  http://gury.atari8.info/effectus/ 
  
  This is open source and freeware project!
}
unit Common;
{$ifdef FPC}{$mode objfpc}{$h+}{$endif}

interface

uses
  sysutils, Classes, Process, INIfiles, strutils, Decl, Core;
 
procedure GetFuncs;
procedure CheckLibProc;
procedure DeviceCheck;
procedure ReadSource;
procedure sc_Lib(RtlDir : String);
procedure RunMads(MADS_dir, MADS_src_dir, MADS_bin_dir, MADS_output_dir : String);
function VarDeclCheck(StrBuf: String) : Boolean;
function ExprCheck(StrBuf: String) : Boolean;
procedure ReadCfg;
function ReadOrig : Boolean;
procedure ReadInclude;
procedure GenerateCode;
procedure MathExpr(VarStr, Str1, Str2 : String; Flag, Index : Byte);
function AsmStrNum(Str : String) : String;
function mvwa(src : String) : String;
function IsNumber(src : Char) : Boolean;
procedure Split(Str, Delimiter : String; Flags : TFlags);
procedure SplitEx2(Str : String; Delim1, Delim2, Delim3, Delim4 : Char);
function Extract(Str, Delimiter : String; Index: Integer) : String;
function ExtractNoTrim(Str, Delimiter : String; Index : Integer) : String;
function Replace(Str : String; Ch1, Ch2 : Char): String;
function Strip(Str : String; Ch : Char) : String;
function ExtractText(Str : String; Ch1, Ch2 : Char) : String;

implementation

{
  Procedure name: GetFuncs
  Description   : List of Action! functions
  Parameters    : None
}
procedure GetFuncs;
begin
  FuncList.Add('Peek');
  FuncList.Add('Stick');
  FuncList.Add('Strig');
  FuncList.Add('Paddle');
  FuncList.Add('Ptrig');
  FuncList.Add('Rand');
  FuncList.Add('PeekC');
  FuncList.Add('GetD');
  FuncList.Add('Locate');
  FuncList.Add('ValB');
  FuncList.Add('ValI');
  FuncList.Add('ValC');
end;

{
  Procedure name: CheckLibProc
  Description   : Check to see which libraries to include
  Parameters    : None
}
procedure CheckLibProc;
var
  i : Byte;
begin
  if System.Pos(';', TextBuf[CR_LF]) > 0 then Exit; 
    
  for i := 0 to GrProcs.Count - 1 do
    if (System.Pos(UpperCase(GrProcs[i]) + '(', UpperCase(TextBuf[CR_LF])) > 0) or
       (System.Pos(UpperCase(GrProcs[i]) + '=', UpperCase(TextBuf[CR_LF])) > 0) then
      lGraphics := True;

  for i := 0 to SoundProcs.Count - 1 do
    if System.Pos(UpperCase(SoundProcs[i]) + '(', UpperCase(TextBuf[CR_LF])) > 0 then
      lSound := True;

  for i := 0 to PrintFProcs.Count - 1 do
    if System.Pos(UpperCase(PrintFProcs[i]) + '(', UpperCase(TextBuf[CR_LF])) > 0 then
      lPrintF := True;

  for i := 0 to IOProcs.Count - 1 do
    if System.Pos(UpperCase(IOProcs[i]) + '(', UpperCase(TextBuf[CR_LF])) > 0 then
      lIO := True;

  for i := 0 to PrintFDProcs.Count - 1 do
    if System.Pos(UpperCase(PrintFDProcs[i]) + '(', UpperCase(TextBuf[CR_LF])) > 0 then
      lPrintFD := True;

  for i := 0 to ControllerProcs.Count - 1 do
    if System.Pos(UpperCase(ControllerProcs[i]) + '(', UpperCase(TextBuf[CR_LF])) > 0 then
      lControllers := True;
end;

{
  Procedure name: DeviceCheck
  Description   : Device housekeeping code (closing the device)
  Parameters    : None
}
procedure DeviceCheck;
var
  i : Byte;
begin
  for i := 0 to 7 do
  begin
    if l_IO_error[i] then
    begin
      WriteLn(fASM, '');
      WriteLn(fASM, 'stop' + IntToStr(i));
      WriteLn(fASM, ' close #$' + IntToStr(i) + '0');
      WriteLn(fASM, '');
    end;
  end;
end;

{
  Procedure name: ReadSource
  Description   : Reads Effectus source code listing
  Parameters    : None
}
procedure ReadSource;
var
  MemBuf, TempX : String[255];
  Count, i, k : LongInt;
  TempBuf, Buffer : String;
  FlagX : Byte;
begin
  CR_LF := 0;
  FlagX := 0;
  TempX := '';

  for k := 0 to 254 do
  begin  
    FileSeek(f, 255 * k, fsFromBeginning);
    Count := FileRead(f, MemBuf, 255);
    TempBuf := '';
    
    for i := 0 to Count - 1 do
    begin
      TempBuf := TempBuf + MemBuf[i];
      
      if (MemBuf[i - 1] + MemBuf[i] = LineEnding) or (MemBuf[i] = LineEnding) then
      begin
        Inc(CR_LF);

        if FlagX = 0 then
          TextBuf[CR_LF] := Copy(TempBuf, 1, Length(TempBuf))          
        else begin
          TextBuf[CR_LF] := TempX + Copy(TempBuf, 1, Length(TempBuf));
          FlagX := 0;          
        end;

        TempBuf := '';

        // Check to see if ELSEIF statement was found in the listing
        if System.Pos('ELSEIF ', UpperCase(TextBuf[CR_LF])) > 0 then
        begin
          flags := flags + [sElseIf];
          //flagElse := True;
        end;

        // Check to see if FOR statement was found in the listing
        if System.Pos('FOR ', UpperCase(TextBuf[CR_LF])) > 0 then
        begin
          flags := flags + [sFor];
        end;

        // Check to see if WHILE statement was found in the listing
        if System.Pos('WHILE ', UpperCase(TextBuf[CR_LF])) > 0 then
        begin
          flags := flags + [sWhile];
        end;

        CheckLibProc;  // search library procedures

        // Retrieve all procedures and functions (PROC, FUNC)
        //
        if (((System.Pos('PROC ', UpperCase(TextBuf[CR_LF])) > 0)
           or (System.Pos('FUNC ', UpperCase(TextBuf[CR_LF])) > 0)) and (System.Pos('"', TextBuf[CR_LF]) < 1)) then
        begin
          Inc(ProcCount);

          if lIncFlag then Inc(ProcCount2);

          if (System.Pos('PROC ', UpperCase(TextBuf[CR_LF])) > 0) then
          begin
            Buffer := Extract(TextBuf[CR_LF], 'PROC ', 2);

            if System.Pos('=', TextBuf[CR_LF]) > 0 then
              Buffer := Extract(Buffer, '=', 1)
            else if System.Pos('=*', TextBuf[CR_LF]) > 0 then
              Buffer := Extract(Buffer, '=*', 1)
            else
              Buffer := Extract(Buffer, '(', 1);

            ProcBuf.Add('PROC' + Buffer);
            PrmBuf.Add(ExtractText(TextBuf[CR_LF], '(', ')'));
          end else begin
            Buffer := Extract(TextBuf[CR_LF], ' FUNC ', 2);
            
            if System.Pos('=*', TextBuf[CR_LF]) > 0 then
              Buffer := Extract(Buffer, '=*', 1)
            else
              Buffer := Extract(Buffer, '(', 1);

            ProcBuf.Add('FUNC' + Buffer);
            PrmBuf.Add(ExtractText(TextBuf[CR_LF], '(', ')'));
          end;
        end;
      end else begin
        if i = 254 then
        begin
          TempX := Copy(TempBuf, 1, Length(TempBuf));
          FlagX := 1;
        end;
      end;   
    end;
  end;
  
  if ProcCount > 0 then Dec(ProcCount);  
end;

{
  Procedure name: sc_Lib
  Description   : Links all necessary libraries
  Parameters    : RtlDir - runtime library directory
}
procedure sc_Lib(RtlDir : String);
begin
  if lGraphics then
    WriteLn(fASM, '.link ' + AnsiQuotedStr(RtlDir + 'graphics.obx', ''''));
  
  if lSound then
    WriteLn(fASM, '.link ' + AnsiQuotedStr(RtlDir + 'sound.obx', ''''));
  
  if lPrintF or lGr then
    WriteLn(fASM, '.link ' + AnsiQuotedStr(RtlDir + 'printf.obx', ''''));
  
  if lIO then
    WriteLn(fASM, '.link ' + AnsiQuotedStr(RtlDir + 'io.obx', ''''));
  
  if lPrintFD then
    WriteLn(fASM, '.link ' + AnsiQuotedStr(RtlDir + 'printfd.obx', ''''));
  
  if lMath then
    WriteLn(fASM, '.link ' + AnsiQuotedStr(RtlDir + 'math.obx', ''''));
  
  if lControllers then
    WriteLn(fASM, '.link ' + AnsiQuotedStr(RtlDir + 'controllers.obx', ''''));
end;

{
  Procedure name: RunMads
  Description   : Executes Mads cross-compiler on Win32 platform
  Parameters    : MADS_dir - Runtime library directory
                  MADS_src_dir
                  MADS_bin_dir
                  MADS_output_dir
}
procedure RunMads(MADS_dir, MADS_src_dir, MADS_bin_dir, MADS_output_dir : String);
var
  AProcess : TProcess;
  AStringList : TStringList;
  i : LongInt;
  logFile : String;
begin
  logFile := GetCurrentDir + PathDelim + 'efflog.txt';

  // Now we will create the TProcess object, and assign it to the var AProcess.
  AProcess := TProcess.Create(nil);
   
  // Create the TStringList object.
  AStringList := TStringList.Create;
   
  AProcess.Parameters.Clear;
  
  // Tell the new AProcess what the command to execute is.
{$ifdef Unix}
  AProcess.Executable := MADS_dir + 'mads';
  AProcess.Parameters.Add(FilenameOrig);
  AProcess.Parameters.Add('-o:' + FilenameBin);  // Generate Atari native executable code
{$else}
  AProcess.Executable := MADS_dir + 'mads.exe';
  AProcess.Parameters.Add(AnsiQuotedStr(MADS_src_dir + FilenameOrig, '"'));  
  AProcess.Parameters.Add('-o:' + AnsiQuotedStr(MADS_bin_dir + FilenameBin, '"'));  // Generate Atari native executable code
{$endif}
  
  AProcess.Parameters.Add('-x');  // Exclude unreferenced procedures

  // We will define an option for when the program is run. This option will make sure that our program
  // does not continue until the program we will launch has stopped running. Also now we will tell it that
  // we want to read the output of the file.
  AProcess.Options := AProcess.Options + [poWaitOnExit, poUsePipes];      
   
  // Now that AProcess knows what the commandline is we will run it.
  AProcess.Execute;

  // Now read the output of the program we just ran into the TStringList.
  AStringList.LoadFromStream(AProcess.Output);
  //ErrList.LoadFromStream(AProcess.Stderr);
  //AStringList.Add(FormatDateTime('c', Now));
  //AStringList.Add('-----------------------------------------');
   
  // Check resulted compile status
  //mStatusLog := '';  //FormatDateTime('c', Now) + LineEnding;
  mStatus := '';

  for i := 0 to AStringList.Count - 1 do
  begin
    mStatusLog := mStatusLog + AStringList[i] + LineEnding;
    
    //if System.Pos('ERROR:', UpperCase(AStringList[i])) > 0 then begin
    //  mStatus := AStringList[i] + LineEnding;
      //lError := True;
    //end;
    
    //lWarnings := System.Pos('WARNING:', UpperCase(AStringList[i])) > 0;
    // else if System.Pos('WARNING:', UpperCase(AStringList[i])) > 0 then begin
    //  lWarnings := True;
    //end;
  end;   
  
  //if lWarnings then
  //  mStatus := mStatus + 'There were some warnings found!' + LineEnding;
  
  //if not lError then
  //  mStatus := mStatus + 'Compiling was successful!' + LineEnding;
      
  AStringList.Add(FormatDateTime('c', Now));
  AStringList.Add('----------------------------------------');
  AStringList.Insert(0, 'Filename: ' + FilenameOrig);
    
  //for i := 0 to AStringList.Count-1 do
  //  AStringList[i] := AStringList[i] + ',';  
      
  // Save the output to a file.
  AStringList.SaveToFile(logFile);
    
  AProcess.Parameters.Clear;
  
  // Redirect current output to the archive log
{$ifdef Unix}
  AProcess.Executable := '/bin/sh';
  AProcess.Parameters.Add('-c');
  AProcess.Parameters.Add('cat ' + logFile + ' >> effectus.log');
{$else}
  AProcess.Executable := 'cmd.exe';
  AProcess.Parameters.Add('/c');
  AProcess.Parameters.Add('"type ' + logFile + ' >> ' + MADS_output_dir + '"');
{$endif}
  AProcess.Options := [poWaitOnExit, poUsePipes];
  AProcess.Execute();  
       
  AStringList.Free;
  AProcess.Free;
end;

{
  Function name: VarDeclCheck
  Description   : Checks for reserved words in variable declarations
  Parameters    : StrBuf - String value to search for
}
function VarDeclCheck(StrBuf : String) : Boolean;
var
  i : ShortInt;
begin
  Result := True;
  
  for i := 1 to 4 do
  begin
    if System.Pos(NotVarDecl[i], UpperCase(StrBuf)) > 0 then
    begin
      Result := False;
      Break;
    end;
  end;
end;

{
  Function name: ExprCheck
  Description   : Checks for reserved words in expression declarations
  Parameters    : StrBuf - String value to search for
}
function ExprCheck(StrBuf : String) : Boolean;
var
  i : ShortInt;
begin
  Result := True;
  
  for i := 1 to 9 do
  begin
    if System.Pos(NotExpr[i], StrBuf) > 0 then
    begin
      Result := False;
      Break;
    end;
  end;
end;

{
  Procedure name: ReadCfg
  Description   : Reads configuration file
  Parameters    : None
}
procedure ReadCfg;
var
  IniFile : TIniFile;
begin
  IniFile := TIniFile.Create( 'config.ini' );
{$ifdef Unix}
  if FileExists(GetCurrentDir + '/config.ini') then
  begin
    IniFile.ReadIniFile;
    meditAddr := IniFile.ReadInteger( 'SETUP', 'ORG', 3200);
    meditMADS_dir := IniFile.ReadString('SETUP', 'MADS_DIR', '');
    meditMADS_src_ext := IniFile.ReadString('SETUP', 'MADS_SRC_EXT', 'asm');
    meditMADS_bin_ext := IniFile.ReadString('SETUP', 'MADS_BIN_EXT', 'xex');
    meditMADS_rtl_dir := IniFile.ReadString('SETUP', 'MADS_RTL_DIR', '');
    meditMLAddr := IniFile.ReadInteger( 'SETUP', 'ML_ORG', 8000);
  end else begin
    meditAddr := 3200;
    meditMADS_dir := '';
    meditMADS_src_ext := 'asm';
    meditMADS_bin_ext := 'xex';
    meditMADS_rtl_dir := GetCurrentDir + '/lib/';
    meditMLAddr := 8000;
  end;
{$else}
  if FileExists(GetCurrentDir + '\config.ini') then
  begin
    IniFile.ReadIniFile;
    meditAddr := IniFile.ReadInteger( 'SETUP', 'ORG', 3200);
    meditMADS_dir := IniFile.ReadString('SETUP', 'MADS_DIR', 'c:\atari\mads\');
    meditMADS_src_ext := IniFile.ReadString('SETUP', 'MADS_SRC_EXT', 'asm');
    meditMADS_bin_ext := IniFile.ReadString('SETUP', 'MADS_BIN_EXT', 'xex');
    meditMADS_rtl_dir := IniFile.ReadString('SETUP', 'MADS_RTL_DIR', 'c:\atari\effectus\lib\');
    meditMLAddr := IniFile.ReadInteger( 'SETUP', 'ML_ORG', 8000);
  end else begin
    meditAddr := 3200;
    meditMADS_dir := '';
    meditMADS_src_ext := 'asm';
    meditMADS_bin_ext := 'xex';
    meditMADS_rtl_dir := GetCurrentDir + '\lib\';
    meditMLAddr := 8000;
  end;
{$endif}  
  IniFile.Destroy;
end;

{
  Function name : ReadOrig
  Description   : Opens and reads Effectus source code file
  Parameters    : None
  Returns       : Boolean True if reading the file was successful, otherwise False
}
function ReadOrig : Boolean;
var
  IsFile : Boolean;
begin
  //if (Copy(meditEff_src_filename, 1, 1) = '\') or (Copy(meditEff_src_filename, 1, 1) = '/') then
  if Copy(meditEff_src_filename, 1, 1) = PathDelim then
    meditEff_src_filename := Copy(meditEff_src_filename, 2, Length(meditEff_src_filename) - 1);

  f := FileOpen(meditEff_src_filename, fmOpenRead);
  try
    if f = -1 then
    begin
      WriteLn('File does not exist!');
      IsFile := False;
    end else begin
      ReadSource;
      IsFile := True;
    end;
  finally
    FileClose(f);
    Result := IsFile;
  end;
end;

{
  Procedure name: ReadInclude
  Description   : Reads and processes the Effectus include source code file
  Parameters    : None
}
procedure ReadInclude;
var
  i : LongInt;
begin
  if not lInclude then Exit;
  
  { Open Effectus source file }
  f := FileOpen(ASM_icl[icl], fmOpenRead);
  if f = -1 then
  begin
    WriteLn('Error reading INCLUDE file!');
    Exit;
  end;
  
  ReadSource;
  FileClose(f);

  FilenameSrc := ExtractFileName(ASM_icl[icl]);
  FilenameSrc := Copy(FilenameSrc, 1, RPos('.', FilenameSrc) - 1) + '.';  
  FilenameSrc := FilenameSrc + meditMADS_src_ext;
  
  AssignFile(fASM_lib, meditMADS_src_dir + FilenameSrc);
  Rewrite(fASM_lib);
  CodeBuf.Clear;
  GenLoop(lInclude);

  for i := 0 to CodeBuf.Count - 1 do
    WriteLn(fASM_lib, CodeBuf[i]);
  
  CloseFile(fASM_lib);
end;

{
  Function name : mvwa
  Description   : Sets mva or mwa mnemonic command depending on the type of source variable
  Parameters    : src - Source variable
  Returns       : Returns mva or mwa mnemonic command depending on the type of source variable
}
function mvwa(src : String) : String;
begin
  if System.Pos(src, 'T1T2') > 0 then
    Result := ' mva '
  else
    Result := ' mwa ';
end;

{
  Function name : IsNumber
  Description   : Determines if source variable is a number (it accepts a number prefixed with $ sign)
  Parameters    : src - Value to be checked
  Returns       : Returns True if source variable is number, otherwise False
}
function IsNumber(src : Char) : Boolean;
begin
  Result := ((src > Chr(47)) and (src < Chr(58))) or (src = '$');
end;

{
  Procedure name : Split
  Description    : Splits string to separate string values delimited by
                   multi-character or one-byte delimiter.
                   It does trim out a space characters.
  Parameters     : Str - String value to be separated
                   Delimiter - Delimiter to be used as separator in Str
                   Flags - two possible values:
                           - []: Trims out all occurrences of a space character
                           - [cNoTrim]: Does not trim out a space character
}
procedure Split(Str, Delimiter : String; Flags : TFlags);
var
  Buffer : String = '';
  i : Integer = 1;
  Len : Byte;
begin
  Len := Length(Delimiter);
  StrBuf.Clear;

  if System.Pos(Delimiter, Str) < 1 then
  begin
    StrBuf.Add(Trim(Str));
    Exit;
  end;
  
  repeat
    if Copy(Str, i, Len) = Delimiter then
    begin
      if System.Pos(Delimiter, Buffer) > 0 then
        Buffer := Copy(Buffer, Len + 1, Length(Buffer) - Len)
      else
        Buffer := Copy(Buffer, 1, Length(Buffer));

      StrBuf.Add(Buffer);
      Buffer := '';
    end;

    Buffer := Buffer + Copy(Str, i, 1);
    Inc(i);
  until i = Length(Str);
  
  Buffer := Buffer + Copy(Str, i, 1);
  Buffer := Copy(Buffer, Len + 1, Length(Buffer) - Len);
  StrBuf.Add(Buffer);
  
  for i := 0 to StrBuf.Count - 1 do
  begin
    StrBuf[i] := IfThen(sNoTrim in Flags, StrBuf[i], Trim(StrBuf[i]));
  end;
end;

{
  Procedure name : SplitEx2
  Description    : Splits string to separate string values
  Parameters     : Str - String value to be separated
                   Delim1
                   Delim2
                   Delim3
                   Delim4
}
procedure SplitEx2(Str : String; Delim1, Delim2, Delim3, Delim4 : Char);
var
  Buffer : String;
  i : Integer;

function test(str: string; delim: char): boolean;
begin  
  if UpperCase(Copy(Str, i, 1)) = UpperCase(Delim) then
  begin
    if UpperCase(Copy(Str, i + 1, 1)) = UpperCase(Delim) then
    begin
      Buffer := Copy(Buffer, 2, Length(Buffer));
      StrBuf2.Add(Buffer);
      Buffer := '';
      result := false;
      Exit;
    end else begin    
      if System.Pos(UpperCase(Delim), UpperCase(Buffer)) > 0 then
        Buffer := Copy(Buffer, 2, Length(Buffer) - 1)
      else
        Buffer := Copy(Buffer, 1, Length(Buffer));
    end;

    Buffer := Strip(Buffer, Delim1);
    Buffer := Strip(Buffer, Delim2);
    Buffer := Strip(Buffer, Delim3);
    Buffer := Strip(Buffer, Delim4);
    StrBuf2.Add(Buffer);
    Buffer := '';
  end;

  result := true;
end;
  
begin
  Buffer := '';
  StrBuf2.Clear;
  
  if (System.Pos(UpperCase(Delim1), UpperCase(Str)) < 1) and (System.Pos(UpperCase(Delim2), UpperCase(Str)) < 1)
     and (System.Pos(UpperCase(Delim3), UpperCase(Str)) < 1) and (System.Pos(UpperCase(Delim4), UpperCase(Str)) < 1) then
  begin
    StrBuf.Add(Str);
    Exit;
  end;

  for i := 1 to Length(Str) do
  begin
    if not test(str, Delim1) then Continue
    else if not test(str, Delim2) then Continue
    else if not test(str, Delim3) then Continue
    else if not test(str, Delim4) then Continue;
    
    if i = Length(Str) then
    begin
      Buffer := Buffer + Copy(Str, i, 1);
      Buffer := Copy(Buffer, 2, Length(Buffer) - 1);
      StrBuf2.Add(Buffer);
    end;

    Buffer := Buffer + Copy(Str, i, 1);
  end;
end;

{
  Procedure name : Extract
  Description    : Extracts string to separate string values delimited by Delimiter
  Parameters     : Str - String value to be separated
                   Delimiter - Delimiter to be used as separator in Str
                   Index
}
function Extract(Str, Delimiter : String; Index : Integer) : String;
var
  Buffer : String = '';
  i, DelimPos : Integer;
  Flag : Boolean = False;
begin
  for i := 1 to Length(Str) do
  begin
    if (UpperCase(Copy(Str, i, Length(Delimiter))) = UpperCase(Delimiter)) and not Flag then
    begin
      Flag := True;
      DelimPos := i;
    end;

    if (UpperCase(Copy(Str, i, Length(Delimiter))) = UpperCase(Delimiter)) then
    begin
      case Index of
        1: Buffer := Copy(Buffer, 1, Length(Buffer));
        2: Buffer := Copy(Str, DelimPos + Length(Delimiter), Length(Str) - DelimPos+Length(Delimiter));
      end;
      
      Break;
    end;
    
    Buffer := Buffer + Str[i];
  end;

  Result := Trim(Buffer);
end;

function ExtractNoTrim(Str, Delimiter : String; Index : Integer) : String;
var
  Buffer : String = '';
  i, DelimPos : Integer;
  Flag : Boolean = False;
begin
  for i := 1 to Length(Str) do
  begin
    if (UpperCase(Copy(Str, i, Length(Delimiter))) = UpperCase(Delimiter)) and not Flag then
    begin
      Flag := True;
      DelimPos := i;
    end;

    if (UpperCase(Copy(Str, i, Length(Delimiter))) = UpperCase(Delimiter)) then
    begin
      case Index of
        1: Buffer := Copy(Buffer, 1, Length(Buffer));
        2: Buffer := Copy(Str, DelimPos + Length(Delimiter), Length(Str) - DelimPos+Length(Delimiter));
      end;
      
      Break;
    end;
    
    Buffer := Buffer + Str[i];
  end;

  Result := Buffer;
end;

{
  Function name : AsmStrNum
  Description   : Checks and processes variable value depending on its type (string or numeric)
  Parameters    : Str - Variable value to be checked
  Returns       : Returns processed variable value
}
function AsmStrNum(Str : String) : String;
var
  Str1, Str2 : String;
begin
  if ((Str[1] > Chr(47)) and (Str[1] < Chr(58))) or (Str[1] = '$') or (Str[1] = '#') then
  begin
    if Str[1] <> '#' then
      Str := '#' + Str;
  end else begin
    if System.Pos('[', Str) > 0 then
    begin
      Str1 := Extract(Str, '[', 1);
      Str2 := Extract(Str, '[', 2);
      Str := Str1 + _EFF + '[' + Str2;
    end else begin
      if (System.Pos('b_param', LowerCase(Str)) < 1)
         and (System.Pos('w_param', LowerCase(Str)) < 1)
         and (System.Pos('store1', LowerCase(Str)) < 1) then
      begin
        Str := Str + _EFF;
      end;
    end;
  end;

  Result := Str;
end;

{
  Function name : Replace
  Description   : Replaces one occurence of a character in a string with another
  Parameters    : Str - String value to be processed
                  Ch1 - Character to be replaced
                  Ch2 - New character
  Returns       : New string value
}
function Replace(Str : String; Ch1, Ch2 : Char) : String;
var
  i : Integer;
begin
  i := System.Pos(Ch1, Str);
  Delete(Str, i, 1);
  Insert(Ch2, Str, i);
  Result := Str;
//  Result := StuffString(Str, System.Pos(Ch1, Str), 1, Ch2);
end;

{
  Function name : Strip
  Description   : Deletes all occurences of a character in a string
  Parameters    : Str - String value to be processed
                  Ch - Character to be deleted from the string
  Returns       : New string value
}
function Strip(Str : String; Ch : Char) : String;
begin
  Result := StringReplace(Str, Ch, '', [rfReplaceAll, rfIgnoreCase]);
end;

{
  Function: ExtractText
  Description:
    Extracts string between characters Ch1 and Ch2  
  Parameters:
    Str - Input string
    Ch1 - Starting character of examined string
    Ch2 - Last character of examined string  
  Examples:
    Str := Between('ProcX(int a, byte b);', '(', ')');
    Str := Between('PrintE("test")', '"', '"');
  Returns:
    Returns string between characters Ch1 and Ch2
}
function ExtractText(Str : String; Ch1, Ch2 : Char) : String;
begin
  result := Copy(Str, System.Pos(Ch1, Str) + 1, RPos(Ch2, Str) - System.Pos(Ch1, Str) - 1);
end;

{
  Procedure name : MathExpr
  Description    : Processes arithmetic expressions 
  Parameters     : VarStr: Type of variable
                   Str1  : Variable name
                   Str2  : Variable value
                   Flag  :
                   Index : Processed element in set or array
}
procedure MathExpr(VarStr, Str1, Str2 : String; Flag, Index : Byte);
var
  n, i : Integer;
  Oper : Char;
  boolAdd8 : Boolean = True;
  OperMemn, OperIncMemn, StrX : String;
begin
  if System.Pos(';', Str2) > 0 then
    Str2 := Extract(Str2, ';', 1);

  if System.Pos('*', Str2) > 0 then Oper := '*'
  else if System.Pos('/', Str2) > 0 then Oper := '/'
  else if System.Pos('+', Str2) > 0 then Oper := '+'
  else if System.Pos('-', Str2) > 0 then Oper := '-'
  else if System.Pos('MOD', UpperCase(Str2)) > 0 then Oper := 'M'
  else if System.Pos('LSH', UpperCase(Str2)) > 0 then Oper := 'L'
  else if System.Pos('RSH', UpperCase(Str2)) > 0 then Oper := 'R'
  else if System.Pos('XOR', UpperCase(Str2)) > 0 then Oper := 'X'
  else if System.Pos('!', Str2) > 0 then Oper := '!'
  else if System.Pos('&', Str2) > 0 then Oper := '&'
  else if System.Pos('%', Str2) > 0 then Oper := '%'
  else Oper := '+';

  if Oper = 'M' then
  begin
    n := System.Pos('MOD', UpperCase(Str2));
    Delete(Str2, n + 1, 1);
    Insert(' ', Str2, n + 1);
    Delete(Str2, n + 2, 1);
    Insert(' ', Str2, n + 1);
    Str2 := Strip(Str2, ' ');
  end else if Oper = 'L' then
  begin
    n := System.Pos('LSH', UpperCase(Str2));
    Delete(Str2, n + 1, 1);
    Insert(' ', Str2, n + 1);
    Delete(Str2, n + 2, 1);
    Insert(' ', Str2, n + 1);
    Str2 := Strip(Str2, ' ');
  end else if Oper = 'R' then
  begin
    n := System.Pos('RSH', UpperCase(Str2));
    Delete(Str2, n + 1, 1);
    Insert(' ', Str2, n + 1);
    Delete(Str2, n + 2, 1);
    Insert(' ', Str2, n + 1);
    Str2 := Strip(Str2, ' ');
  end else if Oper = 'X' then
  begin
    n := System.Pos('XOR', UpperCase(Str2));
    Delete(Str2, n + 1, 1);
    Insert(' ', Str2, n + 1);
    Delete(Str2, n + 2, 1);
    Insert(' ', Str2, n + 1);
    Str2 := Strip(Str2, ' ');
  end;      
  
  Split(UpperCase(Str2), UpperCase(Oper), []);
  if StrBuf.Count < 1 then
  begin
    writeln('Error: Parameter mismatch!');
    Exit;
  end;

  // Check the number of operands
  if StrBuf.Count = 1 then
  begin
    if Flag = 1 then
    begin
      // Processing type variable
      //
      if System.Pos('.', StrBuf[0]) > 0 then
      begin
        StrX := Extract(StrBuf[0], '.', 1);
        Str2 := Extract(StrBuf[0], '.', 2);
        
        for n := 1 to GVarCnt do
        begin
          if (GVar[n].VarName = StrX) and (GVar[n].ParentType = 'T9') then
          begin
            CodeBuf.Add(mvwa(VarStr) + StrX + _EFF + '.' + Str2 + _EFF + ' ' + AsmStrNum(Str1));
            Break;
          end; 
        end;

      // Other variable expression processing
      //
      end else begin
        StrX := '';
        
        // Routing array variable to the vector
        for n := 1 to GVarCnt do
        begin
          if (UpperCase(GVar[n].VarName) = UpperCase(Str1)) and (GVar[n].Value <> '') then
          begin
            for i := 1 to GVarCnt2 do
            begin
              if (UpperCase(GVar2[i].VarName) = UpperCase(StrBuf[0])) and (System.Pos(GVar2[i].VarType, 'T5T6T7T8') > 0) then
              begin
                StrX := 'Addr';
                CodeBuf.Add(mvwa(VarStr) + '#' + AsmStrNum(StrBuf[0]) + ' ' + AsmStrNum(Str1));
                Break;
              end;
            end;
          end;
        end;
        
        // Routing routine to the vector
        for n := 1 to GVarCnt do
        begin
          if (UpperCase(GVar[n].VarName) = UpperCase(Str1)) and (GVar[n].Value <> '') then
          begin
            for i := 0 to ProcCount do
            begin
              if UpperCase(Copy(ProcBuf[i], 5, Length(ProcBuf[i]) - 4)) = UpperCase(StrBuf[0]) then
              begin
                StrX := 'Addr';
                CodeBuf.Add(mvwa(VarStr) + '#' + StrBuf[0] + ' ' + AsmStrNum(Str1));
                Break;
              end;
            end;
          end;
        end;

        // Other variable expression processing
        if StrX = '' then
          CodeBuf.Add(mvwa(VarStr) + AsmStrNum(StrBuf[0]) + ' ' + AsmStrNum(Str1));
      end;
    end else begin
      // BYTE and CHAR ARRAY
      if (System.Pos(VarStr, 'T5T6') > 0) and (GVar2[Index].Location <> 'T5') then
      begin
        if GVar2[Index].Location <> 'SET' then
          CodeBuf.Add(' mva ' + AsmStrNum(StrBuf[0]) + ' ' + Str1 + _EFF)
        else
          CodeBuf.Add(' mva ' + AsmStrNum(StrBuf[0]) + ' ' + Str1)
      
      // INT and CARD ARRAY
      end else if System.Pos(VarStr, 'T7T8') > 0 then
      begin
        if GVar2[Index].Location <> 'SET' then
        begin
          if Flag = 2 then
            CodeBuf.Add(' mwa ' + AsmStrNum(StrBuf[0]) + ' ' +  Str1)
          else begin
            StrX := Extract(Str1, '[', 1);
            Str1 := Extract(Str1, '[', 2);
            Str1 := Copy(Str1, 1, Length(Str1) - 1);
            CodeBuf.Add('');
            CodeBuf.Add(' ldx #0');
            CodeBuf.Add('for_loop_' + IntToStr(PrgVar.Pointer));
            CodeBuf.Add(' mva _str_buffer_' + IntToStr(PrgVar.Pointer) + ',x ' + StrX + 'array_str_' + Str1 + ',x');
            CodeBuf.Add(' inx');
            CodeBuf.Add(' cpx #' + IntToStr(Length(StrBuf[0])));
            CodeBuf.Add(' jcc for_loop_' + IntToStr(PrgVar.Pointer));
            CodeBuf.Add('');
            SData[Cnt] := '_str_buffer_' + IntToStr(PrgVar.Pointer) + ' .byte ' + QuotedStr(StrBuf[0]) + ',$9b';
            Inc(Cnt);
          end;
        end; 
      end;
    end;
  end else begin
    lMath := True;

    // Check to see if any of expression variables is of type CARD
    for n := 1 to GVarCnt do
    begin
      //for i := 0 to strbuf.Count - 1 do
      //begin
      if StrBuf.IndexOf(GVar[n].VarName) >= 0 then
        //if LowerCase(GVar[n].VarName) = LowerCase(StrBuf[i]) then
        begin
          if GVar[n].ML_type = 'word' then
          begin
            boolAdd8 := False;
            Break;
          end;
        end;
      //end;
    end;

    // Check to see if main expression variable is of type CARD  
    for n := 1 to GVarCnt do
    begin
      if LowerCase(GVar[n].VarName) = LowerCase(Str1) then
      begin
        if GVar[n].ML_type = 'word' then
        begin
          boolAdd8 := False;
          Break;
        end;
      end;
    end;

    if ((Copy(StrBuf[0], 1, 1) > Chr(47)) and (Copy(StrBuf[0], 1, 1) < Chr(58))) then
      if StrToInt(StrBuf[0]) > 255 then boolAdd8 := False;

    if ((Copy(StrBuf[1], 1, 1) > Chr(47)) and (Copy(StrBuf[1], 1, 1) < Chr(58))) then
      if StrToInt(StrBuf[1]) > 255 then boolAdd8 := False;

    if System.Pos(VarStr, 'T3T4') > 0 then boolAdd8 := False;

    Str1 := LowerCase(Str1);
    StrBuf[0] := LowerCase(StrBuf[0]);
    StrBuf[1] := LowerCase(StrBuf[1]);

    //
    // Process the arithmetic operation
    //
    
    case Oper of
      '*': begin
             if FuncCheckTrue(StrBuf[0]) then StrBuf[0] := 'STORE1';
             if FuncCheckTrue(StrBuf[1]) then StrBuf[1] := 'STORE1';

             if boolAdd8 then
               CodeBuf.Add(' Mul8 ' + AsmStrNum(StrBuf[0]) + ', ' + AsmStrNum(StrBuf[1]))
             else
               CodeBuf.Add(' Mul16 ' + AsmStrNum(StrBuf[0]) + ', ' + AsmStrNum(StrBuf[1]));
           end;
      '/': begin
             if FuncCheckTrue(StrBuf[0]) then StrBuf[0] := 'STORE1';
             if FuncCheckTrue(StrBuf[1]) then StrBuf[1] := 'STORE1';

             if boolAdd8 then
               CodeBuf.Add(' Div8 ' + AsmStrNum(StrBuf[0]) + ', ' + AsmStrNum(StrBuf[1]))
             else
               CodeBuf.Add(' Div16 ' + AsmStrNum(StrBuf[0]) + ', ' + AsmStrNum(StrBuf[1]));
           end;
      '+': begin
             if boolAdd8 then
             begin
               OperMemn := ' adb ';
               OperIncMemn := ' inc ';
             end else begin
               OperMemn := ' adw ';
               OperIncMemn := ' inw ';
             end;
               
             // a = a + 1
             // a = 1 + a
             if ((Str1 = StrBuf[0]) or (Str1 = StrBuf[1]))
                and (((StrBuf[0] = '1') or (StrBuf[1] = '1'))) then
             begin
               //CodeBuf.Add(' ' + AsmStrNum(StrBuf[0]) + ' ' + AsmStrNum(StrBuf[1]));
               CodeBuf.Add(OperIncMemn + Str1 + _EFF);
             end else begin
               // a = b + c
               if (StrBuf[0] <> StrBuf[1]) and (Str1 <> StrBuf[0]) and (Str1 <> StrBuf[1]) then
               begin
                 if (((Copy(StrBuf[0], 1, 1) > Copy(Chr(47), 1, 1)) and (Copy(StrBuf[0],1,1) < Copy(Chr(58),1,1)))
                    or (Copy(StrBuf[0],1,1) = '$')) and FuncCheck then
                 begin
                   CodeBuf.Add(' mwa ' + AsmStrNum(StrBuf[0]) + ' STORE1');
                   StrBuf[0] := 'STORE1'
                 end;
                 
                 if FuncCheckTrue(StrBuf[0]) then
                   CodeBuf.Add(OperMemn + 'STORE1 ' + AsmStrNum(StrBuf[1]) + ' ' + AsmStrNum(Str1))
                 else if FuncCheckTrue(StrBuf[1]) then
                   CodeBuf.Add(OperMemn + 'STORE1 ' + AsmStrNum(StrBuf[0]) + ' ' + AsmStrNum(Str1))
                 else begin
                   CodeBuf.Add(OperMemn + AsmStrNum(StrBuf[0]) + ' ' + AsmStrNum(StrBuf[1]) + ' ' + AsmStrNum(Str1));
                 end;
               end else if ((Str1 = StrBuf[0]) or (Str1 = StrBuf[1])) then
               begin
                 if (Str1 = StrBuf[0]) then
                   CodeBuf.Add(OperMemn + AsmStrNum(Str1) + ' ' + AsmStrNum(StrBuf[1]))
                 else
                   CodeBuf.Add(OperMemn + AsmStrNum(Str1) + ' ' + AsmStrNum(StrBuf[0]));
               end;
                 
               if StrBuf.Count > 2 then
               begin
                 for i := 2 to StrBuf.Count - 1 do
                   CodeBuf.Add(OperMemn + AsmStrNum(Str1) + ' ' + AsmStrNum(StrBuf[i]));
               end;
             end;
           end;
      '-': begin
             if boolAdd8 then
             begin
               OperMemn := ' sbb ';
               OperIncMemn := ' dec ';
             end else begin
               OperMemn := ' sbw ';
               OperIncMemn := ' dew ';
             end;

             // a = a - 1
             // a = 1 - a
             if ((Str1 = StrBuf[0]) or (Str1 = StrBuf[1]))
                and (((StrBuf[0] = '1') or (StrBuf[1] = '1'))) then
             begin
               CodeBuf.Add(OperIncMemn + Str1 + _EFF);
             end else begin
               // a = b - c
               if (StrBuf[0] <> StrBuf[1]) and (Str1 <> StrBuf[0]) and (Str1 <> StrBuf[1]) then
               begin 
                 if (((Copy(StrBuf[0], 1, 1) > Copy(Chr(47), 1, 1)) and (Copy(StrBuf[0],1,1) < Copy(Chr(58),1,1)))
                    or (Copy(StrBuf[0],1,1) = '$')) and FuncCheck then
                 begin
                   CodeBuf.Add(' mwa ' + AsmStrNum(StrBuf[0]) + ' STORE1');
                   StrBuf[0] := 'STORE1'
                 end;
                          
                 if FuncCheckTrue(StrBuf[0]) then
                   CodeBuf.Add(OperMemn + 'STORE1 ' + AsmStrNum(StrBuf[1]) + ' ' + AsmStrNum(Str1))
                 else if FuncCheckTrue(StrBuf[1]) then
                   CodeBuf.Add(OperMemn + AsmStrNum(StrBuf[0]) + ' STORE1 ' + AsmStrNum(Str1))
                 else
                   CodeBuf.Add(OperMemn + AsmStrNum(StrBuf[0]) + ' ' + AsmStrNum(StrBuf[1]) + ' ' + AsmStrNum(Str1));

                 //if FuncCheckTrue then
                 //  CodeBuf.Add(OperMemn + ' STORE1 ' + AsmStrNum(StrBuf[1]) + ' ' + AsmStrNum(Str1))
                 //else                                 
                 //  CodeBuf.Add(' ' + OperMemn + AsmStrNum(StrBuf[0]) + ' ' + AsmStrNum(StrBuf[1]) + ' ' + AsmStrNum(Str1));

               end else if ((Str1 = StrBuf[0]) or (Str1 = StrBuf[1])) then begin
                 if (Str1 = StrBuf[0]) then
                   CodeBuf.Add(' ' + OperMemn + AsmStrNum(Str1) + ' ' + AsmStrNum(StrBuf[1]))
                 else
                   CodeBuf.Add(' ' + OperMemn + AsmStrNum(Str1) + ' ' + AsmStrNum(StrBuf[0]));
               end;

               if strbuf.Count > 2 then
               begin
                 for i := 2 to strbuf.Count - 1 do
                   CodeBuf.Add(' ' + OperMemn + AsmStrNum(Str1) + ' ' + AsmStrNum(StrBuf[i]));
               end;
             end;
           end;
      'M': CodeBuf.Add(' Mod8 ' + AsmStrNum(StrBuf[0]) + ', ' + AsmStrNum(StrBuf[1]));
      'L':
      begin
        CodeBuf.Add(' lda ' + AsmStrNum(StrBuf[0]));
        for n := 1 to StrToInt(StrBuf[1]) do CodeBuf.Add(' asl @');
        CodeBuf.Add(' sta STORE1');
      end;
      'R':
      begin
        CodeBuf.Add(' lda ' + AsmStrNum(StrBuf[0]));
        for n := 1 to StrToInt(StrBuf[1]) do CodeBuf.Add(' lsr @');
        CodeBuf.Add(' sta STORE1');
      end;
      'X', '!':
      begin
        CodeBuf.Add(' lda ' + AsmStrNum(StrBuf[0]));
        CodeBuf.Add(' eor ' + AsmStrNum(StrBuf[1]));
        CodeBuf.Add(' sta STORE1');
      end;
      '&':
      begin
        CodeBuf.Add(' lda ' + AsmStrNum(StrBuf[0]));
        CodeBuf.Add(' and ' + AsmStrNum(StrBuf[1]));
        CodeBuf.Add(' sta STORE1');
      end;
      '%':
      begin
        CodeBuf.Add(' lda ' + AsmStrNum(StrBuf[0]));
        CodeBuf.Add(' ora ' + AsmStrNum(StrBuf[1]));
        CodeBuf.Add(' sta STORE1');
      end;
    end;

    if (Oper <> '+') and (Oper <> '-') then
    begin    
      if Flag = 1 then
      begin
        CodeBuf.Add(mvwa(VarStr) + ' STORE1 ' + Str1 + _EFF);
      end else begin
        if System.Pos(VarStr, 'T5T6') > 0 then  // BYTE and CHAR array
          CodeBuf.Add(' mva STORE1 ' + Str1 + _EFF)
        else if System.Pos(VarStr, 'T7T8') > 0 then  // INT and CARD array
          CodeBuf.Add(' mwa STORE1 ' + Str1 + _EFF);
      end;
    end;
  end;
end;

{
  Procedure name : GenerateCode
  Description    : Processes and generates Mads source code listing file 
  Parameters     : None
}
procedure GenerateCode;
var
  i, j : LongInt;
  CommentStr, Buffer : String;
begin
  FilenameSrc := ExtractFileName(meditEff_src_filename);
  FilenameSrc := Copy(FilenameSrc, 1, RPos('.',FilenameSrc) - 1) + '.';  
  FilenameBin := FilenameSrc + meditMADS_bin_ext;
  FilenameSrc := FilenameSrc + meditMADS_src_ext;
  FilenameOrig := FilenameSrc;
  
  AssignFile(fASM, meditMADS_src_dir + FilenameSrc);
  Rewrite(fASM);
  
  WriteLn(fASM, ' org $' + IntToStr(meditAddr));
  WriteLn(fASM, '');
  WriteLn(fASM, ' icl ' + QuotedStr(meditMADS_rtl_dir + 'equates.asm'));
  WriteLn(fASM, ' icl ' + QuotedStr(meditMADS_rtl_dir + 'common.asm'));
  WriteLn(fASM, '');

  if sElseIf in flags then  
    WriteLn(fASM, ' .var else_flag .byte');
  
  Cnt := 1;
  Cnt2 := 1;
  ProcML_cnt := 0;
  MemCnt := Hex2Dec(IntToStr(meditMLAddr));  // default 32768 (dec) $8000 (hex)
  lGraphicsFlag := False;
  lGr := False;
  ForCnt := 0;
  lInput := False;
  word_Cnt := 1;
  
  CodeBuf.Clear;

  for i := 1 to CR_LF do
  begin
    CurLine := i;    
    Buffer := Strip(TextBuf[i], ' ');

    if Buffer[1] = ';' then Continue;
    
    sc_Define;
    sc_Var;
    sc_Var2;
    sc_Array;
  end;

  for i := 1 to GVarCnt do
  begin
    // Extract comments from the variable name declaration
    Split(GVar[i].VarName, ';', []);
    
    if StrBuf.Count <= 1 then
      CommentStr := ''
    else begin
      GVar[i].VarName := StrBuf[0];
      CommentStr := '  ; ' + StrBuf[1];
    end;         

    if UpperCase(GVar[i].ParentType) = UpperCase(GVar[i].OrigType) then
    begin
      if GVar[i].Value <> '' then
        CodeBuf.Add(GVar[i].VarName + _EFF + ' equ ' + GVar[i].Value + CommentStr)
      else begin
        if GVar[i].InitValue = -1 then
          CodeBuf.Add(' .var ' + GVar[i].VarName + _EFF + ' .' + GVar[i].ML_type + CommentStr)
        else
          CodeBuf.Add(' .var ' + GVar[i].VarName + _EFF + '=' +
                      IntToStr(GVar[i].InitValue) + ' .' + GVar[i].ML_type + CommentStr);
      end;
    end else begin
      if System.Pos(GVar[i].ParentType, 'T10') > 0 then  // POINTER
      begin
        if LowerCase(GVar[i].ML_type) = 'word' then
          CodeBuf.Add(' .var ' + GVar[i].VarName + _EFF + ' .' + GVar[i].ML_type + CommentStr)
        else
          CodeBuf.Add(' .var ' + GVar[i].VarName + _EFF + ' .byte' + CommentStr);
      end;
    end;
  end;

  for i := 1 to GVarCnt2 do
  begin
    // Extract comments from the variable name declaration
    Split(GVar2[i].VarName, ';', []);
    
    if StrBuf.Count > 1 then
    begin
      GVar2[i].VarName := StrBuf[0];
      CommentStr := '  ; ' + StrBuf[1]
    end else begin
      CommentStr := '';
    end;

    // BYTE and CHAR ARRAY
    //
    if System.Pos(GVar2[i].VarType, 'T5T6') > 0 then
    //if (UpperCase(GVar2[i].VarType) = 'BYTE ARRAY') or (UpperCase(GVar2[i].VarType) = 'CHAR ARRAY') then
    begin
      // BYTE ARRAY str1="Text"
      if GVar2[i].Location = 'T5' then
      begin
        if GVar2[i].Dim = 0 then
          CodeBuf.Add(' .array ' + GVar2[i].VarName + _EFF + ' .byte = $ff')
        else
          CodeBuf.Add(' .array ' + GVar2[i].VarName + _EFF + ' [' + IntToStr(GVar2[i].Dim + 1) + '] .byte = $ff');

        CodeBuf.Add(' [0] = ' + QuotedStr(GVar2[i].Value) + ',$9b');
      end else if GVar2[i].Value <> '' then
      begin
        GVar2[i].Value := Copy(GVar2[i].Value, 1, Length(GVar2[i].Value) - 2);

        if GVar2[i].Dim = 0 then
          CodeBuf.Add(' .array ' + GVar2[i].VarName + _EFF + ' .byte = $ff' + CommentStr)
        else
          //CodeBuf.Add(GVar2[i].VarName + ' .array [' + IntToStr(GVar2[i].Dim) + '] = $ff' + CommentStr);
          CodeBuf.Add(' .array ' + GVar2[i].VarName + _EFF + ' [' + IntToStr(GVar2[i].Dim + 1) + '] .byte = $ff' + CommentStr);
        
        CodeBuf.Add(' ' + GVar2[i].Value);
      end else begin
        //CodeBuf.Add(GVar2[i].VarName + ' .array [' + IntToStr(GVar2[i].Dim) + '] = $ff' + CommentStr);
        CodeBuf.Add(' .array ' + GVar2[i].VarName + _EFF + ' [' + IntToStr(GVar2[i].Dim + 1) + '] .byte = $ff' + CommentStr);
      end;

      CodeBuf.Add(' .end');

      if (sFor in flags) or (sWhile in flags) then
      begin
        CodeBuf.Add(' .var array_buffer_' + GVar2[i].VarName + _EFF + ' .word');
        CodeBuf.Add(' .var array_index_' + GVar2[i].VarName + _EFF + ' .byte');
      end;

    // INT ARRAY or CARD ARRAY
    //
    end else if System.Pos(GVar2[i].VarType, 'T7T8') > 0 then
    begin
      // CARD ARRAY values=[1 2 3 4 5 6 7]
      //if GVar2[i].Value = '' then
        //CodeBuf.Add(GVar2[i].VarName + ' .array [' + IntToStr(GVar2[i].Dim) + '] = $ff' + CommentStr)
        //CodeBuf.Add(' .array ' + GVar2[i].VarName + ' [' + IntToStr(GVar2[i].Dim + 1) + '] .word = $ff' + CommentStr)
      //else begin
      GVar2[i].Value := Copy(GVar2[i].Value, 1, Length(GVar2[i].Value) - 2);

      if GVar2[i].Dim = 0 then
        CodeBuf.Add(' .array ' + GVar2[i].VarName + _EFF + ' .word = $ff' + CommentStr)
      else
        CodeBuf.Add(' .array ' + GVar2[i].VarName + _EFF + ' [' + IntToStr(GVar2[i].Dim + 1) + '] .word = $ff' + CommentStr);

      CodeBuf.Add(' ' + GVar2[i].Value);
      //end;

      CodeBuf.Add(' .end');

      if GVar2[i].Location <> 'SET' then
      begin
        for j := 0 to GVar2[i].Dim do
        begin
          //CodeBuf.Add(GVar2[i].VarName + '_array_str_' + IntToStr(j) + ' .array [40] = $ff' + CommentStr);
          CodeBuf.Add(' .array ' + GVar2[i].VarName + _EFF + 'array_str_' + IntToStr(j) + ' [40] .byte = $ff');
          CodeBuf.Add(' .end');
        end;
      end;

      if (sFor in flags) or (sWhile in flags) then
      begin           
        CodeBuf.Add(' .var array_buffer_' + GVar2[i].VarName + _EFF + ' .word');
        CodeBuf.Add(' .var array_index_' + GVar2[i].VarName + _EFF + ' .byte');
      end;
    end else if System.Pos(GVar2[i].VarType, 'T9') > 0 then  // TYPE
    begin
      CodeBuf.Add(' .struct ' + GVar2[i].VarName + _EFF + CommentStr);

      for j := 1 to 255 do
        if UpperCase(GVar[j].ParentType) = UpperCase(GVar2[i].VarName) then
          CodeBuf.Add(' ' + GVar[j].VarName + _EFF + ' .' + GVar[j].ML_type);

      CodeBuf.Add(' .ends')
    end;
  end;

  for i := 0 to CodeBuf.Count - 1 do
  begin
    WriteLn(fASM, CodeBuf[i]);
  end;

  WriteLn(fASM, '');

  Icl := 0;
  lInclude := False;
  flags := [];
  
  for i := 1 to CR_LF do
  begin
    CurLine := i;
    sc_Include;
  end;
  
  // Read INCLUDE source file
  ReadInclude;
  lInclude := False;

  if lIncludeX and not ReadOrig then Exit;
//  if lIncludeX then
//    if not ReadOrig then Exit;    
    
  CodeBuf.Clear;
  GenLoop(lInclude);    
   
  for i := 0 to CodeBuf.Count - 1 do
    WriteLn(fASM, CodeBuf[i]);

  DeviceCheck;
  
  WriteLn(fASM, ' jmp *');
  WriteLn(fASM, '');
  sc_ML_data;
  WriteLn(fASM, '');
  WriteLn(fASM, '.link ' + AnsiQuotedStr(meditMADS_rtl_dir + 'runtime.obx', ''''));
  sc_Lib(meditMADS_rtl_dir);
  WriteLn(fASM, '');

  sc_Data;

  WriteLn(fASM, '');

  if lIncludeX and not lInclude then
    WriteLn(fASM, ' run ' + Copy(ProcBuf[ProcCount2 - 1], 5, Length(ProcBuf[ProcCount2 - 1]) - 4) + _REFF)
  else
    WriteLn(fASM, ' run ' + Copy(ProcBuf[ProcCount], 5, Length(ProcBuf[ProcCount]) - 4) + _REFF);

  CloseFile(fASM);
end;

end.
