{ 
  Program    : Effectus - Atari MADS cross-assembler/parser for Action! language
  Version    : 0.0.16

  Unit file  : common.pas
  Description: Common library
  
  Author: Bostjan Gorisek, Slovenia

  Program compiled with Free Pascal 2.6.0
  Reference: http://www.freepascal.org/
  
  This is cource code generator for MADS Assembler, which then generates
  executable files from parser of Action! language for 8-bit Atari home
  computers
  References:
  http://mads.atari8.info/
  http://gury.atari8.info/effectus/ 
  
  This is open source and freeware project!
}
unit Common;
{$ifdef FPC}{$mode objfpc}{$h+}{$endif}

interface

uses
 sysutils, Classes, Process, INIfiles, strutils;
 
procedure GetFuncs;
//procedure GetProcs;
procedure CheckLibProc;
procedure DeviceCheck;
procedure ReadSource;
procedure sc_Lib(RtlDir : String);
procedure RunMads(MADS_dir, MADS_src_dir, MADS_bin_dir, MADS_output_dir : String);
procedure RunMadsLinux(MADS_dir, MADS_src_dir, MADS_bin_dir, MADS_output_dir : String);
function VarDeclCheck(StrBuf: String) : Boolean;
function ExprCheck(StrBuf: String) : Boolean;
procedure ReadCfg;
function ReadOrig : Boolean;
procedure ReadInclude;
procedure GenerateCode;
procedure Split(Str : String; Delimiter : String);
procedure SplitEx(Str : String; Delimiter : Char);
function Extract(Str : String; Delimiter : String; Index: Integer) : String;
function ExtractEx(Str : String; Delimiter : Char; Index: Integer) : String;
function AsmStrNum(Str : String) : String;
function Replace(Str : String; Ch1, Ch2 : Char): String;
function Strip(Str : String; Ch : Char) : String;
procedure MathExpr(VarStr, Str1, Str2 : String; Flag, Index : Byte);
function StrBetween(Str : String; Ch1, Ch2 : Char) : String;
function StrBetweenEx(Str : String; Ch1, Ch2 : Char) : String;

implementation

uses
  Decl, Core;

procedure GetFuncs;
begin
  FuncList.Add('Peek');
  FuncList.Add('Stick');
  FuncList.Add('Strig');
  FuncList.Add('Paddle');
  FuncList.Add('Ptrig');
  FuncList.Add('Rand');
  FuncList.Add('PeekC');
  FuncList.Add('GetD');
  FuncList.Add('Locate');
  FuncList.Add('ValB');
  FuncList.Add('ValI');
  FuncList.Add('ValC');
end;

procedure CheckLibProc;
var
  i : Byte;
begin
  if System.Pos(';', TextBuf[CR_LF]) > 0 then Exit; 
    
  for i := 0 to GrProcs.Count-1 do
    if (System.Pos(UpperCase(GrProcs[i]) + '(', UpperCase(TextBuf[CR_LF])) > 0) or
       (System.Pos(UpperCase(GrProcs[i]) + '=', UpperCase(TextBuf[CR_LF])) > 0) then
      lGraphics := True;

  for i := 0 to SoundProcs.Count-1 do
    if System.Pos(UpperCase(SoundProcs[i]) + '(', UpperCase(TextBuf[CR_LF])) > 0 then
      lSound := True;

  for i := 0 to PrintFProcs.Count-1 do
    if System.Pos(UpperCase(PrintFProcs[i]) + '(', UpperCase(TextBuf[CR_LF])) > 0 then
      lPrintF := True;

  for i := 0 to IOProcs.Count-1 do
    if System.Pos(UpperCase(IOProcs[i]) + '(', UpperCase(TextBuf[CR_LF])) > 0 then
      lIO := True;

  for i := 0 to PrintFDProcs.Count-1 do
    if System.Pos(UpperCase(PrintFDProcs[i]) + '(', UpperCase(TextBuf[CR_LF])) > 0 then
      lPrintFD := True;

  for i := 0 to ControllerProcs.Count-1 do
    if System.Pos(UpperCase(ControllerProcs[i]) + '(', UpperCase(TextBuf[CR_LF])) > 0 then
      lControllers := True;                  
end;

procedure DeviceCheck;
var
  i : Byte;
begin
  for i := 0 to 7 do
  begin
    if l_IO_error[i] then
    begin
      WriteLn(fASM, '');
      WriteLn(fASM, 'stop' + IntToStr(i));
      WriteLn(fASM, ' close #$' + IntToStr(i) + '0');
      WriteLn(fASM, '');
    end;
  end;
end;

{ Read Effectus source code and extract every line }
procedure ReadSource;
var
  MemBuf, TempX : String[255];
  Count, i, k : LongInt;
  TempBuf, Buffer : String;
  FlagX : Byte;
begin
  CR_LF := 0;
  FlagX := 0;
  TempX := '';

  for k := 0 to 254 do
  begin  
    FileSeek(f, 255 * k, fsFromBeginning);
    Count := FileRead(f, MemBuf, 255);
    TempBuf := '';
    
    for i := 0 to Count - 1 do
    begin
      TempBuf := TempBuf + MemBuf[i];
      
      if (MemBuf[i-1] + MemBuf[i] = LineEnding) or (MemBuf[i] = LineEnding) then
      begin
        Inc(CR_LF);
        
        if FlagX = 0 then
          TextBuf[CR_LF] := Copy(TempBuf, 1, Length(TempBuf))          
        else
        begin
          TextBuf[CR_LF] := TempX + Copy(TempBuf, 1, Length(TempBuf));
          FlagX := 0;          
        end;
        
        TempBuf := '';
        
        CheckLibProc;  // search library procedures
        
        // Retrieve all procedures and functions (PROC, FUNC)
        //
        if ((System.Pos(UpperCase('PROC '), UpperCase(TextBuf[CR_LF])) > 0)
           or (System.Pos(UpperCase('FUNC '), UpperCase(TextBuf[CR_LF])) > 0)) then
           //and (System.Pos(';', TextBuf[CR_LF]) < 1) and (System.Pos('"', TextBuf[CR_LF]) < 1) then
        begin                  
          Inc(ProcCount);
          
          if lIncFlag then Inc(ProcCount2);
          
          if (System.Pos(UpperCase('PROC '), UpperCase(TextBuf[CR_LF])) > 0) then
          begin
            Buffer := Extract(TextBuf[CR_LF], 'PROC ', 2);
            
            if System.Pos('=*', TextBuf[CR_LF]) > 0 then
              Buffer := Extract(Buffer, '=*', 1)
            else
              Buffer := Extract(Buffer, '(', 1);

            ProcBuf.Add('PROC' + Buffer);
          end
          else
          begin            
            Buffer := Extract(TextBuf[CR_LF], ' FUNC ', 2);
            
            if System.Pos('=*', TextBuf[CR_LF]) > 0 then
              Buffer := Extract(Buffer, '=*', 1)
            else
              Buffer := Extract(Buffer, '(', 1);

            ProcBuf.Add('FUNC' + Buffer);
          end;                                        
        end;
      end
      else
      begin
        if i = 254 then
        begin
          TempX := Copy(TempBuf, 1, Length(TempBuf));
          FlagX := 1;          
        end;
      end;   
    end;
  end;
  
  if ProcCount > 0 then Dec(ProcCount);  
end;

// Link libraries
procedure sc_Lib(RtlDir : String);
begin
  if lGraphics then
    WriteLn(fASM, '.link ' + AnsiQuotedStr(RtlDir + 'graphics.obx', ''''));
  if lSound then
    WriteLn(fASM, '.link ' + AnsiQuotedStr(RtlDir + 'sound.obx', ''''));
  if lPrintF or lGr then
    WriteLn(fASM, '.link ' + AnsiQuotedStr(RtlDir + 'printf.obx', ''''));
  if lIO then
    WriteLn(fASM, '.link ' + AnsiQuotedStr(RtlDir + 'io.obx', ''''));
  if lPrintFD then
    WriteLn(fASM, '.link ' + AnsiQuotedStr(RtlDir + 'printfd.obx', ''''));
  if lMath then
    WriteLn(fASM, '.link ' + AnsiQuotedStr(RtlDir + 'math.obx', ''''));
  if lControllers then
    WriteLn(fASM, '.link ' + AnsiQuotedStr(RtlDir + 'controllers.obx', ''''));
end;

{ Generate MADS binary file from generated source code }
procedure RunMads(MADS_dir, MADS_src_dir, MADS_bin_dir, MADS_output_dir : String);
var
  AProcess : TProcess;
  AStringList : TStringList;
  s1, s2 : String;
  i : LongInt;
  lError, lWarnings : Boolean;
begin
  s1 := MADS_dir + 'mads.exe';
//  s1 := editMADS_dir.Text + 'mads.exe';
  s2 := AnsiQuotedStr(MADS_src_dir + FilenameOrig, '"') +
        ' -o:' + AnsiQuotedStr(MADS_bin_dir + FilenameBin, '"');
//  s2 := AnsiQuotedStr(editMADS_src_dir.Text + FilenameOrig, '"') +
//        ' -o:' + AnsiQuotedStr(editMADS_bin_dir.Text + FilenameBin, '"');
//        ' -b:' + editAddr.Text;
  s1 := s1 + ' ' + s2 + ' -x';  // Exclude unreferenced procedures
  
  // Now we will create the TProcess object, and
   // assign it to the var AProcess.
   AProcess := TProcess.Create(nil);
   
   // Create the TStringList object.
   AStringList := TStringList.Create;
   
   // Tell the new AProcess what the command to execute is.
   // Let's use the FreePascal compiler
   AProcess.CommandLine := s1;
   
   // We will define an option for when the program
   // is run. This option will make sure that our program
   // does not continue until the program we will launch
   // has stopped running. Also now we will tell it that
   // we want to read the output of the file.
   AProcess.Options := AProcess.Options + [poWaitOnExit, poUsePipes];
   
   // Now that AProcess knows what the commandline is 
   // we will run it.
   AProcess.Execute;
   
   // This is not reached until ppc386 stops running.
   
   // Now read the output of the program we just ran
   // into the TStringList.
   AStringList.LoadFromStream(AProcess.Output);
   
   // Check resulted compile status
   mStatusLog := '';
   mStatus := '';
   
   for i := 0 to AStringList.Count-1 do
       mStatusLog := mStatusLog + AStringList[i] + LineEnding;
   
   lError := False;
   
   for i := 0 to AStringList.Count-1 do
     if System.Pos(UpperCase('ERROR:'), AStringList[i]) > 0 then
     begin
       mStatus := AStringList[i] + LineEnding;
       lError := True;
     end else if System.Pos(UpperCase('WARNING:'), AStringList[i]) > 0 then
       lWarnings := True;

   if lWarnings then
     mStatus := mStatus + 'There were some warnings found!' + LineEnding;

   if not lError then
     mStatus := mStatus + 'Compiling was successful!';
   
   // Save the output to a file.
   AStringList.SaveToFile(MADS_output_dir);
//   AStringList.SaveToFile(editMADS_output_dir.Text);
 
   // Now that the file is saved we can free the 
   // TStringList and the TProcess.
   AStringList.Free;
   
   AProcess.Free;
end;

{ Generate MADS binary file from generated source code }
procedure RunMadsLinux(MADS_dir, MADS_src_dir, MADS_bin_dir, MADS_output_dir : String);
var
  Process : TProcess;
  s1, s2 : String;
begin
  s1 := MADS_dir + 'mads';
  s2 := FilenameOrig + ' -o:' + FilenameBin;
  s1 := s1 + ' ' + s2 + ' -x';  // Exclude unreferenced procedures

  Process := TProcess.Create(nil);

  // Run the process
  Process.CommandLine := s1;
  Process.Execute();
	Process.Options := Process.Options + [poWaitOnExit, poUsePipes];
	Process.Free;
end;

{ Check for common words in variable declarations }
function VarDeclCheck(StrBuf : String) : Boolean;
var
  i : ShortInt;
begin
  Result := True;
  
  for i := 1 to 4 do
  begin
    if System.Pos(NotVarDecl[i], UpperCase(StrBuf)) > 0 then
    begin
      Result := False;
      Break;
    end;
  end;
end;

{ Check for common words in variable expressions }
function ExprCheck(StrBuf : String): Boolean;
var
  i : ShortInt;
begin
  Result := True;
  
  for i := 1 to 9 do
  begin
    if System.Pos(NotExpr[i], StrBuf) > 0 then
    begin
      Result := False;
      Break;
    end;
  end;
end;

{ Read configuration file }
procedure ReadCfg;
var
  IniFile : TIniFile;
begin
  IniFile := TIniFile.Create( 'config.ini' );
{$ifdef Unix}
  if FileExists(GetCurrentDir + '/config.ini') then
  begin
    IniFile.ReadIniFile;
    meditAddr := IniFile.ReadInteger( 'SETUP', 'ORG', 3200 );
    meditMADS_dir := IniFile.ReadString('SETUP', 'MADS_DIR', '');
    meditMADS_src_ext := IniFile.ReadString('SETUP', 'MADS_SRC_EXT', 'asm');
    meditMADS_bin_ext := IniFile.ReadString('SETUP', 'MADS_BIN_EXT', 'xex');
    meditMADS_rtl_dir := IniFile.ReadString('SETUP', 'MADS_RTL_DIR', '');
    meditMLAddr := IniFile.ReadInteger( 'SETUP', 'ML_ORG', 8000 );
  end
  else
  begin
    meditAddr := 3200;
    meditMADS_dir := '';
    meditMADS_src_ext := 'asm';
    meditMADS_bin_ext := 'xex';
    meditMADS_rtl_dir := GetCurrentDir + '/lib/';
    meditMLAddr := 8000;
  end;
{$else}
  if FileExists(GetCurrentDir + '\config.ini') then
  begin
    IniFile.ReadIniFile;
    meditAddr := IniFile.ReadInteger( 'SETUP', 'ORG', 3200 );
    meditMADS_dir := IniFile.ReadString('SETUP', 'MADS_DIR', 'c:\atari\mads\');
    meditMADS_src_ext := IniFile.ReadString('SETUP', 'MADS_SRC_EXT', 'asm');
    meditMADS_bin_ext := IniFile.ReadString('SETUP', 'MADS_BIN_EXT', 'xex');
    meditMADS_rtl_dir := IniFile.ReadString('SETUP', 'MADS_RTL_DIR', 'c:\atari\effectus\lib\');
    meditMLAddr := IniFile.ReadInteger( 'SETUP', 'ML_ORG', 8000 );
  end
  else
  begin
    meditAddr := 3200;
    meditMADS_dir := '';
    meditMADS_src_ext := 'asm';
    meditMADS_bin_ext := 'xex';
    meditMADS_rtl_dir := GetCurrentDir + '\lib\';
    meditMLAddr := 8000;
  end;
{$endif}  
  IniFile.Destroy;
end;

{ Open Effectus source file }
function ReadOrig : Boolean;
var
  IsFile : Boolean;
begin
  if (Copy(meditEff_src_filename, 1, 1) = '\') or (Copy(meditEff_src_filename, 1, 1) = '/') then
    meditEff_src_filename := Copy(meditEff_src_filename, 2, Length(meditEff_src_filename)-1);

  f := FileOpen(meditEff_src_filename, fmOpenRead);
  try
    if f = -1 then
    begin
      WriteLn('File does not exist!');
      IsFile := False;
    end
    else
    begin
      ReadSource;
      IsFile := True;
    end;
  finally
    FileClose(f);
    Result := IsFile;
  end;
end;

{ Parse ACTION! source code and generate MADS source code }
procedure ReadInclude;
var
  i : LongInt;
begin
  if not lInclude then Exit;

  //writeln('ASM_icl[icl] = ' + ASM_icl[icl]);
  
  { Open Effectus source file }
  f := FileOpen(ASM_icl[icl], fmOpenRead);
  if f = -1 then
  begin
    WriteLn('Error reading INCLUDE file!');
    Exit;
  end;
  
  ReadSource;
  FileClose(f);

  FilenameSrc := ExtractFileName(ASM_icl[icl]);
  FilenameSrc := Copy(FilenameSrc, 1, RPos('.',FilenameSrc)-1) + '.';  
  FilenameSrc := FilenameSrc + meditMADS_src_ext;
  
  AssignFile(fASM_lib, meditMADS_src_dir + FilenameSrc);
  Rewrite(fASM_lib);
  CodeBuf.Clear;
  GenLoop(lInclude);

  for i := 0 to CodeBuf.Count - 1 do
    WriteLn(fASM_lib, CodeBuf[i]);
  
  CloseFile(fASM_lib);
end;

procedure Split(Str : String; Delimiter : String);
var
  Buffer : String;
  i : Integer;
  Len : Byte;
begin
  Buffer := '';
  Len := Length(Delimiter);
  StrBuf.Clear;

  if System.Pos(Delimiter, Str) < 1 then
  begin
    StrBuf.Add(Trim(Str));
    Exit;
  end;

  for i := 1 to Length(Str) do
  begin
    if Copy(Str, i, Len) = Delimiter then
    begin
      if System.Pos(Delimiter, Buffer) > 0 then
        Buffer := Trim(Copy(Buffer, Len+1, Length(Buffer)-Len))
      else
        Buffer := Trim(Copy(Buffer, 1, Length(Buffer)));
      
      StrBuf.Add(Buffer);
      Buffer := '';
    end;
    
    if i = Length(Str) then
    begin
      Buffer := Buffer + Copy(Str, i, 1);      
      Buffer := Trim(Copy(Buffer, Len+1, Length(Buffer)-Len));
      StrBuf.Add(Buffer);
    end;
    
    Buffer := Buffer + Copy(Str, i, 1);
  end;
end;

procedure SplitEx(Str : String; Delimiter : Char);
var
  Buffer : String;
  i : Integer;
begin
  Buffer := '';
  StrBuf.Clear;
  
  if System.Pos(UpperCase(Delimiter), UpperCase(Str)) < 1 then
  begin
    StrBuf.Add(Str);
    Exit;
  end;

  for i := 1 to Length(Str) do
  begin  
    if UpperCase(Copy(Str, i, 1)) = UpperCase(Delimiter) then
    begin
      if UpperCase(Copy(Str, i + 1, 1)) = UpperCase(Delimiter) then
      begin
        Buffer := Copy(Buffer, 2, Length(Buffer));
        StrBuf.Add(Buffer);
        Buffer := '';
        Continue;                      
      end
      else
      begin    
        if System.Pos(UpperCase(Delimiter), UpperCase(Buffer)) > 0 then
          Buffer := Copy(Buffer, 2, Length(Buffer)-1)
        else
          Buffer := Copy(Buffer, 1, Length(Buffer));
      end;
      
      StrBuf.Add(Buffer);
      Buffer := '';              
    end;
    
    if i = Length(Str) then
    begin
      Buffer := Buffer + Copy(Str, i, 1);      
      Buffer := Copy(Buffer, 2, Length(Buffer)-1);
      StrBuf.Add(Buffer);
    end;
    
    Buffer := Buffer + Copy(Str, i, 1);
  end;
end;

function Extract(Str : String; Delimiter : String; Index : Integer): String;
var
  Buffer : String = '';
  i, DelimPos : Integer;
  Flag : Boolean = False;
begin
  for i := 1 to Length(Str) do
  begin
    if (UpperCase(Copy(Str, i, Length(Delimiter))) = UpperCase(Delimiter)) and not Flag then
    begin
      Flag := True;
      DelimPos := i;
    end;
    
    if (UpperCase(Copy(Str, i, Length(Delimiter))) = UpperCase(Delimiter)) then
    begin
      case Index of
        1: Buffer := Copy(Buffer, 1, Length(Buffer));
        2: Buffer := Copy(Str, DelimPos+Length(Delimiter), Length(Str)-DelimPos+Length(Delimiter));
      end;
      
      Break;
    end;
    
    Buffer := Buffer + Str[i];    
  end;

  Result := Trim(Buffer);
end;

function ExtractEx(Str : String; Delimiter : Char; Index : Integer) : String;
var
  Buffer : String;
  i, DelimPos : Integer;
  Flag : Boolean = False;
begin
  Buffer := '';
  
  for i := 1 to Length(Str) do
  begin
    if (UpperCase(Copy(Str, i, 1)) = UpperCase(Delimiter)) and not Flag then
    begin
      Flag := True;
      DelimPos := i;
    end;
    
    if (UpperCase(Copy(Str, i, 1)) = UpperCase(Delimiter)) and (Index = 1) then
    begin
      Buffer := Copy(Buffer, 1, Length(Buffer));
      Break;
    end;
    
    if (UpperCase(Copy(Str, i, 1)) = UpperCase(Delimiter)) and (Index = 2) then
    begin
      Buffer := Copy(Str, DelimPos+1, Length(Str)-DelimPos+3);
      Break;
    end;
    
    Buffer := Buffer + Copy(Str, i, 1);
  end;
  
  Result := Buffer;
end;

(*
function ExtractEx(Str : String; Delimiter : String; Index : Integer): String;
var
  Buffer : String = '';
  i, DelimPos : Integer;
  Flag : Boolean = False;
begin
  for i := 1 to Length(Str) do
  begin
    if (Copy(Str, i, Length(Delimiter)) = Delimiter) and not Flag then
    begin
      Flag := True;
      DelimPos := i;
    end;
    
    if (Copy(Str, i, Length(Delimiter)) = Delimiter) then
    begin
      case Index of
        1: Buffer := Copy(Buffer, 1, Length(Buffer));
        2: Buffer := Copy(Str, DelimPos+Length(Delimiter), Length(Str)-DelimPos+Length(Delimiter));
      end;
      
      Break;
    end;
    
    Buffer := Buffer + Str[i];    
  end;

  Result := Trim(Buffer);
end;
*)

function AsmStrNum(Str : String) : String;
begin
  if ((Str[1] > Chr(47)) and (Str[1] < Chr(58))) or (Str[1] = '$') then
    Str := '#' + Str;

  Result := Str;
end;

function Replace(Str : String; Ch1, Ch2 : Char) : String;
var
  i : Integer;
begin
  i := System.Pos(Ch1, Str);
  Delete(Str, i, 1);
  Insert(Ch2, Str, i);
  Result := Str;
//  Result := StuffString(Str, System.Pos(Ch1, Str), 1, Ch2);
end;

function Strip(Str : String; Ch : Char): String;
var
  i, j : Integer;
begin
  for j := 1 to Length(Str) do
    for i := 1 to Length(Str) do
      if Copy(Str, i, 1) = Ch then Delete(Str, i, 1);
  
  Result := Str;
end;

{
  Function: StrBetween
  Description:
    Extract string between characters Ch1 and Ch2  
  Parameters:
    Str - Input string
    Ch1 - Starting character of examined string
    Ch2 - Last character of examined string  
  Examples:
    Str := Between('ProcX(int a, byte b);', '(', ')');
    Str := Between('PrintE("test")', '"', '"');
}
function StrBetween(Str : String; Ch1, Ch2 : Char) : String;
var
  n1, n2 : Integer;
begin
  n1 := System.Pos(Ch1, Str);

  if Ch1 = Ch2 then
  begin
    Str := Copy(Str, n1 + 1, Length(Str) - n1);
    n2 := System.Pos(Ch2, Str);
    Result := Copy(Str, 1, Length(Str) - 1 - (Length(Str) - n2));
  end
  else
  begin    
    n2 := System.Pos(Ch2, Str);
    Result := Copy(Str, n1 + 1, Length(Str) - (n1 + 1) - (Length(Str) - n2));
  end;
end;

{
  Function: StrBetweenEx
  Description:
    Extract string between characters Ch1 and Ch2  
  Parameters:
    Str - Input string
    Ch1 - Starting character of examined string
    Ch2 - Last character of examined string  
  Examples:
    Str := Between('ProcX(int a, byte b);', '(', ')');
    Str := Between('PrintE("test")', '"', '"');
}
function StrBetweenEx(Str : String; Ch1, Ch2 : Char) : String;
begin
  result := Copy(Str,
                 System.Pos(Ch1, Str) + 1,
                 RPos(Ch2, Str) - System.Pos(Ch1, Str) - 1);
end;

procedure MathExpr(VarStr, Str1, Str2 : String; Flag, Index : Byte);
var
  ParamInc, n, i : Integer;
  Oper : Char;
  StrX : String;
  boolAdd8 : Boolean = True;
  //boolIncDec : Boolean = False;
  OperMemn, OperIncMemn : String;
begin
  if System.Pos(';', Str2) > 0 then
    Str2 := Extract(Str2, ';', 1);

  if System.Pos('*', Str2) > 0 then Oper := '*'
  else if System.Pos('/', Str2) > 0 then Oper := '/'
  else if System.Pos('+', Str2) > 0 then Oper := '+'
  else if System.Pos('-', Str2) > 0 then Oper := '-'
  else if System.Pos('MOD', UpperCase(Str2)) > 0 then Oper := 'M'
  else if System.Pos('LSH', UpperCase(Str2)) > 0 then Oper := 'L'
  else if System.Pos('RSH', UpperCase(Str2)) > 0 then Oper := 'R'
  else if System.Pos('XOR', UpperCase(Str2)) > 0 then Oper := 'X'
  else if System.Pos('!', Str2) > 0 then Oper := '!'
  else if System.Pos('&', Str2) > 0 then Oper := '&'
  else if System.Pos('%', Str2) > 0 then Oper := '%'
  else Oper := '+';
        
  if Oper = 'M' then
  begin
    n := System.Pos('MOD', UpperCase(Str2));
    Delete(Str2, n + 1, 1);
    Insert(' ', Str2, n + 1);
    Delete(Str2, n + 2, 1);
    Insert(' ', Str2, n + 1);
    Str2 := Strip(Str2, ' ');
  end
  else if Oper = 'L' then
  begin
    n := System.Pos('LSH', UpperCase(Str2));
    Delete(Str2, n + 1, 1);
    Insert(' ', Str2, n + 1);
    Delete(Str2, n + 2, 1);
    Insert(' ', Str2, n + 1);
    Str2 := Strip(Str2, ' ');
  end      
  else if Oper = 'R' then
  begin
    n := System.Pos('RSH', UpperCase(Str2));
    Delete(Str2, n + 1, 1);
    Insert(' ', Str2, n + 1);
    Delete(Str2, n + 2, 1);
    Insert(' ', Str2, n + 1);
    Str2 := Strip(Str2, ' ');
  end
  else if Oper = 'X' then
  begin
    n := System.Pos('XOR', UpperCase(Str2));
    Delete(Str2, n + 1, 1);
    Insert(' ', Str2, n + 1);
    Delete(Str2, n + 2, 1);
    Insert(' ', Str2, n + 1);
    Str2 := Strip(Str2, ' ');
  end;      
  
  Split(UpperCase(Str2), UpperCase(Oper));
  if StrBuf.Count < 1 then
  begin
    writeln('Error: Parameter mismatch!');
    Exit;
  end;

  if StrBuf.Count = 1 then
    ParamInc := 1
  else
    ParamInc := 2;

  if ParamInc = 1 then
  begin
    if Flag = 1 then
    begin
      if System.Pos(VarStr, 'T1T2') > 0 then  // BYTE, CHAR      
        CodeBuf.Add(' mva ' + AsmStrNum(StrBuf[0] + ' ' + Str1))
      else
        CodeBuf.Add(' mwa ' + AsmStrNum(StrBuf[0] + ' ' + Str1));        
    end
    else
    begin
      if (System.Pos(VarStr, 'T5T6') > 0) and (GVar2[Index].Location <> 'T5') then  // BYTE and CHAR ARRAY
      begin
        if GVar2[Index].Location <> 'SET' then
          CodeBuf.Add(' mva ' + AsmStrNum(StrBuf[0]) + ' ' + Str1);
      end
      else if System.Pos(VarStr, 'T7T8') > 0 then  // INT and CARD ARRAY
      begin
        if GVar2[Index].Location <> 'SET' then
        begin
        if Flag = 2 then
          CodeBuf.Add(' mwa ' + AsmStrNum(StrBuf[0]) + ' ' + Str1)
        else
        begin
          StrX := Extract(Str1, '[', 1);
          Str1 := Extract(Str1, '[', 2);
          Str1 := Copy(Str1, 1, Length(Str1) - 1);
          
          CodeBuf.Add('');
          CodeBuf.Add(' ldx #0');
          CodeBuf.Add('for_loop_' + IntToStr(PrgVar.Pointer));
          CodeBuf.Add(' mva _str_buffer_' + IntToStr(PrgVar.Pointer) + ',x ' + StrX + '_array_str_' + Str1 + ',x');
          CodeBuf.Add(' inx');
          CodeBuf.Add(' cpx #' + IntToStr(Length(StrBuf[0])));
          CodeBuf.Add(' jcc for_loop_' + IntToStr(PrgVar.Pointer));
          CodeBuf.Add('');
          
          SData[Cnt] := '_str_buffer_' + IntToStr(PrgVar.Pointer) + ' .byte ' + QuotedStr(StrBuf[0]) + ',$9b';
          Inc(Cnt);
        end;
        end; 
      end;
    end;
  end 
  //    
  else
  //
  begin
    lMath := True;              
    
    // Check to see if any of expression variables is of type CARD
    for n := 1 to GVarCnt do
    begin
      for i := 0 to strbuf.Count - 1 do
      begin
        if LowerCase(GVar[n].VarName) = LowerCase(StrBuf[i]) then
        begin
          if GVar[n].ML_type = 'word' then
          begin
            boolAdd8 := False;
            Break;
          end;
        end;
      end;
    end;

    // Check to see if main expression variable is of type CARD  
    for n := 1 to GVarCnt do
    begin
      if LowerCase(GVar[n].VarName) = LowerCase(Str1) then
      begin
        if GVar[n].ML_type = 'word' then
        begin
          boolAdd8 := False;
          Break;
        end;
      end;
    end;

    if ((Copy(StrBuf[0], 1, 1) > Chr(47)) and (Copy(StrBuf[0], 1, 1) < Chr(58))) then
      if StrToInt(StrBuf[0]) > 255 then boolAdd8 := False;

    if ((Copy(StrBuf[1], 1, 1) > Chr(47)) and (Copy(StrBuf[1], 1, 1) < Chr(58))) then
      if StrToInt(StrBuf[1]) > 255 then boolAdd8 := False;

    (*
    if ((Copy(StrBuf[0], 1, 1) > Chr(47)) and (Copy(StrBuf[0], 1, 1) < Chr(58)))
       and ((Copy(StrBuf[1], 1, 1) > Chr(47)) and (Copy(StrBuf[1], 1, 1) < Chr(58)))
       and (StrToInt(StrBuf[0]) < 256)
       and (StrToInt(StrBuf[1]) < 256)
    *)
    if System.Pos(VarStr, 'T3T4') > 0 then
    begin
      boolAdd8 := False;
    end;
    
    Str1 := LowerCase(Str1);
    StrBuf[0] := LowerCase(StrBuf[0]);
    StrBuf[1] := LowerCase(StrBuf[1]);
    
    //
    // Process the arithmetic operation
    //
        
    case Oper of
      '*': begin             
             if boolAdd8 then
               CodeBuf.Add(' Mul8 ' + AsmStrNum(StrBuf[0]) + ', ' + AsmStrNum(StrBuf[1]))
             else
               CodeBuf.Add(' Mul16 ' + AsmStrNum(StrBuf[0]) + ', ' + AsmStrNum(StrBuf[1]));             
           end;
      '/': begin
             if boolAdd8 then
               CodeBuf.Add(' Div8 ' + AsmStrNum(StrBuf[0]) + ', ' + AsmStrNum(StrBuf[1]))
             else
               CodeBuf.Add(' Div16 ' + AsmStrNum(StrBuf[0]) + ', ' + AsmStrNum(StrBuf[1]));
           end;
      '+': begin
             if boolAdd8 then
             begin
               OperMemn := ' adb ';
               OperIncMemn := ' inc ';
             end
             else
             begin
               OperMemn := ' adw ';
               OperIncMemn := ' inw ';
             end;
               
             // a = a + 1
             // a = 1 + a
             if ((Str1 = StrBuf[0]) or (Str1 = StrBuf[1]))
                and (((StrBuf[0] = '1') or (StrBuf[1] = '1'))) then
             begin
               CodeBuf.Add(OperIncMemn + Str1);
             end
             else
             begin
               // a = b + c
               if (StrBuf[0] <> StrBuf[1]) and (Str1 <> StrBuf[0]) and (Str1 <> StrBuf[1]) then
               begin
                 if ((Copy(StrBuf[0], 1, 1) > Copy(Chr(47), 1, 1)) and (Copy(StrBuf[0],1,1) < Copy(Chr(58),1,1)))
                    or (Copy(StrBuf[0],1,1) = '$') then
                 begin
                   CodeBuf.Add(' mwa ' + AsmStrNum(StrBuf[0]) + ' STORE1');
                   StrBuf[0] := 'STORE1'
                 end;
                               
                 CodeBuf.Add(OperMemn + AsmStrNum(StrBuf[0]) + ' ' + AsmStrNum(StrBuf[1]) + ' ' + Str1);
               end
               else if ((Str1 = StrBuf[0]) or (Str1 = StrBuf[1])) then
               begin
                 if (Str1 = StrBuf[0]) then
                   CodeBuf.Add(OperMemn + AsmStrNum(Str1) + ' ' + AsmStrNum(StrBuf[1]))
                 else
                   CodeBuf.Add(OperMemn + AsmStrNum(Str1) + ' ' + AsmStrNum(StrBuf[0]));
               end;
                 
               if strbuf.Count > 2 then
               begin
                 for i := 2 to strbuf.Count - 1 do
                   CodeBuf.Add(OperMemn + AsmStrNum(Str1) + ' ' + AsmStrNum(StrBuf[i]));
               end;
             end;              
           end;             
      '-': begin
             if boolAdd8 then
             begin
               OperMemn := ' sbb ';
               OperIncMemn := ' dec ';
             end
             else
             begin
               OperMemn := ' sbw ';
               OperIncMemn := ' dew ';
             end;
             
             // a = a - 1
             // a = 1 - a
             if ((Str1 = StrBuf[0]) or (Str1 = StrBuf[1]))
                and (((StrBuf[0] = '1') or (StrBuf[1] = '1'))) then
             begin
               CodeBuf.Add(OperIncMemn + Str1);
             end
             else
             begin
               // a = b - c
               if (StrBuf[0] <> StrBuf[1])
                  and (Str1 <> StrBuf[0])
                  and (Str1 <> StrBuf[1]) then
               begin 
                 if ((Copy(StrBuf[0], 1, 1) > Copy(Chr(47), 1, 1)) and (Copy(StrBuf[0],1,1) < Copy(Chr(58),1,1)))
                    or (Copy(StrBuf[0],1,1) = '$') then
                 begin
                   CodeBuf.Add(' mwa ' + AsmStrNum(StrBuf[0]) + ' STORE1');
                   StrBuf[0] := 'STORE1'
                 end;
                            
                 CodeBuf.Add(OperMemn + AsmStrNum(StrBuf[0]) + ' ' + AsmStrNum(StrBuf[1]) + ' ' + Str1);                               
               end
               else if ((Str1 = StrBuf[0]) or (Str1 = StrBuf[1])) then
               begin
                 if (Str1 = StrBuf[0]) then
                   CodeBuf.Add(OperMemn + AsmStrNum(Str1) + ' ' + AsmStrNum(StrBuf[1]))
                 else
                   CodeBuf.Add(OperMemn + AsmStrNum(Str1) + ' ' + AsmStrNum(StrBuf[0]));
               end;
                 
               if strbuf.Count > 2 then
               begin
                 for i := 2 to strbuf.Count - 1 do
                   CodeBuf.Add(OperMemn + AsmStrNum(Str1) + ' ' + AsmStrNum(StrBuf[i]));
               end;
             end;              
           end;             
      'M': CodeBuf.Add(' Mod8 ' + AsmStrNum(StrBuf[0]) + ', ' + AsmStrNum(StrBuf[1]));
      'L':
      begin        
        CodeBuf.Add(' lda ' + AsmStrNum(StrBuf[0]));
        for n := 1 to StrToInt(StrBuf[1]) do CodeBuf.Add(' asl @');
        CodeBuf.Add(' sta STORE1');
      end;
      'R':
      begin        
        CodeBuf.Add(' lda ' + AsmStrNum(StrBuf[0]));
        for n := 1 to StrToInt(StrBuf[1]) do CodeBuf.Add(' lsr @');
        CodeBuf.Add(' sta STORE1');
      end;
      'X', '!':
      begin        
        CodeBuf.Add(' lda ' + AsmStrNum(StrBuf[0]));
        CodeBuf.Add(' eor ' + AsmStrNum(StrBuf[1]));
        CodeBuf.Add(' sta STORE1');
      end;
      '&':
      begin        
        CodeBuf.Add(' lda ' + AsmStrNum(StrBuf[0]));
        CodeBuf.Add(' and ' + AsmStrNum(StrBuf[1]));
        CodeBuf.Add(' sta STORE1');
      end;
      '%':
      begin        
        CodeBuf.Add(' lda ' + AsmStrNum(StrBuf[0]));
        CodeBuf.Add(' ora ' + AsmStrNum(StrBuf[1]));
        CodeBuf.Add(' sta STORE1');
      end;
    end;
    
    if (Oper <> '+') and (Oper <> '-') then
    begin    
      if Flag = 1 then
      begin
        if System.Pos(VarStr, 'T1T2') > 0 then  // BYTE, CHAR
          CodeBuf.Add(' mva STORE1 ' + Str1)
        else
          CodeBuf.Add(' mwa STORE1 ' + Str1);
      end
      else
      begin
        if System.Pos(VarStr, 'T5T6') > 0 then  // BYTE and CHAR array
          CodeBuf.Add(' mva STORE1 ' + Str1)
        else if System.Pos(VarStr, 'T7T8') > 0 then  // INT and CARD array
          CodeBuf.Add(' mwa STORE1 ' + Str1);
      end;
    end;     
  end;
end;        
    
{ Parse ACTION! source code and generate MADS source code }
procedure GenerateCode;
var
  i, j : LongInt;
  CommentStr, Buffer : String;
begin
  FilenameSrc := ExtractFileName(meditEff_src_filename);
  FilenameSrc := Copy(FilenameSrc, 1, RPos('.',FilenameSrc)-1) + '.';  
  FilenameBin := FilenameSrc + meditMADS_bin_ext;
  FilenameSrc := FilenameSrc + meditMADS_src_ext;
  FilenameOrig := FilenameSrc;
  
  AssignFile(fASM, meditMADS_src_dir + FilenameSrc);
  Rewrite(fASM);
  
  WriteLn(fASM, ' org $' + IntToStr(meditAddr));
  WriteLn(fASM, '');
  WriteLn(fASM, ' icl ' + QuotedStr(meditMADS_rtl_dir + 'equates.asm'));
  WriteLn(fASM, ' icl ' + QuotedStr(meditMADS_rtl_dir + 'common.asm'));
  WriteLn(fASM, '');
  
  Cnt := 1;
  Cnt2 := 1;
  ProcML_cnt := 0;
  MemCnt := Hex2Dec(IntToStr(meditMLAddr));  // default 32768 (dec) $8000 (hex)
  WhileFlag := False;
  ForFlag := False;
  UntilFlag := False;
  lGraphicsFlag := False;
  lGr := False;
  ForCnt := 0;
  lInput := False;
  word_Cnt := 1;
  //lBoolParam := False;
    
  CodeBuf.Clear;

  for i := 1 to CR_LF do
  begin
  (*
    Remarks := Extract(TextBuf[i], ';', 2);
    if Remarks <> TextBuf[i] then CodeBuf.Add('; ' + Remarks);

    TextBuf[i] := Extract(TextBuf[i], ';', 1);
  *)
    Buffer := Strip(TextBuf[i], ' ');
    if Buffer[1] = ';' then Continue;
    //
    //if not ProcCheck(i) then Continue;
    //
//    if System.Pos('PRINT(', Buffer) > 0 then Continue;    
//    if System.Pos('PRINTE(', Buffer) > 0 then Continue;
    
    sc_Define(i);
    sc_Var(i);
    sc_Var2(i);
    sc_Array(i);
  end;

(*
  if BoolType2 then
  begin
    CodeBuf.Add('; Handling TYPE variables');
    CodeBuf.Add(' .var struct_ptr_var .word');
    CodeBuf.Add('');
  end;
*)

  for i := 1 to GVarCnt do
  begin
    // Extract comments from the variable name declaration
    Split(GVar[i].VarName, ';');
    
    if StrBuf.Count <= 1 then
      CommentStr := ''
    else
    begin
      GVar[i].VarName := StrBuf[0];
      CommentStr := '  ; ' + StrBuf[1];
    end;      
      
    if UpperCase(GVar[i].ParentType) = UpperCase(GVar[i].OrigType) then
    begin
      if GVar[i].Value <> '' then
        CodeBuf.Add(GVar[i].VarName + ' equ ' + GVar[i].Value + CommentStr)
      else
      begin
        if GVar[i].InitValue = -1 then
          CodeBuf.Add(' .var ' + GVar[i].VarName + ' .' + GVar[i].ML_type + CommentStr)
        else
          CodeBuf.Add(' .var ' + GVar[i].VarName + '=' +
                      IntToStr(GVar[i].InitValue) + ' .' + GVar[i].ML_type + CommentStr);
      end;
    end
    else
    begin
      if System.Pos(GVar[i].ParentType, 'T10') > 0 then  // POINTER
      //else if Trim(UpperCase(GVar[i].ParentType)) = 'POINTER' then
      begin
        if LowerCase(GVar[i].ML_type) = 'word' then
          CodeBuf.Add(' .var ' + GVar[i].VarName + ' .' + GVar[i].ML_type + CommentStr)
        else
          CodeBuf.Add(' .var ' + GVar[i].VarName + ' .byte' + CommentStr);
      end;      
    end;
  end;

  for i := 1 to GVarCnt2 do
  begin
    // Extract comments from the variable name declaration
    Split(GVar2[i].VarName, ';');
    
    if StrBuf.Count > 1 then
    begin
      GVar2[i].VarName := StrBuf[0];      
      CommentStr := '  ; ' + StrBuf[1]
    end
    else
      CommentStr := '';
    
    if System.Pos(GVar2[i].VarType, 'T5T6') > 0 then  // BYTE and CHAR ARRAY
    //if (UpperCase(GVar2[i].VarType) = 'BYTE ARRAY') or (UpperCase(GVar2[i].VarType) = 'CHAR ARRAY') then
    begin
      // BYTE ARRAY str1="Text"
      if GVar2[i].Location = 'T5' then
      begin
        CodeBuf.Add('.array ' + GVar2[i].VarName + ' ' + IntToStr(GVar2[i].Dim) + ' .byte = $ff');
        CodeBuf.Add(' [0] = ' + QuotedStr(GVar2[i].Value));
      end
      
      else if GVar2[i].Value <> '' then
      begin
        GVar2[i].Value := Copy(GVar2[i].Value, 1, Length(GVar2[i].Value) - 2);
        
        if GVar2[i].Dim = 0 then
          CodeBuf.Add('.array ' + GVar2[i].VarName + ' .byte = $ff' + CommentStr)
        else
          CodeBuf.Add('.array ' + GVar2[i].VarName + ' ' + IntToStr(GVar2[i].Dim) + ' .byte = $ff' + CommentStr);
        
        CodeBuf.Add(' ' + GVar2[i].Value);
      end
      
      else
        CodeBuf.Add('.array ' + GVar2[i].VarName + ' ' + IntToStr(GVar2[i].Dim) + ' .byte = $ff' +
                    CommentStr);
      
      CodeBuf.Add('.enda');
      CodeBuf.Add(' .var array_buffer_' + GVar2[i].VarName + ' .word');
      CodeBuf.Add(' .var array_index_' + GVar2[i].VarName + ' .byte');
    end
    else if System.Pos(GVar2[i].VarType, 'T7T8') > 0 then  // INT and CARD ARRAY
    //else if (UpperCase(GVar2[i].VarType) = 'INT ARRAY') or (UpperCase(GVar2[i].VarType) = 'CARD ARRAY') then
    begin
      // CARD ARRAY values=[1 2 3 4 5 6 7]
      if GVar2[i].Value = '' then
        CodeBuf.Add(' .array ' + GVar2[i].VarName + ' ' + IntToStr(GVar2[i].Dim) + ' .word = $ff' +
                    CommentStr)
      else
      begin
        GVar2[i].Value := Copy(GVar2[i].Value, 1, Length(GVar2[i].Value) - 2);
        
        if GVar2[i].Dim = 0 then
          CodeBuf.Add('.array ' + GVar2[i].VarName + ' .word = $ff' + CommentStr)
        else
          CodeBuf.Add('.array ' + GVar2[i].VarName + ' ' + IntToStr(GVar2[i].Dim) + ' .word = $ff' +
                      CommentStr);
        
        CodeBuf.Add(' ' + GVar2[i].Value);
      end;
                
      CodeBuf.Add('.enda');
      
      if GVar2[i].Location <> 'SET' then
      begin
        for j := 0 to GVar2[i].Dim do
        begin
          CodeBuf.Add('.array ' + GVar2[i].VarName + '_array_str_' + IntToStr(j) + ' 40 .byte = $ff');
          CodeBuf.Add('.enda');
        end;
      end;
           
      CodeBuf.Add(' .var array_buffer_' + GVar2[i].VarName + ' .word');
      CodeBuf.Add(' .var array_index_' + GVar2[i].VarName + ' .byte');
    end
    else if System.Pos(GVar2[i].VarType, 'T9') > 0 then  // TYPE
    begin
      CodeBuf.Add(' .struct ' + GVar2[i].VarName + CommentStr);
      
      for j := 1 to 255 do
        if UpperCase(GVar[j].ParentType) = UpperCase(GVar2[i].VarName) then
          CodeBuf.Add(' ' + GVar[j].VarName + ' .' + GVar[j].ML_type);
      
      CodeBuf.Add(' .ends')
    end;
  end;

  for i := 0 to CodeBuf.Count - 1 do
  begin
    WriteLn(fASM, CodeBuf[i]);
  end;

  WriteLn(fASM, '');

  Icl := 0;
  lInclude := False;
  
  for i := 1 to CR_LF do sc_Include(i);

  ReadInclude; { read INCLUDE source file }

  lInclude := False;

  if lIncludeX and not ReadOrig then Exit;
//  if lIncludeX then
//    if not ReadOrig then Exit;    
  
  CodeBuf.Clear;
  GenLoop(lInclude);
  
  for i := 0 to CodeBuf.Count - 1 do
    WriteLn(fASM, CodeBuf[i]);

  DeviceCheck;
  
  WriteLn(fASM, ' jmp *');
  WriteLn(fASM, '');
  //WriteLn(fASM, ' org $' + IntToStr(MemCnt));
  sc_ML_data;
  WriteLn(fASM, '');
  WriteLn(fASM, '.link ' + AnsiQuotedStr(meditMADS_rtl_dir + 'runtime.obx', ''''));
  sc_Lib(meditMADS_rtl_dir);
  WriteLn(fASM, '');

  sc_Data;

  WriteLn(fASM, '');

  if lIncludeX and not lInclude then
    WriteLn(fASM, ' run ' + Copy(ProcBuf[ProcCount2-1], 5, Length(ProcBuf[ProcCount2-1]) - 4))
  else
    WriteLn(fASM, ' run ' + Copy(ProcBuf[ProcCount], 5, Length(ProcBuf[ProcCount]) - 4));

  CloseFile(fASM);
end;

end.
