{ 
  Program    : Effectus - Atari MADS cross-assembler/parser for Action! language

  Unit file  : routines.pas
  Description: Library of build-in Action! procedures and routines
  
  Author: Bostjan Gorisek, Slovenia
  
  Program compiled with Free Pascal 2.6.4
  Reference: http://www.freepascal.org/
  
  This is cource code generator for MADS Assembler, which then generates
  executable files from parser of Action! language for 8-bit Atari home computers.
  
  References:
  http://mads.atari8.info/
  http://gury.atari8.info/effectus/ 
  
  This is open source and freeware project!  
}
unit Routines;
{$ifdef FPC}{$mode objfpc}{$h+}{$endif}

interface

uses
  SySutils, Classes, StrUtils, INIfiles,
  Decl, Core, Common;

procedure sc_Print(boolCR: Boolean); 
procedure sc_Command(CmdName, Flag : String);
procedure sc_PutE;
procedure sc_PrintX(Proc : String; lEnter : Boolean);
procedure sc_SndRst;
procedure sc_Func1(Proc, Flag : String);
procedure sc_Func2(Proc, Flag : String);
procedure sc_InputS;
procedure sc_SCopy;
procedure sc_SCopyS;
procedure sc_SAssign;
procedure sc_XIO;
procedure sc_Open;
procedure sc_Close;
procedure sc_PrintDX(boolCR : Boolean);
procedure sc_PutDX(Proc: String);
procedure sc_PrintXD(Proc : String; lEnter : Boolean);
procedure sc_InputSMD(Proc, Max, Flags : String);
procedure sc_Point;
procedure sc_Note;
procedure sc_GetD;
procedure sc_Locate;
procedure sc_StrNum(Proc : String);
function GetParams(StrBufX : String; lNum, lConv : Boolean; ParamTypes : String) : Integer;
procedure sc_PrintF;
procedure sc_Input(Proc : String);

implementation

{
  Procedure name : sc_Print
  Description    : Handles Action! Print procedure statement
  Parameters     : boolCR - New line
}
procedure sc_Print(boolCR : Boolean);
var
  n, n1 : LongInt;
  Text_ASM, TextX, Proc : String;
begin
  TextX := Strip(TextBuf[CurLine], ' ');
  
  if boolCR then
    Proc := 'PRINTE'
  else
    Proc := 'PRINT';
  
  if (System.Pos(UpperCase(Proc + '("'), UpperCase(TextX)) > 0) then
  begin
    if System.Pos(UpperCase(Proc) + '("")', UpperCase(TextBuf[CurLine])) > 0 then
    begin
      if boolCr then
        CodeBuf.Add(' PutE');
    end else begin
      Text_ASM := ExtractText(TextBuf[CurLine], '"', '"');
      CodeBuf.Add(' jsr printf');
      
      if boolCR then
        CodeBuf.Add(' dta c' + QuotedStr(Text_ASM) + ',$9b,0')
      else
        CodeBuf.Add(' dta c' + QuotedStr(Text_ASM) + ',0');
    end;
  
  // String variable as parameter
  //
  end else if (System.Pos(UpperCase(Proc + '('), UpperCase(TextX)) > 0) then
  begin    
    CodeBuf.Add(' jsr printf');
    Text_ASM := ExtractText(TextBuf[CurLine], '(', ')');
    
    if cnt_InputS > 0 then
    begin
      n := 1;

      for n1 := 1 to cnt_InputS do
      begin
        if System.Pos(Text_ASM + '_', InputS_buf[n1]) < 1 then
          Continue
        else
          n := n1;

        if InputS_buf[n1] = '' then Break;
      end;
      
      //if (n = 1) and (InputS_buf[n] = '') then
      //begin
      //  Exit;
      //end;

      if (n <> 1) and (InputS_buf[n] <> '') then
      begin
        if boolCR then
          CodeBuf.Add(' dta ' + AnsiQuotedStr('#', '''') + ',$9b,0')
        else
          CodeBuf.Add(' dta ' + AnsiQuotedStr('#', '''') + ',0');
  
        CodeBuf.Add(' dta a(' + InputS_buf[n] + _EFF + ')');
      end;

    end else begin
      if boolCR then
        CodeBuf.Add(' dta c' + AnsiQuotedStr('#', '''') + ',$9b,0')
      else
        CodeBuf.Add(' dta c' + AnsiQuotedStr('#', '''') + ',0');

      // Handle ARRAY variables
      for n1 := 1 to GVarCnt2 do
      begin        
        if System.Pos(UpperCase(GVar2[n1].VarName) + '(', UpperCase(Text_ASM)) > 0 then
        begin
          TextX := ExtractText(Text_ASM, '(', ')');
          Text_ASM := GVar2[n1].VarName + _EFF + 'array_str_' + TextX;
          Break;
        end;
      end;

      if System.Pos(_EFF, LowerCase(Text_ASM)) > 0 then
        CodeBuf.Add(' dta a(' + Text_ASM + ')')
      else
        CodeBuf.Add(' dta a(' + Text_ASM + _EFF + ')');
    end;
  end;
end;

{
  Procedure name : sc_PutE
  Description    : Handles Action! PutE procedure statement
}
procedure sc_PutE;
var
  Buffer : String;
begin
  Buffer := Trim(TextBuf[CurLine]);

  if (System.Pos('PUTE()', UpperCase(Buffer)) > 0) then  
    CodeBuf.Add(' PutE');
end;

{
  Procedure name : sc_Command
  Description    : General code for handling many Action! procedure statements
  Parameters     : CmdName
                   Flag
}
procedure sc_Command(CmdName, Flag : String);
var
  n, n1 : LongInt;
  Proc, mParams, Buffer : String;
  Tmp : String = '';
begin
  Buffer := Strip(TextBuf[CurLine], ' ');
  Proc := CmdName;  

  if System.Pos(UpperCase('=' + Proc), UpperCase(Buffer)) > 0 then
  begin
    Tmp := Extract(Buffer, '=', 1);
  end;

  if Copy(Flag, 1, 1) = '3' then
  begin
    CmdName := '=';
  end else begin
    CmdName := '(';
  end;

  if System.Pos(UpperCase(Proc + CmdName), UpperCase(Buffer)) > 0 then
  begin    
    Buffer := TextBuf[CurLine];
    n1 := GetParams(Buffer, True, False, Flag);
    mParams := mValues[0];
    
    for n := 1 to n1 do
      mParams := mParams + ', ' + mValues[n];

    CodeBuf.Add(' ' + Proc + ' ' + mParams);
    
    if Tmp <> '' then
      CodeBuf.Add(' mva STORE1 ' + Tmp + _EFF);
  end;
end;

{
  Procedure name : sc_PrintX
  Description    : General code for handling several Action! Print procedure statements
  Parameters     : Proc - Print routine name
                   lEnter - Atari carriage return flag
}
procedure sc_PrintX(Proc : String; lEnter : Boolean);
var
  n, n1 : LongInt;
  mValue, mValue2, mValue3, mLabel, VarIndex, Buffer, mTemp : String;

procedure dta(str : String);
var
  r : Byte;
begin
  if lEnter then
    CodeBuf.Add(' dta c' + QuotedStr('%') + ',$9b,0')
  else
    CodeBuf.Add(' dta c' + QuotedStr('%') + ',0');

  if IsNumber(mValue3[1]) then
  begin
    CodeBuf.Add(' dta a(' + mLabel + ')');
    SData[Cnt] := mLabel + ' dta a(' + mValue3 + ')';
    Inc(Cnt);
  end else begin
    if str = 'eff_byte' then begin
      CodeBuf.Add(' dta a($c0)')
    end else begin    
      // Processing the type variable
      //      
      if System.Pos('.', str) > 0 then
      begin
        mValue := Extract(str, '.', 1);
        mValue2 := Extract(str, '.', 2);

        for r := 1 to GVarCnt do
        begin
          if (GVar[r].VarName = mValue) and (GVar[r].ParentType = 'T9') then
          begin
            CodeBuf.Add(' dta a(' + mValue + _EFF + '.' + mValue2 + _EFF + ')');
            Break;
          end; 
        end;

      // Other processing
      end else begin
        CodeBuf.Add(' dta a(' + AsmStrNum(str) + ')')
      end;
    end;
  end;
end;

begin
  Buffer := Strip(TextBuf[CurLine], ' ');

  n1 := System.Pos(UpperCase(Proc + '('), UpperCase(Buffer));
  if (n1 > 0) then
  begin
    mLabel := '_' + Proc + '_' + IntToStr(CurLine) + '_' + IntToStr(n1);
    mValue := ExtractText(Buffer, '(', ')');
    mValue3 := mValue;

    for n := 1 to GVarCnt2 do
    begin
      if Pos(UpperCase(GVar2[n].VarName), UpperCase(mValue)) > 0 then
      begin
        mValue2 := Extract(mValue, '(', 1);
        VarIndex := ExtractText(mValue, '(', ')');
        mValue := Replace(mValue, '(', '[');
        mValue := Replace(mValue, ')', ']');

        if boolArray and (((sFor in flags)(*ForFlag*) and (ForVar = VarIndex))
           or (sWhile in flags) (*WhileFlag*) or (sUntil in flags))  // UntilFlag
           and (UpperCase(GVar2[n].VarName) = UpperCase(mValue2)) then
        begin
          if System.Pos(GVar2[n].VarType, 'T7T8') > 0 then
          begin
            CodeBuf.Add(' ldy array_index_' + GVar2[n].VarName + _EFF);
            CodeBuf.Add(' lda ' + GVar2[n].VarName + _EFF + ',y');
            CodeBuf.Add(' sta array_buffer_' + GVar2[n].VarName + _EFF);
            CodeBuf.Add(' inc array_index_' + GVar2[n].VarName + _EFF);
            CodeBuf.Add(' ldy array_index_' + GVar2[n].VarName + _EFF);
            CodeBuf.Add(' lda ' + GVar2[n].VarName + _EFF + ',y');
            CodeBuf.Add(' sta array_buffer_' + GVar2[n].VarName + _EFF + '+1');
          end else begin
            CodeBuf.Add(' ldy array_index_' + GVar2[n].VarName + _EFF);
            CodeBuf.Add(' lda ' + GVar2[n].VarName + _EFF + ',y');
            CodeBuf.Add(' sta array_buffer_' + GVar2[n].VarName + _EFF);
          end;
        end;

        if boolArray and (((sFor in flags)(*ForFlag*) and (ForVar = VarIndex))
           or (sWhile in flags)(*WhileFlag*) or (sUntil in flags))  //UntilFlag
           and (UpperCase(GVar2[n].VarName) = UpperCase(mValue2)) then
        begin
          CodeBuf.Add(' jsr printf');

          if lEnter then
            CodeBuf.Add(' dta c''%''' + ',$9b,0')
          else
            CodeBuf.Add(' dta c''%''' + ',0');

          CodeBuf.Add(' dta a(array_buffer_' + GVar2[n].VarName  + _EFF + ')');

        // POINTER OR ARRAY
        //
        end else begin        
          if System.Pos(GVar2[n].VarType, 'T10') > 0 then
          begin
            if System.Pos(UpperCase(GVar2[n].VarName) + '.', UpperCase(mValue)) > 0 then
            begin
              VarIndex := Extract(mValue3, '.', 2);
              VarIndex := Copy(VarIndex, 1, Length(VarIndex));

              for n1 := 1 to GVarCnt do
              begin
                if UpperCase(GVar[n1].VarName) = UpperCase(VarIndex) then
                begin
                  CodeBuf.Add(mvwa(GVar[n1].VarType) + PtrData + _EFF + '[' + IntToStr(RecPtrVar.Dim) + '].' + VarIndex + _EFF + ' struct_ptr_var');
                  Break;
                end;
              end;

              CodeBuf.Add(' jsr printf');

              if lEnter then
                CodeBuf.Add(' dta c''%''' + ',$9b,0')
              else
                CodeBuf.Add(' dta c''%''' + ',0');

              CodeBuf.Add(' dta a(struct_ptr_var)'); 
            
            end else begin
              CodeBuf.Add(' jsr printf');

              if lEnter then
                CodeBuf.Add(' dta c''%''' + ',$9b,0')
              else
                CodeBuf.Add(' dta c''%''' + ',0');

              for n1 := 1 to GVarCnt do
              begin
                if (System.Pos(GVar[n1].ParentType, 'T10') > 0)
                   and (UpperCase(GVar[n1].OrigType) = UpperCase(GVar2[n].VarName)) then
                begin
                  CodeBuf.Add(' dta a(' + GVar[n1].VarName + _EFF + ')');
                  Break;
                end;
              end;   
            end;
            
          end else begin
            //mTemp := Copy(Extract(mValue, '[', 2), 1, 1);
            mTemp := AsmStrNum(ExtractText(mValue, '[', ']'));
            mTemp := StringReplace(mTemp, '#', '', [rfReplaceAll]);
            //writeln('mvalue2=' + mvalue2 + ' mTemp=' + mtemp);

            if System.Pos(GVar2[n].VarType, 'T5T6') > 0 then
            begin
              CodeBuf.Add(' mva ' + mvalue2 + _EFF + '[' + mTemp + ']' + ' $c0');
              mValue2 := 'eff_byte';    
            end else if System.Pos(GVar2[n].VarType, 'T7T8') > 0 then begin
              mValue2 := mvalue2 + '[' + mTemp + ']';                     
            end;

            CodeBuf.Add(' jsr printf');
            dta(mvalue2);                
    
            //dta(mvalue2 + '[' + mTemp + ']');
          end;
        end;

        exit;
      end;
    end;

    mValue2 := ExtractText(Buffer, '(', ')');

    if (LowerCase(Proc) = 'printbe') or (LowerCase(Proc) = 'printb') then
    begin
      for n := 1 to GVarCnt do
      begin
        if System.Pos(UpperCase(GVar[n].VarName), UpperCase(mValue2)) > 0 then
        begin
          // Processing the type variable
          if System.Pos('.', mValue2) > 0 then
          begin
            mValue3 := Extract(mValue2, '.', 1);
            VarIndex := Extract(mValue2, '.', 2);
            mValue2 := mValue3 + _EFF + '.' + VarIndex + _EFF;

          // Other processing
          end else begin
            mValue2 := AsmStrNum(mValue2);
          end;

          CodeBuf.Add(' mva ' + mValue2 + ' $c0');
          mValue2 := 'eff_byte';
          Break;
        end;
      end;
    end;

    CodeBuf.Add(' jsr printf');
    dta(mValue2);
  end;
end;

{
  Procedure name : sc_SndRst
  Description    : Code for handling Action! SndRst procedure statement
  Parameters     : None
}
procedure sc_SndRst;
var
  Proc, Buffer : String;
begin
  Buffer := Strip(TextBuf[CurLine], ' ');
  Proc := 'SndRst';

  if (System.Pos(UpperCase(Proc + '('), UpperCase(Buffer)) > 0) then
    CodeBuf.Add(' ' + Proc);
end;

{
  Procedure name : sc_Func1
  Description    :
  Parameters     : Proc - Function name
                   Flag - Parameter type
}
procedure sc_Func1(Proc, Flag : String);
var
  n1, r : Integer;
  Str1, Str2, Buffer, FuncPrm : String;
  arrOper : TStringList;
begin
  Buffer := Strip(TextBuf[CurLine], ' ');

  n1 := System.Pos(UpperCase(Proc + '('), UpperCase(Buffer));
  if (n1 > 0) then
  begin
    arrOper := TStringList.Create;

    try
      Buffer := Trim(TextBuf[CurLine]);
      GetParams(Buffer, False, False, Flag);
      Split(Buffer, '=', []);
      Str1 := StrBuf[0];
      Str2 := StrBuf[1];
      FuncPrm := ExtractText(Str2, '(', ')');

      // Special case - Direct memory address pointer in variable assignment expression
      //
      if System.Pos('+', FuncPrm) > 0 then
      begin
        Str2 := Extract(FuncPrm, '+', 1);
        FuncPrm := Extract(FuncPrm, '+', 2);
        CodeBuf.Add(' ' + Proc + ' ' + Str2 + _EFF + '+' + FuncPrm);
        CodeBuf.Add(' mwa STORE1 ' + Str1 + _EFF);

        if Assigned(arrOper) then FreeAndNil(arrOper);

        Exit;
      end;

      (*
      for n1 := 1 to GVarCnt do
      begin
        if System.Pos(UpperCase(GVar[n1].VarName), UpperCase(FuncPrm)) > 0 then
        begin
          for r := 0 to Length(FuncPrm) - 1 do
          begin
            if FuncPrm[r] = '+' then
              arrOper.Add('1')
            else if FuncPrm[r] = '-' then
              arrOper.Add('2')
            else if FuncPrm[r] = '*' then
              arrOper.Add('3')
            else if Str2[r] = '/' then
              arrOper.Add('4')
          end;
          
          if arrOper.Count = 0 then
          begin
            CodeBuf.Add(mvwa(GVar[n1].VarType) + AsmStrNum(FuncPrm) + ' STORE1' + _EFF);
            Break;
          end;
          
          SplitEx2(FuncPrm, '+', '-', '*', '/');
  
          for r := 0 to arrOper.Count - 1 do
          begin
            if r = 0 then
            begin
              if arrOper[r] = '1' then
                Str3 := strbuf2[0] + '+' + strbuf2[1]
              else if arrOper[r] = '2' then
                Str3 := strbuf2[0] + '-' + strbuf2[1]
              else if arrOper[r] = '3' then
                Str3 := strbuf2[0] + '*' + strbuf2[1]
              else if arrOper[r] = '4' then
                Str3 := strbuf2[0] + '/' + strbuf2[1]
            end else begin
              if arrOper[r] = '1' then
                Str3 := FuncPrm + '+' + strbuf2[r + 1]
              else if arrOper[r] = '2' then
                Str3 := FuncPrm + '-' + strbuf2[r + 1]
              else if arrOper[r] = '3' then
                Str3 := FuncPrm + '*' + strbuf2[r + 1]
              else if arrOper[r] = '4' then
                Str3 := FuncPrm + '/' + strbuf2[r + 1]
            end;

            MathExpr(GVar[n1].VarType, FuncPrm, Str3, 1, 0);
          end;
  
          Break;
        end;
      end;

      arrOper.Clear;
      *)
  
      for n1 := 1 to GVarCnt do
      begin
        if System.Pos(UpperCase(GVar[n1].VarName) + '=', UpperCase(Str1 + '=' + Str2)) > 0 then
        begin
          if Copy(Flag, 1, 1) = '0' then
            CodeBuf.Add(' ' + Proc + ' ' + mValues[0])
          else
            CodeBuf.Add(' ' + Proc + ' ' + AsmStrNum(mValues[0]));

          for r := 0 to Length(Str2) - 1 do
          begin
            if Str2[r] = '+' then
              arrOper.Add('1')
            else if Str2[r] = '-' then
              arrOper.Add('2')
            else if Str2[r] = '*' then
              arrOper.Add('3')
            else if Str2[r] = '/' then
              arrOper.Add('4')
          end;

          // Generate expression output
          //
          if arrOper.Count = 0 then
          begin
            CodeBuf.Add(mvwa(GVar[n1].VarType) + 'STORE1 ' + Str1 + _EFF);
            Break;
          end;

          SplitEx2(Str2, '+', '-', '*', '/');

          for r := 0 to arrOper.Count - 1 do
          begin
            if r = 0 then
            begin
              if arrOper[r] = '1' then
                Str2 := strbuf2[0] + '+' + strbuf2[1]
              else if arrOper[r] = '2' then
                Str2 := strbuf2[0] + '-' + strbuf2[1]
              else if arrOper[r] = '3' then
                Str2 := strbuf2[0] + '*' + strbuf2[1]
              else if arrOper[r] = '4' then
                Str2 := strbuf2[0] + '/' + strbuf2[1]
            end else begin
              if arrOper[r] = '1' then
                Str2 := Str1 + '+' + strbuf2[r + 1]
              else if arrOper[r] = '2' then
                Str2 := Str1 + '-' + strbuf2[r + 1]
              else if arrOper[r] = '3' then
                Str2 := Str1 + '*' + strbuf2[r + 1]
              else if arrOper[r] = '4' then
                Str2 := Str1 + '/' + strbuf2[r + 1]
            end;

            MathExpr(GVar[n1].VarType, Str1, Str2, 1, 0);
          end;

          Break;
        end;
      end;

    finally
      if Assigned(arrOper) then FreeAndNil(arrOper);
    end;
  end;
end;

{
  Procedure name : sc_Func2
  Description    :
  Parameters     : Proc - Function name
                   Flag - Parameter type
}
procedure sc_Func2(Proc, Flag : String);
var
  n1, n2 : LongInt;
  mLabel, Str1, Buffer : String;
begin
  Buffer := Strip(TextBuf[CurLine], ' ');

  n1 := System.Pos(UpperCase(Proc + '('), UpperCase(Buffer));
  if (n1 > 0) then
  begin
    GetParams(Buffer, True, True, Flag);
    Split(Buffer, '=', []);
    Str1 := StrBuf[0];

    for n2 := 1 to GVarCnt do
    begin
      if System.Pos(UpperCase(GVar[n2].VarName) + '=', UpperCase(Buffer)) > 0 then
      begin
        mLabel := '_' + Proc + '_' + IntToStr(CurLine) + '_' + IntToStr(n1);
        SData[Cnt] := mLabel + ' dta a(' + mValues[0] + ')';
        Inc(Cnt);

        if UpperCase(Proc) = 'VALB' then
          CodeBuf.Add(' mva ' + mLabel + ' STORE1')
        else
          CodeBuf.Add(' mwa ' + mLabel + ' STORE1');

        CodeBuf.Add(mvwa(GVar[n2].VarType) + 'STORE1 ' + Str1 + _EFF);
        Break;
      end;
    end;
  end;
end;

{
  Procedure name : sc_Input
  Description    : Handles Action! input of numeric values with functions
                   InputB, InputI and InputC
  Parameters     : Proc - Function name
}
procedure sc_Input(Proc : String);
var
  Buffer : String;
begin
  Buffer := Strip(TextBuf[CurLine], ' ');

  if System.Pos('=' + UpperCase(Proc) + '()', UpperCase(Buffer)) > 0 then
  begin
    Buffer := Extract(Buffer, '=', 1);
    CodeBuf.Add(' getline #hex_num');
    CodeBuf.Add(' ldx #0');
    CodeBuf.Add(' lda hex_num,x');
    CodeBuf.Add(' sbc #48');
    CodeBuf.Add(' sta ' + Buffer + _EFF);
    CodeBuf.Add(' lda #10');  // Currently, only numbers 0 to 9 are shown
    CodeBuf.Add(' cmp ' + Buffer + _EFF);
    CodeBuf.Add(' bcc jfv' + IntToStr(CurLine));
    CodeBuf.Add(' jmp skip' + IntToStr(CurLine));
    CodeBuf.Add('jfv' + IntToStr(CurLine) + ' mva #0 ' + Buffer + _EFF);
    CodeBuf.Add('skip' + IntToStr(CurLine));
  end;
end;

{ *****************
   String routines
  ***************** }

{
  Procedure name : sc_InputS
  Description    : Handles Action! InputS procedure statement
  Parameters     : None
}
procedure sc_InputS;
var
  Buffer : String;
begin
  Buffer := Strip(TextBuf[CurLine], ' ');

  if System.Pos('INPUTS(', UpperCase(Buffer)) > 0 then
  begin
    Buffer := ExtractText(TextBuf[CurLine], '(', ')');
    CodeBuf.Add(' getline #' + Trim(Buffer) + _EFF);
  end;
end;

{
  Procedure name : sc_SCopy
  Description    : Handles Action! SCopy procedure statement
  Parameters     : None

  PROC SCopy(targetString, sourceString)
  
  SCopy(str1,"String1")
  SCopy(str1,str2)
}
procedure sc_SCopy;
var
  n : LongInt;
  Buffer, Str1, Str2 : String;
begin
  Buffer := Strip(TextBuf[CurLine], ' ');

  if System.Pos('SCOPY(', UpperCase(Buffer)) > 0 then
  begin
    Str1 := ExtractText(TextBuf[CurLine], '(', ')');
    Split(TextBuf[CurLine], ',', []);

    if StrBuf.Count <> 2 then Exit;

    Str1 := Extract(StrBuf[0], '(', 2);

    if System.Pos('"', StrBuf[1]) > 0 then
    begin
      Str2 := ExtractText(StrBuf[1], '"', '"');
      SData[Cnt] := '_SCOPY_buffer_' + IntToStr(CurLine) + ' .byte ' + QuotedStr(Str2) + ', $9b';
      Inc(Cnt);
    end else begin
      Str2 := Copy(StrBuf[1], 1, Length(StrBuf[1]) - 1);
    end;

    CodeBuf.Add(' ldx #0');
    CodeBuf.Add('for_loop_' + IntToStr(CurLine));

    if System.Pos('"', StrBuf[1]) > 0 then
      CodeBuf.Add(' mva ' + '_SCOPY_buffer_' + IntToStr(CurLine) + ',x ' + str1 + _EFF + ',x')
    else
      CodeBuf.Add(' mva ' + str2 + _EFF + ',x ' + str1 + _EFF + ',x');

    CodeBuf.Add(' inx');

    if System.Pos('"', StrBuf[1]) > 0 then
    begin
      CodeBuf.Add(' cpx #' + IntToStr(Length(Str2)))
    end else begin
      for n := 1 to GVarCnt2 do
      begin
        if UpperCase(GVar2[n].VarName) = UpperCase(Str2) then
        begin
          CodeBuf.Add(' cpx #' + IntToStr(GVar2[n].Dim));
          Break;
        end;
      end;
    end;

    CodeBuf.Add(' jcc for_loop_' + IntToStr(CurLine));
  end;  
end;

{
  Procedure name : sc_SCopyS
  Description    : Handles Action! SCopyS procedure statement
  Parameters     : None

  PROC SCopyS(targetString, sourceString, BYTE start, stop)

  Examples:
    SCopyS(str1, "LATARIAN", 2, 6)
    SCopyS(str2, str1, 2, 4)
}
procedure sc_SCopyS;
var
  Buffer : String;
begin
  Buffer := Strip(TextBuf[CurLine], ' ');
  
  if System.Pos('SCOPYS(', UpperCase(Buffer)) > 0 then
  begin
    if GetParams(TextBuf[CurLine], True, False, '1111') <> 3 then Exit;

    if System.Pos('#', mValues[1]) > 0 then
      mValues[1] := Extract(mValues[1], '#', 2);

    if System.Pos('#', mValues[3]) < 1 then
    begin
      CodeBuf.Add(' mva ' + mValues[3] + ' b_param1');
      CodeBuf.Add(' dec b_param1');
    end;

    CodeBuf.Add(' ldy ' + mValues[2]);
    CodeBuf.Add(' dey');
    CodeBuf.Add(' ldx #0');
    CodeBuf.Add('for_loop_' + IntToStr(CurLine));

    if System.Pos(_EFF, mValues[1]) > 0 then  
      CodeBuf.Add(' lda ' + mValues[1] + ',y')
    else
      CodeBuf.Add(' lda ' + mValues[1] + _EFF + ',y');

    CodeBuf.Add(' sta ' + mValues[0] + ',x');
    CodeBuf.Add(' iny');
    CodeBuf.Add(' inx');

    if System.Pos('#', mValues[3]) < 1 then
    begin
      CodeBuf.Add(' cpx b_param1')
    end else begin
      mValues[3] := Extract(mValues[3], '#', 2);
      CodeBuf.Add(' cpx #' + IntToStr(StrToInt(mValues[3]) - 1));
    end;

    CodeBuf.Add(' jcc for_loop_' + IntToStr(CurLine));
  end;
end;

{
  Procedure name : sc_SAssign
  Description    : Handles Action! SAssign procedure statement
}
procedure sc_SAssign;
var
  Buffer : String;
begin
  Buffer := Strip(TextBuf[CurLine], ' ');
  
  if System.Pos('SASSIGN(', UpperCase(Buffer)) > 0 then
  begin
    if GetParams(TextBuf[CurLine], True, False, '1111') <> 3 then Exit;
    
    if System.Pos('#', mValues[1]) > 0 then
      mValues[1] := Extract(mValues[1], '#', 2);

    CodeBuf.Add(' ldy ' + mValues[2]);
    CodeBuf.Add(' dey');
    CodeBuf.Add(' ldx #0');
    CodeBuf.Add('for_loop_' + IntToStr(CurLine));
    CodeBuf.Add(' lda ' + mValues[1] + ',y');
    CodeBuf.Add(' sta ' + mValues[0] + ',x');
    CodeBuf.Add(' iny');
    CodeBuf.Add(' inx');
    CodeBuf.Add(' cpx ' + mValues[3]);
    CodeBuf.Add(' jcc for_loop_' + IntToStr(CurLine));
  end;
end;

{
  Procedure name : sc_SCompare
}
procedure sc_SCompare;
var
  Buffer : String;
begin
  // NOT IMPLEMENTED YET
  
  Buffer := Strip(TextBuf[CurLine], ' ');

  if System.Pos('SCOMPARE(', UpperCase(Buffer)) > 0 then
  begin
  end;
end;

{
  Procedure name : sc_XIO
  Description    : Handles Action! XIO procedure statement
  Parameters     : None
  
  PROC XIO(BYTE channel, 0, command, aux1, aux2, fileString)
}
procedure sc_XIO;
var
  n, n1 : LongInt;
  Proc, mParams, Buffer: String;
begin
  Buffer := Strip(TextBuf[CurLine], ' ');
  Proc := 'XIO';

  n1 := System.Pos(UpperCase(Proc + '('), UpperCase(Buffer));  

  if n1 > 0 then
  begin
    Buffer := Trim(TextBuf[CurLine]);
    
    GetParams(Buffer, False, False, '00011111111');
    for n := 0 to n1 do
    begin
      case n of
        1: mParams := mValues[0];
        2: //
        else
          mParams := mParams + ', ' + mValues[n];
      end;
    end;
    
    CodeBuf.Add(' ' + Proc + ' ' + mParams);
  end;
end;

{ I/O routines }

{
  Procedure name : sc_Open
  Description    : Handles Action! Open procedure statement
  Parameters     : None
  
  PROC Open(BYTE channel, fileString, BYTE mode, aux2)
}
procedure sc_Open;
var
  n : LongInt;
  Proc, mParams : String[255];
  Buf : String;
  Num : Integer;
begin
  Buf := Strip(TextBuf[CurLine], ' ');
  Proc := 'Open';

  n := System.Pos(UpperCase(Proc + '('), UpperCase(Buf));
  if (n > 0) then
  begin
    mParams := '';
    Buf := Trim(TextBuf[CurLine]);

    GetParams(Buf, False, False, '211');
    mParams := mValues[0];
    mParams := mParams + ', ' + mValues[2];
    mParams := mParams + ', ' + mValues[1] + _EFF;
    Num := 1;    
    CodeBuf.Add(' ' + Proc + ' ' + mParams);

// BMI - short jump
// JMI - long jump
    l_IO_error[Num] := True;
    CodeBuf.Add(' jmi stop' + IntToStr(Num));
  end;
end;

{
  Procedure name : sc_Close
  Description    : Handles Action! Close procedure statement
  Parameters     : None

  PROC Close(BYTE channel)
}
procedure sc_Close;
var
  Proc, mParams : String[255];
  Buffer : String;
begin
  Buffer := Strip(TextBuf[CurLine], ' ');
  Proc := 'Close';

  if System.Pos(UpperCase(Proc + '('), UpperCase(Buffer)) > 0 then
  begin
    Buffer := Trim(TextBuf[CurLine]);
    GetParams(Buffer, False, False, '2');
    mParams := mValues[0];
    CodeBuf.Add(' ' + Proc + ' ' + mParams);
  end;
end;

{
  Procedure name : sc_PrintDX
  Description    : Handles Action! PrintD and PrintDE procedure statement
  Parameters     : boolCR

  Action! syntax:
    PROC PrintDE(BYTE channel, string)
  
  Examples:
    PrintD(6,"hi, atariage")
    PrintDE(1,text_buf)
}
procedure sc_PrintDX(boolCR : Boolean);
var
  Buf, Str1, Str2, Str3 : String;
  n, Value, Code : Integer;
begin
  Str1 := Strip(TextBuf[CurLine], ' ');
  
  if boolCR then
    Buf := 'PrintDE('
  else
    Buf := 'PrintD(';

  if (System.Pos(UpperCase(Buf), UpperCase(Str1)) > 0) then
  begin
    Str1 := Extract(TextBuf[CurLine], '(', 2);
    Split(Str1, ',', []);

    //StrBuf[0] := Channel(StrBuf[0]);
    Buf := StrBuf[0];
    Val(Buf, Value, Code);
    
    if Code = 0 then
    begin
      if Copy(Buf, 1, 1) = '$' then
      begin
        Buf := Copy(Buf, 2, Length(Buf) - 1);
        n := StrToInt(Buf) * 10;
        StrBuf[0] := '#$' + IntToStr(n);
      end else begin
        Buf := '#$' + Buf + '0';
        StrBuf[0] := Buf;
      end;
    end;

    if System.Pos('"', Str1) > 0 then
    begin
      Str3 := ExtractText(Str1, '"', '"');
      Value := Length(Str3);
      Str2 := '_str_buffer_' + IntToStr(CurLine) + _EFF;

      SData[Cnt] := Str2 + ' .byte ' + QuotedStr(Str3);
      
      if boolCR then
        SData[Cnt] := SData[Cnt] + ', $9b';
        //SData[Cnt] := Str2 + ' .byte ' + QuotedStr(Str3) + ', $9b'
      //else
        
      Inc(Cnt);
      CodeBuf.Add(' Read ' + StrBuf[0] + ', #9, #' + Str2 + ', #' + IntToStr(Value));

    end else begin
      Str3 := Copy(StrBuf[1], 1, Length(StrBuf[1]) - 1);

      for n := 1 to GVarCnt2 do
      begin
        if UpperCase(GVar2[n].VarName) = UpperCase(Str3) then
        begin
          Value := GVar2[n].Dim;
          Break;
        end;
      end;

      CodeBuf.Add(' Read ' + StrBuf[0] + ', #9, #' + Str3 + _EFF + ', #' + IntToStr(Value));
    end;
  end;
end;

{
  Procedure name : sc_PutDX
  Description    : Handles Action! PutD and PutDE procedure statement
  Parameters     : Proc

  Action! syntax:
    PROC PutD(BYTE channel, CHAR character)  
}
procedure sc_PutDX(Proc : String);
var
  mParams, Buf : String;
begin
  Buf := Strip(TextBuf[CurLine], ' ');  
  
  if (System.Pos(UpperCase(Proc + '('), UpperCase(Buf)) > 0) then
  begin
    mParams := '';
    GetParams(Buf, False, False, '21');
    mParams := mValues[0];    
    mParams := mParams + ', ' + mValues[1];
    CodeBuf.Add(' ' + Proc + ' ' + mParams);
  end;
end;

{
  Procedure name : sc_GetD
  Description    : Handles Action! GetD function statement
  Parameters     : None

  Action! syntax:
    CHAR FUNC GetD(BYTE channel)  
}
procedure sc_GetD;
var
  mParams : String = '';
  Buf : String;
begin
  Buf := Strip(TextBuf[CurLine], ' ');

  if (System.Pos('GETD(', UpperCase(Buf)) > 0) then
  begin
    GetParams(Buf, False, False, '2');
    mParams := mValues[0];
    CodeBuf.Add(' GetD ' + mParams);    
    CodeBuf.Add(' mva Store1 ' + Trim(Extract(TextBuf[CurLine], '=', 1)) + _EFF);
  end;
end;

{
  Procedure name : sc_PrintXD
  Description    : Generic Action! print device procedure statement
  Parameters     : Proc - Procedure name value
                   lEnter - New line
}
procedure sc_PrintXD(Proc : String; lEnter : Boolean);
var
  n1 : Integer;
  mLabel, mParams : String[255];
  Buf : String;
begin
  Buf := Strip(TextBuf[CurLine], ' ');
  n1 := System.Pos(UpperCase(Proc + '('), UpperCase(Buf)); 
  
  if (n1 > 0) then
  begin
    mLabel := '_' + Proc + '_' + IntToStr(CurLine) + '_' + IntToStr(n1);
    GetParams(Buf, False, False, '21');
    mParams := mValues[0];
    CodeBuf.Add(' printfd ' + mParams);

    if lEnter then
      CodeBuf.Add(' dta c' + QuotedStr('%') + ',$9b,0')
    else
      CodeBuf.Add(' dta c' + QuotedStr('%') + ',0');

    if Copy(mValues[1], 1, 1) = '#' then 
    begin
      Buf := Copy(mValues[1], 2, Length(mValues[1]) - 1);
      CodeBuf.Add(' dta a(' + mLabel + ')');
      SData[Cnt] := mLabel + ' dta a(' + Buf + ')';
      Inc(Cnt);
    end else begin
      Buf := mValues[1];
      CodeBuf.Add(' dta a(' + Buf + ')');
    end;
  end;
end;

{
  Procedure name : sc_InputSD
  Description    : Handles Action! InputSD procedure statement
  Parameters     : Proc - Procedure name (InputSD, InputMD)
                   Max - Maximum length of buffer
                   Flags - Flags to determine number of parameters
  Action! syntax:
    PROC InputSD(BYTE channel, string)
    PROC InputMD(BYTE channel, string, BYTE max)
}
procedure sc_InputSMD(Proc, Max, Flags : String);
var
  mParams, Buf : String;
begin
  Buf := Strip(TextBuf[CurLine], ' ');
  
  if (System.Pos(Proc + '(', UpperCase(Buf)) > 0) then
  begin
    mParams := '';
    GetParams(Buf, False, False, Flags);
    mParams := mValues[0];  // IOCB channel number
    mParams := mParams + ', #5';  // Get record    
    mParams := mParams + ', ' + mValues[1] + _EFF;  // Get buffer

    if Flags = '211' then Max := mValues[2];

    mParams := mParams + ', ' + Max;  // Length of buffer
    CodeBuf.Add(' Read ' + mParams);    
  end;
end;

{
  Procedure name : sc_Point
  Description    : Handles Action! Point procedure statement
  Parameters     : None

  Action! syntax:
    PROC Point(BYTE channel, CARD sector, BYTE offset)  
}
procedure sc_Point;
var
  Proc, mParams, Buffer : String;
begin
  Buffer := Strip(TextBuf[CurLine], ' ');
  Proc := 'Point';

  if System.Pos(UpperCase(Proc + '('), UpperCase(Buffer)) > 0 then
  begin
    GetParams(Buffer, False, False, '211');
    mParams := mValues[0];
    CodeBuf.Add(' ' + Proc + ' ' + mParams + ', ' + mValues[1] + ', ' + mValues[2]);
  end;
end;

{
  Procedure name : sc_Note
  Description    : Handles Action! Note procedure statement
  Parameters     : None

  Action! syntax:
    PROC Note(BYTE channel, CARD POINTER sector, BYTE POINTER offset)  
}
procedure sc_Note;
var
  Proc, mParams, Buffer : String;
begin
  Buffer := Strip(TextBuf[CurLine], ' ');
  Proc := 'Note';

  if System.Pos(UpperCase(Proc + '('), UpperCase(Buffer)) > 0 then
  begin
    GetParams(Buffer, False, False, '111');
    mParams := mValues[0];        
    CodeBuf.Add(' ' + Proc + ' ' + mParams + ', ' + mValues[1] + ', ' + mValues[2]);
  end;
end;

{
  Procedure name : sc_Locate
  Description    : Handles Action! Locate function statement
  Parameters     : None

  Action! syntax:
    BYTE FUNC Locate(CARD column, BYTE row)  
}
procedure sc_Locate;
var
  Buf : String;
begin
  Buf := Strip(TextBuf[CurLine], ' ');
//  Proc := 'Locate';

  if System.Pos('LOCATE(', UpperCase(Buf)) > 0 then
  begin
    GetParams(Buf, False, False, '11');
    CodeBuf.Add(' Locate ' + mValues[0] + ', ' + mValues[1]);
    Buf := Extract(TextBuf[CurLine], '=', 1) + _EFF;
    CodeBuf.Add(' mva Store1 ' + Buf);
  end;
end;

{
  Procedure name : sc_StrNum
  Description    : Handles Action! StrB, StrI and StrC procedure statements
  Parameters     : Proc - Name of the string conversion routine (StrB, StrI or StrC)

  Action! syntax:
    PROC StrB(BYTE number, string)
}
procedure sc_StrNum(Proc : String);
var
  n : LongInt;
  Str1, Str2 : String;
begin
  Str1 := Strip(TextBuf[CurLine], ' ');

  if System.Pos(UpperCase(Proc + '('), UpperCase(Str1)) > 0 then
  begin
    Str1 := ExtractText(TextBuf[CurLine], '(', ')');
    Split(Str1, ',', []);

    if StrBuf.Count < 2 then Exit;

    Str1 := StrBuf[0];
    Str2 := StrBuf[1];
    CodeBuf.Add(' ldy #0');

    if IsNumber(Str1[1]) then
    begin
      for n := 1 to Length(Str1) do
      begin
        CodeBuf.Add(' mva #''' + Str1[n] + '''' + ' ' + Str2 + _EFF + ',y');
        if n < Length(Str1) then CodeBuf.Add(' iny');
      end;
    end else begin
      CodeBuf.Add(' mva ' + Str1 + _EFF + ' ' + Str2 + _EFF + ',y');
    end;
  end;
end;

{
  Function name : Channel
  Description   : Calculates dec/hex convention if necessary
  Parameters    : Value
  Returns       : Returns string value with recalculated number in dec/hex
}
function Channel(Value : String) : String;
var
  Buf : String;
  n1 : Byte;
begin
  if Copy(Value, 1, 1) <> '#' then
    Result := Value
  else begin
    Buf := Copy(Value, 2, Length(Value) - 1);
    if Buf[1] = '$' then
    begin
      Buf := Copy(Buf, 2, Length(Buf) - 1);
      n1 := StrToInt(Buf) * 10;
      Result := '#$' + IntToStr(n1);
    end else begin
      n1 := StrToInt(Buf) * 16;
      Result := '#' + IntToStr(n1);
    end;
  end;
end;

{
  Function name : GetParams
  Description   : Handles command parameters
  Parameters    : StrBufX,
                  lNum,
                  lConv,
                  ParamTypes
  Returns       : Returns the number of command parameters
}
function GetParams(StrBufX : String; lNum, lConv : Boolean; ParamTypes : String) : Integer;
var
  n, n3, CntX, r : Integer;
  mVal, Buf, StrBuffer : String;
  Flag : Boolean = False;
  arrOper, StrBuf3 : TStringList;
begin
  for n := 1 to 10 do mvalues[n] := '';
  
  StrBuffer := '_str_buffer_' + IntToStr(CurLine);
  
  StrBuf3 := TStringList.Create;
  StrBuf3.Clear;
  
  if ParamTypes = '3' then
    Buf := Extract(strbufx, '=', 2)
  else    
    Buf := ExtractText(strbufx, '(', ')');
  
  // Check for Put(' ) command
  //
  if Buf = Chr(39) + Chr(32) then
  begin
    CntX := 0;
    StrBuf3.Add(Buf);
  
  // otherwise normal processing follows
  //
  end else begin
    if System.Pos('(', Buf) > 0 then Buf := Replace(Buf, '(', '[');
    if System.Pos(')', Buf) > 0 then Buf := Replace(Buf, ')', ']');

    Split(Buf, ',', []);
    CntX := StrBuf.Count - 1;

    for n := 0 to CntX do
      StrBuf3.Add(StrBuf[n]);
  end;

  for n := 0 to CntX do
  begin
    mVal := StrBuf3[n];

    if IsNumber(mVal[1])
       and ((ParamTypes[n + 1] = '1') or (ParamTypes[n + 1] = '2') or (Copy(ParamTypes, 1, 1) = '3')) then
    begin
      mValues[n] := Trim('#' + mVal);
    end else if mVal[1] = '''' then
      mValues[n] := '#' + mVal + ''''
    else begin
      Flag := False;

      for n3 := 1 to GVarCnt2 do
      begin
        if System.Pos(UpperCase(GVar2[n3].VarName), UpperCase(StrBufX)) > 0 then
        begin
          Flag := True;
            
          // Check
          if not lNum then
            mValues[n] := '#' + Trim(mVal)
          else
            mValues[n] := Trim(mVal) + _EFF;

          break;
        end
      end;

      arrOper := TStringList.Create;

      try
        if Copy(mVal, 1, 1) <> '"' then
        begin
          // Process parameters which include operators
  
          // Check and count oparators contained in paramater value
          //
          for r := 0 to Length(mVal) - 1 do
          begin
            if (mVal[r] = '+') or (mVal[r] = '-') or (mVal[r] = '*') or (mVal[r] = '/') then
            begin
              arrOper.Add(mVal[r]);
            end;
          end;
  
          // Handle parameters and their operators
          //
          if arrOper.Count > 0 then
          begin
            SplitEx2(mVal, '+', '-', '*', '/');
  
            for r := 0 to arrOper.Count - 1 do
            begin
              if r = 0 then
                mVal := strbuf2[0] + arrOper[r] + strbuf2[1]
              else
                if ParamTypes[n + 1] = '1' then
                  mVal := 'b_param' + IntToStr(n + 1) + arrOper[r] + strbuf2[r + 1]
                else
                  mVal := 'w_param' + IntToStr(n + 1) + arrOper[r] + strbuf2[r + 1];
  
              if ParamTypes[n + 1] = '1' then
              begin
                mValues[n] := 'b_param' + IntToStr(n + 1);
                MathExpr('T1', mValues[n], mVal, 1, 0);
              end else begin
                mValues[n] := 'w_param' + IntToStr(n + 1);
                MathExpr('T2', mValues[n], mVal, 1, 0);
              end;
            end;
          end;
        end;

        if not Flag and (arrOper.Count = 0) then
        begin
          if IsNumber(mVal[1]) then
            mValues[n] := mVal
          else
            mValues[n] := mVal + _EFF;
        end;

      finally
        if Assigned(arrOper) then FreeAndNil(arrOper);
      end;

      if Copy(mValues[n], 1, 1) = '"' then
      begin
        if lConv then
        //if lNum and not PrgVar.ParamStr then
          mValues[n] := ExtractText(mValues[n], '"', '"')
        else begin
          if Copy(mValues[n], 3, 1) = '"' then
            mValues[n] := '#' + QuotedStr(mValues[n][2])
          else begin
            SData[Cnt] := Strbuffer + _EFF + ' .byte ' +
                          QuotedStr(Copy(mValues[n], 2, Length(mValues[n]) - Length(_EFF) - 2)) + ', $9b';
            Inc(Cnt);
            mValues[n] := '#' + StrBuffer;
          end;
        end;
      end else begin
        if LowerCase(mValues[n]) = 'device' then
          mValues[n] := 'device.devscr';
      end
    end;
  end;
  
  // Calculate dec/hex convention if necessary  
  for n := 0 to CntX do
  begin
    if ParamTypes[n + 1] = '2' then
      mValues[n] := Channel(mValues[n]);
  end;
  
  //PrgVar.ParamStr := False;
  Result := CntX;
  
  StrBuf3.Free;  
end;

{
  Procedure name : sc_PrintF
  Description    : Handles Action! PrintF procedure statement
  Parameters     : None

format char	description for Action!:
%I	INT
%U	CARD (the U stands for Unsigned) and BYTE
%C	print as a character
%H	Hexadecimal number
%E	the RETURN character
%%	output the percent sign
%S	output as a string

Some examples:
  PrintF("%EA%EB%E")
  PrintF("%EThe sum of %U and %U is %U%E",a,b,a+b
  PrintF("The letter %C.%E",65)
  PrintF("Score %U: %U",player,score(player))

MADS method:
 jsr printf
 dta c'tekst #@%',$9b,0
 dta a(string)
 dta a(float)
 dta a(word)
}
procedure sc_PrintF;
var
  n, n1 : LongInt;
  Str1, Str2 : String;
  varx : Array[0..21] of string[10];
  VarxCnt : Integer = 0;
begin
  Str1 := Strip(TextBuf[CurLine], ' ');
  
  if System.Pos('PRINTF("', UpperCase(Str1)) > 0 then
  begin  
    Str2 := ExtractText(TextBuf[CurLine], '"', '"'); 

    (*
    Str3 := ExtractText(Str1, '"', '"');
    Str1 := Copy(Str1, 8 + Length(Str3), Length(Str1) - (8 + Length(Str3)));
    Str1 := Extract(Str1, '"', 2);
    Str1 := Copy(Str1, 2, Length(Str1) - 2);
    *)

    Str1 := ExtractWord(3, TextBuf[CurLine], ['"']);
    Str1 := Extract(Str1, ',', 2);
    Str1 := Extract(Str1, ')', 1);

    Split(Str1, ',', []);
    
    if StrBuf.Count = 0 then Varx[0] := Str1;
    
    for n := 0 to StrBuf.Count - 1 do
    begin      
      varx[n] := StrBuf[n];
    end;

    Str2 := StringReplace(Str2, '%%', '_eff_p_eff_', []);
    Split(Str2, '%', [sNoTrim]);
        
    for n := 0 to StrBuf.Count - 1 do
    begin
      if n = 0 then
      begin
        if StrBuf[n] <> '' then
        begin
          if StrBuf[n] = '_eff_p_eff_' then
            CodeBuf.Add(' put #' + QuotedStr('%'))
          else begin        
            StrBuf[n] := StringReplace(StrBuf[n], '_eff_p_eff_', ':', []);
            CodeBuf.Add(' jsr printf');
            CodeBuf.Add(' dta c' + QuotedStr(StrBuf[n]) + ',0');
          end;
        end;

      end else if UpperCase(Copy(StrBuf[n], 1, 1)) = 'E' then
      begin
        CodeBuf.Add(' PutE');
        //Str2 := ExtractWord(1, UpperCase(StrBuf[n]), ['E']);
        Str2 := ExtractNoTrim(StrBuf[n], 'E', 2);

        if Str2 <> '' then
        begin
          Str2 := StringReplace(Str2, '_eff_p_eff_', ':', []);
          CodeBuf.Add(' jsr printf');
          CodeBuf.Add(' dta c' + QuotedStr(Str2) + ',0');
        end;

      end else if UpperCase(Copy(StrBuf[n], 1, 1)) = 'I' then
      begin
        CodeBuf.Add(' jsr printf');
        CodeBuf.Add(' dta c' + QuotedStr('%') + ',0');
        CodeBuf.Add(' dta a(' + Varx[VarxCnt] + _EFF + ')');
        Inc(VarxCnt);

        //Str2 := ExtractWord(1, UpperCase(StrBuf[n]), ['I']);
        Str2 := ExtractNoTrim(StrBuf[n], 'I', 2);

        if Str2 <> '' then
        begin
          Str2 := StringReplace(Str2, '_eff_p_eff_', ':', []);
          CodeBuf.Add(' jsr printf');
          CodeBuf.Add(' dta c' + QuotedStr(Str2) + ',0');
        end;

      end else if UpperCase(Copy(StrBuf[n], 1, 1)) = 'U' then
      begin
        for n1 := 1 to GVarCnt do
        begin
          // BYTE and CHAR variables
          if (System.Pos(GVar[n1].VarType, 'T1T2') > 0)
             and (UpperCase(GVar[n1].VarName) = UpperCase(Varx[VarxCnt])) then
          begin
            CodeBuf.Add(' mva ' + Varx[VarxCnt] + _EFF + ' $c0');
            CodeBuf.Add(' jsr printf');
            CodeBuf.Add(' dta c' + QuotedStr('%') + ',0');
            CodeBuf.Add(' dta a($c0)');
            Inc(VarxCnt);
            Break;

          // Other variable types
          end else begin    
            CodeBuf.Add(' jsr printf');
            CodeBuf.Add(' dta c' + QuotedStr('%') + ',0');
            CodeBuf.Add(' dta a(' + Varx[VarxCnt] + _EFF + ')');
            Inc(VarxCnt);
            Break;
          end;
        end;

        //Str2 := ExtractWord(1, UpperCase(StrBuf[n]), ['U']);
        Str2 := ExtractNoTrim(StrBuf[n], 'U', 2);

        if Str2 <> '' then
        begin
          Str2 := StringReplace(Str2, '_eff_p_eff_', ':', []);
          CodeBuf.Add(' jsr printf');
          CodeBuf.Add(' dta c' + QuotedStr(Str2) + ',0');
        end;

      end else if UpperCase(Copy(StrBuf[n], 1, 1)) = 'C' then
      begin
        Str2 := StrBuf[n];
        GetParams('Put(' + VarX[VarxCnt] + ')', False, False, '1');
        Inc(VarxCnt);
        CodeBuf.Add(' Put ' + mValues[0]);

        if mValues[0] <> '' then
        begin
          Str2 := ExtractNoTrim(Str2, 'C', 2);

          if Str2 <> '' then
          begin
            Str2 := StringReplace(Str2, '_eff_p_eff_', ':', []);
            CodeBuf.Add(' jsr printf');
            CodeBuf.Add(' dta c' + QuotedStr(Str2) + ',0');
          end;
        end;

        (*
        Str2 := ExtractWord(1, UpperCase(StrBuf[n]), ['C']);
        GetParams('Put(' + VarX[VarxCnt] + ')', False, False, '1');
        Inc(VarxCnt);
        CodeBuf.Add(' Put ' + mValues[0]);
        
        if Str2 <> '' then
        begin
          Str2 := StringReplace(Str2, '_eff_p_eff_', ':', []);
          CodeBuf.Add(' jsr printf');
          CodeBuf.Add(' dta c' + QuotedStr(Str2) + ',0');
        end;
        *)

      end else if UpperCase(Copy(StrBuf[n], 1, 1)) = 'H' then
      begin
        CodeBuf.Add(' ShowHex ' + Varx[VarxCnt] + _EFF);
        CodeBuf.Add(' jsr printf');
        CodeBuf.Add(' dta c' + QuotedStr('$000') + ',0');
        CodeBuf.Add(' jsr printf');
        CodeBuf.Add(' dta c' + QuotedStr('#') + ',0');
        CodeBuf.Add(' dta a(hex_num+1)');
        Inc(VarxCnt);

        //Str2 := ExtractWord(1, UpperCase(StrBuf[n]), ['H']);
        Str2 := ExtractNoTrim(StrBuf[n], 'H', 2);

        if Str2 <> '' then
        begin
          Str2 := StringReplace(Str2, '_eff_p_eff_', ':', []);
          CodeBuf.Add(' jsr printf');
          CodeBuf.Add(' dta c' + QuotedStr(Str2) + ',0');
        end;
      
      //end else if UpperCase(Copy(StrBuf[n], 1, 2)) = '_eff_p_eff_' then
      //begin
      //  CodeBuf.Add(' put #' + QuotedStr('%'));
        //CodeBuf.Add(' Put #37');      

      end else begin
        if StrBuf[n] <> '' then
        begin
          StrBuf[n] := StringReplace(StrBuf[n], '_eff_p_eff_', ':', []);
          CodeBuf.Add(' jsr printf');
          CodeBuf.Add(' dta c' + QuotedStr(StrBuf[n]) + ',0');
        end;
      end;
    end;
  end;
end;

end.
