#include "stdafx.h"

struct ATXFileHeader {
	uint8_t		mSignature[4];			// AT8X
	uint16_t	mVersionMajor;			// 1
	uint16_t	mVersionMinor;			// 1
	uint16_t	mFlags;					// 0x0002
	uint16_t	mMysteryDataCount;
	uint8_t		mFill1[16];
	uint32_t	mTrackDataOffset;
	uint32_t	mTotalSize;
	uint8_t		mFill2[12];
};

struct ATXTrackHeader {
	uint32_t	mSize;
	uint16_t	mType;
	uint16_t	mReserved06;
	uint8_t		mTrackNum;
	uint8_t		mReserved09;
	uint16_t	mNumSectors;
	uint8_t		mFill2[5];
	uint8_t		mMysteryIndex;
	uint8_t		mFill3[2];
	uint32_t	mDataOffset;
	uint8_t		mFill4[8];
};

struct ATXSectorHeader {
	uint8_t		mIndex;
	uint8_t		mFDCStatus;		// not inverted
	uint16_t	mTimingOffset;
	uint32_t	mDataOffset;
};

struct ATXTrackChunkHeader {
	enum {
		kTypeSectorData = 0x00,
		kTypeSectorList = 0x01,
		kTypeWeakBits = 0x10
	};

	uint32_t	mSize;
	uint8_t		mType;
	uint8_t		mNum;
	uint16_t	mData;
};

void read_atx(DiskInfo& disk, const char *path, int track) {
	FILE *fi = fopen(path, "rb");
	if (!fi)
		fatalf("Unable to open file: %s\n", path);

	ATXFileHeader filehdr;
	read_raw(&filehdr, sizeof filehdr, fi, path);

	if (memcmp(&filehdr.mSignature, "AT8X", 4))
		fatalf("Cannot read ATX file %s: incorrect signature; possibly not an ATX file\n", path);

	for(;;) {
		ATXTrackHeader trkhdr;
		uint32_t trkbase = (uint32_t)ftell(fi);

		if (1 != fread(&trkhdr.mSize, 8, 1, fi))
			break;

		if (trkhdr.mSize < 8 || trkhdr.mSize >= 0x8000000U)
			fatal_read();

		if (trkhdr.mType != 0) {
			fseek(fi, trkbase + trkhdr.mSize, SEEK_SET);
			continue;
		}

		// read in the rest of the track header
		if (trkhdr.mSize < sizeof(trkhdr))
			fatal("Invalid track header in ATX file.\n");

		read_raw(&trkhdr.mTrackNum, sizeof(trkhdr) - 8, fi, path);

		// check track number
		if (trkhdr.mTrackNum > 40) {
			printf("WARNING: Ignoring track in ATX file: %u\n", trkhdr.mTrackNum);
			fseek(fi, trkbase + trkhdr.mSize, SEEK_SET);
			continue;
		}

		// read in the track
		TrackInfo& track_info = disk.mTracks[trkhdr.mTrackNum];

		std::vector<uint8_t> rawtrack(trkhdr.mSize);
		read_raw(rawtrack.data() + sizeof(trkhdr), trkhdr.mSize - sizeof(trkhdr), fi, path);

		// parse track chunks
		if (trkhdr.mSize >= sizeof(trkhdr) + 8) {
			std::vector<std::pair<uint8_t, uint16_t>> weakSectorInfo;

			uint32_t tcpos = trkhdr.mDataOffset;
			while(tcpos < trkhdr.mSize - 8) {
				ATXTrackChunkHeader tchdr;
				memcpy(&tchdr, rawtrack.data() + tcpos, 8);

				if (!trkhdr.mSize || trkhdr.mSize - tcpos < tchdr.mSize)
					break;

				if (tchdr.mType == ATXTrackChunkHeader::kTypeSectorList) {
					if (tchdr.mSize < sizeof(ATXTrackChunkHeader) + sizeof(ATXSectorHeader) * trkhdr.mNumSectors)
						fatalf("Invalid ATX image: Sector list at %08x has size %08x insufficient to hold %u sectors.\n", (uint32_t)trkbase + tcpos, tchdr.mSize, trkhdr.mNumSectors);

					// process sectors
					for(uint32_t i=0; i<trkhdr.mNumSectors; ++i) {
						ATXSectorHeader shdr;
						memcpy(&shdr, rawtrack.data() + tcpos + sizeof(ATXTrackChunkHeader) + sizeof(ATXSectorHeader) * i, sizeof(ATXSectorHeader));

						// check for missing sector
						if (shdr.mFDCStatus & 0x10)
							continue;

						// validate data location
						if (shdr.mDataOffset > trkhdr.mSize && trkhdr.mSize - shdr.mDataOffset < 128)
							fatalf("Invalid ATX image: track %u, sector %u extends outside of track chunk.\n", trkhdr.mTrackNum, shdr.mIndex);

						track_info.mSectors.push_back(SectorInfo());
						auto& sec = track_info.mSectors.back();

						memcpy(sec.mData, rawtrack.data() + shdr.mDataOffset, 128);
						memset(sec.mData + 128, 0, sizeof(sec.mData) - 128);

						sec.mIndex = shdr.mIndex;
						sec.mbMFM = false;
						sec.mAddressMark = (shdr.mFDCStatus & 0x20) ? 0xF8 : 0xFB;
						sec.mPosition = (float)shdr.mTimingOffset / 26042.0f;
						sec.mSectorSize = shdr.mFDCStatus & 0x04 ? 256 : 128;
						sec.mWeakOffset = -1;
						sec.mComputedCRC = ComputeInvertedCRC(sec.mData, 128, ComputeCRC(&sec.mAddressMark, 1));
						sec.mRecordedCRC = shdr.mFDCStatus & 0x08 ? ~sec.mComputedCRC : sec.mComputedCRC;
					}
				} else if (tchdr.mType == ATXTrackChunkHeader::kTypeWeakBits) {
					weakSectorInfo.push_back(std::pair<uint8_t, uint16_t>(tchdr.mNum, tchdr.mData));
				}

				tcpos += tchdr.mSize;
			}

			// process weak sector data
			while(!weakSectorInfo.empty()) {
				const auto& wsi = weakSectorInfo.back();

				if (wsi.first >= track_info.mSectors.size()) {
					printf("WARNING: ATX track %u contains extra data for nonexistent sector index %u.\n", track, wsi.first);
				} else {
					SectorInfo& si = track_info.mSectors[wsi.first];

					if (wsi.second >= si.mSectorSize) {
						printf("WARNING: ATX track %u, sector %u has out of bounds weak data offset: %u.\n", track, si.mIndex, wsi.second);
					} else {
						si.mWeakOffset = (int16_t)wsi.second;
					}
				}

				weakSectorInfo.pop_back();
			}
		}

		// next track
		fseek(fi, trkbase + trkhdr.mSize, SEEK_SET);
	}

	fclose(fi);
}

void write_atx(const char *path, DiskInfo& disk, int track) {
	printf("Writing ATX file: %s\n", path);

	FILE *fo = fopen(path, "wb");
	if (!fo)
		fatalf("Unable to open output file: %s.\n", path);

	// write header
	fwrite("AT8X", 4, 1, fo);
	write_u16(1, fo);
	write_u16(1, fo);
	write_u16(2, fo);
	write_u16(0, fo);
	write_pad(16, fo);
	write_u32(48, fo);
	write_u32(0, fo);
	write_pad(12, fo);

	// write tracks
	int phantom_sectors = 0;
	int missing_sectors = 0;
	int error_sectors = 0;

	for(int i=0; i<40; ++i) {
		TrackInfo& track_info = disk.mTracks[i];

		// sort and sift out usable sectors from track
		std::vector<SectorInfo *> secptrs;
		sift_sectors(track_info, i, secptrs);

		const int num_secs = (int)secptrs.size();

		bool sector_map[18] = {0};
		int weak_count = 0;
		for(auto it = secptrs.begin(), itEnd = secptrs.end();
			it != itEnd;
			++it)
		{
			auto *sec_ptr = *it;

			if (sec_ptr->mWeakOffset >= 0)
				++weak_count;

			if (sec_ptr->mIndex >= 1 && sec_ptr->mIndex <= 18) {
				if (sector_map[sec_ptr->mIndex - 1])
					++phantom_sectors;
				else
					sector_map[sec_ptr->mIndex - 1] = true;
			}
		};

		// write track header
		write_u32(32 + 8 + 8 + (128 + 8)*num_secs + 8*weak_count + 8, fo);
		write_u16(0, fo);
		write_u16(0, fo);
		write_u8(i, fo);
		write_u8(0, fo);
		write_u16(num_secs, fo);
		write_pad(5, fo);
		write_u8(0, fo);
		write_pad(2, fo);
		write_u32(32, fo);
		write_pad(8, fo);

		// write sector list header
		write_u32(8 + 8 * num_secs, fo);
		write_u8(1, fo);
		write_pad(3, fo);

		// write out sector headers

		uint32_t data_offset = 32 + 8 + 8*num_secs + 8;
		for(auto it = secptrs.begin(), itEnd = secptrs.end();
			it != itEnd;
			++it)
		{
			auto *sec_ptr = *it;

			write_u8(sec_ptr->mIndex, fo);

			uint8_t fdcStatus = 0;

			fdcStatus += (~sec_ptr->mAddressMark & 3) << 5;

			// set CRC error flag if CRC didn't match
			if (sec_ptr->mComputedCRC != sec_ptr->mRecordedCRC)
				fdcStatus |= 0x08;

			// set CRC error, lost data, and DRQ flags if sector is long
			if (sec_ptr->mSectorSize != 128) {
				printf("WARNING: Long sector of %u bytes found for track %d, sector %d.\n", sec_ptr->mSectorSize, i, sec_ptr->mIndex);
				fdcStatus |= 0x0e;
			}

			if (fdcStatus)
				++error_sectors;

			write_u8(fdcStatus, fo);
			write_u16((uint16_t)((int)(sec_ptr->mPosition * 26042) % 26042), fo);
			write_u32(data_offset, fo);
			data_offset += 128;
		};

		// write out sector data
		write_u32(128 * num_secs + 8, fo);
		write_u8(0, fo);
		write_pad(3, fo);

		for(auto it = secptrs.begin(), itEnd = secptrs.end();
			it != itEnd;
			++it)
		{
			auto *sec_ptr = *it;
			fwrite(sec_ptr->mData, 128, 1, fo);
		};

		// write out weak chunk info
		for(int j=0; j<num_secs; ++j) {
			if (secptrs[j]->mWeakOffset >= 0) {
				write_u32(8, fo);
				write_u8(0x10, fo);
				write_u8((uint8_t)j, fo);
				write_u16((uint16_t)secptrs[j]->mWeakOffset, fo);
			}
		};

		// write end of track chunks
		write_u32(0, fo);
		write_u32(0, fo);

		// report any missing sectors
		if (std::find(std::begin(sector_map), std::end(sector_map), true) == std::end(sector_map)) {
			if (track < 0 || track == i) {
				printf("WARNING: No sectors found for track %d -- possibly unformatted.\n", i);
				missing_sectors += 18;
			}
		} else {
			for(int j=0; j<18; ++j) {
				if (!sector_map[j]) {
					printf("WARNING: Missing track %d, sector %d.\n", i, j + 1);
					++missing_sectors;
				}
			}
		}
	}

	// back-patch size
	uint32_t total_size = ftell(fo);
	fseek(fo, 32, SEEK_SET);
	write_u32(total_size, fo);

	printf("%d missing sector%s, %d phantom sector%s, %d sector%s with errors\n"
		, missing_sectors, missing_sectors == 1 ? "" : "s"
		, phantom_sectors, phantom_sectors == 1 ? "" : "s"
		, error_sectors, error_sectors == 1 ? "" : "s");

	fclose(fo);
}