/*
		FISHING FOR ATARI
		
	a game for 2 players on 1 Lynx
		
					by Dr. Ludos

	I could never have been able to make a Lynx game without the wonderful tools and template created and shared by LynxDev community hanging on AtariAge forums. 
	In particular, I used CC65 "Lynx Edition" and the game template by Karri Kaksonen, the updated template with fadein, EEPROM save by Nop90 and Chipper audio driver and sound creation by Sage.
	
	So here's my little to contribution to the wonderful LYNX community, in the form of the fully commented source of this game.
	It's quite simple so I hope it will useful to someone who is starting Lynxdev like I was myself not so long ago!
*/

#include <lynx.h>
#include <tgi.h>
#include <joystick.h>
#include <stdlib.h>
#include <conio.h>
#include <6502.h>
#include <time.h>
#include <cc65.h>



//========= SPRITES ===========
//The sprite are declared here in the reverse order of the sprite chain (last sprite is on top of this list), so they can reference each other

//Empty sprite data (1px transparent), useful to hide sprite while manipulating their "data" only
extern unsigned char gfx_empty[];

//Player specific message (machines catched + points scored)
extern unsigned char gfx_clock_msg[];
extern unsigned char gfx_lynx_msg[];
extern unsigned char gfx_jaguar_msg[];
extern unsigned char gfx_atari400_msg[];
extern unsigned char gfx_atari800_msg[];
extern unsigned char gfx_atari2600_msg[];
extern unsigned char gfx_atarist_msg[];
static SCB_RENONE spr_score2msg = {
	BPP_4 | TYPE_NORMAL | VFLIP | HFLIP, 
	REUSEPAL, 
	NO_COLLIDE,
	0,
	gfx_empty,
	5, 0,
};
static SCB_RENONE spr_score1msg = {
	BPP_4 | TYPE_NORMAL, 
	REUSEPAL, 
	NO_COLLIDE,
	(char*) &spr_score2msg,
	gfx_empty,
	154, 101,
};


//Score counter (separate digits)
extern unsigned char gfx_0[];
extern unsigned char gfx_1[];
extern unsigned char gfx_2[];
extern unsigned char gfx_3[];
extern unsigned char gfx_4[];
extern unsigned char gfx_5[];
extern unsigned char gfx_6[];
extern unsigned char gfx_7[];
extern unsigned char gfx_8[];
extern unsigned char gfx_9[];
static SCB_RENONE spr_score2D = {
	BPP_4 | TYPE_NORMAL | VFLIP | HFLIP, 
	REUSEPAL, 
	NO_COLLIDE,
	(char*) &spr_score1msg,
	gfx_0,
	5, 97,
};
static SCB_RENONE spr_score2C = {
	BPP_4 | TYPE_NORMAL | VFLIP | HFLIP, 
	REUSEPAL, 
	NO_COLLIDE,
	(char*) &spr_score2D,
	gfx_0,
	5, 91,
};
static SCB_RENONE spr_score2B = {
	BPP_4 | TYPE_NORMAL | VFLIP | HFLIP, 
	REUSEPAL, 
	NO_COLLIDE,
	(char*) &spr_score2C,
	gfx_0,
	5, 85,
};
static SCB_RENONE spr_score2A = {
	BPP_4 | TYPE_NORMAL | VFLIP | HFLIP, 
	REUSEPAL, 
	NO_COLLIDE,
	(char*) &spr_score2B,
	gfx_0,
	5, 79,
};
static SCB_RENONE spr_score1D = {
	BPP_4 | TYPE_NORMAL, 
	REUSEPAL, 
	NO_COLLIDE,
	(char*) &spr_score2A,
	gfx_0,
	154, 4,
};
static SCB_RENONE spr_score1C = {
	BPP_4 | TYPE_NORMAL, 
	REUSEPAL, 
	NO_COLLIDE,
	(char*) &spr_score1D,
	gfx_0,
	154, 10,
};
static SCB_RENONE spr_score1B = {
	BPP_4 | TYPE_NORMAL, 
	REUSEPAL, 
	NO_COLLIDE,
	(char*) &spr_score1C,
	gfx_0,
	154, 16,
};
static SCB_RENONE spr_score1A = {
	BPP_4 | TYPE_NORMAL, 
	REUSEPAL, 
	NO_COLLIDE,
	(char*) &spr_score1B,
	gfx_0,
	154, 22,
};

//Player 1 & 2
extern unsigned char gfx_hook1[];
extern unsigned char gfx_hook2[];
static SCB_RENONE spr_hook2 = {
	BPP_4 | TYPE_NORMAL, 
	REUSEPAL, 
	NO_COLLIDE,
	(char*) &spr_score1A,
	gfx_hook2,
	0, 120,
};
static SCB_RENONE spr_hook1 = {
	BPP_4 | TYPE_NORMAL, 
	REUSEPAL, 
	NO_COLLIDE,
	(char*) &spr_hook2,
	gfx_hook1,
	0, 120,
};

//Foes (a.k.a Fish / Atari machines to catch)
extern unsigned char gfx_clock[];
extern unsigned char gfx_lynx[];
extern unsigned char gfx_jaguar[];
extern unsigned char gfx_atari400[];
extern unsigned char gfx_atari800[];
extern unsigned char gfx_atari2600[];
extern unsigned char gfx_atarist[];
static SCB_RENONE spr_foe7 = {
	BPP_4 | TYPE_NORMAL, 
	REUSEPAL, 
	NO_COLLIDE,
	(char*) &spr_hook1,
	gfx_lynx,
	0, 130,
};
static SCB_RENONE spr_foe6 = {
	BPP_4 | TYPE_NORMAL, 
	REUSEPAL, 
	NO_COLLIDE,
	(char*) &spr_foe7,
	gfx_lynx,
	0, 130,
};
static SCB_RENONE spr_foe5 = {
	BPP_4 | TYPE_NORMAL, 
	REUSEPAL, 
	NO_COLLIDE,
	(char*) &spr_foe6,
	gfx_lynx,
	0, 130,
};
static SCB_RENONE spr_foe4 = {
	BPP_4 | TYPE_NORMAL, 
	REUSEPAL, 
	NO_COLLIDE,
	(char*) &spr_foe5,
	gfx_lynx,
	0, 130,
};
static SCB_RENONE spr_foe3 = {
	BPP_4 | TYPE_NORMAL, 
	REUSEPAL, 
	NO_COLLIDE,
	(char*) &spr_foe4,
	gfx_lynx,
	0, 130,
};
static SCB_RENONE spr_foe2 = {
	BPP_4 | TYPE_NORMAL, 
	REUSEPAL, 
	NO_COLLIDE,
	(char*) &spr_foe3,
	gfx_lynx,
	0, 130,
};
static SCB_RENONE spr_foe1 = {
	BPP_4 | TYPE_NORMAL, 
	REUSEPAL, 
	NO_COLLIDE,
	(char*) &spr_foe2,
	gfx_lynx,
	0, 130,
};
static SCB_RENONE spr_foe0 = {
	BPP_4 | TYPE_NORMAL, 
	REUSEPAL, 
	NO_COLLIDE,
	(char*) &spr_foe1,
	gfx_lynx,
	0, 130,
};


//Background
extern unsigned char gfx_bg[];
SCB_REHV_PAL spr_bg = {
	BPP_3 | TYPE_BACKGROUND, //CAREFUL, the BG is in BPP3 mode instead of BPP4 as it contains only 5 differents colors in total (I guess this is SPR65 automatic optimisations...)
	REHV, 
	NO_COLLIDE,
	(char*) &spr_foe0,
	gfx_bg,
	0, 0,
	0x0100, 0x100,
	{ 0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF }
};

//The following sprites are not part of the main sprite chain (but used in title screen / game over screens for example)

//Message: "Press Buttons" (P1 & P2 - different buttons for each)
extern unsigned char gfx_press1[];
extern unsigned char gfx_press2[];
SCB_REHV_PAL spr_press1 = {
	BPP_4 | TYPE_NORMAL, 
	REHV, 
	NO_COLLIDE,
	0,
	gfx_press1,
	138, 50,
	0x0100, 0x100,
	{ 0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF }
};
SCB_REHV_PAL spr_press2 = {
	BPP_4 | TYPE_NORMAL | VFLIP | HFLIP, 
	REHV, 
	NO_COLLIDE,
	0,
	gfx_press2,
	21, 51,
	0x0100, 0x100,
	{ 0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF }
};

//Message: "You Win!" (P1 & P2)
extern unsigned char gfx_youwin[];
SCB_REHV_PAL spr_youwin1 = {
	BPP_4 | TYPE_NORMAL, 
	REHV, 
	NO_COLLIDE,
	0,
	gfx_youwin,
	127, 50,
	0x0100, 0x100,
	{ 0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF }
};
SCB_REHV_PAL spr_youwin2 = {
	BPP_4 | TYPE_NORMAL | VFLIP | HFLIP, 
	REHV, 
	NO_COLLIDE,
	0,
	gfx_youwin,
	32, 51,
	0x0100, 0x100,
	{ 0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF }
};

//Title Screen Logo
extern unsigned char gfx_titlelogo[];
SCB_REHV_PAL spr_titlelogo = {
	BPP_4 | TYPE_NORMAL, 
	REHV, 
	NO_COLLIDE,
	0,
	gfx_titlelogo,
	28, 2,
	0x0100, 0x100,
	{ 0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF }
};

//Game Over Logo
extern unsigned char gfx_gameover[];
SCB_REHV_PAL spr_gameover = {
	BPP_4 | TYPE_NORMAL, 
	REHV, 
	NO_COLLIDE,
	0,
	gfx_gameover,
	80, 51,
	0x0100, 0x100,
	{ 0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF }
};


//========= VARIABLES ===========

//Input
extern unsigned char checkInput(void);
static unsigned char joy;
extern unsigned char reset; 

//Sound
extern void lynx_snd_init ();
extern void lynx_snd_pause ();
extern void lynx_snd_continue ();
extern void __fastcall__ lynx_snd_play (unsigned char channel, unsigned char *music);
extern void lynx_snd_stop ();

//Get sfx data (must match all the sfx defined in tunes/sfxmusic.s)
typedef struct {
    unsigned char *shoot;
    unsigned char *gotcha;
    unsigned char *ting;
    unsigned char *tingting;
    unsigned char *bonus;
    unsigned char *sfx5;
    unsigned char *sfx6;
    unsigned char *sfx7;
} sfx_data;
extern sfx_data sfx;

//Get music data (must match all the musics defined in tunes/sfxmusic.s)
typedef struct {
    unsigned char *channel0;
    unsigned char *channel1;
    unsigned char *channel2;
    unsigned char *channel3;
} music_data;
extern music_data music;

//Fade in and Fade out function
extern void __fastcall__ fade_in(char *pal, unsigned char interval);
extern void __fastcall__ fade_out(unsigned char interval);
extern unsigned char mainPal[32];


//Text string array buffer to print variables on screen (20chars*8px=160px)
static char text[20];

//Looping variables
static unsigned char i, j;
static unsigned int k;

//FUNCTIONS
static void initGame();
static char spawnFoe();


//Ticks counter (frame counter + modulo computed each frame to trigger 4 "every X frame" counter, so we can move at speed slower than 60 pixel / seconds - 60fps game baby!)
//Frame counter (resetted every 240 frames (4 seconds), so it can be used for countdown too)
static unsigned char ticks;
//Fast speed : triggered every 2 frames
static unsigned char ticks_fast;
//Mid speed : triggered every 4 frames
static unsigned char ticks_mid;
//Slow speed : triggered every 8 frames
static unsigned char ticks_slow;
//Slower speed : triggered every 16 frames
static unsigned char ticks_slower;

//GAMEPLAY

//General variables
static unsigned char STATE; //Current game state (0: title / 1: gameplay / 2: game over)
static unsigned int gametime; //Current time left to play (in ticks)

//Press the buttons only once at a time
static unsigned char pressedP1;
static unsigned char pressedP2;

//Score
static unsigned int P1_score; //Current player score
static unsigned int P2_score; //Current player score

//Player 1
static signed int P1_posX; //X position of the fishing hook (int, as it'll receive 8.8 coordinate from cos function)
static signed int P1_posY; //Y position of the fishing hook (int, as it'll receive 8.8 coordinate from sin function)
static signed int P1_radius; //Current radius of the X/Y position of the hook
static signed int P1_angle; //Current angle of the hook
static signed char P1_speed; //Current moving speed of the hook
static unsigned char P1_catch; //Current id of the fish we catched (255: nothing is catch, and ready to shoot again)
static signed char P1_idle; //Current moving angular speed of the hook when it's idle (ready and waiting to be shot)

//Player 2
static signed int P2_posX; //X position of the fishing hook (int, as it'll receive 8.8 coordinate from cos function)
static signed int P2_posY; //Y position of the fishing hook (int, as it'll receive 8.8 coordinate from sin function)
static signed int P2_radius; //Current radius of the X/Y position of the hook
static signed int P2_angle; //Current angle of the hook
static signed char P2_speed; //Current moving speed of the hook
static unsigned char P2_catch; //Current id of the fish we catched (255: nothing is catch, and ready to shoot again)
static signed char P2_idle; //Current moving angular speed of the hook when it's idle (ready and waiting to be shot)

//Array containing the list of all the "digits" sprites usable on each score display digit
static unsigned char* digits[10];

//Foes (=> Fish to catch, i.e. the "Atari Machines")
typedef struct {
	unsigned char active; //1 if currently active in game
	signed char speedY; //Speed on Y axis
	unsigned char width; //(width / 2) of the sprite (used for collision detection)
	unsigned char height; //(height / 2) of the sprite (used for collision detection)
	unsigned char stuck; //0 = free swiwwing /  1 or 2 : index of the player that currently catched us
	SCB_RENONE *sprite; //pointer to our SCB tie sprite definition (to allow for easy change of appareance if needed)
} Foe;
//Array of foes
static Foe foes[8];
//Current foe pointer
static Foe* foe;


//======== MAIN LOOP ===========
void game() {
	
	//-= INIT GAME =-
	
	//Init random generator
	srand(clock());
	
	//Init gameplay variables
	ticks=1;
	P1_score=0;
	P2_score=0;
	
	//Set the state as Title Screen
	STATE=0;
	
	//Init the Foes
	for( i = 0 ; i < 8 ; i++ ){
		
		//Get current Foe
		foe=&foes[i];
		
		//Disable it
		foe->active=0;
		
		//Set its current SCB
		if( i == 0 ){ foe->sprite=&spr_foe0; }
		else if( i == 1 ){ foe->sprite=&spr_foe1; }
		else if( i == 2 ){ foe->sprite=&spr_foe2; }
		else if( i == 3 ){ foe->sprite=&spr_foe3; }
		else if( i == 4 ){ foe->sprite=&spr_foe4; }
		else if( i == 5 ){ foe->sprite=&spr_foe5; }
		else if( i == 6 ){ foe->sprite=&spr_foe6; }
		else if( i == 7 ){ foe->sprite=&spr_foe7; }
		
		//Hides offscreen
		foe->sprite->hpos=0;
		foe->sprite->vpos=130;
	}
	
	//Hide Player 1
	spr_hook1.hpos=0;
	spr_hook1.vpos=130;
	
	//Hide Player 2
	spr_hook2.hpos=0;
	spr_hook2.vpos=130;
	
	//Define the digits for score display
	digits[0]=gfx_0;
	digits[1]=gfx_1;
	digits[2]=gfx_2;
	digits[3]=gfx_3;
	digits[4]=gfx_4;
	digits[5]=gfx_5;
	digits[6]=gfx_6;
	digits[7]=gfx_7;
	digits[8]=gfx_8;
	digits[9]=gfx_9;

	
	//Update screen (so it can appear for the fade in)
	
	//Draw the background
	tgi_sprite(&spr_bg);
	//All the other sprites are also draw automatically, as they are chained to the BG
	
	//Draw Title Screen
	tgi_sprite(&spr_titlelogo);
	tgi_setcolor(3);
	tgi_outtextxy(42, 50, "a game for");
	tgi_setcolor(12);
	tgi_outtextxy(46, 62, "2 PLAYERS");
	tgi_outtextxy(46, 72, "ON 1 LYNX");
	tgi_setcolor(3);
	tgi_outtextxy(38, 90, "by Dr.LUDOS");
	
	//Force a screen render (ask it, then wait for it to be done before continuing the program)
	tgi_updatedisplay();
	while( tgi_busy() );
		
	//Do a fade in
	fade_in(mainPal, 100);
	
	//DISABLED AS I HAVE NO MUSIC MAKING SKILLS (re-enable it if you want you ears to bleed when you'll listen to the current music tune!)
	//Start the music on channel 0 (sfx will use channels 1, 2 and 3)
	//lynx_snd_play(0, music.channel0);
	
	//Never use while(1). Every loop needs to be halted by the reset event (in current case, the module will quit and goes back to intro module as defined by the resident program)
	while (!reset) { 
		
		//Wait for the VBL to have rendered new screen
		//=> tgi_busy == 0 if the VBL interupt have swapped the draw/display screen (double buffering)
		//=> tgi_update puts tgi_busy==1 to marks that the drawscreen/displayscreen "need to be swapped" when the next VBL interupts fire (this interupts will revert tgi_busy back to 0, making it a sage "waitForNextVBLANK" function)
		if (!tgi_busy()) {
			
			//========================
			//====== GAMEPLAY ========
			//========================
			if( STATE == 1 ){
			
				//=== FRAME & TICKS COUNTER ====
	
				//Increase the frame counter
				++ticks;
				
				//Reset it to zero every 4 seconds (unsigned char > 255 max, so 240 ticks => 4 seconds is ok)
				if (ticks == 241){
					ticks= 1;
				}
				
				//Update the "animation trigger" for everyone
				ticks_fast=!(ticks & 0x1);
				ticks_mid=!(ticks & 0x3);
				ticks_slow=!(ticks & 0x7);
				ticks_slower=!(ticks & 0xF);
				
				
				//-= PLAYER ACTIONS =-
				
				//Retrieve the current keys pressed
				//Instead of calling joy_read() directly, use checkInput() defined in resident.c . 
				//It will return the joy state, but will handle pause and reset events too
				//joy = joy_read(JOY_1);
				joy = checkInput();
	
				
				//-= PLAYER 1 =-
				
				//Press button?
				if( JOY_BTN_FIRE(joy) || JOY_BTN_FIRE2(joy) ){
					
					//Do only once per keypress
					if( pressedP1 == 0 ){
					
						//Shoot the hook if it's ready to shoot
						if( P1_catch == 255 ){
							
							//Launch the hook!
							P1_speed=8;
							
							//Set the hook as launched, without any fish catched currently
							P1_catch=200;
							
							//Play SFX
							lynx_snd_play(1, sfx.shoot);
						}
						
						//Do only once per keypress
						pressedP1=1;
					}
				}
				//Else, button released
				else {
					pressedP1=0;
				}
				
				//Moving hook
				//If the hook is currently launched
				if( P1_catch != 255 ){
					
					//Move the hook
					P1_radius += P1_speed;
					
					//Keep the radius under 127 to avoid issues
					if( P1_radius > 127 ){ 
						P1_radius=127;
					}
					
					//Get back to the player (oh, springy fish hook!), do it slower if we have catch something
					if( (ticks_mid && P1_catch == 200) || (ticks_slower && P1_catch != 200) ){
						--P1_speed;
					}
					
					//Stop it when it's arrived back to us
					if( P1_radius <= 15 ){
						
						//Reset back to idle position (and force radius back to initial value!)
						P1_radius=15;
						P1_speed=0;
						
						//If we have catch something
						if( P1_catch != 200 ){
							
							//Release the catched foe
							foe=&foes[P1_catch];
							
							//Disable it and puts it offscreen
							foe->active=0;
							foe->sprite->vpos=130;
					
							//Earn points (varies for each type)
							//Clock (extra time)
							if( foe->sprite->data == gfx_clock ){
								P1_score += 5;
								//The clock increase gametime too! (+5 sec)
								gametime += 300; 
								//Play time bonus sound
								lynx_snd_play(0, sfx.bonus);
								//Display score info too (machine name and points scored)
								spr_score1msg.data=gfx_clock_msg;
							}
							//Lynx
							else if( foe->sprite->data == gfx_lynx ){
								P1_score += 3;
								//Display score info too (machine name and points scored)
								spr_score1msg.data=gfx_lynx_msg;
							}
							//Jaguar
							else if( foe->sprite->data == gfx_jaguar ){
								P1_score += 2;
								//Display score info too (machine name and points scored)
								spr_score1msg.data=gfx_jaguar_msg;
							}
							//Atari 400
							else if( foe->sprite->data == gfx_atari400 ){
								P1_score += 1;
								//Display score info too (machine name and points scored)
								spr_score1msg.data=gfx_atari400_msg;
							}
							//Atari 800
							else if( foe->sprite->data == gfx_atari800 ){
								P1_score += 2;
								//Display score info too (machine name and points scored)
								spr_score1msg.data=gfx_atari800_msg;
							}
							//Atari 2600
							else if( foe->sprite->data == gfx_atari2600 ){
								P1_score += 1;
								//Display score info too (machine name and points scored)
								spr_score1msg.data=gfx_atari2600_msg;
							}
							//Atari ST
							else if( foe->sprite->data == gfx_atarist ){
								P1_score += 2;
								//Display score info too (machine name and points scored)
								spr_score1msg.data=gfx_atarist_msg;
							}
								
							//Update score display
							//We separate each digit in a very dirty way to display it directly using the "digits" sprite list
							k=P1_score;
							spr_score1A.data=digits[k/1000];
							k-=(k/1000)*1000;
							spr_score1B.data=digits[k/100];
							k-=(k/100)*100;
							spr_score1C.data=digits[k/10];
							k-=(k/10)*10;
							spr_score1D.data=digits[k];
							
							//Play SFX
							lynx_snd_play(3, sfx.gotcha);
						}
	
						//Set player in the "ready to shoot" state
						P1_catch=255;
					}
					
				}
				//Else, we are waiting to be shot
				else {
					//Moves the angle and the idle angular speed
					P1_angle += P1_idle;
					
					//Moves back and forth when the angular limits are reached
					if( P1_angle < 140 ){
						P1_angle=140;
						//Reverse the angular speed
						P1_idle=1;
					}
					//Moves back and forth when the angular limits are reached
					else if( P1_angle > 220 ){
						P1_angle=220;
						//Reverse the angular speed
						P1_idle=-1;
					}		
				}
				
				//Update the Hook position using angle calculation (cc65 has cos and sin function using a LUT with 8.8 fixed values)
				P1_posX=(cc65_cos(P1_angle)*P1_radius) >> 8;
				P1_posY=(cc65_sin(P1_angle)*P1_radius) >> 8;
	
				//Update the sprite position with the new values
				spr_hook1.hpos=160+P1_posX;
				spr_hook1.vpos=51+P1_posY;
				
				
				//-= PLAYER 2 =-
				
				//Press button?
				if( JOY_BTN_LEFT(joy) || JOY_BTN_RIGHT(joy) || JOY_BTN_UP(joy) || JOY_BTN_DOWN(joy) ){
					
					//Do only once per keypress
					if( pressedP2 == 0 ){
					
						//Shoot the hook if it's ready to shoot
						if( P2_catch == 255 ){
							
							//Launch the hook!
							P2_speed=8;
							
							//Set the hook as launched, without any fish catched currently
							P2_catch=200;
							
							//Play SFX
							lynx_snd_play(2, sfx.shoot);
						}
						
						//Do only once per keypress
						pressedP2=1;
					}
				}
				//Else, button released
				else {
					pressedP2=0;
				}
				
				//Moving hook
				//If the hook is currently launched
				if( P2_catch != 255 ){
					
					//Move the hook
					P2_radius += P2_speed;
					
					//Keep the radius under 127 to avoid issues
					if( P2_radius > 127 ){ 
						P2_radius=127;
					}
					
					//Get back to the player (oh, springy fish hook!), do it slower if we have catched something
					if( (ticks_mid && P2_catch == 200) || (ticks_slower && P2_catch != 200) ){
						--P2_speed;
					}
					
					//Stop it when it's arrived back to us
					if( P2_radius <= 15 ){
						
						//Reset back to idle position (and force radius back to initial value!)
						P2_radius=15;
						P2_speed=0;
						
						//If we have catched something
						if( P2_catch != 200 ){
							
							//Release the catched foe
							foe=&foes[P2_catch];
							
							//Disable it and puts it offscreen
							foe->active=0;
							foe->sprite->vpos=130;
					
							//Earn points (varies for each type)
							//Clock (extra time)
							if( foe->sprite->data == gfx_clock ){
								P2_score += 5;
								//The clock increase gametime too! (+10 sec)
								gametime += 300; 
								//Play time bonus sound
								lynx_snd_play(0, sfx.bonus);
								//Display score info too (machine name and points scored)
								spr_score2msg.data=gfx_clock_msg;
							}
							//Lynx
							else if( foe->sprite->data == gfx_lynx ){
								P2_score += 3;
								//Display score info too (machine name and points scored)
								spr_score2msg.data=gfx_lynx_msg;
							}
							//Jaguar
							else if( foe->sprite->data == gfx_jaguar ){
								P2_score += 2;
								//Display score info too (machine name and points scored)
								spr_score2msg.data=gfx_jaguar_msg;
							}
							//Atari 400
							else if( foe->sprite->data == gfx_atari400 ){
								P2_score += 1;
								//Display score info too (machine name and points scored)
								spr_score2msg.data=gfx_atari400_msg;
							}
							//Atari 800
							else if( foe->sprite->data == gfx_atari800 ){
								P2_score += 2;
								//Display score info too (machine name and points scored)
								spr_score2msg.data=gfx_atari800_msg;
							}
							//Atari 2600
							else if( foe->sprite->data == gfx_atari2600 ){
								P2_score += 1;
								//Display score info too (machine name and points scored)
								spr_score2msg.data=gfx_atari2600_msg;
							}
							//Atari ST
							else if( foe->sprite->data == gfx_atarist ){
								P2_score += 2;
								//Display score info too (machine name and points scored)
								spr_score2msg.data=gfx_atarist_msg;
							}
								
							//Update score display
							//We separate each digit in a very dirty way to display it directly using the "digits" sprite list
							k=P2_score;
							spr_score2A.data=digits[k/1000];
							k-=(k/1000)*1000;
							spr_score2B.data=digits[k/100];
							k-=(k/100)*100;
							spr_score2C.data=digits[k/10];
							k-=(k/10)*10;
							spr_score2D.data=digits[k];
							
							//Play SFX
							lynx_snd_play(3, sfx.gotcha);
						}
	
						//Set player in the "ready to shoot" state
						P2_catch=255;
					}
					
				}
				//Else, we are waiting to be shot
				else {
					//Moves the angle and the idle angular speed
					P2_angle += P2_idle;
					
					//Roll back or over 360
					if( P2_angle <= 0 ){ P2_angle += 360; }
					if( P2_angle >= 360){ P2_angle -= 360; }
					
					//Moves back and forth when the angular limits are reached
					if( P2_angle < 320 && P2_angle > 180 ){
						P2_angle=320;
						//Reverse the angular speed
						P2_idle=1;
					}
					//Moves back and forth when the angular limits are reached
					else if( P2_angle > 40 && P2_angle < 180){
						P2_angle=40;
						//Reverse the angular speed
						P2_idle=-1;
					}		
				}
				
				//Update the Hook position using angle calculation (cc65 has cos and sin function using a LUT with 8.8 fixed values)
				P2_posX=(cc65_cos(P2_angle)*P2_radius) >> 8;
				P2_posY=(cc65_sin(P2_angle)*P2_radius) >> 8;
	
				//Update the sprite position with the new values
				spr_hook2.hpos=P2_posX;
				spr_hook2.vpos=51+P2_posY;
				
				
				
				
				//-= FOES =-
				//Spawn new foes regularly, with different rates according to remaining game time (gametime is in ticks, so divide it per 60 to have the time in seconds)
				if ( gametime <= 1200 ){
					if( ticks % 20 == 0 ){
						spawnFoe();
					}
				}
				else if ( gametime > 1500 && gametime <= 1860 ){
					if( ticks % 120 == 0 ){
						spawnFoe();
					}
				}
				else if ( gametime <= 2700 ){
					if( ticks % 60 == 0 ){
						spawnFoe();
					}
				}
				else {
					if( ticks % 80 == 0 ){
						spawnFoe();
					}
				}
	
				
				//For each FOE
				for( i = 0 ; i < 8 ; i++ ){
					
					//Get current Foe
					foe=&foes[i];
			
					//If the foe is active
					if( foe->active == 1 ){
						
						//If it's stuck to Player 1
						if( foe->stuck == 1 ){
							
							//Position our sprite over Player 1
							foe->sprite->hpos = spr_hook1.hpos;
							foe->sprite->vpos = spr_hook1.vpos;
						}
						//If it's stuck to Player 2
						else if( foe->stuck == 2 ){
							
							//Position our sprite over Player 2
							foe->sprite->hpos = spr_hook2.hpos;
							foe->sprite->vpos = spr_hook2.vpos;
						}
						//If it's Swimming freely
						else if( foe->stuck == 0 ){
							
							//Move the foe forward according to its speed
							//If the speed is negative, we apply it every second
							if( foe->speedY <= 0 ){
								foe->sprite->vpos += foe->speedY;
							}
							//Else we do a modulo on the positive speed to move not every frame
							else if( (ticks % foe->speedY) == 0 ){
								--foe->sprite->vpos;
							}
							
							//If Player 1 hasn't catch anything yet
							if( P1_catch == 200 && P1_radius < 127 && P1_speed > 0 ){
							
								//If the foe hits player 1
								//OPTIMIZATION: Manual "inlining" of collision function to save CPU time
								//if ((x1 < (x2+w2)) && ((x1+w1) > x2) && (y1 < (h2+y2)) && ((y1+h1) > y2))
								if( (spr_hook1.hpos < (foe->sprite->hpos + foe->width)) && (spr_hook1.hpos > (foe->sprite->hpos - foe->width)) && (spr_hook1.vpos < (foe->sprite->vpos + foe->height)) && (spr_hook1.vpos > (foe->sprite->vpos - foe->height)) ){
								
									//Get stuck to player 1
									foe->stuck=1;
								
									//Notify Player 1 of the catch too
									P1_catch=i;
								
									//And stop P1 speed so it bounce back quicker
									P1_speed=0;
								}
							}
							
							//If Player 2 hasn't catch anything yet (and that P1 didn't get the foe JUST NOW!)
							if( P2_catch == 200 && P2_radius < 127 && P2_speed > 0 && foe->stuck == 0 ){
							
								//If the foe hits player 2
								//OPTIMIZATION: Manual "inlining" of collision function to save CPU time
								//if ((x1 < (x2+w2)) && ((x1+w1) > x2) && (y1 < (h2+y2)) && ((y1+h1) > y2))
								if( (spr_hook2.hpos < (foe->sprite->hpos+foe->width)) && (spr_hook2.hpos > (foe->sprite->hpos-foe->width)) && (spr_hook2.vpos < (foe->sprite->vpos+foe->height)) && (spr_hook2.vpos > (foe->sprite->vpos-foe->height)) ){
								
									//Get stuck to player 2
									foe->stuck=2;
								
									//Notify Player 2 of the catch too
									P2_catch=i;
								
									//And stop P2 speed so it bounce back quicker
									P2_speed=0;
								}
							}
							
							//If it goes outside of screen, kill it
							if( foe->sprite->vpos < -20 ){
								
								//Disable it and puts it offscreen
								foe->active=0;
								foe->sprite->vpos=130;
							}
							
						}
						
					}
				}
				
				
				
				//-= GAME OVER =-
				
				//Decrease the time left before game over
				if( gametime > 0 ){
					--gametime;
					
					//Play a sound for the last few seconds
					if( gametime == 60 || gametime == 120 || gametime == 180 || gametime == 240 || gametime == 300 ){
						//Play SFX
						lynx_snd_play(0, sfx.ting);
					}
				}
				//If the gametime is out, go to the game over state
				else {
					//Set the state as Game Over
					STATE=2;
					
					//Clear and hide the player sprites
					spr_hook1.hpos=0;
					spr_hook1.vpos=130;
					spr_hook2.hpos=0;
					spr_hook2.vpos=130;
					
					//Clear the score info message too
					spr_score1msg.data=gfx_empty;
					spr_score2msg.data=gfx_empty;
					
					//Activate a "cool down" timer before we can start again
					gametime=(3*60)+30;
					
					//Reset ticks for smooth animation
					ticks=0;
					
					//Play SFX
					lynx_snd_play(0, sfx.tingting);
				}
				
				
				
				//-= RENDER SCREEN =-
				
				//Draw the background
				tgi_sprite(&spr_bg);
				//All the other sprites are also draw automatically, as they are chained to the BG
				
				tgi_setcolor(15);
				//Draw a line for the P1 Hook
				tgi_line(160+P1_posX, 51+P1_posY, 160, 51);
				//Draw a line for the P2 Hook
				tgi_line(P2_posX, 51+P2_posY, 0, 51);
				
				//Draw timer (if we are not if game over state already)
				if( STATE == 1 ){
					tgi_setcolor(3);
					itoa((gametime/60)+1, text, 10);
					tgi_outtextxy(72, 0, text);
				}
				
				//DEBUG: display hitboxes (manually change the foe(s) indexes you'll want to see - beware, these drawing function are very slow as they aren't part of the main sprite chain)
				/*foe=&foes[0];
				tgi_line(foe->sprite->hpos - foe->width,  foe->sprite->vpos - foe->height, foe->sprite->hpos + foe->width, foe->sprite->vpos - foe->height);
				tgi_line(foe->sprite->hpos + foe->width,  foe->sprite->vpos - foe->height, foe->sprite->hpos + foe->width, foe->sprite->vpos + foe->height);
				tgi_line(foe->sprite->hpos + foe->width,  foe->sprite->vpos + foe->height, foe->sprite->hpos - foe->width, foe->sprite->vpos + foe->height);
				tgi_line(foe->sprite->hpos - foe->width,  foe->sprite->vpos + foe->height, foe->sprite->hpos - foe->width, foe->sprite->vpos - foe->height);
				foe=&foes[2];
				tgi_line(foe->sprite->hpos - foe->width,  foe->sprite->vpos - foe->height, foe->sprite->hpos + foe->width, foe->sprite->vpos - foe->height);
				tgi_line(foe->sprite->hpos + foe->width,  foe->sprite->vpos - foe->height, foe->sprite->hpos + foe->width, foe->sprite->vpos + foe->height);
				tgi_line(foe->sprite->hpos + foe->width,  foe->sprite->vpos + foe->height, foe->sprite->hpos - foe->width, foe->sprite->vpos + foe->height);
				tgi_line(foe->sprite->hpos - foe->width,  foe->sprite->vpos + foe->height, foe->sprite->hpos - foe->width, foe->sprite->vpos - foe->height);
				*/
			}
			
			//========================
			//===== TITLE SCREEN =======
			//========================
			else if( STATE == 0 ){
				
				//-= PLAYER ACTIONS =-
				
				//Retrieve the current keys pressed
				//Instead of calling joy_read() directly, use checkInput() defined in resident.c . 
				//It will return the joy state, but will handle pause and reset events too
				//joy = joy_read(JOY_1);
				joy = checkInput();
				
				//If any key is pressed
				if( JOY_BTN_LEFT(joy) || JOY_BTN_RIGHT(joy) || JOY_BTN_UP(joy) || JOY_BTN_DOWN(joy) || JOY_BTN_FIRE(joy) || JOY_BTN_FIRE2(joy) ){
					
					//Initialize the game
					initGame();
					
					//Mark both players buttons as pressed to avoid unintentional action at start
					pressedP1=1;
					pressedP2=1;
					
					//Set the gameplay state
					STATE=1;
				}
				
				//-= RENDER SCREEN =-
				
				//BLINKING ANIMATION
				//Increase the frame counter
				++ticks;
				//Reset it to zero every 1 seconds
				if (ticks == 61){	ticks= 1; }
				
				//Draw the background
				tgi_sprite(&spr_bg);
				//All the other sprites are also draw automatically, as they are chained to the BG
				
				//Draw Title Screen
				tgi_sprite(&spr_titlelogo);
				tgi_setcolor(3);
				tgi_outtextxy(42, 50, "a game for");
				tgi_setcolor(12);
				tgi_outtextxy(46, 62, "2 PLAYERS");
				tgi_outtextxy(46, 72, "ON 1 LYNX");
				tgi_setcolor(3);
				tgi_outtextxy(38, 90, "by Dr.LUDOS");
				
				//Draw the "Press Buttons" messages sprites for both players
				//And makes them blink!
				if( ticks < 31 ){
					tgi_sprite(&spr_press1);
					tgi_sprite(&spr_press2);
				}
			}
			
			//=======================
			//===== GAME OVER =======
			//=======================
			else if( STATE == 2 ){
				
				//-= PLAYER ACTIONS =-
				
				//Retrieve the current keys pressed
				//Instead of calling joy_read() directly, use checkInput() defined in resident.c . 
				//It will return the joy state, but will handle pause and reset events too
				//joy = joy_read(JOY_1);
				joy = checkInput();
				
				//If any key is pressed
				if( JOY_BTN_LEFT(joy) || JOY_BTN_RIGHT(joy) || JOY_BTN_UP(joy) || JOY_BTN_DOWN(joy) || JOY_BTN_FIRE(joy) || JOY_BTN_FIRE2(joy) ){
					
					//Only if the cooldown timer is over
					if( gametime == 0 ){
					
						//Initialize the game
						initGame();
						
						//Mark both players buttons as pressed to avoid unintentional action at start
						pressedP1=1;
						pressedP2=1;
						
						//Set the gameplay state
						STATE=1;
					}
				}
				
				//Decrease the cooldown timer (to avoid erroneous restart, set in the gameplay sequence)
				if( gametime > 0 ){
					--gametime;
				}
				
				
				//-= RENDER SCREEN =-
				
				//BLINKING ANIMATION
				//Increase the frame counter
				++ticks;
				//Reset it to zero every 3 seconds
				if (ticks == 181){ ticks= 1; }
				
				//Draw the background
				tgi_sprite(&spr_bg);
				//All the other sprites are also draw automatically, as they are chained to the BG
				
				//Draw Game Over screen
				tgi_setcolor(3);
				tgi_outtextxy(44, 0, "TIME OUT!");
				
				//Wait a few frames before displaying the game over message
				if( gametime <= 150 ){

					//Reset ticks count to have a smooth appearing animation
					if( gametime == 150 ){ ticks=1; }
					
					//Animate it!
					//Grow (game)
					if( ticks < 31 ){
						spr_gameover.sprctl0=BPP_4 | TYPE_NORMAL | VFLIP | HFLIP;
						spr_gameover.hsize=ticks*8;
					}
					//Stay awhile (game)
					else if( ticks < 61 ){
						spr_gameover.sprctl0=BPP_4 | TYPE_NORMAL | VFLIP | HFLIP;
						spr_gameover.hsize=0x100;
					}
					//Shrink (game)
					else if( ticks < 91 ){
						spr_gameover.sprctl0=BPP_4 | TYPE_NORMAL | VFLIP | HFLIP;
						spr_gameover.hsize=(90-ticks)*8;
					}
					//Grow (over)
					else if( ticks < 121 ){
						spr_gameover.sprctl0=BPP_4 | TYPE_NORMAL; 
						spr_gameover.hsize=(ticks-90)*8;
					}
					//Stay awhile (over)
					else if( ticks < 151 ){
						spr_gameover.sprctl0=BPP_4 | TYPE_NORMAL;
						spr_gameover.hsize=0x100;
					}
					//Shrink (over)
					else {
						spr_gameover.sprctl0=BPP_4 | TYPE_NORMAL;
						spr_gameover.hsize=(181-ticks)*8;
					}
					
					//Display sprite
					tgi_sprite(&spr_gameover);
				}
				
				//Wait a few frames before displaying the winning message
				if( gametime <= 60 ){
				
					//Play SFX (only once!)
					if( gametime == 60 ){
						lynx_snd_play(3, sfx.bonus);
					}
					
					//P1 wins
					if( P1_score > P2_score ){
						//Draw the "You Win" message sprite for this player
						tgi_sprite(&spr_youwin1);
					}
					//P2 wins
					else if( P1_score < P2_score ){
						//Draw the "You Win" message sprite for this player
						tgi_sprite(&spr_youwin2);
					}
					//Draw game
					else {
						//Draw the "You Win" message sprite for both players
						tgi_sprite(&spr_youwin1);
						tgi_sprite(&spr_youwin2);
					}
				}
				
				//Draw the "Press Buttons" messages sprites for both players if they can restart
				if( gametime == 0 ){
					//Makes them blink!
					if( (ticks % 60) < 31 ){
						tgi_sprite(&spr_press1);
						tgi_sprite(&spr_press2);
					}				
				}
			}
			
			
			//-= COMMON BACKEND =-
			
			//Display new rendered screen by swapping it with the draw buffer (this function just asks for it, the actual swap will be performed by the VBL interupt)
			tgi_updatedisplay();
		}
	}
	
	
	//If the game is resetted (soft reset : Pause + Opt1, have to be handled manually by us!)
	if( reset == 1 ){
		//The main game loop will break automatically, so we do some cleanup / manual reset before exiting this module

		//Disable reset variable (it served its purpose by breaking the game loop already)
		reset=0;
		
		//The fade out and back to menu will be handeld by the main loop in resident.c
	}
}


//======== FUNCTIONS  ===========

//This functions init the game, so it can be replayed instantly without reloading the module :)
void initGame() {

	//Init gameplay variables
	ticks=1;
	gametime=60*60; //60 seconds for each game duration
	
	//Init the Foes
	for( i = 0 ; i < 8 ; i++ ){
		
		//Get current Foe
		foe=&foes[i];
		
		//Disable it
		foe->active=0;
		
		//Set its current SCB
		if( i == 0 ){ foe->sprite=&spr_foe0; }
		else if( i == 1 ){ foe->sprite=&spr_foe1; }
		else if( i == 2 ){ foe->sprite=&spr_foe2; }
		else if( i == 3 ){ foe->sprite=&spr_foe3; }
		else if( i == 4 ){ foe->sprite=&spr_foe4; }
		else if( i == 5 ){ foe->sprite=&spr_foe5; }
		else if( i == 6 ){ foe->sprite=&spr_foe6; }
		else if( i == 7 ){ foe->sprite=&spr_foe7; }
		
		//Hides offscreen
		foe->sprite->hpos=0;
		foe->sprite->vpos=130;
	}
	
	//Player 1
	P1_posX=150;
	P1_posY=51;
	P1_radius=10;
	P1_angle=180;
	P1_speed=0;
	P1_catch=255;
	P1_idle=-1;
	P1_score=0;
	
	//Update score display
	//We separate each digit in a very dirty way to display it directly using the "digits" sprite list
	k=P1_score;
	spr_score1A.data=digits[k/1000];
	k-=(k/1000)*1000;
	spr_score1B.data=digits[k/100];
	k-=(k/100)*100;
	spr_score1C.data=digits[k/10];
	k-=(k/10)*10;
	spr_score1D.data=digits[k];
	
	//Clear the score info message too
	spr_score1msg.data=gfx_empty;
	
	
	//Player 2
	P2_posX=100;
	P2_posY=51;
	P2_radius=10;
	P2_angle=0;
	P2_speed=0;
	P2_catch=255;
	P2_idle=1;
	P2_score=0;
	
	//Update score display
	//We separate each digit in a very dirty way to display it directly using the "digits" sprite list
	k=P2_score;
	spr_score2A.data=digits[k/1000];
	k-=(k/1000)*1000;
	spr_score2B.data=digits[k/100];
	k-=(k/100)*100;
	spr_score2C.data=digits[k/10];
	k-=(k/10)*10;
	spr_score2D.data=digits[k];	
	
	//Clear the score info message too
	spr_score2msg.data=gfx_empty;
	
	
	//Launch a foe immediatedly to start game
	spawnFoe();
}


//This functions tries to spawn a new Foe if there are at least one available
//Return the id of the Foe spawned, or 255 if nothing is spawned
char spawnFoe() {
	
	//To check spawned worked or not
	char spawned=255;
	
	//Temp var to randomize foe
	char spawnType;
	
	//For each Foe
	for( j = 0 ; j < 8 ; j++ ){
		
		//If we haven't spawned a Foe yet
		if( spawned == 255 ){
		
			//Get current Foe
			foe=&foes[j];
	
			//If the foe is not active
			if( foe->active == 0 ){
				
				//Mark it as active
				foe->active = 1;
				
				//Make it swim freely
				foe->stuck = 0;
				
				//Position it randomly
				foe->sprite->hpos = 35+(rand()%90);
				foe->sprite->vpos = 130;
				
				//Set its type (speed / appareance) at random
				spawnType=rand()%10;
				
				//Lynx
				if( spawnType < 4 ){
					//Set its speed
					foe->speedY = 2;
					//Change it's appareance
					foe->sprite->data = gfx_lynx;
					foe->width=6;
					foe->height=10;
				}
				//Atari 400
				else if( spawnType == 4 ){
					//Set its speed
					foe->speedY = 6;
					//Change it's appareance
					foe->sprite->data = gfx_atari400;
					foe->width=10;
					foe->height=9;
				}
				//Atari 800XL
				else if( spawnType == 5 ){
					//Set its speed
					foe->speedY = 4;
					//Change it's appareance
					foe->sprite->data = gfx_atari800;
					foe->width=10;
					foe->height=7;
				}
				//Atari 2600
				else if( spawnType == 6 ){
					//Set its speed
					foe->speedY = 6;
					//Change it's appareance
					foe->sprite->data = gfx_atari2600;
					foe->width=10;
					foe->height=8;
				}
				//Atari ST
				else if( spawnType == 7 ){
					//Set its speed
					foe->speedY = 4;
					//Change it's appareance
					foe->sprite->data = gfx_atarist;
					foe->width=10;
					foe->height=8;
				}
				//Clock
				else if( spawnType == 8 ){
					//Set its speed
					foe->speedY = -1;
					//Change it's appareance
					foe->sprite->data = gfx_clock;
					foe->width=8;
					foe->height=8;
				}
				//Jaguar
				else {
					//Set its speed
					foe->speedY = 4;
					//Change it's appareance
					foe->sprite->data = gfx_jaguar;
					foe->width=10;
					foe->height=10;
				}
				
				//Tell that we have spawned this one
				spawned=j;
			}
		}
	}
	
	//Tell whether we spawned a Tie or not
	return spawned;
}
