/*
 * The code and data defined here is placed in resident RAM
 */
#include <lynx.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <tgi.h>
#include <joystick.h>
#include <6502.h>
#include "LynxSD.h"

#define EEPROM_MAGIC "FISH" // Change this to your own string. 3-5 chars are fine.

signed char SDCheck = -1; // tracks if there is a SD cart with saves enabled. -1 = No check yet, 0 = no SD saves, 1 = SD saves possible 
const char SDSavePath[] = "/saves/fishing.sav"; //rename game.sav to yourprogramname.sav, i.e. the same name of your lnx file
unsigned int saveBuf[64]; // Buffer for storing SaveData in memory. Only the first 6 bytes are used for checking if data is valid, for the others there is no predefined data structure. Use it as you want

unsigned char halted = 0; // for pause event
unsigned char reset = 0; // for reset event

// For reading writing EEPROM. The data is always 2 bytes, the address range 0..63
extern int __fastcall__ lnx_eeprom_read(unsigned char pos);
extern void __fastcall__ lnx_eeprom_write(unsigned char pos, int val);

//Sound
void lynx_snd_init ();
void lynx_snd_pause ();
void lynx_snd_continue ();
void __fastcall__ lynx_snd_play (unsigned char channel, unsigned char *music);
void lynx_snd_stop ();


//Game modules functions
extern int intro();
extern int game();

extern int INTRO_FILENR;
extern int GAME_FILENR;
extern int TUNE0_FILENR;

unsigned char backupPal[32]; // stores the palette before graying the screen in pause mode
unsigned char grayPal[32]; // stores the grayed palette for the pause mode derived from the active palette

//Fade in and Fade out function
extern void __fastcall__ fade_in(char *pal, unsigned char interval);
extern void __fastcall__ fade_out(unsigned char interval);

//General game palette
unsigned char mainPal[32] = {
	// green
	0x00ff >> 8, 
	0x0000 >> 8, 
	0x09e0 >> 8, 
	0x0ef3 >> 8, 
	0x0642 >> 8, 
	0x0844 >> 8, 
	0x0c56 >> 8, 
	0x0aaa >> 8, 
	0x0888 >> 8, 
	0x0444 >> 8, 
	0x0ddd >> 8, 
	0x0cbd >> 8, 
	0x0f0f >> 8, 
	0x022f >> 8, 
	0x080f >> 8, 
	0x0fff >> 8, 

	// blue + red
	0x00ff & 0xff, 
	0x0000 & 0xff, 
	0x09e0 & 0xff, 
	0x0ef3 & 0xff, 
	0x0642 & 0xff, 
	0x0844 & 0xff, 
	0x0c56 & 0xff, 
	0x0aaa & 0xff, 
	0x0888 & 0xff, 
	0x0444 & 0xff, 
	0x0ddd & 0xff, 
	0x0cbd & 0xff, 
	0x0f0f & 0xff,
	0x022f & 0xff, 
	0x080f & 0xff, 
	0x0fff & 0xff, 
};



//EEPROM & SD card load/save functions (thanks Karri & Nop90 for your template!)
void eewrite(unsigned char pos, unsigned int val)
{
	unsigned int check;
	check = lnx_eeprom_read(pos);
	if(check!=val)
		lnx_eeprom_write(pos,val);
}

void writeSaveData(void)
{
	unsigned char i;

	if(SDCheck==1) // use SD
	{
		if (LynxSD_OpenFile(SDSavePath) == FR_OK) {
			LynxSD_WriteFile((void *)saveBuf, 128);
			LynxSD_CloseFile();
		}	
	}
	else if(SDCheck==0) // Try to use EEPROM
	{
		for(i=0;i<64;i++)
		{
			eewrite(i,saveBuf[i]);
		}
	}
}

void readSaveData(void)
{
	unsigned char i;
	FRESULT res;

	if(SDCheck==-1) // SD not tested yet
	{
		res = LynxSD_OpenFileTimeout(SDSavePath); 
		if(res == FR_OK)
			SDCheck=2; // SD savefile is ok and next SD access doesn't need to open the file (already opened)
  		else if(res==FR_DISK_ERR) // SD read Timeout -> NO SD
		{
			SDCheck=0; // Try to use EEPROM, if there isn't an EEPROM the functions calls are safe because they don't freeze the code 
		}
		else	
		{
			SDCheck=-2; // The SD answers, but there is no sav file, or some other problem occoured accessing the file -> don't try to use SD and EEPROM anymore (EEPROM calls on SD frezes the code)
		}
	}

	if (SDCheck==0)	// Read the EEPROM	
	{ 
		for(i=0;i<64;i++)
		{
			saveBuf[i]= lnx_eeprom_read(i);
		}
	}
	else if(SDCheck>0) // the value can be 1 or 2
	{
		if(SDCheck==1) //SD sav file enabled
		{
			if (LynxSD_OpenFile(SDSavePath) == FR_OK)
			{
				LynxSD_ReadFile((void *)saveBuf, 128);
				LynxSD_CloseFile();	
			}
		}
		else // //SD sav file enabled and sav file already opened
		{
			SDCheck=1;
			LynxSD_ReadFile((void *)saveBuf, 128);
			LynxSD_CloseFile();	
		}
	}
}

//This function resets the Save Data (write 0 in every byte of the EEPROM / SD card save buffer)
void resetSaveData(void) {
	int i;
	
	//Set 0 to the 4th byte of the save buffer
	saveBuf[3]=0;
	
	//Copy the "magic word" used to identify the save / check that the EEPROM do contains data related to our game
	strcpy((char*)saveBuf, EEPROM_MAGIC);
	
	//And then clear all the remaining slot with zeroes (these are the slots containing the actual save data)
	for( i=14 ; i < 64 ; i++){
		saveBuf[i]=0; //instead of this you could initialize the savedata with some starting values, like a predefined highscores table.
	}
	
	//And write the save buffer to the EEPROM!
	writeSaveData();
}



//-= MAIN PROGRAM LOOP =-
int main(void) {
	
	//Init the lynx screen, sound and overall hardware (thanks CC65!)
	tgi_install(&tgi_static_stddrv);
	tgi_init();
	joy_install(&joy_static_stddrv);
	lynx_snd_init();
	CLI();
	
	//Do an instataneous fadeout to hide the "startup" screen glitches
	fade_out(0);
	
	//Change frame rate
	tgi_setframerate(60);
	//tgi_ioctl(3, 60);
	
	//Disable collision detection => I DISABLED THIS INSTRUCTION AS IT MAKES RETROARCH HANDY FREEZE!
	//tgi_setcollisiondetection(0);

	//Clear the screen first (to remove TGI start up garbage onscreen)
	tgi_clear();
	
	//Init save file
	/*readSaveData();
	if( strcmp( (char*)saveBuf, EEPROM_MAGIC ) != 0){ 
		resetSaveData();
	}
	*/
	
	//Load Sound
	lynx_load((int)&TUNE0_FILENR);
	
	//Init pause / reset vars
	reset = 0;
	halted = 0;
	
	//Run the core loop infinitely to go back and forth between the various game "states"
	while (1) {
		
		//DISABLED: this game didn't need an intro, but you should re-enable it for a larger project. And of course, you can create as many "module" that you want and load them with the lynx_load function
		//Load the introduction screen and run it (it'll fade in automatically)
		//lynx_load((int)&INTRO_FILENR);
		//intro();
		
		//Then, when the menu is finished, do a fadeout
		//fade_out(100);
		
		//Clear the screen
		//tgi_clear();
		
		//load the game and run it (it'll fade in automatically)
		lynx_load((int)&GAME_FILENR);
		game();
		
		//Then, when the game is finished, do a fadeout
		fade_out(200);
		
		//Clear the screen
		tgi_clear();
	}
	return 0;
}

//Function that handles all the input, including special buttons (thanks Karri & Nop90 for the template!)
unsigned char checkInput(void)
{
	unsigned int col;
	unsigned char i;
	const unsigned char* pal;
	 
	if(!reset)
	do
	{
		if (kbhit()) 
		{
			switch (cgetc()) 
			{
			case 'F':
				tgi_flip();
				break;
			case 'P':
				if(halted)
				{
					halted = 0;
					tgi_setpalette(backupPal); // restore normal palette. No need to wait vsync; data in the screenbuffer are valid
					lynx_snd_continue ();					
				}
				else
				{
					pal = tgi_getpalette(); // let's backup the palette
					for (i=0;i<	16;i++) 
					// A simple grayed palette is obtained setting the new r,g, b to the g*0,5 + r*0,25 + b*0,25 . 
					// This is an approximate formula that gives a good result considering that there are only 16 shades of gray.					
					{
						backupPal[i] = pal[i];
 						backupPal[i+16] = pal[i+16];
						col = ((backupPal[i]&0xf)*2 + (backupPal[i+16]&0xf) + (backupPal[i+16]>>4)) >> 2;
						grayPal[i] = col;
						grayPal[i+16] = col | (col<<4);
					}	
					tgi_setpalette(grayPal); // set gray palette
					
					//Write a "pause" message too
					//First draw a black border (display same text four times with 1 pixel offset from actual text)
					tgi_setcolor(1);
					tgi_outtextxy(55, 47, "PAUSED");
					tgi_outtextxy(57, 47, "PAUSED");
					tgi_outtextxy(56, 46, "PAUSED");
					tgi_outtextxy(56, 48, "PAUSED");
					//Then display the real message on top in white color
					tgi_setcolor(15);
					tgi_outtextxy(56, 47, "PAUSED");
					//Update the screen (flip drawbuffer) so we can actually see it onscreen :)
					tgi_updatedisplay();
					
					//Mark the game down as halted (execution will be suspended)
					halted = 1;
					
					//Pause sound too
					lynx_snd_pause ();
				}
				break;
			case 'R':
				if(halted)
				{
					halted = 0;
					tgi_setpalette(backupPal); //restore normal palette
					lynx_snd_continue ();					
				}
				lynx_snd_stop();
				reset=1;
				break;
  
			case '1': 
				break;
			case '2':
				break;

			case '3':// used to clear saves on eeprom pressing Opt1 + Opt2 while game is paused
				if(halted)
				{
					
					//Write a "pause" message too
					//First draw a black border (display same text four times with 1 pixel offset from actual text)
					tgi_setcolor(1);
					tgi_outtextxy(55, 47, "PAUSED");
					tgi_outtextxy(57, 47, "PAUSED");
					tgi_outtextxy(56, 46, "PAUSED");
					tgi_outtextxy(56, 48, "PAUSED");
					//Then display the real message on top in white color
					tgi_setcolor(15);
					tgi_outtextxy(56, 47, "PAUSED");
					
					//First draw a black border (display same text four times with 1 pixel offset from actual text)
					tgi_setcolor(15);
					tgi_outtextxy(3, 67, "HIGHSCORES RESETED!");
					tgi_outtextxy(5, 67, "HIGHSCORES RESETED!");
					tgi_outtextxy(4, 68, "HIGHSCORES RESETED!");
					tgi_outtextxy(4, 66, "HIGHSCORES RESETED!");
					//Then display the real message on top in white color
					tgi_setcolor(1);
					tgi_outtextxy(4, 67, "HIGHSCORES RESETED!");
					//Update the screen (flip drawbuffer) so we can actually see it onscreen :)
					tgi_updatedisplay();
					
					//Reset the saves!
					resetSaveData();
				}
				break;

			case '?': 
				break;

			default:
				break;
			}
		}
	} while(halted && !reset);
	
	return joy_read(JOY_1);
}
