#!/usr/bin/perl
##
## ATARIMAX 'MAXFLASH' Flash Cartridge Product Utility Script
##
## Version 1.0.0 07/10/2003
## Version 1.0.1 08/09/2003
## Version 1.0.2 11/30/2004 Matthias Reichl
## Version 1.0.3 12/15/2004 Matthias Reichl
## Version 1.0.4 03/26/2004 Matthias Reichl
##
## Updates and information at http://www.atarimax.com/
## email: classics@atarimax.com, backup: atari@yahoo.com
## Copyright (C) 2003 Steven J Tucker
##
## ATasm/Linux version (c) 2004 by Matthias Reichl <hias@horus.com>
##
## This program is free software; you can redistribute it and/or
## modify it under the terms of the GNU General Public License
## as published by the Free Software Foundation; either version 2
## of the License, or (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
##

$build_date = '03/26/2005';

use Compress::Zlib;
use File::Basename;

$osb_mathpack		= 'osb/mathpack';			
$osb_mathpack_size	= 2048;					
$osb_os			= 'osb/osb';				
$osb_os_size		= 8192;					

$maxdisk_source		= 'maxdisk.a65';			
$maxdisk_target		= 'maxdisk.com';			
$softdisk_source	= 'softdisk.a65';			
$softdisk_target	= 'softdisk.com';			
$mempack_source 	= 'mempack.a65';			
$mempack_target	  	= 'mempack.com';			
$flash_source 		= 'flash.a65';				
$flash_target	  	= 'flash.com';				
$exepack_menu_source 	= 'exepmenu.a65';			
$exepack_menu_target	= 'exepmenu.com';			
$exepack_up_source	= 'exeunpak.a65';			
$exepack_up_target	= 'exeunpak.com';			
$configuration_file	= 'maxflash.cfg';			

$carttype_1024		= 0x00000026;				
$carttype_4096		= 0x00000027;				
$carttype_8192		= 0x00000028;				
$sector_size		= 128;					
$softdisk_reserved_size = 0x200;				
$atr_header_size	= 16;					
$bank_size		= 8192;					
$bootflash_size		= 2048;					
$flash_titlemarker	= 'AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA';	
$atrheader_1mb		= "\x96\x02\x80\x20\x80\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00";
$atrheader_4mb		= "\x96\x02\x80\x80\x80\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00";
$atrheader_8mb		= "\x96\x02\x80\x00\x80\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00";
$scancodes		= "\x3F\x15\x12\x3A\x2A\x38\x3D\x39\x0D\x01\x05\x00\x25\x23\x08\x0A\x2F\x28\x3E\x2D\x0B\x10\x2E\x16\x2B\x17\x32\x1F\x1E\x1A\x18\x1D\x1B\x33\x35\x30";
$banker			= "\x8D\x00\xD5\x4C\x00\xA0\x00\xA0\x00\x01\xF4\xBF";
$mapkeys		= "ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789 ";
$default_filename	= "untitled.bin";
$static_flash           = "flash/flash.com";

read_configuration();


%options	= ('-LOMEM'	=>	'(D) Force HIMEM to A000 after installation.',
		   '-PALHACK'	=>	'(D) Force OS to detect GITA as PAL.',
		   '-COLORS'	=>	'(*) Fill color registers while working. (debug)',
		   '-BASIC'	=>	'(D) Enable BASIC. (For packing disks that use BASIC)',
		   '-HOLMES'	=>	'(D) Try to follow archive directives (-bas, -osb, etc).',
		   '-OSB'	=>	'(*) Build automatic OS-B translator into cartridge.',
		   '-FLICKER'   =>	'(D) Dont wait for ANTIC. (will cause flickering)',
		   '-TRIGGERS'	=>	'(D) Enable sector triggers.',
		   '-TRANSLATOR'=>	'(D) Replace OS but not not install any drivers.',
		   '-NODISK'	=>	'(D) Create image with blank disk area.',
		   '-CART'	=>	'(*) Write file with Atari800 cartridge header.',
		   '-FLASHER'	=>	'(*) Make bootable ATR image for programming MAXFLASH cart',
		   '-NOBIN'	=>	'(*) Delete raw binary after processing.',
		   '-UNDEROS'	=>	'(M) Unpack memory under OS.',
		   '-BIN2CART'	=>	'(U) Make existing BIN file into Atari800 CART image.',
		   '-BIN2ATR'	=>	'(U) Make existing BIN file into flash programming image.',
		   '-BIN2ALL'	=>	'(U) Make existing BIN file into CART/ATR images.',
		   '-BATCH'	=>	'(*) Never prompt for optional information.',
		   '-RECURSIVE'	=>	'(*) Process sub-directories in directory mode.',
		   '-EXEPACKER'	=>	'(K) Create multi-cart image using directory of EXE files.',
		   '-VERBOSE'	=>	'(*) Display useless information.',
		   '-NOEXE'	=>	'(E) Erase EXE files after packing with -EXEPACKER option.',
		   '-SEGCHECK'	=>	'(U) Display segment information in an Atari EXE file.',
		   '-FIXSEG'	=>	'(U) Combine with -SEGCHECK to try and fix an EXE file.',
		   '-ALTPTR'	=>	'(E) Use alternate address for Atari EXE loader.',
		   '-REINIT'	=>	'(E) Force EXE loader to enter RUN vector via warmstart',
		   '-4MB'	=>	'(*) Create image for 4MB cartridge. (default is 1MB)',
		   '-8MB'	=>	'(*) Create image for 8MB cartridge. (default is 1MB)',
		   '-FASTKEYS'	=>	'(E) Faster selection in EXE loader menu.',
		   '-CHEESE'	=>	'(E) Enable DLIs, sounds and colors in EXE menu.',
		   '-DISKPACKER'=>	'(K) Pack ATR image into cartridge using MAXFLASH driver.',
		   '-MEMPACKER' =>	'(K) Pack Atari800 A8S image into cartridge.',
		   '-CLEAN'	=>	'(U) Remove intermediate files. (lst,bak,atr,com,bin,car)',
		   '-TESTBUILD' =>	'(U) Test assembler by building all source files.',
		   '-NOISE'	=>	'(D) Make a noise during cartridge "disk" access.',

		   '-FRZ32KB'	=>	'(*) Create 32k Freezer image',
		   '-FRZ48KB'	=>	'(*) Create 48k Freezer image',
		   '-FRZ64KB'	=>	'(*) Create 64k Freezer image',
		   '-FRZ80KB'	=>	'(*) Create 80k Freezer image',
		   '-FRZ96KB'	=>	'(*) Create 96k Freezer image',
		   '-FRZ128KB'	=>	'(*) Create 128k Freezer image',
		   '-FRZ192KB'	=>	'(*) Create 192k Freezer image',
		   '-FRZ256KB'	=>	'(*) Create 256k Freezer image',
		   '-FRZ320KB'	=>	'(*) Create 320k Freezer image',
		   '-FRZ384KB'	=>	'(*) Create 384k Freezer image',
		   '-FRZ448KB'	=>	'(*) Create 448k Freezer image',
		   '-FRZBANK=x' =>      '(*) Freezer ROM bank base number (default is 0)');

%hacks		= ('@PORTB'	=>	'(*) Delete writes to PORT B in image.');

print "\nMAXFLASH Multicart Build/Utility Script - c2003 Steven J Tucker\n";
print "Information and updates at http://www.atarimax.com/\n\n";
print "ATasm/Linux and TurboFreezer 4 Cartridge Emulation version\n";
print "(c) 2004-2005 by Matthias Reichl <hias\@horus.com>\n";

if (@ARGV == 0)
{
	print "\nusage: perl $0 [switches] [hacks] [disk.atr] [cartridge.bin] [dump.a8s]\n\n";

	$prefix	= "switches:";

	foreach $key (sort(keys(%options)))
	{
        	printf("%-10.10s%-12.12s%s\n", $prefix, $key, $options{$key});
    		$prefix = '';
    	}
    	print "\n";

	$prefix	= "hacks:";
	while(my ($key, $value) = each(%hacks))
	{
        	printf("%-10.10s%-12.12s%s\n", $prefix, $key, $value);
    		$prefix = '';
    	}
    	print "\n";
    	exit(-1);
}
else
{
	print "\n";
}

foreach (@ARGV)
{
	if (-d $_)
	{
		$directory = $_;
		$directory =~ s/\\/\//igs;
		next;
	}

	if (/^-.*/)
	{
		s/\-/\-D/;
		push(@switches, uc $_);
		next;
	}

	if (/.*atr$/i)
	{
		$image = $_;
		next;
	}

	if (/.*a8s$/i)
	{
		$dump = $_;
		next;
	}

	if (/.*\.bin$/i)
	{
		$target = $_;
		next;
	}

	if (/.*\.(exe|com|xex)$/i)
	{
		$exefile = $_;
		next;
	}

	if (/^\@.*/)
	{
		push(@imagehacks, uc $_);
		next;
	}

	print "Discarded unknown option: $_\n";
}

if ("@switches" =~ /-DCLEAN/igs)
{
	if (-f 'maxflash.pl')
	{
		print "Cleaning up files in source directory.\n";
		unlink <*.lst>;
		unlink <*.bak>;
		unlink <*.com>;
		unlink <*.atr>;
		unlink <*.bin>;
		unlink <*.car>;
		unlink <*.a8s>;
		exit(0);
	}
	else
	{
		print "This option deletes files, please use it only in the source directory.\n";
		exit(-1);
	}
}

if (!("@switches" =~ /\-D([148]MB|FRZ[0-9]*KB)/igs))
{
	push(@switches, '-D1MB');
}

if ("@switches" =~ '-D1MB')
{
	$maximum_banks		= 16;					
	$atari800carttype	= $carttype_1024;			
	$desired_banks		= $maximum_banks - 1;			
	$desired_banks_osb	= $maximum_banks - 2;			
}

if ("@switches" =~ '-D4MB')
{
	$maximum_banks		= 64;					
	$atari800carttype	= $carttype_4096;			
	$desired_banks		= $maximum_banks - 1;			
	$desired_banks_osb	= $maximum_banks - 2;			
}

if ("@switches" =~ '-D8MB')
{
	$maximum_banks		= 128;					
	$atari800carttype	= $carttype_8192;			
	$desired_banks		= $maximum_banks - 1;			
	$desired_banks_osb	= $maximum_banks - 2;			
}

if ("@switches" =~ '-DFRZ32KB')
{
	$maximum_banks		= 4;					
	$atari800carttype	= 0;
	$desired_banks		= $maximum_banks - 1;			
	$desired_banks_osb	= $maximum_banks - 2;			
	push(@switches, '-DFREEZER');
}

if ("@switches" =~ '-DFRZ48KB')
{
	$maximum_banks		= 6;					
	$atari800carttype	= 0;
	$desired_banks		= $maximum_banks - 1;			
	$desired_banks_osb	= $maximum_banks - 2;			
	push(@switches, '-DFREEZER');
}

if ("@switches" =~ '-DFRZ64KB')
{
	$maximum_banks		= 8;					
	$atari800carttype	= 0;
	$desired_banks		= $maximum_banks - 1;			
	$desired_banks_osb	= $maximum_banks - 2;			
	push(@switches, '-DFREEZER');
}

if ("@switches" =~ '-DFRZ80KB')
{
	$maximum_banks		= 10;					
	$atari800carttype	= 0;
	$desired_banks		= $maximum_banks - 1;			
	$desired_banks_osb	= $maximum_banks - 2;			
	push(@switches, '-DFREEZER');
}

if ("@switches" =~ '-DFRZ96KB')
{
	$maximum_banks		= 12;					
	$atari800carttype	= 0;
	$desired_banks		= $maximum_banks - 1;			
	$desired_banks_osb	= $maximum_banks - 2;			
	push(@switches, '-DFREEZER');
}

if ("@switches" =~ '-DFRZ128KB')
{
	$maximum_banks		= 16;					
	$atari800carttype	= 0;
	$desired_banks		= $maximum_banks - 1;			
	$desired_banks_osb	= $maximum_banks - 2;			
	push(@switches, '-DFREEZER');
}

if ("@switches" =~ '-DFRZ192KB')
{
	$maximum_banks		= 24;					
	$atari800carttype	= 0;
	$desired_banks		= $maximum_banks - 1;			
	$desired_banks_osb	= $maximum_banks - 2;			
	push(@switches, '-DFREEZER');
}

if ("@switches" =~ '-DFRZ256KB')
{
	$maximum_banks		= 32;					
	$atari800carttype	= 0;
	$desired_banks		= $maximum_banks - 1;			
	$desired_banks_osb	= $maximum_banks - 2;			
	push(@switches, '-DFREEZER');
}

if ("@switches" =~ '-DFRZ320KB')
{
	$maximum_banks		= 40;					
	$atari800carttype	= 0;
	$desired_banks		= $maximum_banks - 1;			
	$desired_banks_osb	= $maximum_banks - 2;			
	push(@switches, '-DFREEZER');
}

if ("@switches" =~ '-DFRZ384KB')
{
	$maximum_banks		= 48;					
	$atari800carttype	= 0;
	$desired_banks		= $maximum_banks - 1;			
	$desired_banks_osb	= $maximum_banks - 2;			
	push(@switches, '-DFREEZER');
}

if ("@switches" =~ '-DFRZ448KB')
{
	$maximum_banks		= 56;					
	$atari800carttype	= 0;
	$desired_banks		= $maximum_banks - 1;			
	$desired_banks_osb	= $maximum_banks - 2;			
	push(@switches, '-DFREEZER');
}


if (("@switches" =~ '-DOSB') & ("@switches" =~ '-DCOLORS'))
{
	print "WARNING: -osb and -colors options used together.\n";
	print "         (Remove -colors option if you are making a real cartridge)\n\n";
}

if ("@switches" =~ '-DBIN2ALL')
{
	push(@switches, '-DBIN2CART');
	push(@switches, '-DBIN2ATR');
}

if ("@switches" =~ /\-DHOLMES/)
{
	if ($image =~ /\-bas/igs)
	{
		print "Adding -BASIC switch for this image.\n";
		push(@switches, '-DBASIC');
	}

	if ($image =~ /\-osb/igs)
	{
		print "Adding -OSB switch for this image.\n";
		push(@switches, '-DOSB');
	}
}

$extra_options = join(' ', @switches);

if ("@switches" =~ /-DTESTBUILD/igs)
{
	push(@switches, '-DVERBOSE');
	rebuild_all();
	exit(0);
}

if ("@switches" =~ /\-DOSB/)
{
	$desired_banks = $desired_banks_osb;

	if (-r $osb_mathpack)
	{
       		$mathpack_size = -s $osb_mathpack;
	       	($mathpack_size eq $osb_mathpack_size) or die("Mathpack image size $mathpack_size does not equal known correct size of $osb_mathpack_size\n");
       		$osb_mathpack_binary = binslurp_read($osb_mathpack);
        }
        else
        {
        	print "Could not locate OSB Mathpack in $osb_mathpack.\n";
        	exit(-1);
        }

	if (-r $osb_os)
	{
	       	$osb_size = -s $osb_os;
       		($osb_size eq $osb_os_size) or die("OSB image size $osb_size does not equal known correct size of $osb_os_size\n");
       		$osb_os_binary = binslurp_read($osb_os);
	}
	else
	{
		print "Could not location OSB OS in $osb_os.\n";
		exit(-1);
	}

	print "Building OSB Translator into cartridge image.\n\n";
}

if (("@switches" =~ /\-DBIN2ATR/) | ("@switches" =~ /\-DBIN2CART/))
{
	if ("@switches" =~ /\-DBIN2ATR/)
	{
		if ((-d $directory) and ($target eq ''))
		{
			opendir(HANDLE, $directory) or die("Could not read directory contents in $directory\n");
			@files = readdir(HANDLE);
			closedir(HANDLE);

			foreach (@files)
			{
				if (/\.bin$/i)
				{
					$fqn = "$directory\\$_";
					bin2atr($fqn);
				}
			}
		}
		else
		{			
			bin2atr($target);
		}
	}

	if ("@switches" =~ /\-DBIN2CART/)
	{
		bin2car($target);
	}
	exit(0);
}

if ((-d $directory) & (!("@switches" =~ /\-DEXEPACKER/)))
{
	opendir(HANDLE, $directory) or die("Could not read directory contents in $directory\n");
	@files = readdir(HANDLE);
	closedir(HANDLE);

	$spawnoptions = join(' ', @switches) . ' ' . join(' ', @imagehacks);
	$spawnoptions =~ s/\-D/\-/igs;

	foreach (@files)
	{
		if (/^\./)
		{
			next;
		}

		if (("@switches" =~ /\-DRECURSIVE/) & (-d $_))
		{
			print `perl $0 \"$_\" -BATCH $spawnoptions`;
		}

		$fqn = "$directory\\$_";

		if ((/\.atr$/i) & (-r $fqn))
		{
			if (-r $fqn)
			{
				my $binname = $fqn;
				$binname =~ s/.*(\\|\/)//igs;
				$binname =~ s/\.atr$/\.bin/igs;
				print `perl $0 \"$fqn\" \"$binname\" -BATCH $spawnoptions`;
			}
		}
	}
	exit(-1);
}

if ((-f $exefile) & ("@switches" =~ /\-DSEGCHECK/))
{
	($scrubbed_exe, $warncount, $errorcount) = scrubexe($exefile, 1);

	if ("@switches" =~ /\-DFIXSEG/)
	{
		$newexe = $exefile;
		$newexe =~ s/\.exe/_fixseg\.exe/igs;
		$newexe = default_prompt("New filename: ", $newexe, 15);

		binslurp_write($newexe, $scrubbed_exe);

		printf("Wrote $newexe, %i bytes.", -s $newexe);
		print `perl \"$0\" -SEGCHECK \"$newexe\"`;
	}
	exit($errorcount);
}

if ((-f $dump) and ("@switches" =~ /\-DMEMPACKER/))
{
	rebuild($mempack_source, $mempack_target);

	$gz = gzopen($dump, "r") or die ("Could not open $dump for reading.\n");
    	$realsize = $gz->gzread($buffer, 1000000) or die ("Error decompressing $dump.\n");
    	$gz->gzclose();

    	$header_magic	= substr($buffer, 0x00, 8);
    	$header_version	= ord(substr($buffer, 0x08, 1));
    	$header_size	= ord(substr($buffer, 0x0B, 1));
    	$memory		= substr($buffer, 0x40, 0x10000);

	if ($header_magic ne 'ATARI800')
	{
		print "File $dump does not have 'ATARI800' header.\n";
		exit(-1);
	}

	if ($header_version ne 3)
	{
		print "Cant process version $header_version saves yet.\n";
		exit(-1);
	}

	if ($header_size > 1)
	{
		print "This save contains more than 64k of memory contents.\n";
		exit(-1);
	}

	$memory =~ s/..../\x00\xA0\x00\x00/;

	$final_image  = binslurp_read($mempack_target);
	$final_image .= $memory;
	$final_image  = zeropad($final_image, $bank_size * $maximum_banks);
	binslurp_write($target, $final_image);

	print "New image [$target] created.\n";

	if ("@switches" =~ /\-DCART/)
	{
		bin2car($target);
	}

	if ("@switches" =~ /\-DFLASHER/)
	{
		bin2atr($target);
	}

	if ("@switches" =~ /\-DNOBIN/)
	{
		unlink($target) or die("Could not delete file $target, $!\n");
	}
	exit(0);
}

if ("@switches" =~ /\-DEXEPACKER/)
{
	if (!(-d $directory))
	{
		print "\nEXEPACKER option was specified without a directory.  Can't continue.\n\n";
		exit(-1);
	}

	rebuild($exepack_menu_source, $exepack_menu_target);

	rebuild($exepack_up_source, $exepack_up_target);

	$unpack_menu = binslurp_read($exepack_menu_target);
	$unpack_unpacker = zeropad(binslurp_read($exepack_up_target), 256);
	$unpack_menu =~ s/\xFF\xFE\xFD\xFC.*\xFC\xFD\xFE\xFF/$unpack_unpacker/igs;

	if ($desired_banks eq $desired_banks_osb)
	{
		$unpack_menu =~ s/\xFB\xFA\xF9\xF8.*\xFB\xFA\xF9\xF8/$osb_mathpack_binary/;
	}

	@files = dirslurp($directory);

	printf("Looking over EXE files in %s\n\n", shortstring($directory, 50));
	foreach(@files)
	{
		$basefilename = $_;
		$fqn = "${directory}/${basefilename}";

		if (-d $_)
		{
			next;
		}

		my $header = '';
		open(HANDLE, "<$fqn") or die("Could not open $fqn for reading!\n");
		sysread(HANDLE, my $buffer, 2);
		close(HANDLE);

		if ($buffer =~ /^\xFF\xFF/)
		{
			push(@exefiles, $fqn);
		}
		else
		{
			printf("Skipped file \"%s\".  It does not look like an Atari EXE file.\n", basename($fqn));
		}
	}

	$available_space = $desired_banks * $bank_size;

	$reserve_bytes = 0;
	if ("@switches" =~ /\-D[48]MB/)
	{
		$reserve_bytes = length($banker);
	}
	$available_space -= $reserve_bytes;

	$total_files	= 0;
	$skipped_files	= 0;
	$current_bank	= 1;

	if ($desired_banks eq $desired_banks_osb)
	{
		$current_bank++;
	}

	$bankposition	= 0xA000;
	$response_key	= 0;
	$exedata = '';
	$bindata = '';

	foreach (@exefiles)
	{
		$exe_filename	= $_;
		($exe_bindata, $warncount, $errorcount) = scrubexe($exe_filename, 0, 1);
		if ($errorcount)
		{
			printf("Skipped %s, may be damaged, run -SEGCHECK on it.\n", basename($exe_filename));
			next;
		}
		$exe_size = length($exe_bindata);

		if ($exe_size <= $available_space)
		{
			$exedata .= $exe_bindata;

			if ($response_key + 1 > length($scancodes))
			{
				$response_scancode = "\x40";
			}
			else
			{
				$response_scancode = substr($scancodes, $response_key, 1);
				$response_key++;
			}

			$bindata .= $response_scancode;
			$bindata .= pack("n", $bankposition);
			$bindata .= chr($current_bank);

			$total_files++;
			$bankposition += $exe_size;
			while ($bankposition >= 0xC000)
			{
				$bankposition -= 0x2000;
				$current_bank++;
			}
			$available_space -= $exe_size;

			push(@inserted_files, $exe_filename);

			printf("Using %s (%s bytes), %s bytes remaining.\n", basename($exe_filename), $exe_size, $available_space);
		}
		else
		{
			if ("@switches" =~ /\-DVERBOSE/)
			{
				print "Skipping $exe_filename ($exe_size bytes), it wont fit.\n";
			}
		}
	}

	if ($total_files > 0)
	{
		print "\nAssembled $total_files files with $available_space bytes of slack.\n";
	}
	else
	{
		print "\nCould not use any Atari EXE files in $directory!\n";
		exit(-1);
	}

	print "\nBy default the names of the images that appear on the menu of the\n";
	print "flash cartridge will be the same as the files they were taken\n";
	print "from.\n\nYou can choose new menu descriptions here, or just\n";
	print "press Enter to keep what is shown (34 characters maximum).\n\n";

	$response_key = 0;
	foreach (@inserted_files)
	{
		($base, $path, $type) = fileparse($_);
		$base =~ s/\.exe//igs;
		$new_filename = default_prompt("Title ", $base, 34);
		$new_filename = '<' . substr($mapkeys, $response_key, 1) . '> ' . substr($new_filename, 0, 34) . chr(155);
		push(@final_filenames, $new_filename);
		if (++$response_key == length($mapkeys))
		{
			$response_key--;
		}
	}

	print "\nPlease enter a title for this cartridge.  This text will\n";
	print "appear at the top-most line of the multicart menu.  Just\n";
	print "press enter to keep the default description\n\n";
	if ("@switches" =~ /\-DFREEZER/)
	{
		$carttitle = "TurboCartridge EXE Loader";
	}
	else
	{
		$carttitle = "ATARIMAX Flash Multicart EXE Loader";
	}
	$carttitle = default_prompt("Cartridge Title: ", $carttitle, 25);
	$carttitle = substr($carttitle, 0, 40);
	$carttitle = (' ' x (20 - (length($carttitle) / 2))) . $carttitle . (' ' x (20 - (length($carttitle) / 2))) . '  ';
	$carttitle = substr($carttitle, 0, 40);

	foreach (@final_filenames)
	{
		$final_string .= $_;
	}
	$menudatablock = zeropad(chr($total_files) . $final_string, 2000);

	$bindatablock = zeropad($bindata, 1000, "\x40");

	$unpack_menu =~ s/\xD0\xCF\xCE\xCD\xCC.*\xD0\xCF\xCE\xCD\xCC/$menudatablock/igs;
	$unpack_menu =~ s/\xC0\xBF\xBE\xBD\xBC.*\xC0\xBF\xBE\xBD\xBC/$bindatablock/igs;

	$unpack_menu =~ s/AAAAAAAAAAAAZZZZZZZZAAAAAAAAAAAZZZZZZZZZ/$carttitle/igs;

	$final_binary	=  $unpack_menu;

	if ($desired_banks eq $desired_banks_osb)
	{
		$final_binary .= $osb_os_binary;
	}

	$final_binary	.= zeropad($exedata, ($desired_banks * $bank_size) - $reserve_bytes);

	if ($reserve_bytes == length($banker))
	{
		$final_binary .= $banker;
	}

	if ($target eq '')
	{
		print "\nWARNING: You did not enter a destination filename for the raw binary.\n";
		printf("WARNING: Using \"%s\" as the target image filename.\n", $default_filename);
		$target = $default_filename;
	}

	binslurp_write($target, $final_binary);
	printf("\nSaved new binary flash image to %s\n", $target);

	if ("@switches" =~ /\-DCART/)
	{
		bin2car($target);
	}

	if ("@switches" =~ /\-DFLASHER/)
	{
		bin2atr($target);
	}

	if ("@switches" =~ /\-DNOEXE/)
	{
		foreach (@inserted_files)
		{
			unlink($_);
		}
	}
	exit(0);
}

if ("@switches" =~ /\-DDISKPACKER/)
{
	if ($target eq '')
	{
		print "WARNING: You did not enter a destination filename for the raw binary.\n";
		printf("WARNING: Using \"%s\" as the target image filename.\n\n", $default_filename);
		$target = $default_filename;
	}

        $desired_size		= $desired_banks * $bank_size;		
        $desired_atr_size	= $desired_size + $atr_header_size;	

	rebuild($softdisk_source, $softdisk_target);

	rebuild($maxdisk_source, $maxdisk_target);

        if ("@switches" =~ /NODISK/)
        {
        	$size = 92176;
        	$atr = "\0" x $size;
        }
        else
        {
        	$size	= -s $image;
        	$atr    = binslurp_read($image);
        }

        $atr = process_hacks($atr);

       	$sectors = ($size - $atr_header_size) / $sector_size;
       	$desired_sectors = $desired_size / $sector_size;

        if ($size > $desired_atr_size)
        {
		print "\nWARNING: Trimming $sectors sector ATR ($size bytes) to $desired_sectors sectors ($desired_atr_size bytes).\n";
        	$atr = substr($atr, 0, $desired_atr_size);
        	$size = $desired_atr_size;
        }
        if ($size < $desired_atr_size)
        {
 		if ("@switches" =~ /verbose/igs)
		{
	        	print "\nPadding $sectors sector ATR ($size bytes) to $desired_sectors sectors ($desired_atr_size bytes).\n";
		}
	        $atr .= "\0" x ($desired_atr_size - $size);
        	$size = $desired_atr_size;
        }

        $loadersize = -s $maxdisk_target;
        if ($loadersize ne $bank_size)
        {
        	print "\nFatal error: Loader [$maxdisk_target] is not exactly one bank of $bank_size bytes.\n";
        	exit(-1);
        }
        $loader = binslurp_read($maxdisk_target);

	$md_time      = localtime; 
	$md_image     = sprintf("%-33.33s", $image);
	$md_buildtime = sprintf("%-33.33s", $md_time);
	$md_kit	      = sprintf("%-33.33s", $build_date);
	$loader =~ s/IMAGEIMAGEIMAGEIMAGEIMAGEIMAGEIMA/$md_image/igs;
	$loader =~ s/DATEDATEDATEDATEDATEDATEDATEDATED/$md_buildtime/igs;
	$loader =~ s/KITKITKITKITKITKITKITKITKITKITKIT/$md_kit/igs;

	if (length($loader) != $loadersize)
	{
		printf("Fatal error: Loader is wrong size after build data insertion.\n");
		exit(-1);
	}

        $softdisk_size = -s $softdisk_target;
        $softdisk = binslurp_read($softdisk_target);

        $softdisk .= "\xFF" x ($softdisk_reserved_size - $softdisk_size);

        $loader =~ s/\xFF\xFE\xFD\xFC.*\xFF\xFE\xFD\xFC/$softdisk/;

        if ($desired_banks eq $desired_banks_osb)
        {
        	$loader =~ s/\xFB\xFA\xF9\xF8.*\xFB\xFA\xF9\xF8/$osb_mathpack_binary/;
        }

        $image = $loader;

	if (length($loader) != $loadersize)
	{
		printf("Fatal error: Loader is wrong size after ROM image insertions.\n");
		exit(-1);
	}

        if ($desired_banks eq $desired_banks_osb)
        {
		$image .= $osb_os_binary;
        }

	$image .= substr($atr, 16);

	binslurp_write($target, $image);

        print "New binary image [$target] created.\n";

        if ("@switches" =~ /\-DCART/)
        {
        	bin2car($target);
        }

        if ("@switches" =~ /\-DFLASHER/)
        {
        	bin2atr($target)
        }

        if ("@switches" =~ /\-DNOBIN/)
        {
        	unlink($target) or die("Could not delete file $target, $!\n");
        }
	exit(0);
}

print "\nYou must specify at least one (K)ey or (U)tility function.\n\nTry running with no options for help.\n\n";
exit(-1);


sub zeropad
{
	$padding_required = @_[1] - length(@_[0]);
	$padchar = @_[2];
	if (length($padchar) != 1)
	{
		$padchar = "\0";
	}
	$padding = $padchar x $padding_required;
	return(@_[0] . $padding);
}

sub binslurp_read
{
	my $size = -s @_[0];

	open(HANDLE, "<@_[0]") or die("Could not open @_[0] for reading.\n");
	binmode(HANDLE);
	sysread(HANDLE, my $buffer, $size);
	close(HANDLE);
	return($buffer);
}

sub binslurp_write
{
	open(HANDLE, ">@_[0]") or die ("Could not open @_[0] for writing.\n");
	binmode(HANDLE);
	syswrite(HANDLE, @_[1], length(@_[1]));
	close(HANDLE);
}

sub bin2car
{
	if ("@switches" =~ /\-DFRZ[0-9]*KB/)
	{
		print "creating ATR file for freezer is not supported!\n";
		exit(-1);
	}

	$target = @_[0];

	$target =~ /^(.*)\..*$/;
	$target_cartridge = "$1.car";

	$size = -s $target;
	open(HANDLE, "<$target") or die("Could not open $target for reading!\n");
	sysread(HANDLE, $full_binary, $size);
	close(HANDLE);

	$sum = 0;
	while ($full_binary =~ /(.)/gso)
	{
		$sum += ord($1);
	}

	$header = 'CART';
	$header .= pack("N", $atari800carttype);
	$header .= pack("N", $sum);
	$header .= pack("N", 0);

	$header .= $full_binary;
	binslurp_write($target_cartridge, $header);

	printf("Saved new Atari800 \"CART\" image to %s\n", $target_cartridge);
}

sub bin2atr
{
	$target = @_[0];

	if ($static_flash)
	{
		if (-r $static_flash)
		{
			$flash_target = $static_flash;
		}
		else
		{
			printf("[bin2atr] Could not locate flash.com at %s\n", $static_flash);
			exit(-1);
		}
	}
	else
	{
		rebuild($flash_source, $flash_target);
	}

	$sizeof_marker = length($flash_titlemarker);
	$target =~ /^(.*)\..*$/;
	$target_title = $1;
	$target_cartridge = "$1 (maxflash image).atr";

	if ("@switches" =~ /\-D8MB/)
	{
		$flashimage = $atrheader_8mb;
	}
	if ("@switches" =~ /\-D4MB/)
	{
		$flashimage = $atrheader_4mb;
	}
	if ("@switches" =~ /\-D1MB/)
	{
		$flashimage = $atrheader_1mb;
	}

	if ("@switches" =~ /\-DFRZ[0-9]*KB/)
	{
		print "creating ATR file for freezer is not supported!\n";
		exit(-1);
	}

	$flashimage .= zeropad(binslurp_read($flash_target), $bootflash_size);
	$flashimage .= binslurp_read($target);

	if (!("@switches" =~ /\-DBATCH/))
	{
		print "\nWhen you load the flash cartridge programming disk on the Atari Computer\n";
		print "it will display the title of the software to be programmed.  You can change\n";
		print "what is displayed, or just press ENTER to keep what is shown.  Note: you\n";
		print "may use a maximum of 34 characters in this field.\n\n";
		$target_title = default_prompt("Title ", $target_title, 20);
		print "\n";
	}

	$target_title =	sprintf("%-${sizeof_marker}.${sizeof_marker}s", $target_title);
	$flashimage =~ s/$flash_titlemarker/$target_title/igs;

	binslurp_write($target_cartridge, $flashimage);

	printf("Saved new cartridge programming image to \"%s\"\n", $target_cartridge);
}

sub default_prompt
{
        $dp_fixed = @_[0];
        $dp_default = @_[1];
        $dp_size = @_[2];

        $dp_promptsize = 2 + length($dp_fixed) + length($dp_default);

        $dp_prompt = $dp_fixed . (' ' x ($dp_size - $dp_promptsize)) . "[$dp_default] : ";
        print "$dp_prompt";
        $dp_response = <STDIN>;
        chomp($dp_response);
        if ($dp_response eq '')
        {
                $dp_response = $dp_default;
        }

        return($dp_response);
}

sub scrubexe
{
	my $exefile = @_[0];
	my $verbose = @_[1];
	my $minimize= @_[2];

	if ($verbose)
	{
		print "Checking segments in $exefile\n\n";
	}

	my $exedata = binslurp_read($exefile);

	my $errors = 0;
	my $warnings = 0;
	my $exeptr = 0;
	my $newexedata = '';
	my $maximumerrors = 20;
	while (($exeptr < length($exedata)) & ($errors < $maximumerrors))
	{
		if ((length($exedata) - $exeptr) < 4)
		{
			if ($verbose)
			{
				printf("Warning: Ignoring %i bytes of extra data at end of file.\n", length($exedata) - $exeptr);
			}
			$warnings++;
			$exeptr = length($exedata);
			next;
		}

		my $startseg = unpack("v", substr($exedata, $exeptr, 2));
		my $endseg   = unpack("v", substr($exedata, $exeptr + 2, 2));

		if ($startseg == 0xFFFF)
		{
			if ($verbose)
			{
				printf("Header marker 0x%4.4X at 0x%4.4X\n", $startseg, $exeptr);
			}

			if (($exeptr == 0) & (!($minimize)))
			{
				$newexedata .= substr($exedata, $exeptr, 2);
			}
			else
			{
				if ($verbose)
				{
					print "Warning: Ignoring extraneous header embedded in file.\n";
				}
				$warnings++;
			}
			$exeptr += 2;
			next;
		}

		if ($startseg <= $endseg)
		{
			if (((($endseg - $startseg) + 1) + $exeptr + 4) > length($exedata))
			{
				if ($verbose)
				{
					if(ask(sprintf("Segment [0x%4.4X - 0x%4.4X] at 0x%4.4X extends past end of file, keep it?", $startseg, $endseg, $exeptr), 'N'))
					{
						print "Segment retained.\n";
					}
					else
					{
						print "Segment discarded, treating as zero-length.\n";
						$exeptr += 4;
						next;
					}
				}
				else
				{
					$exeptr += 4;
					next;
				}
			}

			if ($verbose)
			{
				printf("Valid segment [0x%4.4X - 0x%4.4X] at 0x%4.4X\n", $startseg, $endseg, $exeptr);
			}

			if (($startseg eq $endseg) & (ord(substr($exedata, $exeptr, 1)) eq (($startseg & 0xFF00) >> 8)) & (ord(substr($exedata, $exeptr, 1)) eq ($endseg & 0xFF)))
			{
				if ($verbose)
				{
					print "Warning: Ignoring junk segment. (transfer padding)\n";
				}
				$warnings++;
			}
			else
			{
				if (!($minimize))
				{
					$newexedata .= substr($exedata, $exeptr, 4 + 1 + ($endseg - $startseg));
				}
				else
				{
					$seglength   = 1 + ($endseg - $startseg);
					$newexedata .= pack("v", $startseg);
					$newexedata .= pack("v", $seglength);
					$newexedata .= substr($exedata, $exeptr + 4, $seglength);
				}

				if (($startseg <= 0x2E0) & ($endseg >= 0x2E1))
				{
					my $runadd = unpack("v", substr($exedata, $exeptr + (0x2E0 - $startseg) + 4, 2));
					if ($verbose)
					{
						printf("Segment sets RUN address of 0x%4.4X\n", $runadd);
					}
				}

				if (($startseg <= 0x2E2) & ($endseg >= 0x2E3))
				{
					my $runadd = unpack("v", substr($exedata, $exeptr + (0x2E2 - $startseg) + 4, 2));
					if ($verbose)
					{
						printf("Segment sets immediate INIT address of 0x%4.4X\n", $runadd);
					}
				}
			}

			$exeptr += 4;
			$exeptr += ($endseg - $startseg) + 1;
			next;
		}
		else
		{
			if ($verbose)
			{
				printf("Invalid segment [0x%4X - 0x%4X] at 0x%4X -- assuming zero length.\n", $startseg, $endseg, $exeptr);
			}
			$exeptr += 4;
			$errors++;
			next;
		}
	}

	if ($exeptr > length($exedata))
	{
		if ($verbose)
		{
			printf("Warning: final segment extends past end of file by %i bytes.\n", ($exeptr - length($exedata)));
		}
		$warnings++;
	}

	if ($errors < $maximumerrors)
	{
		if ($verbose)
		{
			print "\nProcess completed with $warnings warnings and $errors errors.\n\n";
		}
	}
	else
	{
		if ($verbose)
		{
			print "\nProcess aborted after $errors errors.\n\n";
		}
	}

	if ($minimize)
	{
		$newexedata .= "\x00\x00\x00\x00";
	}

	return($newexedata, $warnings, $errors);
}

sub ask
{
	$question = @_[0];
	$default = @_[1];

	print "\n$question ";
	if ((uc $default) eq 'Y')
	{
 		print '[Y/n]: ';
	}
	else
	{
		print '[y/N]: ';
	}

	$answer = <STDIN>;

 	$answer =~ s/\n//g;

	if ($answer eq '')
	{
		$answer = $default;
	}

	if ((uc $answer) eq 'Y')
	{
		return(1);
	}
}

sub dirslurp
{
	opendir(DIRHANDLE, @_[0]);
	my @filelist = readdir(DIRHANDLE);
	closedir(DIRHANDLE);
	return(@filelist);
}

sub process_hacks
{
	my $passed_binary = @_[0];

	foreach $hack (@imagehacks)
	{
		if ($hack =~ /PORTB/igs)
		{
			$passed_binary =~ s/\x8C\x01\xD3/\xEA\xEA\xEA/igs;		
			$passed_binary =~ s/\x8D\x01\xD3/\xEA\xEA\xEA/igs;		
			$passed_binary =~ s/\x8E\x01\xD3/\xEA\xEA\xEA/igs;		
			$passed_binary =~ s/\xEE\x01\xD3/\xEA\xEA\xEA/igs;		
			$passed_binary =~ s/\xCE\x01\xD3/\xEA\xEA\xEA/igs;		
		}
	}

	return($passed_binary);
}

sub shortstring
{
	my $passed_string = @_[0];
	my $maximum_length = @_[1];

	if (length($passed_string) <= $maximum_length)
	{
		return($passed_string);
	}
	return('...' . substr($passed_string, -($maximum_length - 3)));
}
sub read_configuration
{
	undef $atasm_location;
	undef $atasm_options;
	undef $ignore_exit_code;
	undef $redirect_stderr;

        open(IN, "<$configuration_file") or die("Could not read configuration file $configuration_file!\nPlease review the kit documentation on setting the location of the ATASM assembler.\n");
        foreach (<IN>)
        {
        	if (/^#/)
        	{
        		next;
        	}

                if (/.*atasm location(\s*):(\s*)([^\r\n]*)/i)
                {
                        $atasm_location = $3;
                }

		if (/.*atasm options(\s*):(\s*)([^\r\n]*)/i)
		{
			$atasm_options = $3;
		}

		if (/.*ignore exit code(\s*):(\s*)([^\r\n]*)/i)
		{
			$ignore_exit_code = ($3 =~ /on/i);
		}

		if (/.*redirect stderr(\s*):(\s*)([^\r\n]*)/i)
		{
			$redirect_stderr = ($3 =~ /on/i);
		}
        }
        close(IN);

	if ((!(-x $atasm_location)) and (!(-x "{$atasm_location}.exe")))
	{
		printf("\nIn $configuration_file: The ATASM executable does not exist at the location specified \"%s\"!\n", $atasm_location);
		exit(-1);
	}
}

sub rebuild
{
	my $rebuild_source = @_[0];
	my $rebuild_target = @_[1];

	if (!(-e $rebuild_source))
	{
		printf("[Rebuild] The source file %s does not exist.\n", $rebuild_source);
		exit(-1);
	}

	$compiler_command = "$atasm_location $atasm_options $extra_options $rebuild_source -o$rebuild_target";
	if ($redirect_stderr)
	{
		$results = `$compiler_command 2>&1`;
	}
	else
	{
		$results = `$compiler_command`;
	}

	$exit_value  = ($? >> 8);

	if ($ignore_exit_code)
	{
		$exit_value = 0;
	}

	if ($exit_value ne 0)
	{
		printf("\n[Rebuild] Error (%i) calling: %s\n\n", $exit_value, $compiler_command);
		print "$results\n";
		exit($exit_value);
	}
	else
	{
		if (!(-s $rebuild_target))
		{
			printf("[Rebuild] Assembler failed to produce %s.\n", $rebuild_target);
			exit(-1);
		}

		if ("@switches" =~ /verbose/igs)
		{
			$final_size = -s $rebuild_target;
			printf("[Rebuild] Rebuild of %s OK, file is %i bytes.\n", $rebuild_target, $final_size);
		}
	}
}

sub rebuild_all
{
	rebuild($mempack_source, $mempack_target);
	rebuild($exepack_menu_source, $exepack_menu_target);
	rebuild($exepack_up_source, $exepack_up_target);
	rebuild($softdisk_source, $softdisk_target);
	rebuild($maxdisk_source, $maxdisk_target);
	if (!($static_flash))
	{
		rebuild($flash_source, $flash_target);
	}
}
