----------------------------------------------------------------------------------
--  FreezerLogic.vhd - Freezer logic
--
--  Copyright (C) 2011-2012 Matthias Reichl <hias@horus.com>
--
--  This program is free software; you can redistribute it and/or modify
--  it under the terms of the GNU General Public License as published by
--  the Free Software Foundation; either version 2 of the License, or
--  (at your option) any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.
--
--  You should have received a copy of the GNU General Public License
--  along with this program; if not, write to the Free Software
--  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
----------------------------------------------------------------------------------

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.NUMERIC_STD.ALL;

library work;
use work.all;
use FreezerDef.all;

entity FreezerLogic is
Port (		clk: in std_logic;
		clk_register: in std_logic;
		a: in vec16;
		rw: in std_logic;
		reset_n: in std_logic;
		activate_n: in std_logic;
		dualpokey_n: in std_logic;
		output: out mem_output
	);
		
end FreezerLogic;

architecture RTL of FreezerLogic is

signal ram_bank: std_logic_vector(16 downto 12);
signal rom_bank: std_logic_vector(18 downto 13);

subtype state_type is std_logic_vector(2 downto 0);
constant state_disabled: 	state_type := "000";
constant state_half_enabled: 	state_type := "001";
constant state_startup: 	state_type := "010";
constant state_enabled: 	state_type := "100";
constant state_temporary_disabled: state_type := "101";

signal next_state: state_type := state_disabled;
signal state: state_type := state_disabled;

signal vector_access: boolean;
signal vector_a2: std_logic;
signal use_status_as_ram_address: boolean;

begin

vector_access <= (a(15 downto 3) = "1111111111111") and (rw='1');

state_machine: process(clk_register)
begin
	if (falling_edge(clk_register)) then
		if (reset_n = '0') then
			next_state <= state_disabled;
		else
			case next_state is
			when state_disabled =>
				if vector_access and (activate_n = '0') and (a(0) = '0') then
					next_state <= state_half_enabled;
					vector_a2 <= a(2);
				end if;
			when state_half_enabled =>
				if vector_access and (activate_n = '0') and (a(0) = '1') then
					next_state <= state_startup;
				else
					next_state <= state_disabled;
				end if;
			when state_startup =>
				if (a(15 downto 4) = x"D72") then
					next_state <= state_enabled;
				end if;
			when state_enabled =>
				if (a(15 downto 4) = x"D70") then
					if (rw = '1') then
						next_state <= state_disabled;
					else
						next_state <= state_temporary_disabled;
					end if;
				end if;
			when state_temporary_disabled =>
				if (a(15 downto 4) = x"D70") then
					if (rw = '1') then
						next_state <= state_disabled;
					else
						next_state <= state_enabled;
					end if;
				end if;
			when others =>
				next_state <= state_disabled;
			end case;
		end if;
	end if;
end process state_machine;

final_state_machine: process(clk)
begin
	if (falling_edge(clk)) then
		state <= next_state;
	end if;
end process final_state_machine;

set_status_ram_address: process(clk_register)
begin
	if (falling_edge(clk_register)) then
		if (next_state = state_disabled) then
			use_status_as_ram_address <= false;
		else
			if (a(15 downto 4) = x"D71") then
				use_status_as_ram_address <= (rw = '1');
			end if;
		end if;
	end if;
end process set_status_ram_address;

banksel: process(clk_register)
begin
	if (falling_edge(clk_register)) then
		if (next_state = state_disabled) then
			ram_bank <= (others => '0');
			rom_bank <= freezer_def_rom_bank(18 downto 13);
		else
			-- D740-D77F
			if (a(15 downto 6) = "1101011101") then
				rom_bank <= a(5 downto 0);
			end if;
			-- D780-D79F
			if (a(15 downto 5) = "11010111100") then
				ram_bank <= a(4 downto 0);
			end if;
		end if;
	end if;
end process banksel;

access_freezer: process(a, rw, state, vector_access, ram_bank, rom_bank, vector_a2, use_status_as_ram_address, dualpokey_n)
begin
	output.adr <= (others => '0');
	output.ram_access <= false;
	output.rom_access <= false;
	output.disable_atari <= false;
	output.dout <= (others => '0');
	output.dout_enable <= false;

	case state is
	when state_disabled | state_half_enabled =>
		-- shadow writes to D000-D7FF
		if (a(15 downto 11) = "11010") and (rw = '0') then
			output.ram_access <= true;
			output.adr(19 downto 8) <= freezer_def_ram_bank & a(11 downto 8);
			-- GTIA/D000 needs 32 bytes, others 16 bytes
			-- in dualpokey mode also shadow D2xx with 32 bytes
			if (a(10 downto 8) = "000") or (dualpokey_n = '0' and a(10 downto 8) = "010") then
				output.adr(4 downto 0) <= a(4 downto 0);
			else
				output.adr(3 downto 0) <= a(3 downto 0);
			end if;
		end if;
		if (state = state_half_enabled) and (vector_access) and (a(0) = '1') then
			-- re-route interrupt vectors to NOP slide page
			output.dout <= x"21";
			output.dout_enable <= true;
			output.disable_atari <= true;
		end if;
	when state_startup | state_enabled =>
		if (state = state_startup) and (vector_access) and (a(0) = '1') then
			-- re-route interrupt vectors to RTI slide page
			output.dout <= x"20";
			output.dout_enable <= true;
			output.disable_atari <= true;
		end if;
		-- 0000-1FFF: RAM (0000-0FFF fixed, 1000-1FFF switchable)
		if (a(15 downto 13) = "000") then
			output.ram_access <= true;
			output.disable_atari <= true;
			if (a(12) = '0') then
				output.adr(19 downto 12) <= freezer_def_ram_bank;
			else
				output.adr(19 downto 12) <= freezer_def_ram_bank(19 downto 17) & ram_bank;
			end if;
			output.adr(11 downto 0) <= a(11 downto 0);
			if (use_status_as_ram_address) then
				output.adr(4) <= vector_a2;
				output.adr(5) <= not dualpokey_n;
			end if;
		end if;
		-- 2000-3FFF: switched ROM bank
		if (a(15 downto 13) = "001") then
			output.rom_access <= true;
			output.disable_atari <= true;
			output.adr <= freezer_def_rom_bank(19) & rom_bank & a(12 downto 0);
		end if;
		-- D7xx freezer control, disable Atari memory
		if (a(15 downto 8) = x"D7") then
			output.disable_atari <= true;
		end if;
	when state_temporary_disabled =>
		-- D7xx freezer control, disable Atari memory
		if (a(15 downto 8) = x"D7") then
			output.disable_atari <= true;
		end if;
	when others => null;
	end case;
end process access_freezer;

end RTL;

