----------------------------------------------------------------------------------
--  CartEmu.vhd - Cartridge Emulation logic
--
--  Copyright (C) 2011-2012 Matthias Reichl <hias@horus.com>
--
--  This program is free software; you can redistribute it and/or modify
--  it under the terms of the GNU General Public License as published by
--  the Free Software Foundation; either version 2 of the License, or
--  (at your option) any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.
--
--  You should have received a copy of the GNU General Public License
--  along with this program; if not, write to the Free Software
--  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
----------------------------------------------------------------------------------

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.NUMERIC_STD.ALL;

library work;
use work.all;
use FreezerDef.all;

entity CartEmu is
    Port (	clk_register: in std_logic;
		a: in vec16;
		d_in: in std_logic_vector;
		rw: in std_logic;
		powerup_n: in std_logic;
		reset_n: in std_logic;
		cartemu_enable_n: in std_logic;
		cartemu_write_enable_n: in std_logic;

		output: out mem_output
	);
		
end CartEmu;

architecture RTL of CartEmu is

-- general cartemu configuration

-- D590 - bank select
-- bit 7: reserved
-- bit 6..0: bank number
-- writing this register also enables cartemu (cfg_enable/D591)
constant cfg_bank_adr: vec8 := x"90";
signal cfg_bank: std_logic_vector(6 downto 0) := "0000000";

-- D591 - cartemu enable
-- bit 7..1: reserved
-- bit 0: 0=disable 1=enable
constant cfg_enable_adr: vec8 := x"91";
signal cfg_enable: std_logic := '0';

subtype cart_mode_type is std_logic_vector(2 downto 0);
constant cart_mode_off:		cart_mode_type := "000";
constant cart_mode_8k:		cart_mode_type := "001";
constant cart_mode_8k_sram:	cart_mode_type := "010";
constant cart_mode_8k_legacy:	cart_mode_type := "011";
constant cart_mode_16k:		cart_mode_type := "100";
constant cart_mode_oss:		cart_mode_type := "101";
constant cart_mode_atarimax:	cart_mode_type := "110";
constant cart_mode_reserved:	cart_mode_type := "111";

-- D592 - separate RAM bank (for 8k-sram mode)
constant cfg_sram_bank_adr: vec8 := x"92";
-- bit 7..6: reserved
-- bit 5..0: bank number
-- writing this register also enables cartemu (cfg_enable/D591)
signal cfg_sram_bank: std_logic_vector(5 downto 0) := "000000";

-- D593 - separate RAM bank enable
constant cfg_sram_enable_adr: vec8 := x"93";
-- bit 7..1: reserved
-- bit 0: 0=disable 1=enable
signal cfg_sram_enable: std_logic := '0';

-- D594 - cartemu mode
constant cfg_mode_adr: vec8 := x"94";
-- bit 7..4: reserved
-- bit 3..0: mode select
signal cfg_mode: cart_mode_type := cart_mode_off;

-- D595 - misc config
constant cfg_misc_adr: vec8 := x"95";
-- bit 7..3: reserved
-- bit 2: 0=standard cartemu, 1=force cartemu menu
signal cfg_menu: std_logic := '0';
-- bit 1: 0=ROM, 1=RAM
signal cfg_source_ram: std_logic := '0';
-- bit 0: 0=write protect, 1=write enable
signal cfg_write_enable: std_logic := '0';

subtype usdx_mode_type is std_logic_vector(1 downto 0);
constant usdx_mode_off: usdx_mode_type := "00";
constant usdx_mode_ram: usdx_mode_type := "01";
constant usdx_mode_rom1: usdx_mode_type := "10";
constant usdx_mode_rom2: usdx_mode_type := "11";

-- D596 - ultimate 1MB SDX emulation
-- bit 7..2: reserved
-- bit 1..0: usdx mode
constant cfg_usdx_adr: vec8 := x"96";
signal cfg_usdx: usdx_mode_type := usdx_mode_off;

-- D5E0 - ultimate 1MB SDX
constant usdx_adr: vec8 := x"E0";
-- bit 7,6: sdx ctl
-- 0x : SDX  on, cartemu off
-- 10 : SDX off, cartemu  on
-- 11 : SDX off, cartemu off
signal usdx_ctl: std_logic_vector(1 downto 0) := "00";	-- SDX on, cartemu on
-- bit 5..0: sdx bank
signal usdx_bank: std_logic_vector(5 downto 0) := "000000";

-- OSS cart emulation
signal oss_bank: std_logic_vector(1 downto 0) := "00";

signal access_cctl: boolean;
signal access_8xxx: boolean;
signal access_axxx: boolean;

-- I/O to D5xx
signal cctl_disable_atari: boolean;
signal cctl_dout_enable: boolean;
signal cctl_dout: vec8;

-- D013/TRIG3 access
signal trig3_disable_atari: boolean;
signal trig3_dout_enable: boolean;
signal trig3_dout: vec8;

-- I/O to cart
signal cart_address: vec20;
signal cart_ram_access: boolean;
signal cart_rom_access: boolean;
signal cart_disable_atari: boolean;
signal cart_present: boolean;

-- trying to the this beast to fit...
attribute KEEP: String;
attribute KEEP of trig3_disable_atari: signal is "TRUE";

begin

access_cctl <= ( a(15 downto 8) = x"D5" );
access_8xxx <= ( a(15 downto 13) = "100" );
access_axxx <= ( a(15 downto 13) = "101" );

-- disable Atari memory on config access and read back config registers
config_io: process(a, rw, access_cctl,
	cartemu_enable_n, cartemu_write_enable_n,
	cfg_bank, cfg_enable, cfg_sram_bank, cfg_sram_enable,
	cfg_menu, cfg_source_ram, cfg_write_enable, cfg_mode, cfg_usdx,
	usdx_bank, usdx_ctl)
begin
	cctl_disable_atari <= false;
	cctl_dout_enable <= false;
	cctl_dout <= x"00";
	if ( (cartemu_enable_n = '0') or (cartemu_write_enable_n = '0') ) and access_cctl then
		-- freezer config (always)
		case a(7 downto 0) is
		when cfg_bank_adr =>
			cctl_disable_atari <= true;
			if (rw = '1') then
				cctl_dout_enable <= true;
				cctl_dout <= "0" & cfg_bank;
			end if;
		when cfg_enable_adr =>
			cctl_disable_atari <= true;
			if (rw = '1') then
				cctl_dout_enable <= true;
				cctl_dout <= "0000000" & cfg_enable;
			end if;
		when cfg_sram_bank_adr =>
			cctl_disable_atari <= true;
			if (rw = '1') then
				cctl_dout_enable <= true;
				cctl_dout <= "00" & cfg_sram_bank;
			end if;
		when cfg_sram_enable_adr =>
			cctl_disable_atari <= true;
			if (rw = '1') then
				cctl_dout_enable <= true;
				cctl_dout <= "0000000" & cfg_sram_enable;
			end if;
		when cfg_mode_adr =>
			cctl_disable_atari <= true;
			if (rw = '1') then
				cctl_dout_enable <= true;
				cctl_dout <= "00000" & cfg_mode;
			end if;
		when cfg_misc_adr =>
			cctl_disable_atari <= true;
			if (rw = '1') then
				cctl_dout_enable <= true;
				cctl_dout <= "00000" & cfg_menu & cfg_source_ram & cfg_write_enable;
			end if;				
		when cfg_usdx_adr =>
			cctl_disable_atari <= true;
			if (rw = '1') then
				cctl_dout_enable <= true;
				cctl_dout <= "000000" & cfg_usdx;
			end if;
		when others => null;
		end case;

		if (cfg_usdx = usdx_mode_off) then
			null;
		elsif (a(7 downto 0) = usdx_adr) then
			cctl_disable_atari <= true;
			if (rw = '1') then
				cctl_dout_enable <= true;
				cctl_dout <= usdx_ctl & usdx_bank;
			end if;
		end if;
		
		-- other cart specific control registers
		case cfg_mode is
		when cart_mode_8k_legacy =>
			-- D540-D57F, D580-D58F
			if (a(7 downto 6) = "01") or (a(7 downto 4) = "1000") then
				cctl_disable_atari <= true;
			end if;
		when cart_mode_oss =>
			-- D500-D50F
			if (a(7 downto 4) = "0000") then
				cctl_disable_atari <= true;
			end if;
		when cart_mode_atarimax =>
			-- D500-D57F, D580-D58F
			if (a(7) = '0') or (a(7 downto 4) = "1000") then
				cctl_disable_atari <= true;
			end if;
		when others => null;
		end case;
	end if;
end process config_io;

set_config: process(clk_register)
begin
	if falling_edge(clk_register) then
		if (reset_n = '0') then
			cfg_write_enable <= '0';
			cfg_sram_enable <= '0';
			if (powerup_n = '0') then
				cfg_enable <= '0';
				cfg_source_ram <= '0';
				cfg_usdx <= usdx_mode_off;
				usdx_ctl <= "10";
				cfg_mode <= cart_mode_off;
-- registers power up to low, so we can skip initialization
--				cfg_sram_bank <= "000000";
--				cfg_bank <= "0000000";
--				usdx_bank <= "000000";
--				oss_bank <= "00";
				if (cartemu_enable_n = '0') then
					cfg_menu <= '1';
				else
					cfg_menu <= '0';
				end if;
			end if;
		elsif ( (cartemu_enable_n = '0') or (cartemu_write_enable_n = '0') ) and access_cctl then
			if (rw = '0') then
				-- freezer config (always)
				case a(7 downto 0) is
				when cfg_bank_adr =>
					cfg_bank <= d_in(6 downto 0);
					cfg_enable <= '1';	-- automatically enable cartemu on bank access
				when cfg_enable_adr =>
					cfg_enable <= d_in(0);
				when cfg_sram_bank_adr =>
					cfg_sram_bank <= d_in(5 downto 0);
					cfg_sram_enable <= '1';	-- automatically enable sram on bank access
				when cfg_sram_enable_adr =>
					cfg_sram_enable <= d_in(0);
				when cfg_mode_adr =>
					cfg_mode <= d_in(2 downto 0);
				when cfg_misc_adr =>
					cfg_menu <= d_in(2);
					cfg_source_ram <= d_in(1);
					cfg_write_enable <= d_in(0);
				when cfg_usdx_adr =>
					cfg_usdx <= d_in(1 downto 0);
				when others => null;
				end case;
				
				-- usdx config (when usdx is enabled)
				if (cfg_usdx /= usdx_mode_off) and (a(7 downto 0) = usdx_adr) then
					usdx_ctl <= d_in(7 downto 6);
					usdx_bank <= d_in(5 downto 0);
				end if;
			end if;
			
			-- cart config using addresses, ignore read/write
			case cfg_mode is
			when cart_mode_8k_legacy =>
				-- D540-D57F
				if (a(7 downto 6) = "01") then
					cfg_bank(5 downto 0) <= a(5 downto 0);
				end if;
				-- D580/D581: disable/enable
				if (a(7 downto 1) = "1000000") then
					cfg_enable <= a(0);
				end if;
				-- D584/D585: write disable/enable cart
				if (a(7 downto 1) = "1000010") then
					cfg_write_enable <= a(0);
				end if;
			when cart_mode_oss =>
				-- D500-D50F
				if (a(7 downto 4) = "0000") then
					oss_bank <= a(0) & NOT a(3);
				end if;
			when cart_mode_atarimax =>
				-- D500-D57F: set bank, enable
				if (a(7) = '0') then
					cfg_bank <= a(6 downto 0);
					cfg_enable <= '1';
				end if;
				-- D580-D58F: disable
				if (a(7 downto 4) = "1000") then
					cfg_enable <= '0';
				end if;
			when others => null;
			end case;
		end if;
	end if;
end process set_config;

access_cart_data: process(a, rw, access_8xxx, access_axxx,
	cartemu_write_enable_n,
	cfg_bank, cfg_enable, cfg_sram_bank, cfg_sram_enable,
	cfg_menu, cfg_source_ram, cfg_write_enable, cfg_mode, cfg_usdx,
	usdx_ctl, usdx_bank, oss_bank)
variable address: vec20;
variable source_ram: boolean;
variable do_access: boolean;
variable allow_write: boolean;

begin
	-- default values
	cart_address <= (others => '0');
	cart_ram_access <= false;
	cart_rom_access <= false;
	cart_disable_atari <= false;
	cart_present <= false;

	address := cfg_bank & a(12 downto 0);
	source_ram := ( cfg_source_ram = '1');
	do_access := false;
	allow_write := (cartemu_write_enable_n = '0') and (cfg_write_enable = '1');

	if cfg_menu = '1' then
		do_access := access_axxx;
		address(19 downto 13) := cartemu_menu_bank;
		source_ram := false;
		allow_write := false;
		cart_present <= true;
	else
		if (cfg_usdx /= usdx_mode_off) and (usdx_ctl(1) = '0') then
			do_access := access_axxx;
			address(18 downto 13) := usdx_bank;
			cart_present <= true;
			source_ram := (cfg_usdx = usdx_mode_ram);
			if (cfg_usdx = usdx_mode_rom2) then
				address(19) := '1';
			else
				address(19) := '0';
			end if;
		end if;
		if (cfg_usdx = usdx_mode_off) or (usdx_ctl = "10") then
			cart_present <= (cfg_enable = '1');
			do_access := (cfg_enable = '1') and access_axxx;
			case cfg_mode is
			when cart_mode_8k | cart_mode_atarimax =>
				null;
			when cart_mode_8k_legacy =>
				cart_present <= false;	-- Turbo Freezer 2005 doesn't signal cart presence via TRIG3
			when cart_mode_16k =>
				address(13) := a(13);
				do_access := (cfg_enable = '1') and (access_8xxx or access_axxx);
			when cart_mode_oss =>
				if (oss_bank = "00") then
					cart_present <= false;
					do_access := false;
				else
					address(13) := oss_bank(1) and (not a(12));
					address(12) := oss_bank(0) and (not a(12));
				end if;
			when cart_mode_8k_sram =>
				if (cfg_sram_enable = '1') and access_8xxx then
					address(19 downto 13) := "0" & cfg_sram_bank;
					do_access := true;
					allow_write := true;
					source_ram := true;
				end if;
			when others =>
				cart_present <= false;
				do_access := false;
				allow_write := false;
				source_ram := false;
			end case;
		end if;
	end if;

	cart_disable_atari <= do_access;

	if source_ram and
	   ( (address(19 downto 19) = ramdisk_base_bank) or
	     (address(19 downto 14) = freezer_reserved_ram) ) then
		do_access := false;
	end if;
	if  (rw = '0') and (allow_write = false) then
		do_access := false;
	end if;

	if (do_access) then
		cart_address <= address;
		if (source_ram) then
			cart_ram_access <= true;
		else
			cart_rom_access <= true;
		end if;
	end if;
end process access_cart_data;

access_trig3: process(a, rw, cart_present)
begin
	if (a = x"D013") and (rw = '1') and cart_present then
		trig3_disable_atari <= true;
		trig3_dout <= x"01";
		trig3_dout_enable <= true;
	else
		trig3_disable_atari <= false;
		trig3_dout <= x"00";
		trig3_dout_enable <= false;
	end if;
end process access_trig3;

access_cart: process(a, rw, 
	cart_address, cart_ram_access, cart_rom_access, cart_disable_atari, cart_present,
	cctl_disable_atari, cctl_dout, cctl_dout_enable,
	trig3_disable_atari, trig3_dout, trig3_dout_enable)
begin
	output.adr <= cart_address;
	output.ram_access <= cart_ram_access;
	output.rom_access <= cart_rom_access;

	output.disable_atari <= cart_disable_atari or cctl_disable_atari or trig3_disable_atari;
	output.dout <= cctl_dout or trig3_dout;
	output.dout_enable <= cctl_dout_enable or trig3_dout_enable;
end process access_cart;

end RTL;

