----------------------------------------------------------------------------------
--  TurboFreezer.vhd - Turbo Freezer 2011 main file
--
--  Copyright (C) 2011-2012 Matthias Reichl <hias@horus.com>
--
--  This program is free software; you can redistribute it and/or modify
--  it under the terms of the GNU General Public License as published by
--  the Free Software Foundation; either version 2 of the License, or
--  (at your option) any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.
--
--  You should have received a copy of the GNU General Public License
--  along with this program; if not, write to the Free Software
--  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
----------------------------------------------------------------------------------

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;

library work;
use work.all;
use FreezerDef.all;

entity TurboFreezer is
    Port( a : in vec16;
          d : inout vec8;
	  rw: in std_logic;
	  ram_a: out std_logic_vector(7 downto 4);
	  ram_rom_a: out std_logic_vector(18 downto 12);
	  ram_rom_oe: out std_logic := '1';
	  ram_rom_we: out STD_LOGIC := '1';
	  ram0_ce: out std_logic := '1';
	  ram1_ce: out std_logic := '1';
	  rom0_ce: out std_logic := '1';
	  rom1_ce: out std_logic := '1';
	  refresh: out std_logic := 'Z';
	  phi2 : in std_logic;
	  phi2short: in std_logic;
	  activate_n_in: in std_logic;
	  oldos_n: in std_logic;
	  flash_we_n: in std_logic;
	  cartemu_enable_n: in std_logic;
	  ramdisk_enable_n: in std_logic;
	  dualpokey_n: in std_logic;
	  reset_n_in: in std_logic;
	  -- currently unused, but connected I/Os, location defined in UCF file!
	  extsel_in: in std_logic;
	  mpd_in: in std_logic;
	  irq_in: in std_logic;
	  -- connected on MEM2 board, but driven by PBI
	  nc_adr: in std_logic_vector(1 to 4)
	);

	attribute LOC: String;
	attribute LOC of a: signal is "P15 P14 P13 P12 P11 P10 P9 P8 P6 P4 P3 P1 P36 P35 P50 P42";
	attribute LOC of d: signal is "P30 P29 P28 P25 P20 P18 P17 P16";
	attribute LOC of rw: signal is "P33";
	attribute LOC of ram_a: signal is "P95 P94 P93 P92";
	attribute LOC of ram_rom_a: signal is "P91 P90 P89 P87 P86 P85 P82";
	attribute LOC of ram_rom_oe: signal is "P78";
	attribute LOC of ram_rom_we: signal is "P77";
	attribute LOC of ram0_ce: signal is "P81";
	attribute LOC of ram1_ce: signal is "P96";
	attribute LOC of rom0_ce: signal is "P79";
	attribute LOC of rom1_ce: signal is "P97";
	attribute LOC of phi2: signal is "P22";
	attribute LOC of phi2short: signal is "P27";
	attribute LOC of activate_n_in: signal is "P46";
	attribute LOC of oldos_n: signal is "P43";
	attribute LOC of flash_we_n: signal is "P19";
	attribute LOC of cartemu_enable_n: signal is "P7";
	attribute LOC of reset_n_in: signal is "P99";
	attribute LOC of refresh: signal is "P73";
	attribute LOC of ramdisk_enable_n: signal is "P24";
	attribute LOC of dualpokey_n: signal is "P34";

	attribute SLOW: String;
	attribute SLOW of TurboFreezer: entity is "TRUE";

	attribute BUFG: String;
	attribute BUFG of phi2: signal is "clk";
	attribute BUFG of phi2short: signal is "clk";

end TurboFreezer;

architecture RTL of TurboFreezer is

	signal pia_portb: vec8;
	
	signal freezer_mem: mem_output;
	signal cartemu_mem: mem_output;
	signal oldos_mem: mem_output;
	signal ramdisk_mem: mem_output;

	signal merged_out: mem_output;

	signal d_in: vec8;

	-- synchronizers and signals for async inputs
	signal activate_n_sync: std_logic_vector(1 downto 0) := "00";
	signal activate_n: std_logic;
	signal reset_n_sync: std_logic_vector(3 downto 0) := "0000";
	signal reset_n: std_logic;
	signal powerup_n: std_logic := '0';

	signal dout_enable: boolean;
	-- fix combinatorial loop warning of ISE when outputting data
	attribute NOREDUCE: string;
	attribute NOREDUCE of dout_enable: signal is "true";
begin
	use_freezer: entity FreezerLogic
	port map (
		clk => phi2,
		clk_register => phi2short,
		a => a,
		rw => rw,
		reset_n => reset_n,
		activate_n => activate_n,
		dualpokey_n => dualpokey_n,
		output => freezer_mem
	);

	use_cartemu: entity CartEmu
	port map (
		clk_register => phi2short,
		a => a,
		d_in => d_in,
		rw => rw,
		powerup_n => powerup_n,
		reset_n => reset_n,
		cartemu_enable_n => cartemu_enable_n,
		cartemu_write_enable_n => flash_we_n,
		output => cartemu_mem
	);

	use_oldos: entity OldOs
	port map (
		a => a,
		rw => rw,
		oldos_enable_n => oldos_n,
		output => oldos_mem
	);

	use_pia: entity PIA
	port map (
		clk_register => phi2short,
		a => a,
		d_in => d_in,
		rw => rw,
		reset_n => reset_n,
		portb => pia_portb
	);

	use_ramdisk: entity Ramdisk
	port map (
		portb => pia_portb,
		a => a,
		ramdisk_enable_n => ramdisk_enable_n,
		output => ramdisk_mem
	);

	synchronize_inputs: process(phi2)
	begin
		if (falling_edge(phi2)) then
			activate_n_sync <= activate_n_sync(0) & activate_n_in;
			reset_n_sync <= reset_n_sync(2 downto 0) & reset_n_in;
			if (reset_n_sync(3) = '0') and (reset_n_sync(2) = '1') and (powerup_n = '0') then
				powerup_n <= NOT powerup_n;
			end if;
		end if;
	end process synchronize_inputs;
	activate_n <= activate_n_sync(1);
	reset_n <= reset_n_sync(1);

	merge_signals: process(freezer_mem, cartemu_mem, oldos_mem, ramdisk_mem)
	begin
		if (freezer_mem.ram_access or freezer_mem.rom_access or freezer_mem.disable_atari) then
			merged_out <= freezer_mem;
		else
			merged_out <= cartemu_mem or oldos_mem or ramdisk_mem;
		end if;
	end process merge_signals;

	set_ram_rom: process(merged_out, a, rw, phi2, phi2short) 
	begin
		ram0_ce <= '1';
		ram1_ce <= '1';
		rom0_ce <= '1';
		rom1_ce <= '1';
		ram_rom_oe <= '1';
		ram_rom_we <= '1';
		refresh <= 'Z';

		if (phi2 = '1') then
			if (merged_out.ram_access and (merged_out.adr(19) = '0')) then
				ram0_ce <= '0';
			end if;
			if (merged_out.ram_access and (merged_out.adr(19) = '1')) then
				ram1_ce <= '0';
			end if;
			if (merged_out.rom_access and (merged_out.adr(19) = '0')) then
				rom0_ce <= '0';
			end if;
			if (merged_out.rom_access and (merged_out.adr(19) = '1')) then
				rom1_ce <= '0';
			end if;
		end if;
		ram_a <= merged_out.adr(7 downto 4);
		ram_rom_a(18 downto 12) <= merged_out.adr(18 downto 12);
		if (rw = '1') then
			ram_rom_oe <= NOT phi2;
		else
			ram_rom_we <= NOT phi2short;
		end if;
		if merged_out.disable_atari then
			refresh <= '0';
		end if;
	end process set_ram_rom;

	dout_enable <= (phi2 = '1') and (rw = '1') and merged_out.dout_enable;
	d <= merged_out.dout when dout_enable else (others => 'Z');
	d_in <= d;
end RTL;
